/* pioneer_driver.c: command routines for Pioneer LD-V6000A videodisc player */
/* Copyright 1989 Tektronix, Inc. */
/* All Rights Reserved */

#include <stdio.h>
#include "structs.h"
#include "rpd_driver.h"
#include "pioneer.h"
#include "ctrl.h"
typedef unsigned char byte;

#define MAXRETRYS 0                /* # of times we try before giving up */
#define PIONEERDELAY 10            /* 10 ms between chars for pioneer */
#define PIONEER_STD_WAIT 0    /* normal time before a timeout occurs */

extern int ttyutil_debug;
PioneerStatus stat;

int pioneer_debug = 0;

pioneer_cmd(rpd,num)
RPD_ptr    rpd;
int     num;
{
    char *cmd;
    int res;
    int forked = 0;
    int i;

    switch (num) {
    case INDEX_ON_CMD:
	cmd = PIONEER_INDEXON;
	res = pioneer_ttyio(rpd->fd,cmd,5,0);
	rpd->state.indexon = 1;
	break;
    case INDEX_OFF_CMD:
	cmd = PIONEER_INDEXOFF;
	res = pioneer_ttyio(rpd->fd,cmd,5,0);
	rpd->state.indexon = 0;
	break;
    case INDEX_TOGGLE_CMD:
	cmd = PIONEER_TOGGLE;
	res = pioneer_ttyio(rpd->fd,cmd,3,0);
	rpd->state.indexon = !rpd->state.indexon;
	break;
    case A1_ON_CMD:
	cmd = PIONEER_CH1ON;
	res = pioneer_ttyio(rpd->fd,cmd,5,0);
	rpd->state.audio1_on = 1;
	break;
    case A1_OFF_CMD:
	cmd = PIONEER_CH1OFF;
	res = pioneer_ttyio(rpd->fd,cmd,5,0);
	rpd->state.audio1_on = 0;
	break;
    case A2_ON_CMD:
	cmd = PIONEER_CH2ON;
	res = pioneer_ttyio(rpd->fd,cmd,5,0);
	rpd->state.audio2_on = 1;
	break;
    case A2_OFF_CMD:
	cmd = PIONEER_CH2OFF;
	res = pioneer_ttyio(rpd->fd,cmd,5,0);
	rpd->state.audio2_on = 0;
	break;
    case FORKLOAD_CMD:
	forked = 1;
	if (fork())
	    return(0);
    case LOAD_CMD:
	pioneer_getstatus(rpd);
	if ( !stat.motorOff ) {
	    res = PIONEER_ACK;
	    break;
	}
	if ( stat.lidOpen ) {
	    res = PIONEER_NACK;
	    break;
	}

	cmd = PIONEER_PLAY;
	res = pioneer_ttyio(rpd->fd,cmd,3,0);
	for (i=0; i<30; i++) {
	    pioneer_getstatus(rpd);
	    if ( !stat.motorOff ) {
		res = PIONEER_ACK;
		break;
	    }
	    sleep (1);
	}
	/* can get here if timed out, or if loaded */

	if (res == PIONEER_ACK) {
	    rpd->state.volume_loaded = 1;
	    strcpy(rpd->volume,"#ffffff");    /* it's empty */
	    cmd = PIONEER_SETFRAME1;
	    pioneer_ttyio(rpd->fd,cmd,5,0);
	}

	rpd_setframe (rpd,1);
	rpd_setspeed (rpd,0);
	break;
    case UNLOAD_CMD:
	pioneer_getstatus(rpd);
	if ( stat.motorOff || stat.lidOpen ) {
	    res = PIONEER_ACK;
	    break;
	}

	cmd = PIONEER_REJECT;
	pioneer_ttyio(rpd->fd,cmd,3,0);

	pioneer_getstatus(rpd);
	if ( stat.motorOff ) {    /* now set state */
	    rpd->state.volume_loaded = 0;
	    rpd->volume[0] = 0;
	    res = PIONEER_ACK;
	}
	rpd_setframe (rpd,-1);
	rpd_setspeed (rpd,0);
	break;

    default:
	fprintf(stderr,"pioneer_cmd: illegal request %d\n",cmd);
	break;
    }
    if (forked)
	_exit(0);
    if ( res != PIONEER_ACK )
	return(-1);
    else
	return(0);
}

char *
itop(num)
int num;
{
    static char nums[]="30842A6195";
    static char buf1[32];
    static char buf2[32];
    char *bp1,*bp2;

    bp1 = buf1;
    do {
	*bp1++ = 'F';
	*bp1++ = nums[num%10];
    } while (num = num/10);

    buf2[0] = '@';    /* guarantees no nibbles to interfere */
    bp2 = buf2+1;

    while (--bp1 != buf1)
	*bp2++ = *bp1;

    *bp2++ = *bp1;    /* grab the last char */
    *bp2 = 0;        /* null-terminate */

    return buf2;
}

pioneer_varspeed(rpd,num)
RPD_ptr    rpd;
int    num;
{
    char *cmd;
    int    res;
    int realspeed;

    if (num < -120) {
	cmd = PIONEER_RSCAN;
	res = pioneer_ttyio(rpd->fd,cmd,7,0);
	realspeed = -3000;
    }
    if (num < -60 && num > -121 ) {
	cmd = PIONEER_RFAST;
	res = pioneer_ttyio(rpd->fd,cmd,7,0);
	realspeed = -90;
    }
    if (num < -29 && num > -61 ) {
	cmd = PIONEER_RPLAY;
	res = pioneer_ttyio(rpd->fd,cmd,7,0);
	realspeed = -30;
    }
    if (num < 0 && num > -30 ) {
	cmd = itop(30/-num);
	res = pioneer_ttyio(rpd->fd,cmd,strlen(cmd),0);
	cmd = PIONEER_RSLOW;
	res = pioneer_ttyio(rpd->fd,cmd,5,0);
	realspeed = num;
    }      
    if (num == 0) {
	cmd = PIONEER_STILL;
	res = pioneer_ttyio(rpd->fd,cmd,3,0);
	realspeed = 0;
    }
    if (num > 120) {
	cmd = PIONEER_FSCAN;
	res = pioneer_ttyio(rpd->fd,cmd,7,0);
	realspeed = 3000;
    }
    if (num > 60 && num < 121 ) {
	cmd = PIONEER_FFAST;
	res = pioneer_ttyio(rpd->fd,cmd,7,0);
	realspeed = 90;
    }
    if (num > 29 && num < 61 ) {
	cmd = PIONEER_FPLAY;
	res = pioneer_ttyio(rpd->fd,cmd,3,0);
	realspeed = 30;
    }
    if (num > 0 && num < 30 ) {
	cmd = itop(30/num);
	res = pioneer_ttyio(rpd->fd,cmd,strlen(cmd),0);
	cmd = PIONEER_FSLOW;
	res = pioneer_ttyio(rpd->fd,cmd,5,0);
	realspeed = num;
    }      

    if ( res != PIONEER_ACK )
	return(COULDNT_CHANGE_SPEED);
    else {
	rpd_setspeed (rpd, realspeed);
	rpd_setframe (rpd, -1);
	return(NO_ERROR);
    }
}

pioneer_jog(rpd,num)
RPD_ptr    rpd;
int    num;
{
    int    i;
    char *dir;

    if (num > 0)
	dir = PIONEER_FSTEP;
    else
	dir = PIONEER_RSTEP;
    if (num < 0)
	num = -num;

    for ( i=0; i<num; i++ ) {
	if ( pioneer_ttyio(rpd->fd,dir,3,0) )
	    return (-1);
	mpause(33);    /* 33 millisecs */
    }
    rpd_setspeed (rpd,0);
    if (rpd_getframe(rpd) < 0)
	if (pioneer_getframe(rpd) == -1)
	    return (-1);
    if (strcmp(PIONEER_FSTEP,dir) == 0)
	rpd_setframe (rpd, rpd_getframe(rpd)+1);
    else
	rpd_setframe (rpd, rpd_getframe(rpd)-1);

    return (0);
}

pioneer_getframe(rpd)
RPD_ptr    rpd;
{
    char *cmd;
    int frame;

/*     if (rpd_getframe(rpd) > 0) */
/*         return (rpd_getframe(rpd)); */
    cmd = PIONEER_ADDRINQ;
    if ( pioneer_ttyio(rpd->fd,cmd,3,5) == -1 )
	return (-1);
    stat.inchars[5] = '\0';
    sscanf(stat.inchars,"%x",&frame);
    if (frame == 0xffff)
	frame = 0;
    rpd_setframe (rpd, frame);
    return (frame);
}

pioneer_getstatus(rpd)
RPD_ptr    rpd;
{
    char *cmd;

    stat.scanExec = FALSE;
    stat.searchExec = FALSE;
    stat.autoStop = FALSE;
    stat.inputWhilePlaying = FALSE;
    stat.inputAtFreezeFrame = FALSE;
    stat.playNormalForward = FALSE;
    stat.freezeFrame = FALSE;
    stat.playMultiFwd = FALSE;
    stat.playMultiRev = FALSE;
    stat.spinUp = FALSE;
    stat.parked = FALSE;
    stat.rejecting = FALSE;
    stat.lidOpen = FALSE;
    stat.motorOff = FALSE;

    cmd = PIONEER_STATUSINQ;
    if ( pioneer_ttyio(rpd->fd,cmd,3,3) == PIONEER_NACK ) {
	return (-1);
    }
    /* for debugging */
    stat.status[0] = stat.inchars[0];
    stat.status[1] = stat.inchars[1];

    switch (stat.inchars[0]) {
    case '4':
	if (stat.inchars[1] == 'C') {
	    stat.scanExec = TRUE;
	} else {
	    fprintf(stderr,"%s: illegal status 4%c\n",
		rpd->port, stat.inchars[1]);
	    bsdsyslog(LOG_NOTICE, "pioneer_getstatus: illegal status 4%c on %s",
		stat.inchars[1],rpd->port);
	}
	break;
    case '5':
    case 'D':
	if (stat.inchars[1] == '0') {
	    stat.searchExec = TRUE;
	    break;
	} else if (stat.inchars[1] == '5') {
	    stat.autoStop = TRUE;
	    break;
	} else {
	    fprintf(stderr,"%s: illegal status 5/D%c\n",
		rpd->port, stat.inchars[1]);
	    bsdsyslog(LOG_NOTICE,
		"pioneer_getstatus: illegal status 5/D%c on %s",
		stat.inchars[1],rpd->port);
	}
	break;
    case '6':
    case 'E':
	if (stat.inchars[0] == 'E' && stat.inchars[1] == '0') {
	    stat.inputWhilePlaying = TRUE;
	    break;
	} else if (stat.inchars[0] == 'E' && stat.inchars[1] == '1') {
	    stat.inputAtFreezeFrame = TRUE;
	    break;
	} else if (stat.inchars[1] == '4') {
	    stat.playNormalForward = TRUE;
	    break;
	} else if (stat.inchars[1] == '5') {
	    stat.freezeFrame = TRUE;
	    break;
	} else if (stat.inchars[1] == '6') {
	    stat.playMultiFwd = TRUE;
	    break;
	} else if (stat.inchars[1] == '7') {
	    stat.playMultiRev = TRUE;
	    break;
	} else if (stat.inchars[1] == '8') {
	    stat.spinUp = TRUE;
	    break;
	} else {
	    fprintf(stderr,"%s: illegal status 6/E%c\n",
		rpd->port, stat.inchars[1]);
	    bsdsyslog(LOG_NOTICE,
		"pioneer_getstatus: illegal status 6/E%c on %s",
		stat.inchars[1],rpd->port);
	}
	break;
    case '7':
	if (stat.inchars[1] == '8') {
	    stat.parked = TRUE;
	    break;
	} else if (stat.inchars[1] == 'C') {
	    stat.rejecting = TRUE;
	    break;
	} else {
	    fprintf(stderr,"%s: illegal status 7%c\n",
		rpd->port, stat.inchars[1]);
	    bsdsyslog(LOG_NOTICE,
		"pioneer_getstatus: illegal status 7%c on %s",
		stat.inchars[1],rpd->port);
	}
    }

    stat.motorOff = ( stat.spinUp | stat.parked | stat.rejecting );
    stat.lidOpen = ( stat.parked | stat.rejecting );
    return 0;
}

pioneer_search(rpd,frame,wait)
RPD_ptr    rpd;
int    frame;
int    wait;        /* if true, hangs until it gets there or times out */
{
    char *cmd;
    int res;
#ifdef XDEBUG
    int searches = 0;
#endif /* XDEBUG */

    /*
     *    no funny business by trying to go to negative frames
     */
    if (frame < 1)
	frame = 1;

    /*
     *    are we already there?  If so, save some time
     */
    res = pioneer_getframe(rpd);
    if (res == -1)
	return(-1);

    if (frame == res)
	return(frame);

    /*
     * Check for Park or Leadout.  Search will not work here.
     *  Need to RUN to frame 1 then search
     */
    if (res == 0)
	pioneer_cmd(rpd,LOAD_CMD);


    cmd = itop(frame);
    res = pioneer_ttyio(rpd->fd,cmd,strlen(cmd),0);
    if (res == -1)
	return(-1);
    cmd = PIONEER_SEARCH;
    res = pioneer_ttyio(rpd->fd,cmd,3,0);
    if (res == -1)
	return(-1);

    if ( !wait ) {        /* don't wait for reply */
	return(0);
    } else
	while (1) {
	    res = pioneer_getframe(rpd);
	    if (res == -1)
		return(-1);
	    if (res == frame)
		break;
#ifdef XDEBUG
	    searches++;
#endif /* XDEBUG */
	    mpause(33);    /* 33 millisecs */
	}
#ifdef XDEBUG
    pioneer_getstatus(rpd);
    printf("%d srchs, frm %d, stat=%c%c\n",
	searches, frame, stat.status[0], stat.status[1]);
#endif /* XDEBUG */

    rpd_setspeed(rpd,0);
    rpd_setframe(rpd,-1);
    res = pioneer_getframe(rpd);
    if (res == -1)
	return(-1);
    return(0);
}

pioneer_reset(rpd)
RPD_ptr    rpd;
{
    char *cmd;
    int res;

    cmd = PIONEER_REJECT;
    res = pioneer_ttyio(rpd->fd,cmd,3,0);
    rpd_setframe(rpd,0);
    rpd->state.volume_loaded = 0;
    rpd->state.not_responding = 0;
    return (res);
}

pioneer_spinup(rpd, wait)
RPD_ptr    rpd;
int	   wait;
{
    char *cmd;
    int res;

    cmd = itop(1);
    res =pioneer_ttyio(rpd->fd,cmd,strlen(cmd),0);
    if (res == -1)
	return(-1);
    cmd = PIONEER_AUTOSTOP;
    res =pioneer_ttyio(rpd->fd,cmd,3,0);
    if (res == -1)
	return(-1);
    if (!wait)
	return(0);
    while (1) {
	res = pioneer_getframe(rpd);
	if (res == -1)
	    return(-1);
	if (res == 1)
	    break;
	mpause(33);    /* 33 millisecs */
    }

    return 0;
}
pioneer_segplay(rpd,f1,f2,speed,wait)
RPD_ptr    rpd;
int f1,f2;
int speed;
int    wait;        /* if true, hangs until done */
{
    char *cmd;
    int res;

    if ( f1 > f2 || speed < 0) {
	fprintf(stderr,"%s: segplay: can only play forward\n", rpd->port);
	return(-1);
    }

    res = pioneer_getframe(rpd);
    if (res == -1)
	return(-1);
    /*
     * Check for Park or Leadout.  Search will not work here.
     *  Need to RUN to frame 1 then search
     */
    if (res == 0)
	pioneer_cmd(rpd,LOAD_CMD);
    res = pioneer_search( rpd, f1, 1 );    /* wait for the reply */
    if (res == -1)
	return(-1);

    cmd = itop(f2);
    res =pioneer_ttyio(rpd->fd,cmd,strlen(cmd),0);
    if (res == -1)
	return(-1);
    cmd = PIONEER_AUTOSTOP;
    res =pioneer_ttyio(rpd->fd,cmd,3,0);
    if (res == -1)
	return(-1);

    if ( !wait ) {
	return(0);
    } else
	while (1) {
	    res = pioneer_getframe(rpd);
	    if (res == -1)
		return(-1);
	    if (res == f2)
		break;
	    mpause(33);    /* 33 millisecs */
	}

    return(0);
}

pioneer_ttyio( fd, cmdstr, writenum, readnum )
int    fd;
char *cmdstr;
int    writenum;
int    readnum;
{
    int    num = 0;

    ttyutil_debug = pioneer_debug;
    if (ttyutil_clear(fd) == -1)
	return (PIONEER_NACK);

    if ( ttyutil_write( fd,cmdstr,writenum,PIONEERDELAY ) == -1 )
	return (PIONEER_NACK);

    if ( readnum == 0 )
	return (0);

    if ( (num=ttyutil_read(fd,stat.inchars,readnum,PIONEER_STD_WAIT)) == -1) {
	return (PIONEER_NACK);    /* timeout after .5 seconds */
    }

    if ( num==-2 ) {
	if ( pioneer_debug ) {
	    if ( num == -2 )
		fprintf(stderr,
		    "pioneer_ttyio: NO RESPONSE, try %d\n",stat.retry+1);
	    else
		fprintf(stderr,
		    "pioneer_ttyio: ERROR RESPONSE, try %d\n",stat.retry+1);
	}
	if ( stat.retry++ < MAXRETRYS ) {
	    int rv;
	    rv = pioneer_ttyio(fd,cmdstr,writenum,readnum);
	    stat.retry = 0;
	    return (rv);
	} else {
	    fprintf(stderr,"pioneer_ttyio: i/o timeout\n");
	    bsdsyslog(LOG_NOTICE, "pioneer_ttyio: i/o timeout");
	    return (PIONEER_NACK);
	}
    }

    if ( readnum==1 )    /* search - completion, nottarget, error */
	return ((int)stat.inchars[0]);
    return(0);
}

pioneer_setfuncs( disc )
RPD_ptr    disc;
{
    disc->reset = pioneer_reset;
    disc->cmd = pioneer_cmd;
    disc->search = pioneer_search;
    disc->segplay = pioneer_segplay;
    disc->getframe = pioneer_getframe;
    disc->varspeed = pioneer_varspeed;
    disc->jog = pioneer_jog;
    disc->record = NULL;
}
