#!/usr/gnu/bin/perl
# libgnats.pl - a Perl interface to the GNATS bug tracking system

### Modification log:
# 2/1/95  Dan Kegel  Split off from wwwgnats.pl
# 3/1/95  Dan Kegel  Added pr_addr function
### End Modifcation log

#### Configuration begins here
# Gnats
$GNATS_BIN  = "/usr/gnu/bin";
$GNATS_LIB  = "/usr/gnu/lib";
$GNATS_ROOT = "$GNATS_LIB/gnats/gnats-db";
$GNATS_ADM  = "$GNATS_ROOT/gnats-adm";
$PR_EDIT    = "$GNATS_LIB/gnats/pr-edit";
$PR_ADDR    = "$GNATS_LIB/gnats/pr-addr";
$GNATS_VER   = "3.2";
$GNATS_ADDR  = "bugs";

### Configuration ends here

#################### Array definitions
# Query-pr's -i option outputs the following fields numerically.
# Define arrays to map numbers to name.

# Standard GNATS arrays
@nSeverity = ("", "critical", "serious", "non-critical");
@nPriority = ("", "high", "medium", "low");
@nState    = ("", "open", "analyzed", "suspended", "feedback", "closed");
@nClass    = ("", "sw-bug", "doc-bug", "support", "change-request", "mistaken", "duplicate");

# Single-line field names
@fieldnames_single = (
		"Submitter-Id",
		"Originator",
		"Organization",
		"Confidential",
		"Synopsis",
		"Severity",
		"Priority",
		"Category",
		"Class",
		"Release",
		"State",
		"Responsible",
		"Arrival-Date",
	    );
%fieldnames_single = (
		"Submitter-Id", 1,
		"Originator", 1,
		"Organization", 1,	# Doc is ambiguous about whether Org is multi-text 
		"Confidential", 1,
		"Synopsis", 1,
		"Severity", 1,
		"Priority", 1,
		"Category", 1,
		"Class", 1,
		"Release", 1,
		"State", 2,		# The fields with "2" are not submitted in SPRs.
		"Responsible", 2,
		"Arrival-Date", 2,
	    );

# Multiple-line field names
@fieldnames_multi = (
		"Environment",
		"Description",
		"How-To-Repeat",
		"Fix",
		"Audit-Trail",
		"Unformatted",
	    );
%fieldnames_multi = (
		"Environment", 1,
		"Description", 1,
		"How-To-Repeat", 1,
		"Fix", 1,
		"Audit-Trail", 2,
		"Unformatted", 1,
	    );

&read_category;
&read_responsible;

# Convert to lower case
sub tolower
{
    local($str) = $_[0];

    $str =~ tr/A-Z/a-z/;
    return $str;
}

# Parse a problem report into fields.
# Call with problem report in an array.
# Fills the associative array %fieldvalues with the results.
sub parse_pr
{
    local($hdr,$arg,$hdr_nogt);
    local($hdr_multi) = "envelope";  # File is email envelope until first gnats fieldname.
    undef %fieldvalues;
    #print "parse_pr: <pre>\n";
    foreach (@_) {
	# skip non-headers
	if (!/^([>\w\-]+):\s*(.*)\s*$/) {
	    if ($hdr_multi ne "") {
		$fieldvalues{$hdr_multi} .= $_;
		#print "Appending to multi-line header $hdr_multi: $_\n";
	    }
	    next;
	}

	$hdr = $1;
	$arg = $2;
	$hdr_nogt = "*not a valid field name*";
	if ($hdr =~ /^>(.*)$/) {
	    $hdr_nogt = $1;
	}
	#print "<pre>\n";
	#print "hdr = $hdr\t";
	#print "arg = $arg\n";
	#print "</pre>\n";
	if ($fieldnames_single{$hdr_nogt}) {
	    $hdr_multi = "";
	    $fieldvalues{$hdr_nogt} = $arg;
	    #print "storing, hdr = $hdr_nogt, arg = $arg\n";
	} elsif ($fieldnames_multi{$hdr_nogt}) {
	    $hdr_multi = $hdr_nogt;
	    $fieldvalues{$hdr_nogt} = "";
	    #print "Starting multi-line header $hdr_multi\n";
	} elsif ($hdr_multi ne "") {
	    $fieldvalues{$hdr_multi} .= $_;
	    #print "Appending to multi-line header $hdr_multi: $_\n";
	}
	if ($hdr eq "Reply-To" || $hdr eq "From") {
	    # Grab a few fields out of the envelope as it flies by
	    $arg = &tolower($arg);
	    #  Delete everything inside parenthesis and outside <>'s, inclusive.
	    $arg =~ s/\(.*\)//;
	    $arg =~ s/.*<(.*)>.*/$1/;
	    $arg =~ s/^\s+//;
	    $arg =~ s/\s+$//;
	    print "error: internal whitespace in Reply-to: or From: header!" if ($arg =~ /\s/);
	    $fieldvalues{$hdr} = $arg;
	    #print "storing, hdr = $hdr, arg = $arg\n";
	}
    }

}

# Generate a flat pr file from the %fieldvalues array.
# If argument is "send", don't include fields which are not generated by send-pr.
# Usage: $prtext = &unparse_pr("");
sub unparse_pr
{
    local($send) = $_[0];
    local($prtext, $tmp);
    $prtext = $fieldvalues{"envelope"};
    foreach (@fieldnames_single) {
	next if ($send eq "send" && $fieldnames_single{$_} > 1);
	$tmp = ">$_:";
	$tmp .= " " x (17-length($tmp));
	$prtext .= $tmp . $fieldvalues{$_} . "\n";
    }
    foreach (@fieldnames_multi) {
	next if ($send eq "send" && $fieldnames_multi{$_} > 1);
	$prtext .= ">$_:\n" . $fieldvalues{$_};
    }
    #if (!open(RECONSTRUCTED, "|sort > /tmp/recon")) {
	#print "Unable to create /tmp/recon\n";
    #}
    #print RECONSTRUCTED $prtext;
    #close(RECONSTRUCTED);
    #system("sort $fullpr | diff /tmp/recon - > /tmp/pr.diff");

    return $prtext;
}

#################### Read in GNATS data
# Read in possible responsible parties
sub read_responsible {
    #  nametag : fullname : e-mail address
    #  db:Dave Benson:david_benson@ccmail.adventure.com
    # Note: in our installation, the nametags are also valid local
    # e-mail aliases.
    open(RESPON, "$GNATS_ADM/responsible") ||
	die "Couln't get responsible file\n";
    local($nametag, $fullname, $email);
    while (<RESPON>) {
	if (!/^\s*#|^\s*\n$/) {
	    ($nametag, $fullname, $email) = split(/:/);
	    push(@nResponsible, $nametag);
	    # Make a table which maps from fullname to nametag.
	    # This is used when we want to look up bugs by person
	    # without regards to whether the person is the originator
	    # or the responsible party.
	    # In our system, we use short nametags in the responsible field,
	    # and fullname in the originator field.
	    # So I hope the fullname is entered identically in both
	    # the resposible file and the originator file.
	    $fullname = &tolower($fullname);
	    $nametag = &tolower($nametag);
	    $fullname2nametag{$fullname} = $nametag;
	    # Make a table which maps from nametag to fullname.
	    # This is used when counting up bugs according to who they are waiting on.
	    $nametag2fullname{$nametag} = $fullname;
	}
    }
    close(RESPON);
    @nResponsible = sort(@nResponsible);
    unshift(@nResponsible, "");
}

# Read in possible categories
sub read_category {
    # category tag : category name : responsible : other-interested-people
    # ace-bitmap:Accomplish Bitmap Dwg:sr:dank
    open(CATEG, "$GNATS_ADM/categories") ||
	die "Couln't get categories file\n";
    while (<CATEG>) {
	if (!/^\s*#|^\s*\n$/) {
	    ($x, $dummy) = split(/:/);
	    push(@nCategory, $x);
	}
    }
    close(CATEG);
    @nCategory = sort(@nCategory);
    unshift(@nCategory, "");  # needed for 'any'
}

# Given the number of the pr, find $semipr (category/number) and $fullpr (path to datafile)
sub pr_get_path
{
    local($pr) = $_[0];
    # Read in the path of the specified pr from the GNATS index
        # Next line is taken from script edit-pr
    open(PR, "/usr/bin/grep \"/$pr:\" $GNATS_ADM/index | /usr/bin/awk -F: '{print \$1}' - |") || die "Error while browsing through GNATS index";
    chop($semipr = <PR>);
    $fullpr="$GNATS_ROOT/" . $semipr;
    close(PR);
}

# Lock the given pr.  Must specify category/number.
# Usage: $errmsg = &pr_lock($semipr,$who_is_locking);
# Returns "" on success, error message on failure.
sub pr_lock
{
    local($semipr) = $_[0];
    local($locker) = $_[1];
    # Lock the PR
    # This is actually ineffective since the lock must occur in
    # edit_pr(), but it does check against changes made through edit-pr
    #print "locking pr...\n";
    if (!open(PREDIT, "$PR_EDIT --lock=\'$locker\' $semipr 2>&1 |")) {
	return "Error: can't run $PR_EDIT";
    }
    chop($_ = <PREDIT>);
    close(PREDIT);
    if ($_ ne "") {
	if (!/exists/) {
	    s/.*by //g; tr/_/ /;
	    return "Problem report $semipr is being edited by $_.\n";
	} else {
	    return "GNATS is currently locked, try again in a moment\n";
	}
    }
    #print "... lock obtained\n";
    return "";
}

# Lock and read the PR into @oldpr and %fieldvalues
# Also sets $semipr and $fullpr
# If second arg is a user name, lock pr.
# Usage: $err = &read_pr(pr_number [, "who is locking"]);
sub read_pr {
    local($pr) = $_[0];
    local($locker) = $_[1];
    # Find category and filename for that PR
    # Sets $semipr and $fullpr
    &pr_get_path($pr);
    # Check for the existence of that PR
    if (!$semipr) {
	return "Sorry, could not find PR number $pr.\n";
    }
    # lock pr if desired
    if ($locker ne "") {
	local ($err) = "";
	$err = &pr_lock($semipr, $locker);
	return $err if ($err ne "");
    }
    # read in pr
    if (!open(OLDPR, $fullpr)) {
	return "Can't open pr $fullpr\n";
    }
    @oldpr = <OLDPR>;
    close(OLDPR);
    # make sure old pr is sane
    if (!@oldpr) {
	return "Error: old pr file $fullpr is empty!\n";
    }
    # Get field values
    &parse_pr(@oldpr);

    return "";
}

# Convert a responsible-party nickname to an e-mail address
# Usage: ($adr, $err) = &pr_addr($nick);
sub pr_addr
{
    local($nick) = shift(@_);
    local($adr);
    local($err);

    unlink("/tmp/wwwgnats.$$");
    #print "<pre>executing: $PR_ADDR $nick 2&gt; /tmp/wwwgnats.$$ |</pre>\n";
    if (!open(RESP, "$PR_ADDR $nick 2> /tmp/wwwgnats.$$ |")) {
	$err = "Can't get address of responsible party '$nick'";
	return ("", $err);
    }
    $adr = <RESP>;
    chop($adr);
    #print "Got adr = '$adr'\n";
    close(RESP);
    if ($?) {
	$err = "Error: pr-addr returns status $?, and reports:\n";
	if (!open(PRADDR, "/tmp/wwwgnats.$$")) {
	    $err .= "(whoops, no output from pr-addr found; couldn't open /tmp/wwwgnats.$$)\n";
	} else {
	    $err .= join("\n",<PRADDR>);
	    close(PRADDR);
	}
	unlink("/tmp/wwwgnats.$$");
	return ("", $err);
    }
    ($adr, $err);
}
