/* Miscellaneous index routines for use with GNATS. 
   Copyright (C) 1993, 1995 Free Software Foundation, Inc.
   Contributed by Brendan Kehoe (brendan@cygnus.com) and
   by Tim Wicinski (wicinski@barn.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111, USA.  */

#include "config.h"
#include "gnats.h"
#include "gnatsd.h"

/* Where to keep the static index if necessary.  */
Index *index_chain;

/* The last time the index was modified.  */
time_t index_mtime;

/* The filename of the index.  */
char *index_filename = (char *)NULL;

void
create_index_entry (buf)
     char *buf;
{
  char *resp, *r;
#ifdef GNATS_RELEASE_BASED
  char *tim;
  time_t t;
#endif

  resp = field_value (RESPONSIBLE);
  if (! *resp)
    return;

#ifdef GNATS_RELEASE_BASED
  tim = field_value (DATE_REQUIRED);
  if (tim)
    {
      t = get_date (tim, NULL);
      if (! t)
	return;
    }
  else
    t = 0;
#endif

  resp = (char *) strdup (resp);

  r = (char *) strchr (resp, ' ');
  if (r != NULL)
    *r = '\0';

#ifdef GNATS_RELEASE_BASED  	
  sprintf (buf, "%s/%s:%s:%s:%s:%s:%s:%s:%d\n", field_value (CATEGORY),
	   field_value (NUMBER), field_value (SUBMITTER),
	   resp, field_value (STATE), field_value (CONFIDENTIAL), 
	   field_value (SEVERITY), field_value (PRIORITY), t);
#else
  sprintf (buf, "%s/%s:%s:%s:%s:%s:%s:%s\n", field_value (CATEGORY),
	   field_value (NUMBER), field_value (SUBMITTER),
	   resp, field_value (STATE), field_value (CONFIDENTIAL), 
	   field_value (SEVERITY), field_value (PRIORITY));
#endif

  xfree (resp);
}


/* Return the next index entry from FP.  */
Index *
next_index_entry (fp)
     FILE *fp;
{
  char *start, *end, *b;
  char *buf = (char *) xmalloc (STR_MAX);
  Index *p = (Index *) xmalloc (sizeof (Index));

  b = buf;
  do
    if (fgets (buf, STR_MAX, fp) == NULL)
      goto no_entry;
  while (buf[0] == '#');

  start = b;
  end = (char *) strchr (start, '/');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->category = start;

  start = end + 1;
  end = (char *) strchr (start, ':');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->number = start;

  start = end + 1;
  end = (char *) strchr (start, ':');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->submitter = start;

  start = end + 1;
  end = (char *) strchr (start, ':');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->responsible = start;

  start = end + 1;
  end = (char *) strchr (start, ':');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->state = start;

  start = end + 1;
  end = (char *) strchr (start, ':');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->confidential = start;

  start = end + 1;
  end = (char *) strchr (start, ':');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->severity = start;

  start = end + 1;
#ifdef GNATS_RELEASE_BASED
  end = (char *) strchr (start, ':');
#else
  end = (char *) strchr (start, '\n');
#endif
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->priority = start;

#ifdef GNATS_RELEASE_BASED
  start = end + 1;
  end = (char *) strchr (start, '\n');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->date_required = start;
#endif

  p->next = NULL;
  return p;

no_entry:
  xfree ((char *) p);
  xfree (buf);
  return (Index *) NULL;
}

/* Find problem report NUMBER in the index file, returning its category.  */
char *
find_pr_category (fp, number)
     FILE *fp;
     char *number;
{
  char *start, *end, *b;
  static char buf[STR_MAX];
  static char *category;

  b = &buf[0];
  category = (char *)NULL;

  while (1)
    {
      do
	if (fgets (&buf[0], STR_MAX, fp) == NULL)
	  return NULL;
      while (buf[0] == '#');

      start = b;
      end = (char *) strchr (start, '/');
      if (end == NULL)
	return NULL;
      *end = '\0';
      category = start;

      start = end + 1;
      end = (char *) strchr (start, ':');
      if (end == NULL)
	return NULL;
      *end = '\0';

      if (strcmp (start, number) == 0)
	return category;
    }

  return NULL;
}

/* get_index - reads in the entire index. useful for searching or
   modifying index records.  */
Index*
get_index ()
{
  FILE *fp = open_index ();
  Index *i;
  Index *index_chain = NULL;
  Index *index_chain_end = NULL;

  if (fp == (FILE *) NULL)
    return (Index *) NULL;

  while ((i = next_index_entry (fp)))
    {
      if (index_chain_end == NULL)
        index_chain = index_chain_end = i;
      else
        {
          index_chain_end->next = i;
          index_chain_end = i;
        }

      i->next = NULL;
    }

  close_index (fp);
  return (index_chain);
}

time_t
stat_index (name)
     char *name;
{
  struct stat buf;
  char *path;
  int i;

  if (! name)
    path = index_filename;
  else
    path = name;

  i = stat (path, &buf);
  if (i < 0)
    {
      if (! is_daemon)
	fprintf (stderr, "%s: can't stat the index file %s\n",
		 program_name, path);
      return (time_t)-1;
    }

  return buf.st_mtime;
}

int
check_index ()
{
  struct stat buf;
  int i;

  i = stat (index_filename, &buf);
  /* We should probably send something here.  */
  if (i < 0)
    return 0;

  if (buf.st_mtime > index_mtime)
    {
      index_chain = get_index ();
      if (index_chain == NULL)
	{
	  printf ("%d GNATS server cannot read the index.\r\n", CODE_FILE_ERROR);
	  exit (1);
	}
      return 1;
    }

  return 0;
}

FILE *
open_index ()
{
  FILE *fp;
  time_t t;

  if (! index_filename)
    {
      index_filename = (char *) xmalloc (PATH_MAX);
      sprintf (index_filename, "%s/gnats-adm/%s", gnats_root, INDEX);
    }

  fp = fopen (index_filename, "r");
  if (fp == NULL)
    {
      if (! is_daemon)
	fprintf (stderr, "%s: can't open the index file %s\n",
		 program_name, index_filename);
      xfree (index_filename);
      return (FILE *) NULL;
    }

  t = stat_index (index_filename);
  if (t != (time_t)-1)
    index_mtime = t;
  return fp;
}

void
close_index (fp)
     FILE *fp;
{
  fclose (fp);
}
