/* Entry for the GNATS system.
   Copyright (C) 1993, 1994, 1995 Free Software Foundation, Inc.
   Originally written by Tim Wicinski (wicinski@barn.com).
   Further work by Brendan Kehoe (brendan@cygnus.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111, USA.  */

#include "config.h"
#include "gnats.h"
#include "query.h"

static void version		PARAMS((void));
static void usage		PARAMS((void));

/* Name of this program.  */
char *program_name;

/* The name of the file(s) to write to.  */
char *outname = (char *)NULL;

/* Whether we should ignore the index and drop the PR on stdin into the
   database as a new one.  */
int force = 0;

/* If 1, we're running the daemon.  */
int is_daemon = 0;

/* The name of the GNATS server.  */
char *name;

/* If 1, send information as we make the query.  */
int debug = 0;

/* The descriptor for talking to the server in non-stdio ways.  */
extern int sockfd;

enum {
  NONE,
  LOCK,
  UNLOCK,
  LOCKDB,
  UNLOCKDB,
  CHECK,
  MODIFY
} edit_options;

struct option long_options[] =
{
  {"lock", 1, NULL, 'l'},
  {"unlock", 0, NULL, 'u'},
  {"lockdb", 0, NULL, 'L'},
  {"unlockdb", 0, NULL, 'U'},
  {"check", 0, NULL, 'c'},
  {"process", 1, NULL, 'p'},
  {"get-lists", 0, NULL, 'g'},
  {"directory", 1, NULL, 'd'},
  {"filename", 1, NULL, 'f'},
  {"outfile", 1, NULL, 'o'},
  {"host", 1, NULL, 'H'},
  {"debug", 0, NULL, 'D'},
  {"version", 0, NULL, 'V'},
  {"help", 0, NULL, 'h'},
  {NULL, 0, NULL, 0}
};

int
receive_list (cmd, suff)
     char *cmd, *suff;
{
  char *s = (char *) xmalloc (strlen (outname) + 5);
  sprintf (s, "%s%s", outname, suff);
  outfp = fopen (s, "w");
  if (outfp == (FILE *)NULL)
    {
      fprintf (stderr, "%s: error writing to %s: %s\n",
	       program_name, s, strerror (errno));
      return 1;
    }
  if (debug)
    fprintf (stderr, "%s: writing `%s'\n", program_name, cmd);
  fprintf (serv_write, "%s\r\n", cmd);
  get_reply ();
  fclose (outfp);
  xfree (s);
  return 0;
}

int
net_check_pr (fp)
     FILE *fp;
{
  char *p;

  init_gnats ();

  /* Read the message header in.  */
  if (read_header (fp) < 0)
    {
      fprintf (stderr, "%s: end-of-file reading PR\n", program_name);
      return 0;
    }
  read_pr (fp, 0);

  if (debug)
    fprintf (stderr, "%s: writing `CATG %s'\n", program_name,
	     pr[CATEGORY].value);
  fprintf (serv_write, "CATG %s\r\n", pr[CATEGORY].value);
  get_reply ();
  if (debug)
    fprintf (stderr, "%s: writing `SUBM %s'\n", program_name,
	     pr[SUBMITTER].value);
  fprintf (serv_write, "SUBM %s\r\n", pr[SUBMITTER].value);
  get_reply ();
  p = (char *) strchr (pr[RESPONSIBLE].value, ' ');
  if (p != NULL)
    *p = '\0';
  if (debug)
    fprintf (stderr, "%s: writing `RESP %s'\n", program_name,
	     pr[RESPONSIBLE].value);
  fprintf (serv_write, "RESP %s\r\n", pr[RESPONSIBLE].value);
  get_reply ();

#ifdef GNATS_RELEASE_BASED
  if (pr[DATE_REQUIRED].value && pr[DATE_REQUIRED].value[0])
    {
      if (debug)
	fprintf (stderr, "%s: writing `VDAT %s'\n", program_name,
		 pr[DATE_REQUIRED].value);
      fprintf (serv_write, "VDAT %s\r\n", pr[DATE_REQUIRED].value);
      get_reply ();
    }
#endif

#if 0
  enums = check_enum_types (2); /* XXX */
  p = field_value (NUMBER);
  if (enums || *p == '-')
    {
      printf ("%s: invalid fields", program_name);
      if (enums)
	for (e = enums; e; e = e->next)
	  printf ("%s", e->msg);
      /* It was missing its number! */
      if (*p == '-')
	printf ("Number: %s\n", p);
      return 0;
    }
#endif
  return 1;
}

int
net_write (in, out)
     FILE *in, *out;
{
  int i, j;
  char buf[1024 * 8];

  do {
    i = fread (buf, 1, sizeof (buf), in);
    if (i == 0)
      break;
    j = fwrite (buf, 1, i, out);
    if (j == 0)
      break;
  } while (i);

  if (ferror (in))
    ;
  if (ferror (out))
    ;
  return 1;
}

int
net_modify_pr (fp)
     FILE *fp;
{
  char *p;
  if (fp == (FILE *)NULL)
    return 0;

  if (! net_check_pr (fp))
    return 0;

  p = field_value (NUMBER);
  if (debug)
    fprintf (stderr, "%s: writing `EDIT %s'\n", program_name, p);
  fprintf (serv_write, "EDIT %s\r\n", p);
  get_reply (); /* XXX */
  write_header (serv_write, NUM_HEADER_ITEMS);
  fprintf (serv_write, "\r\n");
  write_pr (serv_write, NUM_PR_ITEMS);
  fprintf (serv_write, ".\r\n");
  get_reply ();

  return 1;
}

int
main (argc, argv)
     int argc;
     char **argv;
{

  FILE *fp = stdin;
  int option;
  char *fname = NULL;
  char *username;
  int result = 1, get_lists = 0;
  char *processid = (char *)NULL;

  program_name = basename (argv[0]);
  name = gnats_server;
  edit_options = MODIFY;

  while ((option = getopt_long (argc, argv, "l:uLUcgd:f:o:p:H:DVh",
				long_options, (int *)0)) != EOF)
    {
      switch (option)
	{
	case 'd':
	  gnats_root = optarg;
	  break;

	case 'f':
	  fp = fopen (optarg, "r");
	  /* If they gave a bogus argument, then exit right away; we don't
	     want to send mail to gnats-admin every time someone tries to
	     edit a bogus PR number.  */
	  if (fp == (FILE *)NULL)
	    exit (2);
	  break;

	case 'o':
	  outname = optarg;
	  outfp = fopen (optarg, "w");
	  /* If they gave a bogus argument, then exit right away; we don't
	     want to send mail to gnats-admin every time someone tries to
	     edit a bogus PR number.  */
	  if (outfp == (FILE *)NULL)
	    exit (2);
	  break;

	case 'g':
	  get_lists = 1;
	  break;

	case 'F':
	  force = 1;
	  break;

	case 'p':
	  processid = optarg;
	  break;

	case 'l': 
	  edit_options = LOCK;
	  username = optarg;
	  break;

	case 'u':
	  edit_options = UNLOCK;
	  break;

	case 'L':
	  edit_options = LOCKDB;
	  break;

	case 'U':
	  edit_options = UNLOCKDB;
	  break;

	case 'c':
	  edit_options = CHECK;
	  break; 

	case 'H':
	  name = optarg;
	  break;

	case 'D':
	  debug = 1;
	  break;

	case 'V':
	  version ();
	  exit (0);
	  break;
	  
	case 'h':
	  usage ();
	  exit (0);
	  break;
	  
	default:
	  usage ();
	  exit (3);
	  break;
	}
    }

  if (get_lists && !outname)
    {
      fprintf (stderr, "%s: must use -o with -g\n", program_name);
      exit (1);
    }

  /* if there is another arg, then we take it to be a file name. */
  if (optind < argc)
    fname = argv[optind];

  if (! fname && edit_options == LOCK)
    {
      fprintf (stderr, "%s: must give a PR to lock\n", program_name);
      exit (1);
    }

  client_init ();

  switch (edit_options) {
    case LOCK:
      if (debug)
	fprintf (stderr, "%s: writing `LOCK %s %s\n", program_name,
		 fname, username);
      if (processid)
	fprintf (serv_write, "LOCK %s %s %s\r\n",
		 fname, username, processid);
      else
	fprintf (serv_write, "LOCK %s %s\r\n", fname, username);
      get_reply ();
      break;
    case UNLOCK:
      if (debug)
	fprintf (stderr, "%s: writing `UNLK %s'\n", program_name, fname);
      fprintf (serv_write, "UNLK %s\r\n", fname);
      get_reply ();
      break;
    case CHECK:
      result = net_check_pr (fp);
      break;
    case LOCKDB:
    case UNLOCKDB:
      break;
    default:
      result = net_modify_pr (fp);
      break;
  }

  if (outname)
    {
      fclose (outfp);
      if (receive_list ("LCAT", ".cat"))
	goto done;
      if (receive_list ("LSUB", ".sub"))
	goto done;
      receive_list ("LRES", ".res");
      outfp = (FILE *)NULL;
    }

done:
  fclose (fp);
  if (outfp)
    fclose (outfp);
  unblock_signals ();
  client_exit ();
  exit (!result);
}

static void
usage ()
{
  fprintf (stderr, "\
Usage: %s [-cuLUVFh] [-d directory] [-f filename] [-l username] [-o filename]\n\
          [--directory=directory] [--filename=filename]\n\
          [--lock=user] [--process=pid] [--unlock] [--lockdb] [--unlockdb]\n\
	  [--host=server] [--outfile=filename] [--help] [--version]\n\
          [--check] [PR]\n", program_name);
}

static void
version ()
{
  printf ("%s %s\n", program_name, version_string);
}
