/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB

   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.

   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.	The License grants you the right to
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/* Basic functions neaded by many modules */

#include "mysql_priv.h"
#include <m_ctype.h>
#include <my_dir.h>
#include <hash.h>
#include <nisam.h>
#ifdef	__WIN32__
#include <io.h>
#endif

#define MAX_DBKEY_LENGTH (FN_LEN*2+2)

static int key_cache_used=0;
TABLE *unused_tables;				/* Used by mysql_test */
HASH open_cache;				/* Used by mysql_test */


static int open_unireg_entry(TABLE *entry,const char *db,const char *name,
			     const char *alias);
static bool insert_fields(THD *thd,TABLE_LIST *tables, char *table_name,
			  List_iterator<Item> *it);
static void free_cache_entry(TABLE *entry);
static void mysql_rm_tmp_tables(void);

static byte *cache_key(const byte *record,uint *length)
{
  TABLE *entry=(TABLE*) record;
  *length=entry->key_length;
  return (byte*) entry->table_cache_key;
}

void table_cache_init(void)
{
  VOID(init_hash(&open_cache,table_cache_size,0,0,cache_key,
		 (void (*)(void*)) free_cache_entry));
  mysql_rm_tmp_tables();
}


void table_cache_free(void)
{
  DBUG_ENTER("table_cache_free");
  close_cached_tables();
  if (!open_cache.records)			// Safety first
    free_hash(&open_cache);
  DBUG_VOID_RETURN;
}


uint cached_tables(void)
{
  return open_cache.records;
}

#ifdef EXTRA_DEBUG
static void check_unused(void)
{
  uint count=0,index=0;
  TABLE *link,*start_link;

  if ((start_link=link=unused_tables))
  {
    do
    {
      if (link != link->next->prev || link != link->prev->next)
      {
	DBUG_PRINT("error",("Unused_links aren't linked properly")); /* purecov: inspected */
	return; /* purecov: inspected */
      }
    } while (count++ < open_cache.records && (link=link->next) != start_link);
    if (link != start_link)
    {
      DBUG_PRINT("error",("Unused_links aren't connected")); /* purecov: inspected */
    }
  }
  for (index=0 ; index < open_cache.records ; index++)
  {
    TABLE *entry=(TABLE*) hash_element(&open_cache,index);
    if (!entry->in_use)
      count--;
  }
  if (count != 0)
  {
    DBUG_PRINT("error",("Unused_links dosen't match open_cache: diff: %d", /* purecov: inspected */
			count)); /* purecov: inspected */
  }
}
#else
#define check_unused()
#endif


/******************************************************************************
** Send name and type of result to client.
** Sum fields has table name empty and field_name.
******************************************************************************/

bool
send_fields(THD *thd,List<Item> &list,uint flag)
{
  List_iterator<Item> it(list);
  Item *item;
  byte buff[80];
  String tmp((char*) buff,sizeof(buff)),*res,*packet= &thd->packet;

  if (flag & 1)
  {				// Packet with number of elements
    byte *pos=net_store_length(buff,list.elements);
    (void) my_net_write(&thd->net, buff,(uint) (pos-buff));
  }
  while ((item=it++))
  {
    char *pos;
    Send_field field;
    item->make_field(&field);
    packet->length(0);

    if (net_store_data(packet,field.table_name) ||
	net_store_data(packet,field.col_name) ||
	packet->realloc(packet->length()+10))
      goto err; /* purecov: inspected */
    pos= (char*) packet->ptr()+packet->length();

    if (!(thd->client_capabilities & CLIENT_LONG_FLAG))
    {
      packet->length(packet->length()+9);
      pos[0]=3; int3store(pos+1,field.length);
      pos[4]=1; pos[5]=field.type;
      pos[6]=2; pos[7]=(char) field.flags; pos[8]= (char) field.decimals;
    }
    else
    {
      packet->length(packet->length()+10);
      pos[0]=3; int3store(pos+1,field.length);
      pos[4]=1; pos[5]=field.type;
      pos[6]=3; int2store(pos+7,field.flags); pos[9]= (char) field.decimals;
    }
    if (flag & 2)
    {						// Send default value
      if (!(res=item->str(&tmp)))
      {
	if (net_store_null(packet))
	  goto err;
      }
      else if (net_store_data(packet,(byte*) res->ptr(),res->length()))
	goto err;
    }
    if (my_net_write(&thd->net,(byte*) packet->ptr(),packet->length()))
      break;					/* purecov: inspected */
  }
  send_eof(&thd->net,(test_flags & TEST_MIT_THREAD) ? 0: 1);
  return 0;
 err:
  send_error(&thd->net,ER_OUTOFMEMORY);		/* purecov: inspected */
  return 1;					/* purecov: inspected */
}


/*****************************************************************************
 *	 Functions to free open table cache
 ****************************************************************************/


void intern_close_table(TABLE *table)
{						// Free all structures
  free_io_cache(table);
  if (table->file)
    VOID(closefrm(table));			// close file
  x_free(table->table_cache_key);
}


static void free_cache_entry(TABLE *table)
{
  DBUG_ENTER("free_cache_entry");

  intern_close_table(table);
  if (!table->in_use)
  {
    table->next->prev=table->prev;		/* remove from used chain */
    table->prev->next=table->next;
    if (table == unused_tables)
    {
      unused_tables=unused_tables->next;
      if (table == unused_tables)
	unused_tables=0;
    }
    check_unused();				// consisty check
  }
  my_free((gptr) table,MYF(0));
  DBUG_VOID_RETURN;
}


void free_io_cache(TABLE *table)
{
  if (table->io_cache)
  {
    close_cacheed_file(table->io_cache);
    my_free((gptr) table->io_cache,MYF(0));
    table->io_cache=0;
  }
  if (table->record_pointers)
  {
    my_free((gptr) table->record_pointers,MYF(0));
    table->record_pointers=0;
  }
}

	/* Close all tables with arn't in use by any thread */

void close_cached_tables(void)
{
  VOID(pthread_mutex_lock(&LOCK_open));
  while (unused_tables)
    VOID(hash_delete(&open_cache,(byte*) unused_tables));
  if (!open_cache.records)
  {
    end_key_cache();				/* No tables in memory */
    key_cache_used=0;
  }
  VOID(pthread_mutex_unlock(&LOCK_open));
}


/* Put all tables used by thread in free list */

void close_thread_tables(THD *thd)
{
  TABLE *table,*next;
  DBUG_ENTER("close_thread_tables");

  if (thd->locked_tables)
    DBUG_VOID_RETURN;				// LOCK TABLES in use
  if (thd->lock)
  {
    mysql_unlock_tables(thd->lock);
    thd->lock=0;
  }
  /* VOID(pthread_sigmask(SIG_SETMASK,&thd->block_signals,NULL)); */
  VOID(pthread_mutex_lock(&LOCK_open));

  for (table=thd->open_tables ; table ; table=next)
  {
    next=table->next;
    if (thd->version != reload_version)		/* loaddb or rm has happend */
      VOID(hash_delete(&open_cache,(byte*) table));
    else
    {
      table->in_use=0;
      if (unused_tables)
      {
	table->next=unused_tables;		/* Link in last */
	table->prev=unused_tables->prev;
	unused_tables->prev=table;
	table->prev->next=table;
      }
      else
	unused_tables=table->next=table->prev=table;
    }
  }
  thd->open_tables=0;
  /* Free tables to hold down open files */
  while (open_cache.records >= table_cache_size && unused_tables)
    VOID(hash_delete(&open_cache,(byte*) unused_tables)); /* purecov: tested */
  check_unused();

  VOID(pthread_mutex_unlock(&LOCK_open));
  /*  VOID(pthread_sigmask(SIG_SETMASK,&thd->signals,NULL)); */
  DBUG_VOID_RETURN;
}

	/* move table first in unused links */

static void relink_unused(TABLE *table)
{
  if (table != unused_tables)
  {
    table->prev->next=table->next;		/* Remove from unused list */
    table->next->prev=table->prev;
    table->next=unused_tables;			/* Link in unused tables */
    table->prev=unused_tables->prev;
    unused_tables->prev->next=table;
    unused_tables->prev=table;
    unused_tables=table;
    check_unused();
  }
}


/* Remove one open table from the open list */

TABLE *unlink_open_table(TABLE *list,TABLE *find)
{
  if (find==list)
    return find->next;
  TABLE *start=list;
  for (; list->next ; list=list->next)
  {
    if (list->next == find)
    {
      list->next=find->next;
      break;
    }
  }
  return start;
}


/******************************************************************************
** open a table
** Uses a cache of open tables to find a table not in use.
******************************************************************************/


TABLE *open_table(THD *thd,const char *db,const char *table_name,
		  const char *alias)
{
  reg1	TABLE *table;
  char	key[MAX_DBKEY_LENGTH];
  uint	key_length;
  DBUG_ENTER("open_table");

  /* find a unused table in the open table cache */

  key_length=(uint) (strmov(strmov(key,db)+1,table_name)-key)+1;
  if (thd->locked_tables)
  {						// Using table lock
    for (table=thd->open_tables; table ; table=table->next)
    {
      if (table->key_length == key_length &&
	  !memcmp(table->table_cache_key,key,key_length) &&
	  !my_strcasecmp(table->table_name,alias))
	goto reset;
    }
    my_printf_error(ER_TABLE_NOT_LOCKED,ER(ER_TABLE_NOT_LOCKED),MYF(0),alias);
    DBUG_RETURN(0);
  }


  VOID(pthread_mutex_lock(&LOCK_open));

  for (table=(TABLE*) hash_search(&open_cache,(byte*) key,key_length) ;
       table && table->in_use ;
       table = (TABLE*) hash_next(&open_cache,(byte*) key,key_length)) ;
  if (table)
  {
    if (table == unused_tables)
    {						// First unused
      unused_tables=unused_tables->next;	// Remove from link
      if (table == unused_tables)
	unused_tables=0;
    }
    table->prev->next=table->next;		/* Remove from unused list */
    table->next->prev=table->prev;
    if (strcmp(table->table_name,alias))
    {
      uint i;
      my_free((gptr) table->table_name,MYF(0));
      table->table_name=my_strdup(alias,MYF(MY_WME));
      for (i=0 ; i < table->fields ; i++)
	table->field[i]->table_name=table->table_name;
    }
  }
  else
  {
    /* Free cache if too big */
    while (open_cache.records >= table_cache_size && unused_tables)
      VOID(hash_delete(&open_cache,(byte*) unused_tables)); /* purecov: tested */

    VOID(pthread_mutex_unlock(&LOCK_open));	/* Don't wait for slow open */

    /* make a new table */
    if (!(table=(TABLE*) my_malloc(sizeof(*table),MYF(MY_WME))))
      DBUG_RETURN(NULL);
    if (open_unireg_entry(table,db,table_name,alias))
    {
      table->next=table->prev=table;
      free_cache_entry(table);
      DBUG_RETURN(NULL);
    }
    table->table_cache_key=my_memdup((byte*) key,key_length,MYF(MY_FAE));
    table->key_length=key_length;
    VOID(pthread_mutex_lock(&LOCK_open));
    if (!key_cache_used)
    {
      key_cache_used=1;
      ha_key_cache();
    }
    VOID(hash_insert(&open_cache,(byte*) table));
  }

  table->in_use=thd;
  check_unused();
  VOID(pthread_mutex_unlock(&LOCK_open));
  table->next=thd->open_tables;		/* Link into simple list */
  thd->open_tables=table;
  table->reginfo.update=0;		/* Assume read */

 reset:
  table->version=thd->version;
  table->tmp_table = 0;
  table->tablenr=thd->current_tablenr++;
  table->used_fields=0;
  table->reginfo.ref_fields=0;		/* Some precation */
  table->const_table=0;
  table->status=STATUS_NO_RECORD;
  table->outer_join=table->null_row=table->maybe_null=0;
  DBUG_RETURN(table);
}

/****************************************************************************
** Reopen an already open table because the definition has changed
** Returns 0 if ok.
** If table can't be reopened, the entry is unchanged.
****************************************************************************/

bool reopen_table(THD *thd,TABLE *table,bool locked)
{
  TABLE tmp;
  char *db=table->table_cache_key;
  char *table_name=table->real_name;
  DBUG_ENTER("reopen_table");

  if (open_unireg_entry(&tmp,db,table_name,table->table_name))
    DBUG_RETURN(1);
  if (!locked)
    VOID(pthread_mutex_lock(&LOCK_open));

  free_io_cache(table);

  tmp.table_cache_key=db;
  tmp.key_length=table->key_length;
  tmp.reginfo.update=table->reginfo.update;
  tmp.in_use=table->in_use;
  tmp.next=table->next;
  tmp.prev=table->prev;
  tmp.version=table->version;
  tmp.tmp_table=table->tmp_table;
  tmp.tablenr=table->tablenr;
  tmp.used_fields=table->used_fields;

  if (table->file)
    VOID(closefrm(table));		// close file, free everything

  memcpy((char*) table,(char*) &tmp,sizeof(tmp));

  for (Field **field=table->field ; *field ; field++)
  {
    (*field)->table=table;
    (*field)->table_name=table->table_name;
  }
  for (uint key=0 ; key < table->keys ; key++)
    for (uint part=0 ; part < table->key_info[key].usable_key_parts ; part++)
      table->key_info[key].key_part[part].field->table=table;

  if (!locked)
    VOID(pthread_mutex_unlock(&LOCK_open));
  DBUG_RETURN(0);
}


/* Reopen tables when LOCK TABLES is in used */

bool reopen_tables(THD *thd,const char *table_name)
{
  TABLE *table;
  bool error=0;
  for (table=thd->open_tables; table ; table=table->next)
  {
    if (!strcmp(table->real_name,table_name))
    {
      mysql_lock_remove(thd->locked_tables,table);
      MYSQL_LOCK *lock=0;
      if (reopen_table(thd,table,1) ||
	  !(lock=mysql_lock_tables(thd,&table,1)))
	error=1;
      else
	thd->locked_tables=mysql_lock_merge(thd->locked_tables,lock);
    }
  }
  return error;
}


/****************************************************************************
	open_unireg_entry
**	Purpose : Load a table definition from file and open unireg table
**	Args	: entry with DB and table given
**	Returns : 0 if ok
*/

static int open_unireg_entry(TABLE *entry,const char *db,const char *name,
			     const char *alias)
{
  char path[FN_REFLEN];
  DBUG_ENTER("open_unireg_entry");

  (void)sprintf(path,"%s/%s/%s",mysql_data_home,db,name);
  if (openfrm(path,alias,
	      (uint) (HA_OPEN_KEYFILE | HA_OPEN_RNDFILE | HA_GET_INDEX |
		      HA_TRY_READ_ONLY),
	      READ_KEYINFO | COMPUTE_TYPES | EXTRA_RECORD,
	      entry))
  {
    DBUG_RETURN(1);
  }
  VOID(ha_extra(entry,HA_EXTRA_NO_READCHECK));		// Not needed in SQL
  DBUG_RETURN(0);
}


/*****************************************************************************
** open all tables in list
*****************************************************************************/

int open_tables(THD *thd,TABLE_LIST *tables)
{
  DBUG_ENTER("open_tables");
  for ( ; tables ; tables=tables->next)
  {
    if (!(tables->table=open_table(thd,thd->db,tables->real_name,
				   tables->name)))
      DBUG_RETURN(-1);
    if (tables->flags)
      tables->table->reginfo.update=1;		// Will be updated
  }
  DBUG_RETURN(0);
}


TABLE *open_ltable(THD *thd, TABLE_LIST *table_list, bool update)
{
  TABLE *table;
  if ((table=open_table(thd,table_list->db ? table_list->db : thd->db,
			table_list->real_name,table_list->name)))
  {
    table_list->table=table;
    if (thd->locked_tables)
    {
      if (update && ! table->reginfo.update)
      {
	my_printf_error(ER_TABLE_NOT_LOCKED_FOR_WRITE,
			ER(ER_TABLE_NOT_LOCKED_FOR_WRITE),
			MYF(0),table_list->name);
	return 0;
      }
      return table;
    }
    table->reginfo.update=update;
    if (!(thd->lock=mysql_lock_tables(thd,&table_list->table,1)))
      return 0;
  }
  return table;
}

/*
** Open all tables in list and locks them for read.
** The lock will automaticly be freed by the close_thread_tables
*/

int open_and_lock_tables(THD *thd,TABLE_LIST *tables)
{
  if (open_tables(thd,tables) || lock_tables(thd,tables))
    return -1;					/* purecov: inspected */
  return 0;
}

int lock_tables(THD *thd,TABLE_LIST *tables)
{
  if (tables && !thd->locked_tables)
  {
    uint count=0;
    TABLE_LIST *table;
    for (table = tables ; table ; table=table->next)
      count++;
    TABLE **start,**ptr;
    ptr=start=(TABLE**) sql_alloc(sizeof(TABLE*)*count);
    for (table = tables ; table ; table=table->next)
      *(ptr++)= table->table;
    if (!(thd->lock=mysql_lock_tables(thd,start,count)))
      return -1;				/* purecov: inspected */
  }
  return 0;
}

/*
** Open a single table without table caching and don't set it in open_list
** Used by alter_table to open a temporary table
*/

TABLE *open_tmp_table(THD *thd,const char *db,const char *table_name)
{
  TABLE *tmp_table;
  DBUG_ENTER("open_tmp_table");
  if (!(tmp_table=(TABLE*) my_malloc(sizeof(*tmp_table),MYF(MY_WME))))
    DBUG_RETURN(0);				/* purecov: inspected */
  if (open_unireg_entry(tmp_table,db,table_name,table_name))
  {
    my_free((gptr) tmp_table,MYF(0));		/* purecov: inspected */
    DBUG_RETURN(0);				/* purecov: inspected */
  }
  tmp_table->reginfo.update=1;			// Should be write
  tmp_table->tmp_table = 1;			// For debugging
  tmp_table->table_cache_key=0;
  DBUG_RETURN(tmp_table);
}


/*****************************************************************************
** find field in list or tables. if field is unqualifed and unique,
** return unique field
******************************************************************************/

static Field *
find_field_in_table(THD *thd,TABLE *table,Item_field *item)
{
  char *name=item->field_name;
  Field **ptr=table->field,*field;
  while ((field = *ptr++))
  {
    if (!my_strcasecmp(field->field_name, name))
    {
      if (thd->set_query_id)
      {
	if (field->query_id != thd->query_id)
	{
	  field->query_id=thd->query_id;
	  field->table->used_fields++;
	}
	else
	  thd->dupp_field=field;
      }
      return field;
    }
  }
  my_printf_error(ER_BAD_FIELD_ERROR,ER(ER_BAD_FIELD_ERROR),MYF(0),item->name,thd->where);
  return (Field*) 0;
}


Field *
find_field_in_tables(THD *thd,Item_field *item,TABLE_LIST *tables)
{
  Field *found=0;
  char *table_name=item->table_name;
  char *name=item->field_name;

  if (table_name)
  {						/* Qualified field */
    for (; tables && strcmp(tables->name,table_name) ; tables=tables->next) ;
    if (!tables)
    {
      my_printf_error(ER_UNKNOWN_TABLE,ER(ER_UNKNOWN_TABLE),MYF(0),table_name,
		      thd->where);
      return (Field*) 0;
    }
    return find_field_in_table(thd,tables->table,item);
  }

  for (; tables ; tables=tables->next)
  {
    Field **ptr=tables->table->field,*field;
    while ((field = *ptr++))
    {
      if (!my_strcasecmp(field->field_name, name))
      {
	if (found)
	{
	  if (!thd->where)
	    break;
	  my_printf_error(ER_NON_UNIQ_ERROR,ER(ER_UNKNOWN_TABLE),MYF(0),name,thd->where);
	  return (Field*) 0;
	}
	found=field;
	break;					// check next table
      }
    }
  }
  if (found)
  {
    if (thd->set_query_id)
    {
      if (found->query_id != thd->query_id)
      {
	found->query_id=thd->query_id;
	found->table->used_fields++;
      }
      else
	thd->dupp_field=found;
    }
    return found;
  }
  my_printf_error(ER_BAD_FIELD_ERROR,ER(ER_BAD_FIELD_ERROR),
		  MYF(0),item->full_name(),thd->where);
  return (Field*) 0;
}

Item **
find_item_in_list(Item *find,List<Item> &items)
{
  List_iterator<Item> li(items);
  Item **found=0,*item;
  char *field_name=0,*table_name=0;
  if (find->type() == Item::FIELD_ITEM	|| find->type() == Item::REF_ITEM)
  {
    field_name= ((Item_ident*) find)->field_name;
    table_name= ((Item_ident*) find)->table_name;
  }

  while ((item=li++))
  {
    if (field_name && item->type() == Item::FIELD_ITEM)
    {
      if (!my_strcasecmp(((Item_field*) item)->name,field_name))
      {
	if (!table_name)
	{
	  if (found)
	  {
	    if ((*found)->eq(item))
	      continue;				// Same field twice (Access?)
	    if (current_thd->where)
	      my_printf_error(ER_NON_UNIQ_ERROR,ER(ER_NON_UNIQ_ERROR),MYF(0),
			      find->full_name(), current_thd->where);
	    return (Item**) 0;
	  }
	  found=li.ref();
	}
	else if (!strcmp(((Item_field*) item)->table_name,table_name))
	{
	  found=li.ref();
	  break;
	}
      }
    }
    else if (!my_strcasecmp(item->name,find->name))
    {
      found=li.ref();
      break;
    }
  }
  if (!found && current_thd->where)
    my_printf_error(ER_BAD_FIELD_ERROR,ER(ER_BAD_FIELD_ERROR),MYF(0),
		    find->full_name(),current_thd->where);
  return found;
}


/****************************************************************************
**	Check that all given fields exists and fill struct with current data
****************************************************************************/

int setup_fields(THD *thd, TABLE_LIST *tables, List<Item> &fields,
		 bool set_query_id, List<Item> *sum_func_list)
{
  reg2 Item *item;
  List_iterator<Item> it(fields);
  DBUG_ENTER("setup_fields");

  thd->set_query_id=set_query_id;
  thd->allow_sum_func= test(sum_func_list);
  thd->where="field list";

  /* Remap table numbers if INSERT ... SELECT */
  uint tablenr=0;
  for (TABLE_LIST *table=tables ; table ; table=table->next,tablenr++)
  {
    table->table->tablenr=tablenr;
    table->table->map= (table_map) 1 << tablenr;
  }
  if (tablenr > MAX_TABLES)
  {
    my_error(ER_TOO_MANY_TABLES,MYF(0),MAX_TABLES);
    DBUG_RETURN(-1);
  }
  while ((item=it++))
  {
    if (item->type() == Item::FIELD_ITEM &&
	((Item_field*) item)->field_name[0] == '*')
    {
      if (insert_fields(thd,tables,((Item_field*) item)->table_name,&it))
	DBUG_RETURN(-1); /* purecov: inspected */
    }
    else
    {
      if (item->fix_fields(thd,tables))
	DBUG_RETURN(-1); /* purecov: inspected */
      if (item->with_sum_func)
	item->split_sum_func(*sum_func_list);
    }
  }
  DBUG_RETURN(test(thd->fatal_error));
}


/****************************************************************************
**	This just drops in all fields instead of current '*' field
**	Returns pointer to last inserted field if ok
****************************************************************************/


static bool
insert_fields(THD *thd,TABLE_LIST *tables, char *table_name,
	      List_iterator<Item> *it)
{
  TABLE_LIST *table;
  uint found;
  DBUG_ENTER("insert_fields");

  found=0;
  for (table=tables ; table ; table=table->next)
  {
    if (!table_name || !strcmp(table_name,table->name))
    {
      Field **ptr=table->table->field,*field;
      while ((field = *ptr++))
      {
	Item_field *item= new Item_field(field);
	if (!found++)
	  (void) it->replace(item);
	else
	  it->after(item);
	if (field->query_id == thd->query_id)
	  thd->dupp_field=field;
	field->query_id=thd->query_id;
      }
      /* All fields are used */
      table->table->used_fields=table->table->fields;
    }
  }
  if (!found)
  {
    if (!table_name)
      my_error(ER_NO_TABLES_USED,MYF(0));
    else
      my_error(ER_BAD_TABLE_ERROR,MYF(0),table_name);
  }
  DBUG_RETURN(!found);
}


/*
** Fix all conditions and outer join expressions
*/

int setup_conds(THD *thd,TABLE_LIST *tables,COND *conds)
{
  DBUG_ENTER("setup_conds");
  thd->set_query_id=1;
  thd->cond_count=0;
  thd->allow_sum_func=0;
  if (conds)
  {
    thd->where="where clause";
    if (conds->fix_fields(thd,tables))
      DBUG_RETURN(1);
  }

  /* Check if we are using outer joins */
  for (TABLE_LIST *table=tables ; table ; table=table->next)
  {
    if (table->natural_join)
    {
      /* Make a join of all fields with have the same name */
      TABLE *t1=table->table;
      TABLE *t2=table->natural_join->table;
      Item_cond_and *and=new Item_cond_and();
      if (!and)					// If not out of memory
	DBUG_RETURN(1);
      
      uint i,j;
      for (i=0 ; i < t1->fields ; i++)
      {
	for (j=0 ; j < t2->fields ; j++)
	{
	  if (!my_strcasecmp(t1->field[i]->field_name,t2->field[j]->field_name))
	  {
	    Item_func_eq *tmp=new Item_func_eq(new Item_field(t1->field[i]),
					       new Item_field(t2->field[j]));
	    if (tmp)				// If not out of memory
	    {
	      tmp->fix_length_and_dec();	// Update cmp_type
	      and->list.push_back(tmp);
	    }
	    break;
	  }
	}
      }
      table->on_expr=and;
      and->used_tables_cache= t1->map | t2->map;
      thd->cond_count+=and->list.elements;
    }
    else if (table->on_expr)
    {
      /* Make a join an a expression */
      if (table->on_expr->fix_fields(thd,tables))
	DBUG_RETURN(1);
      thd->cond_count++;
    }
  }
  DBUG_RETURN(test(thd->fatal_error));
}


/******************************************************************************
** Fill a record with data (for INSERT or UPDATE)
** Returns : 1 if some field has wrong type
******************************************************************************/

int
fill_record(List<Item> &fields,List<Item> &values)
{
  List_iterator<Item> f(fields),v(values);
  Item *value;
  Item_field *field;
  DBUG_ENTER("fill_record");

  while ((field=(Item_field*) f++))
  {
    value=v++;
    if (value->save_in_field(field->field))
      DBUG_RETURN(1);
  }
  DBUG_RETURN(0);
}


int
fill_record(TABLE *table,List<Item> &values)
{
  List_iterator<Item> v(values);
  Item *value;
  DBUG_ENTER("fill_record");

  Field **ptr=table->field,*field;
  while ((field = *ptr++))
  {
    value=v++;
    if (value->save_in_field(field))
      DBUG_RETURN(1);
  }
  DBUG_RETURN(0);
}


static void mysql_rm_tmp_tables(void)
{
  uint index;
  char	filePath[FN_REFLEN];
  MY_DIR *dirp;
  FILEINFO *file;
  DBUG_ENTER("mysql_rm_tmp_tables");

  /* See if the directory exists */
  if (!(dirp = my_dir(mysql_tmpdir,MYF(MY_WME | MY_DONT_SORT))))
    DBUG_VOID_RETURN; /* purecov: inspected */

  /*
  ** Remove all SQLxxx tables from directory
  */

  for (index=2 ; index < (uint) dirp->number_off_files	; index++)
  {
    file=dirp->dir_entry+index;
    if (!bcmp(file,"SQL",3))
    {
      sprintf(filePath,"%s/%s",mysql_tmpdir,file->name); /* purecov: inspected */
      VOID(my_delete(filePath,MYF(MY_WME))); /* purecov: inspected */
    }
  }
  my_dirend(dirp);
  DBUG_VOID_RETURN;
}


/*
** mysql can't yet create index on the fly. all index must be created with
** CREATE TABLE.
** CREATE INDEX is implemented as an CHECK for if an index exists.
** If the CREATE INDEX statement fails then the table should be recreated
** if the index is needed
*/

int mysql_create_index(THD *thd,const char *table_name,
		       List<key_part_spec> &columns,
		       bool unique)
{
  TABLE *table;
  DBUG_ENTER("mysql_create_index");
  if (!(table = open_table(thd,thd->db,table_name,table_name)))
  {
    send_error(&thd->net);
    DBUG_RETURN(1);
  }
  List_iterator<key_part_spec> cols(columns);
  for (uint i=0 ; i < table->keys ; i++)
  {
    if (unique)
    {
      if (table->key_info[i].dupp_key ||
	  table->key_info[i].usable_key_parts != columns.elements)
	continue;
    }
    else if (ha_option_flag[(uint) table->db_type] & HA_READ_ORDER)
    {
      if ((uint) table->key_info[i].key_parts < columns.elements)
	continue;
    }
    else if ((uint) table->key_info[i].key_parts != columns.elements)
      continue;

    cols.rewind();
    key_part_spec *name;
    KEY_PART_INFO *key_part=table->key_info[i].key_part;
    for (; (name=cols++) ;key_part++)
    {
      if (my_strcasecmp(name->field_name,key_part->field->field_name))
	break;
      if (name->length && key_part->length != name->length)
	break;
    }
    if (name)
      continue;
    send_ok(&thd->net);				// Found compatible key
    DBUG_RETURN(0);
  }
#ifdef WANT_ERROR_ON_CREATE_INDEX
  my_error(ER_NO_SUCH_INDEX,MYF(0),table_name);
  DBUG_RETURN(-1);
#else
  send_ok(&thd->net);				// Accept anything
  DBUG_RETURN(0);
#endif
}

/*****************************************************************************
	unireg support functions
*****************************************************************************/

	/* make a select from mysql info
	   Error is set as following:
	   0 = ok
	   1 = Got some error (out of memory?)
	   */

SQL_SELECT *make_select(TABLE **table,uint table_count,uint head,
			uint const_tables, COND *conds, int *error)
{
  SQL_SELECT *select;
  DBUG_ENTER("make_select");

  *error=0;
  if (!conds)
    DBUG_RETURN(0);
  if (!(select= new SQL_SELECT))
  {
    *error= 1;
    DBUG_RETURN(0);				/* purecov: inspected */
  }
  if (head == table_count)
    head--;					// All const tables
  select->read_tables=select->const_tables=PREV_BITS(const_tables);
  select->tables=table;
  select->head=table[head];
  select->cond=conds;

  if (table[head]->io_cache)
  {
    memcpy((gptr) &select->file,(gptr) table[head]->io_cache,sizeof(IO_CACHE));
    select->records=(select->file.end_of_file/
		     select->head->keyfile_info.ref_length);
    my_free((gptr) (table[head]->io_cache),MYF(0));
    table[head]->io_cache=0;
  }
  DBUG_RETURN(select);
}


/*****************************************************************************
*	Approximate how many records will be read from table
*****************************************************************************/

ulong get_quick_record_count(SQL_SELECT *select,uint table,uint keys)
{
  int error;
  DBUG_ENTER("get_quick_record_count");
  if (select)
  {
    select->head=select->tables[table];
    if ((error=select->test_quick_select(keys,0L,~0L)) == 1)
      DBUG_RETURN(select->quick->records);
    if (error == -1)
      DBUG_RETURN(0);
    DBUG_PRINT("warning",("Couldn't use record count on const keypart"));
  }
  DBUG_RETURN(~0L);				/* This shouldn't happend */
}

/*
** Invalidate any cache entries that are for some DB
** We can't use hash_delete when looping hash_elements, so mark them and
** delete what's unused.
*/

void remove_db_from_cache(const my_string db)
{
  for (uint index=0 ; index < open_cache.records ; index++)
  {
    TABLE *table=(TABLE*) hash_element(&open_cache,index);
    if (!strcmp(table->table_cache_key,db))
    {
      table->version=0L;			/* Free when thread is ready */
      if (!table->in_use)
	relink_unused(table);
    }
  }
  while (unused_tables && !unused_tables->version)
    VOID(hash_delete(&open_cache,(byte*) unused_tables));
}


void remove_table_from_cache(const char *db,const char *table_name)
{
  char key[MAX_DBKEY_LENGTH];
  uint key_length;
  TABLE *table;

  key_length=(uint) (strmov(strmov(key,db)+1,table_name)-key)+1;
  for (table=(TABLE*) hash_search(&open_cache,(byte*) key,key_length) ;
       table;
       table = (TABLE*) hash_next(&open_cache,(byte*) key,key_length))
  {
      table->version=0L;			/* Free when thread is ready */
      if (!table->in_use)
	relink_unused(table);
      else
	table->in_use->some_tables_deleted=1;
  }
  while (unused_tables && !unused_tables->version)
    VOID(hash_delete(&open_cache,(byte*) unused_tables));
}
