#
# pgp.tcl
#	PGP 2.6 support for exmh.
#	Orginally contributed by Allan Downey
#	Updated by Stefan Monnier, Anders Klemets and William Sproule
#

# future:
# - encrypt a received message (from Sproule's code)
# - deal with Fcc: headers
# - allow to choose automatic pgp decrypt or not
# - generate application/pgp format=keys-only parts
# - rewrite PGP decrypt, to deal with keys, mime stuff, etc...

# Pgp_Init is in extrasInit.tcl
# to avoid auto-loading this whole file.

proc Pgp_Xterm {} {
    global exmh msg mhProfile exwin

    set file $mhProfile(path)/$exmh(folder)/$msg(id)
    Pgp_Interactive $file
}

# display the message vithout keeping the decrypted form
proc Pgp_View {{file ""}} {
    global exmh msg mhProfile exwin

    if {$file == ""} {
	set file $mhProfile(path)/$exmh(folder)/$msg(id)
    }

    set pgppassword [Pgp_GetPass]
    catch {exec pgp -m "+batchmode=on" $file << $pgppassword } message
    regsub -all "\x07" $message "" message
    Pgp_DisplayText "PGP view $exmh(folder)/$msg(id)" $message 40

    return 1
}

# decrypt the current message
proc Pgp_Decrypt {  } {
    global exmh msg mhProfile exwin

    set file $mhProfile(path)/$exmh(folder)/$msg(id)

    set pgppassword [Pgp_GetPass]
    
    Exmh_Status "pgp $exmh(folder)/$msg(id)"
    catch {exec pgp "+batchmode=on" $file -o $file.msg << $pgppassword } message
    regsub -all "\x07" $message "" message
    if {$message != {}} {
	Pgp_DisplayText "PGP decrypt $exmh(folder)/$msg(id)" $message
    }
    set t $exwin(mtext)

    set orig [open $file r]
    if [catch {
	set mess [open $file.msg r]
    } err] {
	return 0
    }
    set comb [open $file.comb w 0600]

    while {[gets $orig line] != -1} {
	if {[regexp {^-+BEGIN PGP MESSAGE} $line]} {
	    puts $comb [read $mess]
	    while {[gets $orig line] != -1} {
		if {[regexp {^-+END PGP MESSAGE} $line]} { break }
	    }
	} else {
	    puts $comb $line
	}
    }
    close $orig
    close $comb
    close $mess

    exec mv $file.comb $file
    exec rm $file.msg

    set msg(dpy) {}
    MsgChange $msg(id)

    return 1
}

# doesn't do anything more than insert a "pgp-action" pseudo-header
# with the specification of the action in it. The pseudo-header must always
# be on the very first line. If another one is already present, it is replaced
proc Pgp_Encrypt { action draft t } {
    global pgp
    if {! $pgp(enabled)} {
	SeditMsg $t "PGP Not enabled"
	return
    }
    # remove any previous pgp-action pseudo-header and replace it by the new one
    if {[regexp -nocase {^pgp-action:} [$t get 1.0 1.end] ]} {
	$t delete 1.0 2.0
    }
    $t insert 1.0 "pgp-action: $action\n"
}

# check the draft to see what kind (if any) of pgp encryption is
# required before sending. Possible return values are:
# none, encrypt, signclear, signtext, encryptsign
proc Pgp_CheckAction { draft } {
    global pgp

    set orig [open $draft r]
    set pgpaction "none"

    # check the whole header
    while {[gets $orig line] != -1 && ![regexp $pgp(headerend) $line]} {
	if [regexp -nocase {^pgp-action: (.+)$} $line {} action] {
	    set action [string tolower $action]
	    if [info exists pgp(action,$action)] {
		set pgpaction $action
	    } else {
		error "<PGP> unknown action '$action'"
	    }
	}
    }
    close $orig
    return "$pgpaction"
}

proc Pgp_Process { draft } {
    global pgp env

    set action [Pgp_CheckAction $draft]

    if {"$action" == "none"} { return }

    set hasfcc "no"
    set mimeheaders {}
    set pgpaction "$pgp(action,$action)"
    set orig [open $draft r]
    set dest [open $draft.dst w 0600]
    set mesg [open $draft.msg w 0600]

    # init the mesg file
    if {$pgp(rfc822)} {
	puts $mesg "Content-Type: message/rfc822\n"
	set mimeversion "Mime-Version: 1.0"
    } else {
	set mimeversion ""
    }

    # process the header
    while {[gets $orig line] != -1 && ![regexp $pgp(headerend) $line]} {
	# filter out the pseudo header "pgp-action"
	if {![regexp -nocase {^pgp-action: } "$line"]} {
	    if {$pgp(rfc822) && ![regexp -nocase {^(b|f)cc:} $line]} {
		puts $mesg $line
	    }
	    if [regexp -nocase {mime-version: } "$line"] {
		set mimeversion $line

	    } elseif [regexp -nocase {^(content-type|content-transfer-encoding|content-description|content-id|content-disposition): } "$line"] {
		append mimeheaders "$line\n"
	    } else {
		if [regexp -nocase {^fcc:} $line] {
		    set hasfcc "yes"
		}
		puts $dest $line
	    }
	}
    }
    if {"$mimeversion" == ""} {
	puts $dest "Mime-Version: 1.0\nContent-Type: application/pgp ; format=text ; x-action=$action\n$line"
    } else {
	puts $dest "$mimeversion\nContent-Type: application/pgp ; format=mime ; x-action=$action\n$line"
	if {! $pgp(rfc822)} {
	    puts $mesg $mimeheaders
	} else {
	    puts $mesg ""
	}
    }

    while {[gets $orig line] != -1} {
	puts $mesg $line
    }
    close $mesg

    # get the list of recipients' keys if necessary
    set ids {}
    if [regexp {encrypt} $action] {
	# get the list of recipients with "whom" (does alias expansion)
	catch {exec whom -nocheck $draft} recipients
	foreach id [split $recipients "\n"] {
	    if [regexp "^ *-" $id] {
		continue
	    }
	    set id [string trim $id]
	    regsub { at } $id {@} id
	    set ids [concat $ids [Pgp_Match $id]]
	}
	if {$hasfcc == "yes"} {
	    set ids [concat $ids [Pgp_Match "$env(USER)"]]
	}
	ExmhLog "<Pgp_Process> Encoding with public key(s): [join $ids ", "]"
    }
    if [regexp {sign} "$action"] {
	set pgppassword [Pgp_GetPass]
    } else {
	set pgppassword {}
    }
    
    set cmd "exec pgp $pgpaction {+batchmode=on} {+armorlines=0} \$draft.msg {[join $ids "} {"]} << \$pgppassword"
    catch {eval $cmd} message
    regsub -all "\x07" $message "" message
    
    if [catch {open $draft.msg.asc r} in] {
	error "<PGP> unexpected error when executing pgp: $message"
	close $dest
	exec rm $draft.dst $draft.msg
	error "<PGP> can't find pgp-generated file"
    } else {
	puts $dest [read $in]
	close $in
	exec rm $draft.msg $draft.msg.asc
    }
    
    close $dest
    
    exec mv $draft.dst $draft
}

# like a "grep id pubring". Returns the unique element of pgpid that matches id
proc Pgp_Match { email } {
    global pgp

    if {$pgp(cacheids) && [info exists pgp(match,$email)]} {
	return $pgp(match,$email)
    }

    if {![regexp "@" $email]} {
	catch {exec uname -n} hostname
	ExmhLog "<Pgp_Match> use hostname: $email at $hostname"
	set id "$email@$hostname"
    } else {
	set id $email
    }

    # get the potentially interesting parts of id
    set subids {}
    foreach subid [split [string tolower $id] {".{}<>()[]@ }] {
	if {$subid != {}} {
	    lappend subids $subid
	}
    }
    
    set ids {}

    # get the list of pgp userids of the public ring and
    # remove all the uninteresting part of the listing displayed by pgp -kv
    # and change the long string into a list of userids
    catch {exec pgp "+batchmode=on" "+verbose=0" -kv [lindex $subids 0] < /dev/null} pgplist
    regexp "\n(pub.*>)\n" $pgplist {} pgplist
    set pgplist [split $pgplist "\n"]
    set pgpids {}
    foreach pgpid $pgplist {
	lappend pgpids [string trim [string range $pgpid 30 end]]
    }

    # all userids are potential contenders at first
    set idsleft $pgpids
    
    # try to use all the subids we have
    foreach subid $subids {
	
	# filter the contenders based on subid
	set newidsleft {}
	foreach pgpid $idsleft {
	    if {[lsearch -exact [split [string tolower $pgpid] "\".{}<>()\[]@ "] $subid] != -1} {
		lappend newidsleft $pgpid
	    }
	}
	# if there are still several matches, keep on filtering
	if {[llength $newidsleft] > 1} {
	    # if there's only 1 match left, it's THE one.
	    set idsleft $newidsleft
	} elseif {[llength $newidsleft] > 0} {
	    # if there aren't any matches left, the filter was too strict.
	    # drop it and keep on filtering with the old list of contenders
	    break
	}
    }
    if {[llength $newidsleft] == 0} {
	ExmhLog "<Pgp_Match> $id doesn't match any known key"
	set newidsleft [Pgp_Keybox $pgpids $id]
    } elseif {[llength $newidsleft] > 1} {
	ExmhLog "<Pgp_Match> $id is ambiguous: [join $newidsleft ", "]"
	# return -1
	set newidsleft [Pgp_Keybox $newidsleft $id]
    }
    set pgp(match,$email) $newidsleft
    return $newidsleft
}

#
# based on fileselect.tcl
#

proc Pgp_Keybox {keylist id} {
    global keyselect
    set w .keySelect
    
    set purpose "$id doesn't correctly match any key"
    if [Exwin_Toplevel $w "Choose key" Dialog no] {
	# path independent names for the widgets
	
	set keyselect(list) $w.key.sframe.list
	set keyselect(scroll) $w.key.sframe.scroll
	set keyselect(ok) $w.but.ok
	set keyselect(cancel) $w.but.cancel
	set keyselect(msg) $w.label
	set keyselect(text) $purpose
	
	# widgets
	Widget_Frame $w but Menubar {top fillx}
	Widget_Label $w label {top fillx pady 10 padx 20}
	Widget_Frame $w key Dialog {bottom expand fill} -bd 10

	Widget_AddBut $w.but ok OK \
		[list keyselect.ok.cmd $w ] {left padx 1}
	Widget_AddBut $w.but cancel Cancel \
		[list keyselect.cancel.cmd $w ] {right padx 1}

	Widget_Frame $w.key sframe

	scrollbar $w.key.sframe.yscroll -relief sunken \
		-command [list $w.key.sframe.list yview]
	listbox $w.key.sframe.list -relief sunken \
		-yscroll [list $w.key.sframe.yscroll set] -setgrid 1 \
		-geometry 48x16
	pack append $w.key.sframe \
		$w.key.sframe.yscroll {right filly} \
		$w.key.sframe.list {left expand fill}
    }
    set keyselect(result) {}
    $keyselect(msg) configure -text $purpose
    $keyselect(list) delete 0 end
    foreach i $keylist {
	$keyselect(list) insert end $i
    }
    bind $keyselect(list) <Double-ButtonPress-1> {
	%W select from [%W nearest %y]
	$keyselect(ok) invoke
    }
    update idletask
    grab $w
    tkwait variable keyselect(result)
    grab release $w
    if {[llength $keyselect(result)] < 1} {
	error "<PGP> error while matching pgp key ids"
    }
    return $keyselect(result)
}

proc keyselect.ok.cmd {w} {
	global keyselect
	if {[selection own] == $keyselect(list) && \
		    ![catch {set sel [selection get]}]} {
		foreach i $sel {
			if {[regexp {^pub} $i] || [regexp {^sec} $i]} {
				lappend keyselect(result) [lrange $i 3 end]
			} else {
				lappend keyselect(result) \
					[string trimleft $i " "]
			}
		}
	} else {
	    set keyselect(result) {}
	}
	Exwin_Dismiss $w
}

proc keyselect.cancel.cmd {w} {
	global keyselect
	Exwin_Dismiss $w
	set keyselect(result) {}
}

# asks for a password, check its correctness and store it in "pgppassowrd"
proc Pgp_GetPass { } {
    global keyselect pgp env ihatetcltk1 ihatetcltk2 .tmppgppassword
    set w .keyselect
    
    set env(PGPPASSFD) 0
    
    if {$pgp(keeppass) && ($pgp(password) != {})} {
	return $pgp(password)
    } else {
	set pgp(password) {}
    }
    
    for {set keyselect(result) "null"} {$keyselect(result) != "pass"} {} {
	if [Exwin_Toplevel $w "Enter your PGP password" Dialog no] {
	    
	    set keyselect(entry) $w.pass.entry
	    set keyselect(ok) $w.but.ok
	    set keyselect(cancel) $w.but.cancel
	    
	    Widget_Frame $w but Menubar {top fill}
	    Widget_AddBut $w.but ok OK {
		set keyselect(result) "pass"
	    } {left padx 1}
	    Widget_AddBut $w.but cancel Cancel {
		set keyselect(result) "dismiss"
	    } {right padx 1}
	    Widget_Frame $w pass sframe {left filly} -bd 10
	    entry $w.pass.entry -state normal -width 50 -relief sunken
	    pack append $w.pass $w.pass.entry {top fill}
	}

	$w.pass.entry delete 0 end
	set .tmppgppassword {}
	
	bind $keyselect(entry) <Return> {
	    set keyselect(result) "pass"
	}
	SeditBind $keyselect(entry) backspace {
	    global .tmppgppassword
	    $keyselect(entry) delete [expr [$keyselect(entry) index end]-1] end
	    set .tmppgppassword [string range ${.tmppgppassword} 0 [expr [string length ${.tmppgppassword}]-2]]
	}
	bind $keyselect(entry) <Any-Key> {
	    global .tmppgppassword
	    if {"%A" != ""} {
		$keyselect(entry) insert insert "*"
		append .tmppgppassword "%A"
	    }
	}
	focus $keyselect(entry)
	update idletasks
	grab -global $w
	tkwait variable keyselect(result)
	Exwin_Dismiss $w
	grab release $w
	set pgppassword ${.tmppgppassword}
	unset .tmppgppassword

	if {$keyselect(result) == "dismiss"} {
	    return {}
	}
	set tmpname /tmp/exmh-[pid]
	set tmpfile [open $tmpname w]
	puts $tmpfile "salut"
	close $tmpfile
	
	if [catch {
	    exec pgp "+batchmode=on" "+verbose=0" -as $tmpname <<$pgppassword
	} err] {
	    if ![file exists $tmpname.asc] {
		set pgppassword {}
		$keyselect(entry) delete 0 end
		set keyselect(result) "null"
		if ![regexp "PGP" $err] {
		    # Probably cannot find pgp to execute.
		    Exmh_Status !${err}!
		    break
		} else {
		    if [regexp {(Error:[^\.]*)\.} $err x match] {
			Exmh_Status ?${match}?
		    }
		    ExmhLog "<Pgp_keyselect.getpass> $err"
		}
	    }
	}
	exec rm $tmpname
	catch { exec rm $tmpname.asc }
    }
    if {$pgp(keeppass)} {
	set pgp(password) $pgppassword
    }
    
    return $pgppassword
}

proc Pgp_DisplayText { title text {height 8}} {
    global mhProfile exmh msg exwin

    if ![info exists msg(tearid)] {
	set msg(tearid) 0
    } else {
	incr msg(tearid)
    }
    set self [Widget_Toplevel .tear$msg(tearid) $title Clip]

    Widget_Frame $self but Menubar {top fill}
    Widget_AddBut $self.but quit "OK" [list destroy $self]
    Widget_Label $self.but label {left fill} -text $exmh(folder)/$msg(id)
    set t [Widget_Text $self 8 -cursor xterm -setgrid true]
    $t configure -height $height
    $t insert 1.0 $text
}

proc Pgp_ShowMessage { tkw part } {
    global mimeHdr mime pgp

    if {! $pgp(enabled)} {
	Mime_ShowText $tkw $part
	return
    }
    if [info exists mimeHdr($part,param,format)] {
	set format $mimeHdr($part,param,format)
    } else {
	set format text
    }
    if [info exists mimeHdr($part,param,x-action)] {
	set action $mimeHdr($part,param,x-action)
    } elseif [regexp "^text/" $mimeHdr($part,type)] {
	set action sign
    } else {
	set action encrypt
    }

    MimeWithDisplayHiding $tkw $part {
	if {($format == "mime") || ($format == "text")} {
	    set tmpfile "$mimeHdr($part,file).msg"
	    if [regexp "encrypt" $action] {
		set pgppassword [Pgp_GetPass]
	    } else {
		set pgppassword ""
	    }
	    catch {exec pgp "+batchmode=on" "+verbose=0" $mimeHdr($part,file) -o $tmpfile << $pgppassword} msg
	    regsub -all "\x07" $msg "" msg
	    set msg [string trim $msg " \t\n"]
	    if {![file exists $tmpfile]} {
		set msg "PGP failed to decrypt the file:\n$msg"
	    }
	    if {$msg != {}} {
		$tkw insert insert "$msg\n"
		MimeInsertSeparator $tkw $part 4
	    }
	    if [catch {open $tmpfile r} fileIO] {
		Exmh_Status "Cannot open body $tmpfile: $fileIO"
		set mimeHdr($part,numParts) 0
		return
	    }
	    if {$format == "mime"} {
		set mimeHdr($part,numParts) [MimeParseSingle $tkw $part $fileIO]
		set mimeHdr($part=1,color) [MimeDarkerColor $tkw $mimeHdr($part,color)]
		MimeShowPart $tkw $part=1 [MimeLabel $part part] 1
	    } else {
		$tkw insert insert [read $fileIO]
	    }
	    MimeClose $fileIO
	    exec rm -f $tmpfile
	} elseif {$format == "keys-only"} {
	    $tkw insert insert "  PGP keys: (use the menu to extract them)\n\n"
	    catch {exec pgp "+batchmode=on" "+verbose=0" $mimeHdr($part,file)} msg
	    regexp "\n(Type.*>)" $msg {} msg
	    $tkw insert insert "$msg\n"
	    MimeMenuAdd $part command \
		    -label "Extract keys into keyring..." \
		    -command "Pgp_ExtractKeys $mimeHdr($part,file)"
	} else {
	    $tkw insert insert "PGP application format '$format' unknown\n"
	    return
	}
    }
}

proc Pgp_ExtractKeys { file } {
    Pgp_Interactive $file
}

proc Pgp_Interactive { file } {
    global env

    if [catch {exec xterm -geometry 80x24 -title "PGP extract key" \
	    -e sh -c "unset PGPPASSFD; pgp $file; read dummy" } err] {
	Exmh_Status $err
    }
}
