# uri.tcl
#
# A little support for Uniform Resource Identifiers.
#
# martin hamilton <martin@mrrl.lut.ac.uk>
# &
# John Robert LoVerso <loverso@osf.org>
# Toivo Pedaste <toivo@ucs.uwa.edu.au>
# Fred Douglis <douglis@research.att.com>

proc URI_Init {} {
    global env uri

    if [info exists uri(init)] {
	return
    }

    set uri(init) 1

    Preferences_Add "URI" \
      "These options control how exmh deals with Uniform Resource
Identifiers, e.g. World-Wide Web URLs.  You can arrange for URIs
embedded in messages to be turned into hyperlinks, and there is an
option to decipher the experimental X-URI header.  This may be used
by the sender of a message to include contact information, such as
the address of their World-Wide Web homepage.  You can add this
header to your messages by editing compcomps, replcomps and so on." {
	{uri(scanForXURIs) uriScanForXURIs	ON {Scan for X-URI: headers}
"This tells exmh whether to look for X-URI: headers in messages when
you read them."}
	{uri(scanForURIs) uriScanForURIs	OFF {Scan for URIs in messages}
"This tells exmh whether to look for URI in the bodies of messages.
If you turn it on, any URIs it finds will be turned into buttons
which you can click on to launch a viewer application.  NB - this can
slow down message displaying somewhat."}
	{uri(viewerApp) uriViewerApp	{Mosaic} {URI tool}
"This is the application which exmh will launch when you click on the
face of your correspondent, or on a hyperlink in the body of a message."}
    }
}

proc URI_StartViewer {xuri} {
    global uri

    regsub -nocase "URL:" $xuri {} xuri
    string trimright $xuri "."
    Exmh_Status $xuri

    if [catch {eval exec $uri(viewerApp) {$xuri &}} err] {
	Exmh_Status $err
    }
}

proc URI_OpenSelection {} {
    if [catch {selection get} xuri] {
        return
    }
    URI_StartViewer $xuri
}

proc Hook_MsgShowParseUri {msgPath hmm} {
    global uri exwin mimeHdr

    $exwin(faceCanvas) configure -relief flat

    if {[info exists mimeHdr(0=1,hdr,x-uri)] && $uri(scanForXURIs)} {
        set temp_uri [MsgParseFrom $mimeHdr(0=1,hdr,x-uri)]
        regsub -all "\[ \t\n\]" $temp_uri {} temp_uri
        $exwin(faceCanvas) configure -relief raised
        bind $exwin(faceCanvas) <Button-1> [list URI_StartViewer $temp_uri]
    } else {
        bind $exwin(faceCanvas) <Button-1> {}
    }

    if !$uri(scanForURIs) {
        return
    }
    URI_ScanMsg $exwin(mtext)
}

proc Hook_MsgClipParseUri {msgPath t} {
    global uri exwin

    if !$uri(scanForURIs) {
        return
    }
    URI_ScanMsg $t
}

proc URI_ScanMsg { {w {}} } {
    global uri exwin
    if {$w == {}} {
	set w $exwin(mtext)
    }
    $w config -cursor watch
 
    set multiline 0

    for {set i 0} {[$w compare $i.0 <= end]} {incr i} {
        set begin 0
        set text [$w get $i.0 "$i.0 lineend"] 

        #######
        # this could take a while - so keep up to date on events!
        update idletasks
 
        #######
        # check for unencapsulated URIs by protocol if no < > present
        if {[string first "<" $text] == -1 && [string first ">" $text] == -1} {
	    catch {unset end}
            if {[regexp -indices -nocase \
     "(ftp|http|gopher|nntp|telnet|wais|file|prospero|finger)://\[^ '\"\n\t\)\]+" \
              $text positions] == 1} {
                 set end [expr [lindex $positions 1] + 1] 
 
            } elseif {[regexp -indices -nocase \
     "(urn|mailto|news|solo|x500):\[^ \n\t\)\]+" \
               $text positions] == 1} {
                 set end [expr [lindex $positions 1] + 1] 
            }
 
            if [info exists end] {
                set start [lindex $positions 0]
                set temp_uri [string range $text $start $end]
  
                # people put all sorts of nasty things at the end of URLs
  	        set temp_uri [string trimright $temp_uri "\.'>\""]

		TextButtonRange $w $i.$start $i.$end \
                  [list URI_StartViewer $temp_uri]
            }
            continue;
        }
 
        ######
        # check for URIs like <protocol: > present
        if {[regexp -indices -nocase \
  "<(ftp|http|gopher|nntp|telnet|wais|file|prospero|finger|urn|mailto|news|solo|x500)://\[^>)\]+" $text positions] == 1} {
            set end_marker [expr [lindex $positions 1] + 1] 

            if {[info exists end_marker]} {
                set start_marker [lindex $positions 0]
                incr start_marker
                set temp_uri [string range $text $start_marker $end_marker]
 	 	 
                # people put all sorts of nasty things at the end of URLs
  	        set temp_uri [string trimright $temp_uri "\.'>"]

                TextButtonRange $w $i.$start_marker $i.$end_marker \
                  [list URI_StartViewer $temp_uri]

                unset start_marker end_marker positions
            }
 	    continue;
 	}
 
        #######
        # match URIs continued from the previous line
        if $multiline {
            set right [string first ">" [string range $text $begin end]]
            if {$right != -1} {
                set right [expr $right + $begin]
                set last $i.$right
                set temp_uri [$w get $mstart $last]
 
		TextButtonRange $w $mstart $last \
                  [list URI_StartViewer $temp_uri]
 
                set begin $right
	        set multiline 0
            }
	    # note: we will continue to look until a close is found
        }
 
        #######
        # match URIs wholly contained on one line
        if {[regexp -indices -nocase "<(URN|URL):\[^>\]+>" \
          [string range $text $begin end] indices] == 1} {
            set start [expr $begin + [lindex $indices 0] + 1]
            set end [expr $begin + [lindex $indices 1]]
            set temp_uri [$w get $i.$start $i.$end]
 
	    TextButtonRange $w $i.$start $i.$end \
              [list URI_StartViewer $temp_uri]

            set begin [lindex $indices 1]
        }
 
        #######
        # match real HTML links
        if {[regexp -indices -nocase "<a href=\[^>\]+>" \
          [string range $text $begin end] indices] == 1} {
            set left [expr $begin + [lindex $indices 0] + 8]
            set right [expr $begin + [lindex $indices 1]]
            set temp_uri [$w get "$i.$left" "$i.$right"]

	    set temp_uri [string trim $temp_uri "\""]

	    TextButtonRange $w $i.$left $i.$right \
              [list URI_StartViewer $temp_uri]

            set begin [lindex $indices 1]
            $w mark unset left right
        }
 
        #######
        # match the start of a URI which is broken over more than one
        # line - must include <URN or <URL
        if {[regexp -indices -nocase "<(URN|URL)" \
          [string range $text $begin end] indices] == 1} {
            set left [expr [lindex $indices 0] + $begin + 1]
            set mstart $i.$left
            set multiline 1
        }
     }

     $w config -cursor xterm
}

