#
# mxedit.utils --
#
# This script defines editing utility operations for the mxedit widget.
#
# Copyright (c) 1992 Xerox Corporation.
# Use and copying of this software and preparation of derivative works based
# upon this software are permitted. Any distribution of this software or
# derivative works must comply with all applicable United States export
# control laws. This software is made available AS IS, and Xerox Corporation
# makes no warranty about the software, its performance or its conformity to
# any specification.

proc mxUtilsInit {} {
    global mxLastPos mxLastCol	;# For relative motion
    global mxFloatingSel 	;# For dragging around detached selections
    global mxSeeJump		;# To control scrolling at top and bottom
    global mxMarkActive		;# Keyboard selection is active
    global mxScrollActive	;# Scrolling action is active

    if ![info exists mxLastPos] { set mxLastPos 0.0 }
    if ![info exists mxLastCol] { set mxLastCol 0 }
    if ![info exists mxFloatingSel] { set mxFloatingSel 0 }
    if ![info exists mxMarkActive] { set mxMarkActive 0 }
    if ![info exists mxScrollActive] { set mxScrollActive 0 }
    if ![info exists mxSeeJump] { set mxSeeJump 0 }

}
########################### EDIT PROCEDURES ###########################

# mxUndo --
#	Undo the previous editing command.  mxedit provides a complete
#	editing log so repeated invocations of undo keeps undoing more.
#	Hint: to roll forward after undoing a lot, insert a space, then
#	go back to using undo repeatedly.  Go ahead, confuse yourself.

proc mxUndo { } {
    global mxedit
    if [catch {$mxedit history next mxHistory [list $mxedit undo more]} msg] {
	mxFeedback $msg
	return $msg
    }
}

# mxRedo --
#	Redo the previous editing sequence.
#	Each sequence is delimited by mouse clicks;
#	there are no explicit "start" and "stop" remembering commands.
#	Note: redo depends on a call to
#	$mxedit history on
#	in order to turn history on

proc mxRedo { } {
    global mxedit mxHistory
    # The following variables end up in the history sequence
    global tk_strictMotif tk_priv
    if [catch {
		$mxedit history add $mxHistory
		$mxedit history ignore {eval $mxHistory}
	      } msg] {
	puts stderr "${msg}\n${mxHistory}"
	mxFeedback $msg
	return $msg
    }
}

# mxDeleteNoSave --
#	Delete the selection without saving it (NOT USED)

proc mxDeleteNoSave { } {
    if [catch {mxDelete sel.left sel.right}] {
	mxFeedback "nothing is selected in this file"
    }
}

# mxDeleteSave --
#	Delete the selection and save it in the X cut buffer

proc mxDeleteSave { } {
    if [mxSelection here] {
	catch {
	    cutbuffer set 0 [selection get]
	    catch {emacsKill sel copy}
	    mxDelete sel.left sel.right
	}
    } else {
	mxFeedback "Selection not in window"
    }
}

# mxCopySave --
#	Copy the selection to the good-old cutbuffer

proc mxCopySave { } {
    if [mxSelection here] {
	if [catch {cutbuffer set 0 [selection get]} msg] {
	    mxFeedback $msg
	}
	catch {emacsKill sel copy}
    } else {
	mxFeedback "Selection not in window"
    }
}

# deleteForwChar --

proc mxDeleteForwChar { } {
    mxBatchDelete caret
}

# mxDeleteBackChar --

proc mxDeleteBackChar { } {
    mxBatchDelete [mxMark caret back 1 char]
}

# mxDeleteBackWord --

proc mxDeleteBackWord { } {
    mxBatchDelete [mxMark caret back 1 word] [mxMark caret back 1 char]
}

# mxDeleteForwWord --

proc mxDeleteForwWord { } {
    mxBatchDelete caret [mxMark [mxMark caret forw 1 word] back 1 char]
}

# mxDeleteEndOfLine --

proc mxDeleteEndOfLine { } {
    if {[string compare [mxMark caret] [mxMark caret char -1]] != 0} {
	mxDelete caret [mxMark [mxMark caret char -1] back 1 char]
    } else {
	mxDelete [mxMark caret]
    }
}

# mxDeleteLine --

proc mxDeleteLine { } {
    if {[string compare [mxMark caret] [mxMark caret char -1]] != 0} {
	mxDelete [mxMark caret char 0] [mxMark caret char -1]
    } else {
	mxDelete [mxMark caret]
    }
}

# mxMoveSel --
#	Move the selection to the insert point

proc mxMoveSel {  } {
    if [mxSelection here] {
	set _t [mxMark caret]
	if [catch {mxInsert [selection get]} msg] {
	    mxFeedback $msg
	} else {
	    set _l [mxMark sel.left]
	    set _r [mxMark sel.right]
	    mxSelection set $_t [mxMark caret back 1 char]
	    mxDelete $_l $_r noviewchange
	}
    } else {
	mxFeedback "selection not in this window"
    }
}

# mxPaste --
#	Insert the selection at the insert point.
#	In order to support things like OpenLook that have a cut-then-paste
#	paradigm, we fall back to inserting the saved selection on the
#	assumption that the user has done a recent cut.  This only
#	works within a single window, however...

proc mxPaste { } {
    if [catch {mxInsert [selection get]} ] {
	if [catch {mxInsert [cutbuffer get]} msg] {
	    mxFeedback "No primary selection or cutbuffer value: $msg"
	}
    }
}

# mxOpenLineBelow --

proc mxOpenLineBelow { } {
    mxCaret [mxMark caret char -1]
    mxNewline
}

# mxOpenLineAbove --

proc mxOpenLineAbove { } {
    mxInsert \n [mxMark caret char 0]
    mxCaret [mxMark [mxMark caret char 0] back 1 char]
}

# mxIndentLine --
#	indent the line with the caret
# 	A (user-settable) indent variable is used to control the amount

proc mxIndentLine { } {
    global mxIndent
    mxIndent caret caret + $mxIndent
}

# mxOutdentLine --
#	outdent the line with the caret

proc mxOutdentLine { } {
    global mxIndent
    mxIndent caret caret - $mxIndent
}
# mxIndentSel --
# Indent the selected region

proc mxIndentSel { } {
    global mxIndent
    if [mxSelection here] {
	mxIndent sel.left sel.right + $mxIndent
    } else {
	mxFeedback "Selection not in window"
    }
}

# mxOutdentSel --
# Outdent the selected region

proc mxOutdentSel { } {
    global mxIndent
    if [mxSelection here] {
	mxIndent sel.left sel.right - $mxIndent
    } else {
	mxFeedback "Selection not in window"
    }
}

#################################### Caret Movement ########################

# Note that these routines use mxCaretMove, not mxCaret, so that
# the mxMarkActive mode variable can be checked.  This mode adds
# selection adjustment to cursor movement so you can select regions
# without the mouse.

# mxBack1char --
#	Move the insert point one character back

proc mxBack1char { } {
    mxCaretMove [mxMark caret back 1 char]
    mxSee caret
}

# mxForw1char --
#	Move the insert point one character forward

proc mxForw1char { } {
    mxCaretMove [mxMark caret forw 1 char]
    mxSee caret
}

# mxBack1Word --
#	Move the insert point one word backward

proc mxBack1Word { } {
    mxCaretMove [mxMark caret back 1 word]
    mxSee caret
}

# mxForw1Word --
#	Move the insert point one word forward

proc mxForw1Word { } {
    mxCaretMove [mxMark caret forw 1 word]
    mxSee caret
}

# mxNextLine --
#	Move the insert point to the begining of the next line

proc mxNextLine {}  {
    mxCaretMove [mxMark [mxMark caret forw 1 line] char 0]
    mxSee caret
}

# mxDown1Line --
#	Move the insert point down one line, maintaining current column
#	If mxSeeJump is not set (the default) then we just scroll
#	down by one line if the caret is at the bottom of the window.
#	Otherwise, we use a naked mxSee caret which will jump
#	the screen so the caret is in the middle if it was at the bottom.

proc mxDown1Line { } {
    global mxLastPos mxLastCol mxSeeJump

    if {[string compare $mxLastPos [mxMark caret]] != 0} {
	set mxLastCol [mxColumn caret]
    }
    set mxLastPos [mxMark [mxMark caret forw 1 line] column $mxLastCol]
    mxCaretMove $mxLastPos

    if {!$mxSeeJump && ($mxLastPos > [mxMark bottom])} {
	mxSee caret bottom
    } else {
	mxSee caret
    }
    return $mxLastPos
}

# mxUp1Line --
#	Move the insert point up 1 line
#	If mxSeeJump is not set (the default) then we just scroll
#	up by one line if the caret is at the top of the window.
#	Otherwise, we use a naked mxSee caret which will jump
#	the screen so the caret is in the middle if it was at the top.

proc mxUp1Line { } {
    global mxLastPos mxLastCol mxSeeJump

    if {[string compare $mxLastPos [mxMark caret]] != 0} {
	set mxLastCol [mxColumn caret]
    }
    set mxLastPos [mxMark [mxMark caret back 1 line] column $mxLastCol]
    mxCaretMove $mxLastPos
    if {!$mxSeeJump && ($mxLastPos < [mxMark top])} {
	mxSee caret top
    } else {
	mxSee caret
    }
    return $mxLastPos
}

# mxBeginOfLine --
#	Move the caret to the beginning of the line

proc mxBeginOfLine { } {
    mxCaretMove [mxMark caret char 0]
    mxSee caret
}

# mxEndOfLine --
#	Move the caret to the end of the line

proc mxEndOfLine { } {
    set _t [mxMark caret char -1]
    if {[string compare [mxMark caret] $_t] == 0} {
	# Caret is already at the end of line
	set _t [mxMark [mxMark caret forw 1 line] char -1]
	mxCaretMove $_t
    } else {
	# Tuned selection setting
	global mxMarkActive mxedit
	$mxedit caret $_t
	if {$mxMarkActive} {
	    $mxedit selection f_adjust [mxMark $_t back 1 char]
	}
    }
    mxSee caret
}

# mxPageUp --
#	Move the caret up, towards the beginning of the file, one screen

proc mxPageUp { } {
    scan [mxGeometry] "%dx%d" _width _height
    mxCaretMove [mxMark caret back $_height lines]
    mxSee caret
}

# mxPageDown --
#	Move the caret down, towards the end of the file, one screen

proc mxPageDown { } {
    scan [mxGeometry] "%dx%d" _width _height
    mxCaretMove [mxMark caret forw $_height lines]
    mxSee caret
}
####################### SELECTION HANDLING ##########################

# mxApplyToSelection --
#	Apply a command to the current selection, catching errors.
#	

proc mxApplyToSelection { prefix } {
    if [catch {selection get} sel] {
	mxFeedback "$prefix: $sel"
    } else {
	return [eval [concat $prefix [list $sel]]]
    }
}

# Building blocks for mouse bindings

# mxRedoBarrier --
#	Set redo/history barriers between mouse clicks
#
proc mxRedoBarrier { widget } {
    global mxHistory
    $widget undo mark
    $widget history next mxHistory	;# shift history into mxHistory variable
}

# mxMouseUp --
#	Reset selection modes upon button release
#
proc mxMouseUp { widget } {
    global mxSelectionMode mxFloatingSel
    set mxFloatingSel 0
}

# mxCaretNoSelChar --
#	sets the caret and initializes selection state, but doesn't
#	actually begin a new selection.
#
proc mxCaretNoSelChar { widget mark } {
    global mxSelectionMode mxFloatingSel mxMarkActive
    $widget caret $mark
    focus $widget
    set mxSelectionMode char
    set mxFloatingSel 0
    set mxMarkActive 0
}

# mxCaretSelChar --
#	sets the caret and initializes selection state
#
proc mxCaretSelChar { widget mark } {
    global mxSelectionMode mxFloatingSel mxMarkActive
    set mxSelectionMode char
    $widget caret $mark
    $widget selection anchor $mark
    $widget selection f_adjust $mark
    focus $widget
    set mxFloatingSel 0
    set mxMarkActive 0
}

# mxCaretSelWord --
#	Interpret a hit as selecting a word
#
proc mxCaretSelWord { widget mark } {
    global mxSelectionMode mxFloatingSel
    set mxSelectionMode word
    $widget selection anchor $mark
    $widget selection f_adjust $mark
    $widget caret sel.left
    focus $widget
    set mxFloatingSel 1
}

# mxCaretSelLine --
#	Interpret a hit as selecting a line
#
proc mxCaretSelLine { widget mark } {
    global mxSelectionMode mxFloatingSel
    set mxSelectionMode line
    $widget selection anchor $mark
    $widget selection f_adjust $mark
    $widget caret sel.left
    focus $widget
    set mxFloatingSel 1
}

# mxCaretSelMove --
#	This moves the caret and/or the selection along under the mouse
#
proc mxCaretSelMove { widget mark } {
    global mxFloatingSel
    if {$mxFloatingSel} {
	$widget selection anchor $mark
	$widget selection f_adjust $mark
	$widget caret sel.left
    } else {
	$widget caret $mark
    }
}

# mxCaretSelAdjust --
#	Adjust the selection anchored to the caret
#
proc mxCaretSelAdjust { widget mark } {
    $widget selection adjust $mark
    focus $widget
}

# mxSelChar --
#	Start a character selection detached from the caret
#
proc mxSelChar { widget mark } {
    global mxSelectionMode
    set mxSelectionMode char
    $widget selection anchor $mark
    $widget selection f_adjust $mark
}

# mxSelMove --
#	Move the selection detached from the caret
proc mxSelMove { widget mark } {
    $widget selection anchor $mark
    $widget selection f_adjust $mark
}

# mxSelWord --
#	Start a word selection detatched from the caret
#
proc mxSelWord { widget mark } {
    global mxSelectionMode
    set mxSelectionMode word
    $widget selection anchor $mark
    $widget selection f_adjust $mark
}
# mxSelLine --
#	Start a line selection detatched from the caret
#
proc mxSelLine { widget mark } {
    global mxSelectionMode
    set mxSelectionMode line
    $widget selection anchor $mark
    $widget selection f_adjust $mark
}

# mxSelAdjust --
#	Adjust the detached selection
#
proc mxSelAdjust { widget mark } {
    $widget selection f_adjust $mark
    focus $widget
}


############################### Scrolling ##########################

# mxScanMark --
#	Put down the marker for a scroll action

proc mxScanMark { widget y } {
    global mxScrollActive
    set mxScrollActive 1
    $widget scan mark $y
}

# mxScanDragto --
#	Scroll it.  mxScrollActive is used because some click-to-type
#	window managers will deliver motion events *before* the
#	button-press events.

proc mxScanDragto { widget y {speed 1} } {
    global mxScrollActive
    if {$mxScrollActive} {
	$widget scan dragto $y $speed
    }
}

# mxScanDone --
#	Done scrolling - clear active flag.

proc mxScanDone { widget } {
    global mxScrollActive
    set mxScrollActive 0
}

############################### Callbacks ##########################

# If the following procedures are defined, they are invoked by
# the mxedit implementation to notify the script-level about
# internal state changes

# mxSizeChangeCallback --
#	Called when the geometry of the window changes
#	This is called as a result of ConfigureNotify X events,
#	which are apparently only generated when the size changes,
#	not the location.

proc mxSizeChangeCallback { } {
#    mxFeedback "New geometry: [geometry] [winfo geometry .]"
}

# mxStateChangeCallback --
#	Called when the clean/dirty state of the file changes

proc mxStateChangeCallback { } {
    global mxFile
    mxNameWindow . $mxFile
}


############################### Abbreviations #######################

# Very weak support for abbreviations that could ultimately be
# expanded to support editing modes (like C or M3 mode)

# mxAbbrev --
#	Set up an abbreviation so that typing the short sequence
#	is equivalent to the longer one.

proc mxAbbrev { abbrev args } {
    mxBind $abbrev "delete caret \[mxMark caret back [expr [string length $abbrev]-1] chars\] ; mxInsertWords $args"
}

# mxInsertWords --
#	Insert a bunch of words.  This won't do the right thing with tabs.

proc mxInsertWords { args } {
    set space {}
    foreach word $args {
	mxBatchInsert $space
	mxBatchInsert $word
	set space " "
    }
}
############################### Miscellany ##########################

# mxLastCommand --
#	Return the last (or N'th to last) top-level command
#

proc mxLastCommand { {back 0} } {
    set curSequence [split [mxHistory info] \n]
    set last [lindex $curSequence [expr [llength $curSequence]-1-$back]]
    return $last
}


# mxGeometry --
#	Set the window's X geometry

proc mxGeometry { { xGeometry none } } {
    if { [string compare $xGeometry none] == 0} {
	return [wm geometry .]
    } else {
	return [wm geometry . $xGeometry]
    }
}

# screenwidth --
#	Return the width of the screen

proc screenwidth {} {
    return [winfo screenwidth .]
}

# screenheight --
#	Return the height of the screen

proc screenheight {} {
    return [winfo screenheight .]
}

# mxLine --
#	Make a particular line visible

proc mxLine { { i _no_i_ } } {
    if {$i != "_no_i_"} {
	if {[scan $i %d _t] != 1} {error [format {bad line number "%s"} $i]}
	set _t [format %d.0 $i]
	mxSee $_t
	mxSelection set $_t [mxMark $_t char -1]
	mxCaret $_t
	return ""
    } else {
	scan [mxMark caret] "%d" line
	return $line
    }
}

# mxTag --
#	Switch files and tag to the given name.

proc mxTag { name } {
    if [catch {mxTaginfo $name} i] {
	mxFeedback $i
    } else {
	mxSwitch [lindex $i 0]
	mxSearch forw [lindex $i 1]
    }
}

# mxTagOpen --
#	Open a new window and tag to the given name.

proc mxTagOpen { name } {
    if [catch {mxTaginfo $name} i] {
	mxFeedback "$i"
    } else {
	set newWindow [mxOpen [lindex $i 0]]
	send $newWindow "mxSearch forw \{[lindex $i 1]\}"
    }
}

# mxCaretInfo --
#	Returns lines and caret information

proc mxCaretInfo { } {
    global mxFile
    scan [mxMark eof] %d _t
    scan [mxMark caret] %d _t2
    return [format {\"%s\" : %d total lines, caret on line %d} $mxFile $_t $_t2]
}

# mxShellSel -- apply a shell command to the selection

proc mxShellSel { args } {
    global tid

    if ![mxSelection here] {
	return "Selection not in window"
    }
    if [catch {selection get} sel] {
	return "Cannot get selection"
    }

    if ![info exists tid] { set tid 1 } else { incr tid }
    while {[file exists /tmp/shellsel$tid]} {
	incr tid
    }
    if [catch {open "/tmp/shellsel$tid" w} tmpfile] {
	return "Cannot open /tmp/shellsel$tid"
    }
    puts $tmpfile $sel
    close $tmpfile

    if [catch {eval "exec $args < /tmp/shellsel$tid > /tmp/shellsel$tid.out"} msg] {
	catch {exec rm $tmpfile $tmpfile.out}
	return $msg
    }
    mxCaret [mxMark sel.left]
    mxDelete [mxMark sel.left] [mxMark sel.right]
    mxRead /tmp/shellsel$tid.out
    catch {exec rm $tmpfile $tmpfile.out}
    return
}

# mxIncrSel - replace a pattern with an incrementing number

proc mxIncrSel { pattern } {
    global myIncrN
    global mxReplaceString mxedit
    if ![info exists myIncrN] {
	set myIncrN 1
    }
    search forw $pattern
    set mxReplaceString $myN
    incr myN
    $mxedit replace selection $mxReplaceString
}

proc mxHideGlobalWindow {} {
    mxGlobalEval wm withdraw .
}
proc mxShowGlobalWindow {} {
    mxGlobalEval wm deiconify .
}

################ Emacs-like dot-and-mark support ##########################

proc mxExchDotMark {} {
    global mxedit

    catch {
	# Note, [mxSelection here] is fooled if you get the
	# selection and then delete it.  Hence this catch instead
	# of a check with mxSelection here.
	if {[mxMark sel.left] < [mxMark caret]} {
	    $mxedit caret [mxMark sel.left]
	    $mxedit selection anchor [mxMark sel.right]
	} else {
	    $mxedit caret [mxMark sel.right]
	    $mxedit selection anchor [mxMark sel.left]
	}
    }
}

################ Routines that dump info to a scratch window ##############

proc mxShowDbUser {} {
    set newWindow [mxOpen {} ]
    set text [exec dbmDump [glob ~/.mxdb]]
    send $newWindow [list mxInsert $text]
}

proc mxShowProcs {args} {
    set newWindow [mxOpen {}]
    send $newWindow {mxInsert Procedure\ information:\n}
    send $newWindow {mxInsert ---------\ -----------}
    if {[llength $args] == 0} {set args [lsort [info procs]]}
    foreach proc $args {
	set space {}
	send $newWindow [list mxInsert [format \n\n%s( $proc]]
	send $newWindow mxClean
	foreach param [info args $proc] {
	    send $newWindow [list mxInsert [format %s%s $space $param]]
	    set space {, }
	    if [info default $proc $param default] {
		send $newWindow [list mxInsert [format { [%s]} $default]]
	    }
	}
	send $newWindow {mxInsert ):\n}
	send $newWindow [list mxInsert [info body $proc]]
    }
    send $newWindow mxClean
    send $newWindow {mxSee 0.0}
}

proc mxShowVars {args} {
    set newWindow [mxOpen {}]
    send $newWindow {mxInsert Variable\ values:\n}
    send $newWindow {mxInsert --------\ -------\n}
    set _maxLength 10
    if {[llength $args] == 0} {set args [lsort [uplevel #0 {info vars}]]}
    foreach _i $args {
	if {[string length $_i] > $_maxLength} {
	    set _maxLength [string length $_i]
	}
    }
    set _maxLength [expr $_maxLength+6]
    set format "\\n%-${_maxLength}s = \"%s\""
    foreach _i $args {
	if [catch {uplevel #0 "set $_i"} value] {
	    # The variable is probably an array
	    set _maxLength 10
	    if {[catch {lsort [uplevel #0 "array names $_i"]} names] == 0} {
#		foreach _j $names {
#		    if {[string length $_j] > $_maxLength} {
#			set _maxLength [string length $_j]
#		    }
#		}
#		set _maxLength [expr $_maxLength+[string length $_i]+2]
#		set format2 "\\n%-${_maxLength}s = \"%s\""
		set format2 $format
		foreach _j $names {
		    if {[catch {uplevel #0 "set ${_i}($_j)"} value] == 0} {
			send $newWindow [list mxInsert \
				[format $format2 \
					[format "%s(%s)" $_i $_j] $value]]
		    }
		}
	    }
	} else {
	    send $newWindow [list mxInsert [format $format $_i $value]]
	}
    }
    send $newWindow mxClean
    send $newWindow {mxSee 0.0}
}

proc mxShowHistory { {when current} } {
    set newWindow [mxOpen {} ]
    case $when {
	current { set text [mxHistory info] }
	last { set text $mxHistory }
	default { set text "Invalid history when $when" }
    }
    send $newWindow [list mxInsert $text]
}

proc mxShowKillBuffers { } {
    set newWindow [mxOpen {} ]
    if [catch {mxGlobalEval set mxKillTop} top] {
	send $newWindow [list mxInsert "
No Kill Buffers
Try emacsKillBindings or mxKillBindings
They provide bindings for:
    emacsKill - push kill buffer, or append to current kill buffer
    emacsYank - insert current kill buffer
    emacsYankPop - pop kill buffer, replace last yank with current buffer
"]
    } else {
	send $newWindow [list mxInsert "Kill Buffer Dump - top is $top\n\n"]
	for {set i $top} {$i >= 0} {incr i -1} {
	    send $newWindow [list mxInsert "Kill Buffer $i\n"]
	    catch {
		set text [mxGlobalEval set mxKillBuf($i)]\n\n
		send $newWindow [list mxInsert $text]
#		send $newWindow mxNewline
#		send $newWindow mxNewline
	    }
	}
    }
}

#############################################################################
#
# Extract procedure headers
#
proc mxDumpProcHeaders { args } {
    global mxLibrary
    if {[llength $args] == 0} {
	set args [lsort [glob $mxLibrary/mxedit.*]]
    }
    set newWindow [mxOpen {} ]
    foreach file $args {
	if {[string match "*.tutorial" $file] ||
	    [string match "*.help" $file]} {
	    continue
	}
	send $newWindow [list mxInsert ".DS\n"]
	send $newWindow [list mxInsert ${file}\n]
	catch {exec egrep ^proc $file | sed s/\{\$//} stuff
	send $newWindow [list mxInsert $stuff]
	send $newWindow [list mxInsert "\n.DE\n"]
	send $newWindow update
	send $newWindow [list mxSee caret]
    }
}
