static char rcsid[]="$Id: tkCanvImage.c,v 1.2 94/02/14 14:52:58 mangin Exp $";

/*
 *
 * Modified from tkCanvBmap.c
 *
 *     Frank Mangin
 *
 *    Wed Dec  8 1993
 */

/* 
 * tkCanvImage.c --
 *
 *	This file implements Image items for canvas widgets.
 *
 * Copyright (c) 1992-1993 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <stdio.h>
#include <sys/types.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <tkInt.h>
#include <tkConfig.h>
#include "tkCanvas.h"
#include <tk.h>

#define strinit(S,V) \
  ((S) = (char *)malloc((1+strlen(V))*sizeof(char)), sprintf((S),(V)))

#define DEG2RAD(A) (M_PI*(A)/180.0)
  
#define ZEROF 0.000001
#define FNULL(A) (((A) > -ZEROF) && ((A) < ZEROF))
#define FEQUAL(A,B) FNULL((A)-(B))
#define F2I(A) ((int)(0.5 + (A)))

/*
 * The structure below defines the record for each Image item
 */

/****  xc, yc, width, height, and angle are theoretical exact parameters  ****/
/****  xregion, regw, and regh are integers approximate values used for display  ****/

typedef struct ImageItem  {
    Tk_Item header;		/* Generic stuff that's the same for all
				 * types.  MUST BE FIRST IN STRUCTURE. */
    double xc, yc;		/* coordinates of image center */
    double width, height;          /* Image dimensions */
    double angle;		/* width side orientation in degrees */
    				/* (0 => width horizontal) */
    double cosine, sine;	/* cos and sin of angle */
    int flip;			/* whether image must be horizontally flipped */
    double brightness;		/* Brightness level between -1.0 and 1.0 */
    double contrast;  		/* contrast level between -1.0 and 1.0 */
    char *path;		        /* Path of image data file */
    FILE *stream;		/* image data file descriptor */
    int offset;	                /* Offset in path to find image data */
    int natw, nath;		/* natural image dimensions */
    Region xregion;		/* oblique rectangle occupied by the rotated image */
    int regw, regh;		/* horiz and vert dims of rotated displayed image */
    				/* (== dims of xregion if not None) */
    GC gc;			/* gc for clipping rotated images */
    XImage *xim;                /* XImage structure */
} ImageItem;

/*
 * Information used for parsing configuration specs:
 */

static int              ParseImageDim _ANSI_ARGS_((ClientData clientData,
                            Tcl_Interp *interp, Tk_Window tkwin, char *value,
                            char *recordPtr, int offset));
static char *           PrintImageDim _ANSI_ARGS_((ClientData clientData,
                            Tk_Window tkwin, char *recordPtr, int offset,
                            Tcl_FreeProc **freeProcPtr));

static Tk_CustomOption imageDimOption =
{ ParseImageDim, PrintImageDim, (ClientData) NULL};
					   
static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_CUSTOM, "-width", (char *) NULL, (char *) NULL,
       (char *) NULL, Tk_Offset(ImageItem, width),
       TK_CONFIG_DONT_SET_DEFAULT, &imageDimOption},
    {TK_CONFIG_CUSTOM, "-height", (char *) NULL, (char *) NULL,
       (char *) NULL, Tk_Offset(ImageItem, height),
       TK_CONFIG_DONT_SET_DEFAULT, &imageDimOption},
    {TK_CONFIG_DOUBLE, "-angle", (char *) NULL, (char *) NULL,
       (char *) NULL, Tk_Offset(ImageItem, angle), TK_CONFIG_NULL_OK},
    {TK_CONFIG_INT, "-flip", (char *) NULL, (char *) NULL,
       (char *) NULL, Tk_Offset(ImageItem, flip), TK_CONFIG_NULL_OK},
    {TK_CONFIG_DOUBLE, "-brightness", (char *) NULL, (char *) NULL,
       (char *) NULL, Tk_Offset(ImageItem, brightness), TK_CONFIG_NULL_OK},
    {TK_CONFIG_DOUBLE, "-contrast", (char *) NULL, (char *) NULL,
       (char *) NULL, Tk_Offset(ImageItem, contrast), TK_CONFIG_NULL_OK},
    {TK_CONFIG_STRING, "-path", (char *) NULL, (char *) NULL,
       (char *) NULL, Tk_Offset(ImageItem, path), TK_CONFIG_NULL_OK},
    {TK_CONFIG_INT, "-offset", (char *) NULL, (char *) NULL,
       (char *) NULL, Tk_Offset(ImageItem, offset), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
       (char *) NULL, 0, TK_CONFIG_NULL_OK, &tkCanvasTagsOption},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
       (char *) NULL, 0, 0}
};

/*
 * Prototypes for procedures defined in this file:
 */

static int	ImageCoords _ANSI_ARGS_((Tk_Canvas *canvasPtr, Tk_Item *itemPtr,
					 int argc, char **argv));
static int	ImageToArea _ANSI_ARGS_((Tk_Canvas *canvasPtr, Tk_Item *itemPtr,
					 double *areaPtr));
static double	ImageToPoint _ANSI_ARGS_((Tk_Canvas *canvasPtr, Tk_Item *itemPtr,
					  double *coordPtr));
static int	ImageToPostscript _ANSI_ARGS_((Tk_Canvas *canvasPtr, Tk_Item *itemPtr,
					       Tk_PostscriptInfo *psInfoPtr));
static void	ComputeImageBbox _ANSI_ARGS_((Tk_Canvas *canvasPtr, ImageItem *ImagePtr));
static int	ConfigureImage _ANSI_ARGS_((Tk_Canvas *canvasPtr, Tk_Item *itemPtr,
					    int argc, char **argv, int flags));
static int	CreateImage _ANSI_ARGS_((Tk_Canvas *canvasPtr, Tk_Item *itemPtr,
					 int argc, char **argv));
static void	DeleteImage _ANSI_ARGS_((Tk_Canvas *canvasPtr, Tk_Item *itemPtr));
static void	DisplayImage _ANSI_ARGS_((Tk_Canvas *canvasPtr, Tk_Item *itemPtr,
					  Drawable drawable));
static void	ScaleImage _ANSI_ARGS_((Tk_Canvas *canvasPtr, Tk_Item *itemPtr,
					double originX, double originY,
					double scaleX, double scaleY));
static void	TranslateImage _ANSI_ARGS_((Tk_Canvas *canvasPtr, Tk_Item *itemPtr,
					    double deltaX, double deltaY));

/**
 **  Prototypes for functions defined in fastimgeom.c
 **/
     
extern void ZoomTurn
  _ANSI_ARGS_((float *src, int srcw, int srch,
	       unsigned char *dst, int dstw, int dsth, int dstlinew,
	       int zflag, float xfact, float yfact,
	       int rflag, float angle, float csn, float sn,
	       Region xregion, float a, float b,
	       int *pixels, int numpixels));

/*
 * The structures below define the image item type
 * by means of procedures that can be invoked by generic item code.
 */

static Tk_ItemType TkImageType = {
    "image",				/* name */
    sizeof(ImageItem),			/* itemSize */
    CreateImage,			/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureImage,			/* configureProc */
    ImageCoords,			/* coordProc */
    DeleteImage,			/* deleteProc */
    DisplayImage,			/* displayProc */
    0,					/* alwaysRedraw */
    ImageToPoint,			/* pointProc */
    ImageToArea,			/* areaProc */
    ImageToPostscript,			/* postscriptProc */
    ScaleImage,			        /* scaleProc */
    TranslateImage,			/* translateProc */
    (Tk_ItemIndexProc *) NULL,		/* indexProc */
    (Tk_ItemCursorProc *) NULL,		/* icursorProc */
    (Tk_ItemSelectionProc *) NULL,	/* selectionProc */
    (Tk_ItemInsertProc *) NULL,		/* insertProc */
    (Tk_ItemDCharsProc *) NULL,		/* dTextProc */
    (Tk_ItemType *) NULL		/* nextPtr */
};

/***************************************************************
 ****	       Canvas image item creation                   ****
 ***************************************************************/

void ImageItemTypeCreate()
{
  Tk_CreateItemType(&TkImageType);
}

/***************************************************************
 ****	    Image format dependent routines                 ****
 ***************************************************************/

int OpenImage(canvasPtr, itemPtr)
     register Tk_Canvas *canvasPtr;	/* Canvas to hold new item. */
     Tk_Item *itemPtr;			/* Record to hold new item; */
{
  register ImageItem *ImagePtr = (ImageItem *) itemPtr;

  ImagePtr->stream = fopen(ImagePtr->path, "r");
  if (ImagePtr->stream == (FILE *)0) {
    Tcl_AppendResult(canvasPtr->interp, "Couldn't open ", ImagePtr->path,
		     " for reading", (char *)0);
    return(TCL_ERROR);
  }

  fseek(ImagePtr->stream, ImagePtr->offset, 0);
  fread((char *)(&ImagePtr->natw), sizeof(int), 1, ImagePtr->stream);
  fread((char *)(&ImagePtr->nath), sizeof(int), 1, ImagePtr->stream);
  return(TCL_OK);
}

int ImageReadIn(canvasPtr, itemPtr, buf, flip)
     register Tk_Canvas *canvasPtr;	/* Canvas to hold new item. */
     Tk_Item *itemPtr;			/* Record to hold new item; */
     float *buf;			/* buffer to hold image data */
     int flip;				/* flip horizontal flag */
{
  register ImageItem *ImagePtr = (ImageItem *) itemPtr;
  
  fseek(ImagePtr->stream, ImagePtr->offset + 2*sizeof(int), 0);
  if (flip) {
    float *src, *psrc;
    int j;

    src = (float *)malloc(ImagePtr->natw * sizeof(float));

    for (j = 0; j < ImagePtr->nath; j++) {
      if (fread((char *)src, sizeof(float),
		ImagePtr->natw, ImagePtr->stream) != ImagePtr->natw) {
	free(src);
	Tcl_AppendResult(canvasPtr->interp, "Error reading image data", (char *)0);
	return(TCL_ERROR);
      }
      psrc = src + ImagePtr->natw-1;
      while (psrc >= src)
	*buf++ = *psrc--;
    }
    free(src);
  } else {
    if (fread((char *)buf, sizeof(float),
	      ImagePtr->natw * ImagePtr->nath,
	      ImagePtr->stream) !=
	ImagePtr->natw * ImagePtr->nath) {
      Tcl_AppendResult(canvasPtr->interp, "Error reading image data", (char *)0);
      return(TCL_ERROR);
    }
  }
  return(TCL_OK);
}  
  
/***************************************************************
 ***************************************************************/

/********  Colorcells  ********/

#define CviNcolors 32 			/* number of colors to use for display */
static int CviGotColors = 0;		/* whether color initialization has been done */
static int *CviPixels;			/* lookup table grey level-> X pixel   */
     
static int InitImageColors(canvasPtr, itemPtr)
     register Tk_Canvas *canvasPtr;	/* Canvas to hold new item. */
     Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
{
  int i;
  unsigned long step, value;
  XColor colors[CviNcolors];

  CviPixels = (int *)malloc(CviNcolors * sizeof(int));
  
  if (!XAllocColorCells
      (Tk_Display(canvasPtr->tkwin),
       DefaultColormap(Tk_Display(canvasPtr->tkwin),
		       DefaultScreen(Tk_Display(canvasPtr->tkwin))),
       True, (unsigned long *)0, 0,
       CviPixels, (unsigned int)CviNcolors)) {
    char buf[32];
    
    sprintf(buf, "%d", CviNcolors);
    Tcl_AppendResult(canvasPtr->interp, "couldn't allocate ",
		     buf, " colorcells", (char *) NULL);
    return(TCL_ERROR);
  }
  
  /**  Store the grey ramp in our colorcells  **/
  step = 65535 / (CviNcolors - 1);
  value = 0;
  for (i = 0; i < CviNcolors; i++) {
    colors[i].red = colors[i].green = colors[i].blue = value;
    colors[i].pixel = CviPixels[i];
    colors[i].flags = DoRed | DoBlue | DoGreen;
    colors[i].pad = 0;
    value += step;
  }
  
  XStoreColors
    (Tk_Display(canvasPtr->tkwin),
     DefaultColormap(Tk_Display(canvasPtr->tkwin),
		     DefaultScreen(Tk_Display(canvasPtr->tkwin))),
     colors, CviNcolors);
  return(TCL_OK);
}

/*
 *--------------------------------------------------------------
 *
 * CreateImage --
 *
 *	This procedure is invoked to create a new Image
 *	item in a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	canvasPtr->interp->result;  in this case itemPtr is
 *	left uninitialized, so it can be safely freed by the
 *	caller.
 *
 * Side effects:
 *	A new Image item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateImage(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas to hold new item. */
    Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing rectangle. */
{
    register ImageItem *ImagePtr = (ImageItem *) itemPtr;
    
    if (argc < 2) {
	Tcl_AppendResult(canvasPtr->interp, "wrong # args:  should be \"",
		Tk_PathName(canvasPtr->tkwin), "\" create ",
		itemPtr->typePtr->name, " x y ?options?",
		(char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Initialize item's record.
     */

    ImagePtr->xc = ImagePtr->yc = 0.0;
    ImagePtr->width = ImagePtr->height = -1.0;
    ImagePtr->angle = 0.0;
    ImagePtr->cosine = 1.0;
    ImagePtr->sine = 0.0;
    ImagePtr->flip = 0;
    ImagePtr->brightness = 0.0;
    ImagePtr->contrast = 0.0;
    strinit(ImagePtr->path, "");
    ImagePtr->stream = (FILE *)0;
    ImagePtr->offset = 0;
    ImagePtr->natw = ImagePtr->nath = -1;
    ImagePtr->xim = (XImage *)0;
    ImagePtr->xregion = None;
    ImagePtr->regw = -1;
    ImagePtr->regh = -1;
    ImagePtr->gc = None;

    /**  If we didn't allocate colorcells yet, do it  **/
    if (! CviGotColors) {
      if (InitImageColors(canvasPtr, itemPtr) == TCL_OK) 
	CviGotColors = 1;
      else {
	DeleteImage(canvasPtr, itemPtr);
	return(TCL_ERROR);
      }
    }

    /*
     * Process the arguments to fill in the item record.
     */

    if ((TkGetCanvasCoord(canvasPtr, argv[0], &ImagePtr->xc) != TCL_OK)
	|| (TkGetCanvasCoord(canvasPtr, argv[1],
			     &ImagePtr->yc) != TCL_OK)) {
      return TCL_ERROR;
    }
    
    if (ConfigureImage(canvasPtr, itemPtr, argc-2, argv+2, 0) != TCL_OK) {
      DeleteImage(canvasPtr, itemPtr);
      return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ImageCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on Image items.  See the user documentation for
 *	details on what it does.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets canvasPtr->interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
ImageCoords(canvasPtr, itemPtr, argc, argv)
    register Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item whose coordinates are to be
					 * read or modified. */
    int argc;				/* Number of coordinates supplied in
					 * argv. */
    char **argv;			/* Array of coordinates: x1, y1,
					 * x2, y2, ... */
{
  register ImageItem *ImagePtr = (ImageItem *) itemPtr;
  char cx[TCL_DOUBLE_SPACE], cy[TCL_DOUBLE_SPACE];
  
  if (argc == 0) {
    Tcl_PrintDouble(canvasPtr->interp, ImagePtr->xc, cx);
    Tcl_PrintDouble(canvasPtr->interp, ImagePtr->yc, cy);
    Tcl_AppendResult(canvasPtr->interp, cx, " ", cy, (char *) NULL);
  } else if ((argc == 2) || (argc == 8)) {
    if ((TkGetCanvasCoord(canvasPtr, argv[0], &ImagePtr->xc) != TCL_OK)
	|| (TkGetCanvasCoord(canvasPtr, argv[1],
			     &ImagePtr->yc) != TCL_OK)) {
      return TCL_ERROR;
    }
    ComputeImageBbox(canvasPtr, ImagePtr);
  } else if ((argc == 1) && (strlen(argv[0]) >= 5) &&
	     !strncmp(argv[0], "-vertices", strlen(argv[0]))) {
    /**  Vertices request  **/
    double sw2, sh2, cw2, ch2;
    
    if (ImagePtr->flip) {
      sw2 = -0.5 * ImagePtr->sine * ImagePtr->width;
      cw2 = -0.5 * ImagePtr->cosine * ImagePtr->width;
    } else {
      sw2 = 0.5 * ImagePtr->sine * ImagePtr->width;
      cw2 = 0.5 * ImagePtr->cosine * ImagePtr->width;
    }

    sh2 = 0.5 * ImagePtr->sine * ImagePtr->height;
    ch2 = 0.5 * ImagePtr->cosine * ImagePtr->height;
    
    Tcl_PrintDouble(canvasPtr->interp,
		    ImagePtr->xc - cw2 - sh2, cx);
    Tcl_PrintDouble(canvasPtr->interp,
		    ImagePtr->yc + sw2 - ch2, cy);
    Tcl_AppendResult(canvasPtr->interp, cx, " ", cy, " ", (char *)0);
    
    Tcl_PrintDouble(canvasPtr->interp,
		    ImagePtr->xc + cw2 - sh2, cx);
    Tcl_PrintDouble(canvasPtr->interp,
		    ImagePtr->yc - sw2 - ch2, cy);
    Tcl_AppendResult(canvasPtr->interp, cx, " ", cy, " ", (char *)0);
    
    Tcl_PrintDouble(canvasPtr->interp,
		    ImagePtr->xc + cw2 + sh2, cx);
    Tcl_PrintDouble(canvasPtr->interp,
		    ImagePtr->yc - sw2 + ch2, cy);
    Tcl_AppendResult(canvasPtr->interp, cx, " ", cy, " ", (char *)0);
    
    Tcl_PrintDouble(canvasPtr->interp,
		    ImagePtr->xc - cw2 + sh2, cx);
    Tcl_PrintDouble(canvasPtr->interp,
		    ImagePtr->yc + sw2 + ch2, cy);
    Tcl_AppendResult(canvasPtr->interp, cx, " ", cy, (char *)0);
    
    return(TCL_OK);
  } else if ((argc == 1) && (strlen(argv[0]) >= 6) &&
	     !strncmp(argv[0], "-config", strlen(argv[0]))) {
    /**  Config request  **/
    Tcl_PrintDouble(canvasPtr->interp,
		    ImagePtr->width, cx);
    Tcl_PrintDouble(canvasPtr->interp,
		    ImagePtr->height, cy);
    Tcl_AppendResult(canvasPtr->interp, "-width ", cx,
		     " -height ", cy, " ", (char *)0);
    
    Tcl_PrintDouble(canvasPtr->interp,
		    ImagePtr->angle, cx);
    Tcl_AppendResult(canvasPtr->interp, "-angle ", cx, " ", (char *)0);

    if (ImagePtr->flip)
      Tcl_AppendResult(canvasPtr->interp, "-flip 1", (char *)0);
    else
      Tcl_AppendResult(canvasPtr->interp, "-flip 0", (char *)0);
    
    return(TCL_OK);
  } else if ((argc == 4) &&
	     (strlen(argv[0]) >= 4) &&
	     !strncmp(argv[0], "-rotate", strlen(argv[0])) ) {
    /* rotate request */
    double xo, yo, x ,y, dalpha, rangle, c, s;
    int no_turn, pos_turn, neg_turn, half_turn;
    int dummy_angle_flag;
    float *src;
    
    if ((Tcl_GetDouble(canvasPtr->interp, argv[1], &xo) != TCL_OK) ||
	(Tcl_GetDouble(canvasPtr->interp, argv[2], &yo) != TCL_OK) ||
	(Tcl_GetDouble(canvasPtr->interp, argv[3], &dalpha) != TCL_OK)) {
      Tcl_AppendResult(canvasPtr->interp,
		       "expected -rotate <double> <double> <double>, got -rotate ",
		       argv[1], argv[2], argv[3], (char *)0);
      return(TCL_ERROR);
    }
      
    ImagePtr->angle += dalpha;
    /**  bring angle in ]-180 , 180]  **/
    while (ImagePtr->angle > 180.0) ImagePtr->angle -= 360.0;
    while (ImagePtr->angle <= -180.0) ImagePtr->angle += 360.0;

    /*  update image center  */
    dalpha *= M_PI / 180.0;
    c = (double)cos(dalpha);
    s = (double)sin(dalpha);
    x = ImagePtr->xc - xo;
    y = ImagePtr->yc - yo;
    ImagePtr->xc = xo + c * x + s * y;
    ImagePtr->yc = yo - s * x + c * y;

    /*  update clip region, cos and sin  */
    no_turn = FNULL(ImagePtr->angle);
    pos_turn = FEQUAL(ImagePtr->angle, 90);
    neg_turn = FEQUAL(ImagePtr->angle, -90);
    half_turn = FEQUAL(fabs(ImagePtr->angle), 180.0);
    dummy_angle_flag =
      no_turn || pos_turn || neg_turn || half_turn;

    if (ImagePtr->xregion != None) {
      XDestroyRegion(ImagePtr->xregion);
      ImagePtr->xregion = None;
    }

    if (dummy_angle_flag) {
      /*  No clip region, just set regw and regh  */
      /*  these are approximate integer dims used */
      /*  for display, and they must be <= to the float dims */
      /*  otherwise fastimgeom routines may try to pick up data */
      /*  outside the bounds of src buffer */
	 
      if (no_turn || half_turn) {
	ImagePtr->regw = (int)ImagePtr->width;
	ImagePtr->regh = (int)ImagePtr->height;
	ImagePtr->cosine = no_turn ? 1.0 : -1.0;
	ImagePtr->sine = 0.0;
      } else {
	ImagePtr->regw = (int)ImagePtr->height;
	ImagePtr->regh = (int)ImagePtr->width;
	ImagePtr->cosine = 0.0;
	ImagePtr->sine = pos_turn ? 1.0 : -1.0 ;
      }
    } else {
      rangle = DEG2RAD(ImagePtr->angle);
      ImagePtr->cosine = (double)cos(rangle);
      ImagePtr->sine = (double)sin(rangle);
      /**  Set clip region  **/
      ComputeImageRegion(canvasPtr, ImagePtr);
    }

    /*  recreate the ximage  */
    if (ImagePtr->xim != (XImage *)0)
      XDestroyImage(ImagePtr->xim); 
    
    ImagePtr->xim =
      XCreateImage(canvasPtr->display,
		   DefaultVisual(canvasPtr->display,
				 DefaultScreen(canvasPtr->display)),
		   8, ZPixmap, 0, 0,
		   (unsigned int)ImagePtr->regw, (unsigned int)ImagePtr->regh,
		   BitmapPad(canvasPtr->display), 0);
    
    ImagePtr->xim->data =
      (char *)malloc(ImagePtr->xim->bytes_per_line * ImagePtr->regh * sizeof(char));
    
    /*  refill ximage  */
    src = (float *)malloc(ImagePtr->natw*ImagePtr->nath*sizeof(float));
    if (ImageReadIn(canvasPtr, itemPtr, src, ImagePtr->flip) != TCL_OK) {
      free(src);
      DeleteImage(canvasPtr, itemPtr);
      return(TCL_ERROR);
    }
    FillImage(canvasPtr, itemPtr, src);
    free(src);
    
    /*  recompute bbox  */
    ComputeImageBbox(canvasPtr, ImagePtr);
    
    return(TCL_OK);
  } else {
    sprintf(canvasPtr->interp->result,
	    "wrong # coordinates:  expected 0 or 2, got %d",
	    argc);
    return TCL_ERROR;
  }
  return TCL_OK;
}

/***************************************************************
 ****	      Given a configured imageptr,                  ****
 ****	       and already read src data,                   ****
 ****	     fills in the ximage structure                  ****
 ***************************************************************/

#define ZEROF 0.000001

static int FillImage(canvasPtr, itemPtr, src)
     Tk_Canvas *canvasPtr;	/* Canvas containing itemPtr. */
     Tk_Item *itemPtr;		/* Image item to reconfigure. */
     float *src;                /* source image data          */
{
  register ImageItem *ImagePtr = (ImageItem *) itemPtr;
  float xfact, yfact, a, b;
  int zoom_flag, turn_flag;

  /**  To bring values between 0.0 and 1.0  **/
  a = 1.0;
  b = 0.0;
  
  /**  Apply brightness  **/

  if ((ImagePtr->brightness < -ZEROF) ||
      (ImagePtr->brightness > ZEROF)) {

    if (ImagePtr->brightness <= -1.0) {
      b -= 1.0;
    } else if (ImagePtr->brightness >= 1.0) {
      b += 1.0;
    } else {
      b += ImagePtr->brightness;
    }
  }

  /**  Apply contrast  **/
  
  if ((ImagePtr->contrast < -ZEROF) ||
      (ImagePtr->contrast > ZEROF)) {
    if (ImagePtr->contrast <= -1.0) {
      a = 0.0;
      b = 0.5;
    } else if (ImagePtr->contrast >= 1.0) {
      /* hack ... */
      a /= ZEROF;
      b = b / ZEROF + 0.5 * (1.0 - ZEROF) / (-ZEROF);
    } else if (ImagePtr->contrast < 0.0) {
      a *= (ImagePtr->contrast + 1);
      b = (ImagePtr->contrast + 1) * b - ImagePtr->contrast/2.0;
    } else if (ImagePtr->contrast > 0.0) {
      a /= (1.0 - ImagePtr->contrast);
      b = b / (1.0 - ImagePtr->contrast) +
	0.5 * ImagePtr->contrast / (ImagePtr->contrast - 1);
    }
  }
  
  /**  To bring values between 0 and CviNcolors **/
  a *= (CviNcolors - ZEROF);
  b *= (CviNcolors - ZEROF);

  /**  zoom parameters  **/
  zoom_flag = (((int)(0.5+ImagePtr->width) != ImagePtr->natw) ||
	       ((int)(0.5+ImagePtr->height) != ImagePtr->nath));
  if (zoom_flag) {
    xfact = ImagePtr->natw / ImagePtr->width;
    yfact = ImagePtr->nath / ImagePtr->height;
  } else {
    xfact = yfact = 1.0;
  }

  /**  rotate parameters  **/
  turn_flag = ! FNULL(ImagePtr->angle);

  /**  Zoom, rotate, and fill in ximage  **/
  ZoomTurn(src, ImagePtr->natw, ImagePtr->nath,
	   (unsigned char *)ImagePtr->xim->data,
	   ImagePtr->regw, ImagePtr->regh,
	   ImagePtr->xim->bytes_per_line,
	   zoom_flag, xfact, yfact,
	   turn_flag, ImagePtr->angle, ImagePtr->cosine, ImagePtr->sine,
	   ImagePtr->xregion,
	   a, b, CviPixels, CviNcolors);
}    

/* ConfigureImage --
 *
 *	This procedure is invoked to configure various aspects
 *	of a Image item, such as its anchor position.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in canvasPtr->interp->result.
 *
 * Side effects:
 *	Configuration information may be set for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int ConfigureImage(canvasPtr, itemPtr, argc, argv, flags)
     Tk_Canvas *canvasPtr;	/* Canvas containing itemPtr. */
     Tk_Item *itemPtr;		/* Image item to reconfigure. */
     int argc;			/* Number of elements in argv.  */
     char **argv;		/* Arguments describing things to configure. */
     int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
  register ImageItem *ImagePtr = (ImageItem *) itemPtr;
  ImageItem savedImagePtr;
  float *src;
  int dim_flag, path_flag, angle_flag, dummy_angle_flag, flip_flag;
  int no_turn, pos_turn, neg_turn, half_turn;
  double rangle;

  /**  Save current ImagePtr state so as to know which actions should be taken  **/
  savedImagePtr.width = ImagePtr->width;
  savedImagePtr.height = ImagePtr->height;
  savedImagePtr.angle = ImagePtr->angle;
  savedImagePtr.flip  = ImagePtr->flip;
  savedImagePtr.brightness = ImagePtr->brightness;
  savedImagePtr.contrast = ImagePtr->contrast;
  strinit(savedImagePtr.path, ImagePtr->path);
  savedImagePtr.offset = ImagePtr->offset;
  
  if (Tk_ConfigureWidget(canvasPtr->interp, canvasPtr->tkwin,
			 configSpecs, argc, argv, (char *) ImagePtr, flags) != TCL_OK) {
    return TCL_ERROR;
  }

  path_flag = strcmp(savedImagePtr.path, ImagePtr->path);
  dim_flag = ((savedImagePtr.width != ImagePtr->width) ||
	      (savedImagePtr.height != ImagePtr->height));
  angle_flag = (((savedImagePtr.angle - ImagePtr->angle) > ZEROF) ||
		((savedImagePtr.angle - ImagePtr->angle) < -ZEROF));

  /**  bring angle in ]-180 , 180]  **/
  while (ImagePtr->angle > 180.0) ImagePtr->angle -= 360.0;
  while (ImagePtr->angle <= -180.0) ImagePtr->angle += 360.0;
  
  /*  angles that require no clipping  */
  no_turn = FNULL(ImagePtr->angle);
  pos_turn = FEQUAL(ImagePtr->angle, 90);
  neg_turn = FEQUAL(ImagePtr->angle, -90);
  half_turn = FEQUAL(fabs(ImagePtr->angle), 180.0);
  flip_flag = (savedImagePtr.flip  != ImagePtr->flip);
  
  dummy_angle_flag =
    no_turn || pos_turn || neg_turn || half_turn;
  
  rangle = DEG2RAD(ImagePtr->angle);
    
  /**  if path changed, reopen the image  **/

  if (path_flag)
    if (OpenImage(canvasPtr, itemPtr) != TCL_OK)
      return(TCL_ERROR);

  /**  set width and height if necessary  **/

  if ((ImagePtr->width < 0.0) && (ImagePtr->height < 0.0)) {
    /*  No requested dimensions  */
    ImagePtr->width = (double)ImagePtr->natw;
    ImagePtr->height = (double)ImagePtr->nath;
  } else if (ImagePtr->width < 0.0) {
    /*  requested height  */
    ImagePtr->width =
      (ImagePtr->height*ImagePtr->natw) / ImagePtr->nath;
  } else if (ImagePtr->height < 0) {
    /*  requested height  */
    ImagePtr->height =
      (ImagePtr->width*ImagePtr->nath) / ImagePtr->natw;
  }

  /**  Update angle and region if necessary  **/

  if (ImagePtr->regw < 0)
    ImagePtr->regw = (int)ImagePtr->width;
  if (ImagePtr->regh < 0)
    ImagePtr->regh = (int)ImagePtr->height;
  
  if (angle_flag || dim_flag) {
    if (ImagePtr->xregion != None) {
      XDestroyRegion(ImagePtr->xregion);
      ImagePtr->xregion = None;
    }

    if (dummy_angle_flag) {
      /*  No clip region, just set regw and regh  */
      /*  these are approximate integer dims used */
      /*  for display, and they must be <= to the float dims */
      /*  otherwise fastimgeom routines may try to pick up data */
      /*  outside the bounds of src buffer */
	 
      if (no_turn || half_turn) {
	ImagePtr->regw = (int)ImagePtr->width;
	ImagePtr->regh = (int)ImagePtr->height;
	ImagePtr->cosine = no_turn ? 1.0 : -1.0;
	ImagePtr->sine = 0.0;
      } else {
	ImagePtr->regw = (int)ImagePtr->height;
	ImagePtr->regh = (int)ImagePtr->width;
	ImagePtr->cosine = 0.0;
	ImagePtr->sine = pos_turn ? 1.0 : -1.0 ;
      }
    } else {
      ImagePtr->cosine = cos(rangle);
      ImagePtr->sine = sin(rangle);
      /**  Set clip region  **/
      ComputeImageRegion(canvasPtr, ImagePtr);
    }
  }
    
  /**  Update the GC **/
  /*  The clip region is set in ComputeImageBbox  */
  /*  (because of coords calls that don't come here) */
  
  if (ImagePtr->gc == None) {
    ImagePtr->gc = XCreateGC(canvasPtr->display,
			     Tk_WindowId(canvasPtr->tkwin),
			     0L, (XGCValues *)0);
  }
  
  /**  if width, height or angle changed, recreate the ximage  **/
  if (dim_flag || angle_flag || (ImagePtr->xim == (XImage *)0)) {
    if (ImagePtr->xim != (XImage *)0)
      XDestroyImage(ImagePtr->xim);

    ImagePtr->xim =
      XCreateImage(canvasPtr->display,
		   DefaultVisual(canvasPtr->display,
				 DefaultScreen(canvasPtr->display)),
		   8, ZPixmap, 0, 0,
		   (unsigned int)ImagePtr->regw, (unsigned int)ImagePtr->regh,
		   BitmapPad(canvasPtr->display), 0);
    
    ImagePtr->xim->data =
      (char *)malloc(ImagePtr->xim->bytes_per_line * ImagePtr->regh * sizeof(char));
  }
  
  /**  if flip, path, width, height, angle, brightness or contrast changed, refill ximage  **/
  if (path_flag || flip_flag || dim_flag || angle_flag ||
      (savedImagePtr.brightness != ImagePtr->brightness) ||
      (savedImagePtr.contrast != ImagePtr->contrast)) {

    /**  Read in image **/
    src = (float *)malloc(ImagePtr->natw*ImagePtr->nath*sizeof(float));
    
    if (ImageReadIn(canvasPtr, itemPtr, src, ImagePtr->flip) != TCL_OK) {
      free(src);
      DeleteImage(canvasPtr, itemPtr);
      return(TCL_ERROR);
    }

    /** Fill in the ximage **/
    
    FillImage(canvasPtr, itemPtr, src);
  
    free(src);
  }
  
  ComputeImageBbox(canvasPtr, ImagePtr);

  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeleteImage --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a Image item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteImage(canvasPtr, itemPtr)
     Tk_Canvas *canvasPtr;		/* Info about overall canvas widget. */
     Tk_Item *itemPtr;			/* Item that is being deleted. */
{
  register ImageItem *ImagePtr = (ImageItem *) itemPtr;

  if (ImagePtr->header.x1 <= canvasPtr->redrawX1) {
    canvasPtr->redrawX1 = ImagePtr->header.x1;
  }
  if (ImagePtr->header.y1 <= canvasPtr->redrawY1) {
    canvasPtr->redrawY1 = ImagePtr->header.y1;
  }
  if (ImagePtr->header.x2 >= canvasPtr->redrawX2) {
    canvasPtr->redrawX2 = ImagePtr->header.x2;
  }
  if (ImagePtr->header.y2 >= canvasPtr->redrawY2) {
    canvasPtr->redrawY2 = ImagePtr->header.y2;
  }

  if (ImagePtr->stream != (FILE *)0)
    fclose(ImagePtr->stream);
  
  if (ImagePtr->xim != (XImage *)0)
    XDestroyImage(ImagePtr->xim);

  if (ImagePtr->xregion != None)
    XDestroyRegion(ImagePtr->xregion);

  if (ImagePtr->gc != None)
    XFreeGC(canvasPtr->display, ImagePtr->gc);
}


/*
 *--------------------------------------------------------------
 *
 * ComputeImagRegion --
 * Given image center, width, height, and angle,
 * computes interger approximates for display : region, regw and regh
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields xregion, regw, and regh are updated in the header
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

 
ComputeImageRegion(canvasPtr, ImagePtr)
     Tk_Canvas *canvasPtr;		/* Canvas that contains item. */
     register ImageItem *ImagePtr;	/* Item whose bbox is to be
					 * recomputed. */
{
  double cw2, sw2, ch2, sh2;
  XPoint points[4];
  XRectangle rect;
  
  cw2 = 0.5 * ImagePtr->width;
  sw2 = - ImagePtr->sine * cw2; /* screen y != math y */
  cw2 *= ImagePtr->cosine;

  ch2 = 0.5 * ImagePtr->height;
  sh2 = - ImagePtr->sine * ch2; /* screen y != math y */
  ch2 *= ImagePtr->cosine;

  /*  use center of image as center of rotation  */
  
  points[0].x = (short)(0.5 - cw2 + sh2);
  points[0].y = (short)(0.5 - sw2 - ch2);
  points[1].x = (short)(0.5 + cw2 + sh2);
  points[1].y = (short)(0.5 + sw2 - ch2);
  points[2].x = (short)(0.5 + cw2 - sh2);
  points[2].y = (short)(0.5 + sw2 + ch2);
  points[3].x = (short)(0.5 - cw2 - sh2);
  points[3].y = (short)(0.5 - sw2 + ch2);
  ImagePtr->xregion =
    XPolygonRegion(points, 4, EvenOddRule);

  XClipBox(ImagePtr->xregion, &rect);
  ImagePtr->regw = rect.width;
  ImagePtr->regh = rect.height;
}

/*
 *--------------------------------------------------------------
 *
 * ComputeImageBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a Image item.
 *	This procedure is where the child Image's placement is
 *	computed.
 *      Use integers regw and regh since bboxes are used for display
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static void
ComputeImageBbox(canvasPtr, ImagePtr)
    Tk_Canvas *canvasPtr;		/* Canvas that contains item. */
    register ImageItem *ImagePtr;	/* Item whose bbox is to be
					 * recomputed. */
{
  ImagePtr->header.x1 = floor(ImagePtr->xc - 0.5*ImagePtr->regw)-1;
  ImagePtr->header.y1 = floor(ImagePtr->yc - 0.5*ImagePtr->regh)-1;

  if (ImagePtr->stream == (FILE *)0) {
    ImagePtr->header.x2 = ImagePtr->header.x1;
    ImagePtr->header.y2 = ImagePtr->header.y1;
    return;
  }

  ImagePtr->header.x2 = ImagePtr->header.x1 + ImagePtr->regw + 2;
  ImagePtr->header.y2 = ImagePtr->header.y1 + ImagePtr->regh + 2;
}

/*
 *--------------------------------------------------------------
 *
 * DisplayImage --
 *
 *	This procedure is invoked to draw a Image item in a given
 *	drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvasPtr.
 *
 *--------------------------------------------------------------
 */

static void
DisplayImage(canvasPtr, itemPtr, drawable)
    register Tk_Canvas *canvasPtr;	/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
{
  register ImageItem *ImagePtr = (ImageItem *) itemPtr;
  XRectangle rect;
  
  /*  Translate the region  */

  if (ImagePtr->xregion != None) {
    XClipBox(ImagePtr->xregion, &rect);
    XOffsetRegion(ImagePtr->xregion,
		  ImagePtr->header.x1+1 - canvasPtr->drawableXOrigin - rect.x,
		  ImagePtr->header.y1+1 - canvasPtr->drawableYOrigin - rect.y);
  }

  /*  update the clip_mask in the GC  */
  /* done here to allow future use of shared GCs */
  
  if (ImagePtr->xregion == None)
    XSetClipMask(canvasPtr->display, ImagePtr->gc, None);
  else
    XSetRegion(canvasPtr->display, ImagePtr->gc, ImagePtr->xregion);

  XPutImage(canvasPtr->display, drawable,
	    ImagePtr->gc,
	    ImagePtr->xim, 0, 0,
	    ImagePtr->header.x1+1 - canvasPtr->drawableXOrigin,
	    ImagePtr->header.y1+1 - canvasPtr->drawableYOrigin,
	    (unsigned int)ImagePtr->regw,
	    (unsigned int)ImagePtr->regh);
}

/*
 *--------------------------------------------------------------
 *
 * ImageToPoint --
 *
 *	Computes the distance from a given point to a given
 *	rectangle, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are coordPtr[0] and coordPtr[1] is inside the Image.  If the
 *	point isn't inside the Image then the return value is the
 *	distance from the point to the Image.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static double
ImageToPoint(canvasPtr, itemPtr, coordPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *coordPtr;		/* Pointer to x and y coordinates. */
{
    register ImageItem *ImagePtr = (ImageItem *) itemPtr;
    double rx, ry;
    double x1, x2, y1, y2, xDiff, yDiff;

    /**  Horizontal image rectangle  **/
    x1 = ImagePtr->xc - 0.5*ImagePtr->width;
    y1 = ImagePtr->yc - 0.5*ImagePtr->height;
    x2 = x1 + ImagePtr->width;
    y2 = y1 + ImagePtr->height;

    /**  Rotate point of -angle around image center  **/
    /**  then compute distance to the horizontal image rectangle  **/
    /** screen y != math y **/
    
    rx = ImagePtr->xc + ImagePtr->cosine*(coordPtr[0] - ImagePtr->xc) -
      ImagePtr->sine*(coordPtr[1] - ImagePtr->yc);
    ry = ImagePtr->yc + ImagePtr->sine*(coordPtr[0] - ImagePtr->xc) +
      ImagePtr->cosine*(coordPtr[1] - ImagePtr->yc);

    /*
     * Point is outside rectangle.
     */

    if (rx < x1) {
	xDiff = x1 - rx;
    } else if (rx > x2)  {
	xDiff = rx - x2;
    } else {
	xDiff = 0;
    }

    if (ry < y1) {
	yDiff = y1 - ry;
    } else if (ry > y2)  {
	yDiff = ry - y2;
    } else {
	yDiff = 0;
    }

    return hypot(xDiff, yDiff);
  }

/*
 *--------------------------------------------------------------
 *
 * ImageToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangle.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the area
 *	given by ImagePtr, 0 if it overlaps, and 1 if it is entirely
 *	inside the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
ImageToArea(canvasPtr, itemPtr, areaPtr)
    Tk_Canvas *canvasPtr;	/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against rectangle. */
    double *areaPtr;		/* Pointer to array of four coordinates
				 * (x1, y1, x2, y2) describing rectangular
				 * area.  */
{
  register ImageItem *ImagePtr = (ImageItem *) itemPtr;

  /*  In any case, image entirely in rect <==> bbox entirely in rect  */
  if ((areaPtr[0] <= ImagePtr->header.x1)
      && (areaPtr[1] <= ImagePtr->header.y1)
      && (areaPtr[2] >= ImagePtr->header.x2)
      && (areaPtr[3] >= ImagePtr->header.y2)) {
    return 1;
  }
  
  if (ImagePtr->xregion == None) {
    /** horizontal rectangle -> use bbox **/
    if ((areaPtr[2] <= ImagePtr->header.x1)
	|| (areaPtr[0] >= ImagePtr->header.x2)
	|| (areaPtr[3] <= ImagePtr->header.y1)
	|| (areaPtr[1] >= ImagePtr->header.y2)) {
      return -1;
    }
 
    return 0;
  } else {
    /**  oblique rectangle => use xregion  **/
    int status;
    XRectangle rect;

    /**  Translate region  **/
    XClipBox(ImagePtr->xregion, &rect);

    XOffsetRegion(ImagePtr->xregion,
		  ImagePtr->header.x1+1 - canvasPtr->drawableXOrigin - rect.x,
		  ImagePtr->header.y1+1 - canvasPtr->drawableYOrigin - rect.y);
    
    status = XRectInRegion(ImagePtr->xregion,
			   (int)(0.5+areaPtr[0]),
			   (int)(0.5+areaPtr[1]),
			   (unsigned int)(areaPtr[2]-areaPtr[0]),
			   (unsigned int)( areaPtr[3]-areaPtr[1]));
    switch (status) {
    case RectangleOut:  return -1;
    case RectangleIn:   return 0;
    case RectanglePart: return 0;
    }
  }
  /*  for compiler warnings  */
  return(0);
}

/*
 *--------------------------------------------------------------
 *
 * ScaleImage --
 *
 *	This procedure is invoked to rescale a rectangle or oval
 *	item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The rectangle or oval referred to by itemPtr is rescaled
 *	so that the following transformation is applied to all
 *	point coordinates:
 *		x' = originX + scaleX*(x-originX)
 *		y' = originY + scaleY*(y-originY)
 *
 *--------------------------------------------------------------
 */

static void ScaleImage(canvasPtr, itemPtr, originX, originY, scaleX, scaleY)
     Tk_Canvas *canvasPtr;		/* Canvas containing rectangle. */
     Tk_Item *itemPtr;			/* Rectangle to be scaled. */
     double originX, originY;		/* Origin about which to scale rect. */
     double scaleX;			/* Amount to scale in X direction. */
     double scaleY;			/* Amount to scale in Y direction. */
{
  register ImageItem *ImagePtr = (ImageItem *) itemPtr;
  float *src;
  double angle;

  /*  new origin  */
  ImagePtr->xc = originX + scaleX*(ImagePtr->xc - originX);
  ImagePtr->yc = originY + scaleY*(ImagePtr->yc - originY);
  
  /*  new dimensions  */
  angle = ImagePtr->angle;
  if ((angle < -135.0) ||
      ((angle > -45.0) && (angle < 45.0)) ||
      (angle > 135.0)) {
    ImagePtr->width *= fabs(scaleX);
    ImagePtr->height *= fabs(scaleY);
    if (scaleX < 0.0)
      ImagePtr->flip = 1 - ImagePtr->flip;
    if (scaleY < 0.0) {
      ImagePtr->flip = 1 - ImagePtr->flip;
      if (angle <= 0.0)
	ImagePtr->angle += 180.0;
      else
	ImagePtr->angle -= 180.0;
    }
  } else {
    ImagePtr->width *= fabs(scaleY);
    ImagePtr->height *= fabs(scaleX);
    if (scaleY < 0.0)
      ImagePtr->flip = 1 - ImagePtr->flip;
    if (scaleX < 0.0) {
      ImagePtr->flip = 1 - ImagePtr->flip;
      if (angle <= 0.0)
	ImagePtr->angle += 180.0;
      else
	ImagePtr->angle -= 180.0;
    }
  }

  if (ImagePtr->xregion != None) {
    XDestroyRegion(ImagePtr->xregion);
    ComputeImageRegion(canvasPtr, ImagePtr);
  } else {
    if (FNULL(ImagePtr->angle) ||
	FEQUAL(fabs(ImagePtr->angle), 180.0)) {
      ImagePtr->regw = F2I(ImagePtr->width);
      ImagePtr->regh = F2I(ImagePtr->height);
    } else {
      ImagePtr->regw = F2I(ImagePtr->height);
      ImagePtr->regh = F2I(ImagePtr->width);
    }
  }
  
  /**  Allocate data buffer  **/
  src = (float *)malloc(ImagePtr->natw*ImagePtr->nath*sizeof(float));
			
  /**  Read in image **/
  if (ImageReadIn(canvasPtr, itemPtr, src, ImagePtr->flip) != TCL_OK) {
    DeleteImage(canvasPtr, itemPtr);
    return;
  }
  
  /**  Recreate ximage  **/
  XDestroyImage(ImagePtr->xim);
  ImagePtr->xim =
    XCreateImage(canvasPtr->display,
		 DefaultVisual(canvasPtr->display,
			       DefaultScreen(canvasPtr->display)),
		 8, ZPixmap, 0, 0,
		 ImagePtr->regw, ImagePtr->regh,
		 BitmapPad(Tk_Display(canvasPtr->tkwin)), 0);
  
  /**  Allocate space for ximage data  **/
  ImagePtr->xim->data =
    (char *)malloc(ImagePtr->xim->bytes_per_line *
		   ImagePtr->regh * sizeof(char));
  
  /** Fill in the ximage **/
  FillImage(canvasPtr, itemPtr, src);
  
  free(src);

  /*  erase old bbox area */
  XClearArea(Tk_Display(canvasPtr->tkwin),
	     Tk_WindowId(canvasPtr->tkwin),
	     ImagePtr->header.x1, ImagePtr->header.y1,
	     ImagePtr->header.x2 - ImagePtr->header.x1,
	     ImagePtr->header.y2 - ImagePtr->header.y1,
	     True);

  if (ImagePtr->header.x1 <= canvasPtr->redrawX1) {
    canvasPtr->redrawX1 = ImagePtr->header.x1;
  }
  if (ImagePtr->header.y1 <= canvasPtr->redrawY1) {
    canvasPtr->redrawY1 = ImagePtr->header.y1;
  }
  if (ImagePtr->header.x2 >= canvasPtr->redrawX2) {
    canvasPtr->redrawX2 = ImagePtr->header.x2;
  }
  if (ImagePtr->header.y2 >= canvasPtr->redrawY2) {
    canvasPtr->redrawY2 = ImagePtr->header.y2;
  }
  
  ComputeImageBbox(canvasPtr, ImagePtr);
}


/*
 *--------------------------------------------------------------
 *
 * TranslateImage --
 *
 *	This procedure is called to move an image by a
 *	given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The position of the rectangle or oval is offset by
 *	(xDelta, yDelta), and the bounding box is updated in the
 *	generic part of the item structure.
 *
 *--------------------------------------------------------------
 */

static void
TranslateImage(canvasPtr, itemPtr, deltaX, deltaY)
    Tk_Canvas *canvasPtr;		/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item that is being moved. */
    double deltaX, deltaY;		/* Amount by which item is to be
					 * moved. */
{
    register ImageItem *ImagePtr = (ImageItem *) itemPtr;

    ImagePtr->xc += deltaX;
    ImagePtr->yc += deltaY;
    ComputeImageBbox(canvasPtr, ImagePtr);
}

/*
 *--------------------------------------------------------------
 *
 * ImageToPostscript --
 *
 *	This procedure is called to generate Postscript for
 *	Image items.
 *
 * Results:
 *	The return value is a standard Tcl result.  If an error
 *	occurs in generating Postscript then an error message is
 *	left in canvasPtr->interp->result, replacing whatever used
 *	to be there.  If no error occurs, then Postscript for the
 *	item is appended to the result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
ImageToPostscript(canvasPtr, itemPtr, psInfoPtr)
    Tk_Canvas *canvasPtr;		/* Information about overall canvas. */
    Tk_Item *itemPtr;			/* Item for which Postscript is
					 * wanted. */
    Tk_PostscriptInfo *psInfoPtr;	/* Information about the Postscript;
					 * must be passed back to Postscript
					 * utility procedures. */
{
  register ImageItem *ImagePtr = (ImageItem *) itemPtr;
  double x, y, rx, ry;
  unsigned int width, height;
  char buffer[256], *pbuf, *bufend;
  float *src, *psrc, *lastPixel;
  float a, b;
  int val;

  if (ImagePtr->stream == (FILE *)0) {
    return TCL_OK;
  }

  /**  Allocate data buffer  **/
  src = (float *)malloc(ImagePtr->natw*ImagePtr->nath*sizeof(float));
  
  /**  Read in image **/
  if (ImageReadIn(canvasPtr, itemPtr, src, ImagePtr->flip) != TCL_OK) {
    return(TCL_ERROR);
  }

  /* Compute mapping coefficients  from [0.0 1.0] to [0 256],
   *   taking brightness and contrast into account
   */
  
  /**  To bring values between 0.0 and 1.0  **/
  a = 1.0;
  b = 0.0;
  
  /**  Apply brightness  **/

  if ((ImagePtr->brightness < -ZEROF) ||
      (ImagePtr->brightness > ZEROF)) {

    if (ImagePtr->brightness <= -1.0) {
      b -= 1.0;
    } else if (ImagePtr->brightness >= 1.0) {
      b += 1.0;
    } else {
      b += ImagePtr->brightness;
    }
  }

  /**  Apply contrast  **/
  
  if ((ImagePtr->contrast < -ZEROF) ||
      (ImagePtr->contrast > ZEROF)) {
    if (ImagePtr->contrast <= -1.0) {
      a = 0.0;
      b = 0.5;
    } else if (ImagePtr->contrast >= 1.0) {
      /* hack ... */
      a /= ZEROF;
      b = b / ZEROF + 0.5 * (1.0 - ZEROF) / (-ZEROF);
    } else if (ImagePtr->contrast < 0.0) {
      a *= (ImagePtr->contrast + 1);
      b = (ImagePtr->contrast + 1) * b - ImagePtr->contrast/2.0;
    } else if (ImagePtr->contrast > 0.0) {
      a /= (1.0 - ImagePtr->contrast);
      b = b / (1.0 - ImagePtr->contrast) +
	0.5 * ImagePtr->contrast / (ImagePtr->contrast - 1);
    }
  }
  
  /**  To bring values between 0 and 256 **/
    
  a *= 256 - ZEROF;
  b *= 256 - ZEROF;

  /*
   * Compute the ps coordinates of the lower-left corner of the rotated Image
   */

  x = - 0.5*ImagePtr->width;
  y = 0.5*ImagePtr->height;

  rx = ImagePtr->xc +
    ImagePtr->cosine * x + ImagePtr->sine * y;
  ry = TkCanvPsY(psInfoPtr, ImagePtr->yc - ImagePtr->sine * x +
		 ImagePtr->cosine * y);  
  
  /*  Write transformation  */
  sprintf(buffer, "%.15g %.15g translate\n%.15g rotate\n%.15g %.15g scale\n",
	  rx, ry, ImagePtr->angle, (double)ImagePtr->width, (double)ImagePtr->height);
  Tcl_AppendResult(canvasPtr->interp, buffer, (char *) NULL);
  
  /*
   * write image commands
   */

  width = ImagePtr->natw;
  height = ImagePtr->nath;
  
  sprintf(buffer, "/picstr %d string def\n%d %d 8 [%d 0 0 %d 0 %d]\n",
	  width, width, height,
	  width, -height, height);
  
  strcat(buffer, "{ currentfile picstr readhexstring pop }\n");
  strcat(buffer, "image\n");    
  Tcl_AppendResult(canvasPtr->interp, buffer, (char *) NULL);
  
  /*
   * write the data by slices of 80 characters
   */
  
  psrc = src;
  lastPixel = src + width * height;
  pbuf = buffer;
  bufend = buffer + 80;

  while (psrc < lastPixel) {
    val = (int)(a * (*psrc++) + b);
    if (val < 0)
      sprintf(pbuf, "%02X", 0);
    else if (val > 255)
      sprintf(pbuf, "%02X", 255);
    else
      sprintf(pbuf, "%02X", (unsigned char)val);
    
    pbuf += 2;
    if (pbuf >= bufend) {
      sprintf(pbuf, "\n");
      Tcl_AppendResult(canvasPtr->interp,
		       buffer, (char *)0);
      pbuf = buffer;
    }
  }
  /*  write last slice  */
  sprintf(pbuf, "\n");
  Tcl_AppendResult(canvasPtr->interp,
		   buffer, (char *)0);

  /**  Cleanup  **/
  free(src);

  return TCL_OK;
}

/**
 **  
 **  Dimension option management
 **  
 **/

static int
ParseImageDim(clientData, interp, tkwin, value, recordPtr, offset)
    ClientData clientData;      /* Not used. */
    Tcl_Interp *interp;         /* Used for error reporting. */
    Tk_Window tkwin;            /* Not used. */
    char *value;                /* Textual specification of arrow shape. */
    char *recordPtr;            /* Pointer to item record in which to
                                 * store arrow information. */
    int offset;                 /* Offset of shape information in widget
                                 * record. */
{
  if (*value == '\0') {
    *(double *)(recordPtr + offset) = -1.0;
    return(TCL_OK);
  }

  return(Tcl_GetDouble(interp, value,
		       (double *)(recordPtr + offset)));
}

static char *
PrintImageDim(clientData, tkwin, recordPtr, offset, freeProcPtr)
    ClientData clientData;      /* Not used. */
    Tk_Window tkwin;            /* Window associated with dlinePtr's widget. */
    char *recordPtr;            /* Pointer to item record containing current
                                 * shape information. */
    int offset;                 /* Offset of arrow information in record. */
    Tcl_FreeProc **freeProcPtr; /* Store address of procedure to call to
                                 * free string here. */
{
  char buffer[TCL_DOUBLE_SPACE];

  Tcl_PrintDouble(interp,
		  *(double *)(recordPtr+offset), buffer);
  return(buffer);
}
