##
## Editor.tcl
##
## This file contains the definition of an tkined editor. The
## procs in this file are mostly callbacks that are called from
## an editor object, which can be found in editor.c.
##
## Copyright (c) 1993, 1994
##
## J. Schoenwaelder
## TU Braunschweig, Germany
## Institute for Operating Systems and Computer Networks
##
## Permission to use, copy, modify, and distribute this
## software and its documentation for any purpose and without
## fee is hereby granted, provided that this copyright
## notice appears in all copies.  The University of Braunschweig
## makes no representations about the suitability of this
## software for any purpose.  It is provided "as is" without
## express or implied warranty.
##


##
## Set up a new editor. Create all the menus and the canvas with
## the scollbars.
##

proc Editor::create { editor } { 
    $editor toplevel ".$editor"
}

proc Editor::delete { editor } { 
    destroy [$editor toplevel]
}

proc Editor::graph { editor } {
    set w [$editor toplevel]
    set top $w.diagram
    if {! [winfo exists $top]} { 
	Diagram::create $editor
    }
    return $top
}

proc Editor::toplevel { editor } {

    global tkined_version

    set w [$editor toplevel]

    catch {destroy $w}
    toplevel $w -class tkined

    wm withdraw $w
    wm iconbitmap $w icon
    wm iconname $w "tkined"
    wm minsize $w 500 300

    set width [$editor attribute width]
    if {$width == ""} { set width 600 }
    set height [$editor attribute height]
    if {$height == ""} { set height 400 }

    # set up the canvas for the graphic
    canvas $w.canvas -borderwidth 1 -relief raised \
	-width $width -height $height
    
    # set up the tool box
    frame $w.tools -borderwidth 1 -relief raised
    
    button $w.tools.select -text "Select" -relief flat \
	-command "Tool::Select $editor"
    button $w.tools.move -text "Move" -relief flat \
	-command "Tool::Move $editor"
    button $w.tools.alter -text "Alter" -relief flat \
	-command "Tool::Alter $editor"
    button $w.tools.text -text "Text" -relief flat \
	-command "Tool::Text $editor"
    button $w.tools.node -bitmap machine -relief flat \
	-command "Tool::Node $editor"
    button $w.tools.network -text Network -relief flat \
	-command "Tool::Network $editor"
    button $w.tools.link -text Link -relief flat \
	-command "Tool::Link $editor"
    button $w.tools.group -bitmap group -relief flat  \
	-command "Tool::Group $editor"
    button $w.tools.reference -bitmap reference -relief flat  \
	-command "Tool::Reference $editor"
    Tool::Select $editor

    label $w.tools.dummy -text "tkined $tkined_version"
    pack $w.tools.dummy -side top -pady 1 -ipadx 5
    pack $w.tools.select    -fill x
    pack $w.tools.move      -fill x
    pack $w.tools.alter     -fill x
    pack $w.tools.text      -fill x
    pack $w.tools.node      -fill x
    pack $w.tools.network   -fill x
    pack $w.tools.link      -fill x
    pack $w.tools.group     -fill x
    pack $w.tools.reference -fill x

    # set up the menu bar
    frame $w.menu -borderwidth 1 -relief raised
    
    # Add some scrollbars to the canvas. We put put the vscrollbar 
    # into a frame with an extra label to get rid of the overlapping
    # scrollbars in the lower right corner. Ugly, but it works.

    scrollbar $w.hscroll -orient horiz -relief sunken \
	-command "$w.canvas xview"
    frame $w.vscroll
    scrollbar $w.vscroll.sc -relief sunken -command "$w.canvas yview"
    button $w.vscroll.l -bitmap corner -relief sunken -command \
	"tkined_useless_button $w.canvas"
    pack  $w.vscroll.l -side bottom
    pack  $w.vscroll.sc -side top -fill y -expand yes

    # set up the application
    pack $w.tools   -side left -fill y
    pack $w.menu    -side top -fill x
    pack $w.vscroll -side right -fill y
    pack $w.hscroll -side bottom -fill x
    pack $w.canvas  -side left -fill both -expand true

    $w.canvas config -xscroll "$w.hscroll set" -yscroll "$w.vscroll.sc set"

    # set up the file menu
    menubutton $w.menu.file -text "File" -menu $w.menu.file.m
    menu $w.menu.file.m
    $w.menu.file.m add command -label "New" \
	-accelerator "  ^N" \
	-command "$w.menu.file configure -state disabled; update idletasks; \
                  Command::New $editor; \
                  $w.menu.file configure -state normal"
    bind $w <Control-n> "$w.menu.file.m invoke 0"
    $w.menu.file.m add command -label "Open" \
	-accelerator "  ^O" \
	-command "$w.menu.file configure -state disabled; update idletasks; \
                  Command::Open $editor; \
                  $w.menu.file configure -state normal"
    bind $w <Control-o> "$w.menu.file.m invoke 1"
    $w.menu.file.m add command -label "Merge" \
	-accelerator "  ^M" \
	-command "$w.menu.file configure -state disabled; update idletasks; \
                  Command::Merge $editor; \
                  $w.menu.file configure -state normal"
    bind $w <Control-m> "$w.menu.file.m invoke 2"
    $w.menu.file.m add command -label "Save" \
	-accelerator "  ^S" \
	-command "$w.menu.file configure -state disabled; update idletasks; \
                  Command::Save $editor; \
                  $w.menu.file configure -state normal"
    bind $w <Control-s> "$w.menu.file.m invoke 3"
    $w.menu.file.m add command -label "Save As" \
	-accelerator "  ^A" \
	-command "$w.menu.file configure -state disabled; update idletasks; \
                  Command::SaveAs $editor; \
                  $w.menu.file configure -state normal"
    bind $w <Control-a> "$w.menu.file.m invoke 4"
    $w.menu.file.m add sep
    $w.menu.file.m add command -label "PostScript" \
	-accelerator "  ^P" \
	-command "$w.menu.file configure -state disabled; update idletasks; \
                  Command::PostScript $editor; \
                  $w.menu.file configure -state normal"
    bind $w <Control-p> "$w.menu.file.m invoke 6"
    $w.menu.file.m add command -label "Print" \
	-accelerator "  ^R" \
	-command "$w.menu.file configure -state disabled; update idletasks; \
                  Command::Print $editor; \
                  $w.menu.file configure -state normal"
    bind $w <Control-r> "$w.menu.file.m invoke 7"
    $w.menu.file.m add sep
    $w.menu.file.m add command -label "Import" \
	-accelerator "  ^I" \
	-command "$w.menu.file configure -state disabled; update idletasks; \
                  Command::Import $editor; \
                  $w.menu.file configure -state normal"
    bind $w <Control-i> "$w.menu.file.m invoke 9"
    $w.menu.file.m add sep
    $w.menu.file.m add command -label "History" \
	-accelerator "  ^H" \
	-command "Command::History $editor"
    bind $w <Control-h> "$w.menu.file.m invoke 11"
    $w.menu.file.m add sep
    $w.menu.file.m add command -label "Quit" \
	-accelerator "  ^Q" \
	-command "Command::Quit $editor"
    bind $w <Control-q> "$w.menu.file.m invoke 13"
    
    # set up the edit menu
    menubutton $w.menu.edit -text "Edit" \
	-menu $w.menu.edit.m
    menu $w.menu.edit.m
    $w.menu.edit.m add command -label "Scale Factor" \
	-accelerator "  F" \
	-command "$w.menu.edit configure -state disabled; update idletasks; \
                  Command::Scale $editor; \
                  $w.menu.edit configure -state normal"
    bind $w <F> "$w.menu.edit.m invoke 0"
    $w.menu.edit.m add sep
    $w.menu.edit.m add command -label "Grid Spacing" \
        -accelerator "  s" \
        -command "$w.menu.edit configure -state disabled; \
                  update idletasks; \
                  Command::GridSpacing $editor; \
                  $w.menu.edit configure -state normal"
    bind $w <s> "$w.menu.edit.m invoke 2"
    $w.menu.edit.m add command -label "Align to Grid" \
        -accelerator "  t" \
        -command "$w.menu.edit configure -state disabled; \
                  update idletasks; \
                  Command::Grid $editor; \
                  $w.menu.edit configure -state normal"
    bind $w <t> "$w.menu.edit.m invoke 3"
    $w.menu.edit.m add sep
    $w.menu.edit.m add command -label "Undo" \
	-accelerator "  U" \
	-command "$w.menu.edit configure -state disabled; update idletasks; \
                  Command::Undo $editor; \
                  $w.menu.edit configure -state normal"
    bind $w <U> "$w.menu.edit.m invoke 5"
    $w.menu.edit.m add command -label "Redo" \
	-accelerator "  R" \
	-command "$w.menu.edit configure -state disabled; update idletasks; \
                  Command::Redo $editor; \
                  $w.menu.edit configure -state normal" 
    bind $w <R> "$w.menu.edit.m invoke 6"
    $w.menu.edit.m add sep
    $w.menu.edit.m add command -label "Delete" \
	-accelerator " ^D" \
	-command "$w.menu.edit configure -state disabled; update idletasks; \
                  Command::Delete $editor; \
                  $w.menu.edit configure -state normal"
    bind $w <Control-d> "$w.menu.edit.m invoke 8"
    $w.menu.edit.m add command -label "Cut" \
	-accelerator "  x" \
	-command "$w.menu.edit configure -state disabled; update idletasks; \
                  $editor cut; \
                  $w.menu.edit configure -state normal"
    bind $w <x> "$w.menu.edit.m invoke 9"
    $w.menu.edit.m add command -label "Copy" \
	-accelerator "  w" \
	-command "$w.menu.edit configure -state disabled; update idletasks; \
                  $editor copy; \
                  $w.menu.edit configure -state normal"
    bind $w <w> "$w.menu.edit.m invoke 10"
    $w.menu.edit.m add command -label "Paste" \
	-accelerator "  v" \
	-command "$w.menu.edit configure -state disabled; update idletasks; \
                  $editor paste; \
                  $w.menu.edit configure -state normal"
    bind $w <v> "$w.menu.edit.m invoke 11"

    # set up the select menu
    menubutton $w.menu.select -text "Select" \
        -menu $w.menu.select.m
    menu $w.menu.select.m
    $w.menu.select.m add command -label "Select All" \
	-accelerator "  a" \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  Command::SelectAll $editor; \
                  $w.menu.select configure -state normal"
    bind $w <a> "$w.menu.select.m invoke 0"
    $w.menu.select.m add sep
    $w.menu.select.m add command -label "Select Neighbours" \
	-accelerator "  n" \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  Command::SelectNeighbours $editor; \
                  $w.menu.select configure -state normal"
    bind $w <n> "$w.menu.select.m invoke 2"
    $w.menu.select.m add command -label "Select Member" \
	-accelerator "  m" \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  Command::SelectMember $editor; \
                  $w.menu.select configure -state normal"
    bind $w <m> "$w.menu.select.m invoke 3"
    $w.menu.select.m add sep
    $w.menu.select.m add cascade -label "Select by Type" \
	-menu $w.menu.select.m.s
    $w.menu.select.m add command -label "Select by Name" \
	-accelerator "  N" \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  Command::SelectName $editor; \
                  $w.menu.select configure -state normal"
    bind $w <N> "$w.menu.select.m invoke 6"
    $w.menu.select.m add command -label "Select by Address" \
	-accelerator "  A" \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  Command::SelectAddress $editor; \
                  $w.menu.select configure -state normal"
    bind $w <A> "$w.menu.select.m invoke 7"
    $w.menu.select.m add command -label "Select by Label" \
        -accelerator "  L" \
        -command "$w.menu.select configure -state disabled; update idletasks; \
                  Command::SelectLabel $editor; \
                  $w.menu.select configure -state normal"
    bind $w <L> "$w.menu.select.m invoke 8"

    menu $w.menu.select.m.s
    $w.menu.select.m.s add command -label node \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  Command::SelectType $editor NODE; \
                  $w.menu.select configure -state normal"
    $w.menu.select.m.s add command -label network \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  Command::SelectType $editor NETWORK; \
                  $w.menu.select configure -state normal"
    $w.menu.select.m.s add command -label link \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  Command::SelectType $editor LINK; \
                  $w.menu.select configure -state normal"
    $w.menu.select.m.s add command -label group \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  Command::SelectType $editor GROUP; \
                  $w.menu.select configure -state normal"
    $w.menu.select.m.s add command -label text \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  Command::SelectType $editor TEXT; \
                  $w.menu.select configure -state normal"
    $w.menu.select.m.s add command -label image \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  Command::SelectType $editor IMAGE; \
                  $w.menu.select configure -state normal"
    $w.menu.select.m.s add command -label reference \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  Command::SelectType $editor REFERENCE; \
                  $w.menu.select configure -state normal"
    $w.menu.select.m.s add command -label stripchart \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  Command::SelectType $editor STRIPCHART; \
                  $w.menu.select configure -state normal"
    $w.menu.select.m.s add command -label barchart \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  Command::SelectType $editor BARCHART; \
                  $w.menu.select configure -state normal"
    $w.menu.select.m.s add command -label graph \
	-command "$w.menu.select configure -state disabled; update idletasks; \
                  Command::SelectType $editor GRAPH; \
                  $w.menu.select configure -state normal"

    # set up the structure menu
    menubutton $w.menu.structure -text "Structure" \
        -menu $w.menu.structure.m
    menu $w.menu.structure.m
    $w.menu.structure.m add cascade -label "Set Label to" \
	-menu $w.menu.structure.m.l
    $w.menu.structure.m add sep
    $w.menu.structure.m add command -label "Bring to Front" \
	-accelerator "  f" \
        -command "$w.menu.structure configure -state disabled; \
                  update idletasks; \
                  Command::Front $editor; \
                  $w.menu.structure configure -state normal"
    bind $w <f> "$w.menu.structure.m invoke 2"
    $w.menu.structure.m add command -label "Send to Back" \
        -accelerator "  b" \
        -command "$w.menu.structure configure -state disabled; \
                  update idletasks; \
                  Command::Back $editor; \
                  $w.menu.structure configure -state normal"
    bind $w <b> "$w.menu.structure.m invoke 3"
    $w.menu.structure.m add sep
    $w.menu.structure.m add command -label "Group" \
        -accelerator "  g" \
        -command "$w.menu.structure configure -state disabled; \
                  update idletasks; \
                  Command::Group $editor; \
                  $w.menu.structure configure -state normal"
    bind $w <g> "$w.menu.structure.m invoke 5"
    $w.menu.structure.m add command -label "Ungroup" \
        -accelerator "  u" \
        -command "$w.menu.structure configure -state disabled; \
                  update idletasks; \
                  Command::Ungroup $editor; \
                  $w.menu.structure configure -state normal"
    bind $w <u> "$w.menu.structure.m invoke 6"
    $w.menu.structure.m add command -label "Collapse" \
        -accelerator "  c" \
        -command "$w.menu.structure configure -state disabled; \
                  update idletasks; \
                  Command::Collapse $editor; \
                  $w.menu.structure configure -state normal"
    bind $w <c> "$w.menu.structure.m invoke 7"
    $w.menu.structure.m add command -label "Expand" \
        -accelerator "  e" \
        -command "$w.menu.structure configure -state disabled; \
                  update idletasks; \
                  Command::Expand $editor; \
                  $w.menu.structure configure -state normal"
    bind $w <e> "$w.menu.structure.m invoke 8"
    $w.menu.structure.m add sep
    $w.menu.structure.m add command -label "Join Group" \
        -accelerator "  j" \
        -command "$w.menu.structure configure -state disabled; \
                  update idletasks; \
                  Command::Join $editor; \
                  $w.menu.structure configure -state normal"
    bind $w <j> "$w.menu.structure.m invoke 10"
    $w.menu.structure.m add command -label "Leave Group" \
        -accelerator "  l" \
        -command "$w.menu.structure configure -state disabled; \
                  update idletasks; \
                  Command::RemoveGroup $editor; \
                  $w.menu.structure configure -state normal"
    bind $w <l> "$w.menu.structure.m invoke 11"

    menu $w.menu.structure.m.l
    $w.menu.structure.m.l add command -label name \
	-command "Command::Label $editor name"
    $w.menu.structure.m.l add command -label address \
	-command "Command::Label $editor address"
    $w.menu.structure.m.l add command -label attribute \
	-command "Command::Label $editor attribute"
    $w.menu.structure.m.l add sep
    $w.menu.structure.m.l add command -label clear \
	-command "Command::Label $editor clear"
    
    # set up the icon menu
    menubutton $w.menu.icon -text "Icon" \
        -menu $w.menu.icon.m
    menu $w.menu.icon.m
    $w.menu.icon.m add cascade -label "Node" \
	-menu $w.menu.icon.m.node
    $w.menu.icon.m add cascade -label "Network" \
	-menu $w.menu.icon.m.network
    $w.menu.icon.m add cascade -label "Group" \
	-menu $w.menu.icon.m.group
    $w.menu.icon.m add cascade -label "Reference" \
	-menu $w.menu.icon.m.reference
    $w.menu.icon.m add cascade -label "Font" \
	-menu $w.menu.icon.m.font
    if {[tk colormodel .]=="color"} { 
	$w.menu.icon.m add cascade -label "Color" \
	    -menu $w.menu.icon.m.color
    }

    # set up the node menu
    menu $w.menu.icon.m.node
    set i 1
    while {[set val [$editor attribute node$i]]!=""} {
	set name [lrange $val 1 end]
	if {$name == ""} {
            $w.menu.icon.m.node add sep
        } else {
	    tkined_makemenu $w.menu.icon.m.node $name last name
	    $last add radio -label " $name" \
		-variable node$w -value $last$name \
		-command "$w.menu.icon configure -state disabled; \
                      update idletasks; \
                      Command::Icon $editor NODE {$name}; \
                      $w.menu.icon configure -state normal"
	}
	incr i
    }
    $w.menu.icon.m.node add radio -label " Machine" \
	-variable node$w -value machine \
	-command "$w.menu.icon configure -state disabled; \
                  update idletasks; \
                  Command::Icon $editor NODE machine; \
                  $w.menu.icon configure -state normal"
    $w.menu.icon.m.node invoke $i

    # set up the network menu
    menu $w.menu.icon.m.network
    set i 1
    while {[set val [$editor attribute network$i]]!=""} {
	set name [lrange $val 1 end]
	if {$name == ""} {
            $w.menu.icon.m.network add sep
        } else {
	    tkined_makemenu $w.menu.icon.m.network $name last name
	    $last add radio -label " $name" \
		-variable network$w -value $last$name \
		-command "$w.menu.icon configure -state disabled; \
                      update idletasks; \
                      Command::Icon $editor NETWORK {$name}; \
                      $w.menu.icon configure -state normal"
	}
        incr i
    }
    $w.menu.icon.m.network add radio -label " Network" \
	-variable network$w -value network \
	-command "$w.menu.icon configure -state disabled;
                  update idletasks; \
                  Command::Icon $editor NETWORK network; \
                  $w.menu.icon configure -state normal"
    $w.menu.icon.m.network invoke $i

    # set up the group menu
    menu $w.menu.icon.m.group
    set i 1
    while {[set val [$editor attribute group$i]]!=""} {
	set name [lrange $val 1 end]
	if {$name == ""} {
            $w.menu.icon.m.group add sep
        } else {
	    tkined_makemenu $w.menu.icon.m.group $name last name
	    $last add radio -label " $name" \
		-variable group$w -value $last$name \
		-command "$w.menu.icon configure -state disabled;
                      update idletasks; \
                      Command::Icon $editor GROUP {$name}; \
                      $w.menu.icon configure -state normal"
	}
        incr i
    }
    $w.menu.icon.m.group add radio -label " Group" \
	-variable group$w -value group \
	-command "$w.menu.icon configure -state disabled;
                  update idletasks; \
                  Command::Icon $editor GROUP group; \
                  $w.menu.icon configure -state normal"
    $w.menu.icon.m.group invoke $i

    # set up the reference menu
    menu $w.menu.icon.m.reference
    set i 1
    while {[set val [$editor attribute reference$i]]!=""} {
	set name [lrange $val 1 end]
	if {$name == ""} {
            $w.menu.icon.m.reference add sep
        } else {
	    tkined_makemenu $w.menu.icon.m.reference $name last name
	    $last add radio -label " $name" \
		-variable reference$w -value $last$name \
		-command "$w.menu.icon configure -state disabled;
                      update idletasks; \
                      Command::Icon $editor REFERENCE {$name}; \
                      $w.menu.icon configure -state normal"
	}
        incr i
    }
    $w.menu.icon.m.reference add radio -label " Reference" \
	-variable reference$w -value reference \
	-command "$w.menu.icon configure -state disabled;
                  update idletasks; \
                  Command::Icon $editor REFERENCE reference; \
                  $w.menu.icon configure -state normal"
    $w.menu.icon.m.reference invoke $i

    # set up the font menu
    menu $w.menu.icon.m.font
    set i 1
    while {[set val [$editor attribute font$i]]!=""} {
	set name [lrange $val 1 end]
	if {$name == ""} {
	    $w.menu.icon.m.font add sep
	} else {
	    tkined_makemenu $w.menu.icon.m.font $name last name
	    $last add radio -label " $name" \
		-variable font$w -value $last$name \
		-command "$w.menu.icon configure -state disabled;
                          update idletasks; \
                          Command::Font $editor {$name}; \
                          $w.menu.icon configure -state normal"
	    $editor attribute font-$name [lindex $val 0]
	}
	incr i
    }
    $w.menu.icon.m.font add radio -label " Fixed" \
	-variable font$w -value default \
	-command "$w.menu.icon configure -state disabled;
                  update idletasks; \
                  Command::Font $editor fixed; \
                  $w.menu.icon configure -state normal"
    $w.menu.icon.m.font invoke $i

    # set up the color menu
    if {[tk colormodel .]=="color"} { 
	menu $w.menu.icon.m.color
	set i 1
	while {[set val [$editor attribute color$i]]!=""} {
	    set name [lrange $val 1 end]
	    if {$name == ""} {
		$w.menu.icon.m.color add sep
	    } else {
		tkined_makemenu $w.menu.icon.m.color $name last name
		$last add radio -label " $name" \
		    -variable color$w -value $name \
		    -command "$w.menu.icon configure -state disabled;
                          update idletasks; \
                          Command::Color $editor {$name}; \
                          $w.menu.icon configure -state normal"
		$editor attribute color-$name [lindex $val 0]
	    }
	    incr i
	}
	$w.menu.icon.m.color add radio -label " Black" \
	    -variable color$w -value Black \
	    -command "$w.menu.icon configure -state disabled;
                  update idletasks; \
                  Command::Color $editor Black; \
                  $w.menu.icon configure -state normal"
	$editor attribute color-Black black
	$w.menu.icon.m.color invoke $i
    }

    # set up the view menu
    menubutton $w.menu.view -text "View" \
	-menu $w.menu.view.m
    menu $w.menu.view.m
    $w.menu.view.m add cascade -label "Media" \
	-menu $w.menu.view.m.page
    $w.menu.view.m add cascade -label "Orientation" \
	-menu $w.menu.view.m.orient
    $w.menu.view.m add sep
    $w.menu.view.m add command -label "New View" \
	-accelerator "  V" \
	-command "$w.menu.view configure -state disabled; update idletasks; \
                  EDITOR; \
                  $w.menu.view configure -state normal"
    bind $w <V> "$w.menu.view.m invoke 3"
    $w.menu.view.m add command -label "Close View" \
	-accelerator "  K" \
	-command "$w.menu.view configure -state disabled; update idletasks; \
                  $editor delete"
    bind $w <K> "$w.menu.view.m invoke 4"
    $w.menu.view.m add sep
    $w.menu.view.m add command -label "Hide Toolbox" \
	-accelerator "  H" \
	-command "$w.menu.view configure -state disabled; update idletasks; \
                  Command::HideShow $editor; \
                  $w.menu.view configure -state normal"
    bind $w <H> "$w.menu.view.m invoke 6"
    bind $w <S> "$w.menu.view.m invoke 6"

    menu $w.menu.view.m.page
    global tkined_pageSize$w
    $w.menu.view.m.page add radio -label "Letter" \
	-variable tkined_pageSize$w -value Letter \
	-command "$w.menu.view configure -state disabled; update idletasks; \
                  $editor pagesize Letter; \
                  $w.menu.view configure -state normal"
    $w.menu.view.m.page add radio -label "Legal" \
	-variable tkined_pageSize$w -value Legal \
	-command "$w.menu.view configure -state disabled; update idletasks; \
                  $editor pagesize Legal; \
                  $w.menu.view configure -state normal"
    $w.menu.view.m.page add radio -label "DIN A4" \
	-variable tkined_pageSize$w -value A4 \
	-command "$w.menu.view configure -state disabled; update idletasks; \
                  $editor pagesize A4; \
                  $w.menu.view configure -state normal"
    $w.menu.view.m.page add radio -label "DIN A3" \
	-variable tkined_pageSize$w -value A3 \
	-command "$w.menu.view configure -state disabled; update idletasks; \
                  $editor pagesize A3; \
                  $w.menu.view configure -state normal"
    $w.menu.view.m.page add radio -label "DIN A2" \
	-variable tkined_pageSize$w -value A2 \
	-command "$w.menu.view configure -state disabled; update idletasks; \
                  $editor pagesize A2; \
                  $w.menu.view configure -state normal"
    $w.menu.view.m.page add radio -label "DIN A1" \
	-variable tkined_pageSize$w -value A1 \
	-command "$w.menu.view configure -state disabled; update idletasks; \
                  $editor pagesize A1; \
                  $w.menu.view configure -state normal"
    $w.menu.view.m.page invoke 2

    menu $w.menu.view.m.orient
    global tkined_orientation$w
    $w.menu.view.m.orient add radio -label "Portrait" \
	-variable tkined_orientation$w -value portrait \
	-command "$w.menu.view configure -state disabled; update idletasks; \
                  $editor orientation portrait; \
                  $w.menu.view configure -state normal"
    $w.menu.view.m.orient add radio -label "Landscape" \
	-variable tkined_orientation$w -value landscape \
	-command "$w.menu.view configure -state disabled; update idletasks; \
                  $editor orientation landscape; \
                  $w.menu.view configure -state normal"
    $w.menu.view.m.orient invoke 1

    # set up the help menu
    menubutton $w.menu.help -text "Help" \
	-menu $w.menu.help.m
    menu $w.menu.help.m
    $w.menu.help.m add command -label "General" \
	-command "$w.menu.help configure -state disabled; update idletasks; \
                  Help::general $editor; \
	          $w.menu.help configure -state normal"
    $w.menu.help.m add command -label "Status" \
	-command "$w.menu.help configure -state disabled; update idletasks; \
                  Help::status $editor; \
                  $w.menu.help configure -state normal"
    $w.menu.help.m add command -label "Key Bindings" \
	-command "$w.menu.help configure -state disabled; update idletasks; \
                  Help::key_bindings $editor; \
                  $w.menu.help configure -state normal"
    $w.menu.help.m add command -label "Man Page (tkined)" \
	-command "$w.menu.help configure -state disabled; update idletasks; \
                  Help::manpage $editor tkined; \
                  $w.menu.help configure -state normal"
    $w.menu.help.m add command -label "Man Page (scotty)" \
	-command "$w.menu.help configure -state disabled; update idletasks; \
                  Help::manpage $editor scotty; \
                  $w.menu.help configure -state normal"
    
    pack $w.menu.file -side left
    pack $w.menu.edit -side left
    pack $w.menu.select -side left
    pack $w.menu.structure -side left
    pack $w.menu.icon -side left
    pack $w.menu.view -side left
    pack $w.menu.help -side right

    # bind the select tool on the left mouse button
    bind $w.canvas <3> \
	"$editor selection clear; \
         Tool::SelectMark $editor \
            \[$w.canvas canvasx %x\] \[$w.canvas canvasy %y\]"
    bind $w.canvas <B3-Motion> \
	"Tool::SelectDrag $editor \
            \[$w.canvas canvasx %x\] \[$w.canvas canvasy %y\]"
    bind $w.canvas <ButtonRelease-3> \
	"Tool::SelectDone $editor"
    bind $w.canvas <Shift-Button-3> \
	"Tool::SelectMark $editor \
            \[$w.canvas canvasx %x\] \[$w.canvas canvasy %y\]"
    bind $w.canvas <Shift-B3-Motion> \
	"Tool::SelectDrag $editor \
            \[$w.canvas canvasx %x\] \[$w.canvas canvasy %y\]"
    bind $w.canvas <Shift-ButtonRelease-3> \
	"Tool::SelectDone $editor"

    # bind the move tool on the middle mouse button
    bind $w.canvas <2> "Tool::MoveMark $editor \
                \[$w.canvas canvasx %x\] \[$w.canvas canvasy %y\]"
    bind $w.canvas <B2-Motion> "Tool::MoveDrag $editor \
                \[$w.canvas canvasx %x\] \[$w.canvas canvasy %y\]"
    bind $w.canvas <ButtonRelease-2> "Tool::MoveDone $editor \
                \[$w.canvas canvasx %x\] \[$w.canvas canvasy %y\]"

    # Key binding to turn on debugging mode
    bind $w <D> "global tkined_debug; set tkined_debug 1"
    bind $w <Q> "global tkined_debug; set tkined_debug 0"

    # Set up for keyboard-based menu traversal
    tk_bindForTraversal $w
    bind $w <Any-Enter> "focus $w"
    eval tk_menuBar $w.menu [pack slaves $w.menu]

    # now its time to map the whole thing on the screen
    update
    wm deiconify $w
    update

    # fire up all interpreters for this editor
    set i 1
    while {[set interp [$editor attribute interpreter$i]]!=""} {
        incr i
	if {[catch {INTERPRETER create $interp} interpreter]} {
	    if {$interp != "manager.tcl"} {
		Dialog::acknowledge $w.canvas \
		    "Sorry, $interp not found !!" "" \
		    "Check your tkined.defaults and your TKINEDPATH."
	    }
	} else {
	    $interpreter editor $editor
	    $interpreter canvas $w.canvas
	}
    }
}

##
## Change the title of the toplevel window to show the new filename.
##

proc Editor::filename { editor } { 
    wm title [$editor toplevel] "$editor: [$editor filename]"
}

##
## Set the canvas to the new width and size. Set the global
## variables tkined_orientation$w and tkined_pageSize$w to
## update the menu.
##

proc Editor::pagesize { editor width height } {

    set c [$editor toplevel].canvas
    $c configure -scrollregion "0 0 $width $height"

    set w [$editor toplevel]
    global tkined_orientation$w tkined_pageSize$w
    set tkined_orientation$w [$editor orientation]
    set tkined_pageSize$w    [$editor pagesize]
}

##
## Create a PostScript dump of the canvas. We need to do some magic here
## to get the background right. I think tk is a bit broken here...
##

proc Editor::postscript { editor } {

    global tkined_ps_map
    set w [$editor toplevel]
    set tkined_ps_map(fixed) [list Courier 10]

    set orientation [$editor orientation]
    foreach option [$w.canvas config] {
	if {[lindex $option 0] == "-scrollregion"} {
	    set i [llength $option]
	    incr i -1
	    set scrolloption [lindex $option $i]
	    set width  [lindex $scrolloption 2]
	    set height [lindex $scrolloption 3]
	}
    }
    foreach item [$w.canvas find all] {
	switch [$w.canvas type $item] {
	    bitmap {
		$w.canvas itemconfigure $item -background White
	    }
	    stripchart {
		$w.canvas itemconfigure $item -background White
	    }
	    barchart {
		$w.canvas itemconfigure $item -background White
	    }
	}
	$w.canvas itemconfigure clip$item -fill White -outline White
    }

    # make all selection marks invisible

    foreach id [$editor selection] {
	foreach item [$id items] {
	    $w.canvas itemconfigure mark$item -outline "" -fill ""
	}
    }

    update

    if {[catch {$w.canvas postscript \
            -fontmap tkined_ps_map \
	    -height $height -width $width \
            -rotate [expr {$orientation == "landscape"}]} result]} {
	Dialog::acknowledge $w.canvas "Generating PostScript failed:" $result
    }

    set color [lindex [$w.canvas configure -background] 4]
    foreach item [$w.canvas find all] {
	switch [$w.canvas type $item] {
	    bitmap {
		$w.canvas itemconfigure $item -background ""
	    }
	    stripchart {
		$w.canvas itemconfigure $item -background $color
	    }
	    barchart {
		$w.canvas itemconfigure $item -background $color
	    }
	}
	$w.canvas itemconfigure clip$item -fill $color -outline $color
    }

    # restore the selection marks

    foreach id [$editor selection] {
	foreach item [$id items] {
	    $w.canvas itemconfigure mark$item -outline black -fill black
	}
    }

    return $result
}

##
## I do not know what to do with this button.
##

proc tkined_useless_button { c } {
    if {[catch {exec /bin/sh -c "( fortune || yow )" 2> /dev/null} txt]} {
	set txt "You do not have fortune or yow on your system? Unbelievable!"
    }
    Dialog::acknowledge $c $txt
}

##
## Make a menu entry. The ultimate test for this proc is to create a
## menu entry like: ined create MENU asdf "aa:bb:cc bb:cc:dd cc:bb:cc"
##

proc tkined_makemenu {w path rlast rname} {

    static widgetpath
    static count

    upvar $rlast last
    upvar $rname name 

    if {![info exists count]} { set count 0 }
    incr count

    set path [split $path :]
    set len  [llength $path]
    set name [lindex $path [incr len -1]]
    set last $w

    for {set i 0} {$i < $len} {incr i} {
	set elem [join [lrange $path 0 $i] :]
	if {![info exists widgetpath($w$elem)]} {
	    set widgetpath($w$elem) $last.$count
	    menu $widgetpath($w$elem)
	    $last add cascade -label " [lindex $path $i]" \
		-menu $widgetpath($w$elem)
	}

	set last $widgetpath($w$elem)
    }
}
