##
## Objects.tcl
##
## This file contains the callbacks that are called whenever and object
## changes its internal state. All calls to procs defined in this file
## should be made by one of the methods defined in methods.c. This file
## implements the View according to the Model-View-Controller paradigm.
##
## Every object can stores its main canvas item in its items attribute.
## All additional items used to create a label etc. are accessed using 
## tags. We use the following tag names (assuming $item has the canvas 
## as stored in the item attribute of an object):
##
##    mark$item   - a tag for all items marking $item as selected
##    label$item  - a tag for the text item containing a label
##    clip$item   - a tag for a box used to clip
##
## More speedups are possible by removing some indirection, but this
## will make the code less readable. So if we need more speed, we 
## should create a real icon canvas object that makes all this tk 
## code superfluous
##
## Copyright (c) 1993, 1994
##
## J. Schoenwaelder
## TU Braunschweig, Germany
## Institute for Operating Systems and Computer Networks
##
## Permission to use, copy, modify, and distribute this
## software and its documentation for any purpose and without
## fee is hereby granted, provided that this copyright
## notice appears in all copies.  The University of Braunschweig
## makes no representations about the suitability of this
## software for any purpose.  It is provided "as is" without
## express or implied warranty.
##


##
## These procs are of general interest and are called by the
## type specific callbacks to save some code.
##

proc ::delete { id } {
    set c [$id canvas]
    if {$c == ""} {
	set c .[$id editor].canvas
    }
    foreach item [$id items] {
	$c delete $item mark$item label$item clip$item
    }
    $id items ""
}

proc ::size { id type } {
    return [[$id canvas] bbox [$id items]]
}

proc ::move { id dx dy } {
    set c [$id canvas]
    set item [$id items]
    $c move $item $dx $dy
    $c move mark$item $dx $dy
    $c move label$item $dx $dy
    $c move clip$item $dx $dy
}

proc ::select { id type } {
    set c [$id canvas]
    set item [$id items]
    $c addtag selected withtag $item
    if {[$c type $item] == "line" && $type != "GROUP"} {
	tkined_mark_points $c $item
    } else {
	tkined_mark_box $c $item
    }
}

proc ::unselect { id } {
    set c [$id canvas]
    set item [$id items]
    $c dtag $item selected
    $c delete mark$item
}

proc ::color { id type color } {
    set c [$id canvas]
    set item [$id items]
    switch [$c type $item] {
	bitmap { $c itemconfigure $item -foreground $color }
	line   { $c itemconfigure $item -fill $color }
	text   { $c itemconfigure $item -fill $color }
	stripchart { $c itemconfigure $item -outline $color }
	barchart   { $c itemconfigure $item -outline $color }
    }
}

proc ::font { id type font} {
    set c [$id canvas]
    set item [$id items]
    if {[$c type $item] == "text"} {
	$c itemconfigure $item -font $font
    }
    catch {$c itemconfigure label$item -font $font}
}

proc ::raise { id } {
    set c [$id canvas]
    set item [$id items]
    $c raise $item
    catch {$c lower clip$item label$item}
}

proc ::lower { id } {
    set c [$id canvas]
    set item [$id items]
    $c lower $item
    catch {$c lower clip$item label$item}
}

proc ::icon { id bitmap } {
    set c [$id canvas]
    set item [$id items]
    if {[$c type $item] == "bitmap"} {
	$c itemconfigure $item -bitmap $bitmap
    }
}

proc ::clearlabel { id } {
    set item [$id items]
    [$id canvas] delete label$item clip$item
}

proc ::label { id type text } {

    set c [$id canvas]
    set item [$id items]

    if {$item == ""} return

    # search for an existing label

    set label [$c find withtag label$item]
    set clip  [$c find withtag clip$item]

    if {$label != ""} {
        $c itemconfigure $label -text $text
        if {$clip != ""} {
            eval $c coords $clip [$c bbox $label]
        }
	return
    }

    # create a new label with and a box behind it

    set color [lindex [$c configure -background] 4]

    if {$type == "NETWORK"} {
	set xy [$id labelxy]
	set x [lindex $xy 0]
	set y [lindex $xy 1]
    } else {
	set bb [$c bbox $item]
	set x1 [lindex $bb 0]
	set x2 [lindex $bb 2]
	set y [expr {[lindex $bb 3]+1}]
	set x [expr {$x1+(($x2-$x1)/2)}]
    }

    set label [$c create text $x $y -anchor n -text $text -font fixed \
	       -justify center -tags [list label$item "id $id"] ]
    set tags [list clip$item "id $id"]
    set clip [eval $c create rectangle [$c bbox $label] \
	      -tags {$tags} -fill $color -outline $color -width 0 ]
    $c lower $clip $label
    $id font [$id font]
}

proc ::postscript { id } {

    global tkined_ps_map
    set tkined_ps_map(fixed) [list Courier 10]

    set c [$id canvas]
    set item [$id items]

    set bb [$c bbox $items $label$item $clip$item]

    set width  [expr {[lindex $bb 2] -[lindex $bb 0]}]
    set height [expr {[lindex $bb 3] -[lindex $bb 1]}]
    set x [lindex $bb 0]
    set y [lindex $bb 1]

    $c postscript -fontmap tkined_ps_map \
	    -height $height -width $width -x $x -y $y
}

##
## The following set of procedures handle node objects.
##

proc NODE::canvas { node } {
    $node items [[$node canvas] create bitmap \
		  [lindex [$node move] 0] [lindex [$node move] 1] \
		      -bitmap machine -tags [list NODE "id $node"] ]
}

proc NODE::delete { node } {
    ::delete $node
}

proc NODE::size { node } {
    ::size $node NODE
}

proc NODE::move { node dx dy } {
    ::move $node $dx $dy
}

proc NODE::select { node } {
    ::select $node NODE
}

proc NODE::unselect { node } {
    ::unselect $node
}

proc NODE::color { node color } {
    ::color $node NODE $color
}

proc NODE::font { node font } {
    ::font $node NODE $font
}

proc NODE::raise { node } {
    ::raise $node
}

proc NODE::lower { node } {
    ::lower $node
}

proc NODE::icon { node bitmap } {
    ::icon $node $bitmap
}

proc NODE::clearlabel { node } {
    ::clearlabel $node
}

proc NODE::label { node text } {
    ::label $node NODE $text
}


##
## The following set of procedures handle group objects.
##

proc GROUP::canvas { group } {
    $group collapse
}

proc GROUP::delete { group } {
    ::delete $group
}

proc GROUP::size { group } {
    ::size $group GROUP
}

proc GROUP::move { group dx dy } {
    ::move $group $dx $dy
}

proc GROUP::select { group } {
    ::select $group GROUP
}

proc GROUP::unselect { group } {
    ::unselect $group
}

proc GROUP::color { group color } {
    ::color $group GROUP $color
}

proc GROUP::font { group font } {
    ::font $group GROUP $font
}

proc GROUP::raise { group } {
    ::raise $group
}

proc GROUP::lower { group } {
    ::lower $group
}

proc GROUP::icon { group bitmap } {
    ::icon $group $bitmap
}

proc GROUP::clearlabel { group } {
    ::clearlabel $group
}

proc GROUP::label { group text} {
    ::label $group GROUP $text
}

proc GROUP::collapse { group } {

    ::delete $group
    foreach id [$group member] {
	::delete $id
    }

    $group items [[$group canvas] create bitmap \
		    [lindex [$group move] 0] [lindex [$group move] 1] \
			-bitmap group -tags [list GROUP "id $group"] ]
}

proc GROUP::expand { group } {

    set c [$group canvas]
    ::delete $group

    set bb [$group move]
    set x1 [expr {[lindex $bb 0]-30}]
    set y1 [expr {[lindex $bb 1]-30}]
    set x2 [expr {[lindex $bb 0]+30}]
    set y2 [expr {[lindex $bb 1]+30}]
    
    set tags [list GROUP "id $group"]
    $group items [eval $c create line $x1 $y1 $x1 $y2 $x2 $y2 $x2 $y1 $x1 $y1 \
		    -width 2 -fill Black -stipple gray50 -joinstyle miter \
			-tags {$tags}]

    if {[$group member] != ""} {
	GROUP::resize $group
    }
}

proc GROUP::resize { group } {

    set c [$group canvas]

    set memberitems ""
    foreach id [$group member] {
	set item [$id items]
        lappend memberitems $item label$item
    }
    if {$memberitems == ""} return

    set bb [eval $c bbox [join $memberitems]]
    set x1 [expr {[lindex $bb 0]-3}]
    set y1 [expr {[lindex $bb 1]-3}]
    set x2 [expr {[lindex $bb 2]+3}]
    set y2 [expr {[lindex $bb 3]+3}]

    $c coords "id $group" $x1 $y1 $x1 $y2 $x2 $y2 $x2 $y1 $x1 $y1
}


##
## The following set of procedures handle network objects.
##

proc NETWORK::canvas { network } {
    set c [$network canvas]
    set points [join [$network points]]
    set tags [list NETWORK "id $network"]
    set item [eval $c create line $points -width 3 -fill black -tags {$tags} ]
    eval $c move $item [$network move]
    $network items $item
}

proc NETWORK::delete { network } {
    ::delete $network
}

proc NETWORK::size { network } {
    ::size $network NETWORK
}

proc NETWORK::move { network dx dy } {
    ::move $network $dx $dy
}

proc NETWORK::select { network } {
    ::select $network NETWORK
}

proc NETWORK::unselect { network } {
    ::unselect $network
}

proc NETWORK::color { network color } {
    ::color $network NETWORK $color
}

proc NETWORK::icon { network width } {
    [$network canvas] itemconfigure [$network items] -width $width
}

proc NETWORK::clearlabel { network } {
    ::clearlabel $network
}

proc NETWORK::label { network text } {
    ::label $network NETWORK $text
}

proc NETWORK::font { network font } {
    ::font $network NETWORK $font
}

proc NETWORK::raise { network } {
    ::raise $network
}

proc NETWORK::lower { network } {
    ::lower $network
}


##
## The following set of procedures handle link objects.
##

proc LINK::canvas {link} {
    set c [$link canvas]
    set points [join [$link points]]
    set len [llength $points]
    if {$len%2 != 0} { 
	incr len -2
	set points [join [lrange $points 1 len]]
	inr len
    }
    set xya [[$link src] move]
    set xyb [[$link dst] move]
    set tags [list LINK "id $link"]
    $link items [eval $c create line $xya $points $xyb \
		 -fill black -tags {$tags}]
}

proc LINK::delete { link } {
    ::delete $link
}

proc LINK::size { link } {
    ::size $link LINK
}

proc LINK::move { link dx dy } {
    ::move $link $dx $dy
}

proc LINK::select { link } {
    ::select $link LINK
}

proc LINK::unselect { link } {
    ::unselect $link
}

proc LINK::color { link color } {
    ::color $link LINK $color
}

proc LINK::raise { link } {
    [$link canvas] raise [$link items]
}

proc LINK::lower { link } {
    [$link canvas] lower [$link items]
}


##
## The following set of procedures handle text objects.
##

proc TEXT::canvas { text } {
    set c [$text canvas]
    set w [winfo parent $c]
    set x [lindex [$text move] 0]
    set y [lindex [$text move] 1]
    set txt [$text text]
    regsub -all "\\\\n" $txt "\n" txt
    $text items [$c create text $x $y -anchor nw -text $txt -font fixed \
		 -tags [list TEXT "id $text"]  ]
}

proc TEXT::delete { text } {
    ::delete $text
}

proc TEXT::size { text } {
    ::size $text TEXT
}

proc TEXT::move { text dx dy } {
    ::move $text $dx $dy
}

proc TEXT::select { text } {
    ::select $text TEXT
}

proc TEXT::unselect { text } {
    ::unselect $text
}

proc TEXT::color { text color } {
    ::color $text TEXT $color
}

proc TEXT::font { text font } {
    ::font $text TEXT $font
}

proc TEXT::text { text } {
    set c [$text canvas]
    regsub -all "\\\\n" [$text text] "\n" txt
    set item [$text items]
    $c itemconfigure $item -text $txt
}

proc TEXT::raise { text } {
    [$text canvas] raise [$text items]
}

proc TEXT::lower { text } {
    [$text canvas] lower [$text items]
}


##
## The following set of procedures handle image objects.
##

proc IMAGE::canvas { image } {

    set c [$image canvas]
    set fname [$image name]

    # expand URL ftp syntax

    if {[string match "ftp://*" $fname] || [string match "file://*" $fname]} {
        set idx [string first "://" $fname]
        incr idx 3
        set fname [string range $fname $idx end]
        set idx [string first "/" $fname]
        set server [string range $fname 0 [expr {$idx-1}]]
        set file [string range $fname $idx end]
        set fname "/tmp/$server:[file tail $file]"

        if {[catch {tkined_ftp $server $file $fname} err]} {
            Dialog::acknowledge $w.canvas "Can not retrieve file $fname." $err
            return ""
        }
    }

    set x [lindex [$image move] 0]
    set y [lindex [$image move] 1]
    set tags [list IMAGE "id $image"]
    if {[catch {$c create bitmap $x $y -bitmap @$fname -tags $tags} item]} {
	Dialog::acknowledge $c "Image file not readable!"
	$image canvas ""
	return
    }
    $image items $item
    $image lower
}

proc IMAGE::delete { image } {
    ::delete $image
}

proc IMAGE::size { image } {
    ::size $image IMAGE
}

proc IMAGE::move { image dx dy } {
    ::move $image $dx $dy
}

proc IMAGE::select { image } {
    ::select $image IMAGE
}

proc IMAGE::unselect { image } {
    ::unselect $image
}

proc IMAGE::color { image color } {
    ::color $image IMAGE $color
}

proc IMAGE::lower { image } {
    [$image canvas] lower [$image items]
}


##
## The following set of procedures handle MENU objects.
##

proc MENU::canvas { menu } {

    set interpreter [$menu interpreter]
    set c [$menu canvas]
    set w [winfo parent $c]
    set name $w.tools.[$menu id]

    set name $w.menu.[$menu id]
    menubutton $name -text [$menu name] -menu $name.m
    menu $name.m
    foreach cmd [$menu items] {
	if {$cmd==""} {
	    $name.m add separator
	} else {
	    tkined_makemenu $name.m $cmd newname newcmd
	    $newname add command -label $newcmd \
		-command [list MENU::send $menu $newcmd]
	}
    }
    pack $name -side left

    # update the traversal bindings
    eval tk_menuBar $w.menu [pack slaves $w.menu]

    $interpreter items "$menu [$interpreter items]"
}

proc MENU::send { menu cmd } {

    set editor [$menu editor]
    set interpreter [$menu interpreter]
    set l ""
    foreach id [$editor selection] {
	lappend l [$id retrieve]
    }
    $interpreter send $cmd $l
}

proc MENU::delete { menu } {

    set interpreter [$menu interpreter]
    set c [$menu canvas]
    set w [winfo parent $c]
    destroy $w.menu.[$menu id]

    # update the traversal bindings
    eval tk_menuBar $w.menu [pack slaves $w.menu]

    # and remove the menu from the interpreter
    set new ""
    foreach t [$interpreter items] {
	if {$t != $menu} { 
	    lappend new $t
	}
    }
    $interpreter items $new
}



##
## The following set of procedures handle interpreter objects.
##

proc INTERPRETER::queue { interpreter qlen } {
    set w [[$interpreter editor] toplevel]
    if {$qlen > 0} { set state disabled } { set state normal }

    foreach menu [$interpreter items] {
	set name $w.menu.$menu
	$name configure -state $state
	set last [$name.m index last]
	if {$last == "none"} continue
	for {set idx 0} {$idx <= $last} {incr idx} {
	    catch {$name.m entryconfigure $idx -state $state}
	}
    }
}


##
## The following set of procedures handle log objects.
##

proc LOG::canvas { log } {

    # The offset used to position log windows automatically.
    static offset
    if {![info exists offset]} {
	set offset 80
    } else {
	incr offset 10
	if {$offset > 180} {set offset 80}
    }

    set c [$log canvas]
    set w [winfo parent $c]
    toplevel $w.$log

    frame $w.$log.button 
    button $w.$log.button.ok    -padx 5 -text dismiss \
	-command "$log delete"
    button $w.$log.button.clear -padx 5 -text clear \
	-command "$log clear"
    button $w.$log.button.save  -padx 5 -text save \
	-command "LOG::save $log"
    button $w.$log.button.load  -padx 5 -text load \
	-command "LOG::load $log"
    button $w.$log.button.print  -padx 5 -text print \
	-command "LOG::print $log"
    button $w.$log.button.email -padx 5 -text email \
	-command "LOG::email $log"
    button $w.$log.button.up -padx 5 -text up \
	-command "LOG::up $log"
    button $w.$log.button.down -padx 5 -text down \
	-command "LOG::down $log"
    button $w.$log.button.freeze -padx 5 -text freeze \
	-command "LOG::freeze $log"
    pack $w.$log.button.clear  -pady 5 -padx 2 -side left
    pack $w.$log.button.up     -pady 5 -padx 2 -side left
    pack $w.$log.button.down   -pady 5 -padx 2 -side left
    pack $w.$log.button.freeze -pady 5 -padx 2 -side left
    pack $w.$log.button.load   -pady 5 -padx 2 -side left
    pack $w.$log.button.save   -pady 5 -padx 2 -side left
    pack $w.$log.button.print  -pady 5 -padx 2 -side left
    pack $w.$log.button.email  -pady 5 -padx 2 -side left
    pack $w.$log.button.ok     -pady 5 -padx 2 -side left
    pack $w.$log.button -anchor w

    scrollbar $w.$log.scrollbar -command "$w.$log.text yview" -relief sunken
    text $w.$log.text -height 24 -width 80 -setgrid true \
	-relief sunken -borderwidth 2 \
	-yscrollcommand "$w.$log.scrollbar set" 
    pack $w.$log.scrollbar -side right -fill y
    pack $w.$log.text -side left -padx 2 -pady 2 -fill both -expand yes

    $log items $w.$log

    # This special purpose binding makes it possible to send
    # complete lines back to the interpreter that created this
    # window. This allows us to use a log window as a simple
    # command frontend.

    bind $w.$log.text <Shift-Return> "LOG::process $log"
    bind $w.$log.text <Control-n>    "LOG::clear $log"
    bind $w.$log.text <Control-o>    "LOG::load $log"
    bind $w.$log.text <Control-s>    "LOG::save $log"
    bind $w.$log.text <Control-q>    "$log delete"

    # Position the log window on the screen.

    wm withdraw $w.$log
    update idletasks
    set top [winfo toplevel $w]

    set rx [expr {[winfo rootx $top]}]
    set ry [expr {[winfo rooty $top]}]

    set cx [expr $rx+[winfo width $top]/4]
    set cy [expr $ry+[winfo height $top]/4]

    set x  [expr $cx+$offset]
    set y  [expr $cy+$offset]

    if {$x < 0} { set x 0 }
    if {$y < 0} { set y 0 }

    wm geometry $w.$log +$x+$y
    wm deiconify $w.$log    

    update idletasks
}

proc LOG::process { log } {

    set w [$log items].text
    set i1 [$w index "insert linestart"]
    set i2 [$w index "insert"]
    set txt [$w get $i1 $i2]
    if {$txt == ""} return

    [$log interpreter] send \
	eval ined append $log \"\[ catch \{ $txt \} err \; set err \]\\n\"

    $w insert insert "\n"
}

proc LOG::name { log } {
    wm title [$log items] [$log name]
    wm iconname [$log items] [$log name]
}

proc LOG::icon { log bitmap } {
    wm iconbitmap [$log items] $bitmap
}

proc LOG::append { log text } {
    set w [$log items]
    $w.text insert end $text
    if {[lindex [$w.button.freeze configure -text] 4] != "melt"} {
	$w.text yview -pickplace end
    }
}

proc LOG::bind { log cmd text } {

    set w [$log items].text

    if {[tk colormodel $w] == "color"} {
        set bold "-foreground red"
        set normal "-foreground {}"
    } else {
        set bold "-foreground white -background black"
        set normal "-foreground {} -background {}"
    }

    $w insert end "<"
    set start [$w index end]
    $w insert $start $text
    $w yview -pickplace end
    set end [$w index end]
    $w insert end ">"
    set tag "tag$start$end"
    $w tag add $tag $start $end

    $w tag configure $tag -underline true
    $w tag bind $tag <Any-Enter> "$w tag configure $tag $bold"
    $w tag bind $tag <Any-Leave> "$w tag configure $tag $normal"
    $w tag bind $tag <Button-3> "[$log interpreter] send $cmd"
    $w tag bind $tag <Button-1> "[$log interpreter] send $cmd"
}

proc LOG::unbind { log } {
    set w [$log items].text
    foreach tag [$w tag names] {
	catch {$w tag delete $tag}
    }
}

proc LOG::clear { log } {
    [$log items].text delete 0.0 end
}

proc LOG::delete { log } {
    destroy [$log items]
}

proc LOG::up { log } {
    [$log items].text yview "@0,0 - 24 lines"
}

proc LOG::down { log } {
    [$log items].text yview "@0,0 + 24 lines"
}

proc LOG::freeze { log } {
    if {[lindex [[$log items].button.freeze configure -text] 4] == "freeze"} {
	[$log items].button.freeze configure -text melt
    } else {
	[$log items].button.freeze configure -text freeze
    }
}

proc LOG::save { log } {

    set fname [Dialog::fileselect [$log items] "Write to file:"]
    if {$fname==""} return

    set mode "w"
    if {[file exists $fname]} {
	set result [Dialog::confirm [$log items] \
		    "File $fname already exists!" [list replace append cancel]]
	switch [lindex $result 0] {
	    "cancel" {
		return
	    }
	    "replace" {
		set mode "w"
	    }
	    "append" {
		set mode "a"
	    }
	}
    }

    if {[catch {open $fname $mode} file]} {
	Dialog::acknowledge [$log items] "Unable to open $fname."
	return
    }

    puts $file [[$log items].text get 1.0 end]
    close $file
}

proc LOG::load { log } {
    set fname [Dialog::fileselect [$log items] "Read from file:"]
    if {$fname==""} return

    if {[catch {open $fname r} file]} {
	Dialog::acknowledge [$log items] "Unable to read from $fname"
	return
    }

    $log clear
    while {![eof $file]} {
	gets $file line
	[$log items].text insert end "$line\n"
    }
    [$log items].text yview -pickplace end
    [$log items].text yview 1.0
    close $file
}

proc LOG::print { log } {

    set fname "/tmp/tkined[pid].log"
    catch {exec /bin/rm -f $fname}

    if {[file exists $fname] && ![file writable $fname]} {
	Dialog::acknowledge [$log items] "Can not write temporary file $fname."
	return
    }

    if {[catch {open $fname w} file]} {
	Dialog::acknowledge [$log items] "Can not open $fname: $file"
	return
    }

    if {[catch {puts $file [[$log items].text get 1.0 end]} err]} {
	Dialog::acknowledge [$log items] "Failed to write $fname: $err"
    }

    catch {close $file}

    tkined_print [$log editor] [$log items] $fname

    catch {exec /bin/rm -f $fname}
}

proc LOG::email { log } {
    global env

    set result [Dialog::request [$log items] \
		"Please enter the email address:" \
		[list [list To: [$log address]] \
		      [list Subject: [$log name]] ] \
		[list send cancel] ]
    if {[lindex $result 0] == "cancel"} return

    set to [lindex $result 1]
    $log address $to
    set subject [lindex $result 2]

    if {[catch {split $env(PATH) :} path]} {
	set path "/usr/bin /bin /usr/ucb /usr/local/bin"
    }

    set mprog ""
    foreach mailer "Mail mail" {
	foreach dir $path {
	    set fname $dir/$mailer
	    if {[file executable $fname] && [file isfile $fname]} {
		set mprog $fname
		break
	    }
	}
	if {$mprog != ""} break
    }

    if {$mprog == ""} {
	Dialog::acknowledge [$log items] "Sorry, can not find mail program."
	return
    }

    if {[catch {open "|$mprog -s \"$subject\" $to" w} file]} {
        Dialog::acknowledge [$log items] "Unable to write to $mprog $to"
        return
    }

    puts $file [[$log items].text get 1.0 end]
    close $file
}

##
## Implementation of references to other tkined maps.
##

proc REFERENCE::canvas { ref } {
    set c [$ref canvas]
    $ref items [$c create bitmap \
		[lindex [$ref move] 0] [lindex [$ref move] 1] \
		    -bitmap reference -tags [list REFERENCE "id $ref"] ]
}

proc REFERENCE::load { reference } {
    set name [$reference address]
    if {$name != ""} {
	set editor [EDITOR]
	foreach w [winfo children .] { $w configure -cursor watch }
	update idletasks
	if {[Command::Open $editor $name] == ""} {
	    $editor delete
	}
	foreach w [winfo children .] { $w configure -cursor top_left_arrow }
    }
}

proc REFERENCE::delete { reference } {
    ::delete $reference
}

proc REFERENCE::size { reference } {
    ::size $reference REFERENCE
}

proc REFERENCE::move { reference dx dy } {
    ::move $reference $dx $dy
}

proc REFERENCE::select { reference } {
    ::select $reference REFERENCE
}

proc REFERENCE::unselect { reference } {
    ::unselect $reference
}

proc REFERENCE::color { reference color } {
    ::color $reference REFERENCE $color
}

proc REFERENCE::font { reference font } {
    ::font $reference REFERENCE $font
}

proc REFERENCE::raise { reference } {
    ::raise $reference
}

proc REFERENCE::lower { reference } {
    ::lower $reference
}

proc REFERENCE::icon { reference bitmap } {
    ::icon $reference $bitmap
}

proc REFERENCE::clearlabel { reference } {
    ::clearlabel $reference
}

proc REFERENCE::label { reference text } {
    ::label $reference REFERENCE $text
}


##
##
##

proc STRIPCHART::canvas { stripchart } {
    set c [$stripchart canvas]
    set color [option get $c background tkined]
    if {$color == ""} { set color white }
    set item [$c create stripchart -31 -21 31 21 \
	       -outline black -background $color \
	       -tags [list STRIPCHART "id $stripchart"]]
    eval $c move $item [$stripchart move]
    $stripchart items $item
}

proc STRIPCHART::delete { stripchart } {
    ::delete $stripchart
}

proc STRIPCHART::size { stripchart } {
    ::size $stripchart STRIPCHART
}

proc STRIPCHART::move { stripchart dx dy } {
    ::move $stripchart $dx $dy
}

proc STRIPCHART::select { stripchart } {
    ::select $stripchart STRIPCHART
}

proc STRIPCHART::unselect { stripchart } {
    ::unselect $stripchart
}

proc STRIPCHART::color { stripchart color } {
    ::color $stripchart STRIPCHART $color
}

proc STRIPCHART::font { stripchart font } {
    ::font $stripchart STRIPCHART $font
}

proc STRIPCHART::raise { stripchart } {
    ::raise $stripchart
}

proc STRIPCHART::lower { stripchart } {
    ::lower $stripchart
}

proc STRIPCHART::clearlabel { stripchart } {
    ::clearlabel $stripchart
}

proc STRIPCHART::label { stripchart text } {
    ::label $stripchart STRIPCHART $text
}

proc STRIPCHART::resize { stripchart x1 y1 x2 y2 } {
    set c [$stripchart canvas]
    set item [$stripchart items]
    $c coords $item $x1 $y1 $x2 $y2
}

proc STRIPCHART::clear { stripchart } {
    set c [$stripchart canvas]
    set item [$stripchart items]
    set bb [$c bbox $item] 
    set lower [lindex $bb 0]
    set upper [lindex $bb 2]
    set values 0
    for {set x $lower} {$x < $upper} {incr x} {
	lappend values 0
    }
    $c itemconfigure $item -values $values
}

proc STRIPCHART::values { stripchart args } {
    set c [$stripchart canvas]
    set item [$stripchart items]
    if {$args == ""} {
	return [lindex [$c itemconfigure $item -values] 4]
    } else {
	$c itemconfigure $item -values [join $args]
    }
}

proc STRIPCHART::scale { stripchart num } {
    set c [$stripchart canvas]
    set item [$stripchart items]
    $c itemconfigure $item -scalevalue $num
}

proc STRIPCHART::jump { stripchart num } {
    set c [$stripchart canvas]
    set item [$stripchart items]
    $c itemconfigure $item -jump $num
}


##
##
##

proc BARCHART::canvas { barchart } {
    set c [$barchart canvas]
    set color [option get $c background tkined]
    if {$color == ""} { set color white }
    if {[tk colormodel .] == "color"} {
	set item [$c create barchart -31 -21 31 21 -autocolor on \
		   -background $color -tags [list BARCHART "id $barchart"]]
    } else {
	set item [$c create barchart -31 -21 31 21 -autocolor off \
		   -background $color -tags [list BARCHART "id $barchart"]]
    }
    eval $c move $item [$barchart move]
    $barchart items $item
}

proc BARCHART::delete { barchart } {
    ::delete $barchart
}

proc BARCHART::size { barchart } {
    ::size $barchart BARCHART
}

proc BARCHART::move { barchart dx dy } {
    ::move $barchart $dx $dy
}

proc BARCHART::select { barchart } {
    ::select $barchart BARCHART
}

proc BARCHART::unselect { barchart } {
    ::unselect $barchart
}

proc BARCHART::color { barchart color } {
    ::color $barchart BARCHART $color
}

proc BARCHART::font { barchart font } {
    ::font $barchart BARCHART $font
}

proc BARCHART::raise { barchart } {
    ::raise $barchart
}

proc BARCHART::lower { barchart } {
    ::lower $barchart
}

proc BARCHART::clearlabel { barchart } {
    ::clearlabel $barchart
}

proc BARCHART::label { barchart text } {
    ::label $barchart BARCHART $text
}

proc BARCHART::resize { barchart x1 y1 x2 y2 } {
    set c [$barchart canvas]
    set item [$barchart items]
    $c coords $item $x1 $y1 $x2 $y2
}

proc BARCHART::clear { barchart } {
    $barchart values 0
}

proc BARCHART::values { barchart args } {
    set c [$barchart canvas]
    set item [$barchart items]
    if {$args == ""} {
	return [lindex [$c itemconfigure $item -values] 4]
    } else {
	$c itemconfigure $item -values [join $args]
    }
}

proc BARCHART::scale { barchart num } {
    set c [$barchart canvas]
    set item [$barchart items]
    $c itemconfigure $item -scalevalue $num
}

## -------------------------------------------------------------------------

proc GRAPH::canvas { graph } {
    set c [$graph canvas]
    $c element create $graph
}

proc GRAPH::delete { graph } {
    set c [$graph canvas]
    $c element delete $graph
}

proc GRAPH::select { graph } {
    set c [$graph canvas]
    $c element configure $graph -symbol circle
    $c element configure $graph -scale 0.5
    $c element configure $graph -linewidth 2
}

proc GRAPH::unselect { graph } {
    set c [$graph canvas]
    $c element configure $graph -symbol line
    $c element configure $graph -linewidth 0
}

proc GRAPH::color { graph color } {
    set c [$graph canvas]
    $c element configure $graph -foreground $color
}

proc GRAPH::icon { graph dashes } {
    [$graph canvas] element configure $graph -dashes $dashes
}

proc GRAPH::clearlabel { graph } {
    [$graph canvas] element configure $graph -label ""
}

proc GRAPH::label { graph text } {
    [$graph canvas] element configure $graph -label $text
}

proc GRAPH::clear { graph } {
    set c [$graph canvas]
    $c element delete $graph
    $c element create $graph
}

proc GRAPH::postscript { graph } {

    global tkined_ps_map
    set tkined_ps_map(fixed) [list Courier 10]

    set c [$graph canvas]

    $c postscript -fontmap tkined_ps_map	    
}

