/*
 * objects.c
 *
 * This file contains all functions that manipulate the state
 * of an editor object. An editor objects keeps track of all
 * information associated with a toplevel view.
 *
 * Copyright (c) 1993, 1994
 *
 * J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/stat.h>

#include "tkined.h"

Tcl_DString clip;                         /* the global clipboard */
static int force = 0;                     /* force a full dump */
static int num_editors = 0;               /* the number of editor objects */
static char *default_name = "noname.tki"; /* the default map name */

/*
 * Forward declarations for procedures defined later in this file:
 */

static int 
tkined_editor_command _ANSI_ARGS_((ClientData clientData, Tcl_Interp *interp,
				   int argc, char **argv));
static void
do_delete             _ANSI_ARGS_((tkined_editor *editor, Tcl_Interp *interp, 
				   Tcl_DString *dstp));
static void
do_dump               _ANSI_ARGS_((tkined_editor *editor, Tcl_Interp *interp, 
				   tkined_object *group, Tcl_DString *dstp));
static int 
do_ined               _ANSI_ARGS_((tkined_editor *editor, Tcl_Interp *interp,
				   char *line));
static int 
do_set                _ANSI_ARGS_((tkined_editor *editor, Tcl_Interp *interp,
				   char *line));
static void
e_default             _ANSI_ARGS_((tkined_editor *editor, Tcl_Interp *interp,
				   int argc, char **argv));
static void
e_readhistory         _ANSI_ARGS_((tkined_editor *editor, Tcl_Interp *interp));

static void
e_writehistory        _ANSI_ARGS_((tkined_editor *editor, Tcl_Interp *interp));

static void
expand_icon_name      _ANSI_ARGS_((tkined_editor *editor, Tcl_Interp *interp,
				   int type, char *str));

/*
 * Create a new editor object. No Parameters are expected.
 */

int 
create_editor (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    tkined_editor *editor;
    static unsigned lastid = 0;

    sprintf(buffer, "tkined%d", lastid++);

    if (argc != 1) {
	interp->result = "wrong # args";
        return TCL_ERROR;
    }

    editor = (tkined_editor *) xmalloc(sizeof(tkined_editor));

    editor->id = xstrdup(buffer);
    editor->toplevel = xstrdup("");

    editor->dirname = xstrdup("");
    editor->filename = xstrdup("");
    editor->psfilename = xstrdup("");

    editor->pagesize = xstrdup("");
    editor->landscape = 0;
    editor->width = 0; editor->height = 0;

    editor->traceCount = 0;

    Tcl_InitHashTable (&(editor->attr), TCL_STRING_KEYS);

    /* create a tcl command for the new object */

    Tcl_CreateCommand(interp, editor->id, tkined_editor_command, 
		      (ClientData) editor, delete_editor);    

    /* load the defaults from the tkined.defaults files */

    e_default (editor, interp, 0, (char **) NULL);

    /* read the old status back */

    e_readhistory (editor, interp);

    /* call the tk procedure to do initialization stuff */

    Tcl_VarEval (interp, "Editor::create ", editor->id, (char *) NULL);
    Tcl_ResetResult (interp);

    /* get the colormodel for this editor */

    Tcl_VarEval (interp, "tk colormodel ", editor->toplevel, (char *) NULL);
    editor->color = (strcmp (interp->result, "color") == 0);
    Tcl_ResetResult (interp);

    /* and initialize the current settings */

    e_new (editor, interp, 0, (char **) NULL);

    num_editors++;

    interp->result = editor->id;

    return TCL_OK;
}

/*
 * Delete an editor object. Free everything allocated before 
 * destroying the structure.
 */

void 
delete_editor (clientData)
     ClientData clientData;
{
    tkined_editor *editor = (tkined_editor *) clientData;
    Tcl_HashEntry *ht_entry;
    Tcl_HashSearch ht_search;

    num_editors--;

    free (editor->id);
    free (editor->toplevel);
    free (editor->dirname);
    free (editor->filename);
    free (editor->psfilename);
    free (editor->pagesize);

    ht_entry = Tcl_FirstHashEntry(&(editor->attr), &ht_search);
    while (ht_entry != NULL) {
	free ((char *) Tcl_GetHashValue (ht_entry));
	ht_entry = Tcl_NextHashEntry (&ht_search);
    }

    Tcl_DeleteHashTable (&(editor->attr));

    free ((char*) editor);
}

/*
 * Read the current config file. It contains the list of last used
 * files.
 */

static void
e_readhistory (editor, interp) 
    Tcl_Interp *interp;
    tkined_editor *editor;
{
    FILE *f;
    char *home = getenv ("HOME");
    char *argv[2];
    Tcl_DString dst;

    if (! home) return;

    Tcl_DStringInit (&dst);
    Tcl_DStringAppend (&dst, home, -1);
    Tcl_DStringAppend (&dst, "/.tkined/.history", -1);
    f = fopen (Tcl_DStringValue (&dst), "r");

    Tcl_DStringFree (&dst);
    if (f) {
        while (fgets(buffer, 1024, f) != NULL) {
	    int len = strlen (buffer);
	    if (buffer[len-1] == '\n') {
	        buffer[len-1] = '\0';
	    }
	    if (access (buffer, R_OK) == 0) {
		Tcl_DStringAppendElement (&dst, buffer);
	    }
	}
    }

    argv[0] = "history";
    argv[1] = Tcl_DStringValue (&dst);
    e_attribute (editor, interp, 2, argv);

    fclose (f);
    Tcl_DStringFree (&dst);
}

/*
 * Prepend the current editor file to the list of last used files.
 * Creates a $HOME/.tkined directory if none exists and writes to
 * $HOME/.tkined/.history . The max. size of the history is HISTSIZE.
 */

static void
e_writehistory (editor, interp) 
    Tcl_Interp *interp;
    tkined_editor *editor;
{
#define HISTSIZE 20
    FILE *f;
    char *fname, *home = getenv ("HOME");
    char *hist[HISTSIZE];
    int i = 0;

    if (! home) return;

    if (strcmp (editor->filename, default_name) == 0) return;

    for (i = 0; i < HISTSIZE; i++) hist[i] = NULL;
    
    fname = xmalloc (strlen(home)+30);
    strcpy (fname, home);
    strcat (fname, "/.tkined/.history");
    f = fopen (fname, "r");

    if (f) {
	i = 0;
	while ((fgets(buffer, 1024, f) != NULL) && (i < HISTSIZE)) {
	    int len = strlen (buffer);
	    if (buffer[len-1] == '\n') {
		buffer[len-1] = '\0';
	    }
	    hist[i++] = xstrdup (buffer);
	}
	fclose (f);
    }

    f = fopen (fname, "w+");
    if (! f) {
	strcpy (fname, home);
	strcat (fname, "/.tkined");
	mkdir (fname, 0755);
	strcat (fname, "/.history");
	f = fopen (fname, "w+");
    }

    if (f) {
	char *name = xmalloc (strlen(editor->dirname) 
			      + strlen(editor->filename) + 2);
	strcpy (name, editor->dirname);
	strcat (name, "/");
	strcat (name, editor->filename);

	fputs (name, f);
	fputs ("\n", f);
	
	for (i = 0; i < HISTSIZE; i++) {
	    if (hist[i] && (strcmp (hist[i], name) != 0)) {
		fputs (hist[i], f);
		fputs ("\n", f);
	    }
	}
	fclose (f);
    }

    free (fname);

    e_readhistory (editor, interp);
}

/*
 * Return the id of an editor object.
 */

int 
e_id (editor, interp, argc, argv)
    tkined_editor *editor;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    interp->result = editor->id;
    return TCL_OK;
}

/*
 * Return or set the toplevel window of an editor object.
 */

int 
e_toplevel (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{

    if (argc > 0 ) {
        STRCOPY (editor->toplevel, argv[0]);
	Tcl_VarEval (interp, "Editor::toplevel ", editor->id, (char *) NULL);
	Tcl_ResetResult (interp);
    }

    interp->result = editor->toplevel;
    return TCL_OK;
}

/*
 * Return the toplevel graph window used by graph objects.
 */

int 
e_graph (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{

    Tcl_VarEval (interp, "Editor::graph ", editor->id, (char *) NULL);
    return TCL_OK;
}

/*
 * Retrieve the objects that belong to this editor.
 */

int 
e_retrieve (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{
    Tcl_HashEntry *ht_entry;
    Tcl_HashSearch ht_search;
    tkined_object *object;

    ht_entry = Tcl_FirstHashEntry(&ht_object, &ht_search);
    while (ht_entry != NULL) {
	object = (tkined_object *) Tcl_GetHashValue (ht_entry);
	if (object->editor == editor) {
	    Tcl_AppendElement (interp, object->id);
	}
	ht_entry = Tcl_NextHashEntry (&ht_search);
    }

    return TCL_OK;
}

/*
 * Return or clear the current selection.
 */

int 
e_selection (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{
    Tcl_HashEntry *ht_entry;
    Tcl_HashSearch ht_search;
    tkined_object *object;
    int clear = 0;

    if (argc > 0 ) {
	if ((argv[0][0] == 'c') && (strcmp(argv[0], "clear") == 0)) {
	    clear++;
	}
    }

    ht_entry = Tcl_FirstHashEntry(&ht_object, &ht_search);
    while (ht_entry != NULL) {

	object = (tkined_object *) Tcl_GetHashValue (ht_entry);
	if (object->editor == editor) {
	    if (clear && object->selected) {
		m_unselect (interp, object, 0, (char **) NULL);
	    }
	    if (object->selected) {
		Tcl_AppendElement (interp, object->id);
	    }
	}

	ht_entry = Tcl_NextHashEntry (&ht_search);
    }

    return TCL_OK;
}

/*
 * Get or set the current directory name.
 */

int 
e_dirname (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{
    if (argc == 1) {
	STRCOPY (editor->dirname, argv[0]);
    }

    interp->result = editor->dirname;

    return TCL_OK;
}

/*
 * Get or set the current filename.
 */

int 
e_filename (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{
    if (argc == 1) {
	STRCOPY (editor->filename, argv[0]);
	Tcl_VarEval (interp, "Editor::filename ", editor->id, (char *) NULL);
	e_writehistory (editor, interp);
    }

    interp->result = editor->filename;

    return TCL_OK;
}

/*
 * Get or set the current postscript filename.
 */

int 
e_psfilename (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{
    if (argc == 1) {
	STRCOPY (editor->psfilename, argv[0]);
    }

    interp->result = editor->psfilename;

    return TCL_OK;
}

/*
 * Get or set the current pagesize. Convert the textual name in the
 * internal width and height in canvas pixels. 
 */

int 
e_pagesize (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{
    if (argc == 1) {

	if (strcmp (argv[0], "Letter") == 0) {
	    STRCOPY (editor->pagesize, "Letter");
	    editor->width  = 216;
	    editor->height = 279;
	} else if (strcmp (argv[0], "Legal") == 0) {
	    STRCOPY (editor->pagesize, "Legal");
            editor->width  = 216;
            editor->height = 256;
	} else if (strcmp (argv[0], "DINA3") == 0
		   || strcmp (argv[0], "A3") == 0) {
	    STRCOPY (editor->pagesize, "A3");
	    editor->width  = 297;
	    editor->height = 510;
	} else if (strcmp (argv[0], "DINA2") == 0
		   || strcmp (argv[0], "A2") == 0) {
	    STRCOPY (editor->pagesize, "A2");
	    editor->width  = 510;
	    editor->height = 594;
	} else if (strcmp (argv[0], "DINA1") == 0
		   || strcmp (argv[0], "A1") == 0) {
	    STRCOPY (editor->pagesize, "A1");
	    editor->width  = 594;
	    editor->height = 1020;
	} else {
	    STRCOPY (editor->pagesize, "A4");
	    editor->width  = 210;
	    editor->height = 297;
	}
	
	/* Flip the page if we should use landscape orientation. */
	if (editor->landscape) {
	    int tmp = editor->width;
	    editor->width = editor->height;
	    editor->height = tmp;
	}

	/* Convert millimeters in canvas pixels */
	sprintf (buffer, "winfo pixels %s %dm", 
		 editor->toplevel, editor->width);
	if (Tcl_Eval(interp, buffer) == TCL_OK) {
	    Tcl_GetInt (interp, interp->result, &editor->width);
	}
	sprintf (buffer, "winfo pixels %s %dm", 
		 editor->toplevel, editor->height);
	if (Tcl_Eval(interp, buffer) == TCL_OK) {
	    Tcl_GetInt (interp, interp->result, &editor->height);
	}

	sprintf (buffer, "Editor::pagesize %s %d %d",
		 editor->id, editor->width, editor->height);
	Tcl_Eval (interp, buffer);
	Tcl_ResetResult (interp);
    }

    interp->result = editor->pagesize;

    return TCL_OK;
}

/*
 * Get or set the current page orientation.
 */

int 
e_orientation (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{
    if (argc == 1) {

	if (strcmp(argv[0], "portrait") == 0) {
	    if (editor->landscape) {
		int tmp = editor->width;
		editor->width = editor->height;
		editor->height = tmp;
	    }
	    editor->landscape = 0;
	    sprintf (buffer, "Editor::pagesize %s %d %d",
		     editor->id, editor->width, editor->height);
	    Tcl_Eval (interp, buffer);
	} else {
	    if (! editor->landscape) {
		int tmp = editor->width;
                editor->width = editor->height;
                editor->height = tmp;
	    }
	    editor->landscape = 1;
	    sprintf (buffer, "Editor::pagesize %s %d %d",
		     editor->id, editor->width, editor->height);
	    Tcl_Eval (interp, buffer);
	}
    }

    if (editor->landscape) {
	interp->result = "landscape";
    } else {
	interp->result = "portrait";
    }

    return TCL_OK;
}

/*
 * Get or set a generic editor attribute. They are used to
 * store default values and arbitrary status information.
 */

int 
e_attribute (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{
    Tcl_HashEntry *ht_entry;

    if (argc == 0) return TCL_OK;

    if (argc == 2) {
	int isnew;

	ht_entry = Tcl_CreateHashEntry (&(editor->attr), argv[0], &isnew);
	if (! isnew) {
	    free ((char *) Tcl_GetHashValue (ht_entry));
	}
	Tcl_SetHashValue (ht_entry, xstrdup(argv[1]));
    }

    ht_entry = Tcl_FindHashEntry (&(editor->attr), argv[0]);
    if (ht_entry != NULL) {
	interp->result = (char *) Tcl_GetHashValue (ht_entry);
    } else {
	Tcl_ResetResult (interp);
    }

    if (debug) {
	if (argc == 2) {
	    fprintf (stderr, "++ %s attribute %s = %s\n", 
		     editor->id, argv[0], argv[1]);
	} else {
	    fprintf (stderr, "-- %s attribute %s (%s)\n",
		      editor->id, argv[0], interp->result);
	}
    }

    return TCL_OK;
}

/*
 * Delete all the objects given by the ids in dstp.
 */

static void
do_delete (editor, interp, dstp)
    Tcl_Interp *interp;
    tkined_editor *editor;
    Tcl_DString *dstp;
{
    int largc, i;
    char **largv;
    
    Tcl_SplitList (interp, Tcl_DStringValue(dstp), &largc, &largv);
    for (i = 0; i < largc; i++) {
	Tcl_VarEval (interp, largv[i], " delete", (char *) NULL);
	Tcl_ResetResult (interp);
    }

    free ((char *) largv);
}

/*
 * Copy the current selection on the clipboard and then call
 * do_delete to actually delete the selection.
 */

int 
e_cut (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{
    Tcl_HashEntry *ht_entry;
    Tcl_HashSearch ht_search;
    tkined_object *object;
    Tcl_DString dst;

    e_copy (editor, interp, argc, argv);

    Tcl_DStringInit (&dst);
 
    ht_entry = Tcl_FirstHashEntry(&ht_object, &ht_search);
    while (ht_entry != NULL) {
	
	object = (tkined_object *) Tcl_GetHashValue (ht_entry);
	if (object->editor == editor && object->selected) {
	    Tcl_DStringAppendElement (&dst, object->id);
	}

	ht_entry = Tcl_NextHashEntry (&ht_search);
    }

    do_delete (editor, interp, &dst);

    Tcl_DStringFree (&dst);

    return TCL_OK;
}

/*
 * Dump an object. Call the appropriate method and write the
 * text to the DString dstp. Group objects are saved recursively.
 * The done flag is used to prevent objects from being saved twice.
 */

static void
do_dump (editor, interp, object, dstp)
    Tcl_Interp *interp;
    tkined_editor *editor;
    tkined_object *object;
    Tcl_DString *dstp;
{

    if (object->done) return;

    switch (object->type) {
      case TKINED_NODE: 
	dump_object (interp, object);
	break;
      case TKINED_NETWORK:
	dump_object (interp, object);
	break;
      case TKINED_LINK:
        {
	    tkined_object *ob = id_to_object (object->src);
	    do_dump (editor, interp, ob, dstp);
	    ob = id_to_object (object->dst);
	    do_dump (editor, interp, ob, dstp);
        }
	dump_object (interp, object);
	break;
      case TKINED_GROUP:
        {
	    int margc, m;
	    char **margv;
	    Tcl_SplitList (interp, object->member, &margc, &margv);
	    for (m = 0; m < margc; m++) {
		tkined_object *member = id_to_object (margv[m]);
		do_dump (editor, interp, member, dstp);
	    }
	    free ((char *) margv);
        }
	dump_object (interp, object);
	break;
      case TKINED_TEXT:
	dump_object (interp, object);
	break;
      case TKINED_IMAGE:
	dump_object (interp, object);
	break;
      case TKINED_INTERPRETER:
	dump_object (interp, object);
	break;
      case TKINED_REFERENCE: 
	dump_object (interp, object);
	break;
      case TKINED_STRIPCHART: 
	dump_object (interp, object);
	break;
      case TKINED_BARCHART: 
	dump_object (interp, object);
	break;
      case TKINED_GRAPH: 
	dump_object (interp, object);
	break;
      default:
	Tcl_ResetResult (interp);
    }
    if (*interp->result != '\0') {
	Tcl_DStringAppend (dstp, interp->result, -1);
	Tcl_DStringAppend (dstp, "\n", 1);
    }

    object->done = 1;

    Tcl_ResetResult (interp);
}

/*
 * Copy the current selection or all objects belonging to this editor
 * (if force is set) in text form on the clipboard. The clipboard may 
 * be used later to save the textual representation in a file or to 
 * paste it in a different editor.
 */

int 
e_copy (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{
    Tcl_HashEntry *ht_entry;
    Tcl_HashSearch ht_search;
    tkined_object *object;

    Tcl_DStringInit (&clip);

    /* Set all the done flag of all objects to false. Ignore
       interpreter objects because they use the done flag to
       indicate a complete command buffer. */

    ht_entry = Tcl_FirstHashEntry(&ht_object, &ht_search);
    while (ht_entry != NULL) {
	object = (tkined_object *) Tcl_GetHashValue (ht_entry);
	if (object->editor == editor) {
	    if (object->type != TKINED_INTERPRETER) object->done = 0;
	}
	ht_entry = Tcl_NextHashEntry (&ht_search);
    }

    ht_entry = Tcl_FirstHashEntry(&ht_object, &ht_search);
    while (ht_entry != NULL) {
	object = (tkined_object *) Tcl_GetHashValue (ht_entry);
	if (object->editor == editor && (object->selected || force)) {
	    if (object->type != TKINED_INTERPRETER) {
		do_dump (editor, interp, object, &clip);
	    }
	}
	ht_entry = Tcl_NextHashEntry (&ht_search);
    }

    return TCL_OK;
}

/*
 * Paste the contents of the clipboard in the window. We `evaluate'
 * the text. A bit tricky because I did not want to write a real parser
 * for this stuff...
 */

static int
do_ined (editor, interp, line)
    tkined_editor *editor;
    Tcl_Interp *interp;
    char *line;
{
    int largc;
    char **largv;
    char *p;
    int i, res;
    tkined_object dummy;

    if (Tcl_SplitList (interp, line, &largc, &largv) != TCL_OK) {
	return TCL_ERROR;
    }
    Tcl_ResetResult (interp);
    
    /* expand variable names */
    
    for (i = 1; i < largc; i++) {
	if (largv[i][0] == '$') {
	    largv[i]++;
	    p = Tcl_GetVar (interp, largv[i], TCL_GLOBAL_ONLY);
	    if (p == NULL) {
		largv[i] = "";
	    } else {
		largv[i] = p;
	    }
	}
    }

    /* fake a dummy interpreter object to process the ined command */
    
    dummy.type = TKINED_INTERPRETER;
    dummy.id = "dummy";
    dummy.editor = editor;
    dummy.canvas = xmalloc(strlen(editor->toplevel)+8);
    strcpy (dummy.canvas, editor->toplevel);
    strcat (dummy.canvas, ".canvas");
    dummy.name = dummy.id;

    res = ined ((ClientData) &dummy, interp, largc, largv);

    free (dummy.canvas);

    return res;
}

static int
do_set (editor, interp, line)
    tkined_editor *editor;
    Tcl_Interp *interp;
    char *line;
{
    int len;
    char *var;

    line += 3;                                       /* skip the word set   */
    while (*line && isspace(*line)) line++;
    if (*line == '\0') return TCL_ERROR;

    var = line;
    while (*line && !isspace(*line)) line++;         /* extract the varname */
    if (*line == '\0') return TCL_ERROR;

    *line++ = '\0';                              /* beginning of the command */
    while (*line && isspace(*line)) line++;
    if (*line == '\0') return TCL_ERROR;

    if (*line != '[') return TCL_ERROR;         /* remove [ and skip spaces */
    line++;
    if (strncmp ("eval", line, 4) == 0)         /* backward compatibility   */
	    line += 4;
    while (*line && isspace(*line)) line++;
    if (*line == '\0') return TCL_ERROR;
    
    len = strlen(line)-1;                       /* remove trailing spaces   */
    while (len > 0 && isspace(line[len])) {
	line[len] = '\0';
	len--;
    }
    if (len == 0) return TCL_ERROR;

    if (line[len] != ']') return TCL_ERROR;     /* remove ]                 */
    line[len] = '\0';

    if (do_ined (editor, interp, line) == TCL_OK) {
	line = Tcl_SetVar (interp, var, interp->result, TCL_GLOBAL_ONLY);
	if (line) {
	    tkined_object *object = id_to_object (interp->result);
	    if (object) object->loaded = 1;
	    return TCL_OK;
	}
    }

    return TCL_ERROR;
}

int 
e_paste (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{
    char *line, *freeme, *p;

    freeme = xstrdup (Tcl_DStringValue(&clip));
    line = freeme;

    while (1) {

	/* find the end of the line if any left */
	p = line;
	while ( *p != '\n' && *p != '\0') p++;
	if (*p == '\0') break;
	*p = '\0';

	/* first remove leading white space characters */
	while (*line && isspace(*line)) line++;
	
	/* evaluate the line ignoring comments and empty lines */
	if ( (*line != '\0') && (*line != '#')) {

	    if (strncmp("set", line, 3) == 0) {
		do_set (editor, interp, line);
	    } else if (strncmp("ined", line, 4) == 0) {
		do_ined (editor, interp, line);
	    } else {
		fprintf (stderr, "** e_paste unknown: %s\n", line);
	    }

	}

	line = ++p;
    }

    free (freeme);

    return TCL_OK;
}

/*
 * Reinitialize the editor. Delete all objects currently associated
 * with this editor.
 */

int
e_new (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{
    Tcl_HashEntry *ht_entry;
    Tcl_HashSearch ht_search;
    tkined_object *object;
    Tcl_DString dst;
    char *cwd;

    Tcl_DStringInit (&dst);

    ht_entry = Tcl_FirstHashEntry (&ht_object, &ht_search);
    while (ht_entry != NULL) {

        object = (tkined_object *) Tcl_GetHashValue (ht_entry);
        if (object->editor == editor
	    && object->type != TKINED_LOG 
	    && object->type != TKINED_MENU
	    && (object->type != TKINED_INTERPRETER
		|| (object->type == TKINED_INTERPRETER && object->loaded))) {

	    Tcl_DStringAppendElement (&dst, object->id);
	}

        ht_entry = Tcl_NextHashEntry (&ht_search);
    }

    do_delete (editor, interp, &dst);

    Tcl_DStringFree (&dst);

    e_filename (editor, interp, 1, &default_name);

    if (! (cwd = getcwd ((char *) NULL, 1024))) cwd = "";
    e_dirname (editor, interp, 1, &cwd);

    return TCL_OK;
}

/*
 * Read the defaults from all possible tkined.defaults files. Store
 * the default values in editor attributes and lookup icon references.
 */

static void
expand_icon_name (editor, interp, type, str)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int type;
    char *str;
{
    char *icon = str;
    char *name = str;
    char *argv[2];
    char *path;

    while (*name && !isspace(*name)) name++;
    if (*name == '\0') return;

    *name = '\0';
    name++;

    while (*name && isspace(*name)) name++;
    if (*name == '\0') return;

    /* skip leading elements that are separated with colons */

    { 
	char *p;
	for (p = name; *p != '\0'; p++) {
	    if (*p == ':') {
		name = p+1;
	    }
	}
    }

    /* make our own copy since findfile will reuse the buffer space */

    icon = xstrdup (icon);
    name = xstrdup (name);
    argv[0] = xmalloc (strlen(name)+20);
    argv[1] = NULL;

    if (type == TKINED_NETWORK) {
	strcpy (argv[0], "NETWORK-icon-");
	strcat (argv[0], name);
	argv[1] = xstrdup (icon);
	e_attribute (editor, interp, 2, argv);
    }

    if (type == TKINED_GRAPH) {
	strcpy (argv[0], "GRAPH-icon-");
	strcat (argv[0], name);
	argv[1] = xstrdup (icon);
	e_attribute (editor, interp, 2, argv);
    }

    path = findfile (icon);
    if (path != NULL) {
	argv[1] = xmalloc (strlen(path)+2);
	argv[1][0] = '@';
	strcpy (argv[1]+1, path);

	if (type == TKINED_NODE) {
	    int len;
	    strcpy (argv[0], "NODE-icon-");
	    strcat (argv[0], name);
	    e_attribute (editor, interp, 2, argv);
	    
	    /* needed to ensure backward compatibility */
	    
	    free (argv[0]);
	    argv[0] = xmalloc (strlen(icon)+15);
	    strcpy (argv[0], "NODE-icon-");
	    strcat (argv[0], icon);
	    e_attribute (editor, interp, 2, argv);
	    
	    if ((len = strlen (argv[0])) > 3
		&& argv[0][len-3] == '.'
		&& argv[0][len-2] == 'b' 
		&& argv[0][len-1] == 'm') {
		argv[0][len-3] = '\0';
		e_attribute (editor, interp, 2, argv);
	    }
	} 
	
	if (type == TKINED_GROUP) {
	    int len;
	    strcpy (argv[0], "GROUP-icon-");
	    strcat (argv[0], name);
	    e_attribute (editor, interp, 2, argv);
	    
	    /* needed to ensure backward compatibility */
	    
	    free (argv[0]);
	    argv[0] = xmalloc (strlen(icon)+15);
	    strcpy (argv[0], "GROUP-icon-");
	    strcat (argv[0], icon);
	    e_attribute (editor, interp, 2, argv);
	    
	    if ((len = strlen (argv[0])) > 3
		&& argv[0][len-3] == '.'
		&& argv[0][len-2] == 'b' 
		&& argv[0][len-1] == 'm') {
		argv[0][len-3] = '\0';
		e_attribute (editor, interp, 2, argv);
	    }
	}

	if (type == TKINED_REFERENCE) {
	    int len;
	    strcpy (argv[0], "REFERENCE-icon-");
	    strcat (argv[0], name);
	    e_attribute (editor, interp, 2, argv);
	    
	    /* needed to ensure backward compatibility */
	    
	    free (argv[0]);
	    argv[0] = xmalloc (strlen(icon)+20);
	    strcpy (argv[0], "REFERENCE-icon-");
	    strcat (argv[0], icon);
	    e_attribute (editor, interp, 2, argv);
	    
	    if ((len = strlen (argv[0])) > 3
		&& argv[0][len-3] == '.'
		&& argv[0][len-2] == 'b' 
		&& argv[0][len-1] == 'm') {
		argv[0][len-3] = '\0';
		e_attribute (editor, interp, 2, argv);
	    }
	}
    }

    free (argv[0]);
    if (argv[1] != NULL) free (argv[1]);
    free (name);
    free (icon);
}

/*
 * Read the defaults file (searching along tkined_path) and set
 * up the editor attributes.
 */

static void
do_readdefaults (editor, interp, filename)
    tkined_editor *editor;
    Tcl_Interp *interp;
    char *filename;
{
    FILE *f;

    if (filename == NULL) return;

    if ((f = fopen(filename, "r")) == NULL) return;
    
    while (fgets(buffer, 1024, f) != NULL) {
	int len;
	char *p, *value;
	char *name = buffer;
	char *largv[2];
	
	while (*name && isspace(*name)) name++;
	if (*name == '\0') continue;
	
	/* ignore lines starting with a comment sign or       */
	/* entries that do not start with a "tkined." prefix. */
	
	if (*name == '#' 
	    || *name == '!' 
	    || strlen(name) < 8 
	    || (strncmp(name, "tkined.", 7) != 0)) continue;
	
	name += 7;
	p = name;
	while (*p && *p != ':') p++;
	if (*p == '\0') continue;
	
	*p = '\0';
	value = ++p;
	
	while (*value && isspace(*value)) value++;
	
	len = strlen(value)-1;           /* remove trailing spaces   */
	while (len > 0 && isspace(value[len])) {
	    value[len] = '\0';
	    len--;
	}
	
	largv[0] = name;
	largv[1] = value;
	e_attribute (editor, interp, 2, largv);
	
	/* expand icon bitmap file names and add expanded entries */
	
	if (strncmp (name, "node", 4) == 0) {
	    expand_icon_name (editor, interp, TKINED_NODE, value);
	} else if (strncmp (name, "group", 5) == 0) {
	    expand_icon_name (editor, interp, TKINED_GROUP, value);
	} else if (strncmp (name, "network", 7) == 0) {
	    expand_icon_name (editor, interp, TKINED_NETWORK, value);
	} else if (strncmp (name, "dashes", 6) == 0) {
	    expand_icon_name (editor, interp, TKINED_GRAPH, value);
	} else if (strncmp (name, "reference", 9) == 0) {
	    expand_icon_name (editor, interp, TKINED_REFERENCE, value);
	}
    }
    
    fclose(f);
}

static void
e_default (editor, interp, argc, argv)
    tkined_editor *editor;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    char *fname;
    char *path;

    fname = xmalloc (strlen(TKINEDLIB)+30);
    strcpy (fname, TKINEDLIB);
    strcat (fname, "/tkined.defaults");
    do_readdefaults (editor, interp, findfile (fname));
    strcpy (fname, TKINEDLIB);
    strcat (fname, "/site/tkined.defaults");
    do_readdefaults (editor, interp, findfile (fname));
    free (fname);

    do_readdefaults (editor, interp, findfile ("~/.tkined/tkined.defaults"));
    do_readdefaults (editor, interp, findfile ("tkined.defaults"));

    if ((path = getenv("TKINED_PATH")) != NULL) {
	char *p, *s;
	path = xstrdup (path);
	for (s = p = path; *p; p++) {
	    if (*p == ':') {
		*p = '\0';
		fname = xmalloc (strlen(s)+20);
		strcpy (fname, s);
		strcat (fname, "/");
		strcat (fname, "tkined.defaults");
		do_readdefaults (editor, interp, findfile (fname));
		free (fname);
		s = ++p;
	    }
	}
	if (*s) {
	    fname = xmalloc (strlen(s)+20);
	    strcpy (fname, s);
	    strcat (fname, "/");
	    strcat (fname, "tkined.defaults");
	    do_readdefaults (editor, interp, findfile (fname));
	    free (fname);
	}
	free (path);
    }
}

/*
 * Load all objects in previously saved in a file. 
 */

int 
e_load (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{
    FILE *f;
    Tcl_DString tmp;
    int isok;
    char *p;

    if (argc != 1) {
	interp->result = "wrong # args";
        return TCL_ERROR;
    }

    if ((f = fopen(argv[0], "r")) == NULL) {
	Tcl_PosixError (interp);
	return TCL_ERROR;
    }

    tmp = clip;
    Tcl_DStringInit (&clip);

    /* check if the header looks like a tkined file */

    isok = 0;
    if (fgets(buffer, 1024, f) != NULL) {
	Tcl_DStringAppend (&clip, buffer, -1);
	if (fgets(buffer, 1024, f) != NULL) {
	    Tcl_DStringAppend (&clip, buffer, -1);
	    for (p = buffer; *p; p++) {
		if (strncmp (p, "tkined version", 14) == 0) {
		    isok = 1;
		    break;		  
		}
	    }
	}
    }

    if (! isok) {
	Tcl_DStringFree (&clip);
	clip = tmp;
	Tcl_SetResult (interp, "not a valid tkined save file", TCL_STATIC);
	return TCL_ERROR;
    }
    
    while (fgets(buffer, 1024, f) != NULL) {
	Tcl_DStringAppend (&clip, buffer, -1);
    }
    fclose(f);
    e_paste (editor, interp, 0, (char **) NULL);
    Tcl_DStringFree (&clip);
    clip = tmp;

    return TCL_OK;
}

/*
 * Save all objects in a file. We copy the whole stuff in good
 * old interviews tradition on the clipboard and write it to a file.
 */

int 
e_save (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{
    FILE *f;
    Tcl_DString tmp;
    Tcl_HashEntry *ht_entry;
    Tcl_HashSearch ht_search;
    tkined_object *object;

    if (argc != 1) {
	interp->result = "wrong # args";
        return TCL_ERROR;
    }

    if ((f = fopen(argv[0], "w")) == NULL) {
	Tcl_PosixError (interp);
	return TCL_ERROR;
    }

    fprintf (f, "##\n## This file was created by tkined version %s\t%s\n",
	     TKINED_VERSION, ">> DO NOT EDIT <<");
    fputs ("##\n", f);
    fputs ("## This may look like TCL code but it is definitely not!\n", f);
    fputs ("##\n\n", f);

    fprintf (f, "ined page %s %s\n\n", editor->pagesize,
	     editor->landscape ? "landscape" : "portrait");

    force = 1;
    tmp = clip;
    e_copy (editor, interp, 0, (char **) NULL);
    if (fputs (Tcl_DStringValue(&clip), f) == EOF) {
	Tcl_PosixError (interp);
	return TCL_ERROR;
    }
    clip = tmp;
    force = 0;

    /* save interpreters with running jobs */

    ht_entry = Tcl_FirstHashEntry(&ht_object, &ht_search);
    while (ht_entry != NULL) {
	object = (tkined_object *) Tcl_GetHashValue (ht_entry);
	if (object->editor == editor && (object->type == TKINED_INTERPRETER)) {
	    if (strlen(object->action) != 0) {
		dump_object (interp, object);
		fputs (interp->result, f);
		fputs ("\n", f);
		Tcl_ResetResult (interp);
	    }
	}
	ht_entry = Tcl_NextHashEntry (&ht_search);
    }

    fclose (f);

    return TCL_OK;
}


/*
 * Delete an editor. Delete all objects that belong to it.
 * We must restart the loop since the hash table may get 
 * clobbered when an object is deleted.
 */

int 
e_delete (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{
    Tcl_HashEntry *ht_entry;
    Tcl_HashSearch ht_search;
    tkined_object *object;

    ht_entry = Tcl_FirstHashEntry(&ht_object, &ht_search);
    while (ht_entry != NULL) {
	object = (tkined_object *) Tcl_GetHashValue (ht_entry);
	if (object->editor == editor) {
	    m_delete (interp, object, 0, (char **) NULL);
	    ht_entry = Tcl_FirstHashEntry(&ht_object, &ht_search);
	}
	ht_entry = Tcl_NextHashEntry (&ht_search);
    }
    
    Tcl_VarEval (interp, "Editor::delete ", editor->id, (char *) NULL);

    Tcl_DeleteCommand (interp, editor->id);

    if (num_editors == 0) e_quit (editor, interp, 0, (char **) NULL);

    return TCL_OK;
}

/*
 * Quit the whole application. This is mainly here as a hook 
 * for future enhancements.
 */

int 
e_quit (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{
    return Tcl_Eval (interp, "destroy .");
}

/*
 * Returns a PostScript dump of the current map.
 */

int
e_postscript (editor, interp, argc, argv)
    Tcl_Interp *interp;
    tkined_editor *editor;
    int argc;
    char **argv;
{
    return Tcl_VarEval (interp, "Editor::postscript ", 
			editor->id, (char *) NULL);

}


/*
 * All methods are dispatched using the following table. Depending
 * on the type and the name of an object, we choose the function
 * to call. The type TKINED_ALL matches any type.
 */

struct tkined_method {
    char *cmd;
    int (*fnx)(); /* (tkined_editor *, Tcl_Interp*, int, char**) */
};

static struct tkined_method tkined_methods[] = {

        { "id",          e_id },
	{ "toplevel",    e_toplevel },
	{ "graph",       e_graph },
	{ "retrieve",    e_retrieve },
	{ "selection",   e_selection },

	{ "dirname",     e_dirname },
	{ "filename",    e_filename },
	{ "psfilename",  e_psfilename },
	{ "pagesize",    e_pagesize },
	{ "orientation", e_orientation },

	{ "attribute",   e_attribute },

	{ "cut",         e_cut },
	{ "copy",        e_copy },
	{ "paste",       e_paste },

	{ "new",         e_new },
	{ "load",        e_load },
	{ "save",        e_save },
	{ "delete",      e_delete },
	{ "quit",        e_quit },

	{ "postscript",  e_postscript },

        { 0, 0 }
    };

/*
 * Process a method of an editor object. Check the table for an
 * appropriate entry and call the desired function. We have an 
 * error if no entry matches.
 */

static int
tkined_editor_command (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    tkined_editor *editor = (tkined_editor *) clientData;
    struct tkined_method *ds;

    if (argc < 2) {
	interp->result = "wrong # args";
	return TCL_ERROR;
    }

    if (strcmp(editor->id, argv[0]) != 0) {
	fprintf (stderr, "** fatal error: %s has illegal id %s\n", 
		 argv[0], editor->id);
	fprintf (stderr, "** while doing: %s %s\n", argv[0], argv[1]);
    }

    for (ds = tkined_methods; ds->cmd; ds++) {
	int res;

	if ((argv[1][0] != *(ds->cmd)) || (strcmp(argv[1], ds->cmd) != 0)) 
		continue;

	res = (ds->fnx)(editor, interp, argc-2, argv+2);
	return res;
    }

    sprintf (interp->result, "%s can not %s", editor->id, argv[1]);

    return TCL_ERROR;
}
