/*
 * methods.c
 *
 * This file contains all functions that manipulate the state
 * of tkined objects. These functions get called whenever a user
 * invokes a command, when applying a tool, or when a command is send
 * from an interpreter. Every method calls a corresponding tk 
 * procedure that is responsible to manipulate the display according
 * to the changes. The naming convention is that a command like `move'
 * is implemented as the method m_move (written in C) which calls
 * <type>::move to let tk take the appropriate action.
 *
 * Copyright (c) 1993, 1994
 *
 * J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <sys/types.h>
#ifdef HAVE_SOCKETPAIR
#include <sys/socket.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <netdb.h>
#include <pwd.h>
#include <sys/time.h>

#include "tkined.h"

/*
 * Forward declarations for procedures defined later in this file:
 */

static void
dump_move             _ANSI_ARGS_((Tcl_Interp *interp, 
				   tkined_object *object));
static void
dump_icon             _ANSI_ARGS_((Tcl_Interp *interp,
				   tkined_object *object));
static void
dump_name             _ANSI_ARGS_((Tcl_Interp *interp, 
				   tkined_object *object));
static void
dump_address          _ANSI_ARGS_((Tcl_Interp *interp, 
				   tkined_object *object));
static void
dump_oid              _ANSI_ARGS_((Tcl_Interp *interp, 
				   tkined_object *object));
static void
dump_font             _ANSI_ARGS_((Tcl_Interp *interp, 
				   tkined_object *object));
static void
dump_color            _ANSI_ARGS_((Tcl_Interp *interp, 
				   tkined_object *object));
static void
dump_label            _ANSI_ARGS_((Tcl_Interp *interp, 
				   tkined_object *object));
static void
dump_scale            _ANSI_ARGS_((Tcl_Interp *interp, 
				   tkined_object *object));
static void
dump_size             _ANSI_ARGS_((Tcl_Interp *interp, 
				   tkined_object *object));
static void
dump_attributes       _ANSI_ARGS_((Tcl_Interp *interp,
                                   tkined_object *object));

static void
parent_resize         _ANSI_ARGS_((Tcl_Interp *interp, 
				   tkined_object *object));
static void
m_network_link_end    _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *network,
				   double *sx, double *sy));
static int 
m_link_update         _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object,
				   int argc, char **argv));

static int 
m_node_create         _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object,
				   int argc, char **argv));
static int 
m_group_create        _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object,
				   int argc, char **argv));
static int 
m_network_create      _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object,
				   int argc, char **argv));
static int 
m_link_create         _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object,
				   int argc, char **argv));
static int 
m_text_create         _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object,
				   int argc, char **argv));
static int 
m_image_create        _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object,
				   int argc, char **argv));
static int 
m_interpreter_create  _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object,
				   int argc, char **argv));
static int 
m_menu_create         _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object,
				   int argc, char **argv));
static int 
m_log_create          _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object,
				   int argc, char **argv));
static int 
m_ref_create          _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object,
				   int argc, char **argv));
static int 
m_strip_create        _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object,
				   int argc, char **argv));
static int 
m_bar_create          _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object,
				    int argc, char **argv));
static int 
m_graph_create        _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object,
				    int argc, char **argv));

static int 
m_node_retrieve       _ANSI_ARGS_((Tcl_Interp* interp, tkined_object *object,
				   int argc, char** argv));
static int 
m_group_retrieve      _ANSI_ARGS_((Tcl_Interp* interp, tkined_object *object,
				   int argc, char** argv));
static int 
m_network_retrieve    _ANSI_ARGS_((Tcl_Interp* interp, tkined_object *object,
				   int argc, char** argv));
static int 
m_link_retrieve       _ANSI_ARGS_((Tcl_Interp* interp, tkined_object *object,
				   int argc, char** argv));
static int 
m_text_retrieve       _ANSI_ARGS_((Tcl_Interp* interp, tkined_object *object,
				   int argc, char** argv));
static int 
m_image_retrieve      _ANSI_ARGS_((Tcl_Interp* interp, tkined_object *object,
				   int argc, char** argv));
static int 
m_interpreter_retrieve _ANSI_ARGS_((Tcl_Interp* interp, tkined_object *object,
				    int argc, char** argv));
static int 
m_menu_retrieve       _ANSI_ARGS_((Tcl_Interp* interp, tkined_object *object,
				   int argc, char** argv));
static int 
m_log_retrieve        _ANSI_ARGS_((Tcl_Interp* interp, tkined_object *object,
				   int argc, char** argv));
static int 
m_ref_retrieve        _ANSI_ARGS_((Tcl_Interp* interp, tkined_object *object,
				   int argc, char** argv));
static int 
m_strip_retrieve      _ANSI_ARGS_((Tcl_Interp* interp, tkined_object *object,
				   int argc, char** argv));
static int 
m_bar_retrieve        _ANSI_ARGS_((Tcl_Interp* interp, tkined_object *object,
				   int argc, char** argv));
static int 
m_graph_retrieve      _ANSI_ARGS_((Tcl_Interp* interp, tkined_object *object,
				   int argc, char** argv));

static void 
m_linked_delete       _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object,
				   int argc, char **argv));
static void 
m_link_delete         _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object,
				   int argc, char **argv));
static void 
m_group_delete        _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object,
				   int argc, char **argv));
static void 
m_interpreter_delete  _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object,
				   int argc, char **argv));

static int 
m_node_dump        _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object));

static int 
m_group_dump       _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object));

static int
m_network_dump     _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object));

static int
m_link_dump        _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object));

static int
m_text_dump        _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object));

static int
m_image_dump       _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object));

static int
m_interpreter_dump _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object));

static int
m_ref_dump         _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object));

static int
m_strip_dump       _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object));

static int
m_bar_dump         _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object));

static int
m_graph_dump       _ANSI_ARGS_((Tcl_Interp *interp, tkined_object *object));


/*
 * The following set of functions is used to write object state
 * into the iterpreter result string. They are used to dump objects
 * into an ascii string.
 */

static void
dump_move (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    sprintf (buffer, "ined -noupdate move $%s %.2f %.2f\n",
	     object->id, object->x, object->y);
    Tcl_AppendResult (interp, buffer, (char *)NULL);
}

static void
dump_icon (interp, object)
     Tcl_Interp *interp;
     tkined_object *object;
{
    if (*(object->icon) != '\0') {

	char *p = strrchr (object->icon, '/');
    
        Tcl_AppendResult (interp, "ined -noupdate icon $",
			  object->id, (char *) NULL);
	Tcl_AppendElement (interp, (p) ? ++p : object->icon);
	Tcl_AppendResult (interp, "\n", (char *) NULL);
    }
}

static void
dump_name (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    Tcl_AppendResult (interp, "ined -noupdate name $", 
		      object->id, (char *) NULL);
    Tcl_AppendElement (interp, object->name);
    Tcl_AppendResult (interp, "\n", (char *) NULL);
}

static void
dump_address (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    if (*(object->address) != '\0') {
        Tcl_AppendResult (interp, "ined -noupdate address $",
			  object->id, (char *) NULL);
	Tcl_AppendElement (interp, object->address);
	Tcl_AppendResult (interp, "\n", (char *) NULL);
    }
}
    
static void
dump_oid (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    if (object->oid != 0) {
	sprintf (buffer, "ined -noupdate oid $%s \"%d\"\n",
		 object->id, object->oid);
	Tcl_AppendResult (interp, buffer, (char *)NULL);
    }
}

static void
dump_font (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    Tcl_AppendResult (interp, "ined -noupdate font $", 
		      object->id, (char *) NULL);
    Tcl_AppendElement (interp, object->font);
    Tcl_AppendResult (interp, "\n", (char *) NULL);
}

static void
dump_color (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    if (strlen(object->color) > 0 && strcmp(object->color, "black") != 0) {
        Tcl_AppendResult (interp, "ined -noupdate color $",
			  object->id, (char *) NULL);
        Tcl_AppendElement (interp, object->color);
        Tcl_AppendResult (interp, "\n", (char *) NULL);
    }
}

static void
dump_label (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    Tcl_AppendResult (interp, "ined -noupdate label $", object->id, 
		      (char *) NULL);
    Tcl_AppendElement (interp, object->label);
    Tcl_AppendResult (interp, "\n", (char *) NULL);
}

static void
dump_scale (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    sprintf (buffer, "ined -noupdate scale $%s %.2f\n", 
	     object->id, object->scale);
    Tcl_AppendResult (interp, buffer, (char *) NULL);
}

static void
dump_size (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    Tcl_AppendResult (interp, "ined -noupdate size $", object->id, " ", 
		      object->size, "\n", (char *) NULL);
}

static void
dump_attributes (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    Tcl_HashEntry *ht_entry;
    Tcl_HashSearch ht_search;

    ht_entry = Tcl_FirstHashEntry(&(object->attr), &ht_search);
    while (ht_entry != NULL) {
	Tcl_AppendResult (interp, "ined -noupdate attribute $", object->id,
			  (char *) NULL);
	Tcl_AppendElement (interp, 
			   Tcl_GetHashKey (&(object->attr), ht_entry));
	Tcl_AppendElement (interp, (char *) Tcl_GetHashValue (ht_entry));
	Tcl_AppendResult (interp, "\n", (char *) NULL);
	ht_entry = Tcl_NextHashEntry (&ht_search);
    }
}

/* 
 * Update the size of an expanded parent objects.
 */
      
static void
parent_resize (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    char *tmp = "reset";

    if (object->parent != '\0') {
        tkined_object *parent = id_to_object (object->parent);
	if ((parent != NULL) && (! (parent->collapsed))) {
	    Tcl_VarEval (interp, type_to_string (parent->type),
			 "::resize ", parent->id, (char *) NULL);
	    m_label (interp, parent, 1, &tmp);
	    parent_resize (interp, parent);
	}
    }
}



/*
 * This utility function gets the coordinates of a network
 * and puts them into two arrays named x and y. The next step 
 * is to find a network segment which allows us to draw a horizontal 
 * or vertical line. If this fails, we connect to the endpoint that 
 * is closest to our initial position.
 */

static void 
m_network_link_end (interp, network, sx, sy)
    Tcl_Interp *interp;
    tkined_object *network;
    double *sx;
    double *sy;
{
    int found = 0;
    int i, j, n;
    int largc;
    char **largv;
    double *x;
    double *y;
    double rx = 0, ry = 0;
    double d = -1;

    Tcl_SplitList (interp, network->member, &largc, &largv);

    x = (double *) xmalloc (largc * sizeof(double));
    y = (double *) xmalloc (largc * sizeof(double));

    if (x == NULL || y == NULL) {
	free ((char *) largv);
	return;
    }

    for (n = 0, i = 0; i < largc; i++) {
	if ((i%2) == 0) {
	    Tcl_GetDouble (interp, largv[i], &x[n]);
	    x[n] += network->x;
	} else {
	    Tcl_GetDouble (interp, largv[i], &y[n]);
	    y[n] += network->y;
	    n++;
	}
    }

    for (i = 1, j = 0, found = 0; i < n; i++, j++) {

	double lo_x = (x[i] < x[j]) ? x[i] : x[j];
	double up_x = (x[i] < x[j]) ? x[j] : x[i];
	double lo_y = (y[i] < y[j]) ? y[i] : y[j];
        double up_y = (y[i] < y[j]) ? y[j] : y[i];

	if ((lo_x <= *sx) && (up_x >= *sx)) {
	    double nd = (*sy > y[i]) ? *sy - y[i] : y[i] - *sy;
	    if (d < 0 || nd < d ) {
		rx = *sx; ry = y[i];
		d = nd;
		found++;
	    }
	}

	if ((lo_y <= *sy) && (up_y >= *sy)) {
	    double nd = (*sx > x[i]) ? *sx - x[i] : x[i] - *sx;
	    if (d < 0 || nd < d ) {
                rx = x[i]; ry = *sy;
                d = nd;
		found++;
	    }
        }
    }

    /* If we can not make a horizontal or vertical link
       or if one of the fixed points is much nearer, we 
       simply make a link to the nearest fixed point */
    
    for (i = 0; i < n; i++) {	
	double nd = ((x[i] > *sx) ? x[i] - *sx : *sx - x[i])
		  + ((y[i] > *sy) ? y[i] - *sy : *sy - y[i]);
	if (!found || (nd < d)) {
	    rx = x[i]; ry = y[i];
	    d = nd;
	    found++;
	}
    }
    
    free ((char *) x);
    free ((char *) y);
    free ((char *) largv);

    *sx = rx; *sy = ry;

    return;
}

/*
 * Update the link position. This must get called whenever the link
 * moves or one of the connected objects moves.
 */

static int 
m_link_update (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    char *tmp;
    char buf_a[20], buf_b[20];
    int n;
    double *x;
    double *y;
    tkined_object *src = NULL, *dst = NULL;
    int selected = object->selected;
    double x_a, y_a, x_b, y_b;

    if (selected) {
        m_unselect (interp, object, 0, (char **) NULL);
    }

    /* Get the objects linked by this link object. Search for 
       parent objects if there is currently no canvas. */

    for (tmp = object->src, src = id_to_object (tmp);
	 src != NULL && (strlen (src->canvas) == 0);
	 src = id_to_object (tmp))
      tmp = src->parent;

    for (tmp = object->dst, dst = id_to_object (tmp);
	 dst != NULL && (strlen (dst->canvas) == 0);
	 dst = id_to_object (tmp))
	    tmp = dst->parent;

    if (src == NULL || dst == NULL) {
	Tcl_SetResult (interp, "update link: can not find linked objects",
		       TCL_STATIC);
	return TCL_ERROR;
    }

    if (src->type == TKINED_NETWORK) {
	x_a = dst->x;
	y_a = dst->y;
	m_network_link_end (interp, src, &x_a, &y_a);
    } else {
	x_a = src->x;
	y_a = src->y;
    }

    if (dst->type == TKINED_NETWORK) {
	x_b = src->x;
	y_b = src->y;
	m_network_link_end (interp, dst, &x_b, &y_b);
    } else {
	x_b = dst->x;
	y_b = dst->y;
    }

    /* handle fixed points if any */

    tmp = NULL;
    if (strlen(object->member) > 0) {
	int i,largc;
	char **largv;
	Tcl_SplitList (interp, object->member, &largc, &largv);

	if (largc > 0) {
	    x = (double *) xmalloc (largc * sizeof(double));
	    y = (double *) xmalloc (largc * sizeof(double));
	
	    if (x == NULL || y == NULL) {
		free ((char*) largv);
		Tcl_ResetResult (interp);
		sprintf (interp->result, "%f %f", object->x, object->y);
		return TCL_OK;
	    }
	    
	    for (n = 0, i = 0; i < largc; i++) {
		if ((i%2) == 0) {
		    Tcl_GetDouble (interp, largv[i], &x[n]);
		    x[n] += object->x;
		} else {
		    Tcl_GetDouble (interp, largv[i], &y[n]);
		    y[n] += object->y;
		    n++;
		}
	    }
	    
	    if (x[0] == x[1]) {
	        y[0] = y_a;
	    } else {
	        x[0] = x_a;
	    }
	    
	    if (x[n-1] == x[n-2]) {
	        y[n-1] = y_b;
	    } else {
	        x[n-1] = x_b;
	    }

	    if (n == 1) {
	        x[0] = x_a;
	        y[0] = y_b;
	    }

	    tmp = xmalloc (n*32);
	    *tmp = 0;
	    for (i = 0; i < n; i++) {
		sprintf (buffer, "%.2f %.2f ", x[i], y[i]);
		strcat (tmp, buffer);
	    }

	    free ((char *) x);
	    free ((char *) y);
	}
	free ((char *) largv);
    }

    sprintf (buf_a, "%.2f %.2f ", x_a, y_a);
    sprintf (buf_b, "%.2f %.2f ", x_b, y_b);

    Tcl_VarEval (interp, "foreach item [", object->id, " items] {",
		 "if {[", object->canvas, " type $item]==\"line\"} break }; ",
		 "eval ", object->canvas, " coords $item ", 
		 buf_a, (tmp == NULL) ? "" : tmp, buf_b,
		 (char *) NULL);

    if (tmp != NULL) free (tmp);

    if (selected) {
        m_select (interp, object, 0, (char **) NULL);
    }
    return TCL_OK;
}


/*
 * Create a NODE object. Initialize the id and name fields.
 */

static int 
m_node_create (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    static unsigned lastid = 0;

    sprintf(buffer, "node%d", lastid++);
    STRCOPY (object->id, buffer);
    STRCOPY (object->name, buffer);

    trace (object->editor, (tkined_object *) NULL, 
	   "ined create NODE", argc, argv, object->id);

    return TCL_OK;
}

/*
 * Create a GROUP object. Initialize the id and name fields.
 * Set the member to the ids given in argv.
 */

static int 
m_group_create (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    static unsigned lastid = 0;

    sprintf(buffer, "group%d", lastid++);
    STRCOPY (object->id, buffer);
    STRCOPY (object->name, buffer);
    object->collapsed = 0;

    m_member (interp, object, argc, argv);

    trace (object->editor, (tkined_object *) NULL, 
	   "ined create GROUP", argc, argv, object->id);
    
    return TCL_OK;
}

/*
 * Create a NETWORK object. Initialize the id and name fields.
 * Store the points in the member field and set the position
 * to the first x and y coordinates.
 */

static int 
m_network_create (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    static unsigned lastid = 0;

    sprintf(buffer, "network%d", lastid++);
    STRCOPY (object->id, buffer);
    STRCOPY (object->name, buffer);

    if (argc > 1) {
	int i;
	Tcl_GetDouble (interp, argv[0], &(object->x));
	Tcl_GetDouble (interp, argv[1], &(object->y));
	buffersize (argc*12);
	*buffer = '\0';
	for (i = 0; i < (argc/2)*2; i++) {
	    char tmp[20];
	    double val;
	    Tcl_GetDouble (interp, argv[i++], &val);
	    sprintf (tmp, "%f ", val - object->x);
	    strcat (buffer, tmp);
	    Tcl_GetDouble (interp, argv[i], &val);
	    sprintf (tmp, "%f ", val - object->y);
	    strcat (buffer, tmp);
	}
	STRCOPY (object->member, buffer);
    } else {
	/* default length and position */
	STRCOPY (object->member, "0 0 130 0");
	object->x = 50;
	object->y = 50;
    }

    trace (object->editor, (tkined_object *) NULL, 
	   "ined create NETWORK", argc, argv, object->id);

    return TCL_OK;
}

/*
 * Create a LINK object. Initialize the id and name fields.
 * Initialize the fields that contain the id of the connected
 * objects.
 */

static int 
m_link_create (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    static unsigned lastid = 0;
    tkined_object *obj;

    if (argc < 2) {
        Tcl_SetResult (interp, "wrong # of args", TCL_STATIC);
	return TCL_ERROR;
    }

    sprintf(buffer, "link%d", lastid++);
    STRCOPY (object->id, buffer);
    STRCOPY (object->name, buffer);
    STRCOPY (object->src, argv[0]);
    STRCOPY (object->dst, argv[1]);

    if (argc > 3) {
	char *freeme = Tcl_Merge (argc-2, argv+2);
	STRCOPY (object->member, freeme);
	free (freeme);
    }

    if ((obj = id_to_object (object->src)) != NULL) {
	lappend (&obj->links, object->id);
    }

    if ((obj = id_to_object (object->dst)) != NULL) {
	lappend (&obj->links, object->id);
    }

    trace (object->editor, (tkined_object *) NULL, 
	   "ined create LINK", argc, argv, object->id);
    
    return TCL_OK;
}

/*
 * Create a TEXT object. Initialize the id and name fields.
 * Map any newline characters to the sequence \n.
 */

static int 
m_text_create (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    static unsigned lastid = 0;

    if (argc == 1) {
	sprintf(buffer, "text%d", lastid++);
	STRCOPY (object->id, buffer);
	
	m_text (interp, object, 1, &argv[0]);

	trace (object->editor, (tkined_object *) NULL, 
	       "ined create TEXT", argc, argv, object->id);
    }

    return TCL_OK;
}

/*
 * Create an IMAGE object. Initialize the id and name fields,
 * which contains the path to the bitmap file.
 */

static int 
m_image_create (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    static unsigned lastid = 0;
    char *file;

    if (argc < 1) {
        Tcl_SetResult (interp, "wrong # of args", TCL_STATIC);
	return TCL_ERROR;
    }

    file = findfile (argv[0]);
    if (file == NULL) {
	file = argv[0];
    }

    STRCOPY (object->name, file);
    sprintf(buffer, "image%d", lastid++);
    STRCOPY (object->id, buffer);

    trace (object->editor, (tkined_object *) NULL, 
	   "ined create IMAGE", argc, argv, object->id);

    return TCL_OK;
}

/*
 * Create an INTERPRETER object. Initialize the id and name fields.
 * Initialize a command buffer and fork a process to do the real work.
 */

static int 
m_interpreter_create (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    static unsigned lastid = 0;
#ifndef HAVE_SOCKETPAIR
    int fd1 [2], fd2 [2];
#endif

    int pid;
    FILE *in;
    char *p;
    char *argw[4];
    char *file;

    if (argc < 1) {
        Tcl_SetResult (interp, "wrong # of args", TCL_STATIC);
	return TCL_ERROR;
    }

    if ((file = findfile (argv[0])) == (char *) NULL) {
	Tcl_ResetResult (interp);
	Tcl_AppendResult (interp, argv[0], " not found", (char *) NULL);
	m_delete (interp, object, 0, (char **) NULL);
        return TCL_ERROR;
    }

    STRCOPY (object->name, file);
    sprintf(buffer, "interpreter%d", lastid++);
    STRCOPY (object->id, buffer);
    Tcl_DStringInit (&object->buffer);
    object->done = 1;
    
#ifdef HAVE_SOCKETPAIR
    if (socketpair (AF_UNIX, SOCK_STREAM, 0, object->xv) < 0) {
	object->xv[0] = object->xv[1] = -1;
	Tcl_ResetResult (interp);
	Tcl_AppendResult (interp, "unable to get socket pair: ",
			  Tcl_PosixError (interp), (char *) NULL);
	return TCL_ERROR;
    }
#else
    if (pipe (fd1) < 0 || pipe (fd2) < 0) {
	object->xv[0] = object->xv[1] = -1;
	Tcl_ResetResult (interp);
	Tcl_AppendResult (interp, "unable to create pipes: ",
			  Tcl_PosixError (interp), (char *) NULL);
	return TCL_ERROR;
    }
#endif

    pid = fork();
    if (pid < 0) {
	object->xv[0] = object->xv[1] = -1;
	Tcl_ResetResult (interp);
        Tcl_AppendResult (interp, "unable to fork process: ",
                          Tcl_PosixError (interp), (char *) NULL);
	return TCL_ERROR;
    }

    object->pid = pid;

    if (pid == 0) {                                        /* child */

	int i;

#ifdef HAVE_SOCKETPAIR
	close (object->xv [0]);
	dup2  (object->xv [1], 0);
	dup2  (object->xv [1], 1);
#else
	dup2  (fd1 [0], 0);
	dup2  (fd2 [1], 1);
#endif

	/* close all open files that we do not need anymore */

	for (i = 3; i < 64; i++) close (i);

	/* start #! scripts by hand to allow long pathes */
	
	if (((in = fopen(object->name, "r")) != NULL) 
	    && (buffer == fgets(buffer, 512, in))) {
	    if ( buffer[0] == '#' && buffer[1] == '!') {
		for (p=buffer+2; isspace(*p); p++) ;
		argw[0] = p;
		while (*p && !isspace(*p)) p++;
		while (*p && isspace(*p)) *p++ = 0;
		argw[1] = p;
		while (*p && !isspace(*p)) p++;
		*p = 0;
		argw[2] = object->name;
		argw[3] = NULL;

		if (strlen(argw[1]) == 0) {
		    argw[1] = argw[2]; argw[2] = NULL;
		}

		fclose (in);
		execv (argw[0], argw);
		
		fprintf (stderr, "Can not execute %s\n", argw[0]);
		printf ("ined acknowledge {Can not execute %s}\n", argw[0]);
		fflush (stdout);
		fgetc(stdin);
		exit (1);
	    }
	    fclose (in);
	}
	
	execl (object->name, object->name, (char *) 0);
	fprintf (stdout, "ined acknowledge {Can not execute %s}\n", 
		 object->name);
	fflush (stdout);
	fgetc(stdin);
	exit (1);
    } else {                                            /* parent */
#ifdef HAVE_SOCKETPAIR
	close (object->xv [1]);
	object->xv [1] = object->xv [0];
#else
	object->xv [0] = fd2 [0];		/* read fd */
	object->xv [1] = fd1 [1];		/* write fd */
#endif
	Tk_CreateFileHandler(object->xv[0], TK_READABLE,
			     receive, (ClientData) object);

	Tcl_DetachPids (1, &pid);

	trace (object->editor, (tkined_object *) NULL, 
	       "ined create INTERPRETER", argc, argv, object->id);
    }

    return TCL_OK;
}

/*
 * Create a MENU object. Initialize the id and name fields.
 * The items field contains the commands of the menu and 
 * the links field contains the id of the interpreter object.
 */

static int 
m_menu_create (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    static unsigned lastid = 0;
    char *freeme;

    if (argc < 2) {
        Tcl_SetResult (interp, "wrong # of args", TCL_STATIC);
	return TCL_ERROR;
    }

    sprintf (buffer, "menu%d", lastid++);
    STRCOPY (object->id, buffer);
    STRCOPY (object->name, argv[0]);
    freeme = Tcl_Merge (argc-1, argv+1);
    STRCOPY (object->items, freeme);
    free (freeme);

    return TCL_OK;
}

/*
 * Create a LOG object (window). Initialize the id and name fields.
 */

static int
m_log_create (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    struct passwd *pwd;
    static unsigned lastid = 0;

    sprintf(buffer, "log%d", lastid++);
    STRCOPY (object->id, buffer);
    STRCOPY (object->name, buffer);

    pwd = getpwuid(getuid());
    STRCOPY (object->address, pwd->pw_name);

    trace (object->editor, (tkined_object *) NULL, 
	   "ined create LOG", argc, argv, object->id);

    return TCL_OK;
}

/*
 * Create a REFERENCE object pointing to another tkined map. 
 * Initialize the id.
 */

static int
m_ref_create (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    static unsigned lastid = 0;

    sprintf(buffer, "reference%d", lastid++);
    STRCOPY (object->id, buffer);

    trace (object->editor, (tkined_object *) NULL, 
	   "ined create REFERENCE", argc, argv, object->id);

    return TCL_OK;
}

/*
 * Create a STRIPCHART object showing statistics in a stripchart.
 */

static int
m_strip_create (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    static unsigned lastid = 0;

    sprintf(buffer, "stripchart%d", lastid++);
    STRCOPY (object->id, buffer);

    trace (object->editor, (tkined_object *) NULL, 
	   "ined create STRIPCHART", argc, argv, object->id);

    return TCL_OK;
}

/*
 * Create a BARCHART object showing statistics in a barchart.
 */

static int
m_bar_create (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    static unsigned lastid = 0;

    sprintf(buffer, "barchart%d", lastid++);
    STRCOPY (object->id, buffer);

    trace (object->editor, (tkined_object *) NULL, 
	   "ined create BARCHART", argc, argv, object->id);

    return TCL_OK;
}

/*
 * Create a GRAPH which uses the BLT graph widget to allow some
 * simple form of data analysis.
 */

static int
m_graph_create (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    static unsigned lastid = 0;

    sprintf(buffer, "graph%d", lastid++);
    STRCOPY (object->id, buffer);

    trace (object->editor, (tkined_object *) NULL, 
	   "ined create GRAPH", argc, argv, object->id);

    return TCL_OK;
}

/*
 * The create method just dispatches to one of those object type
 * specific functions.
 */

int
m_create (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    switch (object->type) {
 case TKINED_NODE: 
	return m_node_create (interp, object, argc, argv);
 case TKINED_GROUP: 
	return m_group_create (interp, object, argc, argv);
 case TKINED_NETWORK: 
	return m_network_create (interp, object, argc, argv);
 case TKINED_LINK:
	return m_link_create (interp, object, argc, argv);
 case TKINED_TEXT:
	return m_text_create (interp, object, argc, argv);
 case TKINED_IMAGE:
	return m_image_create (interp, object, argc, argv);
 case TKINED_INTERPRETER:
	return m_interpreter_create (interp, object, argc, argv);
 case TKINED_MENU:
	return m_menu_create (interp, object, argc, argv);
 case TKINED_LOG:
	return m_log_create (interp, object, argc, argv);
 case TKINED_REFERENCE:
	return m_ref_create (interp, object, argc, argv);
 case TKINED_STRIPCHART:
	return m_strip_create (interp, object, argc, argv);
 case TKINED_BARCHART:
	return m_bar_create (interp, object, argc, argv);
 case TKINED_GRAPH:
	return m_graph_create (interp, object, argc, argv);
    }
    return TCL_OK;
}

/*
 * Retrieve the external representation of a NODE object.
 */

static int 
m_node_retrieve (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    sprintf (buffer, "%u", object->oid);
    Tcl_AppendElement (interp, "NODE");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->name);
    Tcl_AppendElement (interp, object->address);
    Tcl_AppendElement (interp, buffer);
    Tcl_AppendElement (interp, object->links);

    return TCL_OK;
}

/*
 * Retrieve the external representation of a GROUP object.
 */

static int 
m_group_retrieve (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{    
    sprintf (buffer, "%u", object->oid);
    Tcl_AppendElement (interp, "GROUP");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->name);
    Tcl_AppendElement (interp, buffer);
    Tcl_AppendElement (interp, object->member);

    return TCL_OK;
}

/*
 * Retrieve the external representation of a NETWORK object.
 */

static int
m_network_retrieve (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{    
    sprintf (buffer, "%u", object->oid);
    Tcl_AppendElement (interp, "NETWORK");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->name);
    Tcl_AppendElement (interp, object->address);
    Tcl_AppendElement (interp, buffer);
    Tcl_AppendElement (interp, object->links);

    return TCL_OK;
}

/*
 * Retrieve the external representation of a LINK object.
 */

static int
m_link_retrieve (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_AppendElement (interp, "LINK");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->src);
    Tcl_AppendElement (interp, object->dst);

    return TCL_OK;
}

/*
 * Retrieve the external representation of a TEXT object.
 */

static int
m_text_retrieve (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_AppendElement (interp, "TEXT");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->text);

    return TCL_OK;
}

/*
 * Retrieve the external representation of a IMAGE object.
 */

static int
m_image_retrieve (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_AppendElement (interp, "IMAGE");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->name);

    return TCL_OK;
}

/*
 * Retrieve the external representation of a INTERPRETER object.
 */

static int
m_interpreter_retrieve (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_AppendElement (interp, "INTERPRETER");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->name);

    return TCL_OK;
}

/*
 * Retrieve the external representation of a MENU object.
 */

static int
m_menu_retrieve (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_AppendElement (interp, "MENU");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->name);
    Tcl_AppendElement (interp, object->items);

    return TCL_OK;
}

/*
 * Retrieve the external representation of a LOG object.
 */

static int
m_log_retrieve (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_AppendElement (interp, "LOG");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->name);
    Tcl_AppendElement (interp, object->address);

    return TCL_OK;
}

/*
 * Retrieve the external representation of a REFERENCE object.
 */

static int
m_ref_retrieve (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_AppendElement (interp, "REFERENCE");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->name);
    Tcl_AppendElement (interp, object->address);

    return TCL_OK;
}

/*
 * Retrieve the external representation of a STRIPCHART object.
 */

static int
m_strip_retrieve (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_AppendElement (interp, "STRIPCHART");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->name);
    Tcl_AppendElement (interp, object->address);

    return TCL_OK;
}

/*
 * Retrieve the external representation of a BARCHART object.
 */

static int
m_bar_retrieve (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_AppendElement (interp, "BARCHART");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->name);
    Tcl_AppendElement (interp, object->address);

    return TCL_OK;
}

/*
 * Retrieve the external representation of a GRAPH object.
 */

static int
m_graph_retrieve (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_AppendElement (interp, "GRAPH");
    Tcl_AppendElement (interp, object->id);
    Tcl_AppendElement (interp, object->name);
    Tcl_AppendElement (interp, object->address);

    return TCL_OK;
}

/*
 * Retrieve the external representation of an object.
 */

int
m_retrieve (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    switch (object->type) {
 case TKINED_NODE: 
	return m_node_retrieve (interp, object, argc, argv);
 case TKINED_GROUP: 
	return m_group_retrieve (interp, object, argc, argv);
 case TKINED_NETWORK: 
	return m_network_retrieve (interp, object, argc, argv);
 case TKINED_LINK:
	return m_link_retrieve (interp, object, argc, argv);
 case TKINED_TEXT:
	return m_text_retrieve (interp, object, argc, argv);
 case TKINED_IMAGE:
	return m_image_retrieve (interp, object, argc, argv);
 case TKINED_INTERPRETER:
	return m_interpreter_retrieve (interp, object, argc, argv);
 case TKINED_MENU:
	return m_menu_retrieve (interp, object, argc, argv);
 case TKINED_LOG:
	return m_log_retrieve (interp, object, argc, argv);
 case TKINED_REFERENCE:
	return m_ref_retrieve (interp, object, argc, argv);
 case TKINED_STRIPCHART:
	return m_strip_retrieve (interp, object, argc, argv);
 case TKINED_BARCHART:
	return m_bar_retrieve (interp, object, argc, argv);
 case TKINED_GRAPH:
	return m_graph_retrieve (interp, object, argc, argv);
    }

    return TCL_OK;
}

/*
 * Return the type of an object.
 */

int
m_type (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_SetResult (interp, type_to_string (object->type), TCL_STATIC);

    return TCL_OK;
}

/*
 * Return the id of an object.
 */

int 
m_id (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_SetResult (interp, object->id, TCL_STATIC);

    return TCL_OK;
}

/*
 * Return the parent of an object.
 */

int
m_parent (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_SetResult (interp, object->parent, TCL_STATIC);

    return TCL_OK;
}

/*
 * Get and set the name of an object. Call the tk callback if the label 
 * is set to show the name.
 */

int
m_name (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (argc == 1) {

	free (object->name);
	object->name = xstrdupnn (argv[0]);

	if (object->type == TKINED_LOG) {
	    sprintf (buffer, "%s::name %s",
		     type_to_string (object->type), object->id);
	    Tcl_Eval (interp, buffer);
	}

	if (strcmp(object->label, "name") == 0) {
	    notrace (m_label, interp, object, 1, &object->label);
	}

	trace (object->editor, object, 
	       "ined name", argc, argv, object->name);
    }

    Tcl_SetResult (interp, object->name, TCL_STATIC);
    return TCL_OK;
}

/*
 * Return the canvas of an object.
 */

int 
m_canvas (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (argc > 0 ) {

        STRCOPY (object->canvas, argv[0]);

	if (strlen(object->canvas) > 0) {
	    Tcl_VarEval (interp, type_to_string (object->type),
			 "::canvas ", object->id, (char *) NULL);
	    if (object->type == TKINED_LINK) {
		m_link_update (interp, object, 0, (char **) NULL);
	    }

	    if (object->scale != 0) {
		char *tmp = xmalloc(80);
		sprintf (tmp, "%f", object->scale);
		m_scale (interp, object, 1, &tmp);
		free (tmp);
	    }
	}

	if (object->type == TKINED_LINK) {
	    m_lower (interp, object, 0, (char **) NULL);
	}

	/* Update all links connected to this NODE or NETWORK object */

	if (object->type == TKINED_NODE || object->type == TKINED_NETWORK) {
	    int largc, i;
	    char **largv;
	    tkined_object *link;
	    Tcl_SplitList (interp, object->links, &largc, &largv);
	    for (i=0; i<largc; i++) {
	        link = id_to_object (largv[i]);
		if (link != NULL) {
		    m_link_update (interp, link, 0, (char **) NULL);
		    
		}
	    }
	    free ((char*) largv);
	}

    }

    Tcl_SetResult (interp, object->canvas, TCL_STATIC);
    return TCL_OK;
}

/*
 * Get or set the editor of this object. This implementation
 * does not really check if the argument is a real editor object.
 */

int
m_editor (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{

    if (argc == 1) {

	int dotrace = (object->editor == NULL);

	Tcl_CmdInfo info;
	if (Tcl_GetCommandInfo (interp, argv[0], &info) > 0) {
	    object->editor = (tkined_editor *) info.clientData;
	}

	if (dotrace) {
	    trace (object->editor, (tkined_object *) NULL, 
		   (char *) NULL, 0 , (char **) NULL, (char *) NULL);
	}
    }

    if (object->editor != NULL) {
	Tcl_SetResult (interp, object->editor->id, TCL_STATIC);
    } else {
	Tcl_ResetResult (interp);
    }

    return TCL_OK;
}

/*
 * Get and set the tk canvas items belonging to an object.
 */

int
m_items (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (argc == 1) {
        STRCOPY (object->items, argv[0]);
    }
    
    Tcl_SetResult (interp, object->items, TCL_STATIC);
    return TCL_OK;
}

/*
 * Get and set the address of an object. Call the tk callback if the
 * label is set to show the address.
 */

int
m_address (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (argc == 1) {

	free (object->address);
	object->address = xstrdupnn (argv[0]);
	
	if (strcmp(object->label, "address") == 0) {
	    notrace (m_label, interp, object, 1, &object->label);
	}

	trace (object->editor, object, 
	       "ined address", argc, argv, object->address);
    }

    Tcl_SetResult (interp, object->address, TCL_STATIC);
    return TCL_OK;
}


/*
 * Get and set the oid of an object. There is no callback here.
 */

int
m_oid (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    int result;
    
    if (argc == 1) {

	if (Tcl_GetInt (interp, argv[0], &result) != TCL_OK) 
		return TCL_ERROR;

	object->oid = result;

	trace (object->editor, object, 
	       "ined oid", argc, argv, argv[0]);
    }

    Tcl_ResetResult (interp);
    sprintf (interp->result, "%d", object->oid);
    return TCL_OK;
}

/*
 * Get and set the actions bound to an tkined object.
 */

int
m_action (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (argc == 1) {

	STRCOPY (object->action, argv[0]);

	trace (object->editor, object, 
	       "ined action", argc, argv, object->action);
    }

    Tcl_SetResult (interp, object->action, TCL_STATIC);
    return TCL_OK;
}

/*
 * Select an object. Mark it as selected and call the tk
 * procedure to actually do the selection. Skip objects
 * that are invisible (that is have no canvas).
 */

int
m_select (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if ((!object->selected) && (*(object->canvas) != '\0')) {
	object->selected = 1;
	
	Tcl_VarEval (interp, type_to_string (object->type),
		     "::select ", object->id, (char *) NULL);
    }

    Tcl_ResetResult (interp);
    if (object->editor) 
	    e_selection (object->editor, interp, 0, (char **) NULL);
    return TCL_OK;
}

/*
 * Unselect on object. Mark it as unselected and call the tk
 * procedure to actually do the unselection.
 */

int
m_unselect (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (object->selected) {
	object->selected = 0;
	
	Tcl_VarEval (interp, type_to_string (object->type),
		     "::unselect ", object->id, (char *) NULL);
    }

    Tcl_ResetResult (interp);
    if (object->editor) 
	    e_selection (object->editor, interp, 0, (char **) NULL);
    return TCL_OK;
}

/*
 * Return a boolean indicating if the object is selected or not.
 */

int
m_selected (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_ResetResult (interp);
    sprintf (interp->result, "%d", object->selected);
    return TCL_OK;
}

/*
 * Get and set the icon of an object. Call tkined_<TYPE>_icon
 * to let tk update the canvas.
 */

int
m_icon (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    char *tmp = "reset";

    int selected = object->selected;

    if (argc == 1) {

	tkined_editor *editor = object->editor;

	buffersize (strlen(argv[0])+40);
	sprintf (buffer, "%s-icon-%s", 
		 type_to_string (object->type), argv[0]);

	e_attribute (editor, interp, 1, &buffer);
	if (*interp->result != '\0') {
	    STRCOPY (object->icon, interp->result);
	} else {
	    if (argv[0][0] != '\0') {
		STRCOPY (object->icon, argv[0]);
	    } else {
		switch (object->type) {
	    case TKINED_NODE:
		    STRCOPY (object->icon, "node");
		    break;
	    case TKINED_GROUP:
		    STRCOPY (object->icon, "group");
		    break;
	    case TKINED_NETWORK:
		    STRCOPY (object->icon, "network");
		    break;
	    case TKINED_LOG:
		    STRCOPY (object->icon, "");
		    break;
	    case TKINED_REFERENCE:
		    STRCOPY (object->icon, "reference");
		    break;
	    case TKINED_GRAPH:
		    STRCOPY (object->icon, "solid");
		    break;
		}
	    }
	}
	Tcl_ResetResult (interp);

	if (object->type == TKINED_NETWORK) {
	    int width;
	    if (Tcl_GetInt (interp, object->icon, &width) != TCL_OK) {
		STRCOPY (object->icon, "3");
	    }
	}

	if (object->type == TKINED_GRAPH) {
	    int width;
	    if (Tcl_GetInt (interp, object->icon, &width) != TCL_OK) {
		STRCOPY (object->icon, "0");
	    }
	}

	if (selected) {
	    m_unselect (interp, object, 0, (char **) NULL);
	}

        Tcl_VarEval (interp, type_to_string (object->type), "::icon ", 
		     object->id, " ", object->icon, (char *) NULL);

      	notrace (m_label, interp, object, 1, &tmp);

	/* adjust the size of a parent group if necessary */
      
	parent_resize (interp, object);

	if (selected) {
	    m_select (interp, object, 0, (char **) NULL);
	}

	trace (object->editor, object, 
	       "ined icon", argc, argv, object->icon);
    }
    
    Tcl_SetResult (interp, object->icon, TCL_STATIC);
    return TCL_OK;
}

/*
 * Get and set the label of an object. Call tkined_<TYPE>_label
 * to let tk update the canvas.
 */

int
m_label (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (argc > 0) {

	if ( strcmp(argv[0], "clear") == 0 ) {
	    STRCOPY (object->label, argv[0]);
	    Tcl_VarEval (interp, type_to_string (object->type),
			 "::clearlabel ", object->id, (char *) NULL);
	    Tcl_ResetResult (interp);
	    trace (object->editor, object, 
		   "ined label", argc, argv, (char *) NULL);
	} else if ( strcmp(argv[0], "reset") == 0) {
	    Tcl_VarEval (interp, type_to_string (object->type),
			 "::clearlabel ", object->id, (char *) NULL);
	    Tcl_ResetResult (interp);
	    notrace (m_label, interp, object, 1, &object->label);
	} else {
	    char *txt = NULL;
	    if (strcmp(argv[0], "name") == 0) { 
		txt = object->name;
	    } else if (strcmp(argv[0], "address") == 0 ) {
		txt = object->address;
	    } else {
		Tcl_HashEntry *ht_entry;
		ht_entry = Tcl_FindHashEntry (&(object->attr), argv[0]);
		if (ht_entry) {
		    txt = (char *) Tcl_GetHashValue (ht_entry);
		}
	    }
	    if (txt != NULL) {
	        STRCOPY (object->label, argv[0]);
		/* XXX fix this: this quoting breaks any " characters */
		Tcl_VarEval (interp, type_to_string (object->type),
			     "::label ", object->id, " \"", txt, "\"",
			     (char *) NULL);
		Tcl_ResetResult (interp);
		trace (object->editor, object,
                       "ined label", argc, argv, (char *) NULL);
	    }
	}
    }

    Tcl_SetResult (interp, object->label, TCL_STATIC);
    return TCL_OK;
}

/*
 * Get and set the font of an object. Call <TYPE>::font
 * to let tk update the canvas.
 */

int
m_font (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    int selected = (object->selected && object->type == TKINED_TEXT);

    if (argc == 1) {

	tkined_editor *editor = object->editor;

	buffersize (strlen(argv[0])+8);
	sprintf (buffer, "font-%s", argv[0]);

	e_attribute (editor, interp, 1, &buffer);
	if (*interp->result != '\0') {
	    STRCOPY (object->font, interp->result);
	} else {
	    if (argv[0][0] != '\0') {
		STRCOPY (object->font, argv[0]);
	    } else {
		STRCOPY (object->font, "fixed");
	    }
	}
	Tcl_ResetResult (interp);

	if (selected) {
	    m_unselect (interp, object, 0, (char **) NULL);
	}
	
	Tcl_VarEval (interp, type_to_string (object->type), 
		     "::font ", object->id, " ", object->font, 
		     (char *) NULL);

	if (selected) {
            m_select (interp, object, 0, (char **) NULL);
        }

	trace (object->editor, object, 
	       "ined font", argc, argv, object->font);
    }

    Tcl_SetResult (interp, object->font, TCL_STATIC);
    return TCL_OK;
}

/*
 * Get and set the color of an object. Call <TYPE>::color to update the 
 * view. We convert to X11 color name here which makes more sense and is
 * much faster.
 */

int
m_color (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (argc == 1) {

	tkined_editor *editor = object->editor;

	buffersize (strlen(argv[0])+8);
	sprintf (buffer, "color-%s", argv[0]);

	e_attribute (editor, interp, 1, &buffer);
	if (*interp->result != '\0') {
	    STRCOPY (object->color, interp->result);
	} else {
	    if (argv[0][0] != '\0') {
		STRCOPY (object->color, argv[0]);
	    } else {
		STRCOPY (object->color, "black");
	    }
	}
	Tcl_ResetResult (interp);

	if (editor->color) {
	    Tcl_VarEval (interp, type_to_string (object->type),
			 "::color ", object->id, " ", object->color,
			 (char *) NULL);
	} else {
	    Tcl_VarEval (interp, type_to_string (object->type),
			 "::color ", object->id, " black", (char *) NULL);
	}

	trace (object->editor, object,
	       "ined color", argc, argv, object->color);
    }

    Tcl_SetResult (interp, object->color, TCL_STATIC);
    return TCL_OK;
}

/*
 * Move an object. Return the new position and after calling
 * the tk callback <TYPE>::move.
 */

int
m_move (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (argc == 2) {

	double x, y;
	int largc, i;
	char **largv;
	char buf[40];

	if (Tcl_GetDouble (interp, argv[0], &x) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (Tcl_GetDouble (interp, argv[1], &y) != TCL_OK) {
	    return TCL_ERROR;
	}

	/* don't move objects outside of the canvas */

	if (object->editor) {
	    if (object->x+x < 0) x = - object->x;
	    if (object->y+y < 0) y = - object->y;
	    if (object->x+x > object->editor->width)
		    x = object->editor->width - object->x;
	    if (object->y+y > object->editor->height) 
		    y = object->editor->height - object->y;
	}

	object->x += x;
	object->y += y;

	if ( *(object->canvas) != '\0') {
	    if (object->type == TKINED_LINK) {
		m_link_update (interp, object, 0, (char **) NULL);
	    } else {
		sprintf (buffer, "%s::move %s %f %f", 
			 type_to_string (object->type), object->id, x, y);
		Tcl_Eval (interp, buffer);
	    }
	}

	/* move all members of a group object */

	if (object->type == TKINED_GROUP) {
	    int margc;
	    char **margv;

	    Tcl_SplitList (interp, object->member, &margc, &margv);
	    for (i = 0; i < margc; i++) {
		tkined_object *member = id_to_object (margv[i]);
		notrace (m_move, interp, member, argc, argv);
	    }
	    free ((char*) margv);
	}

	/* now update all visible links to this object */

	Tcl_SplitList (interp, object->links, &largc, &largv);
	for (i = 0; i < largc; i++) {
	    tkined_object *link;
	    link = id_to_object (largv[i]);
	    if (link != NULL && *(link->canvas) != '\0') {
		m_link_update (interp, link, 0, (char **) NULL);
	    }
	}
	free ((char*) largv);

	/* adjust the size of a parent group if necessary */
      
	parent_resize (interp, object);
      
	sprintf (buf, "%f %f", object->x, object->y);
	trace (object->editor, object, 
	       "ined move", argc, argv, buf);
    }

    Tcl_ResetResult (interp);
    sprintf (interp->result, "%f %f", object->x, object->y);
    return TCL_OK;
}

/*
 * Raise all items belonging to an object.
 */

int
m_raise (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_VarEval (interp, type_to_string (object->type),
		 "::raise ", object->id, (char *) NULL);

    trace (object->editor, object, 
	   "ined raise", argc, argv, (char *) NULL);

    Tcl_SetResult (interp, object->id, TCL_STATIC);
    return TCL_OK;
}

/*
 * Lower all items belonging to an object. Make sure that images
 * are always in the background. The loop through all objects
 * should be optimized.
 */

int
m_lower (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{

    Tcl_VarEval (interp, type_to_string (object->type),
		 "::lower ", object->id, (char *) NULL);

    if (object->type != TKINED_IMAGE) {
	Tcl_HashEntry *ht_entry;
	Tcl_HashSearch ht_search;

	ht_entry = Tcl_FirstHashEntry(&ht_object, &ht_search);
	while (ht_entry != NULL) {
	    tkined_object *any;
	    any = (tkined_object *) Tcl_GetHashValue (ht_entry);
	    if (any->type == TKINED_IMAGE) {
		notrace (m_lower, interp, any, 0, (char **) NULL);
	    }
	    ht_entry = Tcl_NextHashEntry (&ht_search);
	}
    }

    trace (object->editor, object, 
	   "ined lower", argc, argv, (char *) NULL);

    Tcl_SetResult (interp, object->id, TCL_STATIC);
    return TCL_OK;
}

/*
 * Return the size of an object by the coordinates of the bounding box.
 */

int
m_size (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    int ret;

    if (argc == 4 && (object->type & (TKINED_STRIPCHART | TKINED_BARCHART))) {
	int selected = object->selected;
	char *tmp = "reset";
	double x1, y1, x2, y2;
	
	if (Tcl_GetDouble (interp, argv[0], &x1) != TCL_OK) return TCL_ERROR;
	if (Tcl_GetDouble (interp, argv[1], &y1) != TCL_OK) return TCL_ERROR;
	if (Tcl_GetDouble (interp, argv[2], &x2) != TCL_OK) return TCL_ERROR;
	if (Tcl_GetDouble (interp, argv[3], &y2) != TCL_OK) return TCL_ERROR;

	/* The bounding box is 1 point larger. This is why
	   we make the real size a little bit smaller. */

	x1 += 1; y1 += 1; x2 -= 1; y2 -= 1;

	object->x = (x2+x1)/2;
	object->y = (y2+y1)/2;

	if (selected) {
	    m_unselect (interp, object, 0, (char **) NULL);
	}
	sprintf (buffer, " %f %f %f %f", x1, y1, x2, y2);
	ret = Tcl_VarEval (interp, type_to_string (object->type), "::resize ",
			   object->id, buffer, (char *) NULL);

	if (selected) {
	    m_select (interp, object, 0, (char **) NULL);
	}
	notrace (m_label, interp, object, 1, &tmp);

	trace (object->editor, object, 
	       "ined size", argc, argv, (char *) NULL);
    }

    ret = Tcl_VarEval (interp, type_to_string (object->type),
		       "::size ", object->id, (char *) NULL);

    if (ret == TCL_OK && strlen(interp->result) > 0) {
	STRCOPY (object->size, interp->result);
    }

    Tcl_SetResult (interp, object->size, TCL_STATIC);
    return TCL_OK;
}

/*
 * Return a string of ined commands that rebuilds the node
 * object.
 */

static int 
m_node_dump (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    Tcl_AppendResult (interp, "set ", object->id, 
		      " [ ined -noupdate create NODE ]\n", (char *) NULL);

    dump_move       (interp, object);
    dump_icon       (interp, object);
    dump_font       (interp, object);
    dump_color      (interp, object);
    dump_name       (interp, object);
    dump_address    (interp, object);
    dump_oid        (interp, object);
    dump_attributes (interp, object);
    dump_label      (interp, object);

    return TCL_OK;
}

/*
 * Return a string of ined commands that rebuilds the group
 * object. The user of the string must ensure that the variables 
 * for the member objects exist.
 */

static int
m_group_dump (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    int i;
    int largc;
    char **largv;

    Tcl_AppendResult (interp, "set ", object->id, 
		      " [ ined -noupdate create GROUP ", (char *) NULL);
    Tcl_SplitList (interp, object->member, &largc, &largv);
    for (i = 0; i < largc; i++) {
	Tcl_AppendResult (interp, " $", largv[i], (char *) NULL);
    }
    free ((char*) largv);
    Tcl_AppendResult (interp, " ]\n", (char *) NULL);

    /* Save the position if the group has no members. Otherwise,
       we get the position from the position of the group members */

    if (largc == 0) dump_move (interp, object);
    dump_icon       (interp, object);
    dump_font       (interp, object);
    dump_color      (interp, object);
    dump_name       (interp, object);
    dump_oid        (interp, object);
    dump_attributes (interp, object);
    dump_label      (interp, object);

    if (!object->collapsed) {
	Tcl_AppendResult (interp, "ined -noupdate expand $", 
			  object->id, "\n", (char *) NULL);
    }

    return TCL_OK;
}

/*
 * Return a string of ined commands that rebuilds the network object.
 */

static int
m_network_dump (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    Tcl_AppendResult (interp, "set ", object->id, 
		      " [ ined -noupdate create NETWORK ", 
		      object->member, " ]\n", (char *) NULL);

    dump_move       (interp, object);
    dump_icon       (interp, object);
    dump_font       (interp, object);
    dump_color      (interp, object);
    dump_name       (interp, object);
    dump_address    (interp, object);
    dump_oid        (interp, object);
    dump_attributes (interp, object);
    dump_label      (interp, object);

    return TCL_OK;
}

/*
 * Return a string of ined commands that rebuilds the link object.
 */

static int
m_link_dump (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    Tcl_AppendResult (interp, "set ", object->id, 
		      " [ ined -noupdate create LINK $", object->src, 
		      " $", object->dst, " ", object->member, " ]\n", 
		      (char *) NULL);

    dump_color (interp, object);

    return TCL_OK;
}

/*
 * Return a string of ined commands that rebuilds the text.
 */

static int
m_text_dump (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    Tcl_AppendResult (interp, "set ", object->id, 
		      " [ ined -noupdate create TEXT ", (char *) NULL);
    Tcl_AppendElement (interp, object->text);
    Tcl_AppendResult (interp, " ]\n", (char *) NULL);

    dump_move  (interp, object);
    dump_font  (interp, object);
    dump_color (interp, object);

    return TCL_OK;
}

/*
 * Return a string of ined commands that rebuilds the image.
 */

static int
m_image_dump (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    Tcl_AppendResult (interp, "set ", object->id, 
		      " [ ined -noupdate create IMAGE ", (char *) NULL);
    Tcl_AppendElement (interp, object->name);
    Tcl_AppendResult (interp, " ]\n", (char *) NULL);

    dump_move  (interp, object);
    dump_color (interp, object);

    return TCL_OK;
}

/*
 * Return a string of ined commands that rebuilds the interpreter.
 */

static int
m_interpreter_dump (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    /* remove the absolute path name to make saved files more portable */

    char *p = strrchr (object->name, '/');
    
    Tcl_AppendResult (interp, "set ", object->id, 
		      " [ ined -noupdate create INTERPRETER ", 
		      (p) ? ++p : object->name, 
		      " ]\n", (char *) NULL);

    if (strlen(object->action) > 0) {
	int largc, i;
	char **largv;

	Tcl_SplitList (interp, object->action, &largc, &largv);
	for (i = 0; i < largc; i++) {
	    Tcl_AppendResult (interp, "ined send $", object->id,
			      " ", largv[i], "\n", (char *) NULL);
	}

	free ((char *) largv);
    }

    return TCL_OK;
}

/*
 * Return a string of ined commands that rebuilds the log object.
 */

int
m_log_dump (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    Tcl_AppendResult (interp, "set ", object->id, 
		      " [ ined -noupdate create LOG ]\n", (char *) NULL);

    dump_icon    (interp, object);
    dump_name    (interp, object);

    return TCL_OK;
}

/*
 * Return a string of ined commands that rebuilds the reference.
 */

static int
m_ref_dump (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    Tcl_AppendResult (interp, "set ", object->id, 
		      " [ ined -noupdate create REFERENCE ]\n", (char *) NULL);

    dump_move       (interp, object);
    dump_icon       (interp, object);
    dump_font       (interp, object);
    dump_color      (interp, object);
    dump_name       (interp, object);
    dump_address    (interp, object);
    dump_oid        (interp, object);
    dump_attributes (interp, object);
    dump_label      (interp, object);

    return TCL_OK;
}

/*
 * Return a string of ined commands that rebuilds the stripchart.
 */

static int
m_strip_dump (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    char *values = NULL;

    /* get the current size and the values of the chart */

    m_size (interp, object, 0, (char **) NULL);
    Tcl_VarEval (interp, type_to_string (object->type), "::values ",
		 object->id, (char *) NULL);
    values = xstrdup (interp->result);
    Tcl_ResetResult (interp);

    Tcl_AppendResult (interp, "set ", object->id, 
		      " [ ined -noupdate create STRIPCHART ]\n", 
		      (char *) NULL);

    dump_move       (interp, object);
    dump_font       (interp, object);
    dump_color      (interp, object);
    dump_scale      (interp, object);
    dump_size       (interp, object);
    dump_name       (interp, object);
    dump_address    (interp, object);
    dump_attributes (interp, object);
    dump_label      (interp, object);

    Tcl_AppendResult (interp, "ined -noupdate values $",
		      object->id, " ", values, "\n", (char *) NULL);

    free (values);

    return TCL_OK;
}

/*
 * Return a string of ined commands that rebuilds the barchart.
 */

static int
m_bar_dump (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    char *values = NULL;

    /* get the current size and the values of the chart */

    m_size (interp, object, 0, (char **) NULL);
    Tcl_VarEval (interp, type_to_string (object->type), "::values ",
		 object->id, (char *) NULL);
    values = xstrdup (interp->result);
    Tcl_ResetResult (interp);

    Tcl_AppendResult (interp, "set ", object->id, 
		      " [ ined -noupdate create BARCHART ]\n", (char *) NULL);

    dump_move       (interp, object);
    dump_font       (interp, object);
    dump_color      (interp, object);
    dump_scale      (interp, object);
    dump_size       (interp, object);
    dump_name       (interp, object);
    dump_address    (interp, object);
    dump_attributes (interp, object);
    dump_label      (interp, object);

    Tcl_AppendResult (interp, "ined -noupdate values $",
		      object->id, " ", values, "\n", (char *) NULL);

    free (values);

    return TCL_OK;
}

/*
 * Return a string of ined commands that rebuilds the graph.
 */

static int
m_graph_dump (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    int i;
    
    Tcl_AppendResult (interp, "set ", object->id, 
		      " [ ined -noupdate create GRAPH ]\n", (char *) NULL);

    dump_name       (interp, object);
    dump_address    (interp, object);
    dump_icon       (interp, object);
    dump_color      (interp, object);
    dump_attributes (interp, object);
    dump_label      (interp, object);

    if (object->numValues > 0) {
	Tcl_AppendResult (interp, "ined -noupdate values $",
			  object->id, " ", (char *) NULL);
	for (i = 0; i < object->numValues; i++) {
	    Tcl_PrintDouble (interp, object->valuePtr[i], buffer);
	    Tcl_AppendResult (interp, "{", buffer, " ", (char *) NULL);
	    Tcl_PrintDouble (interp, object->valuePtr[i], buffer);
	    Tcl_AppendResult (interp, buffer, "} ", (char *) NULL);
	}
	Tcl_AppendResult (interp, "\n", (char *) NULL);
    }
	
    return TCL_OK;
}

void
dump_object (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    switch (object->type) {
 case TKINED_NODE:
	m_node_dump (interp, object); 
	break;
 case TKINED_GROUP:
	m_group_dump (interp, object); 
	break;
 case TKINED_NETWORK:
	m_network_dump (interp, object); 
	break;
 case TKINED_LINK:
	m_link_dump (interp, object); 
	break;
 case TKINED_TEXT:
	m_text_dump (interp, object); 
	break;
 case TKINED_IMAGE:
	m_image_dump (interp, object); 
	break;
 case TKINED_INTERPRETER:
	m_interpreter_dump (interp, object); 
	break;
 case TKINED_LOG:
	m_log_dump (interp, object);
        break;
 case TKINED_REFERENCE:
	m_ref_dump (interp, object); 
	break;
 case TKINED_STRIPCHART:
	m_strip_dump (interp, object); 
	break;
 case TKINED_BARCHART:
	m_bar_dump (interp, object); 
	break;
 case TKINED_GRAPH:
	m_graph_dump (interp, object); 
	break;
    }
}

/*
 * Dump an arbitrary object. This function calls the type specific 
 * function and replaces every \n with a ; to seperate the tcl commands. 
 * This conversion is needed to transport the result string through the
 * protocol to applications.
 */

int m_dump (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    char *p;

    dump_object (interp, object);

    for (p = interp->result; *p; p++) {
	if (*p == '\n') *p = ';';
    }

    return TCL_OK;
}

/*
 * Get the starting point (object) of a link.
 */

int
m_src (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_SetResult (interp, object->src, TCL_STATIC);
    return TCL_OK;
}

/*
 * Get the end point (object) of a link.
 */

int
m_dst (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_SetResult (interp, object->dst, TCL_STATIC);
    return TCL_OK;
}

/*
 * Get and set the text of a text object. This calls
 * TEXT::text to update the view.
 */

int
m_text (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (argc == 1) {

	int selected = object->selected;

	free (object->text);
	object->text = xstrdupnn (argv[0]);
	
	if (selected) {
	    m_unselect (interp, object, 0, (char **) NULL);
	}
	Tcl_VarEval (interp, type_to_string (object->type),
		     "::text ", object->id, (char *) NULL);
	if (selected) {
	    m_select (interp, object, 0, (char **) NULL);
	}

	trace (object->editor, object, 
	       "ined text", argc, argv, object->text);
    }

    Tcl_SetResult (interp, object->text, TCL_STATIC);
    return TCL_OK;
}

/*
 * Append some text to the LOG window. The tk callback LOG::append
 * will be called to let the GUI do its actions. A backslash followed
 * by a n character will be expanded to a space and a newline character.
 */

int
m_append (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    int i;
    char *p;

    for (i = 0; i < argc; i++) {

	for (p = argv[i]; (p[0] != '\0') && (p[1] != '\0'); p++) {
	    if ((p[0] == '\\') && (p[1] == 'n')) {
		p[0] = ' ', p[1] = '\n';
	    }
	}

	Tcl_VarEval (interp, type_to_string (object->type),
		     "::append ", object->id, " {", argv[i], "}",
		     (char *) NULL);

	trace (object->editor, object, 
	       "ined append", argc, argv, (char *) NULL);
    }

    return TCL_OK;
}

/*
 * Create a hyperlink in a LOG window. The tk callback LOG::bind
 * will be called to let the GUI do its actions.
 */

int
m_hyperlink (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    int i;
    char *p;

    for (i = 1; i < argc; i++) {

	for (p = argv[i]; (p[0] != '\0') && (p[1] != '\0'); p++) {
	    if ((p[0] == '\\') && (p[1] == 'n')) {
		p[0] = ' ', p[1] = '\n';
	    }
	}

	Tcl_VarEval (interp, type_to_string (object->type),
		     "::bind ", object->id, " {", argv[0], "} ",
		     " {", argv[i], "}", (char *) NULL);

	argv[0][0] = '\0';
	trace (object->editor, object, 
	       "ined append", argc, argv, (char *) NULL);
    }

    return TCL_OK;
}

/*
 * Get the interpreter of a log or menu object. Its name is stored in 
 * the links attribute.
 */

int
m_interpreter (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_SetResult (interp, object->links, TCL_STATIC);
    return TCL_OK;
}

/*
 * Clear the text inside of a LOG window. The tk callback LOG::clear
 * will be called to let the GUI do its actions. This is also understood
 * by stripcharts and barcharts.
 */

int
m_clear (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (object->type == TKINED_GRAPH) {
	if (object->valuePtr) {
	    free (object->valuePtr);
	    object->valuePtr = NULL;
	}
	object->numValues = 0;
    }
    Tcl_VarEval (interp, type_to_string (object->type),
		 "::clear ", object->id, (char *) NULL);

    if (object->type == TKINED_LOG) {
	Tcl_VarEval (interp, type_to_string (object->type),
		     "::unbind ", object->id, (char *) NULL);
    }

    trace (object->editor, object, 
	   "ined clear", argc, argv, (char *) NULL);

    return TCL_OK;
}

/*
 * Get or set the scaling of a strip- or barchart.
 */

int
m_scale (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (argc == 1) {

	if (Tcl_GetDouble (interp, argv[0], &object->scale) != TCL_OK)
		return TCL_ERROR;

	Tcl_VarEval (interp, type_to_string (object->type), "::scale ",
		     object->id, " ", argv[0], (char *) NULL);

	trace (object->editor, object, 
	       "ined scale", argc, argv, (char *) NULL);
    }

    Tcl_ResetResult (interp);
    sprintf (interp->result, "%f", object->scale);
    return TCL_OK;
}

/*
 * Get or set the values of a strip- or barchart.
 */

int
m_values (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
#define MAX_STATIC_POINTS 256
    if (object->type == TKINED_GRAPH) {

	int largc, i;
	char **largv;
	double Xval, Yval;
	char buf[80];
	Tcl_DString dst;

	Tcl_DStringInit (&dst);

	for (i = 0; i < argc; i++) {

	    /* allocate an initial array for data points */
	    
	    if (object->valuePtr == NULL) {
		object->numValues = 0;
		object->valuePtr = (double *) malloc(MAX_STATIC_POINTS *
						     sizeof(double));
		object->allocValues = MAX_STATIC_POINTS;
	    }
	    
	    /* copy the values in the array and add the current time */
	    
	    if (Tcl_SplitList (interp, argv[i], &largc, &largv) != TCL_OK) {
		return TCL_ERROR;
	    }
	    
	    if (largc == 1) {
		time_t clock = time ((time_t *) NULL);
		Xval = clock;
		sprintf (buf, "%ld", clock);
		Tcl_GetDouble(interp, largv[0], &Yval);
		Tcl_DStringAppendElement (&dst, buf);
		Tcl_DStringAppendElement (&dst, argv[0]);
	    } else {
		Tcl_GetDouble(interp, largv[0], &Xval);
		Tcl_GetDouble(interp, largv[1], &Yval); 
		Tcl_DStringAppendElement (&dst, argv[0]);
		Tcl_DStringAppendElement (&dst, argv[1]);
	    }
	
	    if (object->numValues + 2 > object->allocValues) {
		object->allocValues += MAX_STATIC_POINTS;
		object->valuePtr = (double *) xrealloc (
			  (char *) object->valuePtr,
			  (unsigned) (sizeof(double) * object->allocValues));
	    }
	    
	    object->valuePtr[object->numValues] = Xval;
	    object->valuePtr[object->numValues+1] = Yval;
	    object->numValues+=2;

	    free ((char*) largv);
	}

#ifdef HAVE_BLT
	Blt_GraphElement (interp, object->canvas, object->id, 
			  object->numValues, object->valuePtr);
#endif

	Tcl_DStringFree (&dst);
	
    } else {
	
	char *args = Tcl_Merge (argc, argv);
	Tcl_VarEval (interp, type_to_string (object->type),
		     "::values ", object->id, " ", args, (char *) NULL);
	free (args);
    }
	
    trace (object->editor, object, 
	   "ined values", argc, argv, (char *) NULL);

    return TCL_OK;
}

/*
 * Get or set the jump distance of a stripchart.
 */

int
m_jump (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (argc == 1) {

	int num;

	if (Tcl_GetInt (interp, argv[0], &num) != TCL_OK) 
		return TCL_ERROR;

	Tcl_VarEval (interp, type_to_string (object->type),
		     "::jump ", object->id, " ", argv[0], (char *) NULL);

	trace (object->editor, object, 
	       "ined jump", argc, argv, (char *) NULL);
    }

    return TCL_OK;
}

/*
 * Get and set the member variable of a group object. This does not
 * work yet. This function should only affect objects that are removed
 * or added and it must take care that these objects reappear.
 */

int
m_member (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (argc > 0) {

	int i,largc;
	char **largv;
	char *freeme;
	tkined_object *member = NULL;
	Tcl_DString dst;
        int selected = object->selected;

	if (selected) {
	    m_unselect (interp, object, 0, (char **) NULL);
	}

	/* release all members if any */

	Tcl_SplitList (interp, object->member, &largc, &largv);
	for (i = 0; i < largc; i++) {
	    member = id_to_object (largv[i]);
	    if ((member != NULL) && (*member->parent != '\0')) {
		/* should become member of parent group? */
		if (*member->canvas == '\0') {
		    notrace (m_canvas, interp, member, 1, &object->canvas);
		    if (strcmp (member->color, "Black") != 0) {
			notrace (m_color, interp, member, 1, &member->color);
		    }
		    if (strcmp (member->icon, "machine") != 0) {
			notrace (m_icon, interp, member, 1, &member->icon);
		    }
		    if (strcmp (member->font, "default") != 0) {
			notrace (m_font,   interp, member, 1, &member->font);
		    }
		    notrace (m_label,  interp, member, 1, &member->label);
		}
		STRCOPY (member->parent, "");
	    }
	}
	free ((char *) largv);

	/* create a new member list */

	Tcl_DStringInit (&dst);
	freeme = Tcl_Merge (argc, argv);
	Tcl_SplitList (interp, freeme, &largc, &largv);
	for (i = 0; i < largc; i++) {
	    member = id_to_object (largv[i]);
	    if (member && strlen(member->parent) == 0) {
		STRCOPY (member->parent, object->id);
		Tcl_DStringAppendElement (&dst, largv[i]);
	    }
	}
        STRCOPY (object->member, Tcl_DStringValue (&dst));
	Tcl_DStringFree (&dst);
	free (freeme);
	free ((char *) largv);

	if (object->collapsed) {
	    object->collapsed = 0;
	    notrace (m_collapse,  interp, object, 0, (char **) NULL);
	} else {
	    if (member) {
		parent_resize (interp, member);
	    }
	}

	if (selected) {
	    m_select (interp, object, 0, (char **) NULL);
	}

	trace (object->editor, object, 
	       "ined member", argc, argv, (char *) NULL);
    }

    Tcl_SetResult (interp, object->member, TCL_STATIC);
    return TCL_OK;
}

/*
 * Collapse a group to an icon.
 */

int
m_collapse (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (!object->collapsed) {

	int i, largc;
	char **largv;
	double x1 = 0, y1 = 0, x2 = 0, y2 = 0;
        int selected = object->selected;

        object->collapsed = 1;
      
        if (selected) {
	    m_unselect (interp, object, 0, (char **) NULL);
	}

	Tcl_SplitList (interp, object->member, &largc, &largv);
	for (i = 0; i < largc; i++) {
	    tkined_object *member = id_to_object (largv[i]);
	    if (member != NULL) {
	        int sargc;
		char **sargv;

		if (member->selected) {
		    m_unselect (interp, member, 0, (char **) NULL);
		}

		STRCOPY (member->parent, object->id);
		if (member->type == TKINED_GROUP && !member->collapsed) {
		    notrace (m_collapse, interp, member, 0, (char **) NULL);
		}

		/* calculate the initial position of the group icon */

		if (object->x == 0 && object->y == 0) {
		    m_size (interp, member, 0, (char **) NULL);
		    Tcl_SplitList (interp, member->size, &sargc, &sargv);
		    if (sargc == 4) {
			double mx1, my1, mx2, my2;
			Tcl_GetDouble (interp, sargv[0], &mx1);
			Tcl_GetDouble (interp, sargv[1], &my1);
			Tcl_GetDouble (interp, sargv[2], &mx2);
			Tcl_GetDouble (interp, sargv[3], &my2);
			if (x1 == 0 || mx1 < x1) x1 = mx1;
			if (y1 == 0 || my1 < y1) y1 = my1;
			if (mx2 > x2) x2 = mx2;
			if (my2 > y2) y2 = my2;
		    }
		    free ((char *) sargv);
		}

		STRCOPY (member->canvas, "");
	    }
	}
	free ((char*) largv);

	/* set the initial position of the icon */

	if (largc > 0 && object->x == 0 && object->y == 0) {
	    object->x = x1+(x2-x1)/2;
	    object->y = y1+(y2-y1)/2;
	}

	/* make a dummy move to update links pointing to this group */

       {
	   char *tmp = "0";
	   char *foo[2];
	   foo[0] = tmp;
	   foo[1] = tmp;
	   notrace (m_move, interp, object, 2, foo);
       }

	Tcl_VarEval (interp, type_to_string (object->type),
		     "::collapse ", object->id, (char *) NULL);

        notrace (m_icon,  interp, object, 1, &object->icon);
        notrace (m_color, interp, object, 1, &object->color);
        notrace (m_font,  interp, object, 1, &object->font);
        notrace (m_label, interp, object, 1, &object->label);

	if (selected) {
	    m_select (interp, object, 0, (char **) NULL);
	}
	
	trace (object->editor, object, 
	       "ined collapse", argc, argv, (char *) NULL);
    }

    return TCL_OK;
}

/*
 * Expand a group to show its members.
 */

int
m_expand (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (object->collapsed) {

	int i, largc;
	char **largv;
        int selected = object->selected;
	object->collapsed = 0;
      
	if (selected) {
	    m_unselect (interp, object, 0, (char **) NULL);
	}
      
	Tcl_SplitList (interp, object->member, &largc, &largv);
	for (i = 0; i < largc; i++) {
	    tkined_object *member;
	    member = id_to_object (largv[i]);
	    if (member != NULL) {
		if (member->type == TKINED_GROUP && member->collapsed) {
		    member->collapsed = 0;
		}
		notrace (m_canvas, interp, member, 1, &object->canvas);
		if (strcmp (member->color, "Black") != 0) {
		    notrace (m_color, interp, member, 1, &member->color);
		}
		if (strcmp (member->icon, "machine") != 0) {
		    notrace (m_icon, interp, member, 1, &member->icon);
		}
		if (strcmp (member->font, "default") != 0) {
		    notrace (m_font,   interp, member, 1, &member->font);
		}
		notrace (m_label,  interp, member, 1, &member->label);
	    }
	}
	free ((char*) largv);

	Tcl_VarEval (interp, type_to_string (object->type),
		     "::expand ", object->id, (char *) NULL);

	notrace (m_color, interp, object, 1, &object->color);
	notrace (m_font,  interp, object, 1, &object->font);
	notrace (m_label, interp, object, 1, &object->label);

	/* adjust the size of a parent group if necessary */
      
	parent_resize (interp, object);
      
	if (selected) {
	    m_select (interp, object, 0, (char **) NULL);
	}

	trace (object->editor, object, 
	       "ined expand", argc, argv, (char *) NULL);
    }

    return TCL_OK;
}

/*
 * Return a boolean indicating if the group is collapsed or not.
 */

int
m_collapsed (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_ResetResult (interp);
    sprintf (interp->result, "%d", object->collapsed);
    return TCL_OK;
}

/*
 * Ungroup. First expand the group and then delete the group object.
 */

int
m_ungroup (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    int margc, i;
    char **margv;

    if (object->collapsed) {
	m_expand (interp, object, argc, argv);
    }

    Tcl_SplitList (interp, object->member, &margc, &margv);
    for (i = 0; i < margc; i++) {
	tkined_object *member = id_to_object (margv[i]);
	STRCOPY (member->parent, "");
    }
    free ((char *) margv);

    STRCOPY (object->member, "");

    trace (object->editor, object, 
	   "ined ungroup", argc, argv, (char *) NULL);

    return m_delete (interp, object, argc, argv);
}

/*
 * Get and set the links that are connected to a node or network
 * object.
 */

int
m_links (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (argc == 1) {
        STRCOPY (object->links, argv[0]);
    }

    Tcl_SetResult (interp, object->links, TCL_STATIC);
    return TCL_OK;
}

/*
 * Get and set the fixed points stored in the member variable of a 
 * link or network object.
 */

int
m_points (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (argc == 1) {

        STRCOPY (object->member, argv[0]);

	if (object->type == TKINED_NETWORK) {
	    int largc, i;
            char **largv;
            tkined_object *link;
	    char *tmp = "reset";

            Tcl_SplitList (interp, object->links, &largc, &largv);
            for (i = 0; i < largc; i++) {
                link = id_to_object (largv[i]);
                if (link != NULL) {
                    m_link_update (interp, link, 0, (char **) NULL);

                }
            }
	    
	    if (object->selected) {
		m_unselect (interp, object, 0, (char **) NULL);
		m_select (interp, object, 0, (char **) NULL);
	    }

	    notrace (m_label, interp, object, 1, &tmp);

	    parent_resize (interp, object);

	    trace (object->editor, object, 
		   "ined points", argc, argv, (char *) NULL);
	}
    }

    Tcl_SetResult (interp, object->member, TCL_STATIC);
    return TCL_OK;
}

/*
 * Computer the label coordinates of a network object.
 * This is done in C because computations are complicated
 * and slow in TCL.
 *
 * m_network_labelxy searches for the longest horizontal network segment 
 * that is at least longer than 100 points. If there is no such segment, 
 * we return the coordinates of the lowest endpoint of the vertical line
 * endpoint.
 */

int 
m_network_labelxy (interp, network, argc, argv)
    Tcl_Interp *interp;
    tkined_object *network;
    int argc;
    char **argv;
{
    int found = 0;
    int i, j, n;
    int largc;
    char **largv;
    double *x;
    double *y;
    double lx, ly;
    double sx = 0, sy = 0, slen = 0;

    Tcl_SplitList (interp, network->member, &largc, &largv);

    x = (double *) xmalloc (largc * sizeof(double));
    y = (double *) xmalloc (largc * sizeof(double));

    if (x == NULL || y == NULL) {
	free ((char*) largv);
	Tcl_ResetResult (interp);
	sprintf (interp->result, "%f %f", network->x, network->y);
	return TCL_OK;
    }

    for (n = 0, i = 0; i < largc; i++) {
	if ((i%2) == 0) {
	    Tcl_GetDouble (interp, largv[i], &x[n]);
	    x[n] += network->x;
	} else {
	    Tcl_GetDouble (interp, largv[i], &y[n]);
	    y[n] += network->y;
	    n++;
	}
    }

    for (i = 1, j = 0; i < n; i++, j++) {
	lx = (x[i]>x[j]) ? x[i]-x[j] : x[j]-x[i];
	ly = (y[i]>y[j]) ? y[i]-y[j] : y[j]-y[i];
	if (!found) {
            if (y[i] > sy) {
		sx = (x[i]+x[j])/2;
		sy = y[i];
	    }
	    if (y[j] > sy) {
                sx = (x[i]+x[j])/2;
                sy = y[j];
            }
	}
	if (lx > slen) {
            sx = (x[i]+x[j])/2;
	    sy = (y[i]+y[j])/2;
	    slen = lx;
	    found = (slen > 100);
	}
    }
    sy += 3;

    free ((char *) x);
    free ((char *) y);
    free ((char *) largv);

    Tcl_ResetResult (interp);
    sprintf (interp->result, "%f %f", sx, sy+1);
    return TCL_OK;
}

/*
 * Send a message (usually a command) to an interpreter.
 * Append a list of the current selection to it.
 */

int
m_send (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    int len;
    char *args;

    if (argc > 0) {

	args = Tcl_Merge (argc, argv);
	len = strlen (args);
	args[len] = '\n';
	len++;
	if (len != xwrite (object->xv[1], args, len)) {
	    Tcl_ResetResult (interp);
	    Tcl_AppendResult (interp, "write to ", object->id, " failed: ", 
			      Tcl_PosixError (interp), (char *) NULL);
	    free (args);
            return TCL_ERROR;
        }
	free (args);
    }

    return TCL_OK;
}

/*
 * This is called for network and node objects. Before they
 * get deleted, they must delete all links connected to them.
 */

static void
m_linked_delete (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    int i, largc;
    char **largv;
    tkined_object *link;

    Tcl_SplitList (interp, object->links, &largc, &largv);
    for (i = 0; i < largc; i++) {
	link = id_to_object (largv[i]);
	if (link != NULL) {
	    notrace (m_delete, interp, link, 0, (char **) NULL);
	    Tcl_ResetResult (interp);
	}
    }
    free ((char*) largv);
}

/*
 * When deleting a link object, we have to update the links
 * stored by the objects connected by this link.
 */

static void
m_link_delete (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    tkined_object *obj;
    
    if ((obj = id_to_object (object->src)) != NULL) {
	ldelete (interp, obj->links, object->id);
    }

    if ((obj = id_to_object (object->dst)) != NULL) {
	ldelete (interp, obj->links, object->id);
    }
}

/*
 * When deleting a group object, make sure that everything in 
 * it gets deleted.
 */

static void
m_group_delete (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    int largc, i;
    char **largv;
    tkined_object *member;

    Tcl_SplitList (interp, object->member, &largc, &largv);
    for (i = 0; i < largc; i++) {
	member = id_to_object (largv[i]);
        if (member != NULL) {
            notrace (m_delete, interp, member, 0, (char **) NULL);
	    Tcl_ResetResult (interp);
        }
    }
    free ((char *) largv);
}

/*
 * Make sure to delete all menus associated with this interpreter.
 */

static void
m_interpreter_delete (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (object->trace && object->editor) 
	    object->editor->traceCount--;

    if (object->type == TKINED_INTERPRETER) {
	Tcl_HashEntry *ht_entry;
	Tcl_HashSearch ht_search;
	ht_entry = Tcl_FirstHashEntry(&ht_object, &ht_search);
	while (ht_entry != NULL) {
	    tkined_object *obj = (tkined_object *) Tcl_GetHashValue (ht_entry);
	    if ((obj->type == TKINED_MENU) 
		&& (strcmp(obj->links, object->id) == 0)) {
		notrace (m_delete, interp, obj, 0, (char **) NULL);
		Tcl_ResetResult (interp);
	    }
	    if ((obj->type == TKINED_LOG) 
		&& (strcmp(obj->links, object->id) == 0)) {
		Tcl_VarEval (interp, type_to_string (obj->type),
			     "::unbind ", obj->id, (char *) NULL);
		Tcl_ResetResult (interp);
	    }
	    ht_entry = Tcl_NextHashEntry (&ht_search);
	}
    }
}

/*
 * Delete an object. This is understood by all objects.
 */

int
m_delete (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    switch (object->type) {
 case TKINED_NODE: 
 case TKINED_NETWORK: 
	m_linked_delete (interp, object, argc, argv);
	break;
 case TKINED_GROUP: 
	m_group_delete (interp, object, argc, argv);
	break;
 case TKINED_LINK:
	m_link_delete (interp, object, argc, argv);
	break;
 case TKINED_INTERPRETER:
	m_interpreter_delete (interp, object, argc, argv);
	break;
 case TKINED_MENU:
	Tcl_ReapDetachedProcs();
	break;
    }

    if (object->selected) {
	m_unselect (interp, object, 0, (char **) NULL);
    }

    Tcl_VarEval (interp, type_to_string (object->type),
		 "::delete ", object->id, (char *) NULL);

    /* Remove the reference to this object if it is a member
       of a group (that is it has a valid parent) */

    if ( strlen(object->parent) > 0) {
	tkined_object *parent = id_to_object (object->parent);
	if (parent != NULL) {
	    ldelete (interp, parent->member, object->id);
	}
    }

    trace (object->editor, object, 
	   "ined delete", argc, argv, (char *) NULL);

    Tcl_DeleteCommand (interp, object->id);

    return TCL_OK;
}

/*
 * Experimental stuff.
 */

int
m_postscript (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    int rc;

    if (object->type == TKINED_GRAPH) {
	rc = Tcl_VarEval (interp, type_to_string (object->type),
			  "::postscript ", object->id, (char *) NULL);
    } else {
	rc = Tcl_VarEval (interp, "::postscript ", object->id, (char *) NULL);
    }

#ifdef DBMALLOC
    tmp = xstrdupnn (interp->result);
    Tcl_SetResult (interp, tmp, TCL_VOLATILE);
    free (tmp);
#else
    Tcl_SetResult (interp, xstrdupnn(interp->result), TCL_DYNAMIC);
#endif

    return rc;
}

/*
 * Get or set a generic object attribute. They are used to
 * store default values and arbitrary status information.
 */

int 
m_attribute (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    Tcl_HashEntry *ht_entry;

    if (argc == 0) {

	Tcl_HashSearch ht_search;

	ht_entry = Tcl_FirstHashEntry(&(object->attr), &ht_search);
	while (ht_entry != NULL) {
	    Tcl_AppendElement (interp, 
			       Tcl_GetHashKey (&(object->attr), ht_entry));
	    ht_entry = Tcl_NextHashEntry (&ht_search);
	}

	return TCL_OK;
    }

    if (argc == 2) {
	int isnew;

	ht_entry = Tcl_CreateHashEntry (&(object->attr), argv[0], &isnew);
	if (! isnew) {
	    free ((char *) Tcl_GetHashValue (ht_entry));
	}

	if (argv[1][0] == '\0') {
	    char *tmp = "name";
	    Tcl_DeleteHashEntry (ht_entry);
	    if (strcmp(object->label, argv[0]) == 0) {
		notrace (m_label, interp, object, 1, &tmp);
	    }
	} else {
	    Tcl_SetHashValue (ht_entry, xstrdup(argv[1]));
	    if (strcmp(object->label, argv[0]) == 0) {
		notrace (m_label, interp, object, 1, &object->label);
	    }
	}

	trace (object->editor, object, 
	       "ined attribute", argc, argv, argv[0]);
    }

    ht_entry = Tcl_FindHashEntry (&(object->attr), argv[0]);
    if (ht_entry != NULL) {
	interp->result = (char *) Tcl_GetHashValue (ht_entry);
    }

    return TCL_OK;
}

/*
 * Flashing icons to get the attention of the user.
 */

int 
m_flash (interp, object, argc, argv)
    Tcl_Interp *interp;
    tkined_object *object;
    int argc;
    char **argv;
{
    if (argc == 1) {

	int secs;

	if (Tcl_GetInt (interp, argv[0], &secs) != TCL_OK)
		return TCL_ERROR;

	secs *= 2; /* we flash two times in one second */

	if (object->flash > 0) {
	    object->flash = (secs > object->flash) ? secs : object->flash;
	    object->flash = secs;
	} else {
	    object->flash = secs;
	    flash (interp, object);
	}
	
	trace (object->editor, object, 
	       "ined flash ", argc, argv, argv[0]);
    }

    return TCL_OK;
}

