#############################################################################
#
# message.tcl:
#       This file contains the procedures directly related to operations on
#       messages.
#
# ComposeMsg {Address} 
#       Compose a message, bringing up an editor, with the To: field already
#       Filled in.
#
# CopyMsg {FromFol ToFol}
#       Copy $Msg from $FromFol to $ToFol, append as last, use seq copymsg
#
# MoveMsg {FromFol ToFol}
#       Move $Msg from $FromFol to $ToFol, append as last
#
# DeleteMsg {Fol}
#       delete a message, or a range of messages, reading the sequence
#	delmsgs from the .mh_sequences file
#
# ShowMsg {Fol Msg}
#       Bring up a window in which the message is displayed. Add buttons 
#       For reply, forward and so on.
#       TODO: write the stuff for the buttons
#
# EditMsg {Fol Msg} 
#       bring up a window in which the message in the Folder can be edited.
#       needs buttons for save, refile, mail and so
#
# SendMsg {Fol Msg}
#       Send out the Message. Keep determines if it should be kept or 
#       backed up.
#
# SignEdit {w}
#       belongs to EditMsg, appends the Signature file to the edit window w.
#
# ForwardMsg {Fol Msg}
#       forward the current message
#
# PrintMsg {Fol Msgs}
#       Print the message, using the printcommand defined by the user.
#
# ReplyMsg {Fol Msg}
#       Reply to a message
#
#############################################################################

proc ComposeMsg {{Address {}}} {
    global MH cmd

    exec $MH(bindir)/comp -nowhatnowproc
    set File [FindLastFile $MH(Draft-Folder)]
    set FileContents [eval exec $cmd(cat) [RF $MH(Path)/$MH(Draft-Folder)/$File]]
    if {$Address != ""} {
        regsub "To:" $FileContents "To: $Address" FileContents
        set FHandle [open [RF $MH(Path)/$MH(Draft-Folder)/$File] w]
        puts $FHandle $FileContents
        close $FHandle
    }
    append_seq $MH(Draft-Folder) allmsgs $File
    Cache:Append [RF $MH(Path)/$MH(Draft-Folder)/.xmhcache] \
      [list [exec $MH(bindir)/scan +$MH(Draft-Folder) $File]]
    if [winfo exists [FW $MH(Draft-Folder)]] {ScanFol $MH(Draft-Folder)}
    EditMsg $MH(Draft-Folder) $File
}

proc CopyMsg {FromFol ToFol} {
    global MH cmd

    # read in the sequences
    set copymsgs [conv_seq $FromFol copymsgs]
    set firstMsg [expr [FindLastFile $ToFol] + 1]
    
    #copy the messages
    set NewMsg $firstMsg
    foreach Msg $copymsgs {
        eval exec $cmd(cp) [RF $MH(Path)/$FromFol/$Msg] \
                           [RF $MH(Path)/$ToFol/$NewMsg]
        incr NewMsg
    }
    
    # adapt the ToFol seq, cache and display
    del_seq $FromFol copymsgs
    append_seq $ToFol allmsgs $firstMsg-last
    Cache:Append [RF $MH(Path)/$ToFol/.xmhcache] \
      [split [exec $MH(bindir)/scan +$ToFol $firstMsg-last] "\n"]
    ScanFol $ToFol
}

proc MoveMsg {FromFol ToFol} {
    global MH cmd
    
    # read in the sequences
    set movemsgs [conv_seq $FromFol movemsgs]
    set allmsgs [conv_seq $FromFol allmsgs]
    set first [lsearch $allmsgs [lindex $movemsgs 0]]
    set last [lsearch $allmsgs [lindex $movemsgs [expr [llength $movemsgs] -1]]]
    set firstMsg [expr [FindLastFile $ToFol] + 1]
    
    set NewMsg $firstMsg
    foreach Msg $movemsgs {
        eval exec $cmd(mv) [RF $MH(Path)/$FromFol/$Msg] \
                           [RF $MH(Path)/$ToFol/$NewMsg]
        incr NewMsg
        if [winfo exists [SW $FromFol $Msg]] {destroy [SW $FromFol $Msg]}
    }

    del_from_seq $FromFol allmsgs $first-$last
    del_seq $FromFol movemsgs
    Cache:Write [RF $MH(Path)/$FromFol/.xmhcache] \
      [lreplace [Cache:Read [RF $MH(Path)/$FromFol/.xmhcache]] $first $last]

    append_seq $ToFol allmsgs $firstMsg-last
    Cache:Append [RF $MH(Path)/$ToFol/.xmhcache] \
      [split [exec $MH(bindir)/scan +$ToFol $firstMsg-last] "\n"]

    ScanFol $ToFol
    ScanFol $FromFol
}

proc DeleteMsg {Fol} {
    global MH cmd PREF

    # read in the sequences
    set delmsgs [conv_seq $Fol delmsgs]
    set allmsgs [conv_seq $Fol allmsgs]
    set first [lsearch $allmsgs [lindex $delmsgs 0]]
    set last [lsearch $allmsgs [lindex $delmsgs [expr [llength $delmsgs] -1]]]
    
    #delete the files
    eval exec $MH(bindir)/rmm +$Fol $delmsgs
    foreach Msg $delmsgs {
        if [winfo exists [SW $Fol $Msg]] {destroy [SW $Fol $Msg]}
    }
    if {!$PREF(BackupMsg)} {
        foreach Msg $delmsgs {
            if {![file isfile [RF $MH(Path)/$Fol/,$Msg]]} break
            eval exec $cmd(rm) [RF $MH(Path)/$Fol/,$Msg]
        }
    }
    
    #adapt the sequences and the message cache
    del_from_seq $Fol allmsgs $first-$last
    del_seq $Fol delmsgs
    Cache:Write [RF $MH(Path)/$Fol/.xmhcache] \
      [lreplace [Cache:Read [RF $MH(Path)/$Fol/.xmhcache]] $first $last]
    
    # adapt the display if there is one
    ScanFol $Fol
}

proc ShowMsg {Fol Msg} {
    global MH FolTree Aliases
    global FONT cmd PREF BM

    set OldFont $PREF(ShowFont)
    set w [SW $Fol $Msg]
    if [winfo exists $w] {
        $w.t insert end [exec $MH(bindir)/show +$Fol $Msg]
        wm deiconify $w
        return
    }
    toplevel $w -class ShowMsg
    wm title $w "$Fol: $Msg"
    wm iconbitmap $w @$BM(show)

    text $w.t -yscroll "$w.s set" -wrap $PREF(Wrap) -font $PREF(ShowFont)
    scrollbar $w.s -command "$w.t yview"

    frame $w.m -relief raised -borderwidth 2
    checkbutton $w.m.mono -variable PREF(ShowFont) \
      -onvalue $FONT(FixR) -offvalue $FONT(ProR) \
      -command "$w.t configure -font \$PREF(ShowFont)" \
      -text "Mono"
    checkbutton $w.m.wrap -variable PREF(Wrap) \
      -onvalue word -offvalue none \
      -command "$w.t configure -wrap \$PREF(Wrap)" \
      -text "Wrap long lines"

    frame $w.b
    menubutton $w.b.move -text "Move To" -menu $w.b.move.m
    menubutton $w.b.copy -text "Copy To" -menu $w.b.copy.m
    button $w.b.reply -text "Reply" -command "ReplyMsg $Fol $Msg"
    button $w.b.del -text "Delete" \
      -command "destroy $w; write_seq $Fol delmsgs $Msg; DeleteMsg $Fol"
    button $w.b.print -text "Print" -command "PrintMsg $Fol $Msg"
    button $w.b.ok -text "Close" -command "destroy $w"
    button $w.b.help -text "Help" -command "Help Message"

    MakeMenu $w.b.move.m $FolTree 1 {} -command \
      "write_seq $Fol movemsgs $Msg; MoveMsg $Fol %EntryPath"
    MakeMenu $w.b.copy.m $FolTree 1 {} -command \
      "write_seq $Fol copymsgs $Msg; CopyMsg $Fol %EntryPath"
    
    if $PREF(AliMenu) {
      menubutton $w.b.forw -text "Forward To" -menu $w.b.forw.m
      menu $w.b.forw.m
      $w.b.forw.m add command -label *NEW* -command "ForwardMsg $Fol $Msg"
      MakeMenu $w.b.forw.m $Aliases 0 {} -command \
        "if \$PREF(ExpandAlias) {ForwardMsg $Fol $Msg \"\[SearchAlias %Entry 1\]\"} \
         else {ForwardMsg $Fol $Msg %Entry}"
    } else {
      button $w.b.forw -text "Forward To" -command "ForwardMsg $Fol $Msg"
    }

    pack append $w.m \
      $w.m.mono {right} \
      $w.m.wrap {right}

    pack append $w.b \
      $w.b.reply {top expand fill} \
      $w.b.forw {top expand fill} \
      $w.b.copy {top expand fill} \
      $w.b.move {top expand fill} \
      $w.b.print {top expand fill} \
      $w.b.del {top expand fill} \
      $w.b.help {top expand fill} \
      $w.b.ok {top expand fill}

    pack append $w \
      $w.m {top fillx} \
      $w.t {left expand fill} \
      $w.s {left filly} \
      $w.b {left filly}

    $w.t insert end [exec $MH(bindir)/show +$Fol $Msg]
    $w.t configure -state disabled
    focus $w.t
    tkwait window $w
    set PREF(ShowFont) $OldFont
}

proc EditMsg {Fol Msg} {
    global MH cmd FONT PREF env BM FolTree
    global Sent

    set OldKeepSent $PREF(KeepSent)
    set OldFont $PREF(EditFont)
    set w [EW $Fol $Msg]
    if [winfo exists $w] {
        MsgWindow "Already Editing that message!\
                  \nRefreshing Editor contents.\
                  \n\nFolder: $Fol\nMessage: $Msg\n"
        $w.t insert end [eval exec $cmd(cat) [RF $MH(Path)/$Fol/$Msg]]
        wm deiconify $w
        return
    }
    toplevel $w -class EditMsg
    wm title $w "Edit: $Fol:$Msg"
    wm iconbitmap $w @$BM(edit)

    text $w.t -yscroll "$w.s set" -wrap word -font $PREF(EditFont)
    scrollbar $w.s -command "$w.t yview"

    frame $w.m -relief raised -borderwidth 2
    checkbutton $w.m.mime -variable PREF(UseMIME) \
      -text {Use MIME}
    checkbutton $w.m.keepsent -variable PREF(KeepSent) \
      -text {Save when Done}
    checkbutton $w.m.mono -variable PREF(EditFont) \
      -onvalue $FONT(FixR) -offvalue $FONT(ProR) \
      -command "$w.t configure -font \$PREF(EditFont)" \
      -text Mono
    checkbutton $w.m.wrap -variable PREF(Wrap) \
      -onvalue word -offvalue none \
      -command "$w.t configure -wrap \$PREF(Wrap)" \
      -text {Wrap long lines}
    menubutton $w.m.filter -text Filter -menu $w.m.filter.m -width 8
    menu $w.m.filter.m
    foreach Filter $cmd(pipes) {
        if [string compare $Filter separator] {
          $w.m.filter.m add command -label [lindex $Filter 0] \
            -command "Pipe $w.t {[lindex $Filter 1]} [lindex $Filter 2]"
        } else {$w.m.filter.m add separator}}
    $w.m.filter.m add separator
    $w.m.filter.m add command -label {Filter through...} -command \
        "set pipecommand \[Ask {Filter through:}\]
         if {\$pipecommand != {}} {Pipe $w.t \"\$pipecommand\" nonewline}"

    frame $w.b
    button $w.b.ok -text Done -command \
     "SaveText $w.t [RF $MH(Path)/$Fol/$Msg]; destroy $w"
    button $w.b.mail -text {Mail it} -command \
      "SaveText $w.t [RF $MH(Path)/$Fol/$Msg]
       SendMsg $Fol $Msg
       $w.b.mail configure -text {Mailed it}
       set Sent 1"
    button $w.b.sign -text Sign -command "SignEdit $w"
    button $w.b.insert -text {Insert File} -command \
     "set File \[Browse $env(HOME)\]
      if {\$File != {}} {$w.t insert insert \[eval exec \$cmd(cat) \$File\]}"
    button $w.b.help -text Help -command {Help Edit/General}
    menubutton $w.b.save -text {Save to} -menu $w.b.save.m
    MakeMenu $w.b.save.m $FolTree 1 {} -command \
      "set NewMsg \[FindLastFile %EntryPath\]; incr NewMsg
       SaveText $w.t \[RF $MH(Path)/%EntryPath/\$NewMsg\]
       if \[winfo exists \[FW %EntryPath\]\] {ScanFol %EntryPath 1}"
      

    pack append $w.m \
      $w.m.filter {left} \
      $w.m.mono {right} \
      $w.m.wrap {right} \
      $w.m.mime {right} \
      $w.m.keepsent {right}

    pack append $w.b \
      $w.b.mail {top expand fill} \
      $w.b.sign {top expand fill} \
      $w.b.insert {top expand fill} \
      $w.b.help {top expand fill} \
      $w.b.save {top expand fill} \
      $w.b.ok {top expand fill}

    pack append $w \
      $w.m {top fillx} \
      $w.t {left expand fill} \
      $w.s {left filly} \
      $w.b {left filly}

    if {![file isfile $MH(bindir)/mhn]} {$w.m.mime configure -state disabled}
    $w.t insert end [eval exec $cmd(cat) [RF $MH(Path)/$Fol/$Msg]]
    $w.t insert end "\n"
    $w.t mark set insert 0.0
    set Sent 0
    focus $w.t
    tkwait window $w
    if {!$PREF(KeepSent)} {
        write_seq $Fol delmsgs $Msg
        DeleteMsg $Fol
    } else {
        set cache [Cache:Read [RF $MH(Path)/$Fol/.xmhcache]]
        set cache [lrange $cache 0 [expr [llength $cache] - 2]]
        lappend cache [exec $MH(bindir)/scan +$Fol $Msg]
        Cache:Write [RF $MH(Path)/$Fol/.xmhcache] $cache
        if [winfo exists [FW $Fol]] {ScanFol $Fol}
    }
    set PREF(EditFont) $OldFont
    set PREF(KeepSent) $OldKeepSent
}

proc SendMsg {Fol Msg} {
    global MH PREF cmd env

    for {set i 0; set File "#tkmhdraft0"} {[file exists $File]} {incr i} {
        set File "#tkmhdraft$i"
    }
    eval exec $cmd(cp) [RF $MH(Path)/$Fol/$Msg] [RF $MH(Path)/$File]
    if { $PREF(UseMIME) && [file isfile $MH(bindir)/mhn]} {
        set env(mhdraft) [RF $MH(Path)/$File]
        exec $MH(bindir)/mhn [RF $MH(Path)/$File]
        unset env(mhdraft)
    }
    exec $MH(bindir)/send -push -noverbose -draftmessage $File
}

proc SignEdit {w} {
    global MH cmd

    $w.t insert end "\n-----\n"
    $w.t insert end [eval exec $cmd(cat) [RF $MH(SigFile)]]
}

proc ForwardMsg {Fol Msg {Address {}}} {
    global MH cmd

    exec $MH(bindir)/forw $Msg +$Fol -nowhatnowproc
    set File [FindLastFile $MH(Draft-Folder)]
    set FileContents [eval exec $cmd(cat) [RF $MH(Path)/$MH(Draft-Folder)/$File]]
    if {$Address != ""} {
        regsub "To:" $FileContents "To: $Address" FileContents
        set FHandle [open [RF $MH(Path)/$MH(Draft-Folder)/$File] w]
        puts $FHandle $FileContents
        close $FHandle
    }
    append_seq $MH(Draft-Folder) allmsgs $File
    Cache:Append [RF $MH(Path)/$MH(Draft-Folder)/.xmhcache] \
      [list [exec $MH(bindir)/scan +$MH(Draft-Folder) $File]]
    if [winfo exists [FW $MH(Draft-Folder)]] {ScanFol $MH(Draft-Folder)}
    EditMsg $MH(Draft-Folder) $File
}

proc ReplyMsg {Fol Msg} {
    global MH

    exec $MH(bindir)/repl $Msg +$Fol -nowhatnowproc
    set File [FindLastFile $MH(Draft-Folder)]
    append_seq $MH(Draft-Folder) allmsgs $File
    Cache:Append [RF $MH(Path)/$MH(Draft-Folder)/.xmhcache] \
      [list [exec $MH(bindir)/scan +$MH(Draft-Folder) $File]]
    if [winfo exists [FW $MH(Draft-Folder)]] {ScanFol $MH(Draft-Folder)}    
    EditMsg $MH(Draft-Folder) $File
}

proc PrintMsg {Fol Msgs} {
    global cmd MH

    if {![regsub "%f" $cmd(print) {[RF $MH(Path)/$Fol/$Msg]} PrintCmd]} {
        set PrintCmd "$cmd(print) {[RF $MH(Path)/$Fol/$Msg]}"
    }
    foreach Msg $Msgs {
        eval exec $PrintCmd
    }
}
