'\"
'\" Copyright (c) 1990 The Regents of the University of California.
'\" All rights reserved.
'\"
'\" Permission is hereby granted, without written agreement and without
'\" license or royalty fees, to use, copy, modify, and distribute this
'\" documentation for any purpose, provided that the above copyright
'\" notice and the following two paragraphs appear in all copies.
'\"
'\" IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
'\" FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
'\" ARISING OUT OF THE USE OF THIS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
'\" CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
'\"
'\" THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
'\" INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
'\" AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
'\" ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
'\" PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
'\" 
.so man.macros
.HS dp_object tcldp
.BS
.SH NAME
.LP 
dp_objectCreateProc, dp_objectExists, dp_objectFree, dp_objectConfigure
.dp
dp_objectSlot, dp_objectSlotSet, dp_objectSlotAppend, dp_objectSlots 
.dp
  \- dp_object manipulation using Tcl scripting
.SH SYNOPSIS
\fBdp_objectCreateProc\fR \fIclass\fR \fIobject\fR
.br
\fBdp_objectExists\fR \fIobject\fR
.br
\fBdp_objectSlots\fR \fIobject\fR
.br
\fBdp_objectFree\fR \fIobject\fR
.br
\fBdp_objectSlot\fR \fIobject\fR \fIslot\fR
.br
\fBdp_objectSlotSet\fR \fIobject\fR \fIslot\fR \fIvalue\fR
.br
\fBdp_objectSlotAppend\fR \fIobject\fR \fIslot\fR \fIvalue\fR
.br
\fBdp_objectConfigure\fR \fIclass\fR \fIobject\fR \fIoptions\fR
.BE
.SH DESCRIPTION
.LP

A procedural object is a Tcl procedure that represents an object.
The utility commands described in this man page allow one to
easily implement procedural objects using Tcl scripting only.

A procedural object is usually implemented in C (e.g., as a C
struct) for speed but has a Tcl access interface.  For example,
Tk implements its widget classes in C but provides Tcl interfaces
to access those widgets.  Tk uses a procedure to represent each
widget instance.

The first argument of one of these procedural objects is a method
selector.  In other words, if \fIobject\fR is the name of a
procedural object:

\fIobject\fB \fImethod\fR ?\fIargs ...\fR?

If you use the utility procedures described in this man page to
implement a procedural object, that object will have a class and
may have slots.  The class of the object determines what methods
the object will respond to.  No class inheritance is supported.
.SH COMMANDS 
.TP 
\fBdp_objectCreateProc \fIclass\fR \fIobject\fR

This command creates a new Tcl procedure to represent a new \fIobject\fR.
The new procedure will have the name \fIobject\fR and have the given
\fIclass\fR.
.TP
\fBdp_objectExists \fIobject\fR

This command returns 1 if a procedural object named \fIobject\fR
exists in the interpreter and returns 0 otherwise.
.TP
\fBdp_objectFree \fIobject\fR

This command frees up the resources associated with the
\fIobject\fR, which should have been created with
dp_objectCreateProc.  After using dp_objectFree on an
\fIobject\fR, no more references to that \fIobject\fR or its slots
should be made.
.TP
\fBdp_objectSlots \fIobject\fR

This command returns a list of the slots of the procedural
\fIobject\fR, which should have been created with
dp_objectCreateProc.
.TP
\fBdp_objectSlot \fIobject\fR \fIslot\fR

This command returns the value of a \fIslot\fR of the procedural
\fIobject\fR, which should have been created with
dp_objectCreateProc.
.TP
\fBdp_objectSlotSet \fIobject\fR \fIslot\fR \fIvalue\fR

This command sets the \fIvalue\fR of a \fIslot\fR of the
\fIobject\fR, which should have been created with
dp_objectCreateProc.  If \fIslot\fR is not yet a slot of the
\fIobject\fR, this command will make \fIslot\fR a slot of the
object.  Thus, slots can be dynamically added to objects created
using dp_objectCreateProc.  A \fIslot\fR may not be an array.
.TP
\fBdp_objectSlotAppend \fIobject\fR \fIslot\fR \fIvalue\fR

This command uses lappend to append the given \fIvalue\fR
to the current value of the \fIslot\fR of the \fIobject\fR, which
should have been created with dp_objectCreateProc.  If
\fIslot\fR is not yet a slot of the \fIobject\fR, this command
will make \fIslot\fR a slot of the object.  Thus, slots can be
dynamically added to objects created using
dp_objectCreateProc.
.TP
\fBdp_objectConfigure \fIclass\fR \fIobject\fR ?\fIoptions\fR?

This command provides a Tk "configure" style of slot access for
the procedural \fIobject\fR, which should have been created using
dp_objectCreateProc.  (See example usage below.)  The argument
\fIclass\fR should be the class of the \fIobject\fR.  The
optional \fIoptions\fR argument can be used in the following
three formats:

\fBdp_objectConfigure \fIclass\fR \fIobject\fR ?\fI-slot\fR?
.br
\fBdp_objectConfigure \fIclass\fR \fIobject\fR ?\fI-slot val -slot val ...\fR?
.br
\fBdp_objectConfigure \fIclass\fR \fIobject\fR
.br

The first format of dp_objectConfigure retrieves the
configuration entry for a given \fIslot\fR.  A configuration
entry for a \fIslot\fR is a triplet of the format of \fI{-slot
default value}\fR, where \fI-slot\fR is the slot name prepended
with a hyphen, \fIdefault\fR is the default value for the
\fIslot\fR, and \fIvalue\fR is the current value of the
\fIslot\fR.  (Note that this configuration entry format for slots
is different than the configuration entry format of Tk widget
options.)

The second format of dp_objectConfigure sets the current
values of several \fIslot\fRs of an \fIobject\fR.  

The third format of dp_objectConfigure retrieves the
configuration entries of all the slots of the \fIobject\fR.

Note that to use dp_objectConfigure, you must prepend a hyphen
('-') to each slot name.  (e.g., "slot" becomes "-slot".)  This
was designed for pseudo-compatability with the Tk style of widget
option configuration, since widget options in Tk are all prefixed
with a '-'.  Also, any slot names, \fIslot\fR, should be fully
specified and not abbreviated.
.SH "CLASSES AND METHODS"
.LP

The command dp_objectCreateProc creates a new Tcl procedure whose
name is \fIobject\fR to represent a new procedural object.  
You may invoke various methods on the procedural \fIobject\fR,
using the following general form:
.DS C
\fIobject method \fR?\fIarg0 arg1 ...\fR?
.DE

The \fImethod\fR argument and the \fIclass\fR of the \fIobject\fR
determine the method invoked.
.LP
Methods for classes are specified using the following form:
.DS C
proc \fIclass\fR.\fImethod\fR {\fIself arg0 ... argN\fR} {
   ...method body...
}
.DE

The first argument \fIself\fR is required in the method
specification.  The \fIself\fR argument is set to the name of the
\fIobject\fR that the \fImethod\fR will get called upon.
.SH EXAMPLE
.LP
The following is a simple example of using the above commands to
create a procedural object using Tcl scripting only:
.DS C
# Define point creator.
#
proc makePoint {aPoint args} {
   eval dp_objectCreateProc point $aPoint;
   dp_objectSlotSet $aPoint x 0;
   dp_objectSlotSet $aPoint y 0;
   eval $aPoint configure $args
   return $aPoint;
}

# Define methods of point class.
#
proc point.configure {aPoint args} {
   eval dp_objectConfigure point $aPoint $args;
}

proc point.slot-value {aPoint slot} {
  dp_objectSlot $aPoint $slot;
}

proc point.destroy {aPoint slot} {
  dp_objectFree $aPoint $slot;
}

proc point.draw {aPoint} {
   set x [dp_objectSlot $aPoint x];
   set y [dp_objectSlot $aPoint y];
   dot $x $y;
}

# Instatiate a point and test it out.
#
makePoint p;			# Makes the point
p configure -x 10 -y 20;	# sets the value to (10,20)
p configure -x;			# returns {-x {} 10}
p configure;			# returns {{-x {} 10} {-y {} 20}}
p slot-value x;			# returns 10
p draw;				# calls "dot 10 20"
dp_objectExists p;		# returns 1
dp_objectExists p1;		# returns 0
p destroy;			# destroys the object
.DE
.SH KEYWORDS
object, method, class, slots, associative arrays, configure, options,
prototyping
.SH AUTHOR
Brian Smith (bsmith@cs.berkeley.edu)
.br
Steve Yen (syen@cs.berkeley.edu)
.br
Stephen Tu (tus@roger-rabbit.cs.berkeley.edu)
