/*
 * Copyright (c) 1994 Software Research Associates, Inc.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Software Research Associates not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  Software Research
 * Associates makes no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied
 * warranty.
 */

#include <windows.h>
#include <mbstring.h>
#include <string.h>
#include "term.h"

#define SCREEN_IMAGE(x, y) (&(Image[(y)][(x)]))
#define IMAGE_CY 250
#define IMAGE_CX 80
#define BUFFER_LENGTH 256
#define FG_COLOR RGB(255, 255, 255)
#define BG_COLOR RGB(0, 0, 0)
#define INIT_SCREEN_CX	IMAGE_CX
#define INIT_SCREEN_CY	25

#define WM_NO_BUFFERED WM_USER+20

#define NO_SCROLL 0x00
#define V_SCROLL 0x01
#define H_SCROLL 0x02

static int cxFrame, cyFrame, cyCaption, cxVScroll;

void FAR PASCAL
RegisterTerminalClass(HINSTANCE hInstance)
{		    
    WNDCLASS wndclass;

    wndclass.style = CS_HREDRAW | CS_VREDRAW;
    wndclass.lpfnWndProc = TerminalProc;
    wndclass.cbClsExtra = 0;
    wndclass.cbWndExtra = 2;
    wndclass.hInstance = hInstance;
    wndclass.hIcon = LoadIcon(NULL, IDI_APPLICATION);
    wndclass.hCursor = LoadCursor(NULL, IDC_ARROW);
    wndclass.hbrBackground = (HBRUSH) GetStockObject(BLACK_BRUSH);
    wndclass.lpszMenuName = NULL;
    wndclass.lpszClassName = "Terminal";

    RegisterClass(&wndclass);
}

HWND FAR PASCAL
CreateTerminal(HANDLE hInstance)
{
    HDC hdc;
    TEXTMETRIC tm;

    hdc = GetDC((HWND)NULL);
    SelectObject(hdc, GetStockObject(SYSTEM_FIXED_FONT));
    GetTextMetrics(hdc, &tm);
    ReleaseDC((HWND) NULL, hdc);

    cxFrame = GetSystemMetrics(SM_CXFRAME);
    cyFrame = GetSystemMetrics(SM_CYFRAME);
    cyCaption = GetSystemMetrics(SM_CYCAPTION);
    cxVScroll = GetSystemMetrics(SM_CXVSCROLL);

    return  CreateWindow("Terminal", "Terminal", 
    		WS_OVERLAPPED|WS_SYSMENU|WS_MINIMIZEBOX|WS_VSCROLL|WS_THICKFRAME,
    		CW_USEDEFAULT,
    		CW_USEDEFAULT,
    		(tm.tmAveCharWidth * INIT_SCREEN_CX) + (cxFrame * 2) + cxVScroll,
    		(tm.tmHeight * INIT_SCREEN_CY) + cyCaption + (cyFrame * 2), 
    		NULL, NULL, hInstance, NULL);

}


long FAR PASCAL
TerminalProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    static int cxChar, cyChar;		/* character size by pixel */
    static int cxClient, cyClient;	/* cleint area size by pixel */
    static int cxScreen, cyScreen;	/* screen size by character */
    static int xCaret, yCaret;		/* caret posision by character */
    static int nBuffer;			/* input buffer counter */
    static int nTop, nBottom;
    static int nAmount;			/* lines used in Image buffer */ 
    static int nVScrollPos;		/* start position to display when Vscrolled */
    static int nBottomOrg, nTopOrg;	 
    static char Image[IMAGE_CY][IMAGE_CX];
    static char Buffer[BUFFER_LENGTH + 1];
    static WORD wMode = NO_SCROLL;
    static HGLOBAL hGlobal;
    HDC hdc;
    PAINTSTRUCT ps;
    TEXTMETRIC tm;
    MINMAXINFO FAR* lpmmi;
    int y;
    BOOL bBuffered = 1;
    char *pGlobal, *pBuffer;

    switch(message) {
    case WM_CREATE:
        hdc = GetDC(hwnd);
        SelectObject(hdc, GetStockObject(SYSTEM_FIXED_FONT));
        GetTextMetrics(hdc, &tm);
        ReleaseDC(hwnd, hdc);
        cxChar = tm.tmAveCharWidth;
        cyChar = tm.tmHeight;

	hGlobal = GlobalAlloc(GHND, BUFFER_LENGTH+1);
	SetWindowWord(hwnd, 0, (WORD) hGlobal);

	xCaret = 0;
	yCaret = 0;
	cxScreen = INIT_SCREEN_CX;
	cyScreen = INIT_SCREEN_CY;
	nBuffer = 0;
	nBottom = INIT_SCREEN_CY - 1;
	nAmount = 1;

	for (y = 0; y < INIT_SCREEN_CY; y++) {
	    memset(SCREEN_IMAGE(0, y), ' ', INIT_SCREEN_CX);
	}

	SetScrollRange(hwnd, SB_VERT, 0, 1, FALSE);
	SetScrollPos(hwnd, SB_VERT, 1, TRUE);
	nVScrollPos = 1;
        return 0;
        
    case WM_SIZE:
        cxClient = LOWORD(lParam);
        cyClient = HIWORD(lParam);
        return 0;
        
    case WM_GETMINMAXINFO:
	lpmmi = (MINMAXINFO FAR*) lParam;
	lpmmi->ptMinTrackSize.x = (cxChar * INIT_SCREEN_CX) + cxVScroll + (cxFrame * 2);
	lpmmi->ptMinTrackSize.y = (cyChar * INIT_SCREEN_CY) + cyCaption + (cyFrame * 2);
	lpmmi->ptMaxTrackSize.x = (cxChar * INIT_SCREEN_CX) + cxVScroll + (cxFrame * 2);
	lpmmi->ptMaxTrackSize.y = (cyChar * INIT_SCREEN_CY) + cyCaption + (cyFrame * 2);
	return 0;

    case WM_VSCROLL:
    	if (!(wMode & V_SCROLL)) {
    	    wMode  |= V_SCROLL;
	    nTopOrg = nTop;
	    nBottomOrg = nBottom;
	    HideCaret(hwnd);
	}
    	switch (wParam) {
	case SB_LINEUP:
	    nTop = (nTop + IMAGE_CY - 1) % IMAGE_CY;
	    nVScrollPos--;
	    break;
	case SB_LINEDOWN:
	    nTop = (nTop + IMAGE_CY + 1) % IMAGE_CY;
	    nVScrollPos++;
	    break;
	case SB_PAGEUP:
	    nTop = (nTop + IMAGE_CY - (cyScreen - 1)) % IMAGE_CY;
	    nVScrollPos -= cyScreen - 1;
	    break;
	case SB_PAGEDOWN:
	    nTop = (nTop + IMAGE_CY + (cyScreen - 1)) % IMAGE_CY;
	    nVScrollPos += cyScreen - 1;
	    break;
	case SB_THUMBPOSITION:
	    if (nAmount < IMAGE_CY) {
	    	nTop = LOWORD(lParam);
	    } else {
	    	nTop =  (LOWORD(lParam) + nTopOrg + cyScreen) % IMAGE_CY;
	    }
	    nVScrollPos =  LOWORD(lParam);
	default:
	    break;
	}

	if (nVScrollPos >= nAmount - cyScreen) {
	    nVScrollPos = nAmount - cyScreen;
	    nTop = nTopOrg;
	    nBottom = nBottomOrg;
	    wMode &= ~V_SCROLL;
	    ShowCaret(hwnd);
	} else if (nVScrollPos <= 0) {
	    nVScrollPos = 0;
	    nTop = (nAmount < IMAGE_CY)? 0 : (nTopOrg + cyScreen)%IMAGE_CY;
	}
	nBottom = nTop + cyScreen -1;

	if (nVScrollPos != GetScrollPos(hwnd, SB_VERT)) {
	    SetScrollPos(hwnd, SB_VERT, nVScrollPos, TRUE);
	    InvalidateRect(hwnd, NULL, TRUE);
	}

	return 0;

    case WM_NO_BUFFERED:
	bBuffered = 0;

    case WM_CHAR:
	if (wMode & V_SCROLL) {
	    wMode &= ~V_SCROLL;
	    nTop = nTopOrg;
	    nBottom = nBottomOrg;
	    ShowCaret(hwnd);
	}    
	hdc = GetDC(hwnd);
	SelectObject(hdc, GetStockObject(SYSTEM_FIXED_FONT));
	SetBkColor(hdc, BG_COLOR);
	SetTextColor(hdc, FG_COLOR);
        switch ((char) wParam) {
	case '\r':
	case '\n':
	    if (bBuffered) {
		if (nBuffer == BUFFER_LENGTH) {
		    MessageBeep(MB_OK);
		    break;
		}
	    	Buffer[nBuffer] = '\n';
	    	Buffer[nBuffer + 1] = '\0';
		nBuffer = 0;
	    }
	    xCaret = 0;
	    if (nAmount < IMAGE_CY) {
	    	nAmount++;
		nVScrollPos++;
	    }
	    if (nAmount > cyScreen) {
	    	nTop = (nTop + 1) % IMAGE_CY;
	        nBottom = (nBottom + 1) % IMAGE_CY;
		SetScrollRange(hwnd, SB_VERT, 0, nAmount - cyScreen, FALSE);
		SetScrollPos(hwnd, SB_VERT, nAmount - cyScreen, TRUE);
	    }
	    if (yCaret + 1 == cyScreen) {
		HideCaret(hwnd);
	    	for (y = 0; y < cyScreen - 1; y++) {
		    TextOut(hdc, 0, y * cyChar, SCREEN_IMAGE(0, (y + nTop) % IMAGE_CY), cxScreen);
    		}
    		memset(SCREEN_IMAGE(0, nBottom), ' ', 80);
		TextOut(hdc, 0, (cyScreen - 1)  * cyChar, SCREEN_IMAGE(0, nBottom), cxScreen);
	        ShowCaret(hwnd);
	    } else {
	    	yCaret++;
	    }
	    if (bBuffered) {
		pBuffer = Buffer;
		pGlobal = GlobalLock(hGlobal);
	    	while (*pGlobal++ = *pBuffer++);
	    	GlobalUnlock(hGlobal);
		PostMessage(hwnd, WM_USER, hGlobal, 0L);
	    }
	    break;
	case '\b':
	    if(xCaret > 0) {
	        int cnt, flg = 0;
		for (cnt = 0; cnt < xCaret; cnt++) {
		    if (_ismbblead((unsigned char) (*SCREEN_IMAGE(cnt, (nTop + yCaret) % IMAGE_CY))) && flg != 1)
		    	flg = 1;
		    else if (_ismbbtrail((unsigned char) (*SCREEN_IMAGE(cnt, (nTop + yCaret) % IMAGE_CY))) && flg == 1)
			flg = 2;
		    else
		    	flg = 0;
		}
		if (flg == 2 && xCaret > 1 && nBuffer > 1) {
		    xCaret--;
		    nBuffer--;
		    *(SCREEN_IMAGE(xCaret, (nTop + yCaret) % IMAGE_CY)) = ' ';
		}    		 
		if (nBuffer > 0) {
		    nBuffer--;
		} else {
		    break;
		}
	    	xCaret--;
		*(SCREEN_IMAGE(xCaret, (nTop + yCaret) % IMAGE_CY)) = ' ';
		HideCaret(hwnd);
	        TextOut(hdc, 0, yCaret * cyChar, SCREEN_IMAGE(0, (nTop + yCaret) % IMAGE_CY), cxScreen);
	        ShowCaret(hwnd);
	    }
	    break;
	case '\t':
	    do {
	    	if (bBuffered) {
	    	    SendMessage(hwnd, WM_CHAR, ' ', 0L);
		} else {
	    	    SendMessage(hwnd, WM_NO_BUFFERED, ' ', 0L);
		}
	    } while (xCaret % 8 != 0);
	    break;
	default:    
	    if (bBuffered) {
		if (nBuffer == BUFFER_LENGTH) {
		    MessageBeep(MB_OK);
		    break;
		}
	    	Buffer[nBuffer] = (char) wParam;
		nBuffer++;
	    }
	    if (_ismbblead((unsigned short) wParam) && xCaret + 1 == cxScreen) {
	    	SendMessage(hwnd, WM_NO_BUFFERED, ' ', 0L);
	    }

    	    *(SCREEN_IMAGE(xCaret, (nTop + yCaret) % IMAGE_CY)) = (char) wParam;
	    HideCaret(hwnd);
	    TextOut(hdc, 0, yCaret * cyChar, SCREEN_IMAGE(0, (nTop + yCaret) % IMAGE_CY), cxScreen);

	    if (xCaret + 1 == cxScreen) {
	    	xCaret = 0;
		if (nAmount < IMAGE_CY) {
		    nAmount++;
		    nVScrollPos++;
		}
		if (nAmount >=cyScreen) {
	    	    nTop = (nTop + 1) % IMAGE_CY;
	            nBottom = (nBottom + 1) % IMAGE_CY;
		    SetScrollRange(hwnd, SB_VERT, 0, nAmount - cyScreen, FALSE);
		    SetScrollPos(hwnd, SB_VERT, nAmount - cyScreen, TRUE);
		}
	    	if (yCaret + 1 == cyScreen) {
	    	    for (y = 0; y < cyScreen - 1; y++) {
		    	TextOut(hdc, 0, y * cyChar, SCREEN_IMAGE(0, (y + nTop) % IMAGE_CY), cxScreen);
    		    }
    		    memset(SCREEN_IMAGE(0, nBottom), ' ', 80);
		    TextOut(hdc, 0, (cyScreen - 1)  * cyChar, SCREEN_IMAGE(0, nBottom), cxScreen);
	    	} else {
	    	    yCaret++;
	    	}
	    } else {
	    	xCaret++;
	    }
	    ShowCaret(hwnd);
	}        
	ReleaseDC(hwnd, hdc);
	SetCaretPos(xCaret * cxChar, yCaret * cyChar);
    	return 0;

    case WM_SETFOCUS:
    	CreateCaret(hwnd, 0, cxChar, cyChar);
	SetCaretPos(xCaret * cxChar, yCaret * cyChar);
	ShowCaret(hwnd);
	return 0;

    case WM_KILLFOCUS:
    	HideCaret(hwnd);
	DestroyCaret();
	return 0;

    case WM_PAINT:
    	hdc = BeginPaint(hwnd, &ps);
	SelectObject(hdc, GetStockObject(SYSTEM_FIXED_FONT));
	SetBkColor(hdc, BG_COLOR);
	SetTextColor(hdc, FG_COLOR);
	
	for(y = 0; y < cyScreen; y++) {
	    TextOut(hdc, 0, y * cyChar, SCREEN_IMAGE(0, (y + nTop)%IMAGE_CY), cxScreen);
	}
	EndPaint(hwnd, &ps);
	return 0;
	
    case WM_CLOSE:
	DestroyWindow(hwnd);
	return 0;

    case WM_DESTROY:
    	PostQuitMessage(0);
    	return 0;
    }   	    	
    return DefWindowProc(hwnd, message, wParam, lParam);
}

