/*
 * termproc.c
 *
 * Copyright (c) 1994 Software Research Associates, Inc.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Software Research Associates not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  Software Research
 * Associates makes no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied
 * warranty.
 */

#include <X11/Xlib.h>
#ifdef KANJI
#include <mbstring.h>
#endif /* KANJI */
#include <wndproc.h>
#include <util.h>
#include <string.h>
#include <string.h>

#define WM_NO_BUFFERED WM_USER+20

#define MAX_CXSCREEN 64
#define MAX_CYSCREEN 16
#define MIN_CXSCREEN 16
#define MIN_CYSCREEN  4
#define SCREEN_IMAGE(image, x, y) (&((image)[(y)][(x)]))
#define CXIMAGE MAX_CXSCREEN
#define CYIMAGE (MAX_CYSCREEN*5)

#define BUFFER_LENGTH 256

#define FG_COLOR RGB(0, 0, 0)
#define BG_COLOR RGB(255, 255, 255)
#define INIT_SCREEN_CX	CXIMAGE
#define INIT_SCREEN_CY	25

#define NO_SCROLL 0x00
#define EXIST_VSCROLL 0x01
#define EXIST_HSCROLL 0x02
#define V_SCROLLED 0x04
#define H_SCROLLED 0x08

typedef struct {
    int xCaret, yCaret;   	
    int nTop, nBottom;
    int nBottomOrg, nTopOrg;	 
    int nAmount;
    int nVScrollPos;
    int nVScrollRange;
    WORD wMode;
    char Image[CYIMAGE][CXIMAGE];
} SCREEN;

int cxChar, cyChar;
int cxFrame, cyFrame, cyCaption, cxVScroll;
int cxDiff, cyDiff;

XErrorHandler errorHandler;

extern void FAR PASCAL CaregeReturn(HWND, HDC, SCREEN *, int, int, int, int);
extern HWND FAR PASCAL __export CreateTerminal(HANDLE);
extern int FAR PASCAL DefaultErrorHandler(Display *, XErrorEvent *);

void
__far __pascal
RegisterTerminalClass(HINSTANCE hInstance)
{		    
    WNDCLASS wndclass;

    wndclass.style = CS_HREDRAW | CS_VREDRAW;
    wndclass.lpfnWndProc = TerminalProc;
    wndclass.cbClsExtra = 0;
    wndclass.cbWndExtra = 2;
    wndclass.hInstance = hInstance;
    wndclass.hIcon = LoadIcon(NULL, IDI_APPLICATION);
    wndclass.hCursor = LoadCursor(NULL, IDC_ARROW);
    wndclass.hbrBackground = (HBRUSH) GetStockObject(WHITE_BRUSH);
    wndclass.lpszMenuName = NULL;
    wndclass.lpszClassName = "Terminal";

    RegisterClass(&wndclass);
}

HWND 
__far __pascal
CreateTerminal(HANDLE hInstance)
{
    HDC hdc;
    TEXTMETRIC tm;

    hdc = GetDC((HWND)NULL);
    SelectObject(hdc, GetStockObject(SYSTEM_FIXED_FONT));
    GetTextMetrics(hdc, &tm);
    ReleaseDC((HWND) NULL, hdc);

    cxFrame = GetSystemMetrics(SM_CXFRAME);
    cyFrame = GetSystemMetrics(SM_CYFRAME);
    cyCaption = GetSystemMetrics(SM_CYCAPTION);
    cxVScroll = GetSystemMetrics(SM_CXVSCROLL);

    return  CreateWindow("Terminal", "wish", 
    		WS_OVERLAPPED|WS_SYSMENU|WS_MINIMIZEBOX|WS_VSCROLL|WS_THICKFRAME,
    		CW_USEDEFAULT,
    		CW_USEDEFAULT,
    		(tm.tmAveCharWidth * INIT_SCREEN_CX) + (cxFrame * 2) + cxVScroll,
    		(tm.tmHeight * INIT_SCREEN_CY) + cyCaption + (cyFrame * 2), 
    		NULL, NULL, hInstance, NULL);

}


long 
FAR PASCAL
TerminalProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    static SCREEN scr;
    static int cxClient, cyClient;
    static int cxScreen, cyScreen;
    static int nBuffer;
    static char Buffer[BUFFER_LENGTH + 1];
    int y;

    HDC hdc;
    PAINTSTRUCT ps;
    MINMAXINFO FAR * lpmmi;
    WINDOWPOS FAR *lpwp;
    BOOL bBuffered = 1;
    XErrorEvent ee;
    static HGLOBAL hGlobal;
    char *pGlobal, *pBuffer;

    switch(message) {
    case WM_CREATE:
	errorHandler = DefaultErrorHandler;

	scr.xCaret = 0;
	scr.yCaret = 0;
	scr.nTop = 0;
	scr.nBottom = 0;
	scr.nAmount = 1;
	scr.nVScrollPos = 0;
	scr.wMode = NO_SCROLL;
	nBuffer = 0;

	hGlobal = GlobalAlloc(GHND, BUFFER_LENGTH+1);
	SetWindowWord(hwnd, 0, (WORD) hGlobal);

	for (y = 0; y < MAX_CYSCREEN; y++) {
//	for (y = 0; y < CYIMAGE; y++) {
	    memset(SCREEN_IMAGE(scr.Image, 0, y), ' ', CXIMAGE);
	}
        return 0;


    case WM_SIZE:
        if (wParam == SIZE_MINIMIZED) {
            return 0;
        } 
        cxClient = LOWORD(lParam);
        cyClient = HIWORD(lParam);
	cxScreen = cxClient / cxChar;
	cyScreen = cyClient / cyChar;

	if (scr.wMode & V_SCROLLED) {
	    scr.wMode &= ~V_SCROLLED;
	    scr.nTop = scr.nTopOrg;
	    scr.nBottom = scr.nBottomOrg;
	    ShowCaret(hwnd);
	}    
	if (scr.nAmount > cyScreen) {
	    if (!(scr.wMode & EXIST_VSCROLL)) {
		scr.wMode |= EXIST_VSCROLL;
	    	SetWindowPos(hwnd, (HWND) 0, 0, 0, cxClient + cxDiff + cxVScroll, 
			cyClient + cyDiff, SWP_NOZORDER|SWP_NOMOVE);
	    }
	    scr.nTop = (scr.nBottom - (cyScreen - 1) + CYIMAGE) % CYIMAGE;
	    scr.yCaret = cyScreen - 1;
	} else {
	    if (scr.wMode & EXIST_VSCROLL) {
		scr.wMode &= ~EXIST_VSCROLL;
	    	SetWindowPos(hwnd, (HWND) 0, 0, 0, cxClient + cxDiff, 
			cyClient + cyDiff, SWP_NOZORDER|SWP_NOMOVE);
	    }
	    scr.nTop = 0;
	    scr.yCaret = scr.nAmount - 1;
	}
	scr.nVScrollRange = max(0, scr.nAmount - cyScreen);
	SetScrollRange(hwnd, SB_VERT, 0, scr.nVScrollRange, FALSE);
	SetScrollPos(hwnd, SB_VERT, scr.nVScrollRange, TRUE);
	SetCaretPos(scr.xCaret * cxChar, scr.yCaret * cyChar);
        return 0;

    case WM_VSCROLL:
    	if (!(scr.wMode & V_SCROLLED)) {
    	    scr.wMode  |= V_SCROLLED;
	    scr.nTopOrg = scr.nTop;
	    scr.nBottomOrg = scr.nBottom;
	    scr.nVScrollPos = scr.nVScrollRange;
	    HideCaret(hwnd);
	}
    	switch (wParam) {
	case SB_LINEUP:
	    scr.nVScrollPos--;
	    scr.nTop = (scr.nTop + CYIMAGE - 1) % CYIMAGE;
	    break;
	case SB_LINEDOWN:
	    scr.nVScrollPos++; 
	    scr.nTop = (scr.nTop + CYIMAGE + 1) % CYIMAGE;
	    break;
	case SB_PAGEUP:
	    if (scr.nAmount == CYIMAGE) {
	    	scr.nTop = (scr.nTop + CYIMAGE - (cyScreen - 1)) % CYIMAGE;
	    } else {
	    	scr.nTop = max(0, scr.nTop - (cyScreen - 1));
	    }
	    scr.nVScrollPos -= cyScreen - 1;
       	    break;
	case SB_PAGEDOWN:
	    scr.nTop = (scr.nTop + CYIMAGE + (cyScreen - 1)) % CYIMAGE;
	    scr.nVScrollPos += cyScreen - 1;
	    break;
	case SB_THUMBPOSITION:
	    if (scr.nAmount < CYIMAGE) {
	    	scr.nTop = LOWORD(lParam);
	    } else {
	    	scr.nTop =  (LOWORD(lParam) + scr.nTopOrg + cyScreen) % CYIMAGE;
	    }
	    scr.nVScrollPos =  LOWORD(lParam);
	default:
	    break;
	}

	if (scr.nVScrollPos >= scr.nVScrollRange) {
	    scr.nTop = scr.nTopOrg;
	    scr.nBottom = scr.nBottomOrg;
	    scr.wMode &= ~V_SCROLLED;
//	    for(y = 0; y < cyScreen; y++) {
//		TextOut(hdc, 0, y * cyChar, SCREEN_IMAGE(scr.Image, 0, (scr.nTop + y) %CYIMAGE), cxScreen);
//	    }
	    ShowCaret(hwnd);
	} else if (scr.nVScrollPos < 0) {
	    scr.nVScrollPos = 0;
	    scr.nTop = (scr.nAmount < CYIMAGE)? 0 : (scr.nBottomOrg + 1)%CYIMAGE;
	    scr.nBottom = scr.nTop + cyScreen -1;
	} else {
//	    scr.nBottom = scr.nTop + cyScreen -1;
	}
	if (scr.nVScrollPos != GetScrollPos(hwnd, SB_VERT)) {
	    SetScrollPos(hwnd, SB_VERT, scr.nVScrollPos, TRUE);
	    InvalidateRect(hwnd, NULL, TRUE);
	}

	return 0;

    case WM_NO_BUFFERED:
	bBuffered = 0;

    case WM_CHAR:
	hdc = GetDC(hwnd);
	SelectObject(hdc, GetStockObject(SYSTEM_FIXED_FONT));
	SetBkColor(hdc, BG_COLOR);
	SetTextColor(hdc, FG_COLOR);
	if (scr.wMode & V_SCROLLED) {
	    scr.wMode &= ~V_SCROLLED;
	    scr.nTop = scr.nTopOrg;
	    scr.nBottom = scr.nBottomOrg;
	    SetScrollPos(hwnd, SB_VERT, scr.nVScrollRange, TRUE);
	    for(y = 0; y < cyScreen; y++) {
		TextOut(hdc, 0, y * cyChar, SCREEN_IMAGE(scr.Image, 0, (scr.nTop + y) %CYIMAGE), cxScreen);
	    }
	    ShowCaret(hwnd);
	}    
        switch ((char) wParam) {
	case '\r':
	case '\n':
	    if (bBuffered) {
		if (nBuffer == BUFFER_LENGTH) {
		    MessageBeep(MB_OK);
		    break;
		}
	    	Buffer[nBuffer] = '\n';
	    	Buffer[nBuffer + 1] = '\0';
		nBuffer = 0;
	    }
	    CaregeReturn(hwnd, hdc, &scr, cxClient, cyClient, cxScreen, cyScreen);
	    if (bBuffered) {
		pBuffer = Buffer;
		pGlobal = GlobalLock(hGlobal);
	    	while (*pGlobal++ = *pBuffer++);
	    	GlobalUnlock(hGlobal);
		PostMessage(hwnd, WM_USER, hGlobal, 0L);
	    }
	    break;
	case '\b':
	    if(scr.xCaret > 0) {
#ifdef KANJI	        
	        int cnt, flg = 0;

		for (cnt = 0; cnt < scr.xCaret; cnt++) {
		    if (_ismbblead((unsigned char) (*SCREEN_IMAGE(scr.Image, cnt, (scr.nTop + scr.yCaret) % CYIMAGE))) && flg != 1)
		    	flg = 1;
		    else if (_ismbbtrail((unsigned char) (*SCREEN_IMAGE(scr.Image, cnt, (scr.nTop + scr.yCaret) % CYIMAGE))) && flg == 1)
			flg = 2;
		    else
		    	flg = 0;
		}
		if (flg == 2 && scr.xCaret > 1 && nBuffer > 1) {
		    scr.xCaret--;
		    nBuffer--;
		    *(SCREEN_IMAGE(scr.Image, scr.xCaret, (scr.nTop + scr.yCaret) % CYIMAGE)) = ' ';
		}    		 
#endif
		if (nBuffer > 0) {
		    nBuffer--;
		} else {
		    break;
		}
	    	scr.xCaret--;
		*(SCREEN_IMAGE(scr.Image, scr.xCaret, (scr.nTop + scr.yCaret) % CYIMAGE)) = ' ';
		HideCaret(hwnd);
	        TextOut(hdc, 0, scr.yCaret * cyChar, SCREEN_IMAGE(scr.Image, 0, (scr.nTop + scr.yCaret) % CYIMAGE), cxScreen);
	        ShowCaret(hwnd);
	    }
	    break;
	case '\t':
	    do {
	    	if (bBuffered) {
	    	    SendMessage(hwnd, WM_CHAR, ' ', 0L);
		} else {
	    	    SendMessage(hwnd, WM_NO_BUFFERED, ' ', 0L);
		}
	    } while (scr.xCaret % 8 != 0);
	    break;
	default:    
	    if (bBuffered) {
		if (nBuffer == BUFFER_LENGTH) {
		    MessageBeep(MB_OK);
		    break;
		}
	    	Buffer[nBuffer] = (char) wParam;
		nBuffer++;
	    }
#ifdef KANJI	    
	    if (_ismbblead((unsigned short) wParam) && scr.xCaret + 1 == cxScreen) {
	    	SendMessage(hwnd, WM_NO_BUFFERED, ' ', 0L);
	    }
#endif	    
    	    *(SCREEN_IMAGE(scr.Image, scr.xCaret, scr.nBottom)) = (char) wParam;
	    HideCaret(hwnd);
	    TextOut(hdc, 0, scr.yCaret * cyChar, SCREEN_IMAGE(scr.Image, 0, scr.nBottom), cxScreen);

	    if (scr.xCaret + 1 == cxScreen) {
	    	scr.xCaret = 0;
		CaregeReturn(hwnd, hdc, &scr, cxClient, cyClient, cxScreen, cyScreen);
	    } else {
	    	scr.xCaret++;
	    }
	    ShowCaret(hwnd);
	}        
	ReleaseDC(hwnd, hdc);
	SetCaretPos(scr.xCaret * cxChar, scr.yCaret * cyChar);
    	return 0;

	case WM_USER+9 :
	    ee.resourceid = (XID) lParam;
	    ee.error_code = (unsigned char) wParam;
	    (*errorHandler)(NULL, &ee);
	    return 0;
    case WM_PAINT:
    	hdc = BeginPaint(hwnd, &ps);
	SelectObject(hdc, GetStockObject(SYSTEM_FIXED_FONT));
	SetBkColor(hdc, BG_COLOR);
	SetTextColor(hdc, FG_COLOR);
	
	for(y = 0; y < cyScreen; y++) {
	    TextOut(hdc, 0, y * cyChar, SCREEN_IMAGE(scr.Image, 0, (scr.nTop + y)%CYIMAGE), cxScreen);
	}
	EndPaint(hwnd, &ps);
	return 0;
	
    case WM_SETFOCUS:
    	CreateCaret(hwnd, NULL, cxChar, cyChar);
	SetCaretPos(scr.xCaret * cxChar, scr.yCaret * cyChar);
	ShowCaret(hwnd);
	return 0;

    case WM_KILLFOCUS:
    	HideCaret(hwnd);
	DestroyCaret();
	return 0;

    case WM_DESTROY:
    	PostQuitMessage(0);
    	return 0;

    case WM_CLOSE:
	pBuffer = "exit\n";
	pGlobal = GlobalLock(hGlobal);
    	while (*pGlobal++ = *pBuffer++);
    	GlobalUnlock(hGlobal);
	PostMessage(hwnd, WM_USER, hGlobal, 0L);
	return 0;

    case WM_GETMINMAXINFO:
	lpmmi = (MINMAXINFO FAR *) lParam;
#ifdef FIXED_SIZE
	if (scr.wMode & EXIST_VSCROLL) {
	    lpmmi->ptMinTrackSize.x = (cxChar * MAX_CXSCREEN) + cxDiff + cxVScroll;
	    lpmmi->ptMaxTrackSize.x = (cxChar * MAX_CXSCREEN) + cxDiff + cxVScroll;
	} else {
	    lpmmi->ptMinTrackSize.x = (cxChar * MAX_CXSCREEN) + cxDiff;
	    lpmmi->ptMaxTrackSize.x = (cxChar * MAX_CXSCREEN) + cxDiff;
	}
	lpmmi->ptMinTrackSize.y = (cyChar * MAX_CYSCREEN) + cyDiff;
	lpmmi->ptMaxTrackSize.y = (cyChar * MAX_CYSCREEN) + cyDiff;
#else
	if (scr.wMode & EXIST_VSCROLL) {
	    lpmmi->ptMinTrackSize.x = (cxChar * MIN_CXSCREEN) + cxDiff + cxVScroll;
	    lpmmi->ptMaxTrackSize.x = (cxChar * MAX_CXSCREEN) + cxDiff + cxVScroll;
	} else {
	    lpmmi->ptMinTrackSize.x = (cxChar * MIN_CXSCREEN) + cxDiff;
	    lpmmi->ptMaxTrackSize.x = (cxChar * MAX_CXSCREEN) + cxDiff;
	}
	lpmmi->ptMinTrackSize.y = (cyChar * MIN_CYSCREEN) + cyDiff;
	lpmmi->ptMaxTrackSize.y = (cyChar * MAX_CYSCREEN) + cyDiff;
#endif /* FIXED_SIZE */
	return 0;

#ifndef FIXED_SIZE
    case WM_WINDOWPOSCHANGING:
    	lpwp = (WINDOWPOS FAR *) lParam;
	if (lpwp->cx <= 50) {
	    break;
	}
	if (scr.wMode & EXIST_VSCROLL) {
	    lpwp->cx = (lpwp->cx - (cxDiff + cxVScroll)) / cxChar * cxChar + (cxDiff + cxVScroll);	
	} else {
	    lpwp->cx = (lpwp->cx - cxDiff) / cxChar * cxChar + cxDiff;	
	}
	lpwp->cy = (lpwp->cy - cyDiff) / cyChar * cyChar + cyDiff;
    	return 0;	
#endif /* !FIXED_SIZE */
    }
    return DefWindowProc(hwnd, message, wParam, lParam);
}

void
FAR PASCAL
CaregeReturn(HWND hwnd, HDC hdc, SCREEN * lpscr, int cxClient, int cyClient, 
	int cxScreen, int cyScreen)
{
    int y;

    lpscr->nBottom = (lpscr->nBottom + 1) % CYIMAGE;
    if (lpscr->nAmount < CYIMAGE) {
    	lpscr->nAmount++;
    }
    if (lpscr->nAmount > cyScreen) {
    	lpscr->nTop = (lpscr->nTop + 1) % CYIMAGE;
	lpscr->nVScrollRange = lpscr->nAmount - cyScreen;
	if (!(lpscr->wMode & EXIST_VSCROLL)) {
	    lpscr->wMode |= EXIST_VSCROLL;
	    SetWindowPos(hwnd, (HWND) 0, 0, 0, cxClient + cxDiff + cxVScroll, 
		    cyClient + cyDiff, SWP_NOZORDER|SWP_NOMOVE);
	    SetScrollRange(hwnd, SB_VERT, 0, lpscr->nVScrollRange, FALSE);
	    SetScrollPos(hwnd, SB_VERT, lpscr->nVScrollRange, TRUE);
	} else {
	    SetScrollRange(hwnd, SB_VERT, 0, lpscr->nVScrollRange, FALSE);
	}
    }
    lpscr->xCaret = 0;
    if (lpscr->yCaret + 1 == cyScreen) {
	HideCaret(hwnd);
    	for (y = 0; y < cyScreen - 1; y++) {
	    TextOut(hdc, 0, y * cyChar, SCREEN_IMAGE(lpscr->Image, 0, (y + lpscr->nTop) % CYIMAGE), cxScreen);
	}
	memset(SCREEN_IMAGE(lpscr->Image, 0, lpscr->nBottom), ' ', CXIMAGE);
	TextOut(hdc, 0, (cyScreen - 1) * cyChar, SCREEN_IMAGE(lpscr->Image, 0, lpscr->nBottom), cxScreen);
        ShowCaret(hwnd);
    } else {
    	lpscr->yCaret++;
    }
}
