/*  _ __ ___ _
 * | |\ /  /| |  $Id: CdlFile.C,v 1.2 1995/03/03 10:25:07 deans Exp $
 * | | /  / | |  Copyright (C) 1995 IXI Limited.
 * |_|/__/_\|_|  IXI Limited, Cambridge, England.
 *
 * Component   : CdlFile.C
 *
 * Author      : Dean Sheehan (deans@x.co.uk)
 *
 * Description : Contains the implementation for the CdlFile class that 
 *               models the CDL file read.
 *
 * License     :
			Object Tcl License & Copyright
			------------------------------

IXI Object Tcl software, both binary and source (hereafter, Software) is copyrighted by IXI Limited (IXI), and ownership remains with IXI. 

IXI grants you (herafter, Licensee) a license to use the Software for academic, research and internal business purposes only, without a fee. Licensee may distribute the binary and source code (if required) to third parties provided that the copyright notice and this statement appears on all copies and that no charge is associated with such copies. 

Licensee may make derivative works. However, if Licensee distributes any derivative work based on or derived from the Software, then Licensee will (1) notify IXI regarding its distribution of the derivative work, and (2) clearly notify users that such derivative work is a modified version and not the original IXI Object Tcl distributed by IXI. IXI strongly recommends that Licensee provide IXI the right to incorporate such modifications into future releases of the Software under these license terms. 

Any Licensee wishing to make commercial use of the Software should contact IXI, to negotiate an appropriate license for such commercial use. Commercial use includes (1) integration of all or part of the source code into a product for sale or license by or on behalf of Licensee to third parties, or (2) distribution of the binary code or source code to third parties that need it to utilize a commercial product sold or licensed by or on behalf of Licensee. 

IXI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THIS SOFTWARE FOR ANY PURPOSE. IT IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY. IXI SHALL NOT BE LIABLE FOR ANY DAMAGES WHATSOEVER SUFFERED BY THE USERS OF THIS SOFTWARE. 

Copyright (C) 1995, IXI Limited 

By using or copying this Software, Licensee agrees to abide by the copyright law and all other applicable laws of England and the U.S., including, but not limited to, export control laws, and the terms of this license. IXI shall have the right to terminate this license immediately by written notice upon Licensee's breach of, or non-compliance with, any of its terms. Licensee may be held legally responsible for any copyright infringement that is caused or encouraged by Licensee's failure to abide by the terms of this license. 

Comments and questions are welcome and can be sent to
otcl@x.co.uk 

For more information on copyright and licensing issues, contact: 
Legal Department, IXI Limited, Vision Park, Cambridge CB4 4ZR,
ENGLAND. 

 *
 */

static char rcs_id [] = "$Id: CdlFile.C,v 1.2 1995/03/03 10:25:07 deans Exp $";

// System Includes
#include <string.h>

// Tcl Includes
#include <tcl.h>
#include <tclInt.h>

// Local Includes
#include "CdlFile.H"
#include "CdlItem.H"
#include "CdlMethod.H"
#include "CdlArg.H"
#include "CdlRtn.H"

CdlFile::CdlFile (char *n)
{
   name = strdup(n);
   noOfItems = 0;
   interp = Tcl_CreateInterp();
   methodUnderConstruction = NULL;
}

CdlFile::~CdlFile ()
{
   for (int c = 0; c < noOfItems; c++)
   {
      delete item[c];
   }
   Tcl_DeleteInterp(interp);
}

int CdlFile::parse (void)
{
   rmvStdTclCmds();

   addItemCmds();

   int res = Tcl_EvalFile(interp,name);
   if (res != TCL_OK)
   {
      cerr << "Error in input file: \"" << interp->result << "\"" << endl;
      return 0;
   }

   return 1;
}

int CdlFile::genOutput (char *n)
{
   ofstream of(n);
   if (!of)
   {
      return 0;
   }

   genHeader (of);
 
   for (int i = 0; i < noOfItems; i++)
   {
      item[i]->genOutput(of);
      of << endl;
   }

   genTrailer (of);

   return 1;
}

void CdlFile::rmvStdTclCmds (void)
{
   // This is a really nasty way of doing it, but...
   Interp *i = (Interp*)interp;
   Tcl_InitHashTable(&i->commandTable,TCL_STRING_KEYS);
}

void CdlFile::genHeader (ofstream &of)
{
   of << "#include <iostream.h>" << endl;
   of << "#include <stdlib.h>" << endl;
   of << "#include <Otcl.H>" << endl;
   of << "#include <OtclObject.H>" << endl;
   of << "#include <OtclPart.H>" << endl;
   of << endl;
}

void CdlFile::genTrailer (ofstream &)
{
}

void CdlFile::addItemCmds (void)
{
   Tcl_CreateCommand(interp,"class",CdlFile::classCmd,
                     (ClientData)this,(Tcl_CmdDeleteProc*)NULL);
   Tcl_CreateCommand(interp,"pass",CdlFile::passCmd,
                     (ClientData)this,(Tcl_CmdDeleteProc*)NULL);
}

void CdlFile::rmvItemCmds (void)
{
   Tcl_DeleteCommand(interp,"class");
   Tcl_DeleteCommand(interp,"pass");
}

void CdlFile::addMethodCmds (void)
{
   Tcl_CreateCommand(interp,"method",CdlFile::methodCmd,
                     (ClientData)this,(Tcl_CmdDeleteProc*)NULL);
   Tcl_CreateCommand(interp,"classMethod",CdlFile::classMethodCmd,
                     (ClientData)this,(Tcl_CmdDeleteProc*)NULL);
   Tcl_CreateCommand(interp,"constructor",CdlFile::constructorCmd,
                     (ClientData)this,(Tcl_CmdDeleteProc*)NULL);

}

void CdlFile::rmvMethodCmds (void)
{
   Tcl_DeleteCommand(interp,"method");
   Tcl_DeleteCommand(interp,"classMethod");
   Tcl_DeleteCommand(interp,"constructor");
}

void CdlFile::addArgCmds (void)
{
   Tcl_CreateCommand(interp,"void",CdlFile::argCmd,
                     (ClientData)this,(Tcl_CmdDeleteProc*)NULL);
   CdlArg *current = CdlArg::exemplarHead;
   while (current != NULL)
   {
      Tcl_CreateCommand(interp,current->giveCommand(),CdlFile::argCmd,
                        (ClientData)this,(Tcl_CmdDeleteProc*)NULL);
      current = current->nextExemplar;
   }
}

void CdlFile::rmvArgCmds (void)
{
   Tcl_DeleteCommand(interp,"void");
   CdlArg *current = CdlArg::exemplarHead;
   while (current != NULL)
   {
      Tcl_DeleteCommand(interp,current->giveCommand());
      current = current->nextExemplar;
   }
}

void CdlFile::addRtnCmds (void)
{
   Tcl_CreateCommand(interp,"void",CdlFile::rtnCmd,
                     (ClientData)this,(Tcl_CmdDeleteProc*)NULL);
   CdlRtn *current = CdlRtn::exemplarHead;
   while (current != NULL)
   {
      Tcl_CreateCommand(interp,current->giveCommand(),CdlFile::rtnCmd,
                        (ClientData)this,(Tcl_CmdDeleteProc*)NULL);
      current = current->nextExemplar;
   }
}

void CdlFile::rmvRtnCmds (void)
{
   Tcl_DeleteCommand(interp,"void");
   CdlRtn *current = CdlRtn::exemplarHead;
   while (current != NULL)
   {
      Tcl_DeleteCommand(interp,current->giveCommand());
      current = current->nextExemplar;
   }
}

int CdlFile::classCmd (ClientData cd, Tcl_Interp *, int argc, char *argv[])
{
   CdlFile *cdlF = (CdlFile*)cd;
   return cdlF->newClass(argc,argv);
}

int CdlFile::passCmd (ClientData cd, Tcl_Interp *, int argc, char *argv[])
{
   CdlFile *cdlF = (CdlFile*)cd;
   return cdlF->pass(argc,argv);
}

int CdlFile::constructorCmd (ClientData cd, Tcl_Interp *, int argc,
                               char *argv[])
{
   CdlFile *cdlF = (CdlFile*)cd;
   return cdlF->constructor(argc,argv);
}

int CdlFile::methodCmd (ClientData cd, Tcl_Interp *, int argc, char *argv[])
{
   CdlFile *cdlF = (CdlFile*)cd;
   return cdlF->method(argc,argv);
}

int CdlFile::classMethodCmd (ClientData cd, Tcl_Interp *, int argc,
                               char *argv[])
{
   CdlFile *cdlF = (CdlFile*)cd;
   return cdlF->classMethod(argc,argv);
}

int CdlFile::argCmd (ClientData cd, Tcl_Interp *, int argc,
                               char *argv[])
{
   CdlFile *cdlF = (CdlFile*)cd;
   return cdlF->arg(argc,argv);
}

int CdlFile::rtnCmd (ClientData cd, Tcl_Interp *, int argc,
                               char *argv[])
{
   CdlFile *cdlF = (CdlFile*)cd;
   return cdlF->rtn(argc,argv);
}

int CdlFile::newClass (int argc, char *argv[])
{
   if (argc != 3)
   {
      Tcl_SetResult(interp,"wrong # args",TCL_STATIC);
      return TCL_ERROR;
   }

   item[noOfItems] = new CdlClass(argv[1]);
   noOfItems++;

   rmvItemCmds();

   addMethodCmds();

   int res = Tcl_Eval(interp,argv[2]);

   rmvMethodCmds ();

   addItemCmds(); 

   if (res == TCL_OK && ((CdlClass*)item[noOfItems-1])->getConstructor()
       == NULL)
   {
      // Set up default constructor
      char n[50];
      sprintf(n,"%s%s",((CdlClass*)item[noOfItems-1])->giveName(),
              OTCL_PART_SUFFIX);
      ((CdlClass*)item[noOfItems-1])->setConstructor(new CdlConstructor(n));
   }

   return res;
}

int CdlFile::pass (int , char *argv[])
{
   item[noOfItems] = new CdlPass(argv[1]);
   noOfItems++;
   return TCL_OK;
}

int CdlFile::constructor (int argc, char *argv[])
{
   if (argc != 2)
   {
      Tcl_SetResult(interp,"wrong # args",TCL_STATIC);
      return TCL_ERROR;
   }

   CdlClass *c = (CdlClass*)item[noOfItems-1];

   char name[50];
   sprintf(name,"%s%s",c->giveName(),OTCL_PART_SUFFIX);
   methodUnderConstruction = new CdlConstructor(name);

   rmvMethodCmds();
   addArgCmds();

   int res = Tcl_Eval(interp,argv[1]);

   rmvArgCmds();
   addMethodCmds();

   if (res == TCL_OK)
   {
      c->setConstructor((CdlConstructor*)methodUnderConstruction);
      methodUnderConstruction = NULL;
   }

   return res;
}

int CdlFile::method (int argc, char *argv[])
{
   if (argc != 5)
   {
      Tcl_SetResult(interp,"wrong # args",TCL_STATIC);
      return TCL_ERROR;
   }

   CdlInstanceMethod *m = new CdlInstanceMethod(argv[1]);
   methodUnderConstruction = m;

   if (strcmp(argv[2],STATIC_FLAG) == 0)
   {
      m->setBinding(CdlMethod::STATIC);
   }
   else if (strcmp(argv[2],DYNAMIC_FLAG) == 0)
   {
      m->setBinding(CdlMethod::DYNAMIC);
   }
   else
   {
      Tcl_SetResult(interp,"unknown 3rd argument: must be \"static\" or \"dynamic\"",TCL_STATIC);
      return TCL_ERROR;

   }

   rmvMethodCmds();
   addArgCmds();

   int res = Tcl_Eval(interp,argv[3]);

   rmvArgCmds();

   if (res == TCL_OK)
   {
      addRtnCmds();

      res = Tcl_Eval(interp,argv[4]);

      rmvRtnCmds();
   }

   addMethodCmds();

   if (res == TCL_OK)
   {
      ((CdlClass*)item[noOfItems-1])->addInstanceMethod(m);
      methodUnderConstruction = NULL;
   }

   if (m->getRtn() == NULL && !m->returnIsVoid())
   {
      m->setRtn(new CdlIntRtn());
   }

   return res;
}

int CdlFile::classMethod (int argc, char *argv[])
{
   if (argc != 4)
   {
      Tcl_SetResult(interp,"wrong # args",TCL_STATIC);
      return TCL_ERROR;
   }

   CdlClassMethod *m = new CdlClassMethod(argv[1]);
   methodUnderConstruction = m;

   rmvMethodCmds();
   addArgCmds();

   int res = Tcl_Eval(interp,argv[2]);

   rmvArgCmds();

   if (res == TCL_OK)
   {
      addRtnCmds();

      res = Tcl_Eval(interp,argv[3]);

      rmvRtnCmds();
   }
 
   if (res == TCL_OK)
   {
      ((CdlClass*)item[noOfItems-1])->addClassMethod(m);
      methodUnderConstruction = NULL;
   }

   if (m->getRtn() == NULL && !m->returnIsVoid())
   {
      m->setRtn(new CdlIntRtn());
   }

   addMethodCmds();

   return res;
}

int CdlFile::arg (int argc, char *argv[])
{
   if (methodUnderConstruction->argsAreVoid())
   {
      return TCL_ERROR;
   }

   if (strcmp(argv[0],"void") == 0)
   {
      methodUnderConstruction->setVoidArgs();
      return TCL_OK;
   }

   CdlArg *current = CdlArg::exemplarHead;
   while (current != NULL)
   {
      if (strcmp(current->giveCommand(),argv[0]) == 0)
      {
         current = current->instantiate();
         if (!current->setArgs(argc-1,&argv[1]))
         {
            return TCL_ERROR;
         }

         methodUnderConstruction->addArg(current);
         return TCL_OK;
      }
      current = current->nextExemplar;
   }

   return TCL_OK;
}

int CdlFile::rtn (int argc, char *argv[])
{
   if (methodUnderConstruction->getRtn() != NULL)
   {
      return TCL_ERROR;
   }
   if (methodUnderConstruction->returnIsVoid())
   {
      return TCL_ERROR;
   }

   if (strcmp(argv[0],"void") == 0)
   {
      methodUnderConstruction->setVoidReturn();
      return TCL_OK;
   }

   CdlRtn *current = CdlRtn::exemplarHead;
   while (current != NULL)
   {
      if (strcmp(current->giveCommand(),argv[0]) == 0)
      {
         current = current->instantiate();
         if (!current->setArgs(argc-1,&argv[1]))
         {
            return TCL_ERROR;
         }
         
         methodUnderConstruction->setRtn(current);
         return TCL_OK;
      }
      current = current->nextExemplar;
   }

   return TCL_ERROR;
}
