/* lsobj.c - ls for BOS object storage files
 *
 * Copyright (C) 1992,1993 Engineering Design Research Center
 *
 * Author: Sean Levy (snl+@cmu.edu)
 *         n-dim Group
 *         Engineering Design Research Center
 *         Carnegie Mellon University
 *         5000 Forbes Ave / PGH, PA / 51221
 *
 *         Fax: (412) 268-5229
 *         Voice: (412) 268-5226
 */

/*
 * lsobjs.c,v 1.5 1992/08/11 15:22:28 snl Exp
 *
 * lsobjs.c,v
 * Revision 1.5  1992/08/11  15:22:28  snl
 * updated makefiles, finished 0.90 freeze
 *
 * Revision 1.4  1992/08/03  12:35:32  snl
 * bug fixes
 *
 * Revision 1.3  1992/07/29  22:49:55  snl
 * added garbage scanning options and code
 *
 * Revision 1.2  1992/07/29  18:49:49  snl
 * added -F option
 *
 * Revision 1.1  1992/07/29  18:26:47  snl
 * first checkin
 *
 */

#include <stdio.h>
#include <bos.h>
#include <list.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>

static char rcsID[] = "lsobjs.c,v 1.5 1992/08/11 15:22:28 snl Exp";

typedef struct object_dumper {
  List_Links links;
  int version;
  void (*dump_obj_fn)();
  void (*scan_obj_fn)();
} object_dumper;

#define ASIZE(a) (sizeof(a)/sizeof(a[0]))

#define LONG_LISTING		0x0001
#define SHOW_EVANESCENT		0x0002
#define OBJ_NAME_PATTERN	0x0004
#define OBJ_SLOT_PATTERN	0x0008
#define SHOW_ONLY_VERSION	0x0010
#define QUIET			0x0020
#define NO_STATS		0x0040
#define NO_VALUE		0x0080
#define VALUE_SAME_LINE		0x0100
#define DELIMIT_VALUE		0x0200
#define ONLY_STATS		0x0400
#define LOOK_FOR_GARBAGE	0x0800
#define GARBAGE_ONLY		0x1000

#define STORAGE_FILE_NAME "bosobjects.gdbm"
#define LL(s1,s2) ((global_options&LONG_LISTING)?s1:s2)

int global_options = 0;
int global_version = 0;
int global_block_size = 512;
int global_mode = 0x1A4;
char *global_obj_pattern = (char *)0;
char *global_slot_pattern = (char *)0;
char *global_storage_file_name = STORAGE_FILE_NAME;
int global_nbytes = 0;
int global_nslots = 0;
int global_nEslots = 0;
int global_slot_counts[100];
int global_foreign_counts[1000];
int global_nobjs = 0;
int global_suspicious_nslots = 60;
int global_suspicious_foreign_subtype = 20;
int global_suspicious_pri = 10;

List_Links *dumpers = (List_Links *)0;

main(argc, argv)
     int argc;
     char **argv;
{
  void dumpobjs(), scanobjs(), usage(), initialize();
  int c, i;
  extern int optind;
  extern char *optarg;
  char *err_msg;

  /* initialize */
  initialize();
  /* parse args */
  while ((c = getopt(argc, argv, "hv:leO:S:qsVxdtF:gGn:f:p:")) != EOF) {
    switch (c) {
    case 'v':
      if (sscanf(optarg, "%d", &global_version) != 1)
	usage(argv[0], "bad -v option value");
      global_options |= SHOW_ONLY_VERSION;
      break;
    case 'l':
      global_options |= LONG_LISTING;
      break;
    case 'e':
      global_options |= SHOW_EVANESCENT;
      break;
    case 'O':
      if ((err_msg = re_comp(optarg)) != (char *)0)
	usage(argv[0], err_msg);
      global_obj_pattern = optarg;
      global_options |= OBJ_NAME_PATTERN;
      break;
    case 'S':
      if ((err_msg = re_comp(optarg)) != (char *)0)
	usage(argv[0], err_msg);
      global_slot_pattern = optarg;
      global_options |= OBJ_SLOT_PATTERN;
    case 'q':
      global_options |= QUIET;
      break;
    case 's':
      global_options |= NO_STATS;
      break;
    case 'V':
      global_options |= NO_VALUE;
      break;
    case 'x':
      global_options |= VALUE_SAME_LINE | LONG_LISTING;
      global_options &= ~NO_VALUE;
      break;
    case 'd':
      global_options |= DELIMIT_VALUE | LONG_LISTING;
      global_options &= ~NO_VALUE;
      break;
    case 't':
      global_options |= ONLY_STATS;
      global_options &= ~LONG_LISTING;
      global_options &= ~NO_STATS;
      break;
    case 'F':
      global_storage_file_name = optarg;
      break;
    case 'g':
      global_options |= LOOK_FOR_GARBAGE;
      break;
    case 'G':
      global_options |= LOOK_FOR_GARBAGE | GARBAGE_ONLY;
      break;
    case 'n':
    case 'f':
    case 'p':
      { int x;

	if (sscanf(optarg, "%d", &x) != 1) {
	  char errmsg[100];

	  sprintf("-%c takes an integer argument; %s does not scan as int",
		  c, optarg);
	  usage(argv[0], errmsg);
	}
	switch (c) {
	case 'n':
	  global_suspicious_nslots = x;
	  break;
	case 'f':
	  global_suspicious_foreign_subtype = x;
	  break;
	case 'p':
	  global_suspicious_pri = x;
	  break;
	}
      }
      break;
    case 'h':
    default:
      usage(argv[0], "list BOS object file contents");
      break;
    }
  }

  /* dump objects */
  if (!(global_options & GARBAGE_ONLY)) {
    if (optind == argc)
      dumpobjs("OBJECTS");
    else
      for (i = optind; i < argc; i++)
	dumpobjs(argv[i]);
  }
  if (global_options & LOOK_FOR_GARBAGE) {
    if (optind == argc)
      scanobjs("OBJECTS");
    else
      for (i = optind; i < argc; i++)
	scanobjs(argv[i]);
  }  
}

void
usage(prog_name, msg)
     char *prog_name;
     char *msg;
{
  static char *_msg[] = {
    "Usage: %s [-lhqseVxdtgG] [-v vers] [-O pat] [-S pat] [-F file] [dir...]\n",
    "  -h\t\tgive this message",
    "  -l\t\tlong listing",
    "  -q\t\tbe quiet",
    "  -s\t\tdon't print stats",
    "  -e\t\tprocess evanescent slots",
    "  -V\t\tdon't show slot value (implies -l)",
    "  -x\t\tshow slot value on same line (implies -l)",
    "  -d\t\tdelimit slot value (implies -l)",
    "  -t\t\tonly print totals",
    "  -g\t\tscan object store for garbage slots and/or objects",
    "  -G\t\tonly do -g, don't do a normal listing",
    "  -O pat\tonly show objects matching regexp pat",
    "  -S pat\tonly show slots matching regexp pat",
    "  -F file\tuse file as the GDBM filename (def=bosobjects.gdbm)",
    "  -v vers\tonly show objects matching version vers",
    (char *)0
  };
  int i;

  if (msg != (char *)0)
    fprintf(stderr, "%s: %s\n", prog_name, msg);
  fprintf(stderr, _msg[0], prog_name);
  for (i = 1; _msg[i] != (char *)0; i++)
    fprintf(stderr, "%s\n", _msg[i]);
  exit(1);
}

void
dumpobjs(filename)
     char *filename;
{
  GDBM_FILE dbf, open_object_file();
  datum k, d;
  void fatal(), init_stats();
  char file[300];
  int dumpobj();
  struct stat info;

  init_stats();
  sprintf(file, "%s/%s", filename, global_storage_file_name);
  if (stat(file, &info)) {
    perror(file);
    return;
  }
  dbf = gdbm_open(file, global_block_size, GDBM_READER, global_mode, fatal);
  if (dbf == (GDBM_FILE)0) {
    fprintf(stderr, "%s: could not open GDBM database\n", filename);
    return;
  }
  if (!(global_options & LONG_LISTING)) {
    if (!(global_options & ONLY_STATS)) {
      if (global_options & SHOW_EVANESCENT)
	printf("NAME               \tV\tSLOTS\tE\tBYTES\n");
      else
	printf("NAME               \tV\tSLOTS\tBYTES\n");
    }
  }
  k = gdbm_firstkey(dbf);
  while (k.dptr != NULL) {
    d = gdbm_fetch(dbf, k);
    if (d.dptr == NULL) {
      fprintf(stderr, "%s:%s -- key found but not data; skipped.\n",
	      filename, k.dptr);
      continue;
    }
    if (dumpobj(k.dptr, d.dptr))
      global_nobjs++;
    k = gdbm_nextkey(dbf, k);
  }
  gdbm_close(dbf);
  if (!(global_options & NO_STATS)) {
    int i, wasted;
    char *slotTypeString();
    double ovhd;

    wasted = info.st_size - global_nbytes;
    if (!wasted)
      ovhd = 0;
    else {
      ovhd = (double)wasted / (double)info.st_size;
      ovhd *= 100.0;
    }
    if (!(global_options & ONLY_STATS))
      printf("\n");
    printf("totals: %d/%d bytes (%.2g%% ovhd), %d objs, %d/%d slots/E\n",
	   global_nbytes, info.st_size, ovhd,
	   global_nobjs, global_nslots, global_nEslots);
    printf("slot type counts:\n");
    for (i = 0; i < ASIZE(global_slot_counts); i++)
      if (global_slot_counts[i])
	printf("  %s: %d\n", slotTypeString(i), global_slot_counts[i]);
    if (global_slot_counts[Bos_SLOT_FOREIGN]) {
      printf("foreign slot type counts:\n");
      for (i = 0; i < ASIZE(global_foreign_counts); i++)
	if (global_foreign_counts[i])
	  printf("  %d: %d\n", i, global_foreign_counts[i]);
    }
  }
}

int
dumpobj(name, bytes)
     char *name;
     char *bytes;
{
  int v;
  List_Links *dl;

  if (global_options & OBJ_NAME_PATTERN) {
    re_comp(global_obj_pattern);
    if (!re_exec(name))
      return 0;
  }
  v = (int)(*bytes++);
  if (global_options & SHOW_ONLY_VERSION) {
    if (v != global_version)
      return 0;
  }
  LIST_FORALL(dumpers, dl) {
    object_dumper *d = (object_dumper *)dl;

    if (d->version == v) {
      if (!(global_options & ONLY_STATS))
	printf("%s     %sV%d%s", name, LL(" (","\t"), v, LL(") ", "\t"));
      (*d->dump_obj_fn)(bytes);
      return 1;
    }
  }
  if (!(global_options & QUIET))
    printf("WARNING: object %s is a version %d object -- cannot dump.\n",
	   name, v);
  return -1;
}

void
scanobjs(filename)
     char *filename;
{
  GDBM_FILE dbf, open_object_file();
  datum k, d;
  void fatal(), init_stats();
  char file[300];
  struct stat info;

  sprintf(file, "%s/%s", filename, global_storage_file_name);
  if (stat(file, &info)) {
    perror(file);
    return;
  }
  dbf = gdbm_open(file, global_block_size, GDBM_READER, global_mode, fatal);
  if (dbf == (GDBM_FILE)0) {
    fprintf(stderr, "%s: could not open GDBM database\n", filename);
    return;
  }
  k = gdbm_firstkey(dbf);
  while (k.dptr != NULL) {
    d = gdbm_fetch(dbf, k);
    if (d.dptr == NULL) {
      printf("%s key has no data\n", k.dptr);
      continue;
    }
    if (scanobj(k.dptr, d.dptr))
      global_nobjs++;
    k = gdbm_nextkey(dbf, k);
  }
  gdbm_close(dbf);
}

int
scanobj(name, bytes)
     char *name;
     char *bytes;
{
  int v;
  List_Links *dl;

  if (global_options & OBJ_NAME_PATTERN) {
    re_comp(global_obj_pattern);
    if (!re_exec(name))
      return 0;
  }
  v = (int)(*bytes++);
  if (global_options & SHOW_ONLY_VERSION) {
    if (v != global_version)
      return 0;
  }
  LIST_FORALL(dumpers, dl) {
    object_dumper *d = (object_dumper *)dl;

    if (d->version == v) {
      (*d->scan_obj_fn)(name, bytes);
      return 0;
    }
  }
  printf("%s version %d not recognized\n", name, v);
}

void initialize()
{
  void init_dumpers(), init_stats();

  init_dumpers();
  init_stats();
}

void
init_dumpers()
{
  void dump_v1(), scan_v1(), add_dumper();

  dumpers = (List_Links *)malloc(sizeof(List_Links));
  List_Init(dumpers);
  add_dumper(1, dump_v1, scan_v1);
}

void
init_stats()
{
  int i;

  global_nbytes = 0;
  global_nslots = 0;
  global_nEslots = 0;
  for (i = 0; i < ASIZE(global_slot_counts); i++)
    global_slot_counts[i] = 0;
  for (i = 0; i < ASIZE(global_foreign_counts); i++)
    global_foreign_counts[i] = 0;
  global_nobjs = 0;
}

void
add_dumper(version, dump_obj_fn, scan_obj_fn)
     int version;
     void (*dump_obj_fn)();
     void (*scan_obj_fn)();
{
  object_dumper *d;

  d = (object_dumper *)malloc(sizeof(object_dumper));
  List_InitElement((List_Links *)d);
  d->version = version;
  d->dump_obj_fn = dump_obj_fn;
  d->scan_obj_fn = scan_obj_fn;
  List_Insert((List_Links *)d, LIST_ATREAR(dumpers));
}

/*
 * dumpers
 */

#define GETINT4(i,b,bi) memcpy(&i,&b[bi],sizeof(i));i=ntohl(i);bi+=sizeof(i)
#define GETINT2(i,b,bi) memcpy(&i,&b[bi],sizeof(i));i=ntohs(i);bi+=sizeof(i)
#define GETSTRP(s,b,bi) s=b+bi;bi+=strlen(s)+1

void
dump_v1(bytes)			/* dump a version 1 object */
     char *bytes;
{
  int idx, nslots, nEslots;

  idx = nslots = nEslots = 0;
  GETINT4(nslots, bytes, idx);
  if (!(global_options & ONLY_STATS))
    printf("%d%s", nslots, LL(" slots:\n", "\t"));
  if (global_options & LONG_LISTING) {
    printf("  %-20s\t%-12s\t%-12s", "NAME", "TYPE", "PRI");
    if (global_options & VALUE_SAME_LINE)
      printf(" VALUE\n");
  }
  while (nslots--) {
    char *name, *value, *slotTypeString(), *slotPriString();
    Bos_Slot_Type type, plain_type;
    Bos_Slot_Pri pri;
    int evanescent;

    GETSTRP(name, bytes, idx);
    GETINT2(type, bytes, idx);
    GETINT2(pri, bytes, idx);
    plain_type = Bos_PlainSlotType(type);
    evanescent = (type & Bos_SLOT_EVANESCENT_MASK);
    GETSTRP(value, bytes, idx);
    if (evanescent && !(global_options & SHOW_EVANESCENT))
      continue;
    if (global_options & LONG_LISTING) {
      if (global_options & OBJ_SLOT_PATTERN) {
	re_comp(global_slot_pattern);
	if (!re_exec(name))
	  continue;
      }
      printf("  %-20s\t%-12s\t%-12s", name, slotTypeString(type),
	     slotPriString(pri, type));
      if (global_options & NO_VALUE)
	printf("\n");
      else {
	if (global_options & VALUE_SAME_LINE) {
	  if (global_options & DELIMIT_VALUE)
	    printf(" |%s|\n", value);
	  else
	    printf(" %s\n", value);
	} else {
	  if (global_options & DELIMIT_VALUE)
	    printf("\n    |%s|\n", value);
	  else
	    printf("\n    %s\n", value);
	}
      }    
    }
    global_nslots++;
    if (evanescent) {
      global_nEslots++;
      nEslots++;
    }
    global_slot_counts[plain_type]++;
    if (plain_type == Bos_SLOT_FOREIGN)
      global_foreign_counts[pri]++;
  }
  if (!(global_options & ONLY_STATS)){
    if (!(global_options & LONG_LISTING)) {
      if (global_options & SHOW_EVANESCENT)
	printf("%d\t%d\n", nEslots, idx);
      else
	printf("%d\n", idx);
    }    
  }
  global_nbytes += idx;
}

#define BAD_NAME	0x0001
#define BAD_NSLOTS	0x0002

#define BAD_TYPE	0x0008
#define BAD_PRI		0x0010
#define BAD_VALUE	0x0020

void
scan_v1(name, bytes)
     char *name;
     char *bytes;
{
  int idx, nslots, obj_nasties;

  idx = nslots = obj_nasties = 0;
  if (!printable(name)) {
    obj_nasties |= BAD_NAME;
    printf("garbaged object %s:\n  unprintable name\n", name);
  }
  GETINT4(nslots, bytes, idx);
  if (nslots < 0 || nslots > global_suspicious_nslots) {
    obj_nasties |= BAD_NSLOTS;
    if (!(obj_nasties & BAD_NAME))
      printf("garbaged object %s:\n  bad nslots %d\n", name, nslots);
  } else
    while (nslots--) {
      int nasties, evanescent;
      char *sname, *value;
      Bos_Slot_Type type, plain_type;
      Bos_Slot_Pri pri;
      
      nasties = 0;
      GETSTRP(sname, bytes, idx);
      GETINT2(type, bytes, idx);
      GETINT2(pri, bytes, idx);
      GETSTRP(value, bytes, idx);
      plain_type = Bos_PlainSlotType(type);
      evanescent = (type & Bos_SLOT_EVANESCENT_MASK);
      
      if (!printable(sname))
	nasties |= BAD_NAME;
      switch (plain_type) {
      case Bos_SLOT_FOREIGN:
      case Bos_SLOT_CMETHOD:
      case Bos_SLOT_NORMAL:
      case Bos_SLOT_OBJECT:
      case Bos_SLOT_REFERENCE:
      case Bos_SLOT_METHOD:
	break;
      default:
	nasties |= BAD_TYPE;
	break;
      }
      if (plain_type == Bos_SLOT_FOREIGN)
	if (pri < 0 || pri > global_suspicious_foreign_subtype)
	  nasties |= BAD_PRI;
	else if (pri != Bos_PRI_HIGHEST &&
		 pri != Bos_PRI_LOWEST &&
		 (pri < 0 || pri > global_suspicious_pri))
	  nasties |= BAD_PRI;
      if (!printable(value))
	nasties |= BAD_VALUE;
      if (nasties) {
	if (!(obj_nasties & BAD_NAME)) {
	  printf("garbaged object %s:\n", name);
	  obj_nasties |= BAD_NAME; /* XXX cheating */
	}
	printf("  slot %s:", sname);
	if (nasties & BAD_NAME)
	  printf(" unprintable name");
	if (nasties & BAD_TYPE)
	  printf(" bad type %u%s", plain_type, evanescent? "E": "");
	if (nasties & BAD_PRI)
	  printf(" bad pri %d", pri);
	if (nasties & BAD_VALUE)
	  printf(" unprintable value |%s|", value);
	printf("\n");
      }
    }
}

/*
 * utilities
 */

int
printable(str)
     char *str;
{
  char c;

  while ((c = *str++) != '\0')
    if (!isprint(c))
      return 0;
  return 1;
}

char *
slotTypeString(raw_type)
     Bos_Slot_Type raw_type;
{
  static char ret[30];
  Bos_Slot_Type type;

  ret[0] = '\0';
  if (raw_type & Bos_SLOT_EVANESCENT_MASK) {
    strcat(ret, ".");
  }
  type = Bos_PlainSlotType(raw_type);
  switch (type) {
  case Bos_SLOT_NORMAL:
    strcat(ret, "normal");
    break;
  case Bos_SLOT_METHOD:
    strcat(ret, "method");
    break;
  case Bos_SLOT_CMETHOD:
    strcat(ret, "cmethod");
    break;
  case Bos_SLOT_OBJECT:
    strcat(ret, "object");
    break;
  case Bos_SLOT_REFERENCE:
    strcat(ret, "reference");
    break;
  case Bos_SLOT_FOREIGN:
    strcat(ret, "foreign");
    break;
  default:
    sprintf(ret, "UNKNOWN %u", type);
    break;
  }
  return ret;
}

char *
slotPriString(pri, type)
     Bos_Slot_Pri pri;
     Bos_Slot_Type type;
{
  static char ret[100];

  ret[0] = '\0';
  type = Bos_PlainSlotType(type);
  if (type == Bos_SLOT_FOREIGN) {
    char *st;

    sprintf(ret, "subtype %u", pri);
  } else {
    strcpy(ret, "pri ");
    if (pri == Bos_PRI_HIGHEST)
      strcat(ret, "highest");
    else if (pri == Bos_PRI_LOWEST)
      strcat(ret, "lowest");
    else
      sprintf(ret, "%u", pri);
  }
  return ret;
}

void
fatal()
{
  extern gdbm_error gdbm_errno;
  char *_gdbmErrMsg();

  fprintf(stderr, "major problem in GDBM (%d: %s)\n",
	  gdbm_errno, _gdbmErrMsg(gdbm_errno));
  abort();
}

char *
_gdbmErrMsg(code)
     gdbm_error code;
{
  static struct {gdbm_error code; char *text;} _msgs[] = {
    { GDBM_NO_ERROR, "No error" },
    { GDBM_MALLOC_ERROR, "Malloc error" },
    { GDBM_BLOCK_SIZE_ERROR, "Block size error" },
    { GDBM_FILE_OPEN_ERROR, "File open error" },
    { GDBM_FILE_WRITE_ERROR, "File write error" },
    { GDBM_FILE_SEEK_ERROR, "File seek error" },
    { GDBM_FILE_READ_ERROR, "File read error" },
    { GDBM_BAD_MAGIC_NUMBER, "Bad magic number" },
    { GDBM_EMPTY_DATABASE, "Empty database" },
    { GDBM_CANT_BE_READER, "Can't be reader" },
    { GDBM_CANT_BE_WRITER, "Can't be writer" },
    { GDBM_READER_CANT_DELETE, "Reader can't delete" },
    { GDBM_READER_CANT_STORE, "Reader can't store" },
    { GDBM_READER_CANT_REORGANIZE, "Reader can't reorganize" },
    { GDBM_UNKNOWN_UPDATE, "Unknown update" },
    { GDBM_ITEM_NOT_FOUND, "Item not found" },
    { GDBM_REORGANIZE_FAILED, "Reorganize failed" },
    { GDBM_CANNOT_REPLACE, "Cannot replace" },
    { GDBM_NO_ERROR, (char *)0 }
  };
  static char _unknMsg[200];
  int i;

  for (i = 0; _msgs[i].text != (char *)0; i++)
    if (_msgs[i].code == code)
      return _msgs[i].text;
  sprintf(_unknMsg, "Unknown GDBM error code: %d", code);
  return _unknMsg;
}
#ifdef NEED_PANIC
void
panic(str)
  char *str;
{
  fprintf(stderr, "PANIC: %s\n", str);
  fflush(stderr);
  abort();
}
#endif NEED_PANIC
