/* 
 * tkFocus.c (CTk) --
 *
 *	This file contains procedures that manage the input
 *	focus for Tk.
 *
 * Copyright (c) 1990-1994 The Regents of the University of California.
 * Copyright (c) 1994-1995 Sun Microsystems, Inc.
 * Copyright (c) 1994-1995 Cleveland Clinic Foundation
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

static char rcsid[] = "@(#) $Header: /usrs/cvs/root/ctk/tkFocus.c,v 1.1.1.1 1995/03/12 16:27:55 andrewm Exp $";

#include "tkInt.h"
#include "tkPort.h"


/*
 *--------------------------------------------------------------
 *
 * Tk_FocusCmd --
 *
 *	This procedure is invoked to process the "focus" Tcl command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

int
Tk_FocusCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Main window associated with
				 * interpreter. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    Tk_Window tkwin = (Tk_Window) clientData;
    TkWindow *winPtr = (TkWindow *) clientData;
    TkWindow *newPtr, *focusWinPtr;
    char c;
    size_t length;

    /*
     * If invoked with no arguments, just return the current focus window.
     */

    if (argc == 1) {
	focusWinPtr = TkGetFocus(winPtr);
	if (focusWinPtr != NULL) {
	    interp->result = focusWinPtr->pathName;
	}
	return TCL_OK;
    }

    /*
     * If invoked with a single argument beginning with "." then focus
     * on that window.
     */

    if ((argc == 2) && (argv[1][0] == '.')) {
	newPtr = (TkWindow *) Tk_NameToWindow(interp, argv[1], tkwin);
	if (newPtr == NULL) {
	    return TCL_ERROR;
	}
	CtkSetFocus(newPtr);
	return TCL_OK;
    }

    length = strlen(argv[1]);
    c = argv[1][1];
    if ((c == 'd') && (strncmp(argv[1], "-displayof", length) == 0)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " -displayof window\"", (char *) NULL);
	    return TCL_ERROR;
	}
	newPtr = (TkWindow *) Tk_NameToWindow(interp, argv[2], tkwin);
	if (newPtr == NULL) {
	    return TCL_ERROR;
	}
	newPtr = TkGetFocus(newPtr);
	if (newPtr != NULL) {
	    interp->result = newPtr->pathName;
	}
    } else if ((c == 'l') && (strncmp(argv[1], "-lastfor", length) == 0)) {
    	return Ctk_Unsupported(interp, "focus -lastfor");
    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
		"\": must be -displayof or -lastfor", (char *) NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * CtkSetFocus --
 *
 *	This procedure is invoked to change the focus window for a
 *	given display in a given application.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Event handlers may be invoked to process the change of
 *	focus.
 *
 *----------------------------------------------------------------------
 */

void
CtkSetFocus(winPtr)
    TkWindow *winPtr;
{
    TkWindow *focusPtr = winPtr->dispPtr->focusPtr;
    Ctk_Event event;

    if (winPtr == (TkWindow *)NULL || (winPtr->flags & TK_ALREADY_DEAD)) {
    	panic("Attempt to set focus to null/dead window");
    }
    if (winPtr != focusPtr) {
        if (focusPtr && !(focusPtr->flags & TK_ALREADY_DEAD)) {
	    event.type = CTK_UNFOCUS_EVENT;
	    event.window = focusPtr;
	    Tk_HandleEvent(&event);
	}
	winPtr->dispPtr->focusPtr = winPtr;
	event.type = CTK_FOCUS_EVENT;
	event.window = winPtr;
	Tk_HandleEvent(&event);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TkGetFocus --
 *
 *	Given a window, this procedure returns the current focus
 *	window for its application and display.
 *
 * Results:
 *	The return value is a pointer to the window that currently
 *	has the input focus for the specified application and
 *	display, or NULL if none.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

TkWindow *
TkGetFocus(winPtr)
    TkWindow *winPtr;
{
    return winPtr->dispPtr->focusPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * TkFocusDeadWindow --
 *
 *	This procedure is invoked when it is determined that
 *	a window is dead.  It cleans up focus-related information
 *	about the window.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The input focus for the window's display may change.
 *
 *----------------------------------------------------------------------
 */

void
TkFocusDeadWindow(winPtr)
    TkWindow *winPtr;
{
    if (winPtr == winPtr->dispPtr->focusPtr) {
    	/*
    	 * This window has the focus, try to pass focus first to
    	 * top-level then main top-level.  If both of these are
    	 * dead then give up - the application will have exited
    	 * before any more key events will be processed).
    	 */
    	TkWindow *newFocusPtr = Ctk_TopLevel(winPtr);

	if (newFocusPtr->flags & TK_ALREADY_DEAD) {
	    newFocusPtr = Ctk_BottomChild(winPtr->dispPtr->rootPtr);
	    if (newFocusPtr && !(newFocusPtr->flags & TK_ALREADY_DEAD)) {
		CtkSetFocus(newFocusPtr);
	    }
	} else {
	    CtkSetFocus(newFocusPtr);
	}
    }
}
