
# $Id: cmdwinC.tcl,v 1.3 1993/06/08 06:09:35 david Exp $
#
# Copyright 1993 David Herron.
# Permission to use, copy, modify and distribute this software
# and its documentation for any purpose and without fee is
# hereby granted, provided that the above copyright notice
# appear in all copies.  David Herron makes no representations
# about the suitability of this software for any purpose save
# printing it out and using the paper as bird cage lining.
#
# cmdwinC - Class for a TK `command window' which executes TCL commands
#	and nicely displays results.
#
# $Log: cmdwinC.tcl,v $
# Revision 1.3  1993/06/08  06:09:35  david
# Add copyright notice.
#
# Revision 1.2  1993/06/06  06:30:56  david
# Change name to current naming convention (*C.tcl).  Change to new
# format required by require/auto_index_modules/Define_Module.
#
# Revision 1.1.1.1  1993/06/03  06:23:46  david
# Initial revision of `minterp' as its own package.
#
# Revision 1.4  1993/06/01  05:06:22  david
# Update to the newest style of interp.c.
#
# Revision 1.3  1993/04/14  07:20:30  david
# Changes required by new(er) version of interp.  It doesn't work,
# unfortunately.  (Claims that `-exec' is not executable in MainInterp)
#
# Revision 1.2  1993/02/09  07:17:42  david
# Protect against exit.
#
# Revision 1.1  1993/02/09  06:16:38  david
# New command window code.
#
#
# METHODS:
#
# show
#
#	Either creates new command window or unhides & raises
#	an existing one.
#
# execute
#
#	Pulls command from input box, executes it in MainInterp,
#	and plops result strings into result & transcript boxes.
#
# devisualize
#
#	Close down window.
#
# hide
#
#	Withdraw window from screen.
#
# recall
#
#	Pull command from selected text in transcript and insert
#	into input box.  (Necessary?  Will normal cut/paste work?)
#
# clear
#
#	Clear all stuff on screen.

Define_Module CommandWindow {

InterpBase -import CommandWindow InterpBase [InterpBase -get EXPORTS]

# We don't want this interp to go away at all.
rename exit {}
proc exit {} { clear; hide }

Method show {} {
	global RUNNING
	if ![info exists RUNNING] { visualize }
	if [catch {wm deiconify .commandWindow}] {
		catch {unset RUNNING}
		visualize
	}
#	wm raise .commandWindow
}

Method visualize {} {
	global RUNNING
	if [info exists RUNNING] return

	set RUNNING "yes"

	toplevel .commandWindow
	wm iconname .commandWindow "TCL"
	wm title    .commandWindow "Command Window"

	-chain MainInterp .commandWindow

	frame .commandWindow.tr  -borderwidth 1 -relief sunken
	frame .commandWindow.res -borderwidth 1 -relief sunken
	frame .commandWindow.entry
	frame .commandWindow.cmd
	pack append .commandWindow \
		.commandWindow.tr    { top fill expand } \
		.commandWindow.res   { top fill expand } \
		.commandWindow.entry { top fillx } \
		.commandWindow.cmd   { top fillx }

	-chain MainInterp {
		.commandWindow.tr
		.commandWindow.res
		.commandWindow.entry
		.commandWindow.cmd
	}

	text      .commandWindow.tr.t \
			-yscroll { .commandWindow.tr.s set }
	scrollbar .commandWindow.tr.s \
			-command { .commandWindow.tr.t yview } \
			-relief sunken
	pack append .commandWindow.tr \
		.commandWindow.tr.t	{ left fill expand } \
		.commandWindow.tr.s	{ left filly }
	-chain MainInterp {
		.commandWindow.tr.t
		.commandWindow.tr.s
	}

	text      .commandWindow.res.t \
			-yscroll { .commandWindow.res.s set }
	scrollbar .commandWindow.res.s \
			-command { .commandWindow.res.t yview } \
			-relief sunken
	pack append .commandWindow.res \
		.commandWindow.res.t	{ left fill expand } \
		.commandWindow.res.s	{ left filly }
	-chain MainInterp {
		.commandWindow.res.t
		.commandWindow.res.s
	}

	label .commandWindow.entry.l \
			-text "Command"
	entry .commandWindow.entry.e \
			-relief sunken
	pack append .commandWindow.entry \
		.commandWindow.entry.l { left fillx } \
		.commandWindow.entry.e { left fillx expand }

	-chain MainInterp {
		.commandWindow.entry.l
		.commandWindow.entry.e
	}

	bind .commandWindow.entry.e <Return> {CommandWindow execute}

	button .commandWindow.cmd.close \
			-text "Close" \
			-command { CommandWindow devisualize }
	button .commandWindow.cmd.execute \
			-text "Execute" \
			-command { CommandWindow execute }
	button .commandWindow.cmd.hide \
			-text "Hide" \
			-command { CommandWindow hide }
	button .commandWindow.cmd.clear \
			-text "Clear" \
			-command { CommandWindow clear }
	pack append .commandWindow.cmd \
		.commandWindow.cmd.close   { left fillx expand } \
		.commandWindow.cmd.execute { left fillx expand } \
		.commandWindow.cmd.hide    { left fillx expand } \
		.commandWindow.cmd.clear   { left fillx expand }

	-chain MainInterp {
		.commandWindow.cmd.close
		.commandWindow.cmd.execute
		.commandWindow.cmd.hide
		.commandWindow.cmd.clear
	}

	.commandWindow.tr.t tag configure COMMAND -font times-bold
	.commandWindow.tr.t tag configure ERROR   -font times-italic
	.commandWindow.tr.t tag configure NORMAL  -font times-roman

	 wm minsize .commandWindow 1 1
}

#
# execute{} - Execute the command in the input box.
#	Carefully use bold for commands, italic for
#	errors and roman for everything else.
#
Method execute {} {
	global errorInfo

	set cmd [.commandWindow.entry.e get]
	if {$cmd == ""} { beep; return }
	.commandWindow.res.t delete 1.0 end
	.commandWindow.tr.t yview -pickplace end

	set cmd_start [.commandWindow.tr.t index end]
	.commandWindow.tr.t insert end $cmd
	set cmd_end [.commandWindow.tr.t index end]
	.commandWindow.tr.t insert end "\n"

	set err [catch {MainInterp -exec $cmd} val]

	.commandWindow.res.t insert 1.0 $val

	if $err {
		set tr_start [.commandWindow.tr.t index end]
		.commandWindow.tr.t insert end $errorInfo
		set tr_end [.commandWindow.tr.t index end]
		.commandWindow.tr.t insert end "\n"
		set tr_type ERROR
	} else {
		set tr_start [.commandWindow.tr.t index end]
		.commandWindow.tr.t insert end $val
		set tr_end [.commandWindow.tr.t index end]
		.commandWindow.tr.t insert end "\n"
		set tr_type NORMAL
	}

	.commandWindow.tr.t tag add $tr_type $tr_start  $tr_end
	.commandWindow.tr.t tag add COMMAND  $cmd_start $cmd_end
}

Method devisualize {} {
	destroy .commandWindow
	global RUNNING; unset RUNNING
}

Method hide {} {
	wm withdraw .commandWindow
}


Method clear {} {
	.commandWindow.res.t delete 1.0 end
	.commandWindow.tr.t delete 1.0 end
}

Method sels {} {
	foreach tag [.commandWindow.tr.t tag names] {
		puts stdout "$tag: [.commandWindow.tr.t tag ranges $tag]"
	}
}

-chain MainInterp [InterpBase -get TkChain]

}

