'\" 
'\" Copyright 1993 David Herron.
'\" Permission to use, copy, modify and distribute this software
'\" and its documentation for any purpose and without fee is
'\" hereby granted, provided that the above copyright notice
'\" appear in all copies.  David Herron makes no representations
'\" about the suitability of this software for any purpose save
'\" printing it out and using the paper as bird cage lining.
'\" 
'\" $Id: interp.ms,v 1.3 1993/06/08 06:13:59 david Exp $
'\" 
'\" 
'\" NOTE: Format this with groff!
'\" 
'\" 
.ds interp \fIminterp\fP
.ds tcl TCL
.TL
Multiple Interpreters in TCL
.br
and
.br
Object Oriented Extensions to TCL
.AU
David S. Herron
.br
david@davids.mmdf.com (home)
.DA
.AB
.LP
In one of the Usenix papers delivered by Dr. Ousterhout is a
statement to the effect that \*QInterpreters are light weight
things and programmers should feel free to make as many
as s/he likes\*U.  Unfortunately there was no facilities in the
TCL language for manipulating interpreters.  Instead the only
tools to manipulate them are C functions in the library.

This module, called \*[interp], provides access to multiple
interpreters from \*[tcl].  With it \*[tcl] programmers can create and
delete interpreters, and can execute commands or access variables in
other interpreters.

In addition facilities are provided for doing Object Oriented
Programming in \*[tcl].  Guidance in designing the facilities were
taken from \fIObject Oriented Design \s-2with applications\s+2\fP by
Grady Booch.
.AE
.SH
\s+3INTRODUCTION\s-3
.LP
As stated the purpose of \*[interp] is to give TCL programmers
the ability to create and manipulate interpreters.

Creating an interpreter is easy, simply type:
.DS
set var [interp new \fIinterp-name\fP]
.DE
Which creates a new interpreter which can be referred to either as
`interp-name' or `$var' (whichever strikes your fancy; `interp new'
returns the name of the new interpreter).  There is a list of
interpreters kept within \*[interp].  Each is named and this name is
used to refer to each interpreter.  The name of each interpreter is
registered as a command in all the interpreters and is used to send
instructions to the interpreter for execution.

Interpreters contain both commands and variables.  Each interpreter
created has its own namespace for both.  The only shared namespace
anywhere is the list of interpreter names.
.ce 1
\fBNOTE\fP
.QP
In this and the following examples, the \fB%\fP character will be used
for the TCL prompt.  The part typed by the user will be typed in
\fBbold\fP.
.LP
.ce 1
\fBExample 1\fP
.DS
% \fBinterp new i\fP
% \fBi -exec { set pwd [pwd]; set dl [split $pwd "/"] }\fP
{} home david heraldry minterp
% \fBset pwd\fP
Error: can't read "pwd": no such variable
% \fBset dl\fP
Error: can't read "dl": no such variable
% \fBi -exec { set pwd }\fP
/home/david/heraldry/minterp
% \fBi -exec { set dl }\fP
{} home david heraldry minterp
% \fBset p [i -exec { return $pwd } ]\fP
/home/david/heraldry/minterp
% \fBset p\fP
/home/david/heraldry/minterp
% \fBset pwd\fP
Error: can't read "pwd": no such variable
% 
.DE
In this example we see one method for executing commands in another
interpreter (\fB-exec\fP).  We also see that variables are local to
one interpreter, and a couple ways to retrieve values of variables in
another interpreter.

.ce 1
\fBExample 2\fP
.DS
% \fBi -exec {     
\ \ \ \ proc a {} {
\ \ \ \ \ \ \ \ puts stdout "Hello, world!"
\ \ \ \ }
}\fP
% \fBa\fP
Error: ambiguous command name "a": append array auto_execok auto_index_modules auto_load auto_mkindex auto_reset
% \fBi -exec a\fP
Hello, world!
% \fBlsearch [info commands] a\fP
-1
% \fBi -exec { lsearch [info commands] a }\fP
41
% \fBi a\fP
Error: ERROR: a is not a method of interpretor i
% \fBi -exec { -AddMethod a }\fP
% \fBi -exec { -Methods }\fP
a -Methods -exec -IsMethod
% \fBi -Methods\fP
a -Methods -exec -IsMethod
% \fBi a\fP
Hello, world!
% 
.DE

In this example we see that procedures are also local to interpreters.
We also see how to define \fImethods\fP and how they can be called
differently from other commands.

.ce 1
\fBExample 3\fP
.DS
% i -exec { -AllowExec false }
0
% i -exec a
Error: ERROR: Not currently allowed to execute commands (with -exec) in interp i
% i a
Hello, world!
% 
.DE

Here we see how the internals of an interpreter can be protected from
outside influence.  By executing \fB-AllowExec false\fP the only
access into the interpreter is by way of any \fImethods\fP which have
been defined.

A \fImethod\fP is, then, a command (or proc) which can be executed
directly without having to use \fB-exec\fP.  This trivially allows for
all the advantages of modularization to come into play.  If a bug is
found related to a module, since access to the modules internals is
strictly limited the the offending code is likely right there in the
module itself.

There are two forms for the interpreter's command.  One, \fIinterp
-exec command\fP, has already been demonstrated.  The other, \fIinterp
method args\fP, is the basis for many of the object oriented features
discussed below.

Some commands are created in the new interpreter.  We have already
seen \fI-AllowExec\fP and its effects.  One of particular importance
is \fI-exit\fP which simply destroys the interpreter.  If one does not
want the normal \fIexit\fP command to kill the process, then
\fI-exit\fP should be renamed to be \fIexit\fP.
.ce 1
\fBExample 4\fP
.DS
% \fBinterp MainInterp\fP
% \fBinterp new k\fP
% \fBk -exec {
        -destroyHook { puts stdout "AAAARRRGGGHHH!!!" }
}\fP
% \fBk -exec { -exit }\fP
AAAARRRGGGHHH!!!
% \fBk -exec "puts stdout \e"Hello, world!\e""\fP
invalid command name "k"
% \fBecho $errorInfo\fP
invalid command name "k"
    while executing
"error "invalid command name \"$name\"""
    (procedure "unknown" line 53)
    invoked from within
"k -exec "puts stdout \"Hello, world!\"""
%  
.DE

The \*[interp] facilities are described in more detail on the manual page.

.SH
\s+3Elements of Style of Module Creation\s-3
.LP

\fBTO BE WRITTEN.\fP

How to go about designing a module is outside the scope.

How to create modules.  Refer to \fBDefine_Module\fP.

When to use a simple module, and when to use \fIobject orientation\fP.

Making functions appear as if they're local, but actually defined
elsewhere.  Inheritance.

When to protect the internals, when not.

.SH
\s+3What IS this object oriented stuff?\s-3
.LP
Much ado has been made about \fIObject Oriented Programming\fP
recently so there are many books on the subject.  The one I have been
studying is \fIObject Oriented Design \s-2with applications\s+2\fP by
Grady Booch.  It appears to be very good, giving a very in depth
discussion of what it is and how to use it.  Early in the book is
listed the conceptual framework necessary for an object oriented
system.  These are: \fBAbstraction\fP, \fBEncapsulation\fP,
\fBModularity\fP, \fBHierarchy\fP, \fBTyping\fP, \fBConcurrency\fP,
and \fBPersistence\fP.  Of these \*[interp] only implements the first
four.

See \fBinterpBaseC.tcl\fP for the implementation.

These terms mean:
.IP Abstraction
Is the essential characteristics of an object which distinguish it
from all others.  Provides crisply defined conceptual boundries.
Primarily the actions the object performs rather than the data it
holds.

In \*[interp] abstraction is done by defining methods (commands).
These are commands whose names are in the \fImethods list\fP, the list
is manipulated using \fB-AddMethod\fP and \fB-DelMethod\fP.

A popular abstraction-like ability is function/operator overloading.
Since it depends on strong data typing, it is not available in \*[tcl]
nor \*[interp].
.IP Encapsulation
This is also known as Information Hiding.  That is, holding details of
implementation close to the chest so that other bits of software don't
depend on internal details.  Encapsulation is complementary to
abstraction.  Abstraction is the view from outside the object while
encapsulation is from the inside.  Things to be hidden are those which
don't contribute to its essential characteristics.

In \*[interp], a command is hidden by not listing it in the methods
list, and also disallowing \fI-exec\fP.  Once that much is done, only
the methods can be executed through the interpreter's command.

All variables are hidden since \*[interp] provides no direct
facilities to manipulate them.  But it is pretty trivial to write the
necessary TCL code.  Consult the \fB-get\fP, \fB-set\fP, \fB-unset\fP
and \fB-export\fP methods.
.IP Modularity
This is the unit of \fIphysical decomposition\fP.  That is, what
pieces you break the problem down to.

In \*[interp] the unit of modularity is the interpreter.  Place everything
you wish to put into a module into the same interpreter.

.IP Hierarchy
This builds on existing structure to create new things.  There
are two sorts of inheritance, and both require a \fIclass\fP
system to be present.  These are:

.TS
expand allbox;
l l.
Sort of relationship	Sort of hierarchy
T{
\fIkind of\fP == \fIclass structure\fP
T}	T{
\fBinheritance\fP - build on existing object
T}
T{
\fIpart of\fP == \fIobject structure\fP
T}	T{
\fBaggregation\fP - build from many objects
T}
.TE

With \*[interp] one can create either sort of hierarchy.

\fBInheritance:\fP The \fI-execHere\fP pulls command defintions from
other interpreters and executes them within the local interpreter.
Thus there are two interpreters, \fBcaller\fP and \fBtarget\fP.  In
caller you execute
.DS
\fB-execHere\fP \fItarget command-name args\fP
.DE
When it executes, it is done within caller's context (interpreter), but the
actual function is retrieved from target's context (interpreter).  This is
used in \fB-import\fP to have one defining instance of a function,
and being used by many interpreters.

The intent is to create an interpreter which holds the definition
for objects of a particular class.  To create new instantiations
of the class, a command \fInew\fP should be created.  It creates
a new interpreter and then uses \fB-import\fP to link the necessary
procedures into the new interpreter.

\fBAggregation:\fP  This is when you make an object which contains
other objects.  For instance, say you want a file browser and
already have listboxes encapsulated into an object.  File browsers
generally have at least two listboxes so the file browser object
would have a hierarchy like:

.DS
FileBrowser
\ \ \ \ Listbox
\ \ \ \ Listbox
.DE

In \*[interp] part of creating the FileBrowser instance would create
two Listbox instances as well.  Currently there isn't a way to hide
the Listbox instances within the FileBrowser instance, nor is there a
way to record hierarchical information.  It is possible to create
these facilities, but will require changes in order to hide child
interpreters so that only their parent or siblings can access them.
.IP "Typing   "
Not necessarily the same concept as \fIclass\fP.  Defines what kinds
of objects can be interchanged, and how it is done.

Since TCL has no concept of data typing in the first place this
doesn't seem to have a place here.  Further, the objects one would
create with \*[interp] are likely to be \fIlarge\fP, larger than
one would normally do assignments with (for instance).
.IP Concurrency
Multi threading ability.  Not entirely relavent to object oriented
programming, but multi threading can make OOPing easier to accomplish.
This feature distinguishes active objects from inactive ones.

Neither \*[interp] nor TCL have any support for this.
.IP Persistence
Live long and persist.  Persistent objects survive the process which
created & held them exits.  But \*[interp] does not provide any
facilities for persistent objects.  The object itself may have
persistence, and that is up to the objects implementor.

.SH
\s+3Creating classes and objects\s-3
.LP
One of the purposes of \*[interp] is to cover some facets of OOP.  See
the discussion above for more detail.  This section explains how this
is done.

The first step in creating an object class is analysis of what the
object is to do.  There are many fine books on this subject, Grady
Booch's book among them.  Therefore I will not attempt to show you the
analysis and design steps.  Once you have determined the set of
methods the object is to have, you are ready to lay down code.

What you will have in hand is a set of object class descriptions.
Each class may be related to other classes (\fIsubclass\fP or
\fIsuperclass\fP).  Each class has \fImethods\fP (actions the object
can do) and \fIdata\fP (information held within the object).

The general strategy used is to use one interpreter to store the
definition of the methods.  Then to use another interpreter for each
instantiation of the class.  For each method inherited from another
interpreter a small glue procedure is defined using \fB-execHere\fP to
drag over the command definition for local execution.  The one storing
the definitions is called the \fIbase\fP interpreter while the others
are called \fIinstance\fP interpreters.

This strategy is implemented in \fIinterpBaseC.tcl\fP and is the
basis for the following discussion.  Refer to the interpBase manual
page for details.

Each method is one procedure (or other command) defined in the base
interpreter.  An instance interpreter accesses the commands using
\fB-execHere\fP.  One nice implementation of this is the function
\fB-import\fP in \fIinterpBaseC.tcl\fP.  The core of which is in the
lines:
.DS
\ \ \ \ $dest -exec [list Method $method args "
\ \ \ \ \ \ \ \ eval \e"-execHere $src $method \e$args\e"
\ \ \ \ " ]
.DE
Here \fI$dest\fP is the interpreter into which the new function is
being inserted.  It is being told to do the equivalent of:
.DS
\ \ \ \ Method \fImethod-name\fP {args} {
\ \ \ \ \ \ \ \ eval "-execHere \fIsrc-interp method-name\fP \e$args"
\ \ \ \ }
.DE
Thus when \fImethod-name\fP is executed, \fB-execHere\fP is used
to retrieve the command from \fIsrc-interp\fP.  Then the command
is executed giving the complete set of passed arguments to the
retrieved command procedure.

If the class being defined is simple, there might be only one
instantiation and it isn't necessary to use multiple interpreters.
However the encapsulation and abstraction facilities are still very
useful.

For object classes in which there will be multiple instances, two
methods are necessary.  These are:

\fBnew\fP: This creates a new instantiation of the class.  It will
usually be the only proc which is executed in the \fIbase\fP
interpreter.  It must create a new interpreter for each instantiation
and somehow ensure a unique name.  It also populates the interpreter
with commands.

\fBdelete\fP: This destroys the instantiation of the class.  Obviously
it must free up any resources used in the class before it can exit.
Cleaning up local variables is not necessary but things like TK widgets
and expect I/O streams all must be clean up.

An example \fInew\fP procedure is:
.DS
Method new {} {
\ \ \ \ \ \ global <class>_count
\ \ \ \ \ \ if ![info exists <class>_count] {set <class>_count 0}
\ \ \ \ \ \ incr <class>_count
\ \ \ \ \ \ set name "<class>$<class>_count"
\ \ \ \ \ \ interp new $name

\ \ \ \ \ \ -importAll $name InterpBase
\ \ \ \ \ \ -importAll $name ThisClassBase

\ \ \ \ \ \ $name -destroyHook { .. any cleanup code .. }

\ \ \ \ \ \ return $name
}
.DE
It is not necessary to inherit all the commands in another interpreter.
It is also possible to inherit commands from more than one interpreter,
giving one multiple inheritance.  Note that if two commands of the
same name are inherited, the later one will completely replace
the former one.

.SH
\s+3Useful idioms\s-3
.IP "File name conventions"
A file name convention which might be useful to follow is to name the
source file like so: \fI<className>C.tcl\fP.  The C is to indicate it
is a class definition.  On some systems this will limit the length of
class names, so perhaps it is not a useful convention.
.IP "Interpreter name conventions"
If the name for the \fIbase\fP interpreter ends with \fIClass\fP, then
it's a more obvious what is happening.
.IP "Source code layout convention"
In the file a useful idiom is to surround the class definition with
the following code:

.DS
if ![interp exists <ClassName>Class] {

interp new <ClassName>Class

<ClassName>Class -exec {
...define the class...
}
# END <ClassName>Class {...}
}
# END if ![interp exists <ClassName>Class] {...}
.DE

This lets the source file be source'd more than once.

A variant of this is \fIDefine_Module\fP.  It nicely encapsulates the
above idiom into a simple command, \fBand\fP is what
\fBauto_index_modules\fP looks for while building index files.
.IP "Aid in importing the right commands"
In order to aid in importing the right functions from the right place
it helps to have the module list what it is exporting.  One way is to
have an exported variable in the \fIbase\fP interpreter named
\fBEXPORTS\fP.  This variable is a list of methods defined in
\fBthat\fP interpreter.  Thus it is easier to control which things are
being imported by use of:
.DS
\ \ \ \ InterpBase -import \fIdest-interp\fP \fIsrc-interp\fP \e
\ \ \ \ \ \ \ \ [\fIsrc-interp\fP -get EXPORTS]
.DE
.IP "Method names"
For classes encapsulating one or more TK widgets it may be nice to
have a separate function \fIMakeWidgets\fP which creates the visual
part of the class.  If so the destroyHook probably should be created
there rather than in \fInew\fP.

.SH
\s+3Complete example of creating a simple object class\s-3
.LP
For demonstration purposes the following \fIobject\fP is presented.

\fBDescription\fP: The \fB7Segment\fP object creates, on a TK canvas,
the illusion of an 7-Segment-LED style numerical digit display.  It
must be positionable within the canvas.  Its value must be changeable
and retrievable.

From this you get the need for the following methods:
.TS
expand;
l lw(5.0i).
\fBnew\fP	Create a 7Segment object
\fBsetval\fP	Set the value being displayed
\fBgetval\fP	Get the value being displayed
\fBgeometry\fP	Move it around and change its layout
.TE
An outline of the source follows.  Consult \fB7segC.tcl\fP in the
distribution for the complete implementation.  Commentary is
interspersed with the functions.
.DS
Define_Module 7SegmentClass {
InterpBase -import -import 7SegmentClass InterpBase [InterpBase -get EXPORTS]
-chain MainInterp [InterpBase -get TkChain]

set EXPORTS { new setval getval }
export EXPORTS
.DE
This much creates and initializes the interpreter.  It imports
everything which \fIInterpBase\fP exports, and it sets up the TK
commands so they actually execute in \fIMainInterp\fP.  It also
announces what functions are exported from here.
.DS
Method new args {
\ \ \ \ set name [gen_interp_name "7seg"]
\ \ \ \ interp new $name

\ \ \ \ -import $name InterpBase [InterpBase -get EXPORTS]
\ \ \ \ -import $name 7SegmentClass [7SegmentClass -get EXPORTS]
\ \ \ \ $name -exec [list -chain MainInterp [InterpBase -get TkChain]]

\ \ \ \ foreach arg $args {
\ \ \ \ \ \ \ \ case [lindex $arg 0] {
SCAN THROUGH THE ARGUMENTS
\ \ \ \ \ \ \ \ }
\ \ \ \ }
TO BE FINISHED -- DRAW OUT THE LINES.  CALL `geometry' TO POSITION THEM.
}
.DE
This creates a new instance of \fI7Segment\fP.  It creates a new
interpreter, doing it in a way to gaurantee a unique name.  It links
in some procedures from elsewhere.  Scans through command line
arguments to get interesting values.  Then creates the visual
components of the object (the lines making up the 7-segment
illusion).

To create different digits requires manipulating the foreground colors
of the line segments.  To make a line disappear, change its foreground
to be the same as the background of the canvas.  To make the line
appear, make it be the canvas' foreground color.
.DS
Method setval newval {
	global curval
	set curval $newval
	case $newval {
	0	{
TO BE FINISHED - CHANGE THE FOREGROUND COLORS OF THE SEGMENTS SO IT
		 MAKES THE RIGHT SHAPE.
	}
	...
	}
}

Method getval {} { global curval; return $curval }
.DE

Moving or placing the object is a fairly simple set of calculations
done inside here.
.DS
Method geometry {X Y WIDTH HEIGHT} {
CALCULATE THE LAYOUT OF THE LINES GIVEN THE X/Y/WIDTH/HEIGHT
}
}
.DE

Given the \fB7SegmentClass\fP as defined above, a digital clock can
be implemented as follows:

.DS
require 7SegmentClass
canvas .c
pack append . .c { top fill expand }
wm geometry . =300x100

set h1 [7SegmentClass new {{-canvas .c} {-x 20} {-y 20} {-width 20} {-height 50}}]
set h2 [7SegmentClass new {{-canvas .c} {-x 20} {-y 50} {-width 20} {-height 50}}]
set m1 [7SegmentClass new {{-canvas .c} {-x 20} {-y 110} {-width 20} {-height 50}}]
set m2 [7SegmentClass new {{-canvas .c} {-x 20} {-y 140} {-width 20} {-height 50}}]

proc setdate {} {
\ \ \ \ global h1 h2 m1 m2
\ \ \ \ set d [exec date]
\ \ \ \ set word "\e[A-Za-z\e]+"; set num "\e[0-9\e]"
\ \ \ \ regexp "^$word +$word +$num+ +($num)($num):($num)($num).*" $d match \e
\ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ \ hr1 hr2 mn1 mn2
\ \ \ \ $h1 setval $hr1; $h2 setval $hr2; $m1 setval $mn1; $m2 setval $mn2
}

proc dodate {} { setdate ; after 10000 dodate }
dodate
.DE
(Only needs a blinking `:' to be complete ...)

.SH
\s+3Handling unknown commands\s-3
.LP

The \fIunknown\fP command is used by \*[tcl] to perform autoloading of
needed functions.  It also gives a standardized error for when a
command really does not exist.  One assumption is that it, unknown,
will rarely be called.  However \*[interp] changes some assumptions.

In the traditional way of building a \*[tcl] application there is one
interpreter.  In main() there is code to create the set of commands
needed by this application.  Usually the only time commands are added
is in main(), with the exception of extensions like TK which adds
commands for each widget created.  Generally a lot of assumptions
were built in of there being only one interpreter per process.

In \*[interp] each new interpreter has only the ones provided by
\*[tcl] itself.  It does not, for instance, get any of the TK
commands.  This is because \fIinterp new\fP calls Int_CreateInterp()
and it only creates the standard commands and hasn't the foggiest idea
of any extensions available locally.

.SH
\s+3Use within a TK program\s-3
.LP

Discuss special issues because of TK.

Callbacks happen in MainInterp.  So create widgets there.

Callbacks have to explicitly be told to call into the right interpreter.

.SH
\s+3Incorporation into TCL interpreters\s-3
.LP

All the above is nice theory, but not all TCL is solvable through
scripts.  Some of it works through C programming.  Adding any
extension to TCL requires varying amounts of C coding to link the
extension into the interpreter.  Ideally there would be a way to
dynamically load a library and make calls into the library to link it
into the TCL interpreter.  But life just ain't that way, yet.

.IP "\fBMaking the distribution\fP"
\fIMinterp\fP follows the time honored tradition of \fIedit a makefile\fP,
type \fImake\fP, followed by \fImake install\fP.

Instructions are scattered throughout the Makefile marked with the
string \fICONFIGURE\fP.  Sometimes there are multiple choices for the
same variable.  If one is suitable, uncomment it and comment out the
others.  Otherwise comment out the existing ones and insert one of
your own choosing.

NOTE: It comes out of the tarfile configured for davids.mmdf.com (A
SPARC II clone running SunOS 4.1.2 and using gcc v2.4.3).

Once you have the Makefile type \fImake\fP.  Since there is only one
file to compile (interp.c) this will not take long.

If the compile ran OK then become super user (or whatever user ID is
required to install into DEST_xxx) and type \fImake install\fP.

As of this writing (June 3, 1993) minterp has not been ported
anywhere.  I have tried to anticipate portability problems but may not
have done so properly.  If there are problems then send me mail
(david@davids.mmdf.com).

.IP "\fBModifying wish and other main.c's\fP"

There are two frequently used \fImain.c\fP's which I have seen.  One
is for \fIwish\fP and the other for pure TCL.  For both incorporation
of minterp is approximately the same.  Each has a call to either
\fITcl_CreateInterp()\fP or \fITcl_CreateExtendedInterp()\fP at the
beginning of \fImain()\fP.

Shortly afterword is a section in which calls into extension libraries
are to be inserted.  Into this section one inserts the line
.DS
\ \ \ \ init_interp(interp);
.DE

An example taken from exp_main_tk.c (the copy of wish's main.c modified
to include expect):
.DS
int
main(argc, argv)
\ \ \ \ int argc;                           /* Number of arguments. */
\ \ \ \ char **argv;                        /* Array of argument strings. */
{
\ \ \ \ char *args, *p, *msg;
\ \ \ \ char buf[20];
\ \ \ \ int result;
\ \ \ \ Tk_3DBorder border;
\ \ \ \ extern char *exp_argv0;
 
#ifdef TK_EXTENDED
\ \ \ \ tk_mainInterp = interp = Tcl_CreateExtendedInterp();
#else
\ \ \ \ interp = Tcl_CreateInterp();
#endif
#ifdef TCL_MEM_DEBUG
\ \ \ \ Tcl_InitMemory(interp);
#endif
 
\ \ \ \ init_interp(interp);        /* <--- Added to support minterp */
\ \ \ \ exp_init(interp);
\ \ \ \ exp_argv0 = argv[0];
.DE

The source distribution contains a modification (and improvement) on
this scheme.  And it automatically sets up MainInterp, source's in the
\fImodule\fP functions and \fIInterpBase\fP.

.SH
\s+3Performance impacts\s-3
.LP

\fBTO BE WRITTEN.\fP

Need studies of at least:

CPU time costs in sending commands from one interpreter to another.

Interpreter creation & intialization rate (including inheritance
of procedures from elsewhere).

.SH
\s+3Problems in current implementation and possible future work\s-3
.LP

.IP "\fBConcurrently execting interpreters\fP"

.IP "\fBFriendly classes or objects\fP"
In code which grants access to internals of interpreters, give
some other interpreters more access (maybe not full) than others.

.IP "\fBAccess control lists\fP"
Generalize access permissions between interpreters.

.IP "\fBHooks for \fI-exec\fP"
To customize, at TCL level, what \fI-exec\fP does.  For instance
we might want to allow it from some interpreters, and not others.
The command executed should be told the calling interpreter.
It might indicate through \fIreturn\fP value, or it might have
a way to execute the command line, or .. the sky's the limit.

.IP "\fBTracing command passing between interpreters\fP"
To aid in tracing and debugging a multiple interpreter program it
would be useful to see what commands are flying around.  The hook
would be a command to which is appended the names of the calling
and destination interpreters (and command lines?).

.IP "\fBThere are cases in which minterp doesn't call trace procedures\fP"
Due to strange circumstances \fIInt_ExecCommand()\fP is unable
to call any trace procedures.  This may be a bug.  It isn't clear
how serious a problem this is, however.

.IP "\fBSelf test package needs work\fP"

.IP "\fBCan only `link' to a procedure in another interpreter\fP"
To be \fIcomplete\fP (in some sense of that word) one must be able to
copy command definitions between interpreters.  This requires changes
in the TCL core since the \fIclientData\fP data is an opaque pointer
which we don't know how to duplicate properly.  Instead, when the
command is created a \fIduplicate\fP procedure would be given in
addition to the \fIdelete\fP one.

.IP "\fBLearn how to spell interpreter\fP"

.IP "\fBThe toolbox guys defined a \fItoolchest\fB, is one needed here?\fP"
One important split they made was between the \fItoochest\fP and
\fItoolbox\fP.  The toolchest includes fundamentally functions which
do not fit into a complete module.  The toolbox is made up of many
modules, each of which is a complete set of functionality.

\fBInterpBase\fP is currently filling that role as well as storing the
OOP functions.

.SH
\s+3Conclusion\s-3
.LP

\fBTO BE WRITTEN.\fP
