/*
 * $Id: msqltcl.c,v 1.0 1995/01/05 11:13:05 hs Rel $
 *
 * MSQL interface to Tcl
 *
 * Hakan Soderstrom, hs@soderstrom.se
 *
 */

/*
 * Copyright (c) 1994, 1995 Hakan Soderstrom and Tom Poindexter
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice and this permission notice
 * appear in all copies of the software and related documentation.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL HAKAN SODERSTROM OR SODERSTROM PROGRAMVARUVERKSTAD
 * AB BE LIABLE FOR ANY SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL
 * DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF THE POSSIBILITY
 * OF DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <tcl.h>
#include <msql.h>

#include <errno.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>


#define MSQL_HANDLES      15	/* Default number of handles available. */
#define MSQL_BUFF_SIZE	4096	/* Conversion buffer size for various needs. */
#define MSQL_SMALL_SIZE  TCL_RESULT_SIZE /* Smaller buffer size. */
#define MSQL_NAME_LEN     80    /* Max. host, database name length. */

typedef struct MsqlTclHandle {
  int connection ;         /* Connection handle, if connected; -1 otherwise. */
  char host[MSQL_NAME_LEN] ;      /* Host name, if connected. */
  char database[MSQL_NAME_LEN] ;  /* Db name, if selected; NULL otherwise. */
  m_result* result ;              /* Stored result, if any; NULL otherwise. */
  int res_count ;                 /* Count of unfetched rows in result. */
  int col_count ;                 /* Column count in result, if any. */
} MsqlTclHandle;

static MsqlTclHandle   MsqlHandle[MSQL_HANDLES];  

static char *MsqlHandlePrefix = "msql";
/* Prefix string used to identify handles.
 * The following must be strlen(MsqlHandlePrefix).
 */
#define MSQL_HPREFIX_LEN 4

static char *MsqlStatusArr = "msqlstatus";  /* Array for status info. */

/* Options to the 'info', 'result', 'col' combo commands. */
     
static char* MsqlDbOpt[] =
{
  "dbname", "dbname?", "tables", "host", "host?", "databases"
};
#define MSQL_INFNAME_OPT 0
#define MSQL_INFNAMEQ_OPT 1
#define MSQL_INFTABLES_OPT 2
#define MSQL_INFHOST_OPT 3
#define MSQL_INFHOSTQ_OPT 4
#define MSQL_INFLIST_OPT 5

#define MSQL_INF_OPT_MAX 5

static char* MsqlResultOpt[] =
{
  "rows", "rows?", "cols", "cols?"
};
#define MSQL_RESROWS_OPT 0
#define MSQL_RESROWSQ_OPT 1
#define MSQL_RESCOLS_OPT 2
#define MSQL_RESCOLSQ_OPT 3

#define MSQL_RES_OPT_MAX 3

/* Column info definitions. */

static char* MsqlColkey[] =
{
  "table", "name", "type", "length", "prim_key", "non_null"
};

#define MSQL_COL_TABLE_K 0
#define MSQL_COL_NAME_K 1
#define MSQL_COL_TYPE_K 2
#define MSQL_COL_LENGTH_K 3
#define MSQL_COL_PRIMKEY_K 4
#define MSQL_COL_NONNULL_K 5

#define MSQL_COL_K_MAX 5

/* Macro for checking handle syntax; arguments:
 * 'h' must point to the handle.
 * 'H' must be an int variable.
 * RETURNS the handle index; or -1 on syntax conflict.
 * SIDE EFFECT: 'H' will contain the handle index on success.
 * 'h' will point to the first digit on success.
 * ASSUMES constant MSQL_HANDLES >= 10.
 */
#define HSYNTAX(h,H) \
(((strncmp(h,MsqlHandlePrefix,MSQL_HPREFIX_LEN) == 0) && \
(h+=MSQL_HPREFIX_LEN) && isdigit(h[0]))? \
 (((H=h[0]-'0')+1 && h[1]=='\0')?H: \
  ((isdigit(h[1]) && h[2]=='\0' && (H=10*H+h[1]-'0')<MSQL_HANDLES)?H:-1)) \
:-1)


/* Prototypes for all functions. */

extern Tcl_CmdProc  Msqltcl_Connect;
extern Tcl_CmdProc  Msqltcl_Use;
extern Tcl_CmdProc  Msqltcl_Sel;
extern Tcl_CmdProc  Msqltcl_Next;
extern Tcl_CmdProc  Msqltcl_Map;
extern Tcl_CmdProc  Msqltcl_Exec;
extern Tcl_CmdProc  Msqltcl_Close;
extern Tcl_CmdProc  Msqltcl_Info;
extern Tcl_CmdProc  Msqltcl_Result;
extern Tcl_CmdProc  Msqltcl_Col;
extern Tcl_CmdProc  Msqltcl_State;


/*
 *----------------------------------------------------------------------
 * msql_tcl_conflict
 * Sets the status array after an mSQL conflict.
 *
 */

static void
msql_tcl_conflict (interp)
     Tcl_Interp *interp ;
{
    Tcl_SetVar2(interp, MsqlStatusArr, "code", "-1", TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, MsqlStatusArr, "message",
		(msqlErrMsg == NULL) ? "" : msqlErrMsg, TCL_GLOBAL_ONLY);
}


/*
 *----------------------------------------------------------------------
 * msql_prim_conflict
 * Sets the status array after a primitive conflict.
 * If 'mess' is NULL the current result is used.
 * RETURNS 'TCL_ERROR' for convenience.
 *
 */

static int
msql_prim_conflict (interp, mess)
     Tcl_Interp *interp ;
     char* mess ;
{
    Tcl_SetVar2(interp, MsqlStatusArr, "code", "-1", TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, MsqlStatusArr, "message",
		(mess == NULL) ? interp->result : mess, TCL_GLOBAL_ONLY);
    return TCL_ERROR ;
}

  
/*
 *----------------------------------------------------------------------
 * msql_tcl_perror
 * Sets the status array after a Unix system call conflict.
 *
 */

static void
msql_tcl_perror (interp)
     Tcl_Interp *interp ;
{
    char buf[MSQL_BUFF_SIZE];
    char* strerror () ;
    char* msg = strerror (errno) ;

    sprintf(buf, "%d", errno) ;
    Tcl_SetVar2(interp, MsqlStatusArr, "code", buf, TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, MsqlStatusArr, "message",
		(msg == NULL) ? "" : msg, TCL_GLOBAL_ONLY);
}


/*----------------------------------------------------------------------
 * get_handle_plain
 * Check handle syntax (and nothing else).
 * RETURN: MsqlHandle index number or -1 on error.
 */
static int
get_handle_plain (interp, handle) 
     Tcl_Interp *interp ;
     char *handle;
{
  int hi ;
  char* hp = handle ;

  if (HSYNTAX(hp,hi) < 0)
    {
      msql_prim_conflict (interp, "Weird handle") ;
      return -1 ;
    }
  else
    return hi ;
}


/*----------------------------------------------------------------------
 * get_handle_conn
 * Check handle syntax, verify that the handle is connected.
 * RETURN: MsqlHandle index number or -1 on error.
 */
static int
get_handle_conn (interp, handle) 
     Tcl_Interp *interp ;
     char *handle;
{
  int hi ;
  char* hp = handle ;

  if (HSYNTAX(hp,hi) < 0)
    {
      msql_prim_conflict (interp, "Weird handle") ;
      return -1 ;
    }

  if (MsqlHandle[hi].connection < 0)
    {
      msql_prim_conflict (interp, "Handle not connected") ;
      return -1 ;
    }
  else
    return hi ;
}


/*----------------------------------------------------------------------
 * get_handle_db
 * Check handle syntax, verify that the handle is connected and that
 * there is a current database.
 * RETURN: MsqlHandle index number or -1 on error.
 */
static int
get_handle_db (interp, handle) 
     Tcl_Interp *interp ;
     char *handle;
{
  int hi ;
  char* hp = handle ;

  if (HSYNTAX(hp,hi) < 0)
    {
      msql_prim_conflict (interp, "Weird handle") ;
      return -1 ;
    }

  if (MsqlHandle[hi].connection < 0)
    {
      msql_prim_conflict (interp, "Handle not connected") ;
      return -1 ;
    }

  if (MsqlHandle[hi].database[0] == '\0')
    {
      msql_prim_conflict (interp, "No current database") ;
      return -1 ;
    }
  else
    return hi ;
}


/*----------------------------------------------------------------------
 * get_handle_res
 * Check handle syntax, verify that the handle is connected and that
 * there is a current database and that there is a pending result.
 * RETURN: MsqlHandle index number or -1 on error.
 */
static int
get_handle_res (interp, handle) 
     Tcl_Interp *interp ;
     char *handle;
{
  int hi ;
  char* hp = handle ;

  if (HSYNTAX(hp,hi) < 0)
    {
      msql_prim_conflict (interp, "Weird handle") ;
      return -1 ;
    }

  if (MsqlHandle[hi].connection < 0)
    {
      msql_prim_conflict (interp, "Handle not connected") ;
      return -1 ;
    }

  if (MsqlHandle[hi].database[0] == '\0')
    {
      msql_prim_conflict (interp, "No current database") ;
      return -1 ;
    }

  if (MsqlHandle[hi].result == NULL)
    {
      msql_prim_conflict (interp, "No result pending") ;
      return -1 ;
    }
  else
    return hi ;
}


/* 
 *----------------------------------------------------------------------
 * handle_init
 * Initialize the handle array.
 */
static void 
handle_init () 
{
  int i ;

  for (i = 0; i < MSQL_HANDLES; i++) {
    MsqlHandle[i].connection = -1 ;
    MsqlHandle[i].host[0] = '\0' ;
    MsqlHandle[i].database[0] = '\0' ;
    MsqlHandle[i].result = NULL ;
    MsqlHandle[i].res_count = 0 ;
    MsqlHandle[i].col_count = 0 ;
  }
}


/*
 *----------------------------------------------------------------------
 * clear_msg
 *
 * Clears all error and message elements in the global array variable.
 *
 */

static void
clear_msg(interp)
    Tcl_Interp *interp;
{
    Tcl_SetVar2(interp, MsqlStatusArr, "code", "0", TCL_GLOBAL_ONLY);
    Tcl_SetVar2(interp, MsqlStatusArr, "message", "", TCL_GLOBAL_ONLY);
}


/*
 *----------------------------------------------------------------------
 * msql_prologue
 *
 * Does most of standard command prologue, assumes handle is argv[1].
 * RETURNS: Handle index  or -1 on failure.
 * 
 */

static int
msql_prologue (interp, argc, argv, num_args, err_msg, check)
     Tcl_Interp *interp;
     int         argc;
     char      **argv;
     int         num_args;
     char       *err_msg;
     int (*check) () ; /* Pointer to function for checking the handle. */
{
    int         hand;

    /* Check number of minimum args. */

    if (argc < num_args)
      {
	Tcl_AppendResult (interp, "usage: ", argv[0], err_msg, (char*)NULL);
	(void)msql_prim_conflict (interp, (char*)NULL) ;
	return -1 ;
      }

    /* Check the handle.
     * The function is assumed to set the status array on conflict.
     */

    if ((hand = check (interp, argv[1])) < 0)
      {
	Tcl_AppendResult (interp, argv[0], " ", argv[1],
			  ": invalid handle ", (char*)NULL);
	return -1 ;
      }

    /* Clear msqlstatus array for new messages & errors. */
    clear_msg(interp);

    return (hand);
}


/*
 *----------------------------------------------------------------------
 * msql_colinfo
 *
 * Given an m_field struct and a string keyword appends a piece of
 * column info (one item) to the Tcl result.
 * ASSUMES 'fld' is non-null.
 * In case of conflict, replaces the result by a message.
 * RETURNS 0 on success, 1 otherwise.
 */

static int
msql_colinfo (interp, fld, keyw)
     Tcl_Interp  *interp;
     m_field* fld ;
     char* keyw ;
{
  char buf[MSQL_SMALL_SIZE];
  char keybuf[MSQL_SMALL_SIZE];
  int idx ;
  char* res ;
  int retcode ;

  for (idx = 0;
       idx <= MSQL_COL_K_MAX && strcmp (MsqlColkey[idx], keyw) != 0;
       idx++) ;

  switch (idx)
    {
    case MSQL_COL_TABLE_K:
      res = fld->table ;
      break ;
    case MSQL_COL_NAME_K:
      res = fld->name ;
      break ;
    case MSQL_COL_TYPE_K:
      switch (fld->type)
	{
	case INT_TYPE:
	  res = "int" ;
	  break ;
	case CHAR_TYPE:
	  res = "char" ;
	  break ;
	case REAL_TYPE:
	  res = "real" ;
	  break ;
	default:
	  sprintf (buf, "msqlcol: column '%s' has weird datatype", fld->name) ;
	  res = NULL ;
	}
      break ;
    case MSQL_COL_LENGTH_K:
      sprintf (buf, "%d", fld->length) ;
      res = buf ;
      break ;
    case MSQL_COL_PRIMKEY_K:
      sprintf (buf, "%c", (IS_PRI_KEY(fld->flags))?'1':'0') ;
      res = buf ;
      break ;
    case MSQL_COL_NONNULL_K:
      sprintf (buf, "%c", (IS_NOT_NULL(fld->flags))?'1':'0') ;
      res = buf ;
      break ;
    default:
      if (strlen (keyw) >= MSQL_NAME_LEN)
	{
	  strncpy (keybuf, keyw, MSQL_NAME_LEN) ;
	  strcat (keybuf, "...") ;
	}
      else
	strcpy (keybuf, keyw) ;

      sprintf (buf, "msqlcol: unknown option: %s", keybuf) ;
      res = NULL ;
    }

  if (res == NULL)
    {
      Tcl_SetResult (interp, buf, TCL_VOLATILE) ;
      (void)msql_prim_conflict (interp, (char*)NULL) ;
      retcode = 1 ;
    }
  else
    {
      Tcl_AppendElement (interp, res) ;
      retcode = 0 ;
    }

  return retcode ;
}


/*
 *----------------------------------------------------------------------
 * Msqltcl_Kill
 * Close all connections.
 *
 */

void
Msqltcl_Kill (clientData)
    ClientData clientData;
{
  int i ;

  for (i = 0; i < MSQL_HANDLES; i++)
    {
      if (MsqlHandle[i].connection >=0)
	msqlClose (MsqlHandle[i].connection) ;
    }
  
  handle_init () ;
}


/*
 *----------------------------------------------------------------------
 * Msqltcl_Init
 * Perform all initialization for the MSQL to Tcl interface.
 * Adds additional commands to interp, creates message array, initializes
 * all handles.
 *
 * A call to Msqltcl_Init should exist in Tcl_CreateInterp or
 * Tcl_CreateExtendedInterp.
 */

int
Msqltcl_Init (interp)
    Tcl_Interp *interp;
{
  int i;

  /*
   * Initialize mSQL proc structures 
   */
  handle_init () ;

  /*
   * Initialize the new Tcl commands
   */
  Tcl_CreateCommand (interp, "msqlconnect", Msqltcl_Connect, (ClientData)NULL,
		     Msqltcl_Kill);
  Tcl_CreateCommand (interp, "msqluse",     Msqltcl_Use,     (ClientData)NULL,
		     Msqltcl_Kill);
  Tcl_CreateCommand (interp, "msqlsel",     Msqltcl_Sel,     (ClientData)NULL,
		     Msqltcl_Kill);
  Tcl_CreateCommand (interp, "msqlnext",    Msqltcl_Next,    (ClientData)NULL,
		     Msqltcl_Kill);
  Tcl_CreateCommand (interp, "msqlmap",     Msqltcl_Map,     (ClientData)NULL,
		     Msqltcl_Kill);
  Tcl_CreateCommand (interp, "msqlexec",    Msqltcl_Exec,    (ClientData)NULL,
		     Msqltcl_Kill);
  Tcl_CreateCommand (interp, "msqlclose",   Msqltcl_Close,   (ClientData)NULL,
		     Msqltcl_Kill);
  Tcl_CreateCommand (interp, "msqlinfo",    Msqltcl_Info,    (ClientData)NULL,
		     Msqltcl_Kill);
  Tcl_CreateCommand (interp, "msqlresult",  Msqltcl_Result,  (ClientData)NULL,
		     Msqltcl_Kill);
  Tcl_CreateCommand (interp, "msqlcol",     Msqltcl_Col, (ClientData)NULL,
		     Msqltcl_Kill);
  Tcl_CreateCommand (interp, "msqlstate",   Msqltcl_State,   (ClientData)NULL,
		     Msqltcl_Kill);

  /* Initialize msqlstatus global array. */
  clear_msg(interp);

  /* A little sanity check.
   * If this message appears you must change the source code and recompile.
   */
  if (strlen (MsqlHandlePrefix) == MSQL_HPREFIX_LEN)
    return TCL_OK;
  else
    {
      fprintf (stderr, "*** msqltcl (msqltcl.c): handle prefix inconsistency!\n") ;
      return TCL_ERROR ;
    }
}


/*
 *----------------------------------------------------------------------
 *
 * Msqltcl_Connect
 * Implements the msqlconnect command:
 * usage: msqlconnect ?server-host?
 *	                
 * Results:
 *      handle - a character string of newly open handle
 *      TCL_OK - connect successful
 *      TCL_ERROR - connect not successful - error message returned
 */

int
Msqltcl_Connect (clientData, interp, argc, argv)
     ClientData   clientData;
     Tcl_Interp  *interp;
     int          argc;
     char       **argv;
{
  int        hand = -1;
  int        i;
  char       buf[MSQL_BUFF_SIZE];
  int connect ;
  
  /* Can't use msql_prologue, handle is 'created' here. */

  /* Find an unused handle. */

    for (i = 0; i < MSQL_HANDLES; i++) {
	if (MsqlHandle[i].connection <= 0) {
            hand = i;
	    break;
	}
    }

    if (hand == -1) {
	Tcl_AppendResult (interp, argv[0], ": no mSQL handles available",
			  (char*)NULL);
	return msql_prim_conflict (interp, (char*)NULL) ;
    }

  if (argc > 1)
    {
      connect = msqlConnect (argv[1]) ;
      strncpy (MsqlHandle[hand].host, argv[1], MSQL_NAME_LEN) ;
      MsqlHandle[hand].host[MSQL_NAME_LEN - 1] = '\0' ;
    }
  else
    {
      if (gethostname (buf, MSQL_NAME_LEN) == 0)
	{
	  connect = msqlConnect () ;
	  strcpy (MsqlHandle[hand].host, buf) ;
	}
      else
	{
	  msql_tcl_perror (interp) ;
	  Tcl_AppendResult (interp, argv[0], ": cannot get host name",
			  (char*)NULL);
	  return TCL_ERROR;
	}
    }

  if (connect < 0)
    {
      msql_tcl_conflict (interp) ;
      MsqlHandle[hand].connection = -1 ; /* Just to be sure. */
      Tcl_AppendResult (interp, argv[0], ": connect failed",
			  (char*)NULL);
      return TCL_ERROR;
    }
  else
    MsqlHandle[hand].connection = connect ;

    /* Construct handle and return. */
    sprintf(buf, "%s%d", MsqlHandlePrefix, hand);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);

    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * Msqltcl_Use
 *    Implements the msqluse command:
 *    usage: msqluse handle dbname
 *	                
 *    results:
 *	Sets current database to dbname.
 */

Msqltcl_Use (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int hand;
    int toolong ;

    if ((hand = msql_prologue(interp, argc, argv, 3,
			      " handle dbname", get_handle_conn)) < 0)
      return TCL_ERROR;

    toolong = strlen (argv[2]) >= MSQL_NAME_LEN ;
    if (toolong || msqlSelectDB (MsqlHandle[hand].connection, argv[2]) < 0)
      {
	if (toolong)
	  (void)msql_prim_conflict (interp, "Database name too long") ;
	else
	  msql_tcl_conflict (interp) ;
	Tcl_AppendResult(interp, argv[0], ": msqluse failed", (char*)NULL);
	return TCL_ERROR;
      }
    else
      strcpy (MsqlHandle[hand].database, argv[2]) ;

    return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * Msqltcl_Sel
 *    Implements the msqlsel command:
 *    usage: msqlsql handle sel-query
 *	                
 *    results:
 *
 *    SIDE EFFECT: Flushes any pending result, even in case of conflict.
 *    Stores new results.
 */

Msqltcl_Sel (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
  int     hand;
  char    buf[MSQL_BUFF_SIZE];
  char    conv_buf[20];
  int     text_buf_size;


  if ((hand = msql_prologue(interp,argc, argv, 3,
			    " handle sel-query", get_handle_db)) < 0)
    return TCL_ERROR;

  /* Flush any previous result. */
  if (MsqlHandle[hand].result != NULL)
    {
      msqlFreeResult (MsqlHandle[hand].result) ;
      MsqlHandle[hand].result = NULL ;
    }

  if (msqlQuery (MsqlHandle[hand].connection, argv[2]) < 0)
    {
      msql_tcl_conflict (interp) ;
      Tcl_SetResult(interp, "msqlsel failed", TCL_VOLATILE);
      return TCL_ERROR;
    }

  if ((MsqlHandle[hand].result = msqlStoreResult ()) == NULL)
    {
      msql_tcl_conflict (interp) ;
      Tcl_SetResult(interp, "could not get result", TCL_VOLATILE);
      return TCL_ERROR;
    }
  else
    {
      MsqlHandle[hand].res_count = msqlNumRows (MsqlHandle[hand].result) ;
      MsqlHandle[hand].col_count = msqlNumFields (MsqlHandle[hand].result) ;
    }

  return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * Msqltcl_Exec
 * Implements the msqlexec command:
 * usage: msqlexec handle sql-statement
 *	                
 * Results:
 *
 * SIDE EFFECT: Flushes any pending result, even in case of conflict.
 */

int
Msqltcl_Exec (clientData, interp, argc, argv)
     ClientData   clientData;
     Tcl_Interp  *interp;
     int          argc;
     char       **argv;
{
  int     hand;
  
  if ((hand = msql_prologue(interp, argc, argv, 3,
			    " handle sql-statement", get_handle_db)) < 0)
      return TCL_ERROR;

    /* Flush any previous result. */
    if (MsqlHandle[hand].result != NULL)
      {
	msqlFreeResult (MsqlHandle[hand].result) ;
	MsqlHandle[hand].result = NULL ;
      }

    if (msqlQuery (MsqlHandle[hand].connection, argv[2]) < 0)
      {
	msql_tcl_conflict (interp) ;
	Tcl_SetResult(interp, "msqlexec failed", TCL_VOLATILE);
	return TCL_ERROR;
      }

    return TCL_OK ;
}


/*
 *----------------------------------------------------------------------
 *
 * Msqltcl_Next
 *    Implements the msqlnext command:
 *    usage: msqlnext handle
 *	                
 *    results:
 *	next row from pending results as tcl list, or null list.
 */

Msqltcl_Next (clientData, interp, argc, argv)
    ClientData   clientData;
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    int hand;
    int idx ;
    char buf[MSQL_BUFF_SIZE];
    m_row row ;

    if ((hand = msql_prologue(interp, argc, argv, 2,
			      " handle", get_handle_res)) < 0)
      return TCL_ERROR;

    if (MsqlHandle[hand].res_count == 0)
      return TCL_OK ;
    else if ((row = msqlFetchRow (MsqlHandle[hand].result)) == NULL)
      {
	MsqlHandle[hand].res_count = 0 ;
	Tcl_AppendResult(interp, argv[0], ": result counter out of sync", (char*)NULL);
	return msql_prim_conflict (interp, (char*)NULL) ;
      }
    else
      MsqlHandle[hand].res_count-- ;

    for (idx = 0 ; idx < MsqlHandle[hand].col_count ; idx++)
      Tcl_AppendElement (interp, *row++) ;
    
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * Msqltcl_Map
 * Implements the msqlmap command:
 * usage: msqlmap handle binding-list script
 *	                
 * Results:
 * SIDE EFFECT: For each row the column values are bound to the variables
 * in the binding list and the script is evaluated.
 * The variables are created in the current context.
 * NOTE: msqlmap works very much like a 'foreach' construct.
 * The 'continue' and 'break' commands may be used with their usual effect.
 */

int
Msqltcl_Map (clientData, interp, argc, argv)
     ClientData   clientData;
     Tcl_Interp  *interp;
     int          argc;
     char       **argv;
{
  int code ;
  int count ;
  int hand ;
  int idx ;
  int listArgc ;
  char** listArgv ;
  m_row row ;

  if ((hand = msql_prologue(interp, argc, argv, 4,
			    " handle binding-list script", get_handle_res)) < 0)
    return TCL_ERROR;

  if (Tcl_SplitList (interp, argv[2], &listArgc, &listArgv) != TCL_OK)
    return TCL_ERROR ;
  
  if (listArgc > MsqlHandle[hand].col_count)
    {
      Tcl_AppendResult (interp, argv[0], ": too many variables in binding list",
			(char*)NULL) ;
      ckfree ((char*)listArgv) ;
      return msql_prim_conflict (interp, (char*)NULL) ;
    }
  else
    count = (listArgc < MsqlHandle[hand].col_count)?listArgc
      :MsqlHandle[hand].col_count ;
  
  while (MsqlHandle[hand].res_count > 0)
    {
      /* Get next row, decrement row counter. */
      if ((row = msqlFetchRow (MsqlHandle[hand].result)) == NULL)
	{
	  MsqlHandle[hand].res_count = 0 ;
	  Tcl_AppendResult(interp, argv[0], ": result counter out of sync",
			   (char*)NULL);
	  ckfree ((char*)listArgv) ;
	  return msql_prim_conflict (interp, (char*)NULL) ;
	}
      else
	MsqlHandle[hand].res_count-- ;
      
      /* Bind variables to column values. */
      for (idx = 0; idx < count; idx++)
	{
	  if (listArgv[idx][0] != '-')
	    {
	      if (Tcl_SetVar (interp, listArgv[idx], *row++, TCL_LEAVE_ERR_MSG)
		  == NULL)
		{
		  ckfree ((char*)listArgv) ;
		  return TCL_ERROR ;
		}
	    }
	  else
	    row++ ;
	}

      /* Evaluate the script. */
      if ((code = Tcl_Eval (interp, argv[3])) != TCL_OK)
	switch (code) 
	  {
	  case TCL_CONTINUE:
	    continue ;
	    break ;
	  case TCL_BREAK:
	    ckfree ((char*)listArgv) ;
	    return TCL_OK ;
	    break ;
	  default:
	    ckfree ((char*)listArgv) ;
	    return code ;
	  }
    }
  ckfree ((char*)listArgv) ;
  return TCL_OK ;
}


/*
 *----------------------------------------------------------------------
 *
 * Msqltcl_Info
 * Implements the msqldb command:
 * usage: msqldb handle option
 *
 */

int
Msqltcl_Info (clientData, interp, argc, argv)
     ClientData   clientData;
     Tcl_Interp  *interp;
     int          argc;
     char       **argv;
{
  int count ;
  int hand ;
  int idx ;
  m_result* list ;
  m_row row ;
  
  
  /* We can't fully check the handle at this stage. */
  if ((hand = msql_prologue(interp, argc, argv, 3,
			    " handle option", get_handle_plain)) < 0)
    return TCL_ERROR;

  for (idx = 0;
       idx <= MSQL_INF_OPT_MAX && strcmp (argv[2], MsqlDbOpt[idx]) != 0;
       idx++) ;

  /* First check the handle. Checking depends on the option. */
  switch (idx)
    {
    case MSQL_INFNAME_OPT:
    case MSQL_INFTABLES_OPT:
      hand = get_handle_db (interp, argv[1]) ;
      break ;
    case MSQL_INFNAMEQ_OPT:
      if ((hand = get_handle_conn (interp, argv[1])) >= 0)
	{
	  if (MsqlHandle[hand].database[0] == '\0')
	    return TCL_OK ; /* Return empty string if no current db. */
	}
      break ;
    case MSQL_INFHOST_OPT:
    case MSQL_INFLIST_OPT:
      hand = get_handle_conn (interp, argv[1]) ;
      break ;
    case MSQL_INFHOSTQ_OPT:
      if (MsqlHandle[hand].connection < 0)
	return TCL_OK ; /* Return empty string if not connected. */
      break ;
    default: /* unknown option */
      Tcl_AppendResult (interp, argv[0], ": '", argv[2], "' unknown option",
			(char*)NULL);
      return msql_prim_conflict (interp, (char*)NULL) ;
    }

  if (hand < 0)
    {
      /* Status has already been set by the handle checking function. */
      Tcl_AppendResult (interp, argv[0], ": ", argv[1], " invalid handle",
			(char*)NULL) ;
      return TCL_ERROR ;
    }

  /* Handle OK, return the requested info. */
  switch (idx)
    {
    case MSQL_INFNAME_OPT:
    case MSQL_INFNAMEQ_OPT:
      strcpy (interp->result, MsqlHandle[hand].database) ;
      break ;
    case MSQL_INFTABLES_OPT:
      if ((list = msqlListTables (MsqlHandle[hand].connection)) == NULL)
	{
	  Tcl_AppendResult (interp, argv[0], ": could not access table names",
			    "; server may have gone away", (char*)NULL);
	  return msql_prim_conflict (interp, (char*)NULL) ;
	}
      for (count = msqlNumRows (list); count > 0; count--)
	{
	  row = msqlFetchRow (list) ;
	  Tcl_AppendElement (interp, *row) ;
	}
      msqlFreeResult (list) ;
      break ;
    case MSQL_INFHOST_OPT:
    case MSQL_INFHOSTQ_OPT:
      strcpy (interp->result, MsqlHandle[hand].host) ;
      break ;
    case MSQL_INFLIST_OPT:
      if ((list = msqlListDBs (MsqlHandle[hand].connection)) == NULL)
	{
	  Tcl_AppendResult (interp, argv[0], ": could not access database names",
			    "; server may have gone away", (char*)NULL) ;
	  return msql_prim_conflict (interp, (char*)NULL) ;
	}
      for (count = msqlNumRows (list); count > 0; count--)
	{
	  row = msqlFetchRow (list) ;
	  Tcl_AppendElement (interp, *row) ;
	}
      msqlFreeResult (list) ;
      break ;
    default: /* should never happen */
      Tcl_AppendResult (interp, argv[0], ": weirdness in Msqltcl_Info",
			(char*)NULL);
      return msql_prim_conflict (interp, (char*)NULL) ;
    }
  return TCL_OK ;
}


/*
 *----------------------------------------------------------------------
 *
 * Msqltcl_Result
 * Implements the msqlresult command:
 * usage: msqlresult handle option
 *
 */

int
Msqltcl_Result (clientData, interp, argc, argv)
     ClientData   clientData;
     Tcl_Interp  *interp;
     int          argc;
     char       **argv;
{
  int count ;
  int hand ;
  int idx ;
  m_result* list ;
  m_row row ;


  /* We can't fully check the handle at this stage. */
  if ((hand = msql_prologue(interp, argc, argv, 3,
			    " handle option", get_handle_plain)) < 0)
    return TCL_ERROR;

  for (idx = 0;
       idx <= MSQL_RES_OPT_MAX && strcmp (argv[2], MsqlResultOpt[idx]) != 0;
       idx++) ;

  /* First check the handle. Checking depends on the option. */
  switch (idx)
    {
    case MSQL_RESROWS_OPT:
    case MSQL_RESCOLS_OPT:
      hand = get_handle_res (interp, argv[1]) ;
      break ;
    case MSQL_RESROWSQ_OPT:
    case MSQL_RESCOLSQ_OPT:
      if ((hand = get_handle_db (interp, argv[1])) >= 0)
	{
	  if (MsqlHandle[hand].result == NULL)
	    return TCL_OK ; /* Return null string if no pending result. */
	}
      break ;
    default: /* unknown option */
      Tcl_AppendResult (interp, argv[0], ": '", argv[2], "' unknown option",
			(char*)NULL);
      return msql_prim_conflict (interp, (char*)NULL) ;
    }

  if (hand < 0)
    {
      /* Status has already been set by the handle checking function. */
      Tcl_AppendResult (interp, argv[0], ": ", argv[1], " invalid handle",
			(char*)NULL) ;
      return TCL_ERROR ;
    }

  /* Handle OK; return requested info. */
  switch (idx)
    {
    case MSQL_RESROWS_OPT:
    case MSQL_RESROWSQ_OPT:
      sprintf (interp->result, "%d", MsqlHandle[hand].res_count) ;
      break ;
    case MSQL_RESCOLS_OPT:
    case MSQL_RESCOLSQ_OPT:
      sprintf (interp->result, "%d", MsqlHandle[hand].col_count) ;
      break ;
    default:
    }
  return TCL_OK ;
}


/*
 *----------------------------------------------------------------------
 *
 * Msqltcl_Col
 *    Implements the msqlcol command:
 *    usage: msqlcol handle table-name option ?option ...?
 *           msqlcol handle -current option ?option ...?
 * '-current' can only be used if there is a pending result.
 *	                
 *    results:
 *	List of lists containing column attributes.
 *      If a single attribute is requested the result is a simple list.
 *
 * SIDE EFFECT: '-current' disturbs the field position of the result.
 */

Msqltcl_Col (clientData, interp, argc, argv)
     ClientData   clientData;
     Tcl_Interp  *interp;
     int          argc;
     char       **argv;
{
  int coln ;
  int conflict ;
  int current_db ;
  int hand;
  int idx ;
  int listArgc ;
  char** listArgv ;
  m_field* fld ;
  m_result* result ;
  char* sep ;
  int simple ;
  
  if ((hand = msql_prologue(interp, argc, argv, 4,
			    " handle table-name option ?option ...?",
			    get_handle_db)) < 0)
    return TCL_ERROR;

  /* Fetch column info.
   * Two ways: explicit database and table names, or current.
   */
  current_db = strcmp (argv[2], "-current") == 0 ;
  
  if (current_db)
    {
      if ((hand = get_handle_res (interp, argv[1])) < 0)
	{
	  Tcl_AppendResult (interp, argv[0], " ", argv[1],
			    ": invalid handle ", (char*)NULL);
	  return TCL_ERROR ;
	}
      else
	result = MsqlHandle[hand].result ;
    }
  else
    {
      if ((result = msqlListFields (MsqlHandle[hand].connection, argv[2])) == NULL)
	{
	  Tcl_AppendResult (interp, argv[0], ": no column info for table '",
			    argv[2], "'",
			    "; server may have gone away", (char*)NULL) ;
	  return msql_prim_conflict (interp, (char*)NULL) ;
	}
    }

  /* Must examine the first specifier at this point. */
  if (Tcl_SplitList (interp, argv[3], &listArgc, &listArgv) != TCL_OK)
    return TCL_ERROR ;

  conflict = 0 ;
  simple = (argc == 4) && (listArgc == 1) ;

  if (simple)
    {
      msqlFieldSeek (result, 0) ;
      while ((fld = msqlFetchField (result)) != NULL)
	if (msql_colinfo (interp, fld, argv[3]))
	  {
	    conflict = 1 ;
	    break ;
	  }
    }
  else if (listArgc > 1)
    {
      msqlFieldSeek (result, 0) ;
      for (sep = "{"; (fld = msqlFetchField (result)) != NULL; sep = " {")
	{
	  Tcl_AppendResult (interp, sep, (char*)NULL) ;
	  for (coln = 0; coln < listArgc; coln++)
	    if (msql_colinfo (interp, fld, listArgv[coln]))
	      {
		conflict = 1 ;
		break ;
	      }
	  if (conflict)
	    break ;
	  Tcl_AppendResult (interp, "}", (char*)NULL) ;
	}
      ckfree ((char*)listArgv) ;
    }
  else
    {
      ckfree ((char*)listArgv) ; /* listArgv == 1, no splitting */
      for (idx = 3, sep = "{"; idx < argc; idx++, sep = " {")
	{
	  Tcl_AppendResult (interp, sep, (char*)NULL) ;
	  msqlFieldSeek (result, 0) ;
	  while ((fld = msqlFetchField (result)) != NULL)
	    if (msql_colinfo (interp, fld, argv[idx]))
	      {
		conflict = 1 ;
		break ;
	      }
	  if (conflict)
	    break ;
	  Tcl_AppendResult (interp, "}", (char*)NULL) ;
	}
    }
  
  if (!current_db)
    msqlFreeResult (result) ;
  return TCL_OK ;
}


/*
 *----------------------------------------------------------------------
 *
 * Msqltcl_State
 *    Implements the msqlstate command:
 *    usage: msqlstate ?-numeric? handle 
 *	                
 */

Msqltcl_State (clientData, interp, argc, argv)
     ClientData   clientData;
     Tcl_Interp  *interp;
     int          argc;
     char       **argv;
{
  int hi;
  char* hp ;
  int numeric ;
  char* res ;
  
  if (argc < 2)
    {
      Tcl_AppendResult (interp, argv[0], ": too few arguments", (char*)NULL);
      return msql_prim_conflict (interp, (char*)NULL) ;
    }
  else if ((numeric = (strcmp (argv[1], "-numeric") == 0)) && argc < 3)
    {
      Tcl_AppendResult (interp, argv[0], ": too few arguments", (char*)NULL);
      return msql_prim_conflict (interp, (char*)NULL) ;
    }

  hp = (numeric)?argv[2]:argv[1] ;

  if (HSYNTAX(hp,hi) < 0)
    res = (numeric)?"0":"NOT_A_HANDLE" ;
  else if (MsqlHandle[hi].connection < 0)
    res = (numeric)?"1":"UNCONNECTED" ;
  else if (MsqlHandle[hi].database[0] == '\0')
    res = (numeric)?"2":"CONNECTED" ;
  else if (MsqlHandle[hi].result == NULL)
    res = (numeric)?"3":"IN_USE" ;
  else
    res = (numeric)?"4":"RESULT_PENDING" ;

  strcpy (interp->result, res) ;
  return TCL_OK ;
}


/*
 *----------------------------------------------------------------------
 *
 * Msqltcl_Close --
 *    Implements the msqlclose command:
 *    usage: msqlclose ?handle?
 *	                
 *    results:
 *	null string
 */

Msqltcl_Close (clientData, interp, argc, argv)
     ClientData   clientData;
     Tcl_Interp  *interp;
     int          argc;
     char       **argv;
{
  int     hand;
  
  if (argc == 1)
    {
      Msqltcl_Kill ((ClientData)NULL) ;
      return TCL_OK ;
    }
  
  if ((hand = msql_prologue(interp, argc, argv, 2,
			    " handle", get_handle_conn)) < 0)
    return TCL_ERROR;

  msqlClose (MsqlHandle[hand].connection) ;

  MsqlHandle[hand].connection = -1 ;
  MsqlHandle[hand].host[0] = '\0' ;
  MsqlHandle[hand].database[0] = '\0' ;

  if (MsqlHandle[hand].result != NULL)
    msqlFreeResult (MsqlHandle[hand].result) ;
    
  MsqlHandle[hand].result = NULL ;
  MsqlHandle[hand].res_count = 0 ;

  return TCL_OK;
}
