/*	@(#)net_tcl.c	1.2	2/12/93	*/

/*
	net_tcl.c
	
		Tcl sytle calls linked to netCDF functions.

	Created 2/10/92		Ryan McLean

*/

#include <stdio.h>
#include <malloc.h>

#include "tcl.h"
#include "netcdf.h"
#include "net_tcl.h"


#define	MAX_STRING 10	/*  Maximum length of a number to string conversion. */
#define MAX_RETURN 30	/*  Maximum length of a return string.	*/


/*  Initially clobbering is not an option.	*/
int
tcl_nccreate(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int cid;
	char	ret_string[MAX_RETURN];

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" filename\"", (char *) NULL);
		return TCL_ERROR;
	}
	
	cid =  nccreate(argv[1], NC_CLOBBER);
	if (cid == -1) {
		Tcl_AppendResult(interp, "Error creating ", argv[1], 
				(char *) NULL); 
		return TCL_ERROR;
	}

	/*  Generate the return value	*/
	sprintf(ret_string, "%d", cid);
	Tcl_SetResult(interp, ret_string, TCL_VOLATILE);

	return TCL_OK;
}
	


/*  Open everything as read-write to begin with.  Add read-only mode
later.
     Returns the netCDF id as a string value.
*/

int
tcl_ncopen(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int cid;
	char	ret_string[MAX_RETURN];

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" filename\"", (char *) NULL);
		return TCL_ERROR;
	}
	
	cid =  ncopen(argv[1], NC_WRITE);
	if (cid == -1) {
		Tcl_AppendResult(interp, "Error opening ", argv[1], 
				(char *) NULL); 
		return TCL_ERROR;
	}

	/*  Generate the return value	*/
	sprintf(ret_string, "%d", cid);
	Tcl_SetResult(interp, ret_string, TCL_VOLATILE);

	return TCL_OK;
}

int
tcl_ncredef(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int cid, status;

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid\"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (reading cdfid)");
                return TCL_ERROR;
        }

	status = ncredef(cid);
	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	Tcl_SetResult(interp, (char *) NULL, TCL_STATIC);

	return TCL_OK;
}


int
tcl_ncendef(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int cid, status;

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid\"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }

	status = ncendef(cid);
	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	Tcl_SetResult(interp, (char *) NULL, TCL_STATIC);

	return TCL_OK;
}


int
tcl_ncclose(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int cid, status;

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid\"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }

	status = ncclose(cid);
	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	Tcl_SetResult(interp, (char *) NULL, TCL_STATIC);

	return TCL_OK;
}


int
tcl_ncinquire(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int cid, status;
	int ndims, nvars, ngatts, recdim;
	char	ret_string[MAX_RETURN];

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid\"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }

	status = ncinquire(cid, &ndims, &nvars, &ngatts, &recdim);
	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	sprintf(ret_string, "%d %d %d %d", ndims, nvars, ngatts, recdim);
	Tcl_SetResult(interp, ret_string, TCL_VOLATILE);

	return TCL_OK;
}


int
tcl_ncabort(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int cid, status;

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid\"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }

	status = ncabort(cid);
	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	Tcl_SetResult(interp, (char *) NULL, TCL_STATIC);

	return TCL_OK;
}


int
tcl_ncsync(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int cid, status;

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid\"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }

	status = ncsync(cid);
	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	Tcl_SetResult(interp, (char *) NULL, TCL_STATIC);

	return TCL_OK;
}

int
tcl_ncsetfill(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int cid, status, mode;
	char	ret_string[MAX_RETURN];

	if (argc != 3) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid mode\"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }
	if (Tcl_GetBoolean(interp, argv[2], &mode) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (mode)");
                return TCL_ERROR;
        }

	if (mode) {
		status = ncsetfill(cid, NC_FILL);
		sprintf(ret_string, "NC_FILL set");
	}
	if (!mode) {
		status = ncsetfill(cid, NC_NOFILL);
		sprintf(ret_string, "NC_NOFILL set");
	}

	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	Tcl_SetResult(interp, ret_string, TCL_VOLATILE);
	return TCL_OK;
}



int
tcl_ncdimdef(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int cid, dimid;
	int	size;
	char	ret_string[MAX_RETURN];

	if (argc != 4) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid name size\"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }
	if (Tcl_GetInt(interp, argv[3], &size) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (size)");
                return TCL_ERROR;
        }
	
	if (size != -1) {
		dimid = ncdimdef(cid, argv[2], (long) size);
	}
	/*  IF the size is -1, make this the unlimited dimension.  */
	if (size == -1) {
		dimid = ncdimdef(cid, argv[2], NC_UNLIMITED);
	}

	if (dimid == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	sprintf(ret_string, "%d", dimid);
	Tcl_SetResult(interp, ret_string, TCL_VOLATILE);
	return TCL_OK;
}


int
tcl_ncdimid(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int cid, dimid;
	char	ret_string[MAX_RETURN];

	if (argc != 3) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid id\"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }
	
	dimid = ncdimid(cid, argv[2]);

	if (dimid == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	sprintf(ret_string, "%d", dimid);
	Tcl_SetResult(interp, ret_string, TCL_VOLATILE);
	return TCL_OK;
}


int
tcl_ncdiminq(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int cid, dimid;
	char	name[MAX_NC_NAME];
	long	size;
	char	ret_string[MAX_RETURN + MAX_NC_NAME];

	if (argc != 3) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid dimid \"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }
	if (Tcl_GetInt(interp, argv[2], &dimid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (dimid)");
                return TCL_ERROR;
        }
	
	dimid = ncdiminq(cid, dimid, name, &size);

	if (dimid == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	sprintf(ret_string, "%s %ld", name, size);
	Tcl_SetResult(interp, ret_string, TCL_VOLATILE);
	return TCL_OK;
}


int
tcl_ncdimrename(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int cid, dimid, status;

	if (argc != 4) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid dimid name\"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }
	if (Tcl_GetInt(interp, argv[2], &dimid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (dimid)");
                return TCL_ERROR;
        }
	
	
	status = ncdimrename(cid, dimid, argv[3]);

	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	Tcl_SetResult(interp, NULL, TCL_VOLATILE);
	return TCL_OK;
}



int
tcl_ncvardef(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int	cid, status;
	int	ndims, dim[MAX_VAR_DIMS];
	nc_type	type = NC_UNSPECIFIED; 	/* Warning! this is considerd private */
	char	ret_string[MAX_RETURN];
	int	i;
	char	**list;

	if (argc != 5) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid name type dimlist\"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }
	
	/*  Check for the proper type keyword (first letter only).	*/
	if (Tcl_StringMatch(argv[3], "[bB]*")) {
		type = NC_BYTE;
	}
	if (Tcl_StringMatch(argv[3], "[cC]*")) {
		type = NC_CHAR;
	}
	if (Tcl_StringMatch(argv[3], "[sS]*")) {
		type = NC_SHORT;
	}
	if (Tcl_StringMatch(argv[3], "[lL]*")) {
		type = NC_LONG;
	}
	if (Tcl_StringMatch(argv[3], "[fF]*")) {
		type = NC_FLOAT;
	}
	if (Tcl_StringMatch(argv[3], "[dD]*")) {
		type = NC_DOUBLE;
	}

	/*  Break up the last argument into a dimension array.	*/
	if (Tcl_SplitList(interp, argv[4], &ndims, &list) != TCL_OK) {
		Tcl_AppendResult(interp, "Can't split the list of dimensions ",
				argv[0], (char *) NULL);
		return TCL_ERROR;
	}
	
	/*  Split each element into an integer.	*/
	for (i=0; i<ndims; i++) {
		if (Tcl_GetInt(interp, list[i], dim+i) != TCL_OK) {
			Tcl_AddErrorInfo(interp, 
				"\n	(dimension array)");
			return TCL_ERROR;
		}
	}
	
	free((char *) list);
			
	
	status = ncvardef(cid, argv[2], type, ndims, dim);

	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	sprintf(ret_string, "%d", status);
	Tcl_SetResult(interp, ret_string, TCL_VOLATILE);
	return TCL_OK;
}



int
tcl_ncvarid(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int cid, varid;
	char	ret_string[MAX_RETURN];

	if (argc != 3) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid id\"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }
	
	varid = ncvarid(cid, argv[2]);

	if (varid == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	sprintf(ret_string, "%d", varid);
	Tcl_SetResult(interp, ret_string, TCL_VOLATILE);
	return TCL_OK;
}


int
tcl_ncvarinq(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int	cid, status, varid;
	int	ndims, dim[MAX_VAR_DIMS], natts;
	nc_type	type;
	char	name[MAX_NC_NAME];
	char	*merge;
	char	*ret_string;
	char	*ret[4];
	int	i;

	/*  String pointer array for the dimension integers in string form. */
	char	*dimstr[MAX_VAR_DIMS];
	char	nattstr[MAX_STRING], *typestr = NULL;

	if (argc != 3) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid varid \"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }
	if (Tcl_GetInt(interp, argv[2], &varid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (varid)");
                return TCL_ERROR;
        }
	
	
	status = ncvarinq(cid, varid, name, &type, &ndims, dim, &natts);


	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	/*  Get the type	*/
	if (type == NC_BYTE) {
		typestr = "byte";
	}
	if (type == NC_CHAR) {
		typestr = "char";
	}
	if (type == NC_SHORT) {
		typestr = "short";
	}
	if (type == NC_LONG) {
		typestr = "long";
	}
	if (type == NC_FLOAT) {
		typestr = "float";
	}
	if (type == NC_DOUBLE) {
		typestr = "double";
	}
	
	/*   Parse the dims into strings.	*/
	for (i=0; i< ndims; i++) {
		dimstr[i] = calloc(MAX_STRING, sizeof(char));
		sprintf(dimstr[i], "%d", dim[i]);
	}
	sprintf(nattstr, "%d", natts);

	merge = Tcl_Merge(ndims, dimstr);
	ret[0] = name;
	ret[1] = typestr;
	ret[2] = merge;
	ret[3] = nattstr;
	ret_string = Tcl_Merge(4, ret);
	
	
	Tcl_SetResult(interp, ret_string, TCL_VOLATILE);

	/*  Free up the dimension string.	*/
	for (i=0; i<ndims; i++) {
		free(dimstr[i]);
	}
	/*  Free up the merge string.	*/
	free(merge);
	free(ret_string);

	return TCL_OK;
}



int
tcl_ncvarput(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int	cid, status, varid;
	int	dim[MAX_VAR_DIMS];
	long	mindex[MAX_VAR_DIMS];
	char	value[MAX_STRING];
	nc_type	type;
	char	**list;
	int	i, ndims;
	char	*typestr = NULL;

	if (argc != 5) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid varid indices value\"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }
	if (Tcl_GetInt(interp, argv[2], &varid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (varid)");
                return TCL_ERROR;
        }

	
	/*  Get the type of the variable and perform the scan of
	the argument accordingly.	*/
	status = ncvarinq(cid, varid, (char *) 0, &type, (int *) 0,
			(int *) 0, (int *) 0);
	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
	

	/*  Get the type	*/
	if (type == NC_BYTE) {
		typestr = "%c";
	}
	if (type == NC_CHAR) {
		typestr = "%c";
	}
	if (type == NC_SHORT) {
		typestr = "%hd";
	}
	if (type == NC_LONG) {
		typestr = "%ld";
	}
	if (type == NC_FLOAT) {
		typestr = "%f";
	}
	if (type == NC_DOUBLE) {
		typestr = "%lf";
	}


	/*  Scan the string into the carrier.	*/
	sscanf(argv[4], typestr, value);
	
	/*  Process the list of dimensions into a int array.	*/
	/*  Break up the last argument into a dimension array.	*/
	if (Tcl_SplitList(interp, argv[3], &ndims, &list) != TCL_OK) {
		Tcl_AppendResult(interp, "Can't split the list of dimensions ",
				argv[0], (char *) NULL);
		return TCL_ERROR;
	}
	
	/*  Split each element into an integer.	*/
	for (i=0; i<ndims; i++) {
		if (Tcl_GetInt(interp, list[i], dim+i) != TCL_OK) {
			Tcl_AddErrorInfo(interp, 
				"\n	(dimension array)");
			return TCL_ERROR;
		}
		mindex[i] = (long) dim[i];
	}
	
	free((char *) list);
			

	/*  Make the call.	*/
	status = ncvarput1(cid, varid, mindex, value);


	/*  Return value.	*/
	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	return TCL_OK;
}




int
tcl_ncvarget(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int	cid, status, varid;
	int	dim[MAX_VAR_DIMS];
	long	mindex[MAX_VAR_DIMS];
	char	value[MAX_STRING];
	nc_type	type;
	char	**list;
	int	i, ndims;
	char	ret_string[MAX_RETURN];

	if (argc != 4) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid varid indices \"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }
	if (Tcl_GetInt(interp, argv[2], &varid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (varid)");
                return TCL_ERROR;
        }

	
	/*  Get the type of the variable and perform the scan of
	the argument accordingly.	*/
	status = ncvarinq(cid, varid, (char *) 0, &type, (int *) 0,
			(int *) 0, (int *) 0);
	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
	


	/*  Process the list of dimensions into a int array.	*/
	/*  Break up the last argument into a dimension array.	*/
	if (Tcl_SplitList(interp, argv[3], &ndims, &list) != TCL_OK) {
		Tcl_AppendResult(interp, "Can't split the list of dimensions ",
				argv[0], (char *) NULL);
		return TCL_ERROR;
	}
	
	/*  Split each element into an integer.	*/
	for (i=0; i<ndims; i++) {
		if (Tcl_GetInt(interp, list[i], dim+i) != TCL_OK) {
			Tcl_AddErrorInfo(interp, 
				"\n	(dimension array)");
			return TCL_ERROR;
		}
		mindex[i] = (long) dim[i];
	}
	
	free((char *) list);

	/*  Make the call.	*/
	status = ncvarget1(cid, varid, mindex, value);

	/*  Return value.	*/
	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }

	/*  Get the type	*/
	/*  A new first for me.  Aren't these cool casts?  */
	if (type == NC_BYTE) {
		sprintf(ret_string, "%c", *((char *) value));
	}
	if (type == NC_CHAR) {
		sprintf(ret_string, "%c", *((char *) value));
	}
	if (type == NC_SHORT) {
		sprintf(ret_string, "%hd", *((short *) value));
	}
	if (type == NC_LONG) {
		sprintf(ret_string, "%ld", *((long *) value));
	}
	if (type == NC_FLOAT) {
		sprintf(ret_string, "%g", *((float *) value));
	}
	if (type == NC_DOUBLE) {
		sprintf(ret_string, "%lg", *((double *) value));
	}
 
	Tcl_SetResult(interp, ret_string, TCL_VOLATILE);
	return TCL_OK;
}



int
tcl_ncvarrename(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int cid, varid, status;

	if (argc != 4) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid varid name\"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }
	if (Tcl_GetInt(interp, argv[2], &varid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (varid)");
                return TCL_ERROR;
        }
	
	
	status = ncvarrename(cid, varid, argv[3]);

	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	Tcl_SetResult(interp, NULL, TCL_VOLATILE);
	return TCL_OK;
}



int
tcl_nctypelen(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int	status;
	nc_type	type = -1;
	char	ret_string[MAX_RETURN];

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" datatype \"", (char *) NULL);
		return TCL_ERROR;
	}
	
	
	/*  Check for the proper type keyword (first letter only).	*/
	if (Tcl_StringMatch(argv[1], "[bB]*")) {
		type = NC_BYTE;
	}
	if (Tcl_StringMatch(argv[1], "[cC]*")) {
		type = NC_CHAR;
	}
	if (Tcl_StringMatch(argv[1], "[sS]*")) {
		type = NC_SHORT;
	}
	if (Tcl_StringMatch(argv[1], "[lL]*")) {
		type = NC_LONG;
	}
	if (Tcl_StringMatch(argv[1], "[fF]*")) {
		type = NC_FLOAT;
	}
	if (Tcl_StringMatch(argv[1], "[dD]*")) {
		type = NC_DOUBLE;
	}

	status = nctypelen(type);

	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	sprintf(ret_string, "%d", status);
	Tcl_SetResult(interp, ret_string, TCL_VOLATILE);
	return TCL_OK;
}




int
tcl_ncattput(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int	cid, status, varid;
	void	*values;
	nc_type	type;
	char	**list;
	int	i, ndims;

	if (argc != 6) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid varid name type values\"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }

	/*  If the variable id is not a number, assume that
	it is a request for a GLOBAL attribute put.	*/
	if (Tcl_GetInt(interp, argv[2], &varid) != TCL_OK) {
		varid = NC_GLOBAL;
        }

	
	/*  Check for the proper type keyword (first letter only).	*/
	if (Tcl_StringMatch(argv[4], "[bB]*")) {
		type = NC_BYTE;
	}
	if (Tcl_StringMatch(argv[4], "[cC]*")) {
		type = NC_CHAR;
	}
	if (Tcl_StringMatch(argv[4], "[sS]*")) {
		type = NC_SHORT;
	}
	if (Tcl_StringMatch(argv[4], "[lL]*")) {
		type = NC_LONG;
	}
	if (Tcl_StringMatch(argv[4], "[fF]*")) {
		type = NC_FLOAT;
	}
	if (Tcl_StringMatch(argv[4], "[dD]*")) {
		type = NC_DOUBLE;
	}
	

	/*  Break up the last argument into an array of values.	*/
	if (Tcl_SplitList(interp, argv[5], &ndims, &list) != TCL_OK) {
		Tcl_AppendResult(interp, "Can't split the list of dimensions ",
				argv[0], (char *) NULL);
		return TCL_ERROR;
	}

	/*  Make an allocation of the correct type for the values.	*/
	switch(type)  {
		case NC_BYTE:
			values = (void *) calloc(ndims, sizeof(char));
			break;

		case NC_CHAR:
			values = (void *) argv[5];
			ndims = strlen(argv[5]) + 1;
			break;

		case NC_SHORT:
			values = (void *) calloc(ndims, sizeof(short));
			break;

		case NC_LONG:
			values = (void *) calloc(ndims, sizeof(long));
			break;

		case NC_FLOAT:
			values = (void *) calloc(ndims, sizeof(float));
			break;

		case NC_DOUBLE:
			values = (void *) calloc(ndims, sizeof(double));
			break;

		default:
			return TCL_ERROR;
	}
	
	/*  Split each element according to type	*/
	for (i=0; i<ndims; i++) {
	   switch(type)  {
		case NC_BYTE:
			sscanf(list[i], "%c", ((char *) values) + i);
			break;

		case NC_CHAR:
			/*  No conversion necessary, since this
			is a raw string.	*/
			break;

		case NC_SHORT:
			sscanf(list[i], "%hd", ((short *) values) + i);
			break;

		case NC_LONG:
			sscanf(list[i], "%ld", ((long *) values) + i);
			break;

		case NC_FLOAT:
			sscanf(list[i], "%f", ((float *) values) + i);
			break;

		case NC_DOUBLE:
			sscanf(list[i], "%lf", ((double *) values) + i);
			break;
		default:
			return TCL_ERROR;
	   }
	}
	
	/*  The list has been converted, release back to the system.	*/
	free((char *) list);
			

	/*  Make the call.	*/
	status = ncattput(cid, varid, argv[3], type, ndims, values);

	/*  Free the values	*/
	if (type != NC_CHAR) {
		free((char *) values);
	}

	/*  Return value.	*/
	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	return TCL_OK;
}



int
tcl_ncattinq(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int	cid, status, varid;
	nc_type	type;
	int	len;
	char	ret_string[MAX_RETURN];

	if (argc != 4) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid varid name \"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }

	/*  If the variable id is not a number, assume that
	it is a request for a GLOBAL attribute put.	*/
	if (Tcl_GetInt(interp, argv[2], &varid) != TCL_OK) {
		varid = NC_GLOBAL;
        }


	/*  Make the call.	*/
	status = ncattinq(cid, varid, argv[3], &type, &len);

	/*  Return value.	*/
	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	switch(type)  {
		case NC_BYTE:
			sprintf(ret_string, "byte %d", len);
			break;

		case NC_CHAR:
			sprintf(ret_string, "char %d", len);
			break;

		case NC_SHORT:
			sprintf(ret_string, "short %d", len);
			break;

		case NC_LONG:
			sprintf(ret_string, "long %d", len);
			break;

		case NC_FLOAT:
			sprintf(ret_string, "float %d", len);
			break;

		case NC_DOUBLE:
			sprintf(ret_string, "double %d", len);
			break;
		default:
			return TCL_ERROR;
	}
	
	Tcl_SetResult(interp, ret_string, TCL_VOLATILE);
	return TCL_OK;
}



int
tcl_ncattget(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int	cid, status, varid;
	nc_type	type;
	int	len;
	char 	**attlist;
	void	*value;
	char	*list;
	int	i;


	if (argc != 4) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid varid name \"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }
	/*  If the variable id is not a number, assume that
	it is a request for a GLOBAL attribute put.	*/
	if (Tcl_GetInt(interp, argv[2], &varid) != TCL_OK) {
		varid = NC_GLOBAL;
        }
	

	/*  Get the type of the variable and perform the scan of
	the argument accordingly.	*/
	status = ncattinq(cid, varid, argv[3], &type, &len);

	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }

	/*  Make space for the attribute to come in.	*/
	/*  Include space for each to be turned into a string.	*/

	/*  We are building an argc like array for Tcl merge
	to use latter.	*/
	attlist = (char **) calloc(len, sizeof(char *));
	switch(type)  {
		case NC_BYTE:
			value = (void *) calloc(len, sizeof(char));
			break;
		case NC_CHAR:
			value = (void *) calloc(len, sizeof(char));
			break;
		case NC_SHORT:
			value = (void *) calloc(len, sizeof(short));
			break;
		case NC_LONG:
			value = (void *) calloc(len, sizeof(long));
			break;
		case NC_FLOAT:
			value = (void *) calloc(len, sizeof(float));
			break;
		case NC_DOUBLE:
			value = (void *) calloc(len, sizeof(double));
			break;
		default:
			return TCL_ERROR;
	}

	/*  We have the space.  Ask for the attribute and scan into
	strings.	*/
	status = ncattget(cid, varid, argv[3], value);

	/*  Return value.	*/
	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }


	/*  Tricky casting going on.  We have to interpret the
	void pointer as a pointer of the correct type.	*/

	for (i=0; i<len; i++) {
	   switch(type)  {
		case NC_BYTE:
		   *(attlist + i) = (char *) calloc(MAX_STRING, sizeof(char));
		   sprintf(*(attlist + i), "%c", *(((char *) value) + i));
		   break;
		case NC_CHAR:
		   /*  This is a tricky one.  Switch the length
	   	   to one and reassign the first member of the
		   attlist to point here.	*/
		   len = 1;
		   *(attlist) = (char *) value;
			break;
		case NC_SHORT:
		   *(attlist + i) = (char *) calloc(MAX_STRING, sizeof(char));
		   sprintf(*(attlist + i), "%hd", *(((short *) value) + i));
		   break;
		case NC_LONG:
		   *(attlist + i) = (char *) calloc(MAX_STRING, sizeof(char));
		   sprintf(*(attlist + i), "%ld", *(((long *) value) + i));
		   break;
		case NC_FLOAT:
		   *(attlist + i) = (char *) calloc(MAX_STRING, sizeof(char));
		   sprintf(*(attlist + i), "%f", *(((float *) value) + i));
		   break;
		case NC_DOUBLE:
		   *(attlist + i) = (char *) calloc(MAX_STRING, sizeof(char));
		   sprintf(*(attlist + i), "%lf", *(((double *) value) + i));
		   break;
		default:
			return TCL_ERROR;
	   }
	}

	
	/*  Ask for a merge and return that.	*/
	list = Tcl_Merge(len, attlist);

	/*  Free up the alocated space.	*/
	for (i=0; i<len; i++) {
		free(*(attlist +i));
	}

	/*  If this was a string, then the previous call already freed
	the memory.	*/
	if (type != NC_CHAR) {
		free((char *) attlist);
		free((char *) value);
	}
	
	
	Tcl_SetResult(interp, list, TCL_DYNAMIC);
	return TCL_OK;
}


int
tcl_ncattcopy(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int	status;
	int	incid, invar, outcid, outvar;
	
	if (argc != 6) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" incdf invar name outcdf outvar\"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &incid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (incid)");
                return TCL_ERROR;
        }
	if (Tcl_GetInt(interp, argv[2], &invar) != TCL_OK) {
		invar = NC_GLOBAL;
        }
	if (Tcl_GetInt(interp, argv[4], &outcid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (outcid)");
                return TCL_ERROR;
        }
	if (Tcl_GetInt(interp, argv[5], &outvar) != TCL_OK) {
		outvar = NC_GLOBAL;
        }

	status = ncattcopy(incid, invar, argv[3], outcid, outvar);

	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	Tcl_SetResult(interp, NULL, TCL_STATIC);
	return TCL_OK;
}


int
tcl_ncattname(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int	status;
	int	cid, varid, attnum;
	char	name[MAX_NC_NAME];
	
	if (argc != 4) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid varid attnum \"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }
	if (Tcl_GetInt(interp, argv[2], &varid) != TCL_OK) {
		varid = NC_GLOBAL;
        }
	if (Tcl_GetInt(interp, argv[3], &attnum) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (attnum)");
                return TCL_ERROR;
        }

	status = ncattname(cid, varid, attnum, name);

	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	Tcl_SetResult(interp, name, TCL_VOLATILE);
	return TCL_OK;
}


int
tcl_ncattrename(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int	status;
	int	cid, varid;
	
	if (argc != 5) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid varid name newname \"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }
	if (Tcl_GetInt(interp, argv[2], &varid) != TCL_OK) {
		varid = NC_GLOBAL;
        }

	status = ncattrename(cid, varid, argv[3], argv[4]);

	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	Tcl_SetResult(interp, NULL, TCL_STATIC);
	return TCL_OK;
}


int
tcl_ncattdel(ClientData cd, Tcl_Interp *interp, int argc, char *argv[])
{
	int	status;
	int	cid, varid;
	
	if (argc != 4) {
		Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			" cdfid varid name \"", (char *) NULL);
		return TCL_ERROR;
	}
	
	if (Tcl_GetInt(interp, argv[1], &cid) != TCL_OK) {
                Tcl_AddErrorInfo(interp,
                "\n     (cdfid)");
                return TCL_ERROR;
        }
	if (Tcl_GetInt(interp, argv[2], &varid) != TCL_OK) {
		varid = NC_GLOBAL;
        }

	status = ncattdel(cid, varid, argv[3]);

	if (status == -1) {
		Tcl_AppendResult(interp, "Error  ", argv[0],
                                (char *) NULL);
                return TCL_ERROR;
        }
 
	Tcl_SetResult(interp, NULL, TCL_STATIC);
	return TCL_OK;
}


