/*
 * http.c
 *
 * This file contains a simple access method for http documents. It
 * only accepts text/plain and text/html documents. This is based
 * on http Version 1.0.
 *
 * Below is a short syntactic description of what I would like
 * to have:
 *
 * http proxy <host[:port]>
 * http get <url>
 * http head <url>
 * http server <port>
 * 
 * http bind <get|put|post|link> <name> <proc>
 * or
 * http bind <callback> %F %N %A
 *
 * Copyright (c) 1994
 *
 * J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 *
 */

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <tcl.h>

#include "scotty.h"
#include "xmalloc.h"

static char *proxy = NULL;
static int proxyport = 80;
static char *http_version = "HTTP/1.0";

#define HTTP_OK		200
#define HTTP_CREATED	201
#define HTTP_ACCEPTED	202
#define HTTP_PARTIAL	203
#define HTTP_MOVED	301
#define HTTP_FOUND	302
#define HTTP_METHOD	303
#define HTTP_BADREQUEST	400
#define HTTP_UNAUTH	401
#define HTTP_PAYFORIT	402
#define HTTP_FORBIDDEN	403
#define HTTP_NOTFOUND	404
#define HTTP_INTERNAL	500
#define HTTP_NOTYET	501

struct http_errs {
    int errno;
    char *descr;
};

static struct http_errs errors[] = {
    { HTTP_OK,		"Document follows" },
    { HTTP_CREATED,	"Created" },
    { HTTP_ACCEPTED,	"Accepted" },
    { HTTP_PARTIAL,	"Partial Information" },
    { HTTP_MOVED,	"Moved" },
    { HTTP_FOUND,	"Found" },
    { HTTP_METHOD,	"Method" },
    { HTTP_BADREQUEST,	"Bad request" },
    { HTTP_UNAUTH,	"Unauthorized" },
    { HTTP_PAYFORIT,	"PaymentRequired" },
    { HTTP_FORBIDDEN,	"Forbidden" },
    { HTTP_NOTFOUND,	"Not found" },
    { HTTP_INTERNAL,	"Internal Error" },
    { HTTP_NOTYET,	"Not implemented" },
    { 0, NULL },
};

struct http_handle {
    int sock;
    Tcl_Interp *interp;
};

/*
 * Forward declarations for procedures defined later in this file:
 */

static struct hostent*
tcp_gethost	_ANSI_ARGS_((char *name));

static void
http_error	_ANSI_ARGS_((FILE *f, int errno));

static FILE*
http_connect	_ANSI_ARGS_((Tcl_Interp *interp, char **url));

static void
http_close	_ANSI_ARGS_((FILE *f));

static void
http_accept	_ANSI_ARGS_((ClientData clientData, int mask));

static int
http_rcvheader	_ANSI_ARGS_((Tcl_Interp *interp, FILE *f));

static int
http_rcvbody	_ANSI_ARGS_((Tcl_Interp *interp, FILE *f, char *varName));

static int
http_proxy	_ANSI_ARGS_((Tcl_Interp *interp, int argc, char **argv));

static int
http_get	_ANSI_ARGS_((Tcl_Interp *interp, int argc, char **argv));

static int
http_head	_ANSI_ARGS_((Tcl_Interp *interp, int argc, char **argv));

static int
http_server	_ANSI_ARGS_((Tcl_Interp *interp, int argc, char **argv));

/*
 * Get a pointer to a hostent structure. First try gethostbyname.
 * If this fails, try inet_addr and fake a hostent structure.
 */

static struct hostent *
tcp_gethost(name)
    char *name;
{
    struct hostent *hostent;
    static struct hostent _hostent;
    static int hostaddr, hostaddrPtr[2];
	    
    hostent = gethostbyname (name);
    if (hostent != NULL) return hostent;

    hostaddr = inet_addr (name);
    if (hostaddr == -1) return NULL;

    _hostent.h_addr_list = (char **) hostaddrPtr;
    _hostent.h_addr_list[0] = (char *) &hostaddr;
    _hostent.h_addr_list[1] = NULL;
    _hostent.h_length = sizeof(hostaddr);
    _hostent.h_addrtype = AF_INET;
    return &_hostent;
}

/*
 * Write an error message to the output stream.
 */

static void
http_error (f, err)
     FILE *f;
     int err;
{
    struct http_errs *he = errors;

    while (he->errno && he->errno != err) he++;

    if (he->errno) {
	fprintf (f, "%s %d %s\r\n", http_version, he->errno, he->descr);
    } else {
	fprintf (f, "%s 500 Unknown Internal Error\r\n", http_version);
    }
}

/*
 * Open a tcp connection to the host using the given port. Errors 
 * are kept in the interpreter and a NULL pointer is returned.
 */

static FILE *
http_connect (interp, url)
    Tcl_Interp *interp;
    char **url;
{
    char hbuf[1024];
    char *host;
    int port;
    int sock;
    FILE *f;
    struct hostent *hp;
    struct servent servent;
    struct sockaddr_in name;

    if (proxy) {
	host = proxy;
	port = proxyport;
    } else {
	char *p, *q;
	if (strlen (*url) >= 1024) {
	    Tcl_AppendResult (interp, "url too long", (char *) NULL);
	    return (FILE *) NULL;
	}
	strcpy (hbuf, *url);
	p = hbuf;
	if (strncmp (p, "http:", 5) == 0) p += 5;
	if (strncmp (p, "//", 2) == 0) p += 2;
	host = p;
	p = strchr(host, ':');
	if (p) {
	    *p = '\0';
	    q = strchr(p+1, '/');
	    if (q) *q = '\0';
	    if (Tcl_GetInt (interp, ++p, &port) != TCL_OK)
		    return (FILE *) NULL;
	    p = q;
	} else {
	    p = strchr(host, '/');
	    if (p) {
		*p = '\0';
		port = 80;
	    }
	}
	*url = *url + (p - hbuf);
    }
    
    hp = tcp_gethost (host);
    if (hp == NULL) {
	Tcl_AppendResult (interp, "no such host \"", host,
			  "\"", (char *) NULL);
	return (FILE *) NULL;
    }

    servent.s_port = htons (port);
    servent.s_proto = "tcp";

    sock = socket (PF_INET, SOCK_STREAM, 0);
    if (sock < 0) {
        Tcl_AppendResult (interp, "could not create socket: ", 
			  Tcl_PosixError (interp), (char *) NULL);
	return (FILE *) NULL;
    }
    
    memcpy ((char *) &name.sin_addr, (char *) hp->h_addr, hp->h_length);
    name.sin_family = AF_INET;
    name.sin_port = servent.s_port;
     
    if (connect (sock, (struct sockaddr *) &name, sizeof(name)) < 0) {
	Tcl_AppendResult (interp, "can not connect to host \"", host,
			  "\": ", Tcl_PosixError (interp), (char *) NULL);
	close (sock);
	return (FILE *) NULL;
    }
    
    if ((f = fdopen (sock, "w+")) == NULL) {
        Tcl_AppendResult (interp, "couldn't open file: ",
			  Tcl_PosixError (interp), (char *) NULL);
	close (sock);
	return (FILE *) NULL;
    }

    return f;
}

/*
 * Close a http file hanlde. Flushes all buffers, shutdown the tcp
 * connection and close the file handle.
 */

static void
http_close (f)
     FILE *f;
{
    fflush (f);
    shutdown (fileno (f), 2);
    fclose (f);
}

/*
 * Accept a http request. We just read the first bytes send from the
 * client to see what we are supposed to do.
 */

static void
http_accept (clientData, mask)
     ClientData clientData;
     int mask;
{
    struct http_handle *h = (struct http_handle *) clientData;
    struct sockaddr_in sockaddr;
    int len = sizeof sockaddr;
    int fd, rc;
    FILE *f;
    char buf[512];
    char *method = NULL, *name = NULL, *version = NULL, *p;

    if ((fd = accept (h->sock, (struct sockaddr *) &sockaddr, &len)) < 0) {
	return;
    }

    if ((f = fdopen (fd, "w+")) == NULL) {
        close (fd);
        return;
    }

    /* XXX why do we need this? Without it, the linux C library
       will hang because of some ugly lseeks !? */

#ifdef linux
    setbuf (f, NULL);
#endif

    if (fgets (buf, 512, f) == NULL) {
	goto done;
    }

    /* parse the message header - ugly and simple version */
    
    buf[len] = '\0';
    p = buf;
    while (*p) {
	
	while (*p && isspace(*p)) p++;
	for (method = p; *p && !isspace(*p); p++) ;
	if (*p) *p++ = '\0';
	
	while (*p && isspace(*p)) p++;
	for (name = p; *p && !isspace(*p); p++) ;
	if (*p) *p++ = '\0';
	
	while (*p && isspace(*p)) p++;
	for (version = p; *p && !isspace(*p); p++) ;
	if (*p) *p++ = '\0';
	
	while (*p && isspace(*p)) p++;
	
    }

    rewind (f);

    if (method == NULL || name == NULL) {
	http_error (f, HTTP_INTERNAL);
	goto done;
    }

    if (strcmp (method, "GET") != 0) {
	http_error (f, HTTP_NOTYET);
	goto done;
    }

    rc = Tcl_VarEval (h->interp, "httpproc ", method, " ", name, 
		      (char *) NULL);
    if (rc == TCL_OK) {
	http_error (f, HTTP_OK);
	fprintf (f, "MIME-Version: 1.0\r\n");
        fprintf (f, "Server: scotty/%s\r\n", SCOTTY_VERSION);
	fprintf (f, "Content-Type: text/plain\r\n");
	fprintf (f, "Content-Length: %d\r\n\r\n", strlen (h->interp->result));

	fprintf (f, h->interp->result);
    } else {
	http_error (f, HTTP_INTERNAL);
    }

  done:
    http_close (f);
}

/*
 * Receive the header returned from an http server.
 */

static int
http_rcvheader (interp, f)
    Tcl_Interp *interp;
    FILE *f;
{
    char buf[512];
    char *p;
    char *code;
    int len;

    if (fgets (buf, 512, f) == NULL) {
	Tcl_SetResult (interp, "connection closed by peer", TCL_STATIC);
	return TCL_ERROR;
    }
    
    len = strlen (buf);
    while (len > 0 && isspace(buf[len-1])) {
	len --;
	buf[len]='\0';
    }

    /* check the error code */

    for (p = buf; *p && !isspace(*p); p++) ;   /* skip the server id */
    while (*p && isspace(*p)) p++;

    code = p;
    while (*p && isdigit(*p)) p++;
    *p++ = '\0';

    if (atoi (code) != HTTP_OK) {
	Tcl_SetResult (interp, p, TCL_VOLATILE);
	return TCL_ERROR;
    }

    while (fgets(buf, 512, f) != NULL) {
	len = strlen (buf);
	while (len > 0 && isspace(buf[len-1])) {
	    len --;
	    buf[len]='\0';
	}
	if (len == 0) 
		return TCL_OK;
	Tcl_AppendElement (interp, buf);
    }

    return TCL_OK;
}

/*
 * Get the body of the message and write it to varName.
 */

static int
http_rcvbody (interp, f, varName)
    Tcl_Interp *interp;
    FILE *f;
    char *varName;
{
    char buf[512];

    while (fgets(buf, 512, f) != NULL) {
	if (Tcl_SetVar(interp, varName, buf, 
		       TCL_APPEND_VALUE | TCL_LEAVE_ERR_MSG) == NULL) 
		return TCL_ERROR;
    }

    return TCL_OK;
}

/*
 * Set the proxy host. A previously set host will be removed if
 * the argument to the proxy option is empty.
 */

static int
http_proxy (interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    if (argc < 2 && argc > 3) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " proxy host\"", (char *) NULL);
	return TCL_ERROR;
    }
    
    if (argc == 3) {
	if (strlen (argv[2]) == 0) {
	    if (proxy) {
		free (proxy);
		proxy = NULL;
		proxyport = 80;
	    }
	} else {
	    char *p = strchr(argv[2], ':');
	    if (p) {
		*p = '\0';
		if (Tcl_GetInt (interp, ++p, &proxyport) != TCL_OK)
			return TCL_ERROR;
		proxy = xstrdup (argv[2]);
	    } else {
		proxy = xstrdup (argv[2]);
		proxyport = 80;
	    }
	}
    }
    
    if (proxy) {
	Tcl_SetResult (interp, proxy, TCL_STATIC);
    }
    
    return TCL_OK;
}

/*
 *
 */

static int
http_head (interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    char *url;
    FILE *f;

    if (argc != 3) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " head url\"", (char *) NULL);
	return TCL_ERROR;
    }

    url = argv[2];

    if ((f = http_connect (interp, &url)) == NULL)
	return TCL_ERROR;

    fprintf (f, "HEAD %s %s\r\n", url, http_version);
    fputs ("Accept: text/plain; text/html\r\n", f);
    fputs ("Accept-Encoding: \r\n", f);
    fprintf (f, "User-Agent: scotty/%s\r\n", SCOTTY_VERSION);
    fputs ("\r\n", f);
    fflush (f);
    rewind (f);

    if (http_rcvheader (interp, f) != TCL_OK)
	    return TCL_ERROR;

    http_close (f);

    return TCL_OK;
}

/*
 *
 */

static int
http_get (interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    char *url;
    char *varName = (char *) NULL;
    FILE *f;

    if (argc != 4) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " get url varName\"", (char *) NULL);
	return TCL_ERROR;
    }

    url = argv[2];
    varName = argv[3];

    if ((f = http_connect (interp, &url)) == NULL)
	return TCL_ERROR;

    Tcl_UnsetVar (interp, varName, 0);

    fprintf (f, "GET %s %s\r\n", url, http_version);
    fputs ("Accept: text/plain; text/html\r\n", f);
    fputs ("Accept-Encoding: \r\n", f);
    fprintf (f, "User-Agent: scotty/%s \r\n", SCOTTY_VERSION);
    fputs ("\r\n", f);
    fflush (f);
    rewind (f);

    if (http_rcvheader (interp, f) != TCL_OK)
	    return TCL_ERROR;

    if (http_rcvbody (interp, f, varName) != TCL_OK)
            return TCL_ERROR;

    http_close (f);

    return TCL_OK;
}

/*
 * Become a http server. This command creates a listening socket
 * and registers a callback in the event loop to handle incoming
 * requests.
 */

static int
http_server (interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    static int sock = -1;
    static int port;
    struct sockaddr_in name;
    static struct http_handle h;

    if (argc < 2 && argc > 3) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " server ?port?\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (argc == 2) {
	if (sock > 0) {
	    sprintf (interp->result, "%d", port);
	}
	return TCL_OK;
    }

    if (sock > 0) {
	Tk_DeleteFileHandler (sock);
	close (sock);
	sock = -1;
    }

    if (Tcl_GetInt (interp, argv[2], &port) != TCL_OK) return TCL_ERROR;

    sock = socket (PF_INET, SOCK_STREAM, 0);
    if (sock < 0) {
        Tcl_AppendResult (interp, "could not create socket: ", 
			  Tcl_PosixError (interp), (char *) NULL);
        return TCL_ERROR;
    }
    
    name.sin_addr.s_addr = INADDR_ANY;
    name.sin_family = AF_INET;
    name.sin_port = htons (port);
     
    if (bind (sock, (struct sockaddr *) &name, sizeof(name)) < 0) {
	Tcl_AppendResult (interp, "can not listen on port \"", argv[2],
			  "\": ", Tcl_PosixError (interp), (char *) NULL);
	close (sock);
	sock = -1;
	return TCL_ERROR;
    }

    if (listen (sock, 5) < 0) {
	Tcl_AppendResult (interp, "can not listen on port \"", argv[2],
			  "\": ", Tcl_PosixError (interp), (char *) NULL);
	close (sock);
	sock = -1;
	return TCL_ERROR;
    }
    
    h.sock = sock;
    h.interp = interp;

    Tk_CreateFileHandler (sock, TK_READABLE, http_accept, (ClientData) &h);

    return TCL_OK;
}

/*
 * This is the tcp command as described in the scotty documentation.
 * It simply dispatches to the C functions implementing the options
 * understood by the tcp command.
 */

int
httpCmd (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int length;
    char c;

    if (argc < 2) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " option ?arg arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }

    c = argv[1][0];
    length = strlen (argv[1]);

    if ((c == 'p') && (strncmp(argv[1], "proxy", length) == 0)) {
        return http_proxy (interp, argc, argv);
    } else if ((c == 'g') && (strncmp(argv[1], "get", length) == 0)) {
        return http_get (interp, argc, argv);
    } else if ((c == 'h') && (strncmp(argv[1], "head", length) == 0)) {
        return http_head (interp, argc, argv);
    } else if ((c == 's') && (strncmp(argv[1], "server", length) == 0)) {
        return http_server (interp, argc, argv);
    }

    Tcl_AppendResult (interp, "bad option \"", argv[1], "\": should be ",
		      "get, head, proxy, or server",
		      (char *) NULL);
    return TCL_ERROR;
}
