/*
 * icmp.c
 *
 * Extend a tcl command interpreter an icmp command. This module depends
 * on ntping written by Erik Schoenfelder (schoenfr@ibr.cs.tu-bs.de).
 *
 * Copyright (c) 1993, 1994
 *
 * J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#ifndef DBMALLOC
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#endif
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#ifdef HAVE_SOCKETPAIR
#include <sys/socket.h>
#endif
#include <tcl.h>

#include "scotty.h"
#include "xmalloc.h"
#include "xread.h"

/*
 * The default filename where we will find the ntping binary. This
 * is normally overwritten in the Makefile.
 */

#ifndef NTPING
#define NTPING "/usr/local/bin/ntping"
#endif

#define BUF_INCR 1024

static int xv[2] = {-1, -1};      /* read / write fd used for ntping */
static int pid;                   /* the pid of the ntping process   */

static int timeout = 5;
static int retries = 2;
static int size = 64;
static int ttl = -1;
static int trace = -1;
static int delay = 0;

/*
 * Forward declarations for procedures defined later in this file:
 */

static int
icmpOpen	_ANSI_ARGS_((Tcl_Interp *interp));

/*
 * Open the connection to ntping.
 */

static int
icmpOpen (interp)
    Tcl_Interp *interp;
{
#ifdef HAVE_SOCKETPAIR
    if (socketpair (AF_UNIX, SOCK_STREAM, 0, xv) < 0) {
	xv[0] = xv[1] = -1;
	Tcl_AppendResult (interp, "cannot create socketpair: ", 
			  Tcl_PosixError (interp), (char *) NULL);
	return TCL_ERROR;
    }
#else
    int fd1 [2], fd2 [2];
    if (pipe (fd1) < 0 || pipe (fd2) < 0) {
        xv[0] = xv[1] = -1;
	Tcl_AppendResult (interp, "cannot create pipes: ", 
			  Tcl_PosixError (interp), (char *) NULL);
	return TCL_ERROR;
    }
#endif

    pid = fork();
    if (pid < 0) {
	xv[0] = xv[1] = -1;
	Tcl_AppendResult (interp, "fork failed: ",
			  Tcl_PosixError (interp), (char *) NULL);
	return TCL_ERROR;
    }

    if (pid == 0) {                                       /* child */

	int i;

#ifdef HAVE_SOCKETPAIR
	close (xv [0]);
	dup2  (xv [1], 0);
	dup2  (xv [1], 1);
#else
	dup2  (fd1 [0], 0);
	dup2  (fd2 [1], 1);
#endif
	for (i = 3; i < 64; i++) close (i);

	execl (NTPING, NTPING, "-b", (char *) 0);
	perror ("scotty: execl ntping");
	exit (1);
    } else {                                            /* parent */
#ifdef HAVE_SOCKETPAIR
	close (xv [1]);
	xv [1] = xv [0];
#else
	xv [0] = fd2 [0];			/* read fd from ntping */
	xv [1] = fd1 [1];			/* write fd to ntping */
#endif
    }

    return TCL_OK;
}

/*
 * Close the connection to ntping. This will make it finish its
 * loop waiting for commands from here.
 */

void
icmpClose ()
{
    if (xv[0] != -1) {

	close (xv[0]);
#ifndef HAVE_SOCKETPAIR
	close (xv[1]);
#endif
	xv[0] = -1;
	xv[1] = -1;

	kill (pid, SIGTERM);
	Tcl_ReapDetachedProcs();

    }
}

/* 
 * Extend a tcl command interpreter with an icmp command. The
 * result will be a list of hosts and their round trip times
 * including -1 for unreachable hosts. The icmp command will fail,
 * if there are wrong arguments. When using the time to live (ttl)
 * feature, the command will return the host that answerd the udp
 * packet together with the round trip time.
 */

int
icmpCmd (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int new_timeout = -1;
    int new_retries = -1;
    int new_size = -1;
    int new_ttl = -1;
    int new_trace = -1;
    int new_mask = 0;
    int new_tstamp = 0;
    int new_delay = -1;

    Tcl_DString dst;
    int i, buffer_size;
    char *buffer, *p;
    char ch;
    int rc;

    if (argc == 1) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " ?-retries n? ?-size n? ?-timeout n? ?-delay n?",
			  " ?-ttl n? ?-trace n? ?-mask? ?-timestamp?",
			  " host ?host ...?\"", (char *) NULL);
	return TCL_ERROR;
    }

    /* start ntping if not done yet */

    if ((xv[0] == -1) && (icmpOpen (interp) != TCL_OK)) {                 
	return TCL_ERROR;
    }

    /* parse the parameters */

    for (i = 1; i < argc; i++) {
	if (strcmp (argv[i], "-retries") == 0) {
	    if (i == argc-1) break;
	    if (Tcl_GetInt (interp, argv[++i], &new_retries) != TCL_OK)
	        return TCL_ERROR;
	    continue;
	} else 	if (strcmp (argv[i], "-size") == 0) {
	    if (i == argc-1) break;
	    if (Tcl_GetInt (interp, argv[++i], &new_size) != TCL_OK)
                return TCL_ERROR;
	    continue;
	} else 	if (strcmp (argv[i], "-timeout") == 0) {
	    if (i == argc-1) break;
	    if (Tcl_GetInt (interp, argv[++i], &new_timeout) != TCL_OK)
                return TCL_ERROR;
	    continue;
	} else  if (strcmp (argv[i], "-delay") == 0) {
	    if (i == argc-1) break;
	    if (Tcl_GetInt (interp, argv[++i], &new_delay) != TCL_OK)
                return TCL_ERROR;
	    continue;
	} else  if (strcmp (argv[i], "-ttl") == 0) {
	    if (i == argc-1) break;
	    if (Tcl_GetInt (interp, argv[++i], &new_ttl) != TCL_OK)
                return TCL_ERROR;
	    continue;
	} else  if (strcmp (argv[i], "-trace") == 0) {
	    if (i == argc-1) break;
	    if (Tcl_GetInt (interp, argv[++i], &new_trace) != TCL_OK)
                return TCL_ERROR;
	    continue;
	} else  if (strcmp (argv[i], "-mask") == 0) {
	    new_mask++;
	    continue;
	} else  if (strcmp (argv[i], "-timestamp") == 0) {
	    new_tstamp++;
	    continue;
	} else break;
    }

    /* set the static parameters if there are no hosts given */

    if (i == argc) {
	timeout = (new_timeout<0) ? timeout : new_timeout;
	retries = (new_retries<0) ? retries : new_retries;
	size    = ( new_size<0  ) ? size    : new_size;
	delay   = ( new_delay<0 ) ? delay   : new_delay;
	ttl     = (   ttl<0     ) ? ttl     : new_ttl;
	trace   = (  trace<0    ) ? trace   : new_trace;
	return TCL_OK;
    }
    
    /* tell ntping what we are interested in */

    buffer_size = BUF_INCR;
    buffer = xmalloc (buffer_size);

    Tcl_DStringInit (&dst);
    sprintf (buffer, "-t %d -r %d -d %d -s %d ", 
	     (new_timeout<0) ? timeout : new_timeout,
	     (new_retries<0) ? retries : new_retries,
	     (new_delay<0)   ? delay   : new_delay,
	     ( new_size<0  ) ? size    : new_size);
    Tcl_DStringAppend (&dst, buffer, -1);
    if (ttl > 0 || new_ttl > 0) {
	sprintf (buffer, "-ttl %d ", ( ttl > 0 ) ? ttl : new_ttl);
	Tcl_DStringAppend (&dst, buffer, -1);
    } else if (trace > 0 || new_trace > 0) {
	sprintf (buffer, "-trace %d ", ( trace > 0 ) ? trace : new_trace);
	Tcl_DStringAppend (&dst, buffer, -1);
    } if (new_mask > 0) {
	Tcl_DStringAppend (&dst, "-mask ", 6);
    } else if (new_tstamp > 0) {
	Tcl_DStringAppend (&dst, "-timestamp ", 11);
    }
    for (; i < argc; i++) Tcl_DStringAppendElement (&dst, argv[i]);
    Tcl_DStringAppend (&dst, "\n", 1);
    rc = xwrite (xv[1], Tcl_DStringValue (&dst), Tcl_DStringLength (&dst));
    Tcl_DStringFree (&dst);

    if (rc < 0) {
	free (buffer);
	Tcl_AppendResult (interp, "ntping: ", Tcl_PosixError (interp),
                          (char *) NULL);
        return TCL_ERROR;
    }

    p = buffer;
    while (xread (xv[0], &ch, 1) == 1 && (ch != '\n')) {
	*p = ch;
	p++;
	if (p-buffer == buffer_size) {
	    buffer = xrealloc (buffer, buffer_size + BUF_INCR);
	    p = buffer + buffer_size;
	    buffer_size += BUF_INCR;
	}
    }
    *p = '\0';

    if (ch != '\n') {
	free (buffer);
	Tcl_AppendResult (interp, "ntping: ", Tcl_PosixError (interp),
			  (char *) NULL);
	return TCL_ERROR;
    }

#ifdef DBMALLOC
    Tcl_SetResult (interp, buffer, TCL_VOLATILE);
    free (buffer);
#else
    Tcl_SetResult (interp, buffer, TCL_DYNAMIC);
#endif
    return TCL_OK;
}
