/*
 * job.c
 *
 * The simple job scheduler used to implement monitoring scripts.
 * This version is based on the original jobs scheduler written
 * in tcl by Stefan Schoek (schoek@ibr.cs.tu-bs.de).
 *
 * Copyright (c) 1994
 *
 * J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <string.h>
#include <ctype.h>
#include <sys/time.h>

#include <tcl.h>

#include "scotty.h"
#include "xmalloc.h"

typedef struct Job {
    char *id;                /* The id of the job (a unique number) */
    char *cmd;               /* The command to evaluate. */
    int interval;            /* The time interval value in ms. */
    int remtime;             /* The remaining time in ms. */
    int times;               /* The number of evals til the job expires. */
    int status;              /* The status of this job (see below) */
    struct Job *nextPtr;     /* Next event in queue. */
} Job;

static Job *jobList = NULL;

/* These are all possible stati of a job in the jobList */

#define SUSPEND 0
#define WAITING 1
#define RUNNING 2
#define EXPIRED 3

/* The last time the scheduler completed operation */

static struct timeval sched_time;

/* The pointer to the currently running job */

static Job *currentJob = NULL;

/*
 * Forward declarations for procedures defined later in this file:
 */

static void
event_callback   _ANSI_ARGS_((ClientData clientData));

static void 
job_nextschedule _ANSI_ARGS_((Tcl_Interp *interp, int time));

static void 
job_adjustTime   _ANSI_ARGS_((void));

static int
job_create       _ANSI_ARGS_((Tcl_Interp *interp, int argc, char **argv));

static int
job_list         _ANSI_ARGS_((Tcl_Interp *interp, int argc, char **argv));

static int
job_kill         _ANSI_ARGS_((Tcl_Interp *interp, int argc, char **argv));

static int
job_status       _ANSI_ARGS_((Tcl_Interp *interp, int argc, char **argv));

static int
job_command      _ANSI_ARGS_((Tcl_Interp *interp, int argc, char **argv));

static int
job_interval     _ANSI_ARGS_((Tcl_Interp *interp, int argc, char **argv));

static int
job_current      _ANSI_ARGS_((Tcl_Interp *interp, int argc, char **argv));

static int
job_schedule     _ANSI_ARGS_((Tcl_Interp *interp));

/*
 * Make sure job schedule gets called after time seconds.
 */

static void
event_callback (clientData)
    ClientData clientData;
{
    Tcl_Interp *interp = (Tcl_Interp *) clientData;

    job_schedule (interp);
}

static void
job_nextschedule (interp, time)
    Tcl_Interp *interp; 
    int time;
{
    /* The token for the event that calls the scheduler */

    static Tk_TimerToken scheduleEvent = NULL;

    if (scheduleEvent != NULL) {
	Tk_DeleteTimerHandler (scheduleEvent);
	scheduleEvent = NULL;
    }
    scheduleEvent = Tk_CreateTimerHandler (time, event_callback, 
					   (ClientData) interp);
}

/*
 * Update the remaining time of the jobs in the queue and set
 * sched_time to the current time.
 */

static void
job_adjustTime ()
{
    struct timeval time;
    int delta;
    Job *p;

    (void) gettimeofday(&time, (struct timezone *) NULL);

    delta = (time.tv_sec - sched_time.tv_sec) * 1000 
	    + (time.tv_usec - sched_time.tv_usec) / 1000;

    sched_time = time;

    for (p = jobList; p != NULL; p = p->nextPtr) {
        if (p->status != SUSPEND) 
		p->remtime -= delta;
    }
}

/*
 * Create a new job and put it in the jobList.
 */

static int
job_create (interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    static unsigned lastid = 0;
    int interval;
    int times = 0;
    int time;
    Job *job, *p;
    char buf[20];

    if (argc < 4 || argc > 5) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " create command interval ?times?\"", 
			  (char *) NULL);        
        return TCL_ERROR;
    }

    if (Tcl_GetInt (interp, argv[3], &interval) != TCL_OK)
        return TCL_ERROR;

    if ((argc > 4) && (Tcl_GetInt (interp, argv[4], &times) != TCL_OK))
        return TCL_ERROR;

    sprintf (buf, "job%d", lastid++);
    job = (Job *) xmalloc(sizeof(Job));
    job->id = xstrdup (buf);
    job->cmd = xstrdup (argv[2]);
    job->interval = interval;
    job->remtime = 0;
    job->times = times;
    job->status = WAITING;
    job->nextPtr = jobList;
    jobList = job;

    time = -1;
    for (p = jobList; p != NULL; p = p->nextPtr) {
        if (p->status != SUSPEND) {
	    if (time < 0 || p->remtime < time) 
		    time = (p->remtime < 0) ? 0 : p->remtime;
	}
    }

    /* tell the event manager to call us again if needed */

    if (time >= 0) job_nextschedule (interp, time);

    interp->result = job->id;
    return TCL_OK;
}

/*
 * Get a job list. For each job, list the job id, its command
 * the interval and the remaining time, the number or repeations
 * and the job status.
 */

static int
job_list (interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Job *p;
    Tcl_DString dst;
    char buf[20];

    if (argc != 2) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " list\"", (char *) NULL);        
        return TCL_ERROR;
    }

    job_adjustTime ();

    Tcl_DStringInit (&dst);
    
    for (p = jobList; p != NULL; p = p->nextPtr) {

        Tcl_DStringStartSublist (&dst);
	Tcl_DStringAppendElement (&dst, p->id);
	Tcl_DStringAppendElement (&dst, p->cmd);
	sprintf (buf, "%d", p->interval);
	Tcl_DStringAppendElement (&dst, buf);
	sprintf (buf, "%d", p->remtime);
	Tcl_DStringAppendElement (&dst, buf);
	sprintf (buf, "%d", p->times);
	Tcl_DStringAppendElement (&dst, buf);

	switch (p->status) {
        case SUSPEND: 
	    Tcl_DStringAppendElement (&dst, "suspend");
	    break;
	case WAITING:
	    Tcl_DStringAppendElement (&dst, "waiting");
	    break;
	case RUNNING:
	    Tcl_DStringAppendElement (&dst, "running");
	    break;
	case EXPIRED:
	    Tcl_DStringAppendElement (&dst, "expired");
	    break;
	}
	Tcl_DStringEndSublist (&dst);
    }

    Tcl_DStringResult (interp, &dst);    

    return TCL_OK;
}

/*
 * Search for a job and remove it from the job list.
 */

static int
job_kill (interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int i;
    Job *p, *q;

    for (i = 2; i < argc; i++) {

	if (jobList != NULL) {

  	    for (p = jobList, q = NULL; p != NULL; q = p, p = p->nextPtr) {
	        if (strcmp(p->id, argv[i]) == 0) break;
	    }

	    if ((p != NULL) && (strcmp(p->id, argv[i]) == 0)) {

		if (p == currentJob) {

		    currentJob->status = EXPIRED;      /* mark job as dead */

		} else {

		    if (q == NULL) {
			jobList = p->nextPtr;
		    } else {
			q->nextPtr = p->nextPtr;
		    }
		    free (p->id);
		    free (p->cmd);
		    free ((char *) p);
		}
	    }
	}
    }

    return TCL_OK;
}

/*
 * Modify the job status (suspend or waiting/running).
 */

static int
job_status (interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Job *p;
    int status = WAITING;

    if (argc != 3 && argc != 4) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " status job ?value?\"", (char *) NULL);
        return TCL_ERROR;
    }

    if (argc == 4) {
	if (strcmp (argv[3], "suspend") == 0) {
	    status = SUSPEND;
	} else if (strcmp (argv[3], "waiting") == 0) {
	    status = WAITING;
	} else if (strcmp (argv[3], "running") == 0) {
	    /* No typo! A user can not switch to another job. */
	    status = WAITING;
	} else {
	    Tcl_AppendResult (interp, "unknown status \"", 
			      argv[3], "\"", (char *) NULL);
	    return TCL_ERROR;
	}
    }

    job_adjustTime ();

    for (p = jobList; p != NULL; p = p->nextPtr) {
	if (strcmp(p->id, argv[2]) == 0) {
	    if (argc == 4) p->status = status;
	    break;
	}
    }

    if (p == NULL) {
	Tcl_AppendResult (interp, "no job \"", argv[2], "\"", (char *) NULL);
	return TCL_ERROR;
    }

    job_schedule (interp);
    
    switch (p->status) {
    case SUSPEND: 
	interp->result = "suspend";
	break;
    case WAITING:
	interp->result = "waiting";
	break;
    case RUNNING:
	interp->result = "running";
	break;
    case EXPIRED:
	interp->result = "expired";
	break;
    }

    return TCL_OK;
}

/*
 * Modify the command of a job.
 */

static int
job_command (interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Job *p = NULL;

    if (argc != 3 && argc != 4) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " command job ?value?\"", (char *) NULL);
        return TCL_ERROR;
    }

    for (p = jobList; p != NULL; p = p->nextPtr) {
	if (strcmp(p->id, argv[2]) == 0) {
	    if (argc == 4) {
		free (p->cmd);
		p->cmd = xstrdup (argv[3]);
	    }
	    break;
	}
    }

    if (p == NULL) {
	Tcl_AppendResult (interp, "no job \"", argv[2], "\"", (char *) NULL);
	return TCL_ERROR;
    }
    
    interp->result = p->cmd;
    return TCL_OK;
}

/*
 * Modify the interval between two invocations.
 */

static int
job_interval (interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Job *p = NULL;
    int interval;

    if (argc != 3 && argc != 4) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " interval job ?value?\"", (char *) NULL);
        return TCL_ERROR;
    }

    if (argc == 4) {
	if (Tcl_GetInt (interp, argv[3], &interval) != TCL_OK)
		return TCL_ERROR;
    }

    for (p = jobList; p != NULL; p = p->nextPtr) {
	if (strcmp(p->id, argv[2]) == 0) {
	    if (argc == 4) {
		p->interval = interval;
	    }
	    break;
	}
    }

    if (p == NULL) {
	Tcl_AppendResult (interp, "no job \"", argv[2], "\"", (char *) NULL);
	return TCL_ERROR;
    }
    
    sprintf (interp->result, "%d", p->interval);
    return TCL_OK;
}

/*
 * Return the id of the current running job or just an empty string.
 */

static int
job_current (interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    if (argc != 2) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
                          " current\"", (char *) NULL);
        return TCL_ERROR;
    }

    if (currentJob != NULL) 
	    Tcl_SetResult (interp, currentJob->id, TCL_VOLATILE);

    return TCL_OK;
}

/*
 * This function checks for jobs that must be invoked. It finally
 * tells the event mechanism to repeat itself when the next job
 * needs attention.
 */

static int
job_schedule (interp)
    Tcl_Interp *interp;
{
    static int active = 0;
    int time;
    Job *p, *q;

    if (!active) {
	active++;
	(void) gettimeofday (&sched_time, (struct timezone *) NULL);
    }

    /* Flush the queue of pending ined commands */

    inedFlush (interp);

    /* refresh the remaining time of the active jobs */

    job_adjustTime ();
        
    /* subtract the delta from the time of all not suspended jobs */
    /* and execute jobs with remaining time less or equal 0 */

    for (p = jobList; p != NULL; p = p->nextPtr) {

	if ((p->status == WAITING) && (p->remtime <= 0)) {

	  currentJob = p;
	  p->status = RUNNING;
	  Tcl_GlobalEval (interp, p->cmd);
	  Tcl_ResetResult (interp);
	  if (p->status == RUNNING) p->status = WAITING;
	  currentJob = NULL;

	  p->remtime = p->interval;
	  if (p->times > 0) {
	      p->times--; 
	      if (p->times == 0) p->status = EXPIRED;    /* mark job as dead */
	  }
	}
    }

    /* delete all jobs which have expired */

    while (1) {

      for (p = jobList, q = NULL; p != NULL; q = p, p = p->nextPtr) {
          if (p->status == EXPIRED) break;
      }

      if (p == NULL) break;
      
      if (p->status == EXPIRED) {
          if (q == NULL) {
	      jobList = p->nextPtr;
	  } else {
	      q->nextPtr = p->nextPtr;
	  }
	  free (p->id);
	  free (p->cmd);
	  free ((char *) p);
      }
    }
    
    /* compute the time needed to execute the jobs */

    job_adjustTime ();
        
    /* subtract the time from all not suspended jobs */

    time = -1;
    for (p = jobList; p != NULL; p = p->nextPtr) {
        if (p->status != SUSPEND) {
	    if (time < 0 || p->remtime < time) 
		    time = (p->remtime < 0) ? 0 : p->remtime;
	}
    }
    
    /* tell the event manager to call us again if needed */

    if (time < 0) {
        active = 0;
    } else {
        job_nextschedule (interp, time);
    }

    return TCL_OK;
}

/*
 * This is the job command as described in the tkined documentation.
 * It is used to create, delete and modify jobs of the job scheduler.
 */

int
jobCmd (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int length;
    char c;

    if (argc < 2) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " option ?arg arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }

    c = argv[1][0];
    length = strlen (argv[1]);

    if ((c == 'c') && (strncmp(argv[1], "create", length) == 0)) {
	return job_create (interp, argc, argv);
    } else if ((c == 'l') && (strncmp(argv[1], "list", length) == 0)) {
        return job_list (interp, argc, argv);
    } else if ((c == 'k') && (strncmp(argv[1], "kill", length) == 0)) {
        return job_kill (interp, argc, argv);
    } else if ((c == 's') && (strncmp(argv[1], "status", length) == 0)) {
        return job_status (interp, argc, argv);
    } else if ((c == 'c') && (strncmp(argv[1], "command", length) == 0)) {
        return job_command (interp, argc, argv);
    } else if ((c == 'i') && (strncmp(argv[1], "interval", length) == 0)) {
        return job_interval (interp, argc, argv);
    } else if ((c == 'c') && (strncmp(argv[1], "current", length) == 0)) {
        return job_current (interp, argc, argv);
    } else if ((c == 's') && (strncmp(argv[1], "schedule", length) == 0)) {
        return job_schedule (interp);
    }

    Tcl_AppendResult (interp, "bad option \"", argv[1], 
		      "\": should be create, list, kill, schedule,",
		      " status, command, or interval", (char *) NULL);
    return TCL_ERROR;
}
