#! /usr/local/bin/scotty -inf
##
## monitordaemon utility for the [TK]INED editor.
##
## Copyright (c) 1993 St. Schoek
##                    TU Braunschweig, Germany
##                    Institute for Operating Systems and Computer Networks
##
## Permission to use, copy, modify, and distribute this
## software and its documentation for any purpose and without
## fee is hereby granted, provided that this copyright
## notice appears in all copies.  The University of Braunschweig
## makes no representations about the suitability of this
## software for any purpose.  It is provided "as is" without
## express or implied warranty.
##

LoadDefaults netguard

IpInit Netguard-Client

##
## These are the global parameters. The port number and the monitor_name are
## used to connect to our netguard-server.  
##

if {[info exists default(port)]} {
    set port $default(port)
} else {
    set port 6341
}
if {[info exists default(server)]} {
    set monitor_name $default(server)
} else {
    set monitor_name sol.ibr.cs.tu-bs.de
}


##
## global variables used by the procedures 
##

set menus            ""
set to_socket        ""
set sema             ""
set socket           ""

##
## writes errormsg to stderr
##

proc scottyerror { msg } {
    global errorInfo

    catch {puts stderr "$msg\n$errorInfo"; flush stderr}
}

##
## convert datestring in readable string
##

proc date_convert {date} {
    
    ## date == year.month.day in "day month year"
    set date [split $date .]
    return "[lindex $date 2] [lindex $date 1] [lindex $date 0]"  
}

##
## the following procedures are used to build up the connection
## to the monitorserver
##
## proc tries to connect automatically the monitoring server
## when starting tkined
## deletes the menu if connect failed 10 times
##

proc auto_connect {} {
    global port
    global socket
    global monitor_name
    global sema

    set socket ""
    set count  0

    while {$socket == ""} {
	if {$count >= 3} {
	    set answer [ined request \
			"Connect to $monitor_name $port failed $count times." \
			    [list [list "Host:" $monitor_name] \
			     [list "Port:" $port] ] \
			    [list cancel connect] ]
	    if {[lindex $answer 0] == "cancel"} {
		"Delete Netguard-Client" ""
		set socket ""
		set count 3
	    } else {
		set monitor_name [lindex $answer 1]
		set port  [lindex $answer 2]
		if {[catch {tcp connect $monitor_name $port} socket]} {
		    ined acknowledge "Connect to $monitor_name $port failed."
		    "Delete Netguard-Client" ""
		    set socket ""
		    set count 3
		} else {
		    break
		}
	    }
	}
	if {[catch {tcp connect $monitor_name $port} socket]} {
	    catch {writeln "$socket"}
	    set socket ""
	    sleep 5
	}
	incr count
    }
    ## create a job that read the monitor socket and
    ## checks answer from monitor_name

    addinput -read $socket read_socket

    ## create the new menu
    ##

    menu_close

    # here we must set accept_job to 1 because we do not write anything
    # with write_socket until we have got something from monitor
    set sema 1
}

##
## give message about the connect command to monitor
##

proc accept_message {} {
    global to_socket

    ## tell the monitor your name
    if {[catch {exec whoami} name]} {
        set name ""
    }
    lappend to_socket "name $name"
    ## create a job that writes all messages to the monitor socket
    job create write_socket 1000
}

##
## proc asks the monitor after the connection is created about
## the current variables and interfaces measured by the measurement-agents
## of the monitor
##

proc insert_data_now {var_arg} {
    global data_info_now
    
    if {$var_arg == ""} {
	writeln "server do not give result"
	return
    }
    set data_info_now $var_arg
}

##
## Become a slave. Connect to a server
##

proc "Try connect" {list} {
    global monitor_name
    global port
    global socket
    global sema

    if {$socket != ""} {
	# connection already exists
	writeln "Connection to monitor $monitor_name is already open"
	return 
    }
    set result [ined request "Choose hostname and port number of server:" \
		[list [list "Host:" $monitor_name] \
		      [list "Port:" $port] ] \
		[list cancel connect] ]
    if {[lindex $result 0] == "cancel"} return
    set monitor_name [lindex $result 1]
    set port         [lindex $result 2]
    if {[catch {tcp connect $monitor_name $port} socket]} {
	writeln "Can not connect to $monitor_name: $socket"
	set socket ""
	return
    }

    addinput -read $socket read_socket

    menu_close

    # here we must set accept_job to 1 because we do not write anything
    # with write_socket until we have got something from monitor
    set sema 1
}

##
## menu close the connection to the monitor
##

proc "Close connection" {list} {
    global to_socket
    global socket
    
    if {$socket == ""} { 
	ined acknowledge "No connection to close"
	return
    }
    # send close_command to monitor
    set to_socket [linsert $to_socket 0 "close_connect slave"]
}

##
## close the connection to the monitor
##

proc close_connection {} { 
    global data_info_now
    global var_oid_list
    global var_request_list
    global monitor_name
    global socket
    global port
    global to_socket
    global menus

    if {[info exists var_request_list]} {
	catch {puts $socket "data_reset_tag"; flush $socket}
	catch {puts $socket "limit_delete_req all"; flush $socket}
	catch {unset var_oid_list}
	catch {unset data_info_now}
	catch {unset var_request_list}
    }
    catch {removeinput $socket}
    catch {tcp shutdown $socket}
    catch {tcp close $socket}
    set socket ""
    set to_socket ""
    writeln "Connection closed by user at [exec date]."
    job create dummy 10000
    
    menu_connect
}

##
## let the interpreter stay alive
##

proc dummy {} {
    global socket
    
    if {$socket == ""} { return }
    job_kill [job current]
}

##
## the next procedures are used to get information about the status and
## configuration of the monitor server
## e.g. connections, data request, measured variables
##

##
## proc request monitor for a list of all states of the 
## measurementagents served by monitor
## asks for the variables measured by the agents
##

proc "Info agents" {list} {
    global socket
    global to_socket
    global monitor_name
    
    if {$socket == ""} {
	ined acknowledge "No connection to $monitor_name"
	return
    }  
    lappend to_socket "info tools"
}

##
## proc request monitor for the current states of some global variables
##

proc "Info server" {list} {
    global socket
    global to_socket
    global monitor_name
    
    if {$socket == ""} {
	ined acknowledge "No connection to $monitor_name"
	return
    }  
    lappend to_socket "info monitor"
}

##
## proc writes the answers in a log window from tkined
##

proc show_info {answer_arg}  {
    global to_socket
    global monitor_name

    if {$answer_arg == ""} {
	writeln "server do not give result"
	return
    } else { 
	foreach elem $answer_arg {
	    writeln $elem
	}
    }
}

##
## the next procedures are for data requests 
## there are two kind of requests: 
##      1. requests for current data (monitored and aggregated in the
##         last minute before showing in tkined)
##      2. requests for old data    (monitored, aggregated from the
##         date of request to the "begining of the first start of the monitor"
##         statistical data is offered from the "start" to the day before
##         the request)

##
## procedures for an current data request
##
##
## proc asks monitor to send current measured data
## creates a job that build up the stripcharts in tkined
## 

proc "Request current data" {list} {
    global socket
    global new_data_choose
    global data_info_now
    global data_request_now
    global monitor_name
    
    if {$socket == ""} {
	ined acknowledge "No connection to $monitor_name"
	return
    }
    if {![info exists data_info_now]} {
	writeln "server has not sent variable information"
	return
    }
    if {![info exists new_data_choose]} {
	set host *
	set variables *
    } else { 
	set host [list [lindex $new_data_choose 0]]
	set variables [list [lindex $new_data_choose 1]]
    }

    set request [ined request \
		 "Choose host and variable for displaying its data:" \
		 [list [list "Host:" $host] [list "Variables:" $variables] ] \
		 [list cancel ok] ]
    
    if {[lindex $request 0] == "cancel"} return

    set new_data_choose [list [lindex $request 1] [lindex $request 2]]
    set host     [lindex $new_data_choose 0]
    set variables [lindex $new_data_choose 1]

    foreach req_host $host {
	foreach elem $data_info_now {
	    if {![string match $req_host [lindex $elem 0]]} { continue }
	    foreach req_var $variables {
		foreach var_elem [lindex $elem 1] {
		    if {![string match $req_var [lindex $var_elem 0]]} {
			continue 
		    }
		    lappend req_result [format "%10s %15s %3s"  \
				       [lindex $elem 0] [lindex $var_elem 0] \
                                       [lindex $var_elem 1]]
		}		
	    }
	}
    }
    if {![info exists req_result]} { 
	ined acknowledge "No match for \"$host\" and \"$variables\"."
	return
    }
    set header "Select hosts and variables for displaying its data." 
    set res 1
    set request_now ""
    while {$res != ""} {
	set res [ined list $header $req_result [list cancel ok]]
	if {[lsearch $request_now [lindex $res 1]] == -1} {
	    if {[lindex $res 0] != "cancel"} {
		lappend request_now [lindex $res 1]
	    }
	}
    }
    if {$request_now == ""} { return }
    set i 0
    set var_ids ""
    foreach elem $request_now {
	set elem [lindex $elem 0]
	set host [lindex $elem 0]
	set var [lindex $elem 1]
	foreach elem_now $data_info_now { 
	    if {!([lindex $elem_now 0] == $host)} { continue }
	    foreach var_elem_now [lindex $elem_now 1]  {
		if {!([lindex $var_elem_now 0] == $var)} { continue }
		lappend var_ids "$host $var [lindex $var_elem_now 1] \
                                            [lindex $var_elem_now 2]"
	    }
	}
    }
    create_icons $var_ids
}

##
## proc creates stripcharts for every variable in arg and send a data 
## request now to monitor uses a simple layout algo to arrange the stripcharts
##

proc create_icons {var_arg} {
    global x
    global y
    global var_oid_list
    global var_request_list
    global to_socket
        
    set vars ""
    set scale_factor 100 

    foreach var $var_arg {
	set quest ""
	set host [lindex $var 0]
	set varname [lindex $var 1]
	set periode  [lindex $var 2]
	set var_id [lindex $var 3]
	if {[info exists var_request_list]} {
	    if {[lsearch $var_request_list $var_id] != -1} { continue }
	} else {
	    set x 100
	    set y 0
	    job create check_oid 5000
	}
	if {$y == 360} { 
	    set y 60; set x [expr {$x + 150}]
	} else {
	    set y [expr {$y + 60}]
	}
	if {![info exists var_oid_list($var_id)]} {
	    set var_oid_list($var_id) [ined create STRIPCHART]
	    ined -noupdate move $var_oid_list($var_id) $x $y
	    set label_name "$host $varname $periode"
	    ined -noupdate name $var_oid_list($var_id) $label_name
	    ined -noupdate label $var_oid_list($var_id) name
	    lappend var_request_list $var_id
	    lappend vars $var_id
	    lappend quest "scale factor [format "%12s %12s %3d" $host \
                           $varname $periode]:"
	    lappend question "$quest $scale_factor scale 1 100000"
	}
    }
    if {![info exists question]} { return }

    set scaling [ined request "Set initial scale factor for stripcharts:" \
		 $question [list cancel ok] ]
    
    if {[lindex $scaling 0] == "cancel"} {
	foreach var_id $vars {
	    set var_request_list [ldelete var_request_list $var_id]
	    ined -noupdate delete $var_oid_list($var_id)
	    catch {unset var_oid_list($var_id)}
	}
	return
    }

    set scaling [lrange $scaling 1 end]
    set index 0
    foreach var_id $vars {
	ined -noupdate scale $var_oid_list($var_id) [lindex $scaling $index]
	incr index	
    }

    if {$vars != ""} {
	lappend result $vars
	if {$result == ""} { return }
	lappend to_socket "data_insert_tag $result"
    }
}

##
## proc is called when a result about current data is sent from the monitor
## writes the values to the stripcharts (tkined)
## checks if stripcharts are deleted, if so, a data reset about the variable
## corresponding to the stripchart is sent to the monitor
##

proc show_actual_data {value_list} {
    global var_oid_list
    global var_request_list
    global to_socket

    set delete ""
    if {![info exists var_oid_list]} { 
	lappend to_socket "data_reset_tag"
	return
    }
    set value_list [lindex $value_list 1]
    if {$value_list == ""} { return }

    foreach value_elem $value_list {
	set var_id [lindex $value_elem 0]
	set value [lindex $value_elem 1]
	if {![info exists var_request_list]} { break }
	if {![info exists var_oid_list($var_id)]} { 
	    lappend delete $var_id
	    set var_request_list [ldelete var_request_list $var_id]
	    continue
	} 
	if {[ined retrieve $var_oid_list($var_id)] != ""} {
	    ined -noupdate values $var_oid_list($var_id) $value
	} else {
	    unset var_oid_list($var_id)
	    set var_request_list [ldelete var_request_list $var_id]
	    if {$var_request_list == ""} {
		catch {unset var_request_list}
	    }
	    lappend delete $var_id
	}
    }
    if {$delete != ""} {
	lappend result $delete
	lappend to_socket "data_delete_tag $result"
    }
}

##
## proc checks, if all NODE-objects (stripcharts) are still on the canvas
## if there are some deleted sent a data reset about the corresponding
## variable to the monitor
##

proc check_oid {} {
    global var_oid_list
    global var_request_list
    global to_socket
    
    if {![info exists var_request_list]} {
	job_kill [job current]
	return
    }
    foreach var_id $var_request_list {
    	if {[ined retrieve $var_oid_list($var_id)] == ""} {
	    lappend delete $var_id
	    set var_request_list [ldelete var_request_list $var_id]
	    catch {unset var_oid_list($var_id)}
	}
    }
    if {$var_request_list == ""} {
	catch {unset var_request_list}
	catch {unset x}
	catch {unset y}
	lappend to_socket "data_reset_tag"
	return
    }
    if {![info exists delete]} { return }
    lappend to_delete $delete
    lappend to_socket "data_delete_tag $to_delete"
}

##
## procedures for an old data request
##
##
## proc asks for interfaces from which data was measured and could be shown
##

proc "Request statistic data" {arg} {
    global old_data_day
    global to_socket
    global socket
    global monitor_name
    global old_data_sema
    global old_data_choose

    if {$socket == ""} {
	ined acknowledge "No connection to $monitor_name"
	return
    }
    if {[info exists old_data_sema]} {
        ined acknowledge "Function not executable! Waiting for other request"
        return
    }
    set month_list "Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec"
    if {![info exists old_data_day]} {
	set time         [exec date]
	set day          [expr {[lindex $time 2] - 1}]
	set month_string [lindex $time 1]
	set month     [expr {[lsearch $month_list $month_string] + 1}]
	set year      [lindex $time [expr {[llength $time] - 1}]]
	if {!$day} { 
	    ## it is the begining of a new month, 
	    ## set day to the last day of the month before
	    set month [expr {$month - 1}]
	    if {!$month} { 
		## we are changing a year
		set day 31
		set month 12
		set year [expr {$year - 1}]
	    } else {
		set day [calc_day $month $year]
	    }
	}
    } else {
	set year  [lindex [split $old_data_day .] 0]
	set month [expr {[lsearch $month_list [lindex [split \
                  $old_data_day .] 1]] + 1}]
	set day   [lindex [split $old_data_day .] 2]
    }
    if {![info exists old_data_choose]} {
	set host      *
	set variables *
	set statistik_periode 1-hour
    } else {
	set host      [list [lindex $old_data_choose 0]]
	set variables [list [lindex $old_data_choose 1]]
	set statistik_periode [lindex $old_data_choose 2] 
    }

    set request [ined request \
		 "Choose day, host and variable for statistic data:" \
		     [list [list "Day:" $day   scale 1 31] \
                           [list "Month:"     $month scale 1 12] \
                           [list "Year:"      $year  scale 1993 2001] \
                           [list "Host:"      $host] \
                           [list "Variables:" $variables] \
                           [list "Periode:"   $statistik_periode radio\
                                 1-min 15-min 1-hour 24-hour] ] \
		     [list cancel ok] ]

    if {[lindex $request 0] == "cancel"} return

    set day [lindex $request 1]
    set index [expr {[lindex $request 2] - 1}]
    set month [lindex $month_list $index]
    set year [lindex $request 3]

    set old_data_choose [list [lindex $request 4] [lindex $request 5] \
                              [lindex $request 6]]
    if {[llength $old_data_choose] != 3} { 
	set old_data_choose "* * 1-hour"  
	return 
    }
    if {([catch {lindex $old_data_choose 0} result]) || \
        ([catch {lindex $old_data_choose 1} result])} {
	writeln "syntax error in inputstring: $result"
	ined acknowledge "syntax error in inputstring: $result"
	return
    }

    set old_data_day "$year.$month.$day"
    lappend to_socket "data_info_old $old_data_day"
    set old_data_sema 1
}

##
## proc returns a new day
##

proc calc_day {month year} {

    case $month in {
	{1 3 5 7 8 10 12} {
	    return 31
	}
	{4 6 9 11} {
	    return 30
	}
	{2}  {
	    case $year in {
		{1992 1996 2000} {
		    return 29
		}
		{default} {
		    return 28
		}
	    }
	}
    }
}

##
## proc asks the user to selected interfaces and variables for the old data
## request
##

proc request_data_old {varlist} {
    global old_data_specifier
    global old_data_vars
    global old_data_day
    global old_data_sema
    global old_data_choose
    global to_socket
  
    if {$varlist == ""} { 
	catch {unset old_data_sema}
	return 
    }

    set hosts   [lindex $old_data_choose 0]
    set old_var [lindex $old_data_choose 1]

    foreach elem $varlist {
	set found 0
	foreach var_expr $old_var {
	    if {[string match $var_expr [lindex $elem 1]]} { 
		set found 1		
		break
	    }
	}
	if {!$found} { continue }
	foreach host_expr $hosts {
	    if {[string match $host_expr [lindex $elem 0]]} {
		set tmp [format "%10s %15s %3s %s" [lindex $elem 0] \
                        [lindex $elem 1] [lindex $elem 3] [lindex $elem 4]]
		lappend request_old_tmp $tmp
		lappend request_list $elem
		break
	    }
	}
    }
    if {![info exists request_old_tmp]} { 
	ined acknowledge "No match for \"$hosts\" and \"$old_var\"."
	catch {unset old_data_sema}
	return
    }
    set date [split $old_data_day .]
    set request_day [date_convert $old_data_day]
    set header "Select statistic variable at $request_day for display."
    set request_old ""
    set res 1
    while {$res != ""} {
	set res [ined list $header $request_old_tmp [list cancel ok]]
	if {([lindex $res 0] == "cancel") && ($request_old == "")} { 
	    ined acknowledge "No selection for \"$hosts\" and \"$old_var\"." 
	    catch {unset old_data_sema}
	    return 
	}
	if {[lsearch $request_old [lindex $res 1]] == -1} {
	    if {[lindex $res 0] != "cancel" } { 
		lappend request_old [lindex $res 1]
	    }
	}
    }
    set periode [lindex $old_data_choose 2]

    set tmp ""

    foreach elem $request_old {
	set elem [lindex $elem 0]
	set index [lsearch $request_old_tmp $elem]
	lappend tmp [list [lindex $request_list $index]]
    }
    set request_old $tmp

    if {![info exists old_data_specifier]} {
	if {$periode == "1-min"} {
	    set kind_data raw
	} else {
	    set kind_data 1-min-peak 
	}
	set report    no
	set draw      at-once
    } else {
	if {($periode != "1-min") && \
            ([lindex $old_data_specifier 1] != "raw")} {
	    set kind_data [lindex $old_data_specifier 1]
	} else {
	    set kind_data 1-min-peak 
	}
	set report    [lindex $old_data_specifier 3]
	set draw      [lindex $old_data_specifier 4]
    }

    case $periode in {
	{1-min} {
	    set kind_data raw
	    set periode   -1	
	    set report    no
	    set request [ined request \
			 "Select drawing speed for statistic data:" \
                         [list [list "Speed:" $draw radio slow at-once] ] \
			 [list cancel ok] ]
	    if {[lindex $request 0] == "cancel"} { 
		catch {unset old_data_sema}
		return 
	    }
	    set draw [lindex $request 1]	    
	}
	{15-min} { 
	    set kind_data_string "average total 1-min-peak"
	}
	{1-hour} {  
	    set kind_data_string "average total 1-min-peak 15-min-peak"
	} 
	{24-hour} {
	    set kind_data_string "average total 1-min-peak 15-min-peak 1h-peak"
	}
    }
    if {$periode != -1} {

	set request [ined request "Select aggregation and drawing speed for\
                                   statistic data:" \
		     [list "Aggregation: $kind_data radio $kind_data_string" \
                      [list "Create report:" $report radio yes no] \
                      [list "Speed:" $draw radio slow at-once] ] \
		     [list cancel ok] ]

	if {[lindex $request 0] == "cancel"} { 
	    catch {unset old_data_sema}
	    return 
	}
	set kind_data [lindex $request 1]
	set report    [lindex $request 2]
	set draw      [lindex $request 3]
    }
    ##
    ## request_old == variables for which we want data
    ## var_list    == all old variables from server for requested day
    ## request     == specification about requested variables
    ## periode     == intervaltime of variable in statistik
    ##
    catch {unset res}
    foreach elem $request_old {
	set elem [lindex $elem 0]
	lappend res [lindex $elem 2]
	lappend res "[lindex $elem 5] "
	lappend restmp $res
	set res ""
    }
    set result "data_old_tag"
    
    set old_data_specifier "$old_data_day $kind_data $periode $report $draw"
    set old_data_vars      $request_old
    
    lappend res1 $old_data_specifier
    lappend res1 $restmp    
    lappend result $res1
    lappend to_socket $result
}

##
## proc creates stripcharts for the old data values
## creates a job that displays the data in "slow motion" if requested
## writes the data to a log window, if report is selected
##

proc control_old_data {cmd_arg} {
    global old_data_sema
    global old_data_specifier
    global old_data_vars
    global old_data_table
    global old_data_count
    
    if {$cmd_arg == ""} { 
	catch {unset old_data_sema}
	return 
    }

    set kind_data [lindex $old_data_specifier 1]
    set data_periode  [lindex $old_data_specifier 2]
    set report [lindex $old_data_specifier 3]
    
    set x 100
    set y 0
  
    if {$data_periode == -1} { set data_periode ""}
 
    foreach elem $old_data_vars {
	
	set elem [lindex $elem 0]
	
	foreach ext_elem [lindex $elem 5] {
	    
	    set ext_index [lsearch [lindex $elem 5] $ext_elem] 

	    if {![info exists old_data_count]} {
		set old_data_count 0
	    } else {
		incr old_data_count
	    }
	    if {$y == 360} { 
		set y 60; set x [expr {$x + 150}]
	    } else {
		set y [expr {$y + 60}]
	    }
	    set old_data_table($old_data_count,host)       [lindex $elem 0]
	    set old_data_table($old_data_count,var_name)   [lindex $elem 1]
	    set old_data_table($old_data_count,var_id)     [lindex $elem 2]
	    set old_data_table($old_data_count,periode)    [lindex $elem 3]
	    set old_data_table($old_data_count,start_time) [lindex \
                                       [lindex $elem 4] $ext_index]
	    set old_data_table($old_data_count,ext)        $ext_elem
	    set old_data_table($old_data_count,count_strip_value) 0
	    set old_data_table($old_data_count,oid)        [ined create STRIPCHART]
	    
  	    ined -noupdate move $old_data_table($old_data_count,oid) $x $y
	    set old_data_table($old_data_count,coords) [ined size \
                $old_data_table($old_data_count,oid)]
	    set label_name "[lindex $elem 0] [lindex $elem 1] \
                [lindex $elem 3] $kind_data $data_periode"
	    ined -noupdate name $old_data_table($old_data_count,oid) \
                $label_name
	    ined -noupdate label $old_data_table($old_data_count,oid) name
	    
	    set old_data_table($old_data_count,coords) [ined size \
                $old_data_table($old_data_count,oid)]
	}
    }
    set scale_factor 100 
    for {set i 0} {$i <= $old_data_count} {incr i} {
	set quest ""
	lappend quest [format "%12s %12s %3d:" $old_data_table($i,host) \
                $old_data_table($i,var_name) \
		       $old_data_table($i,periode)]
	lappend question "$quest $scale_factor scale 1 100000"	
    }

    set scaling [ined request "Set initial scale factor for stripcharts:" \
		 $question [list cancel ok] ]
    if {[lindex $scaling 0] == "cancel"} {
	for {set i 0} {$i <= $old_data_count} {incr i} {
	    catch {ined -noupdate delete $old_data_table($i,oid)}
	}
	reset_old_vars
	return
    }
    set scaling [lrange $scaling 1 end]
    for {set i 0} {$i <= $old_data_count} {incr i} {
	ined -noupdate scale $old_data_table($i,oid) [lindex $scaling $i]
    }
    set run_count 0

    foreach cmd_arg_elem $cmd_arg {
	set ext         [lindex $cmd_arg_elem 0]
	set varid_list  [lindex $cmd_arg_elem 1]
	set values_list [lindex $cmd_arg_elem 2]
	
	for {set i 0} {$i <= $old_data_count} {incr i} {
	    set var_id1 $old_data_table($i,var_id)
	    set index [lsearch $varid_list $var_id1]
	    if {$index == -1} { continue }
	    if {$old_data_table($i,ext) != $ext} { continue }
	    set old_data_table($i,values) [lindex $values_list $index]
	    set old_run_count [llength $old_data_table($i,values)]
	    if {$run_count < $old_run_count} {
		set run_count $old_run_count
	    }
	}
    }

    if {$report == "yes"} {
	write_report
    }
    
    if {![info exists old_data_count]} {
	catch {unset old_data_sema}
    }
    case [lindex $old_data_specifier 4] in {
	{slow} {
	    set play_time 1
	}
	{at-once} {
	    show_old_data_once
	    return
	}
    }
    job create "show_old_data $run_count" [expr {$play_time * 1000}]
}

##
## proc shows old data, no other request is possible during the run of this
## proc !!!
##
		    
proc show_old_data {run_count} {
    global old_data_specifier
    global old_data_sema 
    global old_data_table
    global old_data_count
    global proc_run_counter

    if {![info exists proc_run_counter]} {
	set proc_run_counter 0
    }
    if {$proc_run_counter >= $run_count} {
	reset_old_vars
	catch {unset proc_run_counter}
	job_kill [job current]
	return
    }
    if {![info exists old_data_count]} {
	reset_old_vars
	catch {unset proc_run_counter}
	job_kill [job current]
	return
    }
    set kind_data [lindex $old_data_specifier 1]
    for {set i 0} {$i <= $old_data_count} {incr i} {
	if {[ined retrieve $old_data_table($i,oid)] == ""} {
	    continue
	}
	set starttime $old_data_table($i,start_time)
	set time  [lindex [lindex $old_data_table($i,values) \
                  $proc_run_counter] 0]
	case $kind_data in {
	    {1-min-peak 15-min-peak 1h-peak} {
		set peak_time [lindex [lindex [lindex \
                         $old_data_table($i,values) $proc_run_counter] 1] 1]
		set value     [lindex [lindex [lindex \
                         $old_data_table($i,values) $proc_run_counter] 1] 0]
	    }
	    {default} {
		set value [lindex [lindex $old_data_table($i,values) \
                         $proc_run_counter] 1]
	    }
	}
	if {$value == ""} { continue }
	if {$proc_run_counter > 57} {
	    set coords $old_data_table($i,coords)
	    set coords [lreplace $coords 2 2 [expr {[lindex $coords 2] + 1}]]
	    set old_data_table($i,coords) $coords
	    eval ined -noupdate size $old_data_table($i,oid) $coords
	}
	ined -noupdate values $old_data_table($i,oid) $value
	ined -noupdate attribute $old_data_table($i,oid) text \
	    "$starttime until $time"
	ined -noupdate label $old_data_table($i,oid) text
    }
    incr proc_run_counter
    return
}

##
## shows old data at once
## 

proc show_old_data_once {} {
    global old_data_specifier
    global old_data_sema 
    global old_data_table
    global old_data_count

    if {![info exists old_data_count]} {
	reset_old_vars
	return
    }
    set kind_data [lindex $old_data_specifier 1]
    for {set i 0} {$i <= $old_data_count} {incr i} {
	if {[ined retrieve $old_data_table($i,oid)] == ""} {
	    continue
	}
	if {![info exists old_data_table($i,values)]} { continue } 
	set starttime $old_data_table($i,start_time)
	set end_index [expr {[llength $old_data_table($i,values)] - 1}]
	set endtime  [lindex [lindex $old_data_table($i,values) $end_index] 0]
	set res ""
	case $kind_data in {
	    {1-min-peak 15-min-peak 1h-peak} {
		foreach value $old_data_table($i,values) {
		    lappend res [lindex [lindex $value 1] 0]
		}
	    }
	    {default} {
		set value ""
		foreach value $old_data_table($i,values) {
		    lappend res [lindex $value 1]
		}
	    }
	}
	if {$res == ""} { continue }
	set coords $old_data_table($i,coords)
	set coords [lreplace $coords 2 2 \
                   [expr {[lindex $coords 2] + $end_index - 50}]]
	set old_data_table($i,coords) $coords
	eval ined size $old_data_table($i,oid) $coords
	eval ined -noupdate values $old_data_table($i,oid) $res
	ined -noupdate attribute $old_data_table($i,oid) text \
	    "$starttime until $endtime"
	ined -noupdate label $old_data_table($i,oid) text
	catch {unset res}
    }
    reset_old_vars
}

##
## proc writes values from old_data_request to log-window
##

proc write_report {} {
    global old_data_table
    global old_data_count
    global old_data_specifier

    set date          [date_convert [lindex $old_data_specifier 0]]
    set kind_data     [lindex $old_data_specifier 1]
    set kind_interval [lindex $old_data_specifier 2]

    writeln " "
    writeln "REPORT for OLD DATA request:"
    writeln "day of aggregated data:     $date"
    writeln "kind of statistic interval: $kind_interval"
    writeln "kind of requested data:     $kind_data"
    set line_count 0
    for {set i 0} {$i <= $old_data_count} {incr i} {
	lappend label [format "%10s %12s" $old_data_table($i,host) \
                                          $old_data_table($i,var_name)]
	set len [llength $old_data_table($i,values)]
	if {$len > $line_count} {
	    set line_count $len
	}
    }
    writeln $label
    for {set i 0} {$i < $line_count} {incr i} { 
	set line ""
	for {set j 0} {$j <= $old_data_count} {incr j} {
	    lappend line [format "%22s " [lindex $old_data_table($j,values) \
                                         $i]]
	}
	writeln $line
    }
}

##
## proc resets all data request of this tool
##

proc "Reset requests" {list} {
    global var_oid_list
    global var_request_list
    global monitor_name
    global socket
    global to_socket
    global proc_run_counter

    if {$socket == ""} {
	ined acknowledge "No connection to $monitor_name"
	return
    }

    set kind_reset all
    set kind_reset [ined request "Select kind of reset:"  \
		    [list [list "" $kind_reset radio actual old all] ] \
		    [list cancel ok] ]

    if {[lindex $kind_reset 0] == "cancel"} return
    set kind_reset [lindex $kind_reset 1]

    if {$kind_reset == "old"} {
	reset_old_vars
	if {[info exists proc_run_counter]} {
	    set proc_run_counter 100000
	}
	return
    }
    if {$kind_reset == "all"} {
	reset_old_vars
	if {[info exists proc_run_counter]} {
	    set proc_run_counter 100000
	}
    }
    if {![info exists var_request_list]} {
	return
    }
    foreach elem $var_request_list {
	catch {unset var_oid_list($elem)}
    }
    catch {unset var_request_list}
    lappend to_socket "data_reset_tag"
}

##
## proc resets old_data_variables
##

proc reset_old_vars {} {
    global old_data_sema 
    global old_data_table
    global old_data_count

    catch {unset old_data_sema}
    if {![info exists old_data_count]} { return }
    for {set i 0} {$i <= $old_data_count} {incr i} {
	catch {unset old_data_table($i,host)}    
	catch {unset old_data_table($i,var_name)}
	catch {unset old_data_table($i,var_id)} 
	catch {unset old_data_table($i,periode)}
	catch {unset old_data_table($i,start_time)}
	catch {unset old_data_table($i,ext)}     
	catch {unset old_data_table($i,count_strip_value)} 
	catch {unset old_data_table($i,oid)}
	catch {unset old_data_table($i,values)}
	catch {unset old_data_table($i,coords)}
    }
    catch {unset old_data_count}
    return 
}

##
## the following proc can be used to close the connection and delete
## the menu entry in tkined
##
##

##
## Delete the menu and this interpreter
##

proc "Delete Netguard-Client" {list} {
    global control_socket
    global menus
    global socket
    global var_request_list
    
    if {[info exists var_request_list]} {
	ined acknowledge "Please reset all current data request, before\
                          deleting the tool."
	return
    }
    # send close_command to monitor
    catch {puts $socket "close_connect slave"; flush $socket}
    catch {removeinput $socket}
    catch {tcp shutdown $socket all}
    catch {tcp close $socket}
    foreach id $menus { ined delete $id }
    exit
}

##
##
##  here starts the section with the limit procedures
##  
##
##
## give infos about limit values and alarm requests
##

proc "Limit Info" {list} {
    global to_socket
    global socket
    global monitor_name
    global limit_info_button
    global data_info_now

    if {$socket == ""} { 
	ined acknowledge "No connection to $monitor_name"
	return
    }
    if {![info exists limit_info_button]} {
	set limit_info_button info
    }

    set request [ined request "Request for variables with limit control." \
		 [list [list "Information about:" $limit_info_button radio \
                   info overflow] ] [list cancel ok] ]

    if {[lindex $request 0] == "cancel"} return

    set limit_info_button [lindex $request 1]
    if {$limit_info_button == "info"} {
	lappend to_socket "limit_info info"
	return
    }
    if {$limit_info_button == "overflow"} {
	lappend to_socket "limit_info all"
	return
    }
    if {$limit_info_button == "some-variables"} {
	return	
    }
}

##
## proc sends a limit request to the momitor
##

proc "Alarm request" {list} {
    global to_socket
    global socket
    global monitor_name
    global data_info_now
    global limit_data_choose
    global limit_info_button

    if {$socket == ""} { 
	ined acknowledge "No connection to $monitor_name"
	return
    }
    if {![info exists data_info_now]} {
	writeln "server has not sent variable information"
	return
    }
    if {![info exists limit_data_choose]} {
	set host *
	set variables *
    } else { 
	set host [list [lindex $limit_data_choose 0]]
	set variables [list [lindex $limit_data_choose 1]]
    }

    set request [ined request \
		 "Choose host and variable for sending overflow messages:" \
		 [list [list "Host:" $host] ["Variables:" $variables] ] \
		 [list cancel ok] ]
    
    if {[lindex $request 0] == "cancel"} return

    set limit_data_choose [list [lindex $request 1] [lindex $request 2]]
    set host     [lindex $limit_data_choose 0]
    set variables [lindex $limit_data_choose 1]

    foreach req_host $host {
	foreach elem $data_info_now {
	    if {![string match $req_host [lindex $elem 0]]} { continue }
	    foreach req_var $variables {
		foreach var_elem [lindex $elem 1] {
		    if {![string match $req_var [lindex $var_elem 0]]} {
			continue 
		    }
		    lappend req_result [format "%10s %15s %3s"  \
		                       [lindex $elem 0] [lindex $var_elem 0] \
                                       [lindex $var_elem 1]]
		}		
	    }
	}
    }
    if {![info exists req_result]} { 
	ined acknowledge "No match for \"$host\" and \"$variables\"."
	return
    }
    set header "Select hosts and variables for sending its overflow data." 
    set res 1
    set request_limit ""
    while {$res != ""} {
	set res [ined list $header $req_result [list cancel ok]]
	if {[lsearch $request_limit [lindex $res 1]] == -1} {
	    if {[lindex $res 0] != "cancel"} {
		lappend request_limit [lindex $res 1]
	    }
	}
    }
    if {$request_limit == ""} { return }
    set i 0
    set var_ids ""
    foreach elem $request_limit {
	set elem [lindex $elem 0]
	set host [lindex $elem 0]
	set var [lindex $elem 1]
	foreach limit_elem $data_info_now { 
	    if {!([lindex $limit_elem 0] == $host)} { continue }
	    foreach var_limit_elem [lindex $limit_elem 1]  {
		if {!([lindex $var_limit_elem 0] == $var)} { continue }
		lappend var_ids [lindex $var_limit_elem 2]
	    }
	}
    }
    lappend to_socket [list limit_insert_req $var_ids]
}

##
## procedure writes alarmmessages in logwindow
##

proc alarm_warning {mess} {
    global data_info_now
    
    set warn_res ""
    if {$mess == ""} { return }
    foreach mess_elem $mess {
	foreach info_elem $data_info_now {
	    foreach var_elem [lindex $info_elem 1] {
		if {[lindex $mess_elem 0] != [lindex $var_elem 2]} { continue }
		set host [lindex $info_elem 0]
		set adress [lindex $info_elem 2]
		set var_name [lindex $var_elem 0]
		set value    [lindex $mess_elem 2]
		set time     [lindex $mess_elem 1] 

		writeln ""
		writeln "LIMIT OVERFLOW: $host $adress $var_name \
		         reached: $value at $time"
		lappend warn_res "$host $adress"
	    }
	}
    }
    if {$warn_res == ""} {return}
    catch {set objects [ined retrieve]}
    foreach host_elem $warn_res {
	foreach elem $objects {
	    if {[ined type $elem] != "NODE"} { continue }
	    if {[lindex $elem 2] == ""} { continue }
	    if {[lindex [split [lindex $elem 2] .] 0] != \
                [lindex [split [lindex $host_elem 0] .] 0]} {
		continue
	    }
	    catch {ined color [lindex $elem 1] Red}
	}
    }
}

##
## proc deletes an entry for sending overflow messages
##

proc "Alarm delete request" {list} {
    global to_socket
    global socket
    global monitor_name
    global limit_data_choose
    global data_info_now

    if {$socket == ""} { 
	ined acknowledge "No connection to $monitor_name"
	return
    }
    if {![info exists data_info_now]} {
	writeln "server has not sent variable information"
	return
    }
    if {![info exists limit_data_choose]} {
	set host *
	set variables *
    } else { 
	set host [list [lindex $limit_data_choose 0]]
	set variables [list [lindex $limit_data_choose 1]]
    }

    set request [ined request \
		 "Choose host and variable for deleting overflow request:" \
		 [list [list "Host:" $host] ["Variables:" $variables] ] \
		 [list cancel ok] ]
    
    if {[lindex $request 0] == "cancel"} return

    set limit_data_choose [list [lindex $request 1] [lindex $request 2]]
    set host     [lindex $limit_data_choose 0]
    set variables [lindex $limit_data_choose 1]

    foreach req_host $host {
	foreach elem $data_info_now {
	    if {![string match $req_host [lindex $elem 0]]} { continue }
	    foreach req_var $variables {
		foreach var_elem [lindex $elem 1] {
		    if {![string match $req_var [lindex $var_elem 0]]} {
			continue 
		    }
		    lappend req_result [format "%10s %15s %3s"  \
					[lindex $elem 0] [lindex $var_elem 0] \
                                        [lindex $var_elem 1]]
		}		
	    }
	}
    }
    if {![info exists req_result]} { 
	ined acknowledge "No match for \"$host\" and \"$variables\"."
	return
    }
    set header "Select hosts and variables for deleting overflow request." 
    set res 1
    set request_limit ""
    while {$res != ""} {
	set res [ined list $header $req_result [list cancel ok]]
	if {[lsearch $request_limit [lindex $res 1]] == -1} {
	    if {[lindex $res 0] != "cancel"} {
		lappend request_limit [lindex $res 1]
	    }
	}
    }
    if {$request_limit == ""} { return }
    set i 0
    set var_ids ""
    foreach elem $request_limit {
	set elem [lindex $elem 0]
	set host [lindex $elem 0]
	set var [lindex $elem 1]
	foreach limit_elem $data_info_now { 
	    if {!([lindex $limit_elem 0] == $host)} { continue }
	    foreach var_limit_elem [lindex $limit_elem 1]  {
		if {!([lindex $var_limit_elem 0] == $var)} { continue }
		lappend var_ids [lindex $var_limit_elem 2]
	    }
	}
    }
    lappend to_socket [list limit_delete_req $var_ids]
}

##
## proc sends a set request to the monitor limit control to set the warn flag
## of the requested vars
##

proc "Alarm suspend request" {list} {
    global to_socket
    global socket
    global monitor_name
    global limit_warn_choose
    global data_info_now

    if {$socket == ""} { 
	ined acknowledge "No connection to $monitor_name"
	return
    }
    if {![info exists data_info_now]} {
	writeln "server has not sent variable information"
	return
    }
    if {![info exists limit_warn_choose]} {
	set host *
	set variables *
    } else { 
	set host [list [lindex $limit_warn_choose 0]]
	set variables [list [lindex $limit_warn_choose 1]]
    }

    set request [ined request "Choose host and variable for changing status\
                               of the warning request:" \
		 [list [list "Host:" $host] ["Variables:" $variables] ] \
		 [list cancel ok] ]
    
    if {[lindex $request 0] == "cancel"} return

    set limit_warn_choose [list [lindex $request 1] [lindex $request 2]]
    set host     [lindex $limit_warn_choose 0]
    set variables [lindex $limit_warn_choose 1]

    foreach req_host $host {
	foreach elem $data_info_now {
	    if {![string match $req_host [lindex $elem 0]]} { continue }
	    foreach req_var $variables {
		foreach var_elem [lindex $elem 1] {
		    if {![string match $req_var [lindex $var_elem 0]]} { 
			continue 
		    }
		    lappend req_result [format "%10s %15s %3s"  \
				       [lindex $elem 0] [lindex $var_elem 0] \
                                       [lindex $var_elem 1]]
		}		
	    }
	}
    }
    if {![info exists req_result]} { 
	ined acknowledge "No match for \"$host\" and \"$variables\"."
	return
    }
    set header "Select hosts and variables for changing status of the warning\
                request."
    set res 1
    set request_limit ""
    while {$res != ""} {
	set res [ined list $header $req_result [list cancel ok]]
	if {[lsearch $request_limit [lindex $res 1]] == -1} {
	    if {[lindex $res 0] != "cancel"} {
		lappend request_limit [lindex $res 1]
	    }
	}
    }
    if {$request_limit == ""} { return }
    set i 0
    set var_ids ""
    foreach elem $request_limit {
	set elem [lindex $elem 0]
	set host [lindex $elem 0]
	set var [lindex $elem 1]
	foreach limit_elem $data_info_now { 
	    if {!([lindex $limit_elem 0] == $host)} { continue }
	    foreach var_limit_elem [lindex $limit_elem 1]  {
		if {!([lindex $var_limit_elem 0] == $var)} { continue }
		lappend var_ids [lindex $var_limit_elem 2]
	    }
	}
    }
    lappend to_socket [list limit_set_warn $var_ids]
}

##
## proc deletes all alarm requests of the client
##

proc "Alarm reset" {list} {
    global socket
    global to_socket
    
    if {$socket == ""} { 
	ined acknowledge "No connection to $monitor_name"
	return
    }
    
    lappend to_socket "limit_delete_req all"
    return
}

##
## proc sets the color of an node object to black 
##      so you can see new incoming alarmmessages
##

proc "Alarm confirm" {list} {
    
    if {$list == ""} { return }

    foreach elem $list {
	if {[lindex $elem 0] == "NODE"} {
	    if {[ined color [lindex $elem 1]] == "Red"} {
		catch {ined color [lindex $elem 1] Black}		
	    }	    
	}	
    }
}

##
## here are procedures for doing the communication with the monitorserver
##
##
## wait for answers from monitor
##
##

proc read_socket {} {
    global socket
    global monitor_name
    global port
    global to_socket 
    global sema

    if {[catch {gets $socket} answer]} {
	writeln "fatal error: can not read from monitorsocket"
	close_connection
	return
    }
    if {[eof $socket]} {
	writeln "error: read_socket connection $monitor_name closed"
	close_connection
	return
    }
    set answer [string trimright $answer "\r\n"]
    if {[llength [split $answer]] < "2"} {
	writeln "error: answer $answer format from monitor is wrong"
	set sema ""
	lappend to_socket "error tkined wrong_format"
	return
    }
    set answer_action [lindex $answer 0]
    set answer_cmd    [lindex $answer 1]
    set answer_arg    [lindex $answer 2]
    process $answer_action $answer_cmd $answer_arg $answer
    set sema ""
}

##
## write to monitorsocket
##

proc write_socket {} {
    global monitor_name
    global port
    global to_socket
    global sema
    global socket

    if {$socket == ""} { 
	job_kill [job current]
	return
    }
    if {$to_socket == ""} { 
	return
    }
    set message [lindex $to_socket 0]
    if {$sema != ""} { 
	return
    }
    set to_socket [ldelete to_socket $message]
    if {[catch {puts $socket $message; flush $socket}]} {
	writeln "Write to monitorsocket $monitor_name $port failed."
	job_kill [job current]
	close_connection
	return
    }
}

##
## proc handle incoming errormsg from server
##

proc error_handle {cmd message} {
    global old_data_sema

    case $cmd in {
	{data_old_tag} { 
	    ined acknowledge "Request of old data failed."
	    catch {unset old_data_sema} 
	    return
	}
	{close} {
	    if {$message == "no agents"} {
		ined acknowledge "No agents to close."
		return
	    } else {
		ined  acknowledge "Wrong message format."
		return
	    }
	}
	{limit_request} {
	    case [lindex [split $message] 0] in {
		{insert} {
		    writeln "Sorry, alarm request failed."
		    return
		}
		{delete} {
		    writeln "Sorry, limit control delete failed."
		    return
		}
		{set_warn} {
		    writeln "Sorry, alarm suspend failed."
		    return
		}
		{default} {
		    writeln $message
		    return
		}
	    }
	}
	{default} {
	    ined  acknowledge "Server returns error. See Netguard-Client\
                               Report."
	    writeln $message
	}
    }
}

##
## proc request monitor for a list of some commands
##

proc show_help {answer_arg} {
    
    if {$answer_arg == ""} {
	writeln "Monitor gives no help ???"
	return
    } else {
	writeln "List of commands for interactive use"
	foreach elem $answer_arg {
	    writeln $elem
	}
    }
}

##
## proc deletes jobs created by scotty`s job command
##

proc job_kill {job_id} {
    
    if {$job_id == ""} { return }
    after 100 "job kill $job_id"
}

##
## parser for answer from monitor
##

proc process {action cmd answer_arg answer} {
    global to_socket
    global monitor_name
    global port
    global old_data_sema

    if {$action == "error"} {
	error_handle $cmd $answer_arg
        return
    }
    if {$action == "result"} {
	case $cmd in {
	    {message} {
		if {$answer_arg != ""} {
		    ined acknowledge "message from monitor"
		    writeln $answer_arg
		}
		return
	    }
	    {secret_message} {
		writeln "server writes to history: $answer_arg"
		return
	    }
	    {info} {
		show_info $answer_arg
		return
	    }
	    {help} {
		show_help $answer_arg
		return
	    }
	    {data_info_now} {
		insert_data_now $answer_arg
		return
	    }
	    {data_tag_now} {
		show_actual_data $answer_arg
		return
	    }
	    {data_info_old} {
		request_data_old $answer_arg
		return
	    }
	    {data_tag_old} {
		control_old_data $answer_arg
		return		
	    }
	    {limit_info_now limit_info_req limit_info_all} {
		show_info $answer_arg
		return 
	    } 
	    {limit_info_tag} {
		writeln $answer_arg
		return
	    }
	    {limit_request_now} {
		alarm_warning $answer_arg
		return		
	    }
	    {limit_request} {
		return
	    }
	    {default} {
		writeln "monitor returns result $cmd"
		return
	    }
	}
    }
    if {$action == "ok"} {
	case $cmd in {
	    {accepted} {
		accept_message
		return
	    }
	    {name} {
		lappend to_socket "data_info now"
		return
	    }
	    {yau} {
		show_info $answer_arg
		return
	    }
	    {close_tools} {
		writeln "All clients of server $monitor_name $port are\
                         closed." 
		return
	    }
	    {close} {
		close_connection
		return
	    }
	    {kill} {
		kill_message
		return
	    }
	    {data_request} {
		return
	    }
	    {limit_request} {
		return
	    }
	    {default} {
		writeln "monitor accepts $cmd."
		return
	    }
	}
    }
    writeln "Unknown action $answer"
}

##
## Display some help about this tool.
##

proc "Help Netguard-Client" {list} {
    ined browse "Help about Netguard-Client" { 
	"The Netguard-Client is a tkined-front-end for a Monitoring-Server" 
	"which collects current data of monitoring agents and stores the data" 
        "in files. This client retrieves data from the server and displays it" 
	"in stripcharts." 
	"" 
	"Info agents:" 
	"    Ask Monitor-Server about the state of its agents" 
	"    and list the variables measured by the agents." 
	"" 
	"Info server:" 
	"    Ask Monitor-Server about its state and active connections." 
	"" 
	"Request current data:" 
	"    Display one or more currently measured variables in stripcharts." 
	"" 
	"Request statistic data:" 
	"    Display old measured statistic data in stripcharts." 
	"    The result can be written to a Report-Window." 
	"" 
	"Reset running data requests:" 
	"    Reset current, statistic or ALL running data requests." 
	"    A running current data request can be reset" 
	"    by deleting the stripchart." 
	"" 
	"Limit Info:" 
	"    Ask Monitor-Server about which variables of hosts are under"  
	"    Limit-control. Show the actual state of each controlled" 
	"    variable." 
	"" 
	"Alarm Request:" 
	"    Insert a request for sending a message, if a Limit-value is" 
	"    reached. The message is displayed in a Log-Window, and an" 
        "    existing stripchart will be marked with another color." 
	"" 
	"Alarm delete request:" 
	"    Delete an existing Alarm-request." 
	"" 
	"Alarm Confirm:" 
	"    Set the color of a marked stripchart back to black." 
	"" 
	"Alarm reset:" 
	"    Reset all actual Alarm requests." 
    }
}

##
## proc creates a menu with the posibility to close a connertion
##

proc menu_close {} {
    global menus
    
    foreach id $menus { ined delete $id }

    set menus [ ined create MENU "Netguard-Client" \
	       "Close connection" "" \
		   "Info agents" "Info server" "" \
		   "Request current data" "Request statistic data" \
		   "Reset requests" "" \
		   "Limit Info" \
		   "Alarm request" "Alarm suspend request" \
		   "Alarm delete request" "Alarm confirm" \
		   "Alarm reset" "" \
		   "Help Netguard-Client" \
		   "Delete Netguard-Client"]
}

##
## proc creates a menu with the posibility to open a connertion
##

proc menu_connect {} {
    global menus

    foreach id $menus { ined delete $id }

    set menus [ ined create MENU "Netguard-Client" \
	       "Try connect" "" \
		   "Info agents" "Info server" "" \
		   "Request current data" "Request statistic data" \
		   "Reset requests" "" \
		   "Limit Info" \
		   "Alarm request" "Alarm suspend request" \
		   "Alarm delete request" "Alarm confirm" \
		   "Alarm reset" "" \
		   "Help Netguard-Client" \
		   "Delete Netguard-Client"] 
}

menu_connect

auto_connect
