/*
 * ntp.c
 *
 * Extend a tcl command interpreter with a command to query
 * NTP server for timestat.
 *
 *
 * Copyright (c) 1994
 *
 * E. Schoenfelder, J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

/* 
 * todo: 
 *    * check about `more' flag.
 *    * cache hostlookups.
 *    * make better error return strings.
 */

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#include <tcl.h>

#include "scotty.h"

/* debugging ? */
static int debug = 0;
#define dprintf if (debug) fprintf

struct ntp_control {
  unsigned char mode;			/* version and mode */
  unsigned char op;			/* opcode */
  unsigned short sequence;		/* sequence # */
  unsigned short status;		/* status */
  unsigned short associd;		/* association id */
  unsigned short offset;		/* data offset */
  unsigned short len;			/* data len */
  unsigned char data[(480 + 20)];	/* data + auth */
};

#ifdef __alpha
/* dec alpha needs an extra cookie... */
typedef unsigned int ipaddr_t;
#else
typedef unsigned long ipaddr_t;
#endif

/* communication socket */
static int sock = -1;

/* # of retries: */
static int retries = 3;

/* timeout in ms */
static int timeo = 3000;

/* sequence number: */
static int seq = 1;

/*
 * Forward declarations for procedures defined later in this file:
 */

static int
sock_init		_ANSI_ARGS_((Tcl_Interp *interp));

static int
is_ready		_ANSI_ARGS_((int sock, int timeo));

static void
make_pkt		_ANSI_ARGS_((struct ntp_control *pkt, 
				     int op, int assoc));
static int
fetch_op		_ANSI_ARGS_((Tcl_Interp *interp, 
				     struct sockaddr_in *daddr, 
				     int op, char *buf, int assoc));
static int
ntp_split		_ANSI_ARGS_((Tcl_Interp *interp, char *varname,
				     char *pfix, char *buf));
static int 
get_peer		_ANSI_ARGS_((char *data, int *assoc));

static int
ntp_query		_ANSI_ARGS_((Tcl_Interp *interp, 
				     int argc, char **argv));


/*
 * create the communication socket:
 */

static int
sock_init (interp)
    Tcl_Interp *interp;
{
    struct sockaddr_in maddr;
    
    if (sock != -1)
	    close (sock);

    if ((sock = socket (AF_INET, SOCK_DGRAM, 0)) < 0) {
	Tcl_AppendResult (interp, "could not create socket: ", 
                          Tcl_PosixError (interp), (char *) NULL);
        return TCL_ERROR;
    }

    maddr.sin_family = AF_INET;
    maddr.sin_addr.s_addr = htonl (INADDR_ANY);
    maddr.sin_port = htons (0);

    if (bind (sock, (struct sockaddr *) &maddr, sizeof (maddr)) < 0) {
	Tcl_AppendResult (interp, "can not bind: ", Tcl_PosixError (interp), 
			  (char *) NULL);
        return TCL_ERROR;
    }

    return TCL_OK;
}


/*
 * return != 1 if sock is ready.
 */

static int
is_ready (sock, timeo)
     int sock, timeo;
{
    fd_set rfd;
    struct timeval tv;
    int rc;
    
    FD_ZERO (&rfd);
    FD_SET (sock, &rfd);
    tv.tv_sec = timeo / 1000;
    tv.tv_usec = (timeo % 1000) * 1000;
    
    do {
	rc = select (sock + 1, &rfd, (fd_set *) 0,  (fd_set *) 0, &tv);
	if (rc == -1 && errno == EINTR)
		continue;
	if (rc == -1)
	{
	    perror ("* select failed; reason");
	    return 0;
	}
    } while (rc < 0);
    
    return rc > 0;
}


static void
make_pkt (pkt, op, assoc)
     struct ntp_control *pkt;
     int op, assoc;
{
    pkt->mode = 0x18 | 6;			/* version 3 | MODE_CONTROL */
    pkt->op = op;				/* CTL_OP_... */
    pkt->sequence = htons (seq);
    pkt->status = 0;
    pkt->associd = htons (assoc);
    pkt->offset = htons (0);

    if (! assoc)
      sprintf (pkt->data, 
	       "precision,peer,system,stratum,rootdelay,rootdispersion,refid");
    else 
      sprintf (pkt->data, 
	       "srcadr,stratum,precision,reach,valid,delay,offset,dispersion");
    pkt->len = htons (strlen (pkt->data));
}


/*
 * fetch op from server, append result to buf.
 * return -1 on error.
 */

static int
fetch_op (interp, daddr, op, buf, assoc)
    Tcl_Interp *interp;
    struct sockaddr_in *daddr;
    int op;
    char *buf;
    int assoc;
{
    struct ntp_control qpkt, pkt;
    struct sockaddr_in saddr;
    int i, rc, slen = sizeof (saddr);
    int timeout = timeo / (retries + 1);

    /* increment to a new sequence number: */
    seq++;
    
    for (i = 0; i < retries + 1; i++)
    {
	make_pkt (&qpkt, op, assoc);		/* CTL_OP_READVAR */
	memset ((char *) &pkt, 0, sizeof (pkt));
	
	rc = sendto (sock, (char *) &qpkt, sizeof (qpkt), 0, 
		     (struct sockaddr *) daddr, sizeof (*daddr));
	if (rc < 0)
	{
	    Tcl_AppendResult (interp, "udp sendto failed: ",
			      Tcl_PosixError (interp), (char *) NULL);
	    return TCL_ERROR;
	}
	
	while (is_ready (sock, timeout))
	{
	    memset ((char *) &pkt, 0, sizeof (pkt));
	    rc = recvfrom (sock, (char *) &pkt, sizeof (pkt), 0, 
			   (struct sockaddr *) &saddr, &slen);
	    if (rc < 0)
	    {
		Tcl_AppendResult (interp, "recvfrom failed: ",
				  Tcl_PosixError (interp), (char *) NULL);
		return TCL_ERROR;
	    }
	    
	    if (rc < 12 + 1)		/* ntp_control + 1 data byte */
	    {
		dprintf (stderr, "* short packet (%d bytes) ignored.\n", rc);
		continue;
	    }
	    
	    dprintf (stderr, "** got status 0x%x\n", pkt.op);

	    if ((pkt.op & 0x80) 
		&& saddr.sin_addr.s_addr == daddr->sin_addr.s_addr
		&& saddr.sin_port == daddr->sin_port
		&& pkt.sequence == qpkt.sequence)
	    {
		dprintf (stderr, "fine: seq %d\n", pkt.sequence);
		dprintf (stderr, "** got: %s\n", pkt.data);
		strcat (buf, pkt.data);
		return TCL_OK;
	    }
	    else 
	    {
		dprintf (stderr, "ignored: seq %d\n", pkt.sequence);
	    }
	}
    }
    
    Tcl_SetResult (interp, "no ntp answer received", TCL_STATIC);
    return TCL_ERROR;
}

static int
ntp_split (interp, varname, pfix, buf)
    Tcl_Interp *interp;
    char *varname;
    char *pfix;
    char *buf;
{
    char *d, *s, *g, *r;
    char var [256];

    for (s = buf, d = buf; *s; s++)
    {
	if (*s == ',') {
	    *s = '\0';
	    for (g = d; *g && (*g != '='); g++) ;
	    if (*g) {
		*g++ = '\0';
		sprintf (var, "%s.%s", pfix, d);
		r = Tcl_SetVar2 (interp, varname, var, g, TCL_LEAVE_ERR_MSG);
		if (!r) return TCL_ERROR;
	    }
	    for (d = s+1; *d && isspace(*d); d++) ;
	}
    }

    if (d != s) {
	if (isspace(*--s)) *s = '\0';
	if (isspace(*--s)) *s = '\0';
	for (g = d; *g && (*g != '='); g++) ;
	if (*g) {
	    *g++ = '\0';
	    sprintf (var, "%s.%s", pfix, d);
	    r = Tcl_SetVar2 (interp, varname, var, g, TCL_LEAVE_ERR_MSG);
	    if (!r) return TCL_ERROR;
	}
    }

    return TCL_OK;
}


/*
 * scan data for a peer=... entry;
 * return the peer entry in peer and rc = 1, if found
 */
 
static int 
get_peer (data, assoc)
     char *data;
     int *assoc;
{
  int i;

  for (i = 0; i < strlen (data); i++)
    if (1 == sscanf (data + i, "peer=%d,", assoc))
	return 1;

  return 0;
}


static int
ntp_query (interp, argc, argv)
    Tcl_Interp *interp;
    int argc;
    char *argv [];
{
    struct sockaddr_in daddr;
    int rc, assoc, a, b, c, d;
    ipaddr_t addr;
    struct hostent *hp;
    char *hname, *varname, data1 [1024], data2 [1024];

    while (argc > 0 && *argv[0] == '-')
    {
	if (! strcmp (argv [0], "-retries"))
	{
	    argc--, argv++;
	    if (argc < 1)
	    {
		Tcl_SetResult (interp, "missing arg for -retries", TCL_STATIC);
		return TCL_ERROR;
	    }
	    if (Tcl_GetInt (interp, argv [0], &retries) != TCL_OK)
		    return TCL_ERROR;
	    if (retries < 0)
	    {
		Tcl_SetResult (interp, "negative retries", TCL_STATIC);
		return TCL_ERROR;
	    }
	}
	else if (! strcmp (argv [0], "-timeout"))
	{
	    argc--, argv++;
	    if (argc < 1)
	    {
		Tcl_SetResult (interp, "missing arg for -timeout", TCL_STATIC);
		return TCL_ERROR;
	    }
	    if (Tcl_GetInt (interp, argv [0], &timeo) != TCL_OK)
                    return TCL_ERROR;
	    if (timeo < 1)
	    {
		Tcl_SetResult (interp, "negative timeout", TCL_STATIC);
                return TCL_ERROR;
	    }
	    timeo *= 1000;
	}
	else
	{
	    Tcl_AppendResult (interp, "unknown option \"", argv [1], "\"", 
			      (char *) NULL);
	    return TCL_ERROR;
	}
	argc--, argv++;
    }
    
    if (argc != 2)
    {
	Tcl_AppendResult (interp, "wrong # args: should be \"", 
			  argv[0], " ?-timeout t? ?-retries r? ",
			  "host varName\"", (char *) NULL);
	return TCL_ERROR;
    }

    hname = argv [0];
    varname = argv[1];
    
    if (sock < 0)
	    if (sock_init (interp) != TCL_OK) return TCL_ERROR;
    
    while (hname && (*hname == ' '|| *hname == '\t'))
	    hname++;
    if (hname[0] >= '0' && hname[0] <= '9'
	&& 4 == sscanf (hname, "%d.%d.%d.%d", &a, &b, &c, &d))
	    addr = htonl (a << 24 | b << 16 | c << 8 | d);
    else if ((hp = gethostbyname (hname)))
	    addr = * (ipaddr_t *) hp->h_addr;
    else
    {
	Tcl_AppendResult (interp, "no such host \"", hname, 
                          "\"", (char *) NULL);
        return TCL_ERROR;
    }
    
    daddr.sin_family = AF_INET;
    daddr.sin_addr.s_addr = addr;
    daddr.sin_port = htons (123);			/* ntp service */
    
    data1 [0] = data2 [0] = 0;
    if (fetch_op (interp, &daddr, 2, data1, 0) != TCL_OK)  /* CTL_OP_READVAR */
      return TCL_ERROR;
    
    /* try to get additional info: */
    if (get_peer (data1, &assoc))
      if (fetch_op (interp, &daddr, 2, data2, assoc) != TCL_OK)
	return TCL_ERROR;

    /*
     * split response buffer:
     */

    rc = ntp_split (interp, varname, "sys", data1);
    if (rc == TCL_OK)
	return ntp_split (interp, varname, "peer", data2);
    return rc;
}

/* 
 * tcl called interface:
 */

int
ntpCmd (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char *argv[];
{
    if (argc < 3) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", 
			  argv[0], " ?-timeout t? ?-retries r? ",
			  "host varName\"", (char *) NULL);
	return TCL_ERROR;
    }
  
    return ntp_query (interp, argc - 1, argv + 1);
}

/* end of ntp.c */
