/*
 * cmipInit.c
 *
 * Implementation of the "cmip" command, with options like:
 * connect, release, get, set, action, create and delete,
 * that implement a CMIP protocol
 *
 * Copyright (c) 1994
 *
 * M. Kernchen
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdlib.h>
#include <string.h>

#if     defined(__GNUG__)
#include <std.h>
extern "C" { int  strncasecmp(const char*, const char*, int); }

#elif   defined(__cplusplus) && defined(AIXV3)
extern "C" { int  strncasecmp(const char*, const char*, int);
             int  strcasecmp(const char*, const char*); }
#endif

/* (schoenfr) extern "C" wrapper added: */
#if defined(__cplusplus) || defined(__GNUG__)
extern "C" {
#include <isode/tailor.h>
}

#else
#include <isode/tailor.h>
#endif

#include "msap.h"
/* zu den naechsten beiden zeilen nochmal erik fragen!!! */
#undef T_NULL
#define T_NULL          (TYPE_DATA + 12)


#include "Syntaxes.h"

#include <tcl.h>

/*
 * These are the definitions found in the TK library and reprogrammed
 * in event.c. They will be replaced by the TK versions once TK 4.0
 * comes out. John Ousterhout announced that the event loop of TK will
 * be usable without any further access to the X11 server.
 */

#define TK_READABLE     1    /* Corresponding to the tk definition */
#define TK_WRITABLE     2    /* Corresponding to the tk definition */
#define TK_EXCEPTION    4    /* Corresponding to the tk definition */

typedef void (Tk_FileProc)  _ANSI_ARGS_((ClientData clientData, int mask));
typedef void (Tk_TimerProc) _ANSI_ARGS_((ClientData clientData));

typedef struct Tk_TimerToken_ *Tk_TimerToken;

extern Tk_TimerToken    Tk_CreateTimerHandler _ANSI_ARGS_((int milliseconds,
                            Tk_TimerProc *proc, ClientData clientData));
extern void             Tk_DeleteTimerHandler _ANSI_ARGS_((
                            Tk_TimerToken token));
extern void             Tk_CreateFileHandler _ANSI_ARGS_((int fd, int mask,
                            Tk_FileProc *proc, ClientData clientData));
extern void             Tk_DeleteFileHandler _ANSI_ARGS_((int fd));


typedef struct cmipHandle {
    char		*cmipHdl;	/* the cmip handle */
    int			 msd;	  /* association descriptor (filedescriptor) */
    char		*agent;		/* name of the agent */
    char		*host;		/* name of the host */
    Tcl_HashTable	*req_table;	/* A Tcl-Hashtable for requests */
    unsigned		 req_nr;	/* id number for the next request */
} cmipHandle;

typedef struct requestHandle {
    char		*reqHdl;	/* the request handle */
    int			 msd;
    unsigned		 request_id;	/* the request id */
    char		*reqId;
    char		*reqcallback;
    char		*reqerrorback;
    Tcl_DString		 dStrResult;	/* result that's being built */
} requestHandle;

typedef struct CmipSession {
    Tcl_Interp		*interp;
    char		*handle;
} CmipSession;

Tcl_HashTable	cmip_table;	/* Tcl-Hashtable for cmip sessions */


/*
 * The ined cmd and its utility functions to flush the ined queue.
 */

extern int
ined_cmd         _ANSI_ARGS_((ClientData clientData,
			      Tcl_Interp *interp,
			      int argc, char **argv));

extern char*
xstrdup          _ANSI_ARGS_((char *s));

/*===========================================================================*/
void
rh_free(rh)
    requestHandle	*rh;
{
    Tcl_DStringFree(&rh->dStrResult);
    free((char *) rh->reqId);
    free((char *) rh->reqHdl);
    if (rh->reqcallback) free((char *) rh->reqcallback);
    if (rh->reqerrorback) free((char *) rh->reqerrorback);
    free((char *) rh);
}

/*===========================================================================*/
/*
 * Trigger responses on a msd (session descriptor), i.e. a file descriptor
 * for a management association to an agent at a host.
 * if the descriptor gets readable, i.e. there is an incomming response, call
 * the wait() routine, to catch the response and evaluate it.
 */

void
trigger_msd(clientdata, mask)
    ClientData	clientdata;
    int		mask;
{
    CmipSession	*cmip_sess = (CmipSession *) clientdata;

    wait(cmip_sess->handle, cmip_sess->interp, (requestHandle *) NULL);
/* return value ignored cause its asynchronous */
}


/*===========================================================================*/

void
cmip_release(clientData)
    ClientData	clientData;	/* is a char* to the cmip handle */
{
    char	*cmipHdl = (char *) clientData;

    cmipHandle		*cmiph;
    Tcl_HashEntry	*ht_entry;
    Tcl_HashSearch	 search;

    ht_entry = Tcl_FindHashEntry(&cmip_table, cmipHdl);
    if (ht_entry == NULL) {
	/* shouldn't really happen */
	/* "cmip_release(): no such cmip handle \"", cmipHdl, "\"" */
    } else {
	cmiph = (cmipHandle *) Tcl_GetHashValue(ht_entry);

	if (m_terminate(cmiph->msd) == NOTOK) {
	    /* nevertheless this association will no longer exist ??? */
	    /* "cmip_release(): unable to release connection from agent \"",
	       cmiph->agent, "\" at host \"", cmiph->host, "\"" */
	}
	Tk_DeleteFileHandler(cmiph->msd);
	Tcl_DeleteHashEntry (ht_entry);

	/* delete structures of all outstanding requests */
	for (ht_entry = Tcl_FirstHashEntry(cmiph->req_table, &search);
	     ht_entry != NULL;
	     ht_entry = Tcl_NextHashEntry(&search)) {
	    rh_free((requestHandle *) Tcl_GetHashValue(ht_entry));
	}
	free(cmiph->cmipHdl);
	free(cmiph->agent);
	free(cmiph->host);
	free((char *) cmiph->req_table);
	free((char *) cmiph);

#ifdef DEBUG_CMIP
    fprintf(stderr, "connection to %s at %s released!\n",
	    cmiph->agent, cmiph->host);
#endif
    }
} /* cmip_release() */

/*===========================================================================*/
int
assoc_cmd(clientData, interp, argc, argv)
    ClientData   clientData;	/* is a char* to the cmip handle */
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    if (argc < 2) {
        Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
                          " option ?arg arg ...?\"", (char *) NULL);
        return TCL_ERROR;
    }

    if (strcmp(argv[1], "get") == 0) {
	return cmip_get(clientData, interp, argc, argv);
    } else if (strcmp(argv[1], "set") == 0) {
	return cmip_set(clientData, interp, argc, argv);
    } else if (strcmp(argv[1], "action") == 0) {
	return cmip_action(clientData, interp, argc, argv);
    } else if (strcmp(argv[1], "create") == 0) {
	return cmip_create(clientData, interp, argc, argv);
    } else if (strcmp(argv[1], "delete") == 0) {
	return cmip_delete(clientData, interp, argc, argv);
    } else if (strcmp(argv[1], "cancelGet") == 0) {
	return cmip_cancelGet(clientData, interp, argc, argv);
    } else if (strcmp(argv[1], "eventReport") == 0) {
        return cmip_eventReport(clientData, interp, argc, argv);
    } else if (strcmp(argv[1], "requests") == 0) {
	/* list all outstanding requests, so the asynchronous requests */
	char	*cmipHdl = (char *) clientData;

	Tcl_HashEntry	*ht_entry;
	Tcl_HashSearch	 search;
	cmipHandle	*cmiph;
	requestHandle	*rh;

	ht_entry = Tcl_FindHashEntry(&cmip_table, cmipHdl);
	if (ht_entry == NULL) {
	    Tcl_AppendResult(interp, "no such cmip handle \"", cmipHdl, "\"",
			      (char *) NULL);
	    return TCL_ERROR;
	}
	cmiph = (cmipHandle *) Tcl_GetHashValue(ht_entry);

	for (ht_entry = Tcl_FirstHashEntry(cmiph->req_table, &search);
	     ht_entry != NULL;
	     ht_entry = Tcl_NextHashEntry(&search)) {
	    rh = (requestHandle *) Tcl_GetHashValue(ht_entry);
	    Tcl_AppendElement(interp,
			      Tcl_GetHashKey(cmiph->req_table, ht_entry));
	}
	return TCL_OK;
    } else if (strcmp(argv[1], "release") == 0) {
	cmip_release(clientData);
	return TCL_OK;
    }
    Tcl_AppendResult(interp, "bad option \"", argv[1], "\": should be ",
		      "get, set, action, create, delete, cancelGet, ",
		      "eventReport, requests, release",
		      (char *) NULL);
    return TCL_ERROR;
} /* assoc_cmd() */

/*===========================================================================*/
/*
 * Connect to a given agent running on host.
 */

int
cmip_connect(interp, argc, argv)
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    /*
     * lastcmipId wird nur beim ersten Aufruf der Funktion auf Null gesetzt 
     */
    static unsigned lastcmipId = 0; /* id number for the cmip handle */
    cmipHandle *cmiph;
    Tcl_HashEntry *ht_entry;
    int flag = 0, msd;

    CmipSession	*cmip_sess;

    char	buf[256];

    if (argc != 4) {
        Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0], " ",
                          argv[1], " agent host\"", (char *)NULL);
        return TCL_ERROR;
    }

/* falls moeglich erstmal pruefen, ob agent und host existieren */
    /* now try to connect to the remote agent at given host */
    if ((msd = m_initialise(argv[2], argv[3])) == NOTOK) {
        Tcl_AppendResult(interp, "can not connect to agent \"", argv[2],
			  "\" at host \"", argv[3], "\"",
			  (char *)NULL);
        return TCL_ERROR;
    }

    cmiph = (cmipHandle *) xmalloc(sizeof(cmipHandle));
    /* the result-string gets a handle for the management association:
     * cmip0, cmip1, cmip2, ...
     */
    sprintf(buf, "cmip%d", lastcmipId++);
    cmiph->cmipHdl	= xstrdup(buf);
    cmiph->msd		= msd;
    cmiph->agent	= xstrdup(argv[2]);
    cmiph->host		= xstrdup(argv[3]);
    cmiph->req_table	= (Tcl_HashTable *) xmalloc(sizeof(Tcl_HashTable));
    Tcl_InitHashTable(cmiph->req_table, TCL_STRING_KEYS);
    cmiph->req_nr	= 1;

    ht_entry = Tcl_CreateHashEntry(&cmip_table, cmiph->cmipHdl, &flag);
    if (flag != 1) {
        /* shouldn't really happen */
	Tcl_AppendResult(interp, "*** cmip handle already exist!? ***",
			  (char *)NULL);
        return TCL_ERROR;
    }
    Tcl_SetHashValue(ht_entry, (ClientData) cmiph);

    cmip_sess = (CmipSession *) xmalloc(sizeof(CmipSession));
    cmip_sess->interp = interp;
    cmip_sess->handle = cmiph->cmipHdl;

    Tk_CreateFileHandler(msd, TK_READABLE, trigger_msd,
			 (ClientData) cmip_sess);

    /* the cmip handle becomes a new Tcl-command */
    interp->result = xstrdup(buf); /* cmip handle */
    Tcl_CreateCommand(interp, interp->result, assoc_cmd,
		      (ClientData) interp->result, cmip_release);
    return TCL_OK;
} /* cmip_connect() */

/*===========================================================================*/
/*
 * This is the cmip command as described in the scotty documentation.
 * It simply dispatches to the C functions implementing the options
 * understood by the cmip command.
 */

int
cmip_cmd(clientData, interp, argc, argv)
    ClientData   clientData; /* bisher nicht genutzt */
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    if (argc < 2) {
        Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
                          " option ?arg arg ...?\"", (char *) NULL);
        return TCL_ERROR;
    }

    if (strcmp(argv[1], "connect") == 0) {
	return cmip_connect(interp, argc, argv);
    } else if (strcmp(argv[1], "associations") == 0) {
	/* list all associations, so the new commands */
	Tcl_HashEntry	*ht_entry;
	Tcl_HashSearch	search;
	cmipHandle	*cmiph;
	for (ht_entry = Tcl_FirstHashEntry(&cmip_table, &search);
	     ht_entry != NULL;
	     ht_entry = Tcl_NextHashEntry(&search)) {
	    cmiph = (cmipHandle *) Tcl_GetHashValue(ht_entry);
	    Tcl_AppendElement(interp, Tcl_GetHashKey(&cmip_table, ht_entry));
	}
	return TCL_OK;
    }

    Tcl_AppendResult(interp, "bad option \"", argv[1], "\": should be ",
		      "connect, associations",
		      (char *) NULL);
    return TCL_ERROR;
} /* cmip_cmd() */

/*===========================================================================*/

/*  <--- f"ur neue Funktion l"oschen
int
...(interp, argc, argv)
    Tcl_Interp* interp;
    int argc;
    char** argv;
{
} * ...() */

/*===========================================================================*/

int
CMIP_Init(interp)
    Tcl_Interp *interp;
{
/* load syntaxes */
    if (initialiseSyntaxes(NULLCP, attrSyntaxes) == NOTOK) {
        Tcl_AppendResult(interp, "CMIP_Init (): cannot initialise syntaxes",
                          (char *) NULL);
	return TCL_ERROR;
    }

/* Init the Hash-table for CMIP-connections */
    Tcl_InitHashTable(&cmip_table, TCL_STRING_KEYS);

/* register new Tcl-command CMIP */
    Tcl_CreateCommand(interp, "cmip", cmip_cmd,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

/* for the use of the ined commands */
#if 0
    Tcl_CreateCommand(interp, "ined", ined_cmd,
                       (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
#endif

/* everything worked fine! */
    return TCL_OK;

} /* CMIP_Init() */
