/*
 * rpc.c
 *
 * Implementation of commands that implement a simple RPC protocol
 * on top of a (reliable) tcp file handle.
 *
 * Copyright (c) 1993, 1994
 *
 * J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#define DIRECT YES

#include <stdio.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif

#include <string.h>
#include <ctype.h>
#include <tcl.h>

#include <scotty.h>
#include "xmalloc.h"
#include "xread.h"

typedef struct rpcHandle {
    char *fileId;               /* the TCL file id of the connection */
    char *rpcId;                /* the rpc handle */
    int  pargc;                 /* number of exported proc names */
    char **pargv;               /* the array of exported proc names */
    struct rpcHandle *server;   /* a pointer to the server handle. it is
			           used by handles representing connection
				   to a server */
} rpcHandle;

/* We use a hash table to map rpc handle ids to rpcHandle pointer. */

static Tcl_HashTable rpc_table;
static int initialized = 0;

/* The types of the RPC messages send between clients and server. */

#define RPC_NONE  0
#define RPC_CALL  1
#define RPC_REPLY 2
#define RPC_MSG   3
#define RPC_ERR   4

/*
 * The clientData used in the callback functions.
 */

typedef struct rpcClientData {
    char *rpcId;
    Tcl_Interp *interp;
} rpcClientData;

/*
 * Forward declarations for procedures defined later in this file:
 */

static rpcHandle*
create_rpc_handle	_ANSI_ARGS_((char *name));

static void 
delete_rpc_handle	_ANSI_ARGS_((ClientData clientData));

static int
read_message		_ANSI_ARGS_((char **msg, FILE *file, int *type));

static int
write_message		_ANSI_ARGS_((char *msg, FILE *file, int type));

static int
rpc_client		_ANSI_ARGS_((ClientData clientData, 
				     Tcl_Interp *interp, 
				     int argc, char **argv));
static int
rpc_server		_ANSI_ARGS_((ClientData clientData, 
				     Tcl_Interp *interp, 
				     int argc, char **argv));
static int
rpc_delete		_ANSI_ARGS_((ClientData clientData, 
				     Tcl_Interp *interp, 
				     int argc, char **argv));
static int 
rpc_make_call		_ANSI_ARGS_((ClientData clientData, 
				     Tcl_Interp *interp,
				     int argc, char **argv));
static int
rpc_register		_ANSI_ARGS_((ClientData clientData, 
				     Tcl_Interp *interp, 
				     int argc, char **argv));
static int
rpc_unregister		_ANSI_ARGS_((ClientData clientData,
				     Tcl_Interp *interp, 
				     int argc, char **argv));
static int
rpc_info		_ANSI_ARGS_((ClientData clientData,
				     Tcl_Interp *interp, 
				     int argc, char **argv));
#ifdef DIRECT

static void
n_rpc_accept		_ANSI_ARGS_((ClientData clientData, int mask));

static void
n_rpc_process		_ANSI_ARGS_((ClientData clientData, int mask));

#else

static int
rpc_accept		_ANSI_ARGS_((ClientData clientData,
				     Tcl_Interp *interp, 
				     int argc, char **argv));

static int
rpc_process		_ANSI_ARGS_((ClientData clientData, 
				     Tcl_Interp *interp, 
				     int argc, char **argv));

#endif

/*
 * Allocate a new rpc handle. Rpc client handles may be used to
 * make rpc calls and rpc server handles are used to register
 * commands for a server port.
 */

static rpcHandle*
create_rpc_handle (file)
    char *file;
{
    char buffer[20];
    static unsigned lastid = 0;
    rpcHandle *rh;
    Tcl_HashEntry *ht_entry;
    int flag;

    sprintf(buffer, "rpc%d", lastid++);

    rh = (rpcHandle *) xmalloc(sizeof(rpcHandle));

    rh->fileId = xstrdup(file);
    rh->rpcId  = xstrdup(buffer);
    rh->pargc = 0;
    rh->pargv = (char **) (xmalloc(1));
    rh->server = (rpcHandle *) NULL;

    /* throw the new handle in the hash table */

    ht_entry = Tcl_CreateHashEntry (&rpc_table, rh->rpcId, &flag);
    Tcl_SetHashValue (ht_entry, (ClientData) rh);
    
    return rh;
}

/*
 * Delete an rpc handle object. Free everything allocated before 
 * destroying the structure.
 */

static void 
delete_rpc_handle (clientData)
     ClientData clientData;
{
    rpcHandle *rh = (rpcHandle *) clientData;
    Tcl_HashEntry *ht_entry;
    Tcl_HashSearch ht_search;
    rpcHandle *crh;

    /* scan through all rpc handles that point to 
       this server rpc handle and set the link to NULL */

    ht_entry = Tcl_FirstHashEntry(&rpc_table, &ht_search);
    while (ht_entry != NULL) {
	crh = (rpcHandle *) Tcl_GetHashValue(ht_entry);
	if (crh->server == rh) {
	    crh->server = (rpcHandle *) NULL;
	}
	ht_entry = Tcl_NextHashEntry (&ht_search);
    }

    free (rh->fileId);
    free (rh->rpcId);

    free ((char *) rh->pargv);
    
    free ((char *) rh);
}

/*
 * Read a message from the stream given by file and return
 * the type and the message.
 */

static int
read_message (msg, file, type)
    char **msg;
    FILE *file;
    int *type;
{
    Tcl_DString dst;
    char buffer[256];
    int len;
    char *p;
    char *b;

    Tcl_DStringInit (&dst);
    do {
	if ((len = xread (fileno (file), buffer, 256)) < 0) {
	    Tcl_DStringFree (&dst);
	    return TCL_ERROR;
	}
	Tcl_DStringAppend (&dst, buffer, len);
    } while (! Tcl_CommandComplete (Tcl_DStringValue (&dst)));
	
    /* Eliminate the leading and the trailing braces, extract the 
     * type information (encoded as the first number) and allocate 
     * a copy of the message.
     */

    p = Tcl_DStringValue (&dst);
    if (*p != '{') {
	Tcl_DStringFree (&dst);
	return TCL_ERROR;
    }

    p++; while (isspace(*p)) p++;        /* remove leading white space */

    b = p; while (isdigit(*p)) p++;      /* scan over the type field */

    *p = '\0'; *type = atoi(b);          /* convert to an integer value */

    *msg = xstrdup (++p);
    (*msg)[strlen(*msg)-1] = '\0';

    Tcl_DStringFree (&dst);

    return TCL_OK;
}

/*
 * Write a message on the stream given by file.
 */

static int
write_message (msg, file, type)
    char *msg;
    FILE *file;
    int type;
{
    char buffer[80];
    Tcl_DString dst;
    int rc, fd = fileno (file);

    sprintf (buffer, "%d", type);

    Tcl_DStringInit (&dst);
    Tcl_DStringStartSublist (&dst);
    Tcl_DStringAppendElement (&dst, buffer);
    Tcl_DStringAppend (&dst, " ", 1);
    Tcl_DStringAppend (&dst, msg, -1);
    Tcl_DStringEndSublist (&dst);

    rc = xwrite (fd, Tcl_DStringValue (&dst), Tcl_DStringLength (&dst));

    Tcl_DStringFree (&dst);

    if (rc < 0) return TCL_ERROR;

    return TCL_OK;
}

/*
 * Create a client rpc handle. The parameter defines the tcl file to use.
 */

static int
rpc_client (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    char *fileId;
    rpcHandle *rh;
    FILE *filePtr;

    if (argc != 4) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", 
			  argv[0], " client host port\"", (char *) NULL);
        return TCL_ERROR;
    }

    if (Tcl_VarEval (interp, "tcp connect ", argv[2], " ", argv[3], 
		     (char *) NULL) != TCL_OK) {
	return TCL_ERROR;
    }
    fileId = xstrdup (interp->result);

    Tcl_ResetResult (interp);

    if (Tcl_GetOpenFile(interp, fileId, 1, 1, &filePtr) != TCL_OK) {
	free (fileId);
 	return TCL_ERROR;
    }

    /* Create a handle and a tcl command for the new object */

    rh = create_rpc_handle (fileId);

    Tcl_CreateCommand (interp, rh->rpcId, rpc_make_call, 
		       (ClientData) rh, delete_rpc_handle);

    Tcl_SetResult (interp, rh->rpcId, TCL_VOLATILE);

    free (fileId);

    return TCL_OK;
}

/*
 * Create a new rpc server. Return a rpc server handle that can
 * be used to register or unregister commands for this service.
 */

static int
rpc_server (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    rpcHandle *rh;
    char *fileId;
    int res;

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: should be \"",
                          argv[0], " server port\"", (char *) NULL);
        return TCL_ERROR;
    }

    res = Tcl_VarEval (interp, "tcp listen ", argv[2], (char *) NULL);
    if (res != TCL_OK) 	return res;

    fileId = xstrdup(interp->result);

    Tcl_ResetResult (interp);

    rh = create_rpc_handle (fileId);

#ifdef DIRECT

    {
	rpcClientData *rcd;
	FILE *filePtr;

	if (Tcl_GetOpenFile (interp, rh->fileId, 1, 1, &filePtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	rcd = (rpcClientData *) xmalloc (sizeof (rpcClientData));
	rcd->rpcId = xstrdup (rh->rpcId);
	rcd->interp = interp;

	Tk_CreateFileHandler (fileno(filePtr), TK_READABLE, 
			      n_rpc_accept, (ClientData) rcd);
    }

#else

    res = Tcl_VarEval (interp, "addinput ", fileId, 
		       " \"rpc accept %F ", rh->rpcId, "\"", 
		       (char *) NULL);

    if (res != TCL_OK) {
	delete_rpc_handle (rh);
	return res;
    }
#endif

    Tcl_SetResult (interp, rh->rpcId, TCL_STATIC);

    free (fileId);

    return TCL_OK;
}

/*
 * Delete an existing rpc handle. No further parameters are expected.
 */

static int
rpc_delete (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Tcl_HashEntry *ht_entry;
    Tcl_CmdInfo info;
    rpcHandle *rh;
#ifdef DIRECT
    FILE *filePtr;
#endif

    if (argc != 3) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", 
			  argv[0], " delete rpchandle\"", (char *) NULL);
        return TCL_ERROR;
    }

    ht_entry = Tcl_FindHashEntry(&rpc_table, argv[2]);
    if (ht_entry == NULL) {
	Tcl_AppendResult (interp, "no rpc handle \"", argv[2], "\"",
			  (char *) NULL);
	return TCL_ERROR;
    }
    
    rh = (rpcHandle *) Tcl_GetHashValue (ht_entry);

    Tcl_VarEval (interp, "tcp close ", rh->fileId, (char *) NULL);
#ifdef DIRECT
    if (Tcl_GetOpenFile (interp, rh->fileId, 1, 1, &filePtr) == TCL_OK) {
	Tk_DeleteFileHandler (fileno (filePtr));
    }
#else
    Tcl_VarEval (interp, "removeinput ", rh->fileId, (char *) NULL);
#endif
    Tcl_ResetResult (interp);

    if (Tcl_GetCommandInfo (interp, argv[2], &info) == 0) {
	delete_rpc_handle ((rpcHandle *) (Tcl_GetHashValue (ht_entry)));
    } else {
	Tcl_DeleteCommand (interp, argv[2]);
    }

    Tcl_DeleteHashEntry (ht_entry);

    return TCL_OK;
}

/*
 * Make an rpc call to the server. The parameters are send
 * to a server proc given by the first mandatory argument.
 *
 * NOTE: Don't change the name to rpc_call! At least solaris
 * uses this name in its C library.
 */

static int 
rpc_make_call (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    rpcHandle *rh = (rpcHandle *) clientData;
    char *msg;
    FILE *filePtr;
    int type;

    if (argc < 2) {	
	Tcl_AppendResult (interp, "wrong # args: should be \"", 
			  argv[0], " ?-async? cmd ?args?\"", 
			  (char *) NULL);
        return TCL_ERROR;
    }

    if (strcmp (argv[1], "-async") == 0) {
	type = RPC_MSG;
	if (argc < 3) {
	    Tcl_AppendResult (interp, "wrong # args: should be \"",
			      argv[0], " ?-async? cmd ?args?\"", 
			      (char *) NULL);
	    return TCL_ERROR;
	}
	argc--;
	argv++;
    } else {
	type = RPC_CALL;
    }

    if (Tcl_GetOpenFile(interp, rh->fileId, 1, 1, &filePtr) != TCL_OK) {
 	return TCL_ERROR;
    }

    /* Write the command to the stream. */

    msg = Tcl_Concat (argc-1, argv+1);
    if (write_message (msg, filePtr, type) != TCL_OK) {
	Tcl_ResetResult (interp);
	Tcl_AppendResult (interp, "can not send RPC message to \"",
			  rh->fileId, "\": ",Tcl_PosixError (interp),
			  (char *) NULL);
        return TCL_ERROR;
    }
    free (msg);

    /* Fetch the answer from the stream. */

    do {
	if (read_message (&msg, filePtr, &type) != TCL_OK) {
	    Tcl_SetResult (interp, "lost connection to rpc server", 
			   TCL_STATIC);
	    return TCL_ERROR;
	}
    } while ((type != RPC_REPLY) && (type != RPC_ERR));

#ifdef DBMALLOC
    Tcl_SetResult (interp, msg, TCL_VOLATILE);
    free (msg);
#else
    Tcl_SetResult (interp, msg, TCL_DYNAMIC);
#endif

    if (type == RPC_REPLY) {
	return TCL_OK;
    } else {
	return TCL_ERROR;
    }
}

/*
 * Export a proc on a rpc handle. The parameter defines the
 * proc to export.
 */

static int
rpc_register (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Tcl_HashEntry *ht_entry;
    rpcHandle *rh;
    char *list;
    Tcl_DString dstr;

    if (argc != 4) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " register rpchandle proc\"", (char *) NULL);
        return TCL_ERROR;
    }

    ht_entry = Tcl_FindHashEntry(&rpc_table, argv[2]);
    if (ht_entry == NULL) {
	Tcl_AppendResult (interp, "no rpc handle \"", argv[2], "\"",
			  (char *) NULL);
	return TCL_ERROR;
    }
    rh = (rpcHandle *) Tcl_GetHashValue(ht_entry);

    Tcl_DStringInit (&dstr);
    list = Tcl_Merge (rh->pargc, rh->pargv);
    Tcl_DStringAppend (&dstr, list, -1);
    Tcl_DStringAppendElement (&dstr, argv[3]);
    free (list);
    free (rh->pargv);
    Tcl_SplitList (interp, Tcl_DStringValue (&dstr), 
		   &(rh->pargc), &(rh->pargv));
    Tcl_DStringFree (&dstr);

    return TCL_OK;
}

/*
 * Remove an exported proc from a rpc handle. The parameter 
 * defines the proc to remove.
 */

static int
rpc_unregister (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Tcl_HashEntry *ht_entry;
    rpcHandle *rh;
    int i;
    Tcl_DString dstr;

    if (argc != 4) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " unregister rpchandle proc\"", (char *) NULL);
        return TCL_ERROR;
    }

    ht_entry = Tcl_FindHashEntry(&rpc_table, argv[2]);
    if (ht_entry == NULL) {
	Tcl_AppendResult (interp, "no rpc handle \"", argv[2], "\"",
			  (char *) NULL);
	return TCL_ERROR;
    }
    rh = (rpcHandle *) Tcl_GetHashValue(ht_entry);

    Tcl_DStringInit (&dstr);
    for (i = 0; i < rh->pargc; i++) {
	if (strcmp(argv[3], rh->pargv[i]) != 0) {
	    Tcl_DStringAppendElement (&dstr, rh->pargv[i]);
	}
    }
    free (rh->pargv);
    Tcl_SplitList (interp, Tcl_DStringValue (&dstr),
                   &(rh->pargc), &(rh->pargv));
    Tcl_DStringFree (&dstr);

    return TCL_OK;
}

/*
 * Return some information about existing rpc handles.
 */

static int
rpc_info (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Tcl_HashEntry *ht_entry;
    Tcl_HashSearch ht_search;

    rpcHandle *rh;

    if (argc < 2 || argc > 3) {
        Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
                          " info ?rpchandle?\"", (char *) NULL);
        return TCL_ERROR;
    }

    if (argc == 3) {

	char *regcmds;
	Tcl_DString dst;
	rpcHandle *arh;

	ht_entry = Tcl_FindHashEntry(&rpc_table, argv[2]);
	if (ht_entry == NULL) {
	    Tcl_AppendResult (interp, "no rpc handle \"", argv[2], "\"",
			      (char *) NULL);
	    return TCL_ERROR;
	}

	rh = (rpcHandle *) Tcl_GetHashValue(ht_entry);

	if (rh->server) {
	    Tcl_AppendResult (interp, "no rpc handle \"", argv[2], "\"",
                              (char *) NULL);
            return TCL_ERROR;
	}

	Tcl_DStringInit (&dst);
	Tcl_DStringAppendElement (&dst, rh->fileId);

	regcmds = Tcl_Merge (rh->pargc, rh->pargv);
	Tcl_DStringAppendElement (&dst, regcmds);
	free (regcmds);

	Tcl_DStringStartSublist (&dst);
	ht_entry = Tcl_FirstHashEntry(&rpc_table, &ht_search);
        while (ht_entry != NULL) {
            arh = (rpcHandle *) Tcl_GetHashValue(ht_entry);
            if (arh->server == rh) {
		Tcl_DStringAppendElement (&dst, arh->fileId);
	    }
            ht_entry = Tcl_NextHashEntry (&ht_search);
        }
	Tcl_DStringEndSublist (&dst);

	Tcl_DStringResult (interp, &dst);
	
    } else {

	/* scan through all rpc handles that point to
	   this server rpc handle and set the link to NULL */
	
	ht_entry = Tcl_FirstHashEntry(&rpc_table, &ht_search);
	while (ht_entry != NULL) {
	    rh = (rpcHandle *) Tcl_GetHashValue(ht_entry);
	    if (! rh->server) Tcl_AppendElement (interp, rh->rpcId);
	    ht_entry = Tcl_NextHashEntry (&ht_search);
	}
    }

    return TCL_OK;
}

/*
 * This command procedure is called whenever a new connection on
 * a rpc file handle is accepted.
 */

#ifdef DIRECT

static void
n_rpc_accept (clientData, mask)
    ClientData clientData;
    int mask;
{
    rpcClientData *rcd = (rpcClientData *) clientData;
    Tcl_Interp *interp = rcd->interp;
    FILE *filePtr;
    Tcl_HashEntry *ht_entry;
    rpcHandle *s_rh;
    rpcHandle *c_rh;

    /* Get the rpc handle if it is vaild. */

    ht_entry = Tcl_FindHashEntry (&rpc_table, rcd->rpcId);
    if (ht_entry == NULL) return;

    s_rh = (rpcHandle *) Tcl_GetHashValue (ht_entry);

    Tcl_ResetResult (interp);
    if (Tcl_VarEval(interp, "tcp accept ", s_rh->fileId, 
		    (char *) NULL) != TCL_OK) {
	return;
    }
    
    c_rh = create_rpc_handle (interp->result);
    c_rh->server = s_rh;

    {
	rpcClientData *rcd;

	if (Tcl_GetOpenFile (interp, c_rh->fileId, 1, 1, &filePtr) != TCL_OK) {
	    return;
	}
	rcd = (rpcClientData *) xmalloc (sizeof (rpcClientData));
	rcd->rpcId = xstrdup (c_rh->rpcId);
	rcd->interp = interp;

	Tk_CreateFileHandler (fileno(filePtr), TK_READABLE, 
			      n_rpc_process, (ClientData) rcd);
    }
}

#else

static int
rpc_accept (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    FILE *filePtr;
    Tcl_HashEntry *ht_entry;
    rpcHandle *s_rh;
    rpcHandle *c_rh;

    if (argc != 4) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0], 
			  " accept file rpchandle\"", (char *) NULL);
        return TCL_ERROR;
    }

    if (Tcl_GetOpenFile(interp, argv[2], 1, 1, &filePtr) != TCL_OK) {
 	return TCL_ERROR;
    }

    ht_entry = Tcl_FindHashEntry(&rpc_table, argv[3]);
    if (ht_entry == NULL) {
        Tcl_AppendResult (interp, "no rpc handle \"", argv[3], "\"",
                          (char *) NULL);
        return TCL_ERROR;
    }
    s_rh = (rpcHandle *) Tcl_GetHashValue(ht_entry);

    Tcl_ResetResult (interp);
    if (Tcl_VarEval(interp, "tcp accept ", argv[2], (char *) NULL) != TCL_OK) {
	return TCL_ERROR;
    }
    
    c_rh = create_rpc_handle (interp->result);
    c_rh->server = s_rh;

    Tcl_VarEval (interp, "addinput ", c_rh->fileId, " \"rpc process ",
		 c_rh->rpcId, "\"", (char *) NULL);

    return TCL_OK;
}

#endif

/*
 * This command procedure is called whenever a rpc request can be 
 * read from the underlying transport connection. The call will be 
 * evaluated and send a return msg depending of the message type.
 */


#ifdef DIRECT

static void
n_rpc_process (clientData, mask)
    ClientData clientData;
    int mask;
{
    rpcClientData *rcd = (rpcClientData *) clientData;
    Tcl_Interp *interp = rcd->interp;
    Tcl_HashEntry *ht_entry;
    rpcHandle *rh;
    FILE *filePtr;
    int type;
    char *msg = NULL;
    int msg_len;
    int i;
    int found;
    rpcHandle *server;

    /* Get the rpc handle if it is vaild. */

    ht_entry = Tcl_FindHashEntry (&rpc_table, rcd->rpcId);
    if (ht_entry == NULL) return;

    rh = (rpcHandle *) Tcl_GetHashValue (ht_entry);

    /* Check if we can get the open file structure. */

    if (Tcl_GetOpenFile (interp, rh->fileId, 1, 1, &filePtr) != TCL_OK) {
	return;
    }

    /* Read in the call message. */

    do {
	if (read_message (&msg, filePtr, &type) != TCL_OK) {
	    goto error;
	}
    } while ((type != RPC_CALL) && (type != RPC_MSG));

    /* Check if the command is valid for the rpc handle. */

    found = 0;
    server = rh->server;
    if (server != (rpcHandle *) NULL) {
	msg_len = strlen(msg);
	for (i=0, found = 0; i < server->pargc; i++) {
	    int len = strlen (server->pargv[i]);
	    if ((len <= msg_len) 
		&& (strncmp (msg, server->pargv[i], len) == 0)) {
		if ((msg[len] == '\0') || isspace (msg[len])) found++;
	    }
	}
    }

    if (!found) {
	if (write_message ("illegal rpc call", filePtr, RPC_ERR) != TCL_OK)
		goto error;
        free (msg);
	return;
    }

    /* Process the call either synchronous or asynchronous. */

    if (type == RPC_CALL) {
	type = (Tcl_Eval (interp, msg) == TCL_OK) ? RPC_REPLY : RPC_ERR;
	if (write_message (interp->result, filePtr, type) != TCL_OK) 
		goto error;
	free (msg);
	
    } else {
	if (write_message ("", filePtr, RPC_REPLY) != TCL_OK) 
		goto error;
        Tcl_Eval (interp, msg);
        free (msg);
    }

    return;

 error:
    if (msg != NULL) free (msg);
#ifdef DIRECT
    if (Tcl_GetOpenFile (interp, rh->fileId, 1, 1, &filePtr) == TCL_OK) {
	Tk_DeleteFileHandler (fileno (filePtr));
    }
#else
    Tcl_VarEval (interp, "removeinput ", rh->fileId, (char *) NULL);
#endif
    Tcl_ResetResult (interp);
    Tcl_AppendResult (interp, "can not send RPC message to \"",
		      rh->fileId, "\": ", Tcl_PosixError (interp),
		      (char *) NULL);
    return;
}

#else

static int
rpc_process (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Tcl_HashEntry *ht_entry;
    rpcHandle *rh;
    FILE *filePtr;
    int type;
    char *msg = NULL;
    int msg_len;
    int i;
    int found;
    rpcHandle *server;

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0], 
			  " process rpchandle\"", (char *) NULL);
        return TCL_ERROR;
    }

    /* Get the rpc handle if it is vaild. */

    ht_entry = Tcl_FindHashEntry (&rpc_table, argv[2]);
    if (ht_entry == NULL) {
	Tcl_AppendResult (interp, "no rpc handle \"", argv[2], "\"",
			  (char *) NULL);
	return TCL_ERROR;
    }
    rh = (rpcHandle *) Tcl_GetHashValue (ht_entry);

    /* Check if we can get the open file structure. */

    if (Tcl_GetOpenFile (interp, rh->fileId, 1, 1, &filePtr) != TCL_OK) {
	goto error;
    }

    /* Read in the call message. */

    do {
	if (read_message (&msg, filePtr, &type) != TCL_OK) {
	    Tcl_VarEval (interp, "removeinput ", rh->fileId, (char *) NULL);
	    Tcl_ResetResult (interp);
	    Tcl_AppendResult (interp, "can not send RPC message to \"",
			      rh->fileId, "\": ", Tcl_PosixError (interp),
			      (char *) NULL);
	    return TCL_ERROR;
	}
    } while ((type != RPC_CALL) && (type != RPC_MSG));

    /* Check if the command is valid for the rpc handle. */

    found = 0;
    server = rh->server;
    if (server != (rpcHandle *) NULL) {
	msg_len = strlen(msg);
	for (i=0, found = 0; i < server->pargc; i++) {
	    int len = strlen (server->pargv[i]);
	    if ((len <= msg_len) 
		&& (strncmp (msg, server->pargv[i], len) == 0)) {
		if ((msg[len] == '\0') || isspace (msg[len])) found++;
	    }
	}
    }

    if (!found) {
	if (write_message ("illegal rpc call", filePtr, RPC_ERR) != TCL_OK)
		goto error;
        free (msg);
	return TCL_OK;
    }

    /* Process the call either synchronous or asynchronous. */

    if (type == RPC_CALL) {
	type = (Tcl_Eval (interp, msg) == TCL_OK) ? RPC_REPLY : RPC_ERR;
	if (write_message (interp->result, filePtr, type) != TCL_OK) 
		goto error;
	free (msg);
	
    } else {
	if (write_message ("", filePtr, RPC_REPLY) != TCL_OK) 
		goto error;
        Tcl_Eval (interp, msg);
        free (msg);
    }

    return TCL_OK;

 error:
    if (msg != NULL) free (msg);
    Tcl_VarEval (interp, "removeinput ", rh->fileId, (char *) NULL);
    Tcl_ResetResult (interp);
    Tcl_AppendResult (interp, "can not send RPC message to \"",
		      rh->fileId, "\": ", Tcl_PosixError (interp),
		      (char *) NULL);
    return TCL_ERROR;
}

#endif

/*
 * This is the rpc command as described in the scotty documentation.
 * It simply dispatches to the C functions implementing the options
 * understood by the rpc command.
 */

int
rpcCmd (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int length;
    char c;

    if (argc < 2) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " option ?arg arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (! initialized) {
	Tcl_InitHashTable (&rpc_table, TCL_STRING_KEYS);
	initialized = 1;
    }

    c = argv[1][0];
    length = strlen (argv[1]);

    if ((c == 'c') && (strncmp(argv[1], "client", length) == 0)) {
	return rpc_client (clientData, interp, argc, argv);
    } else if ((c == 's') && (strncmp(argv[1], "server", length) == 0)) {
        return rpc_server (clientData, interp, argc, argv);
    } else if ((c == 'd') && (strncmp(argv[1], "delete", length) == 0)) {
        return rpc_delete (clientData, interp, argc, argv);
    } else if ((c == 'r') && (strncmp(argv[1], "register", length) == 0)) {
        return rpc_register (clientData, interp, argc, argv);
    } else if ((c == 'u') && (strncmp(argv[1], "unregister", length) == 0)) {
        return rpc_unregister (clientData, interp, argc, argv);
    } else if ((c == 'i') && (strncmp(argv[1], "info", length) == 0)) {
        return rpc_info (clientData, interp, argc, argv); 
#ifndef DIRECT
    } else if ((c == 'a') && (strncmp(argv[1], "accept", length) == 0)) {
        return rpc_accept (clientData, interp, argc, argv);
    } else if ((c == 'p') && (strncmp(argv[1], "process", length) == 0)) {
        return rpc_process (clientData, interp, argc, argv);
#endif
    }

    Tcl_AppendResult (interp, "bad option \"", argv[1], "\": should be ",
		      "client, server, register, unregister, delete, or info",
		      (char *) NULL);
    return TCL_ERROR;
}
