/*
 * scotty.c
 *
 * This is scotty, a simple tcl interpreter with some special commands
 * to get information about TCP/IP networks. 
 *
 * Copyright (c) 1993, 1994
 *
 * J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <errno.h>

#include <tcl.h>

#include "scotty.h"
#include "xmalloc.h"

#ifdef HAVE_TCLX
extern int
TclX_Init		_ANSI_ARGS_((Tcl_Interp *interp));
#endif

static int norc = 0;            /* norc -- dont read the initialization file */
static int ined = 0;            /* dont initialize for tkined */
static char *progname;          /* the name of the game */

static char *prompt = "";       /* the prompt */
static Tcl_Interp *interp;      /* the main tcl interpreter */

Tcl_Interp *sc_interp;          /* the main tcl interpreter used in event.c */

/*
 * Get one command and process it. This is the default stdin handler.
 */

static void
doOneCommand (clientData, mask)
    ClientData clientData;
    int mask;
{
    Tcl_Interp *interp = (Tcl_Interp *) clientData;
    char *line = NULL;
    char *cmd;
    int result;
    Tcl_DString buffer;
    Tcl_DStringInit (&buffer);

    while (1) {
	
	line = (char *) xmalloc(256);
	if (fgets(line, 256, stdin) == NULL) {
	    free (line);
	    Tcl_DStringFree (&buffer);
	    Tk_DeleteFileHandler (0);
            return;
	}

	cmd = Tcl_DStringAppend (&buffer, line, -1);
	cmd = Tcl_DStringAppend (&buffer, " ", 1);

	if (Tcl_CommandComplete (cmd)) break;
	if (isatty(0)) {
	    printf("> ");
	    fflush(stdout);
	}
    }
	
    if (!strcmp (cmd, "exit")) {
	Tcl_DStringFree (&buffer);
	Tk_DeleteFileHandler (0);
	return;
    }

#ifdef DBMALLOC
    /* lets have a look about memory leaks: */
    { static unsigned long mark1, mark2, size1, size2;
      size1 = malloc_inuse (&mark1);
#endif /* DBMALLOC */
    
    result = Tcl_Eval (interp, cmd);

#ifdef DBMALLOC
      size2 = malloc_inuse (&mark2);
      if (size1 != size2)
	  malloc_list (fileno (stderr), mark1, mark2);
    }
#endif /* DBMALLOC */

    if (result == TCL_OK) {
	if (*interp->result != 0) {
	    printf ("%s\n", interp->result);
	}
    } else {
	printf("Error ");
	if (*interp->result != 0) {
	    printf (": %s\n", interp->result);
	} else {
	    printf ("\n");
	}
    }

    printf(prompt);
    fflush(stdout);

    Tcl_DStringFree (&buffer);

    if (line != (char *) NULL) free (line);

    return;
}

/*
 * execute commands in $HOME/.scottyrc
 */

static void
read_scottyrc()
{
    char *home, *cmd;
    
    if ((home = getenv("HOME")) != NULL) {
	if ((cmd = (char *) xmalloc (strlen(home)+15))) {
	    sprintf (cmd, "%s/.scottyrc", home);
	    Tcl_EvalFile (interp, cmd);
	    free (cmd);
	}
    }
}

/*
 * Initialize a new interpreter.
 */

int
Tcl_AppInit (interp)
    Tcl_Interp *interp;
{
    if (Tcl_Init (interp) != TCL_OK) {
        return TCL_ERROR;
    }

#ifdef HAVE_TCLX
    if (TclX_Init (interp) != TCL_OK) {
        return TCL_ERROR;
    }
#endif

    /* these are here since tk < 4.0 needs its own version */

    Tcl_CreateCommand (interp, "after", afterCmd,
                       (ClientData) NULL, (void (*)()) NULL);
    Tcl_CreateCommand (interp, "addinput", addInputCmd,
                       (ClientData) NULL, (void (*)()) NULL);
    Tcl_CreateCommand (interp, "removeinput", remInputCmd,
                       (ClientData) NULL, (void (*)()) NULL);

    if (Scotty_Init (interp) != TCL_OK) {
        return TCL_ERROR;
    }
    
    return TCL_OK;
}

/*
 * here we start
 */

int
main(argc, argv)
    int argc;
    char **argv;
{
    char *filename = NULL;
    char *query = NULL;
    char buffer[20];
    Tcl_DString sc_argv;
    int i, c;
    int sc_argc = 0;
    int errflg = 0;
    int res;

    if ((progname = strrchr(argv[0], '/')) == NULL) {
	progname = argv[0];
    } else {
	progname++;
    }
    
    if (isatty(0)) {
	prompt = (char*) xmalloc (strlen(progname)+4);
	sprintf(prompt, "%s > ", progname);
    }

    Tcl_DStringInit (&sc_argv);

    for (i = 1; i < argc; i++) {
        if (argv[i][0] != '-') {
	    Tcl_DStringAppendElement (&sc_argv, argv[i]); 
	    sc_argc++;
	    continue;
	}
	for (c = 1; (i < argc) && (c < strlen(argv[i])); c++) {
	    switch (argv[i][c]) {
	      case 'c' :
		if ( (argv[i][c+1] == '\0') && (++i < argc) ) {
		    query = argv[i];
		    c = strlen(argv[i]);
		} else errflg++;
		break;
	      case 'f':
		if ( (argv[i][c+1] == '\0') && (++i < argc) ) {
		    filename = argv[i]; 
		    c = strlen(argv[i]);
		} else errflg++;
		break;
	      case 'n':
		norc++;
		break;
	      case 'i':
		ined++;
		break;
	      case '?':
		errflg++;
		break;
	      case '-':
		c = strlen(argv[i]);
		for (i++; i < argc; i++) {
		    Tcl_DStringAppendElement (&sc_argv, argv[i]);
		    sc_argc++;
		}
		break;
	      default:
		c = strlen(argv[i]);
		Tcl_DStringAppendElement (&sc_argv, argv[i]);
		sc_argc++;
		break;
	    }
	}
    }

    if (errflg) {
	fprintf (stderr, "usage: %s [-c query] [-f file] [-n] [-i] [-v]\n", 
		 progname);
	return 2;
    }


    /* Create and initialize the Tcl interpreter */

    interp = Tcl_CreateInterp();
    sc_interp = interp;

    /* Set the tcl_interactive variable */

    Tcl_SetVar (interp, "tcl_interactive", "0", TCL_GLOBAL_ONLY);

    /* Register all new commands. */

    if (Tcl_AppInit(interp) == TCL_ERROR) {
	fprintf(stderr, "Tcl_AppInit failed: %s\n", interp->result);
        return TCL_ERROR;
    }

    /* Set the global variables argc and argv */

    Tcl_SetVar (interp, "argv", Tcl_DStringValue(&sc_argv), TCL_GLOBAL_ONLY);
    Tcl_DStringFree (&sc_argv);
    sprintf (buffer, "%d", sc_argc);
    Tcl_SetVar (interp, "argc", buffer, TCL_GLOBAL_ONLY);

    /* first process the $HOME/.scottyrc */

    if (!norc) read_scottyrc();
    
    /* process a single query */

    if (query) {
	int result = Tcl_Eval (interp, query);
	if (result == TCL_OK) {
	    if (*interp->result != 0) {
		printf ("%s\n", interp->result);
	    }
	} else {
	    fprintf (stderr, "Error: %s\n%s\n", interp->result,
		     Tcl_GetVar (interp, "errorInfo", TCL_GLOBAL_ONLY));
	}
	eventLoop (interp);
	icmpClose ();
	return 0;
    }
    
    /* process queries from a file */

    if (filename) {

	if (ined) {
	    if (Tcl_Eval (interp, "ined size") != TCL_OK) {
		fprintf (stderr, "Fatal: can not talk to tkined: %s\n",
			 interp->result);
		exit (32);
	    }
	}

	res = Tcl_EvalFile (interp, filename);
	if (res != TCL_OK) {
	    fprintf (stderr, "Error: %s\n%s\n", interp->result,
		     Tcl_GetVar (interp, "errorInfo", TCL_GLOBAL_ONLY));
	}

	/* fall into the event loop. */
	eventLoop (interp);

	icmpClose ();
	return res;
    }
    
    /* ok, we are interactive -- that's where the fun beginns */

    if (isatty(0)) {
	Tcl_SetVar (interp, "tcl_interactive", "1", TCL_GLOBAL_ONLY);
    }
    printf (prompt);
    fflush (stdout);

    Tk_CreateFileHandler (0, TK_READABLE, doOneCommand, interp);    

    eventLoop (interp);
    icmpClose ();
    return 0;
}
