/*
 * snmp_agent.c
 *
 * This is the SNMP agent interface of scotty.
 *
 * Copyright (c) 1994
 *
 * J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <tcl.h>

#include "snmp.h"
#include "agent.h"
#include "xmalloc.h"
#include "misc.h"
#include "asn1.h"
#include "mib.h"

#define DEBUG

static struct instance *instTree = NULL;

/*
 * Forward declarations for procedures defined later in this file:
 */

static struct instance *
TreeAdd				_ANSI_ARGS_((char *id, char *syntax, 
					     int access, char *tclvar));
static void
TreeRemove			_ANSI_ARGS_((struct instance *tree,
					     char *varname));
static struct instance*
TreeFind			_ANSI_ARGS_((struct instance *tree,
					     oid *id, int len));
static struct instance*
TreeFindNext			_ANSI_ARGS_((struct instance *tree,
					     oid *id, int len));
static char*
Trace_SysUpTime			_ANSI_ARGS_((ClientData clientData, 
					     Tcl_Interp *interp, 
					     char *name1, char *name2, 
					     int flags));

static char*
Trace_UnsignedInt		_ANSI_ARGS_((ClientData clientData,
					     Tcl_Interp *interp, 
					     char *name1, char *name2, 
					     int flags));

static char*
Trace_DeleteInstance		_ANSI_ARGS_((ClientData clientData, 
					     Tcl_Interp *interp, 
					     char *name1, char *name2, 
					     int flags));

static void
Cache_Init			_ANSI_ARGS_((void));

static struct snmp_pdu*
Cache_Get			_ANSI_ARGS_((struct session *session));

static int
Cache_Hit			_ANSI_ARGS_((Tcl_Interp *interp, 
					     struct session *session, 
					     int id));

/*
 * The following structure is used to implement a cache that
 * is used to remember queries so that we can respond to
 * retries quickly. This is needed because side effects can
 * break the agent down if we do them for each retry.
 */

struct cache_elem {
    struct session *session;
    struct snmp_pdu pdu;
};

#define CACHE_SIZE 64
static struct cache_elem cache[CACHE_SIZE];

struct stat_regs {
    char *name;
    unsigned int *value;
};

static struct stat_regs stat_regs[] = {
    { "snmpInPkts.0",			&snmp_stats.snmpInPkts },
    { "snmpOutPkts.0",			&snmp_stats.snmpOutPkts },
    { "snmpInBadVersions.0",		&snmp_stats.snmpInBadVersions },
    { "snmpInBadCommunityNames.0",	&snmp_stats.snmpInBadCommunityNames },
    { "snmpInBadCommunityUses.0",	&snmp_stats.snmpInBadCommunityUses },
    { "snmpInASNParseErrs.0",		&snmp_stats.snmpInASNParseErrs },
    { "snmpInTooBigs.0",		&snmp_stats.snmpInTooBigs },
    { "snmpInNoSuchNames.0",		&snmp_stats.snmpInNoSuchNames },
    { "snmpInBadValues.0",		&snmp_stats.snmpInBadValues },
    { "snmpInReadOnlys.0",		&snmp_stats.snmpInReadOnlys },
    { "snmpInGenErrs.0",		&snmp_stats.snmpInGenErrs },
    { "snmpInTotalReqVars.0",		&snmp_stats.snmpInTotalReqVars },
    { "snmpInTotalSetVars.0",		&snmp_stats.snmpInTotalSetVars },
    { "snmpInGetRequests.0",		&snmp_stats.snmpInGetRequests },
    { "snmpInGetNexts.0",		&snmp_stats.snmpInGetNexts },
    { "snmpInSetRequests.0",		&snmp_stats.snmpInSetRequests },
    { "snmpInGetResponses.0",		&snmp_stats.snmpInGetResponses },
    { "snmpInTraps.0",			&snmp_stats.snmpInTraps },
    { "snmpOutTooBigs.0",		&snmp_stats.snmpOutTooBigs },
    { "snmpOutNoSuchNames.0",		&snmp_stats.snmpOutNoSuchNames },
    { "snmpOutBadValues.0"	,	&snmp_stats.snmpOutBadValues },
    { "snmpOutGenErrs.0",		&snmp_stats.snmpOutGenErrs },
    { "snmpOutGetRequests.0",		&snmp_stats.snmpOutGetRequests },
    { "snmpOutGetNexts.0",		&snmp_stats.snmpOutGetNexts },
    { "snmpOutSetRequests.0",		&snmp_stats.snmpOutSetRequests },
    { "snmpOutGetResponses.0",		&snmp_stats.snmpOutGetResponses },
    { "snmpOutTraps.0",			&snmp_stats.snmpOutTraps },
#if 0
    unsigned int snmpStatsPackets },
    unsigned int snmpStats30Something },
    unsigned int snmpStatsUnknownDstParties },
    unsigned int snmpStatsDstPartyMismatches },
    unsigned int snmpStatsUnknownSrcParties },
    unsigned int snmpStatsBadAuths },
    unsigned int snmpStatsNotInLifetimes },
    unsigned int snmpStatsWrongDigestValues },
    unsigned int snmpStatsUnknownContexts },
    unsigned int snmpStatsBadOperations },
    unsigned int snmpStatsSilentDrops },
#endif
    { "snmpV1BadCommunityNames.0",	&snmp_stats.snmpInBadCommunityNames },
    { "snmpV1BadCommunityUses.0",	&snmp_stats.snmpInBadCommunityUses },
    { 0, 0 }
};

/*
 * The global variable to keep the snmp statistics is defined here.
 */

struct snmp_stats snmp_stats;

#if 0
static void
dump_instances (root, deep)
     struct instance *root;
     int deep;
{
    int i;
    struct instance *brother;

    /*
     * print all nodes of this tree level
     */

    deep++;
    
    for (brother = root; brother != NULL; brother = brother->next) {
	for (i = 0; i < deep; i++) printf( "   " );
	printf ("%s : %s\n", brother->label, brother->tclvar);

	/*
         * for each node print the subtree
	 */

	if (brother->child != NULL)
	  dump_instances (brother->child, deep);
    }
}

void
agent_dump ()
{
    dump_instances (instTree, 0);
}
#endif

/*
 * AddToTree() add a new instance to the instTree.
 */

static struct instance *
TreeAdd (id, syntax, access, tclvar)
     char *id;
     char *syntax;
     int access;
     char *tclvar;
{
    oid *oidbuf;
    int i, oidlen;
    struct instance *p, *q = NULL, *n;

    if (instTree == NULL) {
	instTree = (struct instance *) xmalloc (sizeof (struct instance));
	memset ((char *) instTree, '\0', sizeof (struct instance));
	instTree->subid = 1;
    }

    oidbuf = SNMP_Str2Oid (id, &oidlen);
    if (oidbuf[0] != 1) return NULL;
    for (p = instTree, i = 1; i < oidlen; p = q, i++) {
	for (q = p->child; q; q = q->next) {
	    if (q->subid == oidbuf[i]) break;
	}
	if (! q) {
	    
	    /*
	     * Create new intermediate nodes.
	     */

	    n = (struct instance *) xmalloc (sizeof (struct instance));
	    memset ((char *) n, '\0', sizeof (struct instance));
	    n->subid = oidbuf[i];

	    if (! p->child) {				/* first node  */
		p->child = n;

	    } else if (p->child->subid > oidbuf[i]) {	/* insert head */
		    n->next = p->child;
		    p->child = n;

	    } else {					/* somewhere else */
		for (q = p->child; q->next && q->next->subid < oidbuf[i]; 
		     q = q->next) ;
		if (q->next && q->next->subid == oidbuf[i]) {
		    return q;				/* exists already */
		}
		n->next = q->next;
		q->next = n;
	    }

	    q = n;

	    /*
	     * Set the values if we have reached to leaf node.
	     */

	    if (i == oidlen - 1) {
		q->label  = id;
		q->syntax = syntax;
		q->access = access;
		q->tclvar = tclvar;
	    }
	}
    }

    return q;
}

/*
 * TreeFindNext() locates the lexikographic next instance in the tree. 
 */

static struct instance *
TreeFindNext (tree, id, len)
     struct instance *tree;
     oid *id;
     int len;
{
    struct instance *p, *inst;

    p = tree;
    if (len && id) {
	while (p && p->subid < id[0]) p = p->next;
    }

    while (p) {
	if (p->child) {
	    if (len > 0 && p->subid == id[0]) {
		inst = TreeFindNext (p->child, id + 1, len - 1);
	    } else {
		inst = TreeFindNext (p->child, NULL, 0);
	    }
	    if (inst && inst->syntax) return inst;
	} else {
	    if (len != 1 || p->subid != id[0]) return p;
	}
	p = p->next;
    }

    return NULL;
}

/*
 * TreeFind() locates an instance in the tree.
 */

static struct instance *
TreeFind (tree, id, len)
     struct instance *tree;
     oid *id;
     int len;
{
    struct instance *p, *q = NULL;
    int i;
    
    if (id[0] != 1) return NULL;
    for (p = tree, i = 1; i < len; p = q, i++) {
	for (q = p->child; q; q = q->next) {
	    if (q->subid == id[i]) break;
	}
	if (!q) {
	    return NULL; 
	}
    }
    return (q && q->syntax) ? q : NULL;
}

/*
 * TreeRemove() deletes all nodes from the tree that are associated
 * with a given tcl variable.
 */

static void
TreeRemove (tree, varname)
     struct instance *tree;
     char *varname;
{
    struct instance *p, *q;

    if (!tree) return;

    for (p = tree; p; p = p->next) {
	if (p->child) {
	    q = p->child;
	    TreeRemove (q, varname);
	    if (q->tclvar && (strcmp (q->tclvar, varname) == 0)) {
		p->child = q->next;
		free (q->label);
		free (q->syntax);
		free (q->tclvar);
		free ((char *) q);
	    }
	}
	if (p->next) {
	    q = p->next;
	    if (q->tclvar && (strcmp (q->tclvar, varname) == 0)) {
		p->next = q->next;
		free (q->label);
		free (q->syntax);
		free (q->tclvar);
		free ((char *) q);
	    }
	}
    }
}

/*
 * Trace_SysUpTime() is a trace callback which is called by the
 * tcl interpreter whenever the sysUpTime variable is read.
 */

static char*
Trace_SysUpTime (clientData, interp, name1, name2, flags)
     ClientData clientData;
     Tcl_Interp *interp;
     char *name1;
     char *name2;
     int flags;
{
    char buf[20];
    sprintf (buf, "%u", SNMP_SysUpTime());
    Tcl_SetVar (interp, name1, buf, TCL_GLOBAL_ONLY);
    return NULL;
}

/*
 * Trace_UnsignedInt writes the unsigned value pointed to by
 * clientData into the tcl variable under trace. Used to implement 
 * snmp statistics.
 */

static char*
Trace_UnsignedInt (clientData, interp, name1, name2, flags)
     ClientData clientData;
     Tcl_Interp *interp;
     char *name1;
     char *name2;
     int flags;
{
    char buf[20];
    sprintf (buf, "%u", *(unsigned *) clientData);
    Tcl_SetVar (interp, name1, buf, TCL_GLOBAL_ONLY);
    return NULL;    
}

/*
 * Trace_DeleteInstance() is a trace callback which is called by the
 * tcl interpreter whenever a MIB variable is removed. We have run through
 * the whole tree to discard these variables.
 */

static char*
Trace_DeleteInstance (clientData, interp, name1, name2, flags)
     ClientData clientData;
     Tcl_Interp *interp;
     char *name1;
     char *name2;
     int flags;
{
    int len = strlen (name1);
    char *foo;
			 
    if (name2) len += strlen (name2);
    foo = xmalloc (len + 3);
    strcpy (foo, name1);
    if (name2) {
	strcat (foo,"(");
	strcat (foo, name2);
	strcat (foo,")");
    }

    TreeRemove (instTree, foo);
    free (foo);
    return NULL;
}


/*
 * Cache_Init() initializes the cache. This is called only once when
 * we become an agent.
 */

static void
Cache_Init ()
{
    int i;

    memset ((char *) cache, '\0', sizeof (cache));
    for (i = 0; i <  CACHE_SIZE; i++) {
	Tcl_DStringInit (&cache[i].pdu.varbind);
    }
}


/*
 * Cache_Get() gets a free cache element. We search for a free entry
 * and if that fails, we remove everything older than 2 seconds. If
 * this fails, we remove a entry choosen by nearly random order.
 */

static struct snmp_pdu*
Cache_Get (session)
     struct session *session;
{
    static last = 0;

    last = (last + 1 ) % CACHE_SIZE;
    Tcl_DStringFree (&cache[last].pdu.varbind);
    cache[last].session = session;
    return &(cache[last].pdu);
}


/*
 * Cache_Hit() checks if the request identified by session and
 * request id is in the cache so we can send the answer wihtout
 * further processing. It returns a standard TCL result if found
 * and -1 if not hit was made.
 */

static int
Cache_Hit (interp, session, id)
     Tcl_Interp *interp;
     struct session *session;
     int id;
{
    int i;

    for (i = 0; i < CACHE_SIZE; i++) {
	if (cache[i].pdu.request_id == id && cache[i].session == session) {
#ifdef DEBUG
	    fprintf (stderr, "cache response (%d): %s\n", id,
		     Tcl_DStringValue (&cache[i].pdu.varbind));
#endif
	    return SNMP_Encode (interp, cache[i].session, &cache[i].pdu, 
				NULL, NULL);
	}
    }
    return -1;
}


/*
 * SNMP_AgentInit() initializes the agent by registering some 
 * default variables.  */

void
SNMP_AgentInit (interp)
     Tcl_Interp *interp;
{
    static done = 0;
    char tclvar[80];
    struct stat_regs *p;

    if (done) return;
    done = 1;

    Cache_Init();

    SNMP_CreateInstance (interp, "sysDescr.0", "sysDescr",
			 "scotty midlevel agent");
    SNMP_CreateInstance (interp, "sysObjectID.0", "sysObjectID",
			 "1.3.6.1.4.1.1701.1");
    SNMP_CreateInstance (interp, "sysUpTime.0", "sysUpTime", "0");
    Tcl_TraceVar (interp, "sysUpTime", TCL_TRACE_READS | TCL_GLOBAL_ONLY, 
		  Trace_SysUpTime, (ClientData) NULL);
    SNMP_CreateInstance (interp, "sysContact.0", "sysContact", "");
    SNMP_CreateInstance (interp, "sysName.0", "sysName", "");
    SNMP_CreateInstance (interp, "sysLocation.0", "sysLocation", "");
    SNMP_CreateInstance (interp, "sysServices.0", "sysServices", "72");
    SNMP_CreateInstance (interp, "scottyVersion.0", "scotty_version", NULL);

    for (p = stat_regs; p->name; p++) {
	strcpy (tclvar, p->name);
	SNMP_CreateInstance (interp, p->name, tclvar, "");
	Tcl_TraceVar (interp, tclvar, TCL_TRACE_READS | TCL_GLOBAL_ONLY,
		      Trace_UnsignedInt, (ClientData) p->value);
    }

    /* XXX snmpEnableAuthenTraps.0 should be implemented */

    Tcl_ResetResult (interp);
}


/*
 * CreateInstance() creates a new instance in the instance tree and
 * a tcl array variable that will be used to access and modify the
 * instance from within tcl.
 */

int
SNMP_CreateInstance (interp, label, tclvar, defval)
     Tcl_Interp *interp;
     char *label;
     char *tclvar;
     char *defval;
{
    char *id = NULL, *syntax = NULL;
    struct tree *node = MIB_FindNode (label, NULL, 0);
    int access;

    if (!node) {
	Tcl_AppendResult (interp, "no object \"", label, "\"", (char *) NULL);
	return TCL_ERROR;
    }

    fprintf (stderr, "SNMP_CreateInstance %s -> %s\n", label, tclvar);
    
    id = xstrdup (MIB_Oid (label, 0));

    if (node->tt && node->tt->name) {
	syntax = xstrdup (ASN1_Sntx2Str (node->tt->syntax));
    } else {
	syntax = xstrdup (ASN1_Sntx2Str (node->syntax));
    }

    access = node->access;
    if (access == M_NOACCESS) {
	Tcl_AppendResult (interp, "object \"", label, "\" is not accessible",
			  TCL_STATIC);
	goto errorExit;
    }

    if (defval) {
	if (Tcl_SetVar (interp, tclvar, defval, 
			TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG) == NULL) {
	    goto errorExit;
	}
    }

    TreeAdd (id, syntax, access, xstrdup (tclvar));
    Tcl_TraceVar (interp, tclvar, TCL_TRACE_UNSETS | TCL_GLOBAL_ONLY, 
		  Trace_DeleteInstance, (ClientData) NULL);
    Tcl_ResetResult (interp);
    return TCL_OK;

  errorExit:
    if (id) free (id);
    if (syntax) free (syntax);
    return TCL_ERROR;
}


/*
 * SNMP_AgentRequest() gets called when the agent receives a get or
 * set message. It split the varbind, looks up the variables and
 * assembles an answer.
 */

int
SNMP_AgentRequest (interp, session, pdu)
     Tcl_Interp *interp;
     struct session *session;
     struct snmp_pdu *pdu;
{
    char **vblv;
    int i, vblc;
    int exact = (pdu->type == GET_REQUEST || pdu->type == SET_REQUEST);
    int rc;
    struct snmp_pdu *reply;

#ifdef DEBUG
    switch (pdu->type) {
      case GET_REQUEST:		fprintf (stderr, "get"); break;
      case GET_NEXT_REQUEST:	fprintf (stderr, "getnext"); break;
      case GET_BULK_REQUEST:	fprintf (stderr, "getbulk"); break;
      case SET_REQUEST:		fprintf (stderr, "set"); break;
      case INFORM_REQUEST:	fprintf (stderr, "inform"); break;
    }
    fprintf (stderr, " (%d): %s\n", pdu->request_id, 
	     Tcl_DStringValue(&pdu->varbind));
#endif

    switch (pdu->type) {
      case GET_REQUEST:		snmp_stats.snmpInGetRequests++; break;
      case GET_NEXT_REQUEST:	snmp_stats.snmpInGetNexts++; break;
      case GET_BULK_REQUEST:	break;
      case SET_REQUEST:		snmp_stats.snmpInSetRequests++; break;
      case INFORM_REQUEST:	break;
    }

    if ((rc = Cache_Hit (interp, session, pdu->request_id)) >= 0) {
	return rc;
    }

    reply = Cache_Get (session);
    reply->request_id = 0;		/* set if everything is ok */
    reply->error_status = E_NOERROR;
    reply->error_index  = 0;

    Tcl_SplitList (interp, Tcl_DStringValue(&pdu->varbind), &vblc, &vblv);

    for (i = 0; i < vblc; i++) {

	oid *oidbuf;
	struct instance *inst;	
	char **vbv;
	int oidlen, vbc;
	char *value;
	
	if (Tcl_SplitList (interp, vblv[i], &vbc, &vbv) != TCL_OK) {
	    fprintf (stderr, "XXX bogus argument list\n");
	    return TCL_ERROR;
	}
	if (vbc != 3) {
	    fprintf (stderr, "XXX bogus argument list\n");
	    return TCL_ERROR;
	}

	oidbuf = SNMP_Str2Oid (vbv[0], &oidlen);
	if (exact) {
	    inst = TreeFind (instTree, oidbuf, oidlen);
	} else {
	    inst = TreeFindNext (instTree, oidbuf, oidlen);
	}

	if (!inst && pdu->type == SET_REQUEST) {
	    char *access = MIB_Access (vbv[0], 0);
	    if (access && (strcmp (access, "read-create") == 0)) {
		char *name = MIB_Name (vbv[0], 0);
		char *tmp = xmalloc (strlen(name) + 2);
		char *c = tmp;
		strcpy (tmp, name);
		for (c = tmp; *c && *c != '.'; c++);
		if (*c) *c = '(';
		while (*c) c++;
		*c++ = ')';
		*c = '\0';
		SNMP_CreateInstance (interp, vbv[0], tmp, "");
		free (tmp);
		inst = TreeFind (instTree, oidbuf, oidlen);
	    }
	}

	if (!inst) {
	    fprintf (stderr, "XXX instance %s not found\n", vbv[0]);
	    reply->error_status = E_NOSUCHNAME;
	    reply->error_index = i+1;
	    free ((char *) vbv);
	    break;
	}

	if (pdu->type == SET_REQUEST) {
	    if (inst->access == M_READONLY) {
		reply->error_status = E_NOTWRITABLE;
		reply->error_index = i+1;
		free ((char *) vbv);
		break;
	    }
	    Tcl_SetVar (interp, inst->tclvar, vbv[2], TCL_GLOBAL_ONLY);
	    snmp_stats.snmpInTotalSetVars++;
	}

	Tcl_DStringStartSublist (&reply->varbind);
	Tcl_DStringAppendElement (&reply->varbind, inst->label);
	Tcl_DStringAppendElement (&reply->varbind, inst->syntax);
	value = Tcl_GetVar (interp, inst->tclvar, 
			    TCL_GLOBAL_ONLY | TCL_LEAVE_ERR_MSG);
	if (!value) {
	    reply->error_status = E_GENERR;
	    reply->error_index = i+1;
	    free ((char *) vbv);
	    break;
	}
	Tcl_DStringAppendElement (&reply->varbind, value);
	Tcl_ResetResult (interp);

	if (pdu->type != SET_REQUEST) snmp_stats.snmpInTotalReqVars++;

	Tcl_DStringEndSublist (&reply->varbind);
	free ((char *) vbv);

    }
    free ((char *) vblv);

    reply->type = RESPONSE;
    if (reply->error_status != E_NOERROR) {
	Tcl_DStringFree (&reply->varbind);
	Tcl_DStringAppend (&reply->varbind,
			   Tcl_DStringValue (&pdu->varbind),
			   Tcl_DStringLength (&pdu->varbind));
    }
 
    reply->request_id = pdu->request_id;
#ifdef DEBUG
    fprintf (stderr, "response (%d): %s\n", reply->request_id,
	     Tcl_DStringValue (&reply->varbind));
#endif
    return SNMP_Encode (interp, session, reply, NULL, NULL);
}

