/*
 * asn1.c
 *
 * This is the implementation of the ASN1/BER encoding and 
 * decoding functions.
 *
 * Copyright (c) 1994
 *
 * Sven Schmidt, J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdio.h>
#include <string.h>
#include <tcl.h>
#include <sys/types.h>
#include <netinet/in.h>

#ifdef DBMALLOC
#include <dbmalloc.h>
#endif

#include "snmp.h"
#include "asn1.h"
#include "xmalloc.h"

/*
 * The static error string is modified whenever we have found an error.
 * ASN1_ErrorString() can be used to retrieve the error message from other
 * modules.
 */

static char error[256];

char*
ASN1_ErrorString ()
{
    return error;
}


/*
 * The following table is used to convert syntax names to syntax token.
 * Access to the table is done by str_to_syntax() and syntax_to_str().
 */

struct asn1_type {
   char			*name;
   int			syntax;
   struct asn1_type	*next;
};

static struct asn1_type syntax_table[] =
{
   { "ASN1_OTHER",		ASN1_OTHER 		},
   { "1",			ASN1_OTHER 		},
   { "INTEGER",			ASN1_INTEGER 		},
   { "BIT STRING",		ASN1_BIT_STRING		},
   { "OCTET STRING",		ASN1_OCTET_STRING 	},
   { "ASN1 NULL",		ASN1_NULL		},
   { "OBJECT IDENTIFIER",	ASN1_OBJECT_IDENTIFIER	},
   { "7",			ASN1_OTHER 		},
   { "8",			ASN1_OTHER 		},
   { "9",			ASN1_OTHER 		},
   { "10",			ASN1_OTHER 		},
   { "11",			ASN1_OTHER 		},
   { "12",			ASN1_OTHER 		},
   { "13",			ASN1_OTHER 		},
   { "14",			ASN1_OTHER 		},
   { "15",			ASN1_OTHER 		},
   { "SEQUENCE",		ASN1_SEQUENCE		},
   { "SEQUENCE OF",		ASN1_NULL		},
   { "18",			ASN1_OTHER 		},
   { "19",			ASN1_OTHER 		},
   { "Integer32",		ASN1_INTEGER		},
   { "NetworkAddress",		ASN1_IpAddress		},
   { "Counter32",		ASN1_Counter		},
   { "Gauge32",			ASN1_Gauge		},
   { "DisplayString",		ASN1_OCTET_STRING	},
   { "PhysAddress",		ASN1_OCTET_STRING	},
   { "OwnerString",		ASN1_OCTET_STRING	},
   { "EntryStatus",		ASN1_INTEGER		},
   { "Party",			ASN1_OBJECT_IDENTIFIER	},
   { "TAddress",		ASN1_OCTET_STRING	},
   { "30",			ASN1_OTHER 		},
   { "31",			ASN1_OTHER 		},
   { "32",			ASN1_OTHER 		},
   { "33",			ASN1_OTHER 		},
   { "34",			ASN1_OTHER 		},
   { "35",			ASN1_OTHER 		},
   { "36",			ASN1_OTHER 		},
   { "37",			ASN1_OTHER 		},
   { "38",			ASN1_OTHER 		},
   { "39",			ASN1_OTHER 		},
   { "40",			ASN1_OTHER 		},
   { "41",			ASN1_OTHER 		},
   { "42",			ASN1_OTHER 		},
   { "43",			ASN1_OTHER 		},
   { "44",			ASN1_OTHER 		},
   { "45",			ASN1_OTHER 		},
   { "46",			ASN1_OTHER 		},
   { "47",			ASN1_OTHER 		},
   { "48",			ASN1_OTHER 		},
   { "49",			ASN1_OTHER 		},
   { "50",			ASN1_OTHER 		},
   { "51",			ASN1_OTHER 		},
   { "52",			ASN1_OTHER 		},
   { "53",			ASN1_OTHER 		},
   { "54",			ASN1_OTHER 		},
   { "55",			ASN1_OTHER 		},
   { "56",			ASN1_OTHER 		},
   { "57",			ASN1_OTHER 		},
   { "58",			ASN1_OTHER 		},
   { "59",			ASN1_OTHER 		},
   { "60",			ASN1_OTHER 		},
   { "61",			ASN1_OTHER 		},
   { "62",			ASN1_OTHER 		},
   { "63",			ASN1_OTHER 		},
   { "IpAddress",		ASN1_IpAddress		},
   { "Counter",			ASN1_Counter		},
   { "Gauge",			ASN1_Gauge		},
   { "TimeTicks",		ASN1_TimeTicks		},
   { "Opaque",			ASN1_Opaque		},
   { "NsapAddress",		ASN1_NsapAddress	},
   { "Counter64",		ASN1_Counter64		},
   { "UInteger32",		ASN1_UInteger32		},
   { NULL }
};

/*
 * ASN1_Sntx2Str() converts a syntax token into a string representation
 * which is some static storage. asn1_to_str() returns NULL if the 
 * argument is not one of the known syntaxes.
 */

char*
ASN1_Sntx2Str (syntax)
     int syntax;
{
    if ((syntax >= 0) && (syntax <= ASN1_UInteger32))
      return syntax_table[syntax].name;
    else
      return NULL;
}


/* 
 * ASN1_Str2Sntx() converts the string to an ASN1 syntax token. 
 * ASN1_Str2Sntx() returns 0 if the string is not a know syntax.
 */

int
ASN1_Str2Sntx (str)
     char *str;
{
    struct asn1_type *tp;

    for (tp = syntax_table; tp->name; tp++) {
	if (!strcmp (str, tp->name)) return tp->syntax;
    }
    
    return 0;
}


/* 
 * ASN1_EncodeLength() sets the length field of any ASN1 encoded
 * type. If length is > 0x7F the array is shifted to get enough space
 * to hold the value.
 */

u_char*
ASN1_EncodeLength (packet, packetlen, len_fld, length)
     u_char	*packet;
     int	*packetlen;
     u_char	*len_fld;
     int	length;

{
    int i;

    if (length <= 0x7F) {

	*len_fld = length;
	
    } else if (length <= 0xFF) {

        for (i = packet - len_fld; i > 0; i--) {
	    len_fld[i] = len_fld[i - 1];
	}
	packet     += 1;
	*packetlen += 1;
	*len_fld++ = ( BIT_8 | 0x01 );
	*len_fld++ = length;      

    } else if (length <= 0xFFFF) {

	for (i = packet - len_fld - 1; i > 0; i--) {
	    len_fld[i] = len_fld[i - 2];
	}
	packet     += 2;
	*packetlen += 2;
	*len_fld++ = ( BIT_8 | 0x02 );
	*len_fld++ = ( ( length >> 8 ) & 0xFF );
	*len_fld++ = ( length & 0xFF );

   } else {

       strcpy (error, "failed to encode very long ASN1 length");
       return NULL;
   }

   return packet;
}



/*
 * ASN1_DecodeLength() decodes the length field of any ASN1 encoded type.
 * If length field is in indefinite length form or longer than
 * the size of an unsigned int, an error is reported.
 */

u_char*
ASN1_DecodeLength (packet, packetlen, length)
     u_char	*packet;
     int	*packetlen;
     u_int	*length;
{
    u_char	*cp;
    
    /* check if length field is longer than one byte */
    
    if (*packet & BIT_8) {

	cp = packet;
	*packet &= ~BIT_8;
	if (*packet == 0) {
	    strcpy (error, "indefinite length format not supported");
	    return NULL;
	}
	if (*packet > sizeof (int)) {
	    strcpy (error, "data lengths of this size not supported");
	    return NULL;
	}
	memcpy ((char *) length, (char *) packet + 1, (int) *packet);
	*length = ntohl( *length );
	*length >>= ( 8 * ( (sizeof *length) - (int) *packet));
	
	*packetlen += ( 1 + *packet );
	packet     += ( 1 + *packet );
	*cp |= BIT_8;

    } else {

	*length    = (int) *packet++;
	*packetlen += 1;
    }

    return( packet );
}



/*
 * ASN1_EncodeInt() encodes an ASN1 Integer value (means a int) to an
 * octet string by using the primitive, definite length encoding
 * method. Returns a pointer to the first byte past the end of this 
 * object or NULL on any error.
 */

u_char*
ASN1_EncodeInt (packet, packetlen, type, value)
     u_char	*packet;
     int	*packetlen;
     u_char	type;
     int	value;
{
    int		asnlen  = 0;
    int		intsize = sizeof( int );
    int		mask;
    u_char	*length;
    
    /* encode type and reserve space for length */

    *packet++  = type;
    *packetlen += 1;
    
    length = packet++;   
    *packetlen += 1;
    
    /* set the leftmost 9 bits of mask to 1 and check if the 
       leftmost bits of value are 0 or 1  */

    mask = 0x1FF << ( ( 8 * ( sizeof( int ) - 1 ) ) - 1 );
    
    while ((((value & mask) == 0)
	    || ((value & mask) == mask )) && intsize > 1) {
	intsize--;
	value <<= 8;
    }

    /* set the leftmost 8 bits of mask to 1 and build the 
       two's complement of value */

    mask = 0xFF << ( 8 * ( sizeof( int ) - 1 ) );
    while (intsize--) {
	*packet++ = (( value & mask ) >> ( 8 * ( sizeof( int ) - 1 )));
	*packetlen += 1;
	value <<= 8;
	asnlen += 1;
    }

    /* encode length field and return */

    packet = ASN1_EncodeLength( packet, packetlen, length, asnlen );
    return packet;
}


/*
 * ASN1_DecodeInt() decodes an ASN integer value.
 */

u_char*
ASN1_DecodeInt (packet, packetlen, type, value)
     u_char	*packet;
     int	*packetlen;
     u_char	type;
     int	*value;
{
    u_int asnlen = 0;

    /* handle invalid tag field */

    if (*packet++ != type) {
	sprintf (error, "invalid tag: 0x%.2x, expecting 0x%.2x",
		 *--packet, type);
	return NULL;
    }
    *packetlen += 1;

   /* handle invalid integer size */

    packet = ASN1_DecodeLength (packet, packetlen, &asnlen);
    if (packet == NULL) return NULL;
    
    if (asnlen == 0) {
	*value = 0;
	return packet;
    }
    
    while (asnlen > 4) {
	++packet;
	*packetlen += 1;
	--asnlen;
    }
    
    /* check if it's a negative value and decode data */
    
    if ((type == ASN1_INTEGER) && (*packet & 0x80))
      *value = -1;
    else
      *value = 0;
    
    while (asnlen-- > 0) {
	*value = ( *value << 8 ) | (*packet++ & 0xff);
	*packetlen += 1;
    }
    
    return packet;
}


/*
 * ASN1_EncodeNull() encodes a NULL value.  On success, a pointer to
 * the first byte past the end of this object is returned, or NULL on
 * any error.
 */

u_char*
ASN1_EncodeNull (packet, packetlen)
     u_char 	*packet;
     int	*packetlen;
{
    *packet++  = ASN1_NULL;
    *packet++  = '\0';
    *packetlen += 2;
    return packet;
}


/*
 * ASN1_EncodeOID() encodes an OBJECT IDENTIFIER to an octet string by
 * using the primitive, definite length encoding method. On success, a
 * pointer to the first byte past the end of this object is returned,
 * or NULL on any error.
 */

u_char*
ASN1_EncodeOID (packet, packetlen, objid, oidlen)
     u_char	*packet;
     int	*packetlen;
     oid	*objid;
     int	oidlen;
{
    int		asnlen = 0;
    long	mask, bits;
    oid		*op = objid;
    u_char	*length;
    
    /* check for valid length */

    if (oidlen < 2) return NULL;

    /* encode type and reserve space for length */

    *packet++  = ASN1_OBJECT_IDENTIFIER;
    *packetlen += 1;

    length = packet++;   
    *packetlen += 1;

    /* encode the first two components using the formula (X * 40) + Y */

    *packet++  = objid[0] * 40 + objid[1];
    *packetlen += 1;
    asnlen += 1;
    oidlen -= 2;
    op     += 2;

    /* encode the other components */

    while (oidlen-- > 0) {

      /* are seven bits enough for this component */
	
	if (*op <= 0x7F) {
	    
	    asnlen += 1;
	    *packet++  = *op++;
	    *packetlen += 1;
	    
	} else {
	    
	    /* we need two or more octets for encoding */
	    
	    /* check nr of bits for this component */
	    
	    int n = sizeof (*op) * 8;		/* max bit of op */
	    
	    mask = 1 << (n - 1);
	    for (bits = n; bits > 0; bits--, mask >>= 1) {
		if (*op & mask) break;
	    }
	    
	    /* round # of bits to multiple of 7: */
	    
	    bits = ((bits + 6) / 7) * 7;
	    mask = (1 << bits) - 1;
	    
	    /* encode the mostleft 7 bits and shift right */
	    
	    for (; bits > 7; mask >>= 7 ) {
		asnlen += 1;
		bits -= 7;
		*packet++  = ((( *op & mask ) >> bits ) | BIT_8 );
		*packetlen += 1;
	    }
	    
	    asnlen += 1;
	    *packet++ = ( *op++ & mask );
	    *packetlen += 1;
	}
    }

    /* encode length field */

    packet = ASN1_EncodeLength (packet, packetlen, length, asnlen);
    return packet;
}



/* 
 * ASN1_DecodeOID() decodes and object identifier. The caller of this
 * function is responsible to provide enough memory to hold the
 * object identifier.
 */

u_char*
ASN1_DecodeOID (packet, packetlen, objid, oidlen)
     u_char		*packet;
     int		*packetlen;
     oid		*objid;
     int		*oidlen;
{
    u_int	asnlen;
    oid		*op = objid;

    /* handle invalid tag field */
    
    if (*packet++ != ASN1_OBJECT_IDENTIFIER) {
	sprintf (error, "invalid tag: 0x%.2x, expecting 0x%.2x",
		 *--packet, ASN1_OBJECT_IDENTIFIER );
	return NULL;
    }
    *packetlen += 1;
    
    /* handle invalid length field */
    
    packet = ASN1_DecodeLength (packet, packetlen, &asnlen);
    if (packet == NULL) return NULL;
    
    if (asnlen == 0) {
	strcpy (error, "didn't expect OBJECT IDENTIFIER of zero length");
	return NULL;
    }
    
    if (asnlen == 1) {
	*objid     = *packet++;
	*oidlen    = 1;
	*packetlen += 1;
	return packet;
    }
    
    /* decode the first component to the first two subidentifiers */
    
    objid[1] = (u_char)( *packet % 40 );
    objid[0] = (u_char)(( *packet++ - objid[1] ) / 40 );
    op         += 2;
    *oidlen    = 2;
    asnlen     -= 1;
    *packetlen += 1;
    
    /* decode the subidentifer */

    while (asnlen > 0) {
	memset ((char *)op, '\0', sizeof (oid));
	while (*packet > 0x7F) {
	    *op = ( *op << 7 ) + ( *packet++ & ~BIT_8 );
	    asnlen     -= 1;
	    *packetlen += 1;
	}

	*op = ( *op << 7 ) + ( *packet++ );
	op         += 1;
	*oidlen    += 1;
	asnlen     -= 1;
	*packetlen += 1;
    }

    return packet;
}


/*
 * ASN1_EncodeOctetString() encodes an OCTET STRING to an octet string by
 * using the primitive, definite length encoding method. On success, a
 * pointer to the first byte past the end of this object is returned,
 * or NULL on any error.
 */

u_char*
ASN1_EncodeOctetString (packet, packetlen, type, octets, len)
     u_char	*packet;
     int	*packetlen;
     u_char	type;
     u_char	*octets;
     int	len;
{
    int		i, asnlen = 0;
    u_char	*length, *op = octets;

    /* encode type, reserve space for length field and encode octets */

    *packet++  = type;
    *packetlen += 1;
    length     = packet++;   
    *packetlen += 1;

#if 0    
    if (op != NULL) {
	while (*op != '\0') {
	    *packet++  = *op++;
	    *packetlen += 1;
	    asnlen     += 1;
	}
    }
#else
    for (i = 0; i < len; i++) {
	*packet++ = *op++;
    }
    *packetlen += len;
    asnlen     += len;
#endif

    /* encode length field and return */

    packet = ASN1_EncodeLength (packet, packetlen, length, asnlen);
    return packet;
}



/*
 * ASN1_DecodeOctetString() decodes an octet string. On success, a
 * malloced pointer to the first byte of the octet string is set and
 * the length is set. It is up to the caller to free this memory.
 * In case of an error, no memory is allocated.
 */

u_char*
ASN1_DecodeOctetString (packet, packetlen, type, octets, len)
     u_char		*packet;
     int		*packetlen;
     u_char		type;
     u_char		**octets;
     int		*len;
{
    u_int	asnlen;
    int		i;

    /* handle invalid tag field */

    if (*packet++ != type) {
	sprintf (error, "invalid tag: 0x%.2x, expecting 0x%.2x",
		 *--packet, type );
	return NULL;
    }
    *packetlen += 1;

    /* handle zero length octet string of the form 0x04 0x00 */

    packet = ASN1_DecodeLength (packet, packetlen, &asnlen);
    if (packet == NULL) return NULL;

    if (asnlen == 0 && octets != NULL) {
        *octets = NULL;
	*len	= 0;
	return packet;
    }

    /* decode value fields */

    if (octets == NULL) {
	*packetlen += asnlen;
	return (packet + asnlen);
    }

    *octets = (u_char *) xmalloc (asnlen + 1);
    *len = asnlen;

    for( i = 0; i < asnlen; i++) {
	(*octets)[i] = packet[i];
    }
    (*octets)[i] = '\0';
    *packetlen += asnlen;
    return (packet + asnlen);
}


