/*
 * event.c
 *
 * This file contains all functions that handle events.
 *
 * Copyright (c) 1994
 *
 * Sven Schmidt, J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdio.h>
#include <tcl.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef DBMALLOC
#include <dbmalloc.h>
#endif

#include "snmp.h"
#include "udp.h"
#include "misc.h"

/*
 * SNMP_TimeoutEvent() handles timeouts by retransmitting the message
 * or calling the error callback function if the number of retries
 * are already done.
 */

void
SNMP_TimeoutEvent (clientData)
     ClientData	clientData;
{
    struct request_timeout *timeout = (struct request_timeout *) clientData;
    struct request_list	  *request = NULL;
    struct session	  *session = NULL;
    
    Tcl_Interp *interp = timeout->interp;
    
    /* check # of retransmissions for the request that timed out */
    
    request = SNMP_LookupRequest (timeout->request_id, &session);
    if (request == NULL) return;
    
    if (request->retr_nr < session->retries) {
	
	/* reinstall TimerHandler for this request */
	
        request->retr_nr += 1;
	request->timeout_token = Tk_CreateTimerHandler(
				( session->timeout * 1000 ) / session->retries,
			 	SNMP_TimeoutEvent,
				(ClientData *)timeout );
	
	/* retransmit this request (keeping the original oid) */
	SNMP_Send (interp, session, request->packet, request->packetlen);

    } else {

	/* # of retransmissions reached: delete this request */
    
	Tcl_DString dst;
	
	Tcl_DStringInit (&dst);
	if (request->error_func != NULL) {
	    SNMP_EvalCallback (interp, session, timeout->request_id,
			       -1, 0, request->error_func, &dst);
	}
	Tcl_DStringFree (&dst);
	
	SNMP_DeleteRequest (session, request);
	
	Tcl_ResetResult (interp);
    }
}

/*
 * This function is called from the event dispatcher whenever a
 * response to a management message is received.
 */

void
SNMP_ResponseEvent (clientData, mask)
     ClientData	clientData;
     int mask;
{
    Tcl_Interp *interp = (Tcl_Interp *) clientData;
    u_char packet[BUFSIZE];
    int packetlen = BUFSIZE;
    int reqid;

    Tcl_ResetResult (interp);
    if (SNMP_Recv (interp, packet, &packetlen) != TCL_OK) return;

    if (SNMP_Decode (interp, &reqid, packet, packetlen) != TCL_OK) {
	fprintf (stderr, "SNMP_ResponseEvent: %s\n", interp->result);
    }
}

/*
 * SNMP_TrapEvent() is called from the event dispatcher when a
 * trap message is received.
 */

void
SNMP_TrapEvent (clientData, mask)
     ClientData clientData;
     int mask;
{
    Tcl_Interp *interp = (Tcl_Interp *) clientData;
    u_char packet[BUFSIZE];
    int packetlen = BUFSIZE;
    int reqid;

    Tcl_ResetResult (interp);
    if (SNMP_TrapRecv (interp, packet, &packetlen) != TCL_OK) return;

    if (SNMP_Decode (interp, &reqid, packet, packetlen) != TCL_OK) {
	fprintf (stderr, "SNMP_TrapEvent: %s\n", interp->result);
    }
}

/*
 * SNMP_AgentEvent() is called whenever a request to the SNMP agent
 * side is received.
 */

void
SNMP_AgentEvent (clientData, mask)
     ClientData clientData;
     int mask;
{
    struct agentToken *token = (struct agentToken *) clientData;
    Tcl_Interp *interp = token->interp;
    struct session *session = token->session;
    u_char packet[BUFSIZE];
    int packetlen = BUFSIZE;
    int reqid;
    
    Tcl_ResetResult (interp);
    if (SNMP_AgentRecv (interp, session, packet, &packetlen) != TCL_OK) return;
    
    if (SNMP_Decode (interp, &reqid, packet, packetlen) != TCL_OK) {
	fprintf (stderr, "SNMP_AgentEvent: %s\n", interp->result);
    }
}
