/*
 * oid.c
 *
 * Some small utilities that deal with object identifier.
 *
 * This file contains all functions that handle the UDP communication.
 * There is also some code here to talk to the straps daemon which is
 * used to receive and forward SNMP traps send to the privileged 162
 * port.
 *
 * Copyright (c) 1994
 *
 * Sven Schmidt, J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <tcl.h>

#ifdef DBMALLOC
#include <dbmalloc.h>
#endif

#include "oid.h"
#include "xmalloc.h"

/*
 * SNMP_OidDup() duplicates an object identifier. The length is 
 * returned in buflen.
 */

oid*
SNMP_OidDup (buflen, Oid, OidLen)
     int *buflen;
     oid *Oid;
     int OidLen;
{
    oid	*buf = (oid *) xmalloc (sizeof (oid) * OidLen);
    *buflen = OidLen;

    memcpy (buf, Oid, OidLen * sizeof (oid));
    
    return buf;
}

/*
 * SNMP_Oid2Str() converts an object identifier to string in 
 * dotted notation. The string is alloced using malloc and must be
 * freed by the caller.
 */

char*
SNMP_Oid2Str (objid, len)
     oid *objid;
     int len;
{
    int	 i;
    oid	 *op;
    char *buf, *cp;

    if (objid == NULL) return NULL;

    buf = xmalloc (sizeof(char) * OID_MAXLEN * 8);
    cp  = buf;
    
    for (i = 0, op = objid; i < len; i++) {
	sprintf (cp, "%d.", *op++);
	while (*cp) cp++;
    }
    *--cp = '\0';
    
    return buf;
}

/*
 * SNMP_Str2Oid() converts a string with an oid in dotted representation
 * into an object identifier vector. The vector is static memory and must
 * not be freed by the caller. 
 */

oid*
SNMP_Str2Oid (str, len)
     char *str;
     int  *len;
{
    static oid oidbuf[OID_MAXLEN];
    char buf[20], *cp, *val;
    oid	*op;

    *len  = 0;    
    if (str == NULL) return NULL;

    op    = oidbuf;
    val   = buf;

    for (cp = str; *cp; cp++) {
	if (*cp == '.') {
	    *val = '\0';
	    val  = buf;
	    if (buf[0] != '\0') {
	        *op++ = atoi (buf);
		*len += 1;
	    }
	}
	else {
	    *val++ = *cp;
	}
    }
    
    *val = '\0';
    if (buf[0] != '\0') {
	*op++ = atoi (buf);
	*len += 1;
    }
    
    *op = '\0';
    return oidbuf;
}

/*
 * SNMP_IsOid() tests the given string, whether it consists of dots and
 * digits only. If the string is an OID, 1 is returned, 0 otherwise.
 */

int
SNMP_IsOid (str)
     char *str;
{
    char *cp;

    for (cp = str; *cp != '\0'; cp++) {
        if (!isdigit (*cp) && *cp != '.') return 0;
    }
    
    return 1;
}

