/*
 * query.c
 *
 * Provides functions to search in the MIB tree.
 *
 * Copyright (c) 1994
 *
 * Sven Schmidt, J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <tcl.h>

#include "snmp.h"
#include "asn1.h"
#include "mib.h"
#include "xmalloc.h"

#ifdef DBMALLOC
#include <dbmalloc.h>
#endif

/*
 * module global variables
 */

static struct tree *tree;

/*
 * Forward declarations for procedures defined later in this file:
 */

static struct tree *
lookup_oid		_ANSI_ARGS_((struct tree *root, char *label,
				     char *soid, int exact));
static struct tree*
lookup_name		_ANSI_ARGS_((struct tree *root, char *label,
				     char *soid, int exact, int fuzzy));
static char *
textconv_format		_ANSI_ARGS_((char *val, char *fmt));

static char *
textconv_deformat	_ANSI_ARGS_((char *val, char *fmt));


/*
 * MIB_Load() reads another MIB definition and adds the objects to the
 * internal MIB tree.
 */

int
MIB_Load (interp, file)
     Tcl_Interp	*interp;
     char *file;
{
    Tcl_DString dst;
    char *fname;

    if ((fname = Tcl_TildeSubst (interp, file, &dst)) == NULL) {
	Tcl_DStringFree (&dst);
	return TCL_ERROR;
    }

    if (access (fname, R_OK) != 0) {
	Tcl_AppendResult (interp, "couldn't open MIB file \"", file,
			  "\": ", Tcl_PosixError (interp), (char *) NULL);
	Tcl_DStringFree (&dst);
	return TCL_ERROR;
    }
    tree = MIB_Parse (fname, tree);

    Tcl_DStringFree (&dst);

    if (tree == NULL) {
	Tcl_SetResult (interp, "loading MIB file failed", TCL_STATIC);
	return TCL_ERROR;   
    }

    return TCL_OK;
}


/*
 * lookup_oid() searches for the node given by soid. It converts the
 * oid from string to binary representation and uses to subids to
 * go from the root to the requested node. Returns the node or NULL
 * if the node was not found.
 */

static struct tree *
lookup_oid (root, label, soid, exact)
     struct tree *root;
     char *label;
     char *soid;
     int exact;
{
    oid *id;
    int i, len;
    struct tree *p, *q = NULL;
    char buf[20];

    id = SNMP_Str2Oid (label, &len);

    for (p = root; p ; p = p->next) {
	if (id[0] == p->subid) break;
    }
    if (!p) return NULL;

    for (i = 1; i < len; p = q, i++) {
	for (q = p->child; q; q = q->next) {
	    if (q->subid == id[i]) break;
	}
	if (!q) {
	    if (! exact) {
		if (soid) {
		    strcpy (soid, p->label);
		    while (i < len) {
			sprintf (buf, ".%d", id[i++]);
			strcat (soid, buf);
		    }
		}
		return p;
	    } else {
		return NULL; 
	    }
	}
    }

    if (q && soid) {
	strcpy (soid, q->label);
    }
    return q;
}


/*
 * lookup_name() extracts the first element (head) of the label
 * argument and recursively searches for a tree node named head. If
 * found, it checks whether the remaining label elements (tail) can be
 * resolved from this node.
 */

static struct tree*
lookup_name (root, label, soid, exact, fuzzy)
     struct tree *root;
     char *label;
     char *soid;
     int exact;
     int fuzzy;
{
    char head[STATIC_BUFFER_SIZE];
    char buf[20];
    char *tail = label, *p = head;
    struct tree *tp = NULL, *brother;
    int num = 1;

    if (!root) return NULL;

    while (*tail && *tail != '.') {
	num = isdigit (*tail) ? num : 0;
	*p++ = *tail++;
    }
    *p = '\0';
    if (*tail == '.') tail++;
    num = num ? atoi (head) : -1;

    for (brother = root; brother; brother = brother->next) {
	if ((strcmp (head, brother->label) == 0) || (num == brother->subid)) {
	    if (! *tail) {
		tp = brother;
	    } else if (brother->child) {
		tp = lookup_name (brother->child, tail, soid, exact, 0);
	    } else if (! exact) {
		tp = brother;
	    }
	    if (tp) {
		if (soid) {
		    char *p, *s = soid + strlen (soid);
		    if (!*soid && *tail) {
			p = tail + strlen (tail) - 1;
			while (p >= tail) *s++ = *p--;
		    }
		    if (*soid && *s != '.') *s++ = '.';
		    sprintf (buf, "%d", brother->subid);
		    p = buf + strlen (buf) - 1;
		    while (p >= buf) *s++ = *p--;
		    *s = '\0';
		}
		return tp;
	    }
	}
	if (fuzzy && brother->child) {
	    tp = lookup_name (brother->child, label, soid, exact, 1);
	    if (tp) {
		if (soid) {
		    char *p, *s = soid + strlen (soid);
		    if (*soid) *s++ = '.';
		    sprintf (buf, "%d", brother->subid);
		    p = buf + strlen (buf) - 1;
		    while (p >= buf) { *s++ = *p--; }
		    *s = '\0';
		}
		return tp;
	    }
	}
    }

    return NULL;
}


/*
 * MIB_FindNode() takes a MIB name and searches for the corresponding
 * node in the MIB tree. The object identifier of the node is written
 * to soid if soid is not NULL. soid must be large enough to hold the
 * complete path. MIB_FindNode() calls lookup_oid() if the name is an
 * object identier for fast MIB searches. Otherwise we call slow
 * lookup_name() to compare the tree node labels.  
 */

struct tree *
MIB_FindNode (name, soid, exact)
     char *name;
     char *soid;
     int exact;
{
    struct tree *tp = NULL;

    if (SNMP_IsOid (name)) {
	tp = lookup_oid (tree, name, soid, exact);
    } else {
	tp = lookup_name (tree, name, soid, exact, 1);
	if (tp && soid) {		/* reverse the string in soid */
	    int i, len;
	    char c;
	    for (i = 0, len = strlen (soid); i < len / 2; i++) {
		c = soid[i];
		soid[i] = soid[len-i-1];
		soid[len-i-1] = c;
	    }
	}
    }

    return tp;
}


/*
 * MIB_Oid() returns the oid that belongs to label. The exact switch
 * turns exact matching on, which only allows object identifier which
 * exactly match an object type definition.
 */

char*
MIB_Oid (label, exact)
     char *label;
     int exact;
{
    static char soid[STATIC_BUFFER_SIZE];
    struct tree *node;

    memset (soid, '\0', STATIC_BUFFER_SIZE);
    if ((node = MIB_FindNode (label, soid, exact))) {
	if (SNMP_IsOid (label)) {
	    return label;
	}
	return soid;
    }

    return NULL;
}

/*
 * MIB_Name() returns the name that belongs to label. This is the 
 * reverse operation to MIB_Oid().
 */

char*
MIB_Name (label, exact)
     char *label;
     int exact;
{
    static char soid[STATIC_BUFFER_SIZE];
    struct tree *node;

    memset (soid, '\0', STATIC_BUFFER_SIZE);
    if ((node = MIB_FindNode (label, soid, exact))) {
	if (SNMP_IsOid (label)) {
	    return soid;
	}
#if 1
	/* must append trailing instance identifiers */
	/* XXX fprintf (stderr, "** perhaps %s ?\n", soid); */
	return node->label;
#else
	return MIB_Name (soid, exact);
#endif
    }

    return NULL;
}

/*
 * MIB_Description() extracts the description of a MIB object.
 * White spaces following newline characters are removed so that
 * the resulting text looks like the text in the MIB definition.
 * Note however, that all leading white spaces are removed which
 * may make problems in some rare situations.
 */

char*
MIB_Description (name, exact)
     char *name;
     int exact;
{
    FILE *fp;
    int	ch;
    char line[81];
    int len = 0;
    struct tree *node = MIB_FindNode (name, NULL, exact);
    static Tcl_DString *result = NULL;
    
    if (result == NULL) {
	result = (Tcl_DString *) xmalloc (sizeof(Tcl_DString));
	Tcl_DStringInit (result);
    } else {
	Tcl_DStringFree (result);
    }
    
    line[0] = '\0';
    if (node) {
	if (node->description > 0 && node->filename != NULL) {
	    fp = fopen (node->filename, "r");
	    if (fp == NULL) return (NULL);
	    fseek (fp, node->description, 0);

	    while ((ch = getc (fp)) != EOF) {
		if (ch == '"') break;
	    }
	    
	    len = 0;
            line[0] = '\0';
	    while ((ch = getc (fp)) != EOF) {
		if (ch == '"') break;
		line[len++] = ch;
		if (ch == '\n' || len == 80) {
		    line[len] = '\0';
		    Tcl_DStringAppend (result, line, len);
		    len = 0;
		    if (ch == '\n') {
			while ((ch = getc (fp)) != EOF) {
			    if (!isspace (ch)) break;
			}
			if (ch == EOF || ch == '"') break;
			line[len++] = ch;
		    }
		}
	    }
	    if (len != 0) {
		line[len] = '\0';
		Tcl_DStringAppend (result, line, len);
	    }
	    
	    fclose (fp);
	    return (Tcl_DStringValue (result));
	} else {
	    Tcl_DStringAppend (result, "no description for \"", -1);
	    Tcl_DStringAppend (result, name, -1);
	    Tcl_DStringAppend (result, "\"", 1);
	    return (Tcl_DStringValue (result));
	}
    }
    return NULL;
}

/*
 * MIB_Succ()
 */

char*
MIB_Succ (name)
     char *name;
{
    struct tree	*node = MIB_FindNode (name, NULL, 1);
    int retoid = SNMP_IsOid (name);
    static Tcl_DString *result = NULL;
    char buf[20];
    
    if (result == NULL) {
	result = (Tcl_DString *) xmalloc (sizeof(Tcl_DString));
	Tcl_DStringInit (result);
    } else {
	Tcl_DStringFree (result);
    }
    
    if (node) {
	if (node->child) {
	    for (node = node->child; node; node = node->next) {
		if (retoid) {
		    Tcl_DStringAppendElement (result, name);
		    sprintf (buf, ".%d", node->subid);
		    Tcl_DStringAppend (result, buf, -1);
		} else {
		    Tcl_DStringAppendElement (result, node->label);
		}
	    }
	}
	return Tcl_DStringValue (result);
    }
    return NULL;
}


/*
 * MIB_Syntax() returns the syntax as defined in the OBJECT-TYPE
 * definition. This is used to implement the mib syntax command.
 */

char*
MIB_Syntax (name, exact)
     char *name;
     int exact;
{
    struct tree *node = MIB_FindNode (name, NULL, exact);

    if (node) {
	if (node->tt && node->tt->name) {
	    if (*node->tt->name == '_')
	      return ASN1_Sntx2Str (node->tt->syntax);
	    else 
	      return node->tt->name;
	}
	return ASN1_Sntx2Str (node->syntax);
    }

    return NULL;
}


/*
 * MIB_ASN1() returns the transfer syntax actually used to encode a
 * value. This may be different from the syntax defined in the
 * OBJECT-TYPE macro as textual conventions are based on a set of
 * primitive types.
 */

int
MIB_ASN1 (name, exact)
     char *name;
     int exact;
{
    struct tree *node = MIB_FindNode (name, NULL, exact);

    if (node) {
	if (node->tt && node->tt->name) {
	    return node->tt->syntax;
	} else {
	    return node->syntax;
	}
    }
    return ASN1_OTHER;
}

/*
 * MIB_Access() return the max access field of the OBJECT-TYPE
 * macro.
 */

char*
MIB_Access (name, exact)
     char *name;
     int exact;
{
    struct tree *node = MIB_FindNode (name, NULL, exact);

    if (node) {
	switch (node->access) {
	  case M_NOACCESS:   return ("not-accessible");
	  case M_READONLY:   return ("read-only");
	  case M_READCREATE: return ("read-create");
	  case M_READWRITE:  return ("read-write");
	  case M_WRITEONLY:  return ("write-only");
	}
    }
    
    return NULL;
}


/*
 * format val according to fmt;
 * returns the parameter or a pointer to static buffer space 
 * (0 terminated), or 0 on error.
 * XXX: may parts untested...
 */

static char *
textconv_format (val, fmt)
     char *val;
     char *fmt;
{
    int pfx, have_pfx;			/* counter prefix */
    char *last_fmt;			/* save ptr to last seen fmt */
    static char *ret = 0;
    static int ret_len = 0;
    int idx;

    /* sanity check; */
    if (! fmt)
      return val;

    /* a simple one: */
    if (! strcmp (fmt, "1x:"))
      return val;

    if (! ret)
      ret = xmalloc (ret_len = 100);
    idx = 0;			/* idx into ret buffer */

    while (*fmt && *val)
      {
	  last_fmt = fmt;	/* save for loops */

	  /* scan prefix: */
	  have_pfx = pfx = 0;
	  while (*fmt && isdigit(*fmt))
	    pfx = pfx * 10 + *fmt - '0', have_pfx = 1, fmt++;
	  if (! have_pfx)
	    pfx = 1;

	  /* scan format: */
	  if (*fmt == 'a')
	    {
		while (*val && pfx > 0) {
		    char c = *val++ & 0xff;
		    int v = c >= 'a' ?  c - 87 : (c >= 'A' ? c - 55 : c - 48);
		    if (! *val)
		      break;
		    c = *val++ & 0xff;
		    v = (v << 4) + (c >= 'a' ?  c - 87 :
				    (c >= 'A' ? c - 55 : c - 48));
		    if (idx + 100 >= ret_len)
		      ret = xrealloc (ret, ret_len += 100);
		    ret [idx++] = v;
		    pfx--;
		    if (*val == ':')
		      val++;
		}
		fmt++;
	    }
	  else if (*fmt == 'd' || *fmt == 'o' || *fmt == 'b')
	    {
		int c, vv = 0, v;

		/* collect octets and format: */
		while (pfx > 0)
		  { 
		      if (! *val || ! val [1])
			break;
		      /* collect next byte from octal buffer and move to vv: */
		      c = *val++ & 0xff;
		      v = c >= 'a' ?  c - 87 : 
					(c >= 'A' ? c - 55 : c - 48);
		      c = *val++ & 0xff;
		      v = (v << 4) + (c >= 'a' ?  c - 87 :
					(c >= 'A' ? c - 55 : c - 48));
		      vv = vv * 256 + v;

		      if (*val == ':')
			val++;

		      pfx--;
		  }
		
		if (idx + 100 >= ret_len)
		  ret = xrealloc (ret, ret_len += 100);
		if (*fmt == 'd')
		  {
		      sprintf (ret + idx, "%d", vv);
		      idx += strlen (ret + idx);
		  }
		else if (*fmt == 'o')
		  {
		      /* XXX: completely untested */
		      sprintf (ret + idx, "%o", vv);
		      idx += strlen (ret + idx);
		  }
		else if (*fmt == 'b')
		  {
		      /* XXX: completely untested */
		      int i; 
		      for (i = (sizeof (int) * 8 - 1); i >= 0
			   && ! (vv && (1 << i)); i--);
		      for (; i >= 0; i--)
			ret [idx++] = vv && (1 << i) ? '1' : '0';
		  }
		fmt++;
	    }
	  else if (*fmt == 'x')
	    {
		/* a simple one: copy the following hex-digits: */
		while (pfx > 0)
		  {		
		      if (! *val || ! val [1])
			break;
		      if (idx + 100 >= ret_len)
			ret = xrealloc (ret, ret_len += 100);
		      ret [idx++] = *val++;
		      ret [idx++] = *val++;
		      if (*val == ':')
			val++;
		      pfx--;
		  }
		fmt++;
	    }
	  else {
	      fprintf (stderr, "scotty: oerg ? unknown textual-format `%c'\n",
		       *fmt); 
	      fmt++;
	    }

	  /* look about a separator: */
	  if (*fmt && ! isdigit(*fmt) && *fmt != '*')
	    {
		if (have_pfx && *val)
		  {
		      if (idx + 100 >= ret_len)
			ret = xrealloc (ret, ret_len += 100);
		      ret [idx++] = *fmt;
		  }
		fmt++;
	    }

	  /* repeat with last format, if datas avail: */
	  if (! *fmt && *val)
	    fmt = last_fmt;
      }

    ret [idx] = 0;

    return ret;
}


/*
 * de-format val according to fmt;
 * returns the parameter or a pointer to static buffer space 
 * (0 terminated), or 0 on error.
 * eg: this converts readable strings according to fmt to hex format.
 * XXX: may parts untested...
 */

static char *
textconv_deformat (val, fmt)
     char *val;
     char *fmt;
{
    int pfx, have_pfx;			/* counter prefix */
    char *last_fmt;			/* save ptr to last seen fmt */
    static char *ret = 0;
    static int ret_len = 0;
    int idx;

    /* sanity check; */
    if (! fmt)
      return val;

    if (! ret)
      ret = xmalloc (ret_len = 100);

    *ret = '\0';
    idx = 0;			/* idx into ret buffer */

    /* a simple one: */
    if (! strcmp (fmt, "1x:"))
      return val;
    
    /* quick ``255a'' formatting: */
    if (! strcmp (fmt, "255a"))
      {
	  while (*val)
	    {
		if (idx + 100 >= ret_len)
		  ret = xrealloc (ret, ret_len += 100);
		sprintf (ret + idx, "%02x", *val & 0xff);
		idx += 2;
		if (*++val)
	    ret [idx++] = ':';
	    }
	  return ret;
      }


    /* and now to something different: */

    while (*fmt && *val)
      {
	  last_fmt = fmt;	/* save for loops */
	  /* scan prefix: */
	  have_pfx = pfx = 0;
	  while (*fmt && isdigit(*fmt))
	    pfx = pfx * 10 + *fmt - '0', have_pfx = 1, fmt++;
	  if (! have_pfx)
	    pfx = 1;

	  /* scan format: */
	  if (*fmt == 'a')
	    {
		while (*val && pfx > 0) {
		    char c = *val;
		    int c1 = (c & 0xf0) >> 4;
		    int c2 = c & 0x0f;
		    if ((c1 += '0') > '9') c1 += 7;
		    if ((c2 += '0') > '9') c2 += 7;
		    
		    if (idx + 100 >= ret_len)
		      ret = xrealloc (ret, ret_len += 100);

		    ret [idx++] = c1, ret [idx++] = c2;
		    if (*++val)
		      ret [idx++] = ':';
		    pfx--;
		}
		fmt++;
	    }
	  else if (*fmt == 'd' || *fmt == 'o' || *fmt == 'b')
	    {
		int vv = 0, v, got_val = 0;

		if (*fmt == 'd' && 1 == sscanf (val, "%d", &vv))
		  {
		      got_val = 1;
		      while (isdigit(*val))
			val++;
		  }
		else if (*fmt == 'o' && 1 == sscanf (val, "%o", &vv))
		  {
		      /* XXX: completely untested */
		      got_val = 1;
		      while (*val >= '0' && *val <= '7')
			val++;
		  }
		else if (*fmt == 'b')
		  {
		      /* XXX: completely untested */
		      while (*val == '0' || *val == '1')
			{
			    got_val = 1;
			    vv = (vv << 1) | (*val - '0');
			    val++;
			}
		  }

		/* XXX: tell more about unscanable strings !!! */
		if (! got_val)
		  break;
		
		/* now put pfx-num bytes to the output buffer: */
		while (pfx > 0)
		  {
		      int c1, c2;

		      v = vv >> ((pfx - 1) * 8);
		      
		      if (idx + 100 >= ret_len)
			ret = xrealloc (ret, ret_len += 100);
		      
		      c1 = (v & 0xf0) >> 4;
		      c2 = v & 0x0f;
		      if ((c1 += '0') > '9') c1 += 7;
		      if ((c2 += '0') > '9') c2 += 7;
		      		      
		      ret [idx++] = c1, ret [idx++] = c2;
		      if (*val)
			ret [idx++] = ':';
		      pfx--;
		  }
		
		fmt++;
	    }
	  else if (*fmt == 'x')
	    {
		/* a simple one: copy the following hex-digits: */
		while (pfx > 0)
		  {		
		      if (! *val || ! val [1])
			break;
		      if (idx + 100 >= ret_len)
			ret = xrealloc (ret, ret_len += 100);
		      ret [idx++] = *val++;
		      ret [idx++] = *val++;
		      if (*val == ':')
			val++, ret [idx++] = ':';
		      pfx--;
		  }
		fmt++;
	    }
	  else {
	      fprintf (stderr, "scotty: oerg ? unknown textual-format `%c'\n",
		       *fmt); 
	      fmt++;
	    }

	  /* look about a separator: */
	  if (*fmt && ! isdigit(*fmt) && *fmt != '*')
	    {
		if (have_pfx && *val)
                      val++;
		fmt++;
	    }

	  /* repeat with last format, if datas avail: */
	  if (! *fmt && *val)
	    fmt = last_fmt;
      }

    ret [idx] = 0;
	  
    return ret;
}


/*
 * MIB_Scan() converts value given any textual conventions that are
 * defined for name. Value will be returned unmodified if no
 * convention exists. The result is alway a pointer into static buffer
 * space so there is no need to worry about freeing it.
 */

char *
MIB_Format (name, exact, value)
     char *name;
     int exact;
     char *value;
{
    struct tree	*node = MIB_FindNode (name, NULL, exact);
   
    if (node) {
	if (node->tt) {
	    char *t;
	    struct idesc *i = node->tt->desc;
	    while (i) {
		if (! strcmp (i->val, value)) return i->desc;
		i = i->next;
	    }
	    t = node->tt->textconv;
	    if (t) {
		char *ret = textconv_format (value, t);
		if (ret)
		  return ret;
	    }
	}
	return value;
    }

    return NULL;
}

/*
 * MIB_Scan() converts value given any textual conventions that are
 * defined for name. Value will be returned unmodified if no
 * convention exists. This is the counterpart to MIB_Format().
 */

char*
MIB_Scan (name, exact, value)
     char *name;
     int exact;
     char *value;
{
    struct tree	*node = MIB_FindNode (name, NULL, exact);
    
    if (node) {
	if (node->tt) {
	    char *t;
	    struct idesc *i = node->tt->desc;	    
	    while (i) {
		if (! strcmp (i->desc, value)) return i->val;
		i = i->next;
	    }
	    
	    /* for textual conventions scan value and return hex: */
	    t = node->tt->textconv;
            if (t) {
                char *ret = textconv_deformat (value, t);
                if (ret)
                  return ret;
            }
	}
	return value;
    }

    return NULL;
}
