/*
 * request.c
 *
 * This file contains all functions that take a request to send a
 * SNMP packet over the network.
 *
 * Copyright (c) 1994
 *
 * Sven Schmidt, J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <tcl.h>

#ifdef DBMALLOC
# include <dbmalloc.h>
#endif

#include "snmp.h"
#include "asn1.h"
#include "misc.h"
#include "xmalloc.h"
#include "memory.h"
#include "udp.h"
#include "mib.h"
#include "agent.h"

/*
 * Forward declarations for procedures defined later in this file:
 */

static int
encode_SnmpV1Message _ANSI_ARGS_((Tcl_Interp *interp,
				  struct session *sess,
				  struct snmp_pdu *pdu,
				  u_char *packet, int *packetlen));

static int
encode_SnmpV2Message _ANSI_ARGS_((Tcl_Interp *interp,
				  struct session *sess,
				  struct snmp_pdu *pdu,
				  u_char *packet, int *packetlen));

static u_char*
encodePDU	     _ANSI_ARGS_((Tcl_Interp *interp,
				  struct snmp_pdu *pdu,
				  u_char *packet, int *packetlen));


/*
 * SNMP_Encode() converts the pdu into BER transfer syntax and sends it 
 * from the management party on this system to a remote agent.
 */

int
SNMP_Encode (interp, sess, pdu, callback, error_callback)
     Tcl_Interp *interp;
     struct session *sess;
     struct snmp_pdu *pdu;
     char *callback;
     char *error_callback;
{
    int	retry = 0, packetlen = 0, rc = 0;
    struct request_list	*request = NULL;
    u_char packet[BUFSIZE];

    memset ((char *) packet, '\0', sizeof (packet));
    packetlen = 0;

    /*
     * Turn V2 getbulk requests into getnext request for V2 agents.
     */

    if (sess->version == SNMPv1 && pdu->type == GET_BULK_REQUEST) {
	pdu->type = GET_NEXT_REQUEST;
	pdu->error_status = E_NOERROR;
	pdu->error_index  = 0;
    }

    /*
     * Encode message into ASN1 BER transfer syntax. authentication or
     * encryption is done within the following procedures if it is an
     * authentic or private message.
     */

    if (sess->version == SNMPv1)
      rc = encode_SnmpV1Message (interp, sess, pdu, packet, &packetlen);
    else
      rc = encode_SnmpV2Message (interp, sess, pdu, packet, &packetlen);
    if (rc != TCL_OK) return TCL_ERROR;

    switch (pdu->type) {
      case GET_REQUEST:		snmp_stats.snmpOutGetRequests++; break;
      case GET_NEXT_REQUEST:	snmp_stats.snmpOutGetNexts++; break;
      case SET_REQUEST:		snmp_stats.snmpOutSetRequests++; break;
      case RESPONSE:		snmp_stats.snmpOutGetResponses++; break;
      case SNMPv1_TRAP:		snmp_stats.snmpOutTraps++; break;
    }
    
    /*
     * a trap message or a response? - send it and we are done
     */
    
    if (pdu->type == SNMPv1_TRAP || pdu->type == SNMPv2_TRAP 
	|| pdu->type == RESPONSE) {
	if (SNMP_Send (interp, sess, packet, packetlen) != TCL_OK) {
	    return TCL_ERROR;
	}
	Tcl_ResetResult (interp);
	return TCL_OK;
    }
  
   /*
    * asychronous request: record request and send packet to destination
    */

    if (callback) {
	request = SNMP_RecordRequest (interp, sess, pdu->request_id, 
				      packet, packetlen, 
				      callback, error_callback);
	
	if (SNMP_Send (interp, sess, packet, packetlen) == TCL_ERROR) {
	    SNMP_DeleteRequest (sess, request);
	    return TCL_ERROR;
	}
	sprintf (interp->result, "%d", (int) pdu->request_id);
	return TCL_OK;
    }

   /*
    * synchronous request: send packet and wait for response
    */

    for (retry = 0; retry <= sess->retries; retry++) {
	
	int id;
	
	if (SNMP_Send (interp, sess, packet, packetlen) == TCL_ERROR) {
	    return TCL_ERROR;
	}

	while (SNMP_Wait (sess->timeout * 1000 / (sess->retries + 1)) > 0) {
	    u_char packet[BUFSIZE];
	    int packetlen = BUFSIZE;
	    int rc;
	    
	    if (SNMP_Recv (interp, packet, &packetlen) != TCL_OK) {
		return TCL_ERROR;
	    }
	    
	    rc = SNMP_Decode (interp, &id, packet, packetlen);
	    if (rc == TCL_OK) {
		if (id == pdu->request_id) {
		    return TCL_OK;
		}
		rc = TCL_CONTINUE;
	    }
	    
	    if (rc == TCL_CONTINUE) continue;
	    if (rc == TCL_ERROR) return TCL_ERROR;
	}
    }
    
    Tcl_ResetResult (interp);
    sprintf (interp->result, "no response after %d retries", sess->retries);
    return TCL_ERROR;
}

/*
 * encode_SnmpV1Message() takes a session and a PDU and serializes the
 * ASN1 pdu as an octet string into the buffer pointed to by packet
 * using the "Basic Encoding Rules".
 */

static int
encode_SnmpV1Message (interp, sess, pdu, packet, packetlen)
     Tcl_Interp *interp;
     struct session *sess;
     struct snmp_pdu *pdu;
     u_char *packet;
     int *packetlen;
{
    u_char *Message_len;

    /*
     * encode Message ( SEQUENCE )
     */

    *packet++ = (UNIVERSAL | CONSTRUCTED | ASN1_SEQUENCE);
    Message_len = packet++;
    *packetlen += 2;

    /*
     * encode SNMPv1 type
     */

    packet = ASN1_EncodeInt (packet, packetlen, ASN1_INTEGER, 0);

    /*
     * encode community string
     */

    packet = ASN1_EncodeOctetString (packet, packetlen,
				     ASN1_OCTET_STRING,
				     sess->community, 
				     strlen (sess->community));
    /*
     * encode PDU
     */
    
    if ((packet = encodePDU (interp, pdu, packet, packetlen)) == NULL) {
	return TCL_ERROR;
    }

    /*
     * set the message length field 
     */
    
    packet = ASN1_EncodeLength (packet, packetlen, Message_len,
                               packet - (Message_len + 1));
    return TCL_OK;
}

/*
 * encode_SnmpV2Message() takes a session and a PDU and serializes the
 * ASN1 pdu as an octet string into the buffer pointed to by packet
 * using the "Basic Encoding Rules".
 */

static int
encode_SnmpV2Message (interp, sess, pdu, packet, packetlen)
     Tcl_Interp *interp;
     struct session *sess;
     struct snmp_pdu *pdu;
     u_char *packet;
     int *packetlen;
{
    int digest_len = 0;
    
    u_char *PrivMsg_len	= NULL,	*PrivData_len = NULL;
    u_char *AuthMsg_len = NULL, *AuthData_len = NULL;

    u_char md5Digest[17], *digest_begin = NULL, *digest_end = NULL;
    
    /*
     * encode "SnmpPrivMsg" ( tag: [1] IMPLICIT SEQUENCE )
     */

    *packet++ = (CONTEXT_SPECIFIC | CONSTRUCTED | TAG_1);
    PrivMsg_len = packet++;
    *packetlen += 2;   

    /*
     * encode the "privDst" field
     */

    packet = ASN1_EncodeOID (packet, packetlen, sess->dstParty.Identity,
			     sess->dstParty.IdentityLen);
    
    /*
     * encode the "privData" field ( tag: [1] IMPLICIT OCTET STRING )
     */

    *packet++ = (CONTEXT_SPECIFIC | PRIMITIVE | TAG_1);
    PrivData_len = packet++;
    *packetlen += 2;
    
    /*
     * encode the "SnmpAuthMsg" ( tag: [1] IMPLICIT SEQUENCE )
     */

    *packet++ = (CONTEXT_SPECIFIC | CONSTRUCTED | TAG_1);
    AuthMsg_len = packet++;
    *packetlen += 2;

    if (sess->srcParty.AuthProtocol) {
	digest_begin = packet;
	packet = ASN1_EncodeOctetString (packet, packetlen, ASN1_OCTET_STRING,
					 sess->srcParty.AuthPrivate, 16);
	digest_end = packet;
	packet = ASN1_EncodeInt (packet, packetlen, ASN1_INTEGER,
				 sess->dstParty.AuthClock);
	packet = ASN1_EncodeInt (packet, packetlen, ASN1_INTEGER,
				 sess->srcParty.AuthClock);
    } else {
	packet = ASN1_EncodeOctetString (packet, packetlen, ASN1_OCTET_STRING,
					 NULL, 0);
    }
    
    /*
     * encode the "SnmpMgmtCom" ( tag: [1] IMPLICIT SEQUENCE )
     */

    *packet++ = (CONTEXT_SPECIFIC | CONSTRUCTED | TAG_2);
    AuthData_len = packet++;
    *packetlen += 2;

    /*
     * encode the "dstParty", "srcParty" and "context" field
     */

    packet = ASN1_EncodeOID (packet, packetlen, 
			     sess->dstParty.Identity, 
			     sess->dstParty.IdentityLen);

    packet = ASN1_EncodeOID (packet, packetlen,
			     sess->srcParty.Identity,
			     sess->srcParty.IdentityLen);
    
    packet = ASN1_EncodeOID (packet, packetlen,
			     sess->context.Identity,
			     sess->context.IdentityLen);

    /*
     * encode PDU
     */

    if ((packet = encodePDU (interp, pdu, packet, packetlen)) == NULL) {
	return TCL_ERROR;
    }

    /*
     * Set the various length fields with their values. If one field
     * is longer than it's reserved space, make sure to shift the
     * other pointers.
     */

    packet = ASN1_EncodeLength (packet, packetlen,
				AuthData_len, packet - (AuthData_len + 1));
    packet = ASN1_EncodeLength (packet, packetlen,
				AuthMsg_len, packet - (AuthMsg_len + 1));
    packet = ASN1_EncodeLength (packet, packetlen, 
				PrivData_len, packet - (PrivData_len + 1));
    packet = ASN1_EncodeLength (packet, packetlen,
				PrivMsg_len, packet - (PrivMsg_len + 1));

    /*
     * call message digest computation if it's an authenticated message
     */

    if (sess->srcParty.AuthProtocol) {
	SNMP_MD5_digest (digest_begin, packet - digest_begin, md5Digest);
	digest_begin = ASN1_EncodeOctetString (digest_begin, &digest_len,
					       ASN1_OCTET_STRING, 
					       md5Digest, 16);

	if (digest_begin != digest_end) {
	    Tcl_SetResult (interp, "error computing message digest", 
			   TCL_STATIC);
	    return TCL_ERROR;
	}
    }

    return TCL_OK;
}

/*
 * encodePDU() takes a session and a PDU and serializes the ASN1 PDU
 * as an octet string into the buffer pointed to by packet using the
 * "Basic Encoding Rules".
 */

static u_char*
encodePDU (interp, pdu, packet, packetlen)
     Tcl_Interp		*interp;
     struct snmp_pdu	*pdu;
     u_char		*packet;
     int		*packetlen;
{    
    u_char *PDU_len = NULL, *VarBind_len = NULL, *VarBindList_len = NULL;
    
    int i, vblc, vbc;
    char **vblv, **vbv;
    char *value;

    oid *oidbuf;
    int oidlen;

    /*
     * encode pdu type ( tag: [pdu_type] IMPLICIT PDU )
     */
    
    *packet++  = (CONTEXT_SPECIFIC | CONSTRUCTED | pdu->type);
    PDU_len    = packet++;
    *packetlen += 2;

    if (pdu->type == SNMPv1_TRAP) {

	int generic = 0, specific = 0, len;
	char hostname[64];
	struct hostent *he;

	oidbuf = SNMP_Str2Oid (pdu->trapOID, &oidlen);
	if (strncmp (pdu->trapOID, "1.3.6.1.6.3.1.1.5", 17) == 0) {
	    oid *tmp = SNMP_Str2Oid ("1.3.6.1.4.1.1701", &len);
	    generic = oidbuf[oidlen-1] - 1;
	    specific = 0;
	    packet = ASN1_EncodeOID (packet, packetlen, tmp, len);
	} else {
	    generic = 6;
	    specific = oidbuf[oidlen-1];
	    packet = ASN1_EncodeOID (packet, packetlen, oidbuf, oidlen-2);
	}
	if (packet == NULL) {
	    Tcl_SetResult (interp, 
			   "failed to encode enterprise object identifier", 
			   TCL_STATIC);
	    return NULL;
	}
 
	if (gethostname (hostname, 64) < 0) {
	    strcpy (hostname, "loopback");
	}
	he = gethostbyname (hostname);

	packet = ASN1_EncodeOctetString (packet, packetlen, ASN1_IpAddress,
					 *he->h_addr_list, he->h_length);

	packet = ASN1_EncodeInt (packet, packetlen, ASN1_INTEGER, generic);
	packet = ASN1_EncodeInt (packet, packetlen, ASN1_INTEGER, specific);
	packet = ASN1_EncodeInt (packet, packetlen, ASN1_TimeTicks, 
				 SNMP_SysUpTime());

    } else {
    
	packet = ASN1_EncodeInt (packet, packetlen,
				 ASN1_INTEGER, pdu->request_id);
	packet = ASN1_EncodeInt (packet, packetlen,
				 ASN1_INTEGER, pdu->error_status);
	switch (pdu->error_status) {
	  case E_TOOBIG:	snmp_stats.snmpOutTooBigs++; break;
	  case E_NOSUCHNAME:	snmp_stats.snmpOutNoSuchNames++; break;
	  case E_BADVALUE:	snmp_stats.snmpOutBadValues++; break;
	  case E_READONLY:	break; /* not used */
	  case E_GENERR:	snmp_stats.snmpOutGenErrs++; break;
	}
	packet = ASN1_EncodeInt (packet, packetlen,
				 ASN1_INTEGER, pdu->error_index);
    }

    /*
     * encode VarBindList ( SEQUENCE of VarBind )
     */
    
    *packet++       = (UNIVERSAL | CONSTRUCTED | ASN1_SEQUENCE);
    VarBindList_len = packet++;
    *packetlen      += 2;
    
    /*
     * split the varbind list and loop over all elements
     */
    
    if (Tcl_SplitList (interp, Tcl_DStringValue (&pdu->varbind), &vblc, &vblv)
	!= TCL_OK) {
	return NULL;
    }

    if (pdu->type == SNMPv2_TRAP) {

	/* 
	 * encode two default var binds: sysUpTime.0 and snmpTrapOID.0
	 * as defined in RFC 1448
	 */
	   
	*packet++   = (UNIVERSAL | CONSTRUCTED | ASN1_SEQUENCE);
	VarBind_len = packet++;
	*packetlen  += 2;

	oidbuf = SNMP_Str2Oid ("1.3.6.1.2.1.1.3.0", &oidlen);
	packet = ASN1_EncodeOID (packet, packetlen, oidbuf, oidlen);

	packet = ASN1_EncodeInt (packet, packetlen, ASN1_TimeTicks, 
				 SNMP_SysUpTime());
	packet = ASN1_EncodeLength (packet, packetlen,
				    VarBind_len,
				    packet - (VarBind_len + 1));

	*packet++   = (UNIVERSAL | CONSTRUCTED | ASN1_SEQUENCE);
	VarBind_len = packet++;
	*packetlen  += 2;

	oidbuf = SNMP_Str2Oid ("1.3.6.1.6.3.1.1.4.1.0", &oidlen);
	packet = ASN1_EncodeOID (packet, packetlen, oidbuf, oidlen);

	oidbuf = SNMP_Str2Oid (pdu->trapOID, &oidlen);
	packet = ASN1_EncodeOID (packet, packetlen, oidbuf, oidlen);
	
	packet = ASN1_EncodeLength (packet, packetlen,
				    VarBind_len,
				    packet - (VarBind_len + 1));
    }
    
    for (i = 0; i < vblc; i++) {
	
	int asn1_type = ASN1_OTHER;
	
	/*
	 * split a single varbind into its components
	 */
	
	if (Tcl_SplitList (interp, vblv[i], &vbc, &vbv) != TCL_OK) {
	    free ((char *) vblv);
	    return NULL;
	}

	if (vbc == 0) {
	    Tcl_SetResult (interp, "missing OBJECT IDENTIFIER", TCL_STATIC);
	    free ((char *) vblv);
            return NULL;
	}
	
	/*
	 * encode each VarBind ( SEQUENCE name, value )
	 */
	
	*packet++   = (UNIVERSAL | CONSTRUCTED | ASN1_SEQUENCE);
	VarBind_len = packet++;
	*packetlen  += 2;

	/*
	 * encode the object identifier, perhaps consulting the MIB
	 */
	
	if (SNMP_IsOid (vbv[0])) {
	    oidbuf = SNMP_Str2Oid (vbv[0], &oidlen);
	} else {
	    char *tmp = MIB_Oid (vbv[0], 0);
	    if (! tmp) {
		Tcl_ResetResult (interp);
		Tcl_AppendResult (interp, "invalid arg \"", vbv[0],
				  ": expecting OBJECT IDENTIFIER",
				  (char *) NULL);
		free ((char *) vbv);
		free ((char *) vblv);
		return NULL;
	    }
	    oidbuf = SNMP_Str2Oid (tmp, &oidlen);
	}
	packet = ASN1_EncodeOID (packet, packetlen, oidbuf, oidlen);

	/*
	 * guess the asn1 type field and the value
	 */

	switch (vbc) {
	  case 1:
	    value = "";
	    asn1_type = ASN1_NULL;
	    break;
	  case 2:
	    value = vbv[1];
	    asn1_type = MIB_ASN1 (vbv[0], 0);
	    break;
	  default:
	    value = vbv[2];
	    asn1_type = ASN1_Str2Sntx (vbv[1]);
	    break;
	}

	/*
         * if it's a SET request, we'll have to encode the VALUE
	 */

	if (pdu->type == SET_REQUEST || pdu->type == RESPONSE
	    || pdu->type == SNMPv1_TRAP || pdu->type == SNMPv2_TRAP) {

	    switch (asn1_type) {
	      case ASN1_INTEGER:
	      case ASN1_Counter:
	      case ASN1_Gauge:
	      case ASN1_UInteger32:
		{   int int_val, rc;
		    char *tmp = MIB_Scan (vbv[0], 0, value);
		    if (tmp && *tmp) {
			rc = Tcl_GetInt (interp, tmp, &int_val);
		    } else {
			rc = Tcl_GetInt (interp, value, &int_val);
		    }
		    if (rc != TCL_OK) return NULL;
		    packet = ASN1_EncodeInt (packet, packetlen,
					     ASN1_INTEGER, int_val);
		}
		break;
	      case ASN1_BIT_STRING:
		packet = ASN1_EncodeNull (packet, packetlen);
		break;
	      case ASN1_TimeTicks:
		{   int d, h, m, s, f, n, val;
		    n = sscanf (value, "%dd %d:%d:%d.%d", 
				&d, &h, &m, &s, &f);
		    if (n == 5) {
			val = d * 8640000 + h * 360000 
					+ m * 6000 + s * 100 + f;
		    } else {
			val = atoi (value);
		    }
		    packet = ASN1_EncodeInt (packet, packetlen,
					     ASN1_TimeTicks, val);
		}
                break;
	      case ASN1_IpAddress:
		{   int addr = inet_addr (value);
		    ASN1_EncodeOctetString (packet, packetlen, ASN1_IpAddress,
					    (char *) &addr, 4);
		}
		break;
	      case ASN1_OCTET_STRING:
		{   char *buffer = NULL, *fmt = value;
		    int len;
		    char *tmp = MIB_Scan (vbv[0], 0, value);
		    if (tmp) fmt = tmp;
		    if (*fmt) {
			buffer = xmalloc (strlen (fmt));
			hex_to_bin (fmt, buffer, &len);
		    } else {
			len = 0;
		    }
		    packet = ASN1_EncodeOctetString (packet, packetlen,
						     ASN1_OCTET_STRING,
						     buffer, len);
		    if (buffer) free (buffer);
		}
		break;
	      case ASN1_OBJECT_IDENTIFIER:
		if (! SNMP_IsOid (value)) {
		    char *tmp = MIB_Oid (value, 0);
		    if (!tmp) {
			Tcl_AppendResult (interp, 
					  "illegal object identifier \"",
					  value, "\"", (char *) NULL);
			return NULL;
		    }
		    oidbuf = SNMP_Str2Oid (tmp, &oidlen);
		} else {
		    oidbuf = SNMP_Str2Oid (value, &oidlen);
		}
		packet = ASN1_EncodeOID (packet, packetlen, oidbuf, oidlen);
		break;
	      default:
		packet = ASN1_EncodeNull (packet, packetlen);
		break;
	    }
	} else {
	    packet = ASN1_EncodeNull (packet, packetlen);
	}

	packet = ASN1_EncodeLength (packet, packetlen,
				    VarBind_len,
				    packet - (VarBind_len + 1));

	free ((char **) vbv);
    }

    free ((char **) vblv);

    packet = ASN1_EncodeLength (packet, packetlen, VarBindList_len,
				packet - (VarBindList_len + 1));
    packet = ASN1_EncodeLength (packet, packetlen, PDU_len,
				packet - (PDU_len + 1));
    return packet;
}
