/*
 * straps.c:						Sept 1994
 *
 * A simple SNMP trap-sink. Mainly for scotty's snmp code, but
 * commonly useable by other clients.
 *
 * The trap demon listens to the snmp-trap port 162/udp and forwards
 * the received event to connected clients (like scotty).  
 *
 * Because the port 162 needs root access and the port can be opend
 * only one time, the use of a simple forwarding demon is a good choice.
 *
 * The client can connect to the AF_UNIX domain stream socket
 * /tmp/.straps and will get the trap-packets in raw binary form:
 * 4 bytes data-length (in host-byte-order) followed by the data-bytes
 * of the packet.
 *
 */

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <sys/un.h>
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_SYS_SELECT_H
# include <sys/select.h>
#endif
#ifndef FD_SETSIZE
# define FD_SETSIZE 32
#endif


/* default trap service name: */
static char *snmp_trap_service_name = "snmp-trap";

/* default trap service port (if service_name failed): */
static short snmp_trap_service_port = 162;

/* default path to the unix-stream service socket: */
static char *serv_path = "/tmp/.straps";


#ifdef SIGPIPE
/* ignore broken pipes */
static void
ign_pipe ()
{
    /* restart signalhandler for all these bozo's outside: */
    signal (SIGPIPE, ign_pipe);
}
#endif


int
main (argc, argv)
int argc;
char *argv[];
{
    struct servent *se;
    struct sockaddr_in taddr;
    struct sockaddr_un saddr, daddr;
    int trap_s, serv_s, slen, dlen, rc, i;
    fd_set fds;
    static int cl_addr [FD_SETSIZE];
    char buf [2048];
    int go_on;
    
    /*
     * check args:
     */

    if (argc > 1)
      {
	  fprintf (stderr, "use:  straps\nto run the snmp-trap demon.\n");
	  exit (1);
      }
    
    /*
     * fetch trap port:
     */
    
    if ((se = getservbyname (snmp_trap_service_name, "udp")))
      snmp_trap_service_port = se->s_port; 
    else
      snmp_trap_service_port = htons (snmp_trap_service_port);
      

    /*
     * open trap port:
     */

    if ((trap_s = socket (AF_INET, SOCK_DGRAM, 0)) < 0)
      {
	  perror ("straps: cannot get trap-socket; reason");
	  exit (1);
      }
    
    taddr.sin_family = AF_INET;
    taddr.sin_port = snmp_trap_service_port;
    taddr.sin_addr.s_addr = INADDR_ANY;

    if (bind (trap_s, (struct sockaddr *) &taddr, sizeof (taddr)) < 0)
      {
	  perror ("straps: cannot bind trap-socket; reason");
	  exit (1);
      }

    /*
     * open client socket:
     */

    /* remove old socket: */
    unlink (serv_path);

    if ((serv_s = socket (AF_UNIX, SOCK_STREAM, 0)) < 0)
      {
	  perror ("straps: cannot get serv-socket; reason");
	  exit (1);
      }

    memset ((char *) &saddr, 0, sizeof(saddr));
    
    saddr.sun_family = AF_UNIX;
    strcpy (saddr.sun_path, serv_path);
    slen = sizeof(saddr.sun_family) + strlen (saddr.sun_path);
#ifdef NETBSD
    slen += sizeof(saddr.sun_len);
#endif

    if (bind (serv_s, (struct sockaddr *) &saddr, slen) < 0)
      {
	  perror ("straps: cannot bind serv-socket; reason");
	  exit (1);
      }
    
    if (listen (serv_s, 5) < 0)
      {
          perror ("straps: cannot listen on serv-socket; reason");
          exit (1);
      }

#ifdef SIGPIPE
	  /* ignore broken pipes */
	  signal (SIGPIPE, ign_pipe);
#endif
    
    /*
     * fine anything is ready; lets listen for events: 
     */

    for (;;)
      {
	  FD_ZERO (&fds);
	  FD_SET (trap_s, &fds);
	  FD_SET (serv_s, &fds);

	  rc = select (FD_SETSIZE, &fds, (fd_set *) 0, (fd_set *) 0, 
		       (struct timeval *) 0);
	  if (rc < 0)
	    {
		if (errno == EINTR || errno == EAGAIN)
		  continue;
		perror ("straps: select failed; reason");
	    }
	  
	  if (FD_ISSET (trap_s, &fds))
	    {
		if ((rc = recv (trap_s, buf, sizeof (buf), 0)) < 0)
		  {

		      perror ("straps: recv failed; reason");
		      continue;
		  }
		
		go_on = 0;

		for (i = 0; i < FD_SETSIZE; i++)
		  {
		      if (cl_addr [i] > 0)
			{
			    /* XXX: writeable */
			    if (write (i, (char *) &rc, 4) != 4
				|| write (i, buf, rc) != rc)
			      {
				  cl_addr [i] = 0;
				  close (i);
			      }

			    go_on++;
			}
		  }

		if (! go_on) break;
	    }
	  else
	    {
		rc = accept (serv_s, (struct sockaddr *) &daddr, &dlen);
		if (rc < 0)
		  {
		      perror ("straps: accept failed; reason");
		      continue;
		  }
		cl_addr [rc] = 1;
	    }

      } /* end for (;;) */

    unlink (serv_path);

    return 0;
}

/* end of straps.c */
