/*
 * udp_dom.c
 *
 * This file contains all functions that handle transport over UDP.
 * There is also some code here to talk to the straps daemon which is
 * used to receive and forward SNMP traps send to the privileged 162
 * port.
 *
 * Copyright (c) 1994
 *
 * Sven Schmidt, J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif

#include <tcl.h>

#ifdef DBMALLOC
#include <dbmalloc.h>
#endif

#include "snmp.h"
#include "asn1.h"
#include "xmalloc.h"
#include "udp.h"

/*
 * local variables:
 */

extern int	hexdump;		/* flag that controls hexdump */
static int	sock = -1;		/* socket to send/receive messages */
static int	trap_sock = -1;		/* socket to receive traps */

static char	*serv_path = "/tmp/.straps";

/*
 * The default filename where we will find the straps binary. This
 * is normally overwritten in the Makefile.
 */

#ifndef STRAPS
#define STRAPS "/usr/local/bin/straps"
#endif

/*
 * Forward declarations for procedures defined later in this file:
 */

static int
xread			_ANSI_ARGS_((int fd, char *buf, int len));

static int
straps			_ANSI_ARGS_((Tcl_Interp *interp));

static void
packetdump		_ANSI_ARGS_((u_char *packet, int packetlen,
				     char *msg));

#ifdef SIGPIPE
static void
ign_pipe		_ANSI_ARGS_((void));
#endif

/*
 * Ignore borken pipe signals. Restart signalhandler for all these 
 * bozo's outside.
 */

#ifdef SIGPIPE
static void
ign_pipe ()
{
    signal (SIGPIPE, ign_pipe);
}
#endif

/*
 * Read a buffer from a file descriptor. This wrapper is needed on
 * broken SYS V machines to handle interrupted system calls.
 */

static int
xread (fd, buf, len)
     int fd;
     char *buf;
     int len;
{
    int rc;
    
    while ((rc = read (fd, buf, len)) < 0
           && (errno == EINTR || errno == EAGAIN))
            continue;

    return rc;
}

/*
 * straps() starts the straps forwarder daemon.
 */

static int
straps (interp)
     Tcl_Interp *interp;
{
    int pid;
    
    pid = fork();
    if (pid < 0) {
        Tcl_AppendResult (interp, "fork failed: ",
                          Tcl_PosixError (interp), (char *) NULL);
        return TCL_ERROR;
    }
    
    if (pid == 0) {                                       /* child */

        int i;
        for (i = 3; i < 64; i++) close (i);
	
        execl (STRAPS, STRAPS, (char *) 0);
        perror ("SNMP: execl straps");
        exit (1);
    }
    
    return TCL_OK;
}

/*
 * packetdump() prints a hex dump of a packet. Useful for debugging 
 * this code.  The message given in the third parameter should be used 
 * to identify the received packet.
 */

static void
packetdump (packet, packetlen, msg)
     u_char *packet;
     int packetlen;
     char *msg;
{
    u_char *cp = packet;
    int	i = 0;
    
    if (msg) {
	printf ("%s\n", msg);
    }

    while (i++ < packetlen) {
	printf ("%02x", *cp++);
	
	if (i++ < packetlen)
	  printf ("%02x ", *cp++);
	
	if ((i % 16) == 0)
	  printf ("\n");
    }
    printf ("\n\n");
}

/*
 * SNMP_ManagerSocket() opens a UDP-Port for the sessions and
 * associates the processing function with that file handle for
 * asychronous response handling.
 */

int
SNMP_ManagerSocket (interp)
     Tcl_Interp *interp;
{
    struct sockaddr_in myaddr;

    if (sock > 0) {
	return TCL_OK;
    }

    if ((sock = socket (AF_INET, SOCK_DGRAM, 0)) < 0) {
        Tcl_AppendResult (interp, "can not create socket: ", 
			  Tcl_PosixError (interp), (char *) NULL);
	return TCL_ERROR;
    }
    
    myaddr.sin_family = AF_INET;
    myaddr.sin_port = 0;
    myaddr.sin_addr.s_addr = INADDR_ANY;
    
    if (bind (sock, (struct sockaddr *) &myaddr, sizeof (myaddr)) < 0) {
        Tcl_AppendResult (interp, "can not bind socket: ", 
			  Tcl_PosixError (interp), (char *) NULL);
	close (sock);
	sock = -1;
	return TCL_ERROR;
    }
    
    Tk_CreateFileHandler (sock, TK_READABLE, SNMP_ResponseEvent,
			  (ClientData *) interp);
    return TCL_OK;
}

/*
 * SNMP_ManagerClose() closes the socket used to handle manager
 * initiated SNMP communication.
 */

void
SNMP_ManagerClose ()
{
    Tk_DeleteFileHandler (sock);
    close (sock);
    sock = -1;
}

/*
 * SNMP_Wait() waits for an answer. This is used to implement 
 * synchronous operations. The argument defines the number of
 * milliseconds to wait.
 */

int
SNMP_Wait (ms)
     int ms;
{
    struct timeval wait;
    int width = sock + 1;
    fd_set readfds;

    wait.tv_sec  = ms / 1000;
    wait.tv_usec = (ms % 1000) * 1000;
    FD_ZERO (&readfds);
    FD_SET (sock, &readfds);

    return select (width, &readfds, (fd_set *)0, (fd_set *)0, &wait);
}

/*
 * SNMP_TrapSocket() creates a socket to the straps trap daemon. 
 */

int
SNMP_TrapSocket (interp)
     Tcl_Interp *interp;
{
    int i, rc;
    struct sockaddr_un saddr;
    int slen;
    
    if ((trap_sock = socket (AF_UNIX, SOCK_STREAM, 0)) < 0) {
	Tcl_AppendResult (interp, "can not create straps socket: ",
			  Tcl_PosixError (interp), (char *) NULL);
	return TCL_ERROR;
    }
    
    memset ((char *) &saddr, 0, sizeof(saddr));
    
    saddr.sun_family = AF_UNIX;
    strcpy (saddr.sun_path, serv_path);
    slen = sizeof(saddr.sun_family) + strlen (saddr.sun_path);
#ifdef NETBSD
    slen += sizeof(saddr.sun_len);
#endif
    
    if (connect (trap_sock, (struct sockaddr *) &saddr, slen) < 0) {
	
	if (straps (interp) != TCL_OK) return TCL_ERROR;
	
	for (i = 0; i < 5; i++) {
	    sleep (1);
	    rc = connect (trap_sock, (struct sockaddr *) &saddr, slen);
	    if (rc >= 0) break;
	}
	
	if (rc < 0) {
	    Tcl_AppendResult (interp, "can not connect straps socket: ",
			      Tcl_PosixError (interp), (char *) NULL);
	    close (trap_sock);
	    trap_sock = -1;    
	    return TCL_ERROR;
	}
    }

#ifdef SIGPIPE
    signal (SIGPIPE, ign_pipe);
#endif

    Tk_CreateFileHandler (trap_sock, TK_READABLE, SNMP_TrapEvent,
			  (ClientData *) interp);
    return TCL_OK;
}

/*
 * SNMP_TrapClose() closes the connection to the straps daemon.
 */

void
SNMP_TrapClose ()
{
    Tk_DeleteFileHandler (trap_sock);
    close (trap_sock);
    trap_sock = -1;    
}

/*
 * SNMP_AgentSocket() creates a socket for the agent part on a given 
 * port. Note, we currently only allow one agent on one port. This may 
 * change if we feel need for it.
 */

int
SNMP_AgentSocket (interp, session)
     Tcl_Interp *interp;
     struct session *session;
{
    struct sockaddr_in myaddr;
    struct agentToken *token;
    
    if (session->agentSocket > 0) return TCL_OK;

    if ((session->agentSocket = socket (AF_INET, SOCK_DGRAM, 0)) < 0) {
        Tcl_AppendResult (interp, "can not create socket: ",
                          Tcl_PosixError (interp), (char *) NULL);
        return TCL_ERROR;
    }
    
    myaddr.sin_family = AF_INET;
    myaddr.sin_port = htons (session->srcParty.TPort);
    myaddr.sin_addr.s_addr = INADDR_ANY;
    
    if (bind (session->agentSocket, (struct sockaddr *) &myaddr, 
	      sizeof (myaddr)) < 0) {
        Tcl_AppendResult( interp, "can not bind socket: ",
                         Tcl_PosixError (interp), (char *) NULL);
        return TCL_ERROR;
    }

    token = (struct agentToken *) xmalloc (sizeof (struct agentToken));
    token->interp = interp;
    token->session = session;

    Tk_CreateFileHandler (session->agentSocket, TK_READABLE, 
			  SNMP_AgentEvent, (ClientData *) token);
    
    return TCL_OK;    
}

/*
 * SNMP_AgentClose() closes the socket for incoming requests.
 */

void
SNMP_AgentClose (session)
     struct session *session;
{
    Tk_DeleteFileHandler (session->agentSocket);
    close (session->agentSocket);
    session->agentSocket = 0;
}

/*
 * SNMP_Host2Addr() returns the network address for the named host.  A
 * null pointer is returned if gethostbyname doesn't find a matching
 * hostname in /etc/hosts. Otherwise the structure containing the
 * network address is returned.
 */

int
SNMP_Host2Addr	(interp, host, port, addr)
     Tcl_Interp	*interp;
     char *host;
     unsigned short port;
     struct sockaddr_in *addr;
{
    struct hostent *hp;

    if ((hp = gethostbyname (host))) {
	memcpy ((char *) &addr->sin_addr, (char *) hp->h_addr, hp->h_length);
    } else {
	int hostaddr = inet_addr (host);
	if (hostaddr == -1) return 0;
	memcpy( (char *) &addr->sin_addr, (char *) &hostaddr, 4);
    } 

    addr->sin_family = AF_INET;
    addr->sin_port = htons (port);
    
    return 1;
}

/*
 * SNMP_Send() takes a serialized packet pointed to by packet
 * and sends it to the destination defined in the socket descriptor of
 * this session.
 */

int
SNMP_Send (interp, sess, packet, packetlen)
     Tcl_Interp	*interp;
     struct session *sess;
     u_char *packet;
     int packetlen;
{
    if (sendto (sock, packet, packetlen, 0,
		(struct sockaddr *) &sess->addr, sizeof (sess->addr)) < 0) {
        Tcl_AppendResult (interp, "sendto failed: ", 
			  Tcl_PosixError (interp), (char *) NULL);
	return TCL_ERROR;
    }

    snmp_stats.snmpOutPkts++;

    if (hexdump) {
	packetdump (packet, packetlen, "Manager -> Agent (Request)");
    }

    return TCL_OK;
}

/*
 * SNMP_Recv() reads from the socket, if an incoming response is expected.
 */

int
SNMP_Recv (interp, packet, packetlen)
     Tcl_Interp	*interp;
     u_char	*packet;
     int	*packetlen;
{
    struct sockaddr_in from;
    int	fromlen = sizeof (from);

    if ((*packetlen = recvfrom (sock, packet, *packetlen, 0,
				(struct sockaddr *)&from,
				&fromlen)) < 0) {
	Tcl_AppendResult (interp, "recvfrom failed: ",
			  Tcl_PosixError (interp), (char *) NULL);
	return TCL_ERROR;
    }

    snmp_stats.snmpInPkts++;

    if (hexdump) {
	packetdump (packet, *packetlen, "Agent -> Manager (Response)");
    }

    return TCL_OK;
}

/*
 * SNMP_TrapRecv() reads a trap message from the straps daemon socket
 * and returns the packet.
 */

int
SNMP_TrapRecv (interp, packet, packetlen)
     Tcl_Interp *interp;
     u_char     *packet;
     int        *packetlen;
{
    int len, rlen, four;
    char c;

    /* read the length of the trap packet and then the packet itself */

    if ((four = xread (trap_sock, (char *) &len, 4)) < 0
        || xread (trap_sock, packet, 
		  rlen = (len < *packetlen ? len : *packetlen)) < 0) {
	Tcl_AppendResult (interp, "read failed: ",
			  Tcl_PosixError (interp), (char *) NULL);
	return TCL_ERROR;
    }

    if (four != 4) {
        Tk_DeleteFileHandler (trap_sock);
        close (trap_sock);
        trap_sock = -1;
	Tcl_AppendResult (interp, "lost connection to straps daemon");
	return TCL_ERROR;
    }

    /* eat up remaining data-bytes */

    while (len > *packetlen) {
        if (xread (trap_sock, &c, 1) != 1) {
	    Tcl_AppendResult (interp, "lost connection to straps daemon");
	    return TCL_ERROR;
	}
        len--;
    }

    *packetlen = rlen;

    snmp_stats.snmpInPkts++;

    if (hexdump) {
	packetdump (packet, *packetlen, "Agent -> Manager (Trap)");
    }

    return TCL_OK;
}

/*
 * SNMP_AgentRecv() reads from the socket, if an incoming request 
 * is expected.
 */

int
SNMP_AgentRecv (interp, session, packet, packetlen)
     Tcl_Interp	*interp;
     struct session *session;
     u_char *packet;
     int *packetlen;
{
    struct sockaddr_in from;
    int	fromlen = sizeof (from);

    if ((*packetlen = recvfrom (session->agentSocket, packet, *packetlen, 0,
				(struct sockaddr *) &session->addr,
				&fromlen)) < 0) {
	Tcl_AppendResult (interp, "recvfrom failed: ",
			  Tcl_PosixError (interp), (char *) NULL);
	return TCL_ERROR;
    }

    snmp_stats.snmpInPkts++;

    if (hexdump) {
	packetdump (packet, *packetlen, "Agent -> Manager (Response)");
    }

    return TCL_OK;
}

