/*
 * sunrpc.c
 *
 * Extend a tcl command interpreter with some sunrpc commands.
 *
 * Copyright (c) 1993, 1994
 *
 * J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 *
 */

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <string.h>
#include <stdio.h>
#include <netdb.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <rpc/rpc.h>
#include <rpc/pmap_prot.h>
#include <rpc/pmap_clnt.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <tcl.h>

#include "scotty.h"
#include "xmalloc.h"

#include "rstat.h"

#undef h_addr					/* fix suggested by Erik */
#include "ether.h"
#define h_addr h_addr_list[0]

#include "mount.h"

#ifdef NO_RPCENT
struct rpcent {
        char    *r_name;        /* name of server for this rpc program */
        char    **r_aliases;    /* alias list */
        int     r_number;       /* rpc program number */
};
#endif

static char str[512];

/*
 * Forward declarations for procedures defined later in this file:
 */

static char*
gethostnamebyany _ANSI_ARGS_((Tcl_Interp *interp,  char *str));

static int
init_socket      _ANSI_ARGS_((Tcl_Interp *interp, 
			      struct sockaddr_in *rsocket, char *host));

static int 
open_etherd      _ANSI_ARGS_((Tcl_Interp *interp, char *host));

static int
close_etherd     _ANSI_ARGS_((Tcl_Interp *interp, char *host));

static int
do_etherd        _ANSI_ARGS_((Tcl_Interp *interp, char *host));

static int
do_rstat         _ANSI_ARGS_((Tcl_Interp *interp, char *host));

static int
do_info          _ANSI_ARGS_((Tcl_Interp *interp, char *host));

static int
do_mount         _ANSI_ARGS_((Tcl_Interp *interp, char *host));

static int
do_exports       _ANSI_ARGS_((Tcl_Interp *interp, char *host));

static int
do_probe         _ANSI_ARGS_((Tcl_Interp *interp, char *host, 
			      unsigned long prognum, unsigned long version, 
			      unsigned protocol));


/*
 * Make sure that str points to a hostname and not to an ip address.
 */

static char* 
gethostnamebyany (interp, str)
    Tcl_Interp *interp;
    char *str;
{
    struct hostent *hostent;
    int hostaddr;

    hostent = gethostbyname (str);
    if (hostent != NULL) return xstrdup (str);

    /* this may be an IP address in numerical form */

    if ((hostaddr = inet_addr(str)) == -1) {
        Tcl_AppendResult (interp, "unknown host name ", str, (char *) NULL);
        return NULL;    
    }
    hostent = gethostbyaddr ((char *) &hostaddr, sizeof(hostaddr), AF_INET);
    if (hostent == NULL) {
        Tcl_AppendResult (interp, "unknown host name ", str, (char *) NULL);
        return NULL;
    }
    
    return xstrdup ((char *) hostent->h_name);
}

static int 
init_socket (interp, rsocket, host)
    Tcl_Interp *interp;
    struct sockaddr_in *rsocket;
    char* host;
{
    struct hostent *hp;

    if ((host = gethostnamebyany (interp, host)) == NULL)
        return TCL_ERROR;

    rsocket->sin_family = AF_INET;
    rsocket->sin_port = 0;
    rsocket->sin_addr.s_addr = INADDR_ANY;

    if ((hp = gethostbyname (host)) == NULL) {
	Tcl_AppendResult (interp, "unknown host name ", host, (char *) NULL);
	return TCL_ERROR;
    }
    rsocket->sin_addr.s_addr = * (unsigned long *) hp->h_addr;

    free (host);
    return TCL_OK;
}

/*
 * Get some network monitoring information from suns etherd.
 */

struct ether_handle {
    char   *name;
    CLIENT *client;
    etherstat estat;
    struct ether_handle *next;
};
typedef struct ether_handle ether_handle;

static ether_handle *eh_list = NULL;

static int 
open_etherd (interp, host)
    Tcl_Interp *interp;
    char *host;
{
    int dummy;
    ether_handle *p;
    etherstat *res;

    if ((host = gethostnamebyany (interp, host)) == NULL)
        return TCL_ERROR;

    for (p = eh_list; p != NULL; p = p->next) {
	if (!strcmp (host, p->name)) {
	    free (host);
	    return TCL_OK;
	}
    }

    p = (ether_handle *) xmalloc (sizeof(ether_handle));

#ifdef NO_CLNT_CREATE
    p->client = NULL;
#else
    p->client = clnt_create (host, ETHERPROG, ETHERVERS, "udp" );
#endif
    if (p->client == NULL) {
	Tcl_AppendResult (interp, "can not connect to ", host, (char *) NULL);
	free (host);
        return TCL_ERROR;
    }

    etherproc_on_1 (&dummy, p->client);

    res = etherproc_getdata_1 (&dummy, p->client);
    if (res == NULL) {
	Tcl_AppendResult (interp, "can not connect to ", host, (char *) NULL);
	free (host);
	return TCL_ERROR;
    }
    p->estat = *res;
    p->name = xstrdup (host);
    p->next = eh_list;
    eh_list = p;
    
    free (host);
    return TCL_OK;
}

static int 
close_etherd (interp, host)
    Tcl_Interp *interp;
    char *host;
{
    int dummy;
    ether_handle *p;
    ether_handle *q = NULL;

    if ((host = gethostnamebyany (interp, host)) == NULL)
        return TCL_ERROR;

    for (p = eh_list; p != NULL; p = p->next) {
        if (!strcmp (host,p->name)) {
	    etherproc_off_1 (&dummy, p->client);
	    if (q == NULL) {
		eh_list = p->next;
	    } else {
		q->next = p->next;
	    }
	    free (p->name);
	    free (p);
        }
	q = p;
    }
    
    free (host);
    return TCL_OK;
}

static int
do_etherd (interp, host)
    Tcl_Interp *interp;
    char *host;
{
    int dummy, tdiff;
    ether_handle *p;
    etherstat *res;

    if ((host = gethostnamebyany (interp, host)) == NULL)
        return TCL_ERROR;

    for (p = eh_list; p != NULL; p = p->next) {
	if (!strcmp (host, p->name)) {
	    res = etherproc_getdata_1 (&dummy, p->client);
	    if (res == NULL) {
		Tcl_AppendResult (interp, "can not connect to ", host,
				  (char *) NULL);
		return TCL_ERROR;
	    }
	    tdiff = res->e_time.tv_useconds>p->estat.e_time.tv_useconds ?
		    (res->e_time.tv_useconds - p->estat.e_time.tv_useconds) :
	            (1000000 - p->estat.e_time.tv_useconds - res->e_time.tv_useconds);
	    tdiff = tdiff/1000 + 1000 * 
		    (res->e_time.tv_seconds - p->estat.e_time.tv_seconds);
	    sprintf(str,"{time TimeTicks %u} {bytes Gauge %u} {packets Gauge %u} {bcast Gauge %u}",
		    tdiff, 
		    res->e_bytes - p->estat.e_bytes, 
		    res->e_packets - p->estat.e_packets, 
		    res->e_bcast - p->estat.e_bcast);
	    Tcl_AppendResult (interp, str, (char*) NULL);
	    
	    p->estat = *res;
	    free (host);
	    return TCL_OK;
	}
    }

    Tcl_AppendResult (interp, "ether: first open ", host, (char*) NULL);

    free (host);
    return TCL_ERROR;
}

/* 
 * Get some information about the status of a host using the rstat RPC.
 * The result is a list of name type value triples as returned by the
 * snmp interface.
 */

static int
do_rstat (interp, host)
    Tcl_Interp *interp;
    char *host;
{
    int rc, dummy;
    struct statstime statp;
    
    if ((host = gethostnamebyany (interp, host)) == NULL)
        return TCL_ERROR;
    
    rc = callrpc (host, RSTATPROG, RSTATVERS_TIME, RSTATPROC_STATS,
		  xdr_void, (char *) &dummy,
		  xdr_statstime, (char *) &statp);

    if (rc != 0) {
	char *p = clnt_sperrno (rc);
	if (strncmp(p, "RPC: ", 5) == 0) p += 5;
	Tcl_AppendResult (interp, p, (char *) NULL);
	free (host);
	return TCL_ERROR;
    }

    sprintf (str,"cp_user Counter %d", statp.cp_time[0]);
    Tcl_AppendElement (interp, str);
    sprintf (str,"cp_nice Counter %d", statp.cp_time[1]);
    Tcl_AppendElement (interp, str);
    sprintf (str, "cp_system Counter %d", statp.cp_time[2]);
    Tcl_AppendElement (interp, str);
    sprintf (str, "cp_idle Counter %d", statp.cp_time[3]);
    Tcl_AppendElement (interp, str);

    sprintf (str, "dk_xfer_0 Counter %d", statp.dk_xfer[0]);
    Tcl_AppendElement (interp, str);
    sprintf (str, "dk_xfer_1 Counter %d", statp.dk_xfer[1]);
    Tcl_AppendElement (interp, str);
    sprintf (str, "dk_xfer_2 Counter %d", statp.dk_xfer[2]);
    Tcl_AppendElement (interp, str);
    sprintf (str, "dk_xfer_3 Counter %d", statp.dk_xfer[3]);
    Tcl_AppendElement (interp, str);

    sprintf (str, "v_pgpgin Counter %d", statp.v_pgpgin);
    Tcl_AppendElement (interp, str);
    sprintf (str, "v_pgpgout Counter %d", statp.v_pgpgout);
    Tcl_AppendElement (interp, str);
    sprintf (str, "v_pswpin Counter %d", statp.v_pswpin);
    Tcl_AppendElement (interp, str);
    sprintf (str, "v_pswpout Counter %d", statp.v_pswpout);
    Tcl_AppendElement (interp, str);

    sprintf (str, "v_intr Counter %d", statp.v_intr);
    Tcl_AppendElement (interp, str);
    sprintf (str, "v_swtch Counter %d", statp.v_swtch);
    Tcl_AppendElement (interp, str);

    sprintf (str, "if_ipackets Counter %d", statp.if_ipackets);
    Tcl_AppendElement (interp, str);
    sprintf (str, "if_ierrors Counter %d", statp.if_ierrors);
    Tcl_AppendElement (interp, str);
    sprintf (str, "if_opackets Counter %d", statp.if_opackets);
    Tcl_AppendElement (interp, str);
    sprintf (str, "if_oerrors Counter %d", statp.if_oerrors);
    Tcl_AppendElement (interp, str);

    sprintf (str, "avenrun_0 Gauge %d", statp.avenrun[0]);
    Tcl_AppendElement (interp, str);
    sprintf (str, "avenrun_1 Gauge %d", statp.avenrun[1]);
    Tcl_AppendElement (interp, str);
    sprintf (str, "avenrun_2 Gauge %d", statp.avenrun[2]);
    Tcl_AppendElement (interp, str);

    sprintf (str, "boottime TimeTicks %d", statp.boottime.tv_sec);
    Tcl_AppendElement (interp, str);
    sprintf (str, "curtime TimeTicks %d", statp.curtime.tv_sec);
    Tcl_AppendElement (interp, str);

    free (host);
    return TCL_OK;
}

/*
 * Ask the portmapper about registered RPC services.
 */

static int
do_info (interp, host)
    Tcl_Interp *interp;
    char *host;
{
    struct sockaddr_in _socket;
    struct sockaddr_in *rsocket = &_socket;
    struct pmaplist *portmapperlist;

    if (init_socket(interp, rsocket, host) != TCL_OK) 
        return TCL_ERROR;

    portmapperlist = pmap_getmaps (rsocket);
    while (portmapperlist) {
	int prog = portmapperlist->pml_map.pm_prog;
	struct rpcent *re = getrpcbynumber(prog);
	sprintf (str, "%lu %2lu %s %5lu %s",
		 portmapperlist->pml_map.pm_prog,
		 portmapperlist->pml_map.pm_vers,
		 portmapperlist->pml_map.pm_prot==IPPROTO_UDP ? "udp" : "tcp",
		 portmapperlist->pml_map.pm_port,
		 (re!=NULL) ? re->r_name : "(unknown)" );
	Tcl_AppendElement(interp, str);
	portmapperlist = portmapperlist->pml_next;
    }
    return TCL_OK;
}

/*
 * Ask the mount daemon about mounted filesystems.
 */

static int
do_mount (interp, host)
    Tcl_Interp *interp;
    char *host;
{
    mountlist ml = NULL;
    int res;

    if ((host = gethostnamebyany (interp, host)) == NULL)
        return TCL_ERROR;
    
    res = callrpc (host, MOUNTPROG, MOUNTVERS, MOUNTPROC_DUMP,
		   xdr_void, (char*) 0, xdr_mountlist, (char *) &ml);

    if (res != RPC_SUCCESS) {
	char *p = clnt_sperrno(res);
	if (strncmp(p, "RPC: ", 5) == 0) p += 5;
	Tcl_AppendResult (interp, p, (char *) NULL);
	free (host);
	return TCL_ERROR;
    }

    while (ml) {
	sprintf (str, "%s %s", ml->ml_directory, ml->ml_hostname);
	Tcl_AppendElement (interp, str);
	ml = ml->ml_next;
    }

    free (host);
    return TCL_OK;
}

/*
 * Ask the mount daemon about exported filesystems.
 */

static int
do_exports (interp, host)
    Tcl_Interp *interp;
    char *host;
{
    exports ex = NULL;
    groups gr;
    int res;
    char *grstr;
    char buf[512];
    int size = 0;

    if ((host = gethostnamebyany (interp, host)) == NULL)
        return TCL_ERROR;

    res = callrpc (host, MOUNTPROG, MOUNTVERS, MOUNTPROC_EXPORT,
		   xdr_void, (char*) 0, xdr_exports, (char *) &ex);

    if (res != RPC_SUCCESS) {
	char *p = clnt_sperrno(res);
	if (strncmp(p, "RPC: ", 5) == 0) p += 5;
        Tcl_AppendResult (interp, p, (char *) NULL);
	free (host);
        return TCL_ERROR;
    }
    
    while (ex) {
	size = 1;
	gr = ex->ex_groups;
	while (gr) {
	    size += strlen (gr->gr_name);
	    size++;
	    gr = gr->gr_next;
	}
	gr = ex->ex_groups;
	grstr = (char *) xmalloc(size);
	strcpy (grstr, "");
	while (gr) {
	    sprintf (buf, "%s ", gr->gr_name);
	    strcat (grstr, buf);
	    gr = gr->gr_next;
	}
	sprintf (str, "%s {%s}",
		 ex->ex_dir ? ex->ex_dir : "\"\"",
		 ex->ex_groups ? grstr : "");
	Tcl_AppendElement (interp, str);
	free (grstr);
	ex = ex->ex_next;
    }

    free (host);
    return TCL_OK;
}

/*
 * Test a registered RPC service by calling procedure 0.
 */

static int
do_probe (interp, host, prognum, version, protocol)
    Tcl_Interp *interp;
    char *host;
    unsigned long prognum;
    unsigned long version;
    unsigned protocol;
{
    struct sockaddr_in _socket;
    struct sockaddr_in *rsocket = &_socket;
    CLIENT *cl;
    int sock = RPC_ANYSOCK;    
    struct timeval timeout;
    enum clnt_stat res;
    u_short port;
    struct timeval tvs, tve;
    int time;
    char buf[40];
    char *p;

    timeout.tv_sec = 10; timeout.tv_usec = 0;  /* timeout for this RPC call */

    if (init_socket (interp, rsocket, host) != TCL_OK) return TCL_ERROR;

    if ((protocol != IPPROTO_UDP) && (protocol != IPPROTO_TCP)) {
	interp->result = "unknown protocol";
	return TCL_ERROR;
    }

    port = pmap_getport (rsocket, prognum, version, protocol);
    rsocket->sin_port = port;
    
    if (protocol == IPPROTO_TCP) {
	cl = clnttcp_create (rsocket, prognum, version, &sock, 0, 0); 
    } else {
	struct timeval wait;
	wait.tv_sec = 1; wait.tv_usec = 0;
	cl = clntudp_create (rsocket, prognum, version, wait, &sock);
    }
    if (cl == NULL) {
	p = clnt_spcreateerror ("");
	if (strncmp(p, ": RPC: ", 7) == 0) p += 7;
	Tcl_AppendResult (interp, p, (char *) NULL);
	return TCL_ERROR;
    }

    (void) gettimeofday(&tvs, (struct timezone *) NULL);

    res = clnt_call (cl, NULLPROC, xdr_void, (char *)NULL, 
		     xdr_void, (char *) NULL, timeout);

    (void) gettimeofday(&tve, (struct timezone *) NULL);

    clnt_destroy (cl);

    time = (tve.tv_sec - tvs.tv_sec) * 1000;
    time += (tve.tv_usec - tvs.tv_usec) / 1000;
    sprintf (buf, "%d ", time);

    p = clnt_sperrno(res);
    if (strncmp(p, "RPC: ", 5) == 0) p += 5;
    Tcl_AppendResult (interp, buf, p, (char *) NULL);
    return TCL_OK;
}

/* 
 * Extend a tcl command interpreter with a sunrpc command.
 */

int
sunrpcCmd (clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    if (argc < 3) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0], 
			  " option host ?args?\"", (char *) NULL);
	return TCL_ERROR;
    }
    
    if ((argc == 3) && (strcmp (argv[1], "info") == 0)) {

	return do_info (interp, argv[2]);

    } else if ((argc == 3) && (strcmp (argv[1], "stat") == 0)) {

	return do_rstat (interp, argv[2]);

    } else if ((argc == 3) && (strcmp (argv[1], "mount") == 0)) {
	
	return do_mount (interp, argv[2]);
	
    } else if ((argc == 3) && (strcmp (argv[1], "exports") == 0)) {
	
	return do_exports (interp, argv[2]);
	
    } else if (strcmp (argv[1], "probe") == 0) {

        int program, version;
	unsigned protocol = 0;

	if (argc < 6) { 
	    Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			      " probe host program version protocol\"",
			      (char *) NULL);
	    return TCL_ERROR;
	}
	if (Tcl_GetInt (interp, argv[3], &program) != TCL_OK) 
		return TCL_ERROR;
	if (Tcl_GetInt (interp, argv[3], &version) != TCL_OK) 
		return TCL_ERROR;
	if (!strcmp (argv[5], "tcp")) protocol = IPPROTO_TCP; else 
		if (!strcmp (argv[5], "udp")) protocol = IPPROTO_UDP; else {
		    Tcl_AppendResult (interp, "unknown protocol ",
				      argv[5], (char *) NULL);
		    return TCL_ERROR;
		}

	return do_probe (interp, argv[2], program, version, protocol);

    } else if (!strcmp (argv[1], "ether")) {
	
	if (argc == 4) {
	    if (!strcmp (argv[3], "open"))
		    return open_etherd (interp, argv[2]);
	    if (!strcmp (argv[3], "close"))
		    return close_etherd (interp, argv[2]);
	    Tcl_AppendResult (interp, "bad arg \"", argv[3],
			      "\": should be open or close",
			      (char *) NULL);
	    return TCL_ERROR;
	}
	if (argc == 3) {
	    return do_etherd (interp, argv[2]);
	} else {
	    Tcl_AppendResult (interp, "wrong # args: should be \"",
			      argv[0], " ether host ?open? ?close?\"",
			      (char *) NULL);
	    return TCL_ERROR;
	}
	
    }

    if (argc > 3) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0], 
			  " option host\"", (char *) NULL);
    } else {
	Tcl_AppendResult (interp, "bad option \"", argv[1], 
			  "\": should be info, probe, ",
			  "mount, exports, stat, or ether", 
			  (char *) NULL);
    }

    return TCL_ERROR;
}
