static char copyright []
= "$Id: xessConnCmd.c,v 1.3 1994/11/09 14:38:22 johnsonm Exp $\n\
   Copyright (c) 1992 General Electric.  All rights reserved.\n\
   Copyright (C) 1994 Applied Information Systems.  All rights reserved.";

/*
 *   Permission to use, copy, modify, and distribute this
 *   software and its documentation for any purpose and without
 *   fee is hereby granted, provided that the above copyright
 *   notice appear in all copies and that both that copyright
 *   notice and this permission notice appear in supporting
 *   documentation, and that the name of General Electric not be used in
 *   advertising or publicity pertaining to distribution of the
 *   software without specific, written prior permission.
 *   General Electric makes no representations about the suitability of
 *   this software for any purpose.  It is provided "as is"
 *   without express or implied warranty.
 *
 *   This work was supported by the DARPA Initiative in Concurrent
 *   Engineering (DICE) through DARPA Contract MDA972-88-C-0047.
 *
 * $Log: xessConnCmd.c,v $
 * Revision 1.3  1994/11/09  14:38:22  johnsonm
 * *** empty log message ***
 *
 * Revision 1.2  1994/10/13  16:55:01  johnsonm
 * bzero ==> memset
 *
 * Revision 1.1  1994/08/26  13:40:16  johnsonm
 * Initial revision
 *
 *
 * Old log:
 * Revision 1.4  1994/03/09  22:17:38  kennykb
 * Repaired reference to deallocated memory in Xess_connect_cmd (newWin's
 * display has to be extracted before newWin is deallocated.)
 *
 * Revision 1.3  1994/03/02  22:53:26  kennykb
 * Corrected typo in prototype for makeDummyWindow.
 *
 * Revision 1.2  1994/02/28  20:47:13  kennykb
 * Display management completely revised, in order to make the `-display'
 * option work (it was previously untested).  Instead of using XOpenDisplay
 * directly, displays are opened by calling Tk_CreateWindow to make a
 * dummy top-level window on the display, extracting the display pointer
 * using Tk_Display, and destroying the dummy window again.  This roundabout
 * procedure has the side effect of establishing Tk event management for
 * the display, allowing the callbacks to come through Tk_DoOneEvent
 * correctly into the generic handler.
 *
 * Revision 1.1  1992/10/09  19:03:22  kennykb
 * Initial revision
 *
 *
 * xessConnCmd --
 *
 *	This file contains the code needed to process the `xess connect'
 * Tcl command, and the related code that configures the connection object.
 *
 */

#include "tclXessInt.h"

static Tk_Window makeDummyWindow _ANSI_ARGS_((XessConnection *));


/*
 * xessConnOpts
 *
 *	The following table gives the options for the `xess connect' command
 * in a form suitable for Tk_ConfigureWidget
 */

/* CAN_SET is set if the option may be set when the connection is opened.
 * CAN_CHANGE is set if the option may be changed after the connection is
 * opened.
 */

#define CAN_SET		TK_CONFIG_USER_BIT
#define CAN_CHANGE	(TK_CONFIG_USER_BIT << 1)

static Tk_ConfigSpec xessConnOpts [] = {
  {TK_CONFIG_STRING, "-closecommand", "xessCloseCommand", "XessCloseCommand",
     (char *) NULL, Tk_Offset (XessConnection, closeCmd),
     TK_CONFIG_NULL_OK | CAN_SET | CAN_CHANGE},
/*  {TK_CONFIG_CUSTOM, "-display", "@display", "@display",
 *     (char *) NULL, Tk_Offset (XessConnection, disp),
 *     TK_CONFIG_NULL_OK | CAN_SET, &XessDisplayOption},
 */
  {TK_CONFIG_STRING, "-messagecommand", "xessMessageCommand",
     "XessMessageCommand",
     (char *) NULL, Tk_Offset (XessConnection, messageCmd),
     TK_CONFIG_NULL_OK | CAN_SET | CAN_CHANGE},
  {TK_CONFIG_STRING, "-name", "xessName", "XessName",
     "Tk", Tk_Offset (XessConnection, name),
     CAN_SET},
  {TK_CONFIG_INT, "-port", "@port", "@port", "0",
     Tk_Offset (XessConnection, port), 0},
  {TK_CONFIG_INT, "-timeout", "xessTimeout", "XessTimeout",
     "30", Tk_Offset (XessConnection, timeout),
     CAN_SET},
  {TK_CONFIG_INT, "-window", "xessWindow", "XessWindow",
     "0", Tk_Offset (XessConnection, window),
     CAN_SET},
  {TK_CONFIG_END}
};

/*
 * Table of Xess connections by port
 */

XessConnection * XessConnTable [64];


/*
 * Xess_connect_cmd --
 *
 * Process the `xess connect' Tcl command.
 *
 * The command takes the form:
 *
 * 	xess connect connectionName ?-option value?...
 *
 * connectionName is a user-assigned string that will identify the
 * particular Xess connection to the Tcl program.
 *
 * The options are:
 *
 *	-closecommand TEXT
 *
 *	Records TEXT to be a command to execute when the connection is
 *	closed unexpectedly.  TEXT may contain any of a number of standard
 *	substitution keys; see xessBind.c for descriptions of these.  Default
 *	is no callback; this behavior may be selected explicitly by
 *	presenting an empty string as the TEXT parameter.
 *
 * 	-display NAME
 *
 *	Attempts to connect to a copy of Xess running on the display called
 *	NAME.  The default is the display on which the Tk application's main
 *	window appears.
 *
 *	-messagecommand TEXT
 *
 *	Records TEXT to be a command to execute when a message is received
 *	from another client connected to the same XESS instance.  TEXT may
 *	contain a number of standard substitution keys; see xessBind.c for
 *	descriptions of these.  The default is no callback; this behavior
 *	may be selected explicitly by presenting an empty string as the
 *	TEXT parameter.
 *
 *	-name NAME
 *
 *	Assigns a name of NAME to the Xess connection.  Default is "Tk".
 *
 *	-timeout N
 *
 *	Sets the timeout to N seconds.  If the instance of Xess does not
 *	respond in that time, the connection attempt gives up.  Default is
 *	30 seconds.
 *
 *	-window WINDOWID
 *
 *	Requests a `back door' connection to the Xess instance whose main
 *	window is identified by the hexadecimal string WINDOWID.  The default
 *	is not to use WINDOWID; this behavior may also be selected explicitly
 *	by presenting a null string as a WINDOWID parameter.
 *
 *	When WINDOWID is specified, there is additional change to the behavior:
 *		+ The calling process must be registered as the Xess connection
 *		  manager on the specified display.
 *		+ The specified instance of Xess need not be accepting
 *		  connections for the conneciton to be established.
 *		+ The connection is silent; no message appears on the Xess
 *		  status line when the conneciton is established.
 *	These changes are a side effect of the fact that this mechanism uses
 *	the mgr_connect() call in the Xess API rather than the
 *	xess_establish_connection() call.  Refer to the Xess API Users' Guide
 *	for a further discussion of connection management.
 *
 * Return value:
 *	The connectionName parameter is returned.
 *
 * Further side effects:
 *	The conneectionName parameter is bound as a Tcl command.  See
 *	xessConnObjCmd.c for its operation.
 */

int
Xess_connect_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  char * screen = "";
  Tk_Uid screenUid;
  int src, dst;
  Tk_Window newWin;
  Display * theDisplay;
  XessConnection * conn;
  static Bool callbacksSet = False;
  int status;

  /* Check that sufficient arguments are supplied */

  if (argc < 2) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], " name ?-option value?...\"",
		      (char *) NULL);
    return TCL_ERROR;
  }

  /* Break out the `-display' option separately. */
  
  for (src = 2, dst = 2; src < argc;  src += 2) {
    char c;

    c = argv[src][1];
    if (c == 'd'
	&& (strncmp(argv[src], "-display", strlen(argv[src])) == 0)) {
      screen = argv[src+1];
    } else {
      argv[dst] = argv[src];
      argv[dst+1] = argv[src+1];
      dst += 2;
    }
  }
  argc -= src-dst;

  /* Allocate the data structure for the Xess connection, and initialize it.
   */

  conn = (XessConnection *) ckalloc (sizeof (XessConnection));
  memset ((char *) conn, 0, sizeof (*conn));
  conn -> interp = interp;
  conn -> flags = 0;
  conn -> firstMenu = (XessMenu *) NULL;
  conn -> firstFunction = (XessFunction *) NULL;
  conn -> screenUid = Tk_GetUid (screen);

  /* Configuring the widget determines port, timeout, message command,
     and so on. */

  newWin = makeDummyWindow (conn);
  if (newWin == NULL) {
    (void) ckfree (conn);
    return TCL_ERROR;
  }
  theDisplay = Tk_Display (newWin);
  Tk_DestroyWindow (newWin);
  status = Tk_ConfigureWidget (interp, Tk_MainWindow(interp),
			       xessConnOpts, argc-2, argv+2,
			       (char *) conn, CAN_SET);

  if (status == TCL_OK) {

    /* Make sure that the Xess library has the display open */

    conn -> dispID = xess_new_display (theDisplay);
    if (conn -> dispID == 0) {
      Tcl_AppendResult (interp,
			"xess connection library refused to register display ",
			"\"", DisplayString (theDisplay), "\"",
			(char *) NULL);
    } else if (xess_set_display (conn -> dispID) == 0) {
      Tcl_AppendResult (interp,
			"xess connection library lost registration for ",
			"display \"", DisplayString (theDisplay),
			"\"", (char *) NULL);
    } else {

      /* Make the connection to Xess */

      if (conn -> window == 0) {
	conn -> port = xess_establish_connection (conn -> name,
						  conn -> timeout);
      } else {
	conn -> port = mgr_connect (conn -> dispID, conn -> window,
				    conn -> name);
      }
      if (conn -> port == 0) {
	Tcl_SetResult (interp, "connection refused or timed out.", TCL_STATIC);
      } else {

	/* Find the connection's window ID. */

	conn -> window = xess_window (conn -> port);

	/* Store the connection's address so that it can be looked up
	   by port */

	XessConnTable [conn -> port] = conn;

	/* Set up the connection object command */

	conn -> objectName = malloc (strlen (argv [1]) + 1);
	(void) strcpy (conn -> objectName, argv [1]);
	Tcl_CreateCommand (interp, conn -> objectName,
			  XessConnObj_cmd, (ClientData) conn,
			  XessConnObj_Destroy);

	/* Set up the Xess global callbacks, if they're not set up already */

	if (!callbacksSet) {
	  Tk_CreateGenericHandler (XessEventFilter, (ClientData) interp);
	  xess_set_close_notify (XessCloseCallback);
	  xess_set_receiver (XessReceiveCallback);
	  callbacksSet = True;
	}

	Tcl_SetResult (interp, conn -> objectName, TCL_STATIC);
	return TCL_OK;
      }
    }
  }

  /* If an error occurs, destroy the connection data structure and the
     dummy window. */

  (void) ckfree ((char *) conn);
  Tk_DestroyWindow (newWin);
  return TCL_ERROR;
}

/*
 * xessConnObj_configure_cmd --
 *
 *	Handles the `configure' command on an Xess connection object
 */

int
XessConnObj_configure_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;

  if (argc == 1) {
    status = Tk_ConfigureInfo (interp, Tk_MainWindow (interp), xessConnOpts,
			       (char *) conn, (char *) NULL, 0);
  }
  else if (argc == 2) {
    status =  Tk_ConfigureInfo (interp, Tk_MainWindow (interp), xessConnOpts,
			     (char *) conn, argv [1], 0);
  }
  else {
    status =  Tk_ConfigureWidget (interp, Tk_MainWindow (interp), xessConnOpts,
				  argc-1, argv+1, (char *) conn,
				  TK_CONFIG_ARGV_ONLY | CAN_CHANGE);
  }
  return status;
}

/* makeDummyWindow --
 *
 *   Temporarily create a dummy window for configuring a connection object
 */

static Tk_Window
makeDummyWindow (conn)
     XessConnection * conn;
{
  Tk_Window newWin;
  XSetWindowAttributes atts;
  newWin = Tk_CreateWindow (conn -> interp,
			    Tk_MainWindow(conn -> interp),
			    "_tclxess",
			    conn -> screenUid);
  if (newWin == NULL) {
    return NULL;
  }
  Tk_SetClass (newWin, "TclXess");
  atts.override_redirect = True;
  Tk_ChangeWindowAttributes (newWin, CWOverrideRedirect, &atts);
  Tk_MakeWindowExist (newWin);
  return newWin;
}

