static char copyright []
= "$Id: xessEncode.c,v 1.1 1994/08/26 13:50:13 johnsonm Exp $\n\
   Copyright (c) 1992 General Electric.  All rights reserved.";

/*
 *   Permission to use, copy, modify, and distribute this
 *   software and its documentation for any purpose and without
 *   fee is hereby granted, provided that the above copyright
 *   notice appear in all copies and that both that copyright
 *   notice and this permission notice appear in supporting
 *   documentation, and that the name of General Electric not be used in
 *   advertising or publicity pertaining to distribution of the
 *   software without specific, written prior permission.
 *   General Electric makes no representations about the suitability of
 *   this software for any purpose.  It is provided "as is"
 *   without express or implied warranty.
 *
 *   This work was supported by the DARPA Initiative in Concurrent
 *   Engineering (DICE) through DARPA Contract MDA972-88-C-0047.
 *
 * $Log: xessEncode.c,v $
 * Revision 1.1  1994/08/26  13:50:13  johnsonm
 * Initial revision
 *
 *
 * Old log:
 * Revision 1.2  1994/02/28  20:52:39  kennykb
 * Removed #includes that are now obtained from tclXessInt.h
 *
 * Revision 1.1  1992/10/09  19:05:13  kennykb
 * Initial revision
 *
 *
 * xessEncode.c
 *
 *	This file contains the C source for the `xess' subcommands that
 * 	encode row, column, range, and row specifications.
 */

#include "tclXessInt.h"

/*
 * Xess_encode_column_cmd
 *
 *	Xess subcommand to encode a column specifier
 *
 * Syntax:
 *	xess encode_column column# absFlag
 * -or- xess encode_column {column# absFlag}
 *
 * Where:
 *	column# is the column number to convert.
 *	absFlag is 1 if the column number is an absolute address,
 *		or 0 otherwise.
 *
 * Result:
 *	Returns a standard Tcl result that normally contains the
 *	column specifier in alphabetic form.
 */

/* ARGSUSED */
int
Xess_encode_column_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  int unpacked = 0;		/* Flag == 1 if argc/argv had to be unpacked */
  int status = TCL_OK;
  int column;
  int absflag;
  int flags;

  /* If only one argument has been presented, split the list. */

  if (argc == 2) {
    status = Tcl_SplitList (interp, argv [1], &argc, &argv);
    unpacked = 1;
  } else {
    ++argv;
    --argc;
  }

  /* Check syntax */

  if (status == TCL_OK) {
    if (argc != 2) {
      Tcl_AppendResult (interp,
			"wrong # args, should be \"xess encode_column ",
			"column flags\"", (char *) NULL);
      status = TCL_ERROR;
    }
  }

  /* Get the column number and absolute flag */
  
  if (status == TCL_OK) {
    status = Tcl_GetInt (interp, argv [0], &column);
  }
     
  if (status == TCL_OK) {
    status = Tcl_GetBoolean (interp, argv [1], &absflag);
    if (absflag)
      flags = XESS_CELL_ABSCOL;
    else
      flags = 0;
  }
  
  /* Format the column number */
  
  if (status == TCL_OK) {
    status = xessStoreColumn (interp, column, flags, interp -> result,
			      (char * *) NULL);
  }

  if (unpacked)
    (void) ckfree (argv);
  return status;
}

/*
 * Xess_encode_row_cmd
 *
 *	Xess subcommand to encode a row specifier
 *
 * Syntax:
 *	xess encode_row row# absFlag
 * -or- xess encode_row {row# absFlag}
 *
 * Where:
 *	row# is the row number to convert.
 *	absFlag is 1 if the row number is an absolute address,
 *		or 0 otherwise.
 *
 * Result:
 *	Returns a standard Tcl result that normally contains the
 *	row specifier in alphanumeric form.
 */

/* ARGSUSED */
int
Xess_encode_row_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  int unpacked = 0;		/* Flag == 1 if argc/argv had to be unpacked */
  int status = TCL_OK;
  int row;
  int absflag;
  int flags;

  /* If only one argument has been presented, split the list. */

  if (argc == 2) {
    status = Tcl_SplitList (interp, argv [1], &argc, &argv);
    unpacked = 1;
  } else {
    ++argv;
    --argc;
  }

  /* Check syntax */

  if (status == TCL_OK) {
    if (argc != 2) {
      Tcl_AppendResult (interp, "wrong # args, should be \"xess encode_row",
			" row flags\"", (char *) NULL);
      status = TCL_ERROR;
    }
  }

  /* Get the row number and absolute flag */
  
  if (status == TCL_OK) {
    status = Tcl_GetInt (interp, argv [0], &row);
  }
     
  if (status == TCL_OK) {
    status = Tcl_GetBoolean (interp, argv [1], &absflag);
    if (absflag)
      flags = XESS_CELL_ABSROW;
    else
      flags = 0;
  }
  
  /* Format the row number */
  
  if (status == TCL_OK) {
    status = xessStoreRow (interp, row, flags, interp -> result,
			   (char * *) NULL);
  }

  if (unpacked)
    (void) ckfree (argv);
  return status;
}

/*
 * Xess_encode_cell_cmd
 *
 *	Xess subcommand to encode a cell specifier
 *
 * Syntax:
 *	xess encode_cell row# column# absFlags
 * -or- xess encode_cell {row# column# absFlags}
 *
 * Where:
 *	row# is the row number to convert
 *	column# is the column number to convert
 *	absFlags is a string comprising two binary digits, brow, and bcolumn,
 *	 	which are 0 if the row (column) number is relative and
 *			  1 if it is absolute
 *
 * Result:
 *	Returns a standard Tcl result that normally contains the
 *	cell specifier in alphanumeric form.
 */

/* ARGSUSED */
int
Xess_encode_cell_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  int unpacked = 0;		/* Flag == 1 if argc/argv had to be unpacked */
  int status = TCL_OK;
  int row, col;
  int flags;

  /* If only one argument has been presented, split the list. */

  if (argc == 2) {
    status = Tcl_SplitList (interp, argv [1], &argc, &argv);
    unpacked = 1;
  } else {
    ++argv;
    --argc;
  }

  /* Check syntax */

  if (status == TCL_OK) {
    if (argc != 3) {
      Tcl_AppendResult (interp, "wrong # args, should be \"xess encode_cell ",
			"row col flags\"", (char *) NULL);
      status = TCL_ERROR;
    }
  }

  /* Get the cell coordinates and absolute flags */
  
  if (status == TCL_OK) {
    status = Tcl_GetInt (interp, argv [0], &row);
  }
  if (status == TCL_OK) {
    status = Tcl_GetInt (interp, argv [1], &col);
  }
  if (status == TCL_OK) {
    switch (argv [2] [0])
      {
      case '1':
	flags = XESS_CELL_ABSROW;
	break;
      case '0':
	flags = 0;
	break;
      default:
	Tcl_SetResult (interp, "bad absolute-address flag in row specifier",
		       TCL_STATIC);
	status = TCL_ERROR;
	break;
      }
  }
  if (status == TCL_OK) {
    switch (argv [2] [1])
      {
      case '1':
	flags |= XESS_CELL_ABSCOL;
	break;
      case '0':
	break;
      default:
	Tcl_SetResult (interp, "bad absolute-address flag in column specifier",
		       TCL_STATIC);
	status = TCL_ERROR;
	break;
      }
  }
  if (status == TCL_OK) {
    if (argv [2] [2] != '\0') {
      Tcl_SetResult (interp, "bad absolute-address flags", TCL_STATIC);
      status = TCL_ERROR;
    }
  }
  
  /* Format the cell number */
  
  if (status == TCL_OK) {
    status = xessStoreCell (interp, row, col, flags, interp -> result,
			      (char * *) NULL);
  }

  if (unpacked)
    (void) ckfree (argv);
  return status;
}

/*
 * Xess_encode_range_cmd
 *
 *	Xess subcommand to encode a range specifier
 *
 * Syntax:
 *	xess encode_range cell1 cell2
 * -or- xess encode_range {cell1 cell2}
 *
 * Where:
 *	cell1 and cell2 are cell numbers to convert, much as with
 *		xessEncodeCellCommand
 *
 * Result:
 *	Returns a standard Tcl result that normally contains the
 *	range specifier in alphanumeric form.
 */

/* ARGSUSED */
int
Xess_encode_range_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  char * cell1 = NULL;
  char * cell2 = NULL;
  int status = TCL_OK;
  int unpacked = 0;
  int free1 = 0;

  /* If only one argument has been presented, split the list. */

  if (argc == 2) {
    status = Tcl_SplitList (interp, argv [1], &argc, &argv);
    unpacked = 1;
  } else {
    ++argv;
    --argc;
  }

  /* Check syntax */

  if (status == TCL_OK) {
    if (argc != 2) {
      Tcl_AppendResult (interp, "wrong # args, should be \"xess encode_range",
			" cell1 cell2\"", (char *) NULL);
      status = TCL_ERROR;
    }
  }

  /* Encode first cell */

  if (status == TCL_OK) {
    status = Tcl_VarEval (interp, "xess encode_cell ", argv [0],
			  (char *) NULL);
  }

  /* Encode second cell */

  if (status == TCL_OK) {
    cell1 = (char *) ckalloc (strlen (interp -> result) + 1);
    strcpy (cell1, interp -> result);
    free1 = 1;
    Tcl_ResetResult (interp);
    status = Tcl_VarEval (interp, "xess encode_cell ", argv [1],
			  (char *) NULL);
  }

  /* Catenate the two cell addresses */

  if (status == TCL_OK) {
    cell2 = (char *) ckalloc (strlen (interp -> result) + 1);
    strcpy (cell2, interp -> result);
    Tcl_ResetResult (interp);
    sprintf (interp -> result, "%s..%s", cell1, cell2);
    (void) ckfree (cell2);
  }

  /* Free storage and return */

  if (unpacked)
    (void) ckfree (argv);
  if (free1)
    (void) ckfree (cell1);
  return status;
}
