static char copyright []
= "$Id: xessEval.c,v 1.1 1994/08/26 13:51:23 johnsonm Exp $\n\
   Copyright (c) 1992 General Electric.  All rights reserved.";

/*
 *   Permission to use, copy, modify, and distribute this
 *   software and its documentation for any purpose and without
 *   fee is hereby granted, provided that the above copyright
 *   notice appear in all copies and that both that copyright
 *   notice and this permission notice appear in supporting
 *   documentation, and that the name of General Electric not be used in
 *   advertising or publicity pertaining to distribution of the
 *   software without specific, written prior permission.
 *   General Electric makes no representations about the suitability of
 *   this software for any purpose.  It is provided "as is"
 *   without express or implied warranty.
 *
 *   This work was supported by the DARPA Initiative in Concurrent
 *   Engineering (DICE) through DARPA Contract MDA972-88-C-0047.
 *
 * $Log: xessEval.c,v $
 * Revision 1.1  1994/08/26  13:51:23  johnsonm
 * Initial revision
 *
 *
 * Old log:
 * Revision 1.2  1994/02/28  20:52:54  kennykb
 * Removed #includes that are now obtained from tclXessInt.h
 * Changed Tk_ParseArgv call to use Tk_MainWindow rather than keeping the
 * main wondow pointer in the XessConnection structure.
 *
 * Revision 1.1  1992/10/09  19:05:30  kennykb
 * Initial revision
 *
 *
 * xessEval.c --
 *
 *	This file contains the Tcl function bindings for the Xess
 *	evaluation and recalculation functions.
 */

#include "tclXessInt.h"

/* Command parsing options */

static int forceFlag;

static Tk_ArgvInfo xessRecalcOpts [] = {
  {"-force", TK_ARGV_CONSTANT, (char *) 1, (char *) &forceFlag,
     "Force recalculation even if recalc mode is manual"},
  {(char *) NULL, TK_ARGV_END, (char *) NULL, (char *) NULL, (char *) NULL}
};

/*
 * XessConnObj_recalculate_cmd--
 *
 *	Recalculate one or more ranges on a spreadsheet.
 *
 * Syntax:
 *	connectionName recalculate ?-option value?... ?range?...
 *
 * Where:
 *	connectionName is the name of an Xess connection object
 *	range is a range of cells in the spreadsheet, e.g., AA6..BX14
 *
 * Options:
 *	-force
 *		If `-force' is specified, then the recalculation is
 *		forced even if the current recalculation mode is manual.
 *		Default is to follow the user's recalculation mode.
 *
 * Notes:
 *	If no ranges are specified, the entire sheet is recalculated.
 */

int
XessConnObj_recalculate_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;

  int status;
  Range * ranges;
  int r;

  /* Parse the arguments to determine the `force' flag */

  forceFlag = 0;
  status = Tk_ParseArgv (interp, Tk_MainWindow (interp), &argc, argv,
			 xessRecalcOpts, 0);
  if (status != TCL_OK)
    return status;

  /* Parse the ranges if there are any */

  if (argc <= 1) {
    ranges = (Range *) NULL;
  }
  else {
    ranges = (Range *) ckalloc ((argc-1) * sizeof * ranges);
    for (r = 1; r < argc; ++r) {
      status = xessGetRange (interp, argv [r], ranges + r - 1);
      if (status != TCL_OK)
	break;
    }
  }

  /* Make sure that we're not running recursively */

  if (status == TCL_OK) {
    if (conn -> flags & XESS_FUNCTION_PENDING) {
      Tcl_AppendResult (interp, "recursive use of \"", argv [-1], " ",
			argv [0], "\" is forbidden.", (char *) NULL);
      status =  TCL_ERROR;
    }
  }

  /* Do the recalculation */

  if (status == TCL_OK)
    xess_recalc (conn -> port, forceFlag, argc-1, ranges);

  if (ranges)
    (void) ckfree (ranges);

  return status;
}

/*
 * XessConnObj_evaluate_cell_cmd
 *
 *	Evaluate a single cell in an Xess spreadsheet
 *
 * Syntax:
 *	connectionName evaluate_cell addr
 *
 * Where:
 *	connectionName is the name of an Xess connection object.
 *	addr is the address of a cell in the spreadsheet.
 *
 * Results:
 *	Returns a standard Tcl result, whose value is the value contained
 *	in the specified cell after evaluation.
 */

int
XessConnObj_evaluate_cell_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;

  XessCell cell;
  int status;
  double value;
  char * string;

  /* Check syntax */

  if (argc != 2) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], " address\"", (char *) NULL);
    return TCL_ERROR;
  }

  /* Parse cell address */

  status = xessGetCell (interp, argv [1], &cell);
  if (status != TCL_OK) 
    return status;

  /* Evaluate the cell */

  status = xess_evaluate_cell (conn -> port, cell.row, cell.col,
			       &value, &string);
  switch (status)
    {
    case XESS_NUMBER:
      sprintf (interp -> result, "%#g", value);
      return TCL_OK;
    case XESS_STRING:
      Tcl_SetResult (interp, string, TCL_DYNAMIC);
      return TCL_OK;
    default:
      Tcl_AppendResult (interp, "cell ", argv[1], " is empty", (char *) NULL);
      return TCL_ERROR;
    }
}

/*
 * XessConnObj_evaluate_expression_cmd
 *
 *	Evaluate a single expression in an Xess spreadsheet
 *
 * Syntax:
 *	connectionName evaluate_expression expr
 *
 * Where:
 *	connectionName is the name of an Xess connection object.
 *	expr is the expresion to evaluate
 *
 * Results:
 *	Returns a standard Tcl result, whose value is the value of the
 *	specified expression.
 */

int
XessConnObj_evaluate_expression_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;

  int status;
  double value;
  char * string;

  /* Check syntax */

  if (argc != 2) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], " expression\"", (char *) NULL);
    return TCL_ERROR;
  }

  /* Evaluate the expression */

  status = xess_evaluate_expr (conn -> port, argv [1], &value, &string);
  switch (status)
    {
    case XESS_NUMBER:
      sprintf (interp -> result, "%#g", value);
      return TCL_OK;
    case XESS_STRING:
      Tcl_SetResult (interp, string, TCL_DYNAMIC);
      return TCL_OK;
    default:
      Tcl_SetResult (interp, "error in evaluation", TCL_STATIC);
      return TCL_ERROR;
    }
}
