static char copyright []
= "$Id: xessFile.c,v 1.5 1994/11/09 14:39:15 johnsonm Exp $\n\
   Copyright (c) 1992 General Electric.  All rights reserved.\n\
   Copyright (C) 1994 Applied Information Systems.  All rights reserved.";

/*
 *   Permission to use, copy, modify, and distribute this
 *   software and its documentation for any purpose and without
 *   fee is hereby granted, provided that the above copyright
 *   notice appear in all copies and that both that copyright
 *   notice and this permission notice appear in supporting
 *   documentation, and that the name of General Electric not be used in
 *   advertising or publicity pertaining to distribution of the
 *   software without specific, written prior permission.
 *   General Electric makes no representations about the suitability of
 *   this software for any purpose.  It is provided "as is"
 *   without express or implied warranty.
 *
 *   This work was supported by the DARPA Initiative in Concurrent
 *   Engineering (DICE) through DARPA Contract MDA972-88-C-0047.
 *
 * $Log: xessFile.c,v $
 * Revision 1.5  1994/11/09  14:39:15  johnsonm
 * *** empty log message ***
 *
 * Revision 1.4  1994/10/13  16:51:44  johnsonm
 * ifder ==> ifdef
 *
 * Revision 1.3  1994/08/26  15:20:52  johnsonm
 * Temporary kludge to get around bug in xess connections library.
 * Remove the #if 0 when the library returns the correct return
 * codes for xess_print().
 *
 * Revision 1.2  1994/08/26  15:15:00  johnsonm
 * ifdef'ed out fprintf's.
 *
 * Revision 1.1  1994/08/26  13:56:19  johnsonm
 * Initial revision
 *
 *
 * Old log:
 * Revision 1.3  1994/03/02  22:55:03  kennykb
 * Fixed typos in strrchr() calls.
 *
 * Revision 1.2  1994/02/28  20:55:37  kennykb
 * Removed #includes that are now obtained from tclXessInt.h
 * Changed calls to Tk_ParseArgv to use standard method of obtaining
 * applicaiton main window.
 *
 * Revision 1.1  1992/10/09  19:06:41  kennykb
 * Initial revision
 *
 *
 * xessFile.c --
 *
 *	This file contains the definitions of the Tcl commands that
 *	perform file and printing operations on Xess connection objects
 */

#include "tclXessInt.h"


/*
 * XessConnObj_read_file_cmd --
 *
 *	This function processes the `read_file' command on an Xess connection
 *	object.
 *
 * Syntax:
 *	connection read_file filename
 *
 * Where:
 *	connection is the name of an Xess connection object.
 *	filename is the name of a file to read into the spreadsheet.  Previous
 *		contents of the sheet are lost.
 *
 * Result:
 *	Returns a standard Tcl result, normally empty.
 */

int
XessConnObj_read_file_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;

  /* Check syntax */

  if (argc != 2) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], " filename\"", (char *) NULL);
    return TCL_ERROR;
  }

  /* Read the file */

  status = xess_read_file (conn -> port, argv [1]);

  /* Return status */

  switch (status)
    {
    case 0:
      Tcl_AppendResult (interp, "couldn't read \"", argv [1], "\"",
			(char *) NULL);
      return TCL_ERROR;
    case 1:
      return TCL_OK;
    default:
      sprintf (interp -> result,
	       "xess_read_file returned unexpected status %d",
	       status);
      return TCL_ERROR;
    }
}

/*
 * XessConnObj_write_file_cmd --
 *
 *	This function processes the `write_file' command on an Xess connection
 *	object.
 *
 * Syntax:
 *	connection write_file filename
 *
 * Where:
 *	connection is the name of an Xess connection object.
 *	filename is the name of a file in which to write the spreadesheet's
 *	contents.  The sheet itself is left unchanged.
 *
 * Result:
 *	Returns a standard Tcl result, normally empty.
 */

int
XessConnObj_write_file_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;

  /* Check syntax */

  if (argc != 2) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], " filename\"", (char *) NULL);
    return TCL_ERROR;
  }

  /* Write the file */

  status = xess_write_file (conn -> port, argv [1]);

  /* Return status */

  switch (status)
    {
    case 0:
      Tcl_AppendResult (interp, "couldn't write \"", argv [1], "\"",
			(char *) NULL);
      return TCL_ERROR;
    case 1:
      return TCL_OK;
    default:
      sprintf (interp -> result,
	       "xess_write_file returned unexpected status %d",
	       status);
      return TCL_ERROR;
    }
}

/*
 * XessConnObj_print_cmd --
 *
 *	This function handles the `print' Tcl command on an Xess connection
 *	object.
 *
 * Syntax:
 *	connection print range ?-into filename? ?-type filetype?
 *
 * Where:
 *	connection is the name of an Xess connection object.
 *	range is the range of cells to be printed.
 *
 * Options:
 *	-into filename
 *		Specifies the name of a file to receive the print data.
 *		Default is to use the printer specified by the `Printer
 *		Characteristics' dialogue.
 *	-type filetype
 *		Specifies the type of printing to be done.  Default is
 *		`ascii'; the other option is `ps', which specifies PostScript
 *		output.
 *
 * Result:
 *	Returns a standard Tcl result, normally empty.
 */

/* Define command options */

static char * printFileName;	/* Name of the print file or NULL */
static int printFileType;	/* Type of the print file */
static Tk_ArgvInfo xessPrintOpts [] = {
  {"-into", TK_ARGV_STRING, (char *) NULL, (char *) & printFileName,
     "Filename to receive print data. If not specifed, send to printer."},
  {"-type", TK_ARGV_GENFUNC, (char *) xessGetPrinttypeArgv,
     (char *) &printFileType, "Type of print data (ascii/ps)."},
  {(char *) NULL, TK_ARGV_END, (char *) NULL, (char *) NULL, (char *) NULL}
};

int
XessConnObj_print_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  Range range;
  int status;
  int result;

  /* Parse the command line */

  printFileName = (char *) NULL;
  printFileType = PRINT_ASCII;
  status = Tk_ParseArgv (interp, Tk_MainWindow (interp),
			 &argc, argv, xessPrintOpts, 0);
  if (status == TCL_OK && argc != 2) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], " range ?-option value?...\"", (char *) NULL);
    status = TCL_ERROR;
  }
  if (status == TCL_OK) {
    status = xessGetRange (interp, argv [1], &range);
  }

  /* Make the Xess call to print the data */

  if (status == TCL_OK) {
    result = xess_print (conn -> port, printFileType, range, printFileName);
#if 0 /* bug in Xess connections library; put this back in when fixed */
    switch (result)
      {
      case 0:
	Tcl_SetResult (interp, "couldn't print", TCL_STATIC);
	status = TCL_ERROR;
	break;
      case 1:
	break;
      default:
	sprintf (interp -> result,
		 "xess_print returned unexpected status %d",
		 result);
	status = TCL_ERROR;
	break;
      }
#endif
  }

  return status;
}

/*
 * XessConnObj_import_cmd --
 *
 *	This function handles the `import' Tcl command on an Xess connection
 *	object.
 *
 * Syntax:
 *	connection import ?-at cell? ?-from filename? ?-type filetype?
 *
 * Where:
 *	connection is the name of an Xess connection object.
 *
 * Options:
 *	-from filename
 *		Specifies the name of a file from which to import data.
 *		This option must be supplied.
 *	-type filetype
 *		Specifies the type of file.  Default is to use the file
 *		extension as the type of the file, if it is a valid file
 *		type.  Otherwise, the file is assumed to be text.
 *	-at cell
 *		Specifies the cell where the data are to begin.  The size of
 *		the file determines the range of data to be imported.  This
 *		option must be supplied.
 *
 * Result:
 *	Returns a standard Tcl result, normally empty.
 */

/* Define command options */

static char * importFileName;	/* Name of the import file or NULL */
static int importFileType;	/* Type of the import file */
static XessCell importFileCell; /* Cell to receive imported data */
static Tk_ArgvInfo xessImportOpts [] = {
  {"-from", TK_ARGV_STRING, (char *) NULL, (char *) & importFileName,
     "Filename from which to import data."},
  {"-type", TK_ARGV_GENFUNC, (char *) xessGetFiletypeArgv,
     (char *) &importFileType, "Type of file."},
  {"-at", TK_ARGV_GENFUNC, (char *) xessGetCellArgv,
     (char *) &importFileCell, "Cell where imported data will begin"},
  {(char *) NULL, TK_ARGV_END, (char *) NULL, (char *) NULL, (char *) NULL}
};

int
XessConnObj_import_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  Range range;
  int status;
  int result;

  /* Parse the command line */

  importFileName = (char *) NULL;
  importFileType = -1;
  importFileCell.row = 0;
  status = Tk_ParseArgv (interp, Tk_MainWindow (interp), &argc, argv,
			 xessImportOpts, 0);
  if (status == TCL_OK && argc != 1) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], " ?-option value?...\"", (char *) NULL);
    status = TCL_ERROR;
  }

  /* Cell to place data */

  if (status == TCL_OK && importFileCell.row == 0) {
    Tcl_SetResult (interp, "-at option must be supplied", TCL_STATIC);
    status = TCL_ERROR;
  }

  /* File to read */

  if (status == TCL_OK && importFileName == (char *) NULL) {
    Tcl_SetResult (interp, "-file option must be supplied", TCL_STATIC);
    status = TCL_ERROR;
  }

  /* Type of file */

  if (status == TCL_OK && importFileType == -1) {
    char * ext = strrchr (importFileName, '.');
    if (ext != NULL) {
      status = xessGetFiletype (interp, ext+1, &importFileType);
      if (status != TCL_OK) {
	status = TCL_OK;
	Tcl_ResetResult (interp);
      }
    }
    if (importFileType == -1) {
      importFileType = TEXT_FILE;
    }
  }

  /* Make the Xess call to import the data */

  if (status == TCL_OK) {
#ifdef DEBUG_PRINTF
    fprintf (stderr, "xess_import (%d, %d, %s, %d, %d);\n",
	     conn -> port, importFileType, importFileName,
	     importFileCell.row, importFileCell.col);
#endif
    result = xess_import (conn -> port, importFileType, importFileName,
			  importFileCell.row, importFileCell.col);
#ifdef DEBUG_PRINTF
    fprintf (stderr, "    result = %d\n", result);
#endif
    switch (result)
      {
      case 0:
	Tcl_SetResult (interp, "couldn't import", TCL_STATIC);
	status = TCL_ERROR;
	break;
      case 1:
	break;
      default:
	sprintf (interp -> result,
		 "xess_import returned unexpected status %d",
		 result);
	status = TCL_ERROR;
	break;
      }
  }

  return status;
}

/*
 * XessConnObj_export_cmd --
 *
 *	This function handles the `export' Tcl command on an Xess connection
 *	object.
 *
 * Syntax:
 *	connection export ?-from range? ?-to filename? ?-type filetype?
 *
 * Where:
 *	connection is the name of an Xess connection object.
 *
 * Options:
 *	-from range
 *		Specifies the range of cells to be exported. This option
 *		must be specified.
 *	-to filename
 *		Specifies the name of a file from which to export data.
 *		This option must be supplied.
 *	-type filetype
 *		Specifies the type of file.  Default is to use the file
 *		extension as the type of the file, if it is a valid file
 *		type.  Otherwise, the file is assumed to be text.
 *
 * Result:
 *	Returns a standard Tcl result, normally empty.
 */

/* Define command options */

static char * exportFileName;	/* Name of the export file or NULL */
static int exportFileType;	/* Type of the export file */
static Range exportFileRange;	/* Range of  exported data */
static Tk_ArgvInfo xessExportOpts [] = {
  {"-to", TK_ARGV_STRING, (char *) NULL, (char *) & exportFileName,
     "Filename to which to export data."},
  {"-type", TK_ARGV_GENFUNC, (char *) xessGetFiletypeArgv,
     (char *) &exportFileType, "Type of file."},
  {"-from", TK_ARGV_GENFUNC, (char *) xessGetRangeArgv,
     (char *) &exportFileRange, "Range to export"},
  {(char *) NULL, TK_ARGV_END, (char *) NULL, (char *) NULL, (char *) NULL}
};

int
XessConnObj_export_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;
  int result;

  /* Parse the command line */

  exportFileName = (char *) NULL;
  exportFileType = -1;
  exportFileRange.r0 = 0;
  status = Tk_ParseArgv (interp, Tk_MainWindow (interp), &argc, argv,
			 xessExportOpts, 0);
  if (status == TCL_OK && argc != 1) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], " ?-option value?...\"", (char *) NULL);
    status = TCL_ERROR;
  }

  /* Range data */

  if (status == TCL_OK && exportFileRange.r0 == 0) {
    Tcl_SetResult (interp, "-from option must be supplied", TCL_STATIC);
    status = TCL_ERROR;
  }

  /* File to read */

  if (status == TCL_OK && exportFileName == (char *) NULL) {
    Tcl_SetResult (interp, "-to option must be supplied", TCL_STATIC);
    status = TCL_ERROR;
  }

  /* Type of file */

  if (status == TCL_OK && exportFileType == -1) {
    char * ext = strrchr (exportFileName, '.');
    if (ext != NULL) {
      status = xessGetFiletype (interp, ext+1, &exportFileType);
      if (status != TCL_OK) {
	status = TCL_OK;
	Tcl_ResetResult (interp);
      }
    }
    if (exportFileType == -1) {
      exportFileType = TEXT_FILE;
    }
  }

  /* Make the Xess call to export the data */

  if (status == TCL_OK) {
#ifdef DEBUG_PRINTF
    fprintf (stderr, "xess_export (%d, %d, %s, (%d,%d)..(%d,%d))\n",
	     conn -> port, exportFileType, exportFileName,
	     exportFileRange.r0, exportFileRange.c0,
	     exportFileRange.r1, exportFileRange.c1);
#endif
    result = xess_export (conn -> port, exportFileType, exportFileName,
			  exportFileRange);
#ifdef DEBUG_PRINTF
    fprintf (stderr, "   result = %d\n", result);
#endif
    switch (result)
      {
      case 0:
	Tcl_SetResult (interp, "couldn't export", TCL_STATIC);
	status = TCL_ERROR;
	break;
      case 1:
	break;
      default:
	sprintf (interp -> result,
		 "xess_export returned unexpected status %d",
		 result);
	status = TCL_ERROR;
	break;
      }
  }

  return status;
}
