/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */
/*
 * Copyright 1992 by the National Optical Astronomy Observatories(*)
 *
 * Permission to use, copy, and distribute
 * is hereby granted without fee, providing that the above copyright
 * notice appear in all copies and that both the copyright notice and this
 * permission notice appear in supporting documentation.
 *
 * This software is provided "as is" without any express or implied warranty.
 *
 * (*) Operated by the Association of Universities for Research in
 *     Astronomy, Inc. (AURA) under cooperative agreement with the
 *     National Science Foundation.
 */
/* Program: Sender.c
 *      This file contains the low level creation and deletion of Senders.
 *      These functions are based on George Drapeau's MAEstro Sender.c.
 *
 * Created: K. Gillies 26 June 1992
 * SCCS INFO
 *      @(#)Sender.c	1.1 9/11/92
 *
 */
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <string.h>
#include <stdio.h>
#include "tcl.h"
#include "tclIPC.h"            /* RPC stuff */
#include "tclipcP.h"           /* Private structures */
#include "tclipc.h"            /* Public interface */

/*
 *--------------------------------------------------------------
 *
 * NewSender
 *
 *      Create a new Sender structure and return it to caller.
 *      A TCP connection will be established to the senderPort.
 *
 * Results:
 *      NULL if there is a problem or a Sender structure.
 *--------------------------------------------------------------
 */
Sender NewSender(interp, senderPort)
Tcl_Interp *interp;
Port* senderPort;
{
  char*			tempHostName;
  struct hostent*	hostEntry;
  struct sockaddr_in	remoteAddress;
  struct timeval	timeoutParams;
  Sender		newSender;

  /* Initialize local variables */  
  tempHostName = senderPort->hostName;
  newSender = (Sender)ckalloc(sizeof(SenderR));
  newSender->clientSocket = RPC_ANYSOCK;
  newSender->portNumber = senderPort->portNumber;
  newSender->clientPtr = (CLIENT*)NULL;
  newSender->interp = interp;
  newSender->localReceiver = (Receiver)NULL; 

  /* If the portNumber passed in is 0, then this...*/
  /* Sender was not meant to be used, so just return.  This happens when
   * the PortManager App starts up and needs a Sender to make a Receiver.
   */
  if (newSender->portNumber == 0) {
    return newSender;
  }

  newSender->localReceiver = ipcCheckForLocal(senderPort);
  if (newSender->localReceiver != NULL) {
    /* It's a local sender so just return with no actual connection. */
    return newSender;
  }

  /* Get host address from the hostname */
  hostEntry = gethostbyname(tempHostName);
  if (hostEntry == (struct hostent*) NULL) {
    /* gethostbyname failed, print error and exit this function */
    Tcl_AppendResult(interp, 
	    "IPC Protocol: Could not get host information for host named.",
		     tempHostName, ".\n",
		     (char *)NULL);
    /* Free space taken by newSender, which is no longer valid */
    ckfree((void *)newSender);
    return ((Sender)NULL);
  }

  /* Set up remoteAddress's port number, address family,...*/
  /* ...and Internet address.*/
  remoteAddress.sin_port = htons(newSender->portNumber);
  remoteAddress.sin_family = hostEntry->h_addrtype;
  bcopy(hostEntry->h_addr,
	(char*)&remoteAddress.sin_addr,
	hostEntry->h_length);

  /* Try to create connection to remote application's receiving port */
  /* With my program and version.  0,0 mean default buffer sizes */
  newSender->clientPtr = clnttcp_create(&remoteAddress,
					TCLIPC,
					FIRSTTESTVERSION,
					&newSender->clientSocket,
					0,0);
  /* Was the client creation successful? */
  if (newSender->clientPtr == (CLIENT*)NULL) {
    if (interp != NULL) {
      Tcl_AppendResult(interp, 
        "IPC Protocol: Could not create a new Sender.\n", 
        (char *)NULL);
    }
    /* Release space allocated for the invalid Sender */
    ckfree((void *)newSender);
    return((Sender)NULL);
  }
  /* Tell the client to stop retrying after 25 seconds, ...*/
  timeoutParams.tv_usec = (long)0;
  timeoutParams.tv_sec = (long)25;
  clnt_control(newSender->clientPtr, CLSET_TIMEOUT, (char*)&timeoutParams);
  return newSender;
}

/*
 *--------------------------------------------------------------
 *
 * SenderAppCheck
 *
 *      Used by programs that wish to check whether or not another
 *      program is still running (for instance, the PortManager).
 *      Almost the same as NewSender but uses no Tcl_Interp.
 *
 * Results:
 *      Returns -1 if no app exists or 0 if ok.
 *--------------------------------------------------------------
 */
int SenderAppCheck(senderPort)
Port* senderPort;
{
  char*			tempHostName;
  struct hostent*	hostEntry;
  struct sockaddr_in	remoteAddress;
  Sender		newSender;

  /* If it's a local connection, return */
  if(ipcCheckForLocal(senderPort) != NULL) {
    return TCL_OK;
  }

  /* Initialize local variables */  
  tempHostName = senderPort->hostName;
  newSender = (Sender)ckalloc(sizeof(SenderR));
  newSender->clientSocket = RPC_ANYSOCK;
  newSender->portNumber = senderPort->portNumber;
  newSender->clientPtr = (CLIENT*)NULL;
  newSender->interp = NULL;
  newSender->localReceiver = (Receiver)NULL; 

  /* Get host address from the hostname */
  /* gethostbyname failed, print error and exit this function */
  hostEntry = gethostbyname(tempHostName);
  if (hostEntry == (struct hostent*) NULL) {
    /* Free space taken by newSender, which is no longer valid */
    ckfree((void *)newSender);
    return TCL_ERROR;
  }

  /* Set up remoteAddress's port number, address family,...*/
  /* ...and Internet address.*/
  remoteAddress.sin_port = htons(newSender->portNumber);
  remoteAddress.sin_family = hostEntry->h_addrtype;
  bcopy(hostEntry->h_addr, (char*)&remoteAddress.sin_addr,
	hostEntry->h_length);

  /* Try to create connection to remote application's receiving port */
  newSender->clientPtr = clnttcp_create(&remoteAddress,
					TCLIPC,
					FIRSTTESTVERSION,
					&newSender->clientSocket,
					0,0);
  /* Was the client creation successful? */
  if (newSender->clientPtr == (CLIENT*)NULL) {
    /* Release space allocated for the invalid Sender */
    ckfree((void *)newSender);
    return TCL_ERROR;
  }

  /* Get rid of the test sender, clientPtr is good */
  DestroySender(newSender);

  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DestroySender
 *
 *      Close a sender and deallocate its resources.
 *
 * Results:
 *      void
 *--------------------------------------------------------------
 */
void DestroySender(sender)
Sender sender;
{
  /* Was the Sender passed in valid? */
  if (sender != (Sender)NULL) {
    /* First check to see if it's a local sender */
    if (sender->localReceiver != NULL) {
      /* Only release the memory */
      ckfree((void *)sender); 
    } else { 
      /* Destroy means of communication with remote application */
      clnt_destroy(sender->clientPtr);
      /* Close low-level socket on which the clientPtr was based*/
      close(sender->clientSocket);
      /* Free space taken by the Sender */
      ckfree((void *)sender);
    }
  }
}

/*
 *--------------------------------------------------------------
 *
 * SenderPortNumber
 *
 *      Return a valid sender's port.
 *
 * Results:
 *      an integer port number
 *--------------------------------------------------------------
 */
int SenderPortNumber(sender)
Sender sender;
{
  return(sender->portNumber);
}

/*
 *--------------------------------------------------------------
 *
 * ipcGetSenderSocket
 *
 *      Return a valid Sender's socket.
 *
 * Results:
 *      an integer socket
 *--------------------------------------------------------------
 */
int ipcGetSenderSocket(sender)
Sender sender;
{
  return(sender->clientSocket);
}
