/*-
 * Copyright (c) 1993, 1994 Michael B. Durian.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Michael B. Durian.
 * 4. The name of the the Author may be used to endorse or promote 
 *    products derived from this software without specific prior written 
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#ifdef USE_MPU401THREAD
#ifndef USE_PTHREADS
You must define USE_PTHREADS if you are also defining USE_MPU401THREAD
#endif
#include <assert.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <strstream.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <pthread.h>

#ifdef SVR4
#include <sys/filio.h>
#include <sys/midi.h>
#endif

#ifdef linux
#include <linux/termios.h>
#include <linux/midi.h>
#endif

#ifdef BSD
#include <sys/midiioctl.h>
#endif

#include "401Thrd.h"
#include "EvntUtil.h"
#include "Note.h"


void *
PlayThread(void *arg)
{
	MPU401Thread *mpu = (MPU401Thread *)arg;

cerr << "Calling PlayThread\n";
	return (mpu->PlayLoop());
}

void *
RecordThread(void *arg)
{
	MPU401Thread *mpu = (MPU401Thread *)arg;

cerr << "Calling RecordThread\n";
	mpu->RecordLoop();
	return (0);
}

MPU401Thread::MPU401Thread() : fd(-1), last_play_time(0), last_rec_time(0), finished(0),
    last_record_rs(0)
{

	assert(pthread_mutex_init(&finished_mutex, 0) == 0);
	assert(pthread_cond_init(&finished_cond, 0) == 0);
}

MPU401Thread::MPU401Thread(const char *dev) : MidiDevice(dev), fd(-1), last_play_time(0),
    last_rec_time(0), finished(0), last_record_rs(0)
{

	assert(pthread_cond_init(&finished_cond, 0) == 0);
	assert(pthread_mutex_init(&finished_mutex, 0) == 0);
}

MPU401Thread::~MPU401Thread()
{
	
	if (fd != -1)
		Stop();
	if (curr_event != 0)
		delete curr_event;
	assert(pthread_cond_destroy(&finished_cond) == 0);
	assert(pthread_mutex_destroy(&finished_mutex) == 0);
}

int
MPU401Thread::Play(Song *s, int r)
{
	ostrstream err;
	const char *name;
	char *str;
	int arg, i;

	// open the device
	if (fd != -1) {
		SetError("Device already open");
		return (0);
	}
	if ((name = GetName()) == 0) {
		SetError("No device set");
		return (0);
	}

	// set repeat
	SetRepeat(r);

	play_song = s;

	// initialize track/event info
	last_play_time = 0;
	finished = 0;
	curr_event = new Event *[play_song->GetNumTracks()];
	if (curr_event == 0) {
		SetError("No more memory");
		return (0);
	}
	for (i = 0; i < play_song->GetNumTracks(); i++)
		curr_event[i] = play_song->GetTrack(i).NextEvent(0);

	// open device
	if ((fd = open(name, O_WRONLY)) == -1) {
		err << "open failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	// set division
	arg = play_song->GetDivision();
	if (ioctl(fd, MSDIVISION, &arg) == -1) {
		err << "ioctl MSDIVISION failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	arg = GetMidiThru();
	if (ioctl(fd, MTHRU, &arg) == -1) {
		err << "ioctl MTHRU failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	if (pthread_create(&thread, 0, PlayThread, this)
	    != 0) {
		err << "Couldn't spawn thread: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

cerr << "returning from Play\n";
	return (1);
}

int
MPU401Thread::Record(Song *rs, Song *ps, int r)
{
	ostrstream err;
	const char *name;
	char *str;
	int arg, i;

	if (fd != -1)
		return (0);
	if ((name = GetName()) == 0)
		return (0);

	rec_song = rs;
	last_play_time = 0;
	last_rec_time = 0;
	last_record_rs = 0;
	if (ps == 0) {
		play_song = 0;
		if ((fd = open(name, O_RDONLY)) == -1) {
			err << "open failed: " << strerror(errno) << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}
		arg = rec_song->GetDivision();
		if (ioctl(fd, MSDIVISION, &arg) == -1) {
			err << "ioctl MSDIVISION failed: " << strerror(errno)
			    << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}
	} else {
		play_song = ps;

		// initialize track/event info
		curr_event = new Event *[play_song->GetNumTracks()];
		if (curr_event == 0) {
			SetError("No more memory");
			return (0);
		}
		for (i = 0; i < play_song->GetNumTracks(); i++)
			curr_event[i] = play_song->GetTrack(i).NextEvent(0);

		if ((fd = open(name, O_RDWR)) == -1) {
			err << "open failed: " << strerror(errno) << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}

		// set division
		arg = play_song->GetDivision();
		if (ioctl(fd, MSDIVISION, &arg) == -1) {
			err << "ioctl MSDIVISION failed: " << strerror(errno)
			    << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}

		// start record timer when first event is scheduled to play
		arg = 1;
		if (ioctl(fd, MRECONPLAY, &arg) == -1) {
			err << "ioctl MRECONPLAY failed: " << strerror(errno)
			    << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}

		finished = 0;
	}

	// set repeat
	SetRepeat(r);

	arg = GetMidiThru();
	if (ioctl(fd, MTHRU, &arg) == -1) {
		err << "ioctl MTHRU failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	if (pthread_create(&thread, 0, PlayThread, this)
	    != 0) {
		err << "Couldn't spawn thread: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	if (pthread_create(&thread, 0, RecordThread, this)
	    != 0) {
		err << "Couldn't spawn thread: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}

	return (1);
}

int
MPU401Thread::Stop(void)
{
	ostrstream err;
	char *str;
	int arg;

cerr << "About to lock finish\n";
	if (pthread_mutex_lock(&finished_mutex) != 0) {
		err << "Couldn't lock finished mutex: " << strerror(errno)
		    << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}
cerr << "locked finish\n";
	finished = 1;
cerr << "about to unlock finish\n";
	if (pthread_mutex_unlock(&finished_mutex) != 0) {
		err << "Couldn't unlock finished mutex: " << strerror(errno)
		    << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}
cerr << "unlocked finish\n";

	if (fd == -1)
		return (1);

	if (ioctl(fd, MGPLAYQ, &arg) == -1) {
		err << "ioctl MGPLAYQ failed: " << strerror(errno) << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}
cerr << "got queue play size of " << arg << "\n";
	// flush queues
	if (arg != 0) {
cerr << "About to drain queue\n";
		if (ioctl(fd, MDRAIN, NULL) == -1) {
			err << "ioctl MDRAIN failed: " << strerror(errno)
			    << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}
cerr << "drained queue\n";
	}

	close (fd);
cerr << "closed fd\n";
	fd = -1;

	delete curr_event;
	curr_event = 0;
	play_song = 0;
	rec_song = 0;
	return (1);
}

int
MPU401Thread::Wait(void)
{
	ostrstream err;
	char *str;

	// XXX Perhaps I can just wait with pthread_join?
	// My only concern is checking to verify that we aren't
	// already finished and calling pthread_join atomically.
	// Or will it work to join on a non-existant thread?
	if (pthread_mutex_lock(&finished_mutex) != 0) {
		err << "Couldn't lock finished mutex: " << strerror(errno)
		    << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}
	while (!finished) {
		if (pthread_cond_wait(&finished_cond, &finished_mutex) != 0) {
			err << "Couldn't block on finished conditional: "
			    << strerror(errno) << ends;
			str = err.str();
			SetError(str);
			delete str;
			return (0);
		}
	}
	if (pthread_mutex_unlock(&finished_mutex) != 0) {
		err << "Couldn't unlock finished mutex: " << strerror(errno)
		    << ends;
		str = err.str();
		SetError(str);
		delete str;
		return (0);
	}
	return (1);
}

void *
MPU401Thread::PlayLoop(void)
{
	void *foo;

	while (!finished) {
		// we really want to select here, but we'll try faking
		// the block by doing a write (0);
		if (write(fd, foo, 0) == -1) {
			cerr << "write (select) failed: " << strerror(errno)
			    << endl;
			continue;
		}
		if (finished)
			break;
		if (WriteEvents() == 0) {
			// wait for queued events to flush
			if (ioctl(fd, MFLUSH) == -1)
				cerr << "ioctl MFLUSH failed: " <<
				    strerror(errno) << "\n";
			// No events to write - finished
			if (pthread_mutex_lock(&finished_mutex) != 0) {
				cerr << "Couldn't lock finished mutex: "
				    << strerror(errno) << "\n";
				pthread_exit(0);
			}
			finished = 1;
			if (pthread_mutex_unlock(&finished_mutex) != 0) {
				cerr << "Couldn't unlock finished mutex: "
				    << strerror(errno) << "\n";
				pthread_exit(0);
			}
			pthread_cond_broadcast(&finished_cond);
			// XXX Should I pass back error?
			pthread_exit(0);
		}
	}
	Stop();
	pthread_exit(0);
	return (0);
}

// Reads events from device
void
MPU401Thread::RecordLoop(void)
{
	MidiDeviceCallback callback;
	SMFTrack track;
	int num_read;
	EventType etype;
	unsigned char event[MaxEventSize];
	const char *errstr;
	Event *e, *first_e;

	while (!finished) {
		do {
			if ((num_read = read(fd, event, MaxEventSize)) == -1) {
				cerr << "error reading MPU401Thread event: " <<
				    strerror(errno) << "\n";
				return;
			}
			track.PutData(event, num_read);
		} while (num_read == MaxEventSize);

		track.SetRunningState(last_record_rs);
		first_e = 0;
		while ((e = ReadEventFromSMFTrack(track, last_rec_time,
		    errstr)) != 0) {
			Event *new_e;

			if (first_e == 0)
				first_e = e;
			new_e = rec_song->PutEvent(0, *e);
			delete e;
			if (new_e == 0)
				continue;
			etype = new_e->GetType();
			// put links on noteoffs
			if ((etype == NOTEON &&
			    ((NoteEvent *)new_e)->GetVelocity() == 0) ||
			    etype == NOTEOFF)
				rec_song->SetNotePair(0, new_e);
		}
		if (errstr != 0)
			cerr << "Error while recording: " << errstr << "\n";
		last_record_rs = track.GetRunningState();
		if ((callback = GetRecordCallback()) != 0)
			callback(first_e);
	}
}

int
MPU401Thread::WriteEvents(void)
{
	SMFTrack smf;
	unsigned long low_time;
	long len;
	int bytes_written, i, low_index, numtowrite, numwritten;
	const char *errstr;

	if (ioctl(fd, MGQAVAIL, &numtowrite) == -1) {
		cerr << "MPU401Thread: ioctl MGQAVAIL failed: "
		    << strerror(errno) << "\n";
		return (-1);
	}
cerr << "numtowrite = " << numtowrite << "\n";
	for (numwritten = 0; numwritten < numtowrite;) {
		// find track with lowest timed event
		low_index = -1;
		for (i = 0; i < play_song->GetNumTracks() - 1; i++) {
			if (curr_event[i] == 0)
				continue;
			if (low_index == -1) {
				low_time = curr_event[i]->GetTime();
				low_index = i;
				continue;
			} else {
				if (curr_event[i]->GetTime() < low_time) {
					low_time = curr_event[i]->GetTime();
					low_index = i;
				}
			}
		}
		// Check to see if we're done
		if (low_index == -1) {
			if (!GetRepeat()) {
				// if we don't repeat, we're done
				if (numwritten != 0) {
					len = smf.GetLength();
					bytes_written = write(fd,
					    smf.GetData(len), len);
					if (bytes_written != len) {
						cerr << "MPU401Thread: write "
						    "wrote wrong amount";
						if (bytes_written == -1)
							cerr << ": " <<
							    strerror(errno);
						else {
							cerr << bytes_written
							    << " of  " <<
							    len << "\n";
						}
					}
				}
				return (numwritten);
			}
			// otherwise reset event pointers
			for (i = 0; i < play_song->GetNumTracks(); i++)
				curr_event[i] = play_song->GetTrack(i).
				    NextEvent(0);
			last_play_time = 0;
			continue;
		}
		// Write events until time changes
		while (curr_event[low_index] != 0 &&
		    curr_event[low_index]->GetTime() == low_time) {
			if (!WriteEventToSMFTrack(smf, last_play_time,
			    curr_event[low_index], errstr)) {
				cerr << "error playing: " << errstr << "\n";
				return (-1);
			}
			numwritten++;
			curr_event[low_index] = play_song->GetTrack(low_index).
			    NextEvent(curr_event[low_index]);
		}
	}
	if (numwritten != 0) {
		len = smf.GetLength();
		bytes_written = write(fd, smf.GetData(len), len);
		if (bytes_written != len) {
			cerr << "MPU401Thread: write wrote wrong amount";
			if (bytes_written == -1)
				cerr << ": " << strerror(errno);
			else
				cerr << bytes_written << " of " << len << "\n";
		}
	}
	return (numwritten);
}

ostream &
operator<<(ostream &os, const MPU401Thread &mpu)
{

	os << mpu.GetName();
	if (mpu.play_song != 0)
		os << " Playing";
	if (mpu.rec_song != 0)
		os << " Recording";
	return (os);
}
#endif
