/*-
 * Copyright (c) 1993, 1994 Michael B. Durian.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Michael B. Durian.
 * 4. The name of the the Author may be used to endorse or promote 
 *    products derived from this software without specific prior written 
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <linux/types.h>
#include <linux/major.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/fcntl.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/string.h>
#include <linux/malloc.h>
#include <linux/sys.h>
#include <linux/termios.h>
#include <linux/ioctl.h>
#include "midi.h"

#include <asm/io.h>
#include <asm/segment.h>
#include <asm/system.h>

#ifdef MODULE
#include "release.h"
#endif

extern struct midi_softc midi_sc[];
extern struct midi_config midi_conf[];
extern int NumMidi;

extern volatile u_long jiffies;
extern struct task_struct *current;

#if defined(__STDC__) || defined(__cplusplus)
#define __P(protos)	protos
#else
#define __P(protos)	()
#endif

#ifndef MIN
#define MIN(a, b)	(((a) < (b)) ? (a) : (b))
#endif

/* I've been burned too many times by different byte ordering */
#define OUTB(x, y) outb((y), (x))
#define INB(x) inb((x))


/* now in midi.h so mem.c can see it
 * long midiinit __P((long kmem_start));
 */
int midiopen __P((struct inode *inode, struct file *file));
void midirelease __P((struct inode *inode, struct file *file));
int midiread __P((struct inode *inode, struct file *file, char *buf, int len));
int midiwrite __P((struct inode *inode, struct file *file, char *buf, int len));
int midiioctl __P((struct inode *inode, struct file *file, unsigned int cmd,
    unsigned long arg));
int midiselect __P((struct inode *inode, struct file *file, int type,
    select_table *wait));
void midiintr __P((int irq));

static struct file_operations midi_fops = {
	NULL,		/* lseek */
	midiread,	/* read */
	midiwrite,	/* write */
	NULL,		/* readdir */
	midiselect,	/* select */
	midiioctl,	/* ioctl */
	NULL,		/* mmap */
	midiopen,	/* open */
	midirelease,	/* release */
	NULL		/* fsync */
};

static void midi_copy_event __P((struct event *, struct event *));
static void midi_initq __P((struct event_queue *));
static int midi_deq __P((struct event_queue *, struct event **));
static int midi_peekq __P((struct event_queue *, struct event **));
static int midi_browseq __P((struct event_queue *, u_long, struct event **));
static int midi_enq __P((struct event_queue *, struct event *));
static int midi_fullreset __P((struct midi_softc *softc));
static int midi_wait_rdy_rcv __P((struct midi_softc *softc));
static int midi_send_command __P((struct midi_softc *softc,
    /* u_char */ int comm));
static int midi_send_command_with_response __P((struct midi_softc *softc,
    /* u_char */ int comm, void *response, int resp_size));
static int midi_reset __P((struct midi_softc *));
static int midi_uart __P((struct midi_softc *));
static int midi_enter_smpte_mode __P((struct midi_softc *));
static int midi_event2smf __P((struct midi_softc *, struct event *event,
    struct stynamic *smf));
static int midi_next_byte __P((struct midi_softc *, char **buf, char *buf_end));
static int midi_str2event __P((struct midi_softc *softc, char **buf,
    char *buf_end, struct event *event));
static int midi_fix2var __P((u_long, u_char *));
static int midi_var2fix __P((u_char *, u_long *));
static u_long midi_smf2timing_tick __P((struct midi_softc *, long));
static u_long midi_timing2smf_tick __P((struct midi_softc *, long, long));
static int midi_midi2SMPTE __P((struct event *, struct SMPTE_frame *));
static int midi_SMPTE_framerate __P((struct SMPTE_frame *, /* u_char */ int));
static void midi_timeout __P((unsigned long arg));
static void midi_add_complete_event __P((struct midi_softc *));
static void midi_schedule_timeout __P((struct midi_softc *, u_long));
static void midi_write_event __P((struct midi_softc *, struct event *, int));
static void midi_reset_devices __P((struct midi_softc *));
static void midi_parse_MTC_event __P((struct midi_softc *, int /* u_char */));
static void midi_register_SMPTE_sync __P((struct midi_softc *, int /* u_char */));
static void midi_dropped_SMPTE_sync  __P((unsigned long arg));
static u_long midi_SMPTE2timing __P((struct SMPTE_frame *));
static int midi_compare_SMPTE_time __P((struct SMPTE_frame *t1,
    struct SMPTE_frame *t2));
static void midi_send_sigio __P((struct midi_softc *softc));
static void midi_time_warp __P((struct midi_softc *softc));
static u_long midi_get_timing_clock __P((struct midi_softc *));
static u_long midi_get_smf_time __P((struct midi_softc *softc));
static int midi_get_byte __P((char **buf, char *buf_end));
static void stynamic_add_byte __P((struct stynamic *, int /* u_char */));
static void stynamic_add_bytes __P((struct stynamic *, u_char *, int));
static u_char stynamic_get_byte __P((struct stynamic *, int));
static void stynamic_copy __P((struct stynamic *, void *, int));
static void stynamic_append __P((struct stynamic *, struct stynamic *));
static void stynamic_release __P((struct stynamic *));
static void stynamic_shift __P((struct stynamic *, int));
static void stynamic_init __P((struct stynamic *));

/* Unused functions
static int midi_get_bytes __P((char **buf, char *buf_end, char *dst, int len));
static void stynamic_copy_from __P((struct stynamic *, int, void *, int));
static void stynamic_print __P((struct stynamic *));
*/

#ifdef MODULE
int
init_module(void)
{

	printk("midi.c: init_module called\n");
	return (midiinit(0));
}

void
cleanup_module(void)
{
	struct midi_softc *softc;
	int i;

	/* don't free anything until all devices are closed */
	for (i = 0; i < NumMidi; i++) {
		softc = &midi_sc[i];
		if (softc->status & MIDI_OPEN) {
			printk("midi device #%d is in use.  Try again later\n",
			    i);
			return;
		}
	}
	for (i = 0; i < NumMidi; i++) {
		softc = &midi_sc[i];
		midi_fullreset(softc);
		kfree_s(softc->wqueue, sizeof(struct event_queue));
		kfree_s(softc->rqueue, sizeof(struct event_queue));
		free_irq(softc->intr);
	}
	if (unregister_chrdev(MIDI_MAJOR, "midi") != 0)
		printk("cleanup_module failed\n");
	else
		printk("cleanup_module succeeded\n");
}
#endif

long
midiinit(kmem_start)
	long kmem_start;
{
	struct midi_softc *softc;
	int i;
	u_char rev;

	if (register_chrdev(MIDI_MAJOR, "midi", &midi_fops)) {
		printk("unable to get major %d for MPU401 MIDI\n",
		    MIDI_MAJOR);
		return (kmem_start);
	}
	for (i = 0; i < NumMidi; i++) {
		softc = &midi_sc[i];
		softc->addr = midi_conf[i].addr;
		softc->intr = midi_conf[i].intr;
		softc->status = MIDI_RD_BLOCK;
		rev = 0x01;
		if (!midi_reset(softc)) {
			if (!midi_reset(softc)) {
				printk("Couldn't reset MPU401 #%d\n", i);
				softc->status |= MIDI_UNAVAILABLE;
				continue;
			}
		}
		if (!midi_wait_rdy_rcv(softc)) {
			printk("MPU401 not acknowledging!\n");
			softc->status |= MIDI_UNAVAILABLE;
			continue;
		}
		if (!midi_send_command_with_response(softc, MIDI_REVISION,
		     &rev, 1))
			printk("Couldn't get MPU401 revision\n");
		if (rev & 0x04)
			softc->features |= SMPTE_EQUIP;
		else
			softc->features &= ~SMPTE_EQUIP;
			
		if (!midi_uart(softc)) {
			printk("Couldn't put MPU401 #%d into UART mode!\n", i);
			softc->status |= MIDI_UNAVAILABLE;
			continue;
		}

		/* allocate memory for event queues */
#ifndef MODULE
		softc->rqueue = (struct event_queue *)kmem_start;
		kmem_start += sizeof(struct event_queue);
		softc->wqueue = (struct event_queue *)kmem_start;
		kmem_start += sizeof(struct event_queue);
#else
		softc->rqueue = kmalloc(sizeof(struct event_queue),
		    GFP_KERNEL);
		if (softc->rqueue == 0) {
			printk("Out of memory for read queue\n");
			return (-1);
		}
		softc->wqueue = kmalloc(sizeof(struct event_queue),
		    GFP_KERNEL);
		if (softc->wqueue == 0) {
			printk("Out of memory for write queue\n");
			kfree_s(softc->rqueue, sizeof(struct event_queue));
			return (-1);
		}
#endif
		memset(softc->rqueue, 0, sizeof(struct event_queue));
		memset(softc->wqueue, 0, sizeof(struct event_queue));
		stynamic_init(&softc->rpartial);
		stynamic_init(&softc->wpartial);
		softc->wpartialpos = 0;

		/* register the intr */
		if (request_irq(softc->intr, midiintr, SA_INTERRUPT, "midi")) {
			printk("MPU401 #%d unable to use interrupt %d\n",
			    i, softc->intr);
#ifndef MODULE
			kmem_start -= 2 * sizeof(struct event_queue);
#else
			kfree_s(softc->rqueue, sizeof(struct event_queue));
			kfree_s(softc->wqueue, sizeof(struct event_queue));
#endif
			softc->status |= MIDI_UNAVAILABLE;
			continue;
		}
		printk("Found MPU401 #%d at 0x%02x irq %d", i, softc->addr,
		    softc->intr);
		if (softc->features & SMPTE_EQUIP)
			printk(" SMPTE Equipped");
		printk("\n");
	}
	return (kmem_start);
}

int
midiopen(inode, file)
	struct inode *inode;
	struct file *file;
{
	register struct midi_softc *softc;
	register int unit;
	register int smpte;
 
	/* high bit is for SMPTE sync */
	unit  = MINOR(inode->i_rdev);
	smpte = unit & 0x80;
	unit &= 0x7f;

	if (unit >= NumMidi)
		return (-ENXIO);
	softc = &midi_sc[unit];
	if (softc->status & MIDI_UNAVAILABLE)
		return (-EIO);
	
	if (softc->status & MIDI_OPEN)
		return (-EBUSY);
	else
		softc->status = MIDI_OPEN;

	/*
	 * reads will block until something appears
	 */
	softc->status |= MIDI_RD_BLOCK;

	/* initialize the queues */
	midi_initq(softc->rqueue);
	midi_initq(softc->wqueue);

	softc->partial_event.event.len = 0;
	softc->partial_event.event.datad = NULL;

	/* make sure we are in UART mode */
	if (!midi_fullreset(softc)) {
		printk("Couldn't put MPU401 into UART mode!\n");
		softc->status |= MIDI_UNAVAILABLE;
		return (-EIO);
	}
	/* if high bit is set, open device in SMPTE mode */
	if (!smpte)
		softc->status &= ~MIDI_EXTCLK;
	else {
		if (!(softc->features & SMPTE_EQUIP))
			return (-ENXIO);
		/* reset again so we can issue commands */
		/* try twice since we were in UART mode */
		(void)midi_reset(softc);
		if (!midi_enter_smpte_mode(softc)) {
			printk("Couldn't enter SMPTE mode\n");
			return (-EIO);
		}
		if (!midi_uart(softc)) {
			/* GJW maybe figure out which MQ board it is.. */
			printk("Couldn't put MQX-32M into UART mode\n");
			return (-EIO);
		}
		softc->status |= MIDI_EXTCLK;
	}

	softc->pgid = current->pid;
	softc->owner = current;

	/* are we going to read, write or both? */
	switch (file->f_flags & O_ACCMODE) {
	case O_RDONLY:
		softc->status &= ~MIDI_WRITING;
		softc->status |= MIDI_READING;
		break;
	case O_WRONLY:
		softc->status |= MIDI_WRITING;
		softc->status &= ~MIDI_READING;
		break;
	case O_RDWR:
		softc->status |= MIDI_READING;
		softc->status |= MIDI_WRITING;
		break;
	}

	/* XXX Do this the right way! */
	/*
	 * GJW i.e. change midiintr() so we can read 0xf1's but
	 * ignore everything else when in SMPTE play mode.. ??
	 */
	if (softc->status & MIDI_EXTCLK)
		softc->status |= MIDI_READING;

	if (file->f_flags & O_NONBLOCK)
		softc->status |= MIDI_NONBLOCK;
	else
		softc->status &= ~MIDI_NONBLOCK;

	softc->status |= MIDI_FIRST_WRITE;
	softc->status |= MIDI_FIRST_SMPTE;
	return (0);
}

void
midirelease(inode, file)
	struct inode *inode;
	struct file *file;
{
	register struct midi_softc *softc;
	register int unit;

	unit  = MINOR(inode->i_rdev) & 0x7f;
	softc = &midi_sc[unit];

	/*
	 * we're not going to finish closing until everything has
	 * been played.
	 */
	if (softc->status & MIDI_WRITING) {
		/* same as MFLUSHQ ioctl */
		if (softc->wqueue->count != 0) {
			softc->status |= MIDI_FLUSH_SLEEP;
			interruptible_sleep_on(&softc->flush_waitq);
			/* if the sleep was interrupted, clear the flag */
			softc->status &= ~MIDI_FLUSH_SLEEP;
		}
	}

	/* turn off any notes that might be stuck on */
	midi_reset_devices(softc);
	midi_fullreset(softc);

	softc->status &= ~MIDI_OPEN;
	return;
}

int
midiread(inode, file, buf, len)
	struct inode *inode;
	struct file *file;
	char *buf;
	int len;
{
	struct event *event;
	register struct midi_softc *softc;
	int init_count, unit, num_to_move;

	unit  = MINOR(inode->i_rdev) & 0x7f;
	softc = &midi_sc[unit];
	init_count = len;

	if (softc->rqueue->count == 0 && softc->rpartial.len == 0) {
		if (softc->status & MIDI_NONBLOCK)
			return (-EWOULDBLOCK);
		else {
			softc->status |= MIDI_RD_SLEEP;
			interruptible_sleep_on(&softc->read_waitq);
			/* XXX maybe check for abort here */
		}
	}
	while (len) {
		/*
		 * dequeue an event if partial is empty
		 */
		if (softc->rpartial.len == 0) {
			if (!midi_deq(softc->rqueue, &event)) {
				softc->status |= MIDI_RD_BLOCK;
				return (init_count - len);
			}
			midi_event2smf(softc, event, &softc->rpartial);
			stynamic_release(&event->data);
		}
		/* read out as much of rpartial as possible */
		num_to_move = MIN(softc->rpartial.len, len);
		if (softc->rpartial.len <= STYNAMIC_SIZE)
			memcpy_tofs(buf, softc->rpartial.datas, num_to_move);
		else
			memcpy_tofs(buf, softc->rpartial.datad, num_to_move);
		stynamic_shift(&softc->rpartial, num_to_move);
		buf += num_to_move;
		len -= num_to_move;
	}
	return (init_count - len);
}

int
midiwrite(inode, file, buf, len)
	struct inode *inode;
	struct file *file;
	char *buf;
	int len;
{
	register struct midi_softc *softc;
	struct event event, *e;
	int convert, init_count, unit, empty_queue;
	char *buf_end;

	unit  = MINOR(inode->i_rdev) & 0x7f;
	softc = &midi_sc[unit];
	init_count = len;

	if (softc->wqueue->count == 0)
		empty_queue = 1;
	else
		empty_queue = 0;

	/* check to see if we'll block */
	if (softc->status & MIDI_WR_BLOCK) {
		if (softc->status & MIDI_NONBLOCK)
			return (-EWOULDBLOCK);
		else {
			softc->status |= MIDI_WR_SLEEP;
			interruptible_sleep_on(&softc->write_waitq);
		}
	}
	/* if returns from sleep and should abort because of DRAIN */
	if (softc->status & MIDI_WR_ABORT) {
		softc->status &= ~MIDI_WR_ABORT;
		return (init_count - len);
	}

	buf_end = buf + len;
	stynamic_init(&event.data);
	while (len) {
		/* block if queue is full */
		if (softc->wqueue->count == MIDI_Q_SIZE) {
			softc->status |= MIDI_WR_BLOCK;
			if (empty_queue) {
				if (!midi_peekq(softc->wqueue, &e))
					return (-EIO);
				midi_schedule_timeout(softc, e->time);
				empty_queue = 0;
			}
			if (softc->status & MIDI_NONBLOCK) {
				return (init_count - len);
			} else {
				softc->status |= MIDI_WR_SLEEP;
				interruptible_sleep_on(&softc->write_waitq);
			}
		}

		/*
		 * 1) get a complete event off queue
		 * 2) convert it from SMF to board format
		 * 3) deal with it
		 */
		convert = midi_str2event(softc, &buf, buf_end, &event);
		len = buf_end - buf;
		switch (convert) {
		case 0:
			break;
		case -1:
			/* not a complete event - we'll get it next time */
			if (empty_queue) {
				if (!midi_peekq(softc->wqueue, &e))
					return (-EIO);
				midi_schedule_timeout(softc, e->time);
			}
			return (init_count - len);
		default:
			return (convert);
		}
		if (midi_enq(softc->wqueue, &event) == -1)
			return (-EIO);
		stynamic_release(&event.data);
		/* set flag so next time we cross LOW water we will SIGIO */
		if (softc->wqueue->count >= MIDI_LOW_WATER)
			softc->status |= MIDI_SENDIO;
	}
	/*
	 * we set the flag if the last write filled the queue, but
	 * we don't need to block
	 */
	if (softc->wqueue->count == MIDI_Q_SIZE)
		softc->status |= MIDI_WR_BLOCK;

	if (empty_queue) {
		if (!midi_peekq(softc->wqueue, &e))
			return (-EIO);
		midi_schedule_timeout(softc, e->time);
	}

	return (init_count - len);
}

void
midiintr(irq)
	int irq;
{
	register struct midi_softc *softc;
	register u_char code;
	register struct partial_event *pe;
	int i;

	for (i = 0; i < NumMidi && midi_sc[i].intr != irq; i++);
	if (i == NumMidi) {
		printk("midi: ivec(%d) is not a valid MPU401 intr number\n",
		    irq);
		return;
	}
	softc = &midi_sc[i];

	/* don't read the data in response to a command */
	if (softc->status & MIDI_NEED_DATA)
		return;
	code = INB(softc->addr + MIDI_DATA);
	if (code == 0xfe && (softc->status & MIDI_NEED_ACK)) {
		softc->status &= ~MIDI_NEED_ACK;
		return;
	}

	/* throw away data if no one has the device open */
	if (!(softc->status & MIDI_READING))
		return;

 	/* pass input data to out port if necessary */
 	if (softc->status & MIDI_THRU) {
 		if (!midi_wait_rdy_rcv(softc))
 			printk("Couldn't pass data thru\n");
		else
			OUTB(softc->addr + MIDI_DATA, code);
 	}
	pe = &softc->partial_event;

	/* check for realtime events */
	if ((code & 0xf8) == 0xf8) {
		switch (code) {
		case 0xfa:	/* start */
			/* reset prev_incoming */
			softc->prev_incoming = midi_get_timing_clock(softc);
			break;
		case 0xff:	/* reset */
			/*
			 * I don't really want to do a full reset
			 * I'll just clear any event in progress
			 */
			stynamic_release(&pe->event);
			pe->state = START;
			break;
		case 0xf8:	/* timing clock */
		case 0xf9:	/* undefined */
		case 0xfb:	/* continue */
		case 0xfc:	/* stop */
		case 0xfd:	/* undefined */
		case 0xfe:	/* active sensing */
			break;
		}
		return;
	}
INTR_SWITCH:
	switch (pe->state) {
	case START:
		/* record the time when the event started */
		pe->time = midi_get_timing_clock(softc);
		pe->tempo = softc->rtempo;
		/* start collecting the input */
		stynamic_release(&pe->event);
		/* decide what state is next */
		if (!(code & 0x80)) {
			stynamic_add_byte(&pe->event, code);
			switch (pe->rs) {
			case 0x80:
			case 0x90:
			case 0xa0:
			case 0xb0:
			case 0xe0:
				/*
				 * code is the first data byte, but
				 * we still need to get the second
				 */
				pe->state = NEEDDATA2;
				break;
			case 0xc0:
			case 0xd0:
				/* code is the only data byte */
				pe->state = START;
				midi_add_complete_event(softc);
				break;
			default:
				break;
			}
		} else {
			switch (code & 0xf0) {
			case 0x80:
			case 0x90:
			case 0xa0:
			case 0xb0:
			case 0xe0:
				pe->rs = code & 0xf0;
				stynamic_add_byte(&pe->event, code);
				pe->state = NEEDDATA1;
				break;
			case 0xc0:
			case 0xd0:
				pe->rs = code & 0xf0;
				stynamic_add_byte(&pe->event, code);
				pe->state = NEEDDATA2;
				break;
			default:
				switch (code) {
				case 0xf0: /* sysex */
					stynamic_add_byte(&pe->event, code);
					pe->state = SYSEX;
					break;
				case 0xf1: /* MTC quarter-frame */
					if (softc->status & MIDI_EXTCLK) {
						pe->state = MTC;
						softc->extclock++;
					}
					break;
				case 0xf4: /* undefined */
				case 0xf5: /* undefined */
				case 0xf6: /* tune request */
				case 0xf7: /* EOX (terminator) */
					/* ignore these */
					break;
				case 0xf2: /* song position */
					pe->state = SYSTEM2;
					break;
				case 0xf3: /* song select */
					pe->state = SYSTEM1;
					break;
				}
				break;
			}
		}
		break;
	case NEEDDATA1:
		stynamic_add_byte(&pe->event, code);
		pe->state = NEEDDATA2;
		break;
	case NEEDDATA2:
		stynamic_add_byte(&pe->event, code);
		pe->state = START;
		midi_add_complete_event(softc);
		break;
	case SYSEX:
		/* any non-data byte ends sysex */
		if (!(code & 0x80))
			stynamic_add_byte(&pe->event, code);
		else {
			stynamic_add_byte(&pe->event, 0xf7);
			midi_add_complete_event(softc);
			pe->state = START;
			if (code != 0xf7)
				goto INTR_SWITCH;
		}
		break;
	case SYSTEM1:
		/* throw away one data byte of a system message */
		pe->state = START;
		break;
	case SYSTEM2:
		/* throw away two data bytes of a system message */
		pe->state = SYSTEM1;
		break;
	case MTC:
		pe->state = START;
		midi_parse_MTC_event(softc, code);
		break;
	}
	return;
}

int
midiioctl(inode, file, cmd, arg)
	struct inode *inode;
	struct file *file;
	unsigned int cmd;
	unsigned long arg;
{
	struct event *event;
	struct midi_softc *softc;
	u_long ulval;
	register int unit;
	int val;

	unit  = MINOR(inode->i_rdev) & 0x7f;
	softc = &midi_sc[unit];

	switch (cmd) {
	case FIOASYNC:
	case MASYNC:
		/*
		 * Linux doesn't pass FIOASYNC to the driver, so
		 * we need a separate ioctl entry point to do
		 * exactly what FIOASYNC does.  Thus MASYNC.
		 */
		memcpy_fromfs(&val, (int *)arg, sizeof(int));
		if (!val)
			softc->status &= ~MIDI_ASYNC;
		else {
			softc->status |= MIDI_ASYNC;
			if (softc->wqueue->count < MIDI_LOW_WATER
			    && softc->status & MIDI_WRITING) {
				if (softc->pgid < 0)
					kill_pg(-softc->pgid, SIGIO, 1);
				else
					kill_proc(softc->pgid, SIGIO, 1);
			}
		}
		break;
	case TIOCSPGRP:
		memcpy_fromfs(&softc->pgid, (int *)arg, sizeof(int));
		break;
	case TIOCGPGRP:
		memcpy_tofs((int *)arg, &softc->pgid, sizeof(int));
		break;
	case MRESET:
		if (!midi_fullreset(softc))
			return (-EIO);
		break;
	case MSDIVISION:
		memcpy_fromfs(&val, (int *)arg, sizeof(int));
 		/* must recalculate play remainder */
 		softc->premainder = softc->premainder * val / softc->division;
 		softc->division = val;
		break;
	case MGDIVISION:
		memcpy_tofs((int *)arg, &softc->division, sizeof(int));
		break;
	case MDRAIN:
		/* dequeue all everything */
		while (midi_deq(softc->rqueue, &event))
			stynamic_release(&event->data);
		while (midi_deq(softc->wqueue, &event))
			stynamic_release(&event->data);
		/* remove any events already being timed */
		del_timer(&softc->timer);
		softc->status &= ~MIDI_WR_BLOCK;
		softc->status |= MIDI_RD_BLOCK;
		if (softc->status & MIDI_WR_SLEEP) {
			softc->status &= ~MIDI_WR_SLEEP;
			softc->status |= MIDI_WR_ABORT;
			wake_up_interruptible(&softc->write_waitq);
		}
		midi_reset_devices(softc);
		break;
	case MFLUSH:
		if (softc->wqueue->count != 0) {
			softc->status |= MIDI_FLUSH_SLEEP;
			interruptible_sleep_on(&softc->flush_waitq);
		}
		break;
	case MGPLAYQ:
		memcpy_tofs((int *)arg, &softc->wqueue->count, sizeof(int));
		break;
	case MGRECQ:
		memcpy_tofs((int *)arg, &softc->rqueue->count, sizeof(int));
		break;
	case MGQAVAIL:
		val = MIDI_Q_SIZE - softc->wqueue->count;
		memcpy_tofs((int *)arg, &val, sizeof(int));
		break;
 	case MTHRU:
		memcpy_fromfs(&val, (int *)arg, sizeof(int));
 		if (val)
 			softc->status |= MIDI_THRU;
 		else
 			softc->status &= ~MIDI_THRU;
  		break;
 	case MRECONPLAY:
		memcpy_fromfs(&val, (int *)arg, sizeof(int));
 		if (val)
 			softc->status |= MIDI_RECONPLAY;
 		else
 			softc->status &= ~MIDI_RECONPLAY;
  		break;
	case MGSMPTE:
		if (!(softc->status & MIDI_EXTCLK))
			return (-ENODEV);
		memcpy_tofs((struct SMPTE_frame *)arg, &softc->SMPTE_current,
		    sizeof(struct SMPTE_frame));
		break;
	case MGSMFTIME:
		ulval = midi_get_smf_time(softc);
		memcpy_tofs((u_long *)arg, &ulval, sizeof(u_long));
		break;
	default:
		return (-ENOTTY);
	}
	return (0);
}

int
midiselect(inode, file, type, wait)
	struct inode *inode;
	struct file *file;
	int type;
	select_table *wait;
{
	register struct midi_softc *softc;
	int unit;

	unit  = MINOR(inode->i_rdev) & 0x7f;
	softc = &midi_sc[unit];

	switch (type) {
	case SEL_IN:
		if (!(softc->status & MIDI_RD_BLOCK))
			return (1);
		else {
			softc->status |= MIDI_SELIN;
			select_wait(&softc->selin_waitq, wait);
			return (0);
		}
		break;
	case SEL_OUT:
		if (!(softc->status & MIDI_WR_BLOCK))
			return (1);
		else {
			softc->status |= MIDI_SELOUT;
			select_wait(&softc->selout_waitq, wait);
			return (0);
		}
		break;
	case SEL_EX:
		softc->status |= MIDI_SELEX;
		select_wait(&softc->selex_waitq, wait);
		return (0);
	default:
		return (0);
	}
}

int
midi_fullreset(softc)
	struct midi_softc *softc;
{
	u_char pitch;
	struct event *event;
	struct tempo_change *tch, *nextch;

	/* dequeue all everything */
	while (midi_deq(softc->rqueue, &event))
		stynamic_release(&event->data);
	while (midi_deq(softc->wqueue, &event))
		stynamic_release(&event->data);
	/* free any storage allocated for tempo changes */
	for (tch = softc->tempo_changes; tch != NULL; tch = nextch) {
		nextch = tch->next;
		kfree_s(tch, sizeof(struct tempo_change));
	}
	softc->tempo_changes = NULL;
	/* remove any events already being timed */
	del_timer(&softc->timer);
	/* mark start time */
	softc->start_time = jiffies;
	softc->prev_incoming = midi_get_timing_clock(softc);
	softc->prev_outgoing = 0;
	/* zero smf position counter */
	softc->write_smf_time = 0;
	/* reset fractional count */
	softc->premainder = 0;
	softc->rremainder = 0;
	/* initialize some variables */
	/* this is 120 bpm when a quarter note == 1 beat */
	softc->ptempo = 500000;
	softc->rtempo = 500000;
 	softc->prev_rtempo = 500000;

	/* reset external clock */
	softc->extclock = 0;

	/* clear noteon */
	for (pitch = 0; pitch <= 0x7f; pitch++)
		softc->noteon[pitch] = 0;
	softc->noteonrs = 0;

	/* clear running play state */
	softc->writers = 0;

	/* reset partial event stuff */
	stynamic_release(&softc->partial_event.event);
	softc->partial_event.state = START;
	stynamic_release(&softc->rpartial);
	stynamic_release(&softc->wpartial);
	softc->wpartialpos = 0;

	/* defaults to 120 clocks per beat */
	softc->division = 120;
	/* default to kernel timing 100 */
	softc->hz = HZ;
	/* reset and enter uart mode */
	(void)midi_reset(softc);
	return (midi_uart(softc));
}

int
midi_wait_rdy_rcv(softc)
	struct midi_softc *softc;
{
	int flag;
	register int i;

	for (i = 0; i < MIDI_TRIES; i++) {
		flag = INB(softc->addr + MIDI_STATUS) & MIDI_RDY_RCV;
		if (flag == 0)
			break;
	}
	if (i == MIDI_TRIES)
		return (0);
	return(1);
}

int
midi_send_command(softc, comm)
	struct midi_softc *softc;
	u_char comm;
{
	int flag;
	register int i;
	unsigned char ack;

	/* writing command and setting flag must be atomic */
	cli();
	OUTB(softc->addr + MIDI_COMMAND, comm);
	softc->status |= MIDI_NEED_ACK;
	sti();

	/* time out after MIDI_TRIES times */
	for (i = 0; i < MIDI_TRIES; i++) {
		/* did we pick up the ack via midiintr? */
		if (!(softc->status & MIDI_NEED_ACK))
			break;
		/* can we read a data byte? */
		flag = INB(softc->addr + MIDI_STATUS) & MIDI_DATA_AVL;
		if (flag == 0)
			break;
	}
	if (i == MIDI_TRIES)
		return (0);

	if (softc->status & MIDI_NEED_ACK) {
		ack = INB(softc->addr + MIDI_DATA);
		if (ack != MIDI_ACK)
			return (0);
	}
	softc->status &= ~MIDI_NEED_ACK;
	return (1);
}

int
midi_send_command_with_response(softc, comm, response, resp_size)
	struct midi_softc *softc;
	u_char comm;
	void *response;
	int resp_size;
{
	int flag;
	register int i, j;
	unsigned char ack, *resp;

	/* writing command and setting ack flag must be atomic */
	cli();
	OUTB(softc->addr + MIDI_COMMAND, comm);
	softc->status |= MIDI_NEED_ACK;
	softc->status |= MIDI_NEED_DATA;
	sti();

	/* time out after MIDI_TRIES times */
	for (i = 0; i < MIDI_TRIES; i++) {
		/* did we pick up the ack via midiintr? */
		if (!(softc->status & MIDI_NEED_ACK))
			break;
		/* can we read a data byte? */
		flag = INB(softc->addr + MIDI_STATUS) & MIDI_DATA_AVL;
		if (flag == 0)
			break;
	}
	if (i == MIDI_TRIES)
		return (0);

	if (softc->status & MIDI_NEED_ACK) {
		ack = INB(softc->addr + MIDI_DATA);
		if (ack != MIDI_ACK)
			return (0);
	}
	softc->status &= ~MIDI_NEED_ACK;
	resp = response;
	for (j = 0; j < resp_size; j++) {
		for (i = 0; i < MIDI_TRIES; i++) {
			flag = INB(softc->addr + MIDI_STATUS) & MIDI_DATA_AVL;
			if (flag == 0)
				break;
		}
		if (i == MIDI_TRIES)
			return (0);
		resp[j] = INB(softc->addr + MIDI_DATA);
	}
	softc->status &= ~MIDI_NEED_DATA;
	return (1);
}

int
midi_reset(softc)
	struct midi_softc *softc;
{

	if (!midi_wait_rdy_rcv(softc))
		return (0);

	if (!midi_send_command(softc, MIDI_RESET))
		return (0);

	return (1);
}

int
midi_uart(softc)
	struct midi_softc *softc;
{

	if (!midi_wait_rdy_rcv(softc))
		return (0);

	if (!midi_send_command(softc, MIDI_UART))
		return (0);
	return (1);
}

int
midi_enter_smpte_mode(softc)
	struct midi_softc *softc;
{

	if (!(softc->features & SMPTE_EQUIP))
		return (0);
	memset(&softc->SMPTE_current, 0, sizeof(softc->SMPTE_current));
	softc->SMPTE_current.rate = 30;
	memset(&softc->SMPTE_partial, 0, sizeof(softc->SMPTE_partial));
	softc->SMPTE_partial.rate = 30;
	/* first guess for hz is 30-frame SMPTE ==> 120 Hz */
	softc->hz = 120;

	/* MQX on-board clock */
	if (!midi_wait_rdy_rcv(softc))
		return (0);
	if (!midi_send_command(softc, 0x80))
		return (0);
	/* SMPTE tape sync */
	if (!midi_wait_rdy_rcv(softc))
		return (0);
	if (!midi_send_command(softc, 0x3d))
		return (0);
	/* SMPTE reading active */
	if (!midi_wait_rdy_rcv(softc))
		return (0);
	if (!midi_send_command(softc, 0x2d))
		return (0);
	/* system-exclusive messages to PC */
	if (!midi_wait_rdy_rcv(softc))
		return (0);
	if (!midi_send_command(softc, 0x97))
		return (0);
	/* system common, real-time messages to PC */
	if (!midi_wait_rdy_rcv(softc))
		return (0);
	if (!midi_send_command(softc, 0x38))
		return (0);
	if (!midi_wait_rdy_rcv(softc))
		return (0);
	if (!midi_send_command(softc, 0x39))
		return (0);
	/* clock messages to PC */
	if (!midi_wait_rdy_rcv(softc))
		return (0);
	if (!midi_send_command(softc, 0x95))
		return (0);

	return (1);
}

int
midi_event2smf(softc, event, smf)
	struct midi_softc *softc;
	struct event *event;
	struct stynamic *smf;
{
	long tempo, smf_ticks;
	int tmp_len;
	u_char tmp_buf[4];

	/* convert from timing ticks to SMF ticks */
	smf_ticks = midi_timing2smf_tick(softc, event->tempo, event->time);
	tmp_len = midi_fix2var(smf_ticks, tmp_buf);
	stynamic_add_bytes(smf, tmp_buf, tmp_len);

	switch (event->type) {
	case NORMAL:
		stynamic_append(smf, &event->data);
		break;
	case TEMPO:
		/* this case just won't occur */
		stynamic_copy(&event->data, (void *)&tempo, sizeof(tempo));
		stynamic_add_byte(smf, 0xff);
		stynamic_add_byte(smf, 0x51);
		stynamic_add_byte(smf, 0x03);
		stynamic_add_byte(smf, (tempo & 0xff000000) >> 16);
		stynamic_add_byte(smf, (tempo & 0xff00) >> 8);
		stynamic_add_byte(smf, tempo & 0xff);
		break;
	case SYSX:
		stynamic_add_byte(smf, 0xf0);
		/* skip over the leading 0xf0 */
		stynamic_shift(&event->data, 1);
		tmp_len = midi_fix2var(event->data.len, tmp_buf);
		stynamic_add_bytes(smf, tmp_buf, tmp_len);
		stynamic_append(smf, &event->data);
		break;
	default:
		break;
	}
	return (1);
}

int
midi_get_byte(buf, buf_end)
	char **buf, *buf_end;
{
	u_char byte;

	if (*buf >= buf_end)
		return (-1);
	else {
		memcpy_fromfs(&byte, *buf, 1);
		*buf += 1;
		return (byte);
	}
}

#if 0
/* unused function */
int
midi_get_bytes(buf, buf_end, dst, len)
	char **buf, *buf_end, *dst;
	int len;
{

	if (*buf + len >= buf_end)
		return (0);
	memcpy_fromfs(dst, *buf, len);
	*buf += len;
	return (1);
}
#endif

int
midi_next_byte(softc, buf, buf_end)
	struct midi_softc *softc;
	char **buf, *buf_end;
{
	int byte;

	/* if we're not at the end of a partial event, read from it */
	if (softc->wpartialpos < softc->wpartial.len) {
		byte = stynamic_get_byte(&softc->wpartial,
		    softc->wpartialpos++);
		return (byte);
	} else {
		/* read from str and copy uio onto partial event */
		if ((byte = midi_get_byte(buf, buf_end)) == -1) {
			/*
			 * reset partialpos so next time through
			 * we'll read from the partial event if
			 * it is non-zero in length
			 */
			softc->wpartialpos = 0;
			return (-1);
		}
		stynamic_add_byte(&softc->wpartial, byte);
		softc->wpartialpos = softc->wpartial.len;
		return (byte);
	}
}

int
midi_str2event(softc, buf, buf_end, event)
	struct midi_softc *softc;
	char **buf, *buf_end;
	struct event *event;
{
	u_long smf_ticks, ulen;
	long tempo;
	int byte, extra_byte, i, len, num_data_bytes;
	int rs_change;
	u_char meta_type, tmp_buf[256];
#ifdef GJWDEBUG
	struct SMPTE_frame tmp_frame;
#endif

	/* copy in timing portion */
	len = 0;
	do {
		if ((byte = midi_next_byte(softc, buf, buf_end)) == -1) {
			stynamic_release(&event->data);
			return (-1);
		}
		tmp_buf[len++] = byte;
	} while (byte & 0x80);

	/* compute time in smf ticks */
	midi_var2fix(tmp_buf, &smf_ticks);

	/* now convert from smf to timing */
	event->time = midi_smf2timing_tick(softc, smf_ticks);
	softc->write_smf_time += smf_ticks;
	event->smf_time = softc->write_smf_time;

	/* get first byte of event data */
	if ((byte = midi_next_byte(softc, buf, buf_end)) == -1) {
		stynamic_release(&event->data);
		return (-1);
	}
	switch (byte) {
	case 0xf0:
		/* basic sysex type */
		event->type = SYSX;
		len = 0;
		do {
			if ((byte = midi_next_byte(softc, buf, buf_end))
			    == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[len++] = byte;
		} while (byte & 0x80);
		midi_var2fix(tmp_buf, &ulen);
		stynamic_add_byte(&event->data, 0xf0);
		for (; ulen > 0; ulen--) {
			if ((byte = midi_next_byte(softc, buf, buf_end))
			    == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			stynamic_add_byte(&event->data, byte);
		}
		break;
	case 0xf7:
		/* continued sysex type */
		event->type = SYSX;
		len = 0;
		do {
			if ((byte = midi_next_byte(softc, buf, buf_end))
			    == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[len++] = byte;
		} while (byte & 0x80);
		midi_var2fix(tmp_buf, &ulen);
		for (; ulen > 0; ulen--) {
			if ((byte = midi_next_byte(softc, buf, buf_end))
			    == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			stynamic_add_byte(&event->data, byte);
		}
		break;
	case 0xff:
		/* meta events */
		if ((byte = midi_next_byte(softc, buf, buf_end)) == -1) {
			stynamic_release(&event->data);
			return (-1);
		}
		meta_type = byte;
		/* get length of meta data */
		len = 0;
		do {
			if ((byte = midi_next_byte(softc, buf, buf_end))
			    == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[len++] = byte;
		} while (byte & 0x80);
		midi_var2fix(tmp_buf, &ulen);
		/* read it in - meta events are not over 256 in size*/
		for (i = 0; i < ulen; i++) {
			if ((byte = midi_next_byte(softc, buf, buf_end))
			    == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[i] = byte;
		}
		switch (meta_type) {
		default:
			/*
			 * we'll skip these events, but we need to
			 * save the timing info
			 */
			event->type = NOP;
			break;
		case 0x51:
			/* tempo event */
			event->type = TEMPO;
			tempo = 0;
			for (i = 0; i < 3; i++) {
				tempo = tempo << 8;
				tempo |= tmp_buf[i];
			}
			stynamic_add_bytes(&event->data, (u_char *)&tempo,
			    sizeof(tempo));
			/*
			 * change ptempo now, rtempo will change when
			 * the event's time comes up
			 */
			softc->ptempo = tempo;
			break;
		case 0x54:
			/* MetaSMPTE */
			event->type = SMPTE;
			stynamic_add_bytes(&event->data, tmp_buf, 5);

			/* 
			 * MetaSMPTE must come before any non-zero delta
			 * events.  Therefore let MetaSMPTE *define* the
			 * SMF zero point
			 */
			softc->write_smf_time = 0;
#ifdef GJWDEBUG
			/*
			 * convert the MIDI format SMPTE frame into our
			 * format
			 */
			midi_midi2SMPTE(event, &tmp_frame);

			printk("MetaSMPTE %2d:%2d:%2d:%2d:%2d\n",
			    tmp_frame.hour, tmp_frame.minute,
			    tmp_frame.second, tmp_frame.frame,
			    tmp_frame.fraction);
#endif
			break;
		}
		break;
	default:
		if ((byte & 0xf0) == 0x80) {
			u_char rs_chan, rs_type;

			/* check note off events separately */
			tmp_buf[0] = byte;
			if ((byte = midi_next_byte(softc, buf, buf_end))
			    == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[1] = byte;
			if ((byte = midi_next_byte(softc, buf, buf_end))
			    == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[2] = byte;
			len = 3;
			/*
			 * check to see if we can collapse and use
			 * running state
			 */
			rs_type = softc->writers & 0xf0;
			rs_chan = softc->writers & 0x0f;
			/*
			 * collapse to running state if time is 0
			 * and the running state is the same
			 * or the running state is note on for the
			 * same channel and the note off velocity is
			 * zero.
			 */
			if (event->time == 0 && (softc->writers == tmp_buf[0]
			    || (tmp_buf[2] == 0 && rs_type == 0x90
			    && rs_chan == (tmp_buf[0] & 0x0f)))) {
				tmp_buf[0] = tmp_buf[1];
				tmp_buf[1] = tmp_buf[2];
				len = 2;
			} else {
				softc->writers = tmp_buf[0];
			}
		} else {
			extra_byte = 0;
			rs_change = 0;
			if ((byte & 0x80) && (byte != softc->writers)) {
				softc->writers = byte;
				rs_change = 1;
			}
			len = 0;
			if (event->time != 0 || rs_change) {
				/*
				 * stick in a mode byte if time is non-zero
				 * This is so we don't confuse hardware that
				 * is turned on while we're playing
				 * also add it if the running state changes
				 */
				tmp_buf[0] = softc->writers;
				len = 1;
			}
			if (byte & 0x80)
				extra_byte = 1;
			else {
				tmp_buf[len] = byte;
				len++;
			}

			switch (softc->writers & 0xf0) {
			case 0x80:
			case 0x90:
			case 0xa0:
			case 0xb0:
			case 0xe0:
				num_data_bytes = 1;
				break;
			default:
				num_data_bytes = 0;
			}
			for (i = 0; i < num_data_bytes + extra_byte; i++) {
				if ((byte = midi_next_byte(softc, buf, buf_end))
				    == -1) {
					stynamic_release(&event->data);
					return (-1);
				}
				tmp_buf[len++] = byte;
			}
		}
		event->type = NORMAL;
		stynamic_add_bytes(&event->data, tmp_buf, len);
	}

	stynamic_release(&softc->wpartial);
	softc->wpartialpos = 0;
	return (0);
}

int
midi_fix2var(fix, var)
	u_long fix;
	u_char *var;
{
	int i;
	unsigned char buf[4], *bptr;

	buf[0] = buf[1] = buf[2] = buf[3] = 0;
	bptr = buf;
	*bptr++ = fix & 0x7f;
	while ((fix >>= 7) > 0) {
		*bptr |= 0x80;
		*bptr++ += (fix & 0x7f);
        }

	i = 0;
	do {
		*var++ = *--bptr;
		i++;
	} while (bptr != buf);

	return (i);
}

int
midi_var2fix(var, fix)
	u_char *var;
	u_long *fix;
{
	int delta;

	*fix = 0;
	delta = 0;
	do {
		*fix = (*fix << 7) + (*var & 0x7f);
		delta++;
	} while (*var++ & 0x80);

	return (delta);
}

u_long
midi_smf2timing_tick(softc, smf)
	struct midi_softc *softc;
	long smf;
{
 	long long denominator, numerator;
 	u_long timing;
  
	numerator = (long long)softc->ptempo * softc->hz * smf
	    + softc->premainder;
 	denominator = 1000000 * (long long)softc->division;
 	timing = numerator / denominator;
 	softc->premainder = numerator % denominator;
  	return (timing);
}

u_long
midi_timing2smf_tick(softc, tempo, timing)
	struct midi_softc *softc;
	long tempo, timing;
{
 	long long numerator, denominator;
 	u_long smf;
  
 	if (softc->prev_rtempo != tempo) {
 		/*
 		 * also update the rremainder to reflect tempo
 		 * change
 		 */
 		softc->rremainder = softc->rremainder * tempo
 		    / softc->prev_rtempo;
 		softc->prev_rtempo = tempo;
 	}
	numerator = (long long)softc->division * 1000000 * timing
	    + softc->rremainder;
 	denominator = (long long)tempo * softc->hz;
 	smf = numerator / denominator;
 	softc->rremainder = numerator % denominator;
  	return (smf);
}

void
midi_add_complete_event(softc)
	struct midi_softc *softc;
{
	struct event event;
	struct partial_event *pe;
	long ticks;

	stynamic_init(&event.data);
	pe = &softc->partial_event;

	ticks = pe->time - softc->prev_incoming;
	if (ticks < 0) {
		ticks = 0;
		pe->time = softc->prev_incoming;
	}
	event.time = ticks;
	event.tempo = pe->tempo;
	switch (stynamic_get_byte(&pe->event, 0)) {
	case 0xf0:
		/* sysex */
		event.type = SYSX;
		break;
	default:
		event.type = NORMAL;
		break;
	}
	stynamic_append(&event.data, &softc->partial_event.event);
	/* enqueue new event */
	midi_enq(softc->rqueue, &event);
	/* readjust previous event time */
	softc->prev_incoming = pe->time;

	softc->status &= ~MIDI_RD_BLOCK;
	if (softc->status & MIDI_RD_SLEEP) {
		softc->status &= ~MIDI_RD_SLEEP;
		wake_up_interruptible(&softc->read_waitq);
	}
	if (softc->status & MIDI_ASYNC) {
		if (softc->pgid < 0)
			kill_pg(-softc->pgid, SIGIO, 1);
		else
			kill_proc(softc->pgid, SIGIO, 1);
	}

	/* notify select that there is data to be read */
	if (softc->status & MIDI_SELIN) {
		softc->status &= ~MIDI_SELIN;
		wake_up_interruptible(&softc->selin_waitq);
	}

	stynamic_release(&event.data);
	return;
}

void
midi_schedule_timeout(softc, t)
	struct midi_softc *softc;
	u_long t;
{

	/* don't need this with external timing */
	if (!(softc->status & MIDI_EXTCLK)) {
		if (softc->status & MIDI_FIRST_WRITE) {
			softc->start_time = jiffies;
			softc->status &= ~MIDI_FIRST_WRITE;
		}
		softc->timer.expires = t;
		softc->timer.data = (unsigned long)softc;
		softc->timer.function = midi_timeout;
		add_timer(&softc->timer);
	}
}

void
midi_timeout(arg)
	unsigned long arg;
{
	struct event *event;
	struct midi_softc *softc = (struct midi_softc *)arg;
	u_long sched_time, t;
	int quiet;
#ifdef GJWDEBUG
	struct SMPTE_frame *cf = &softc->SMPTE_current;
#endif

	if (!midi_peekq(softc->wqueue, &event))
		return;

	while ((softc->prev_outgoing + event->time) <= 
	    (t = midi_get_timing_clock(softc))) {
		midi_deq(softc->wqueue, NULL);

#ifdef GJWDEBUG
		/* Useful for debugging sync problems */
		printk("Writing: S:%2d:%2d:%2d:%2d P:%4lu E:%4lu/%3lu M:%4lu\n",
		    cf->minute, cf->second, cf->frame, cf->fraction,
		    softc->prev_outgoing, softc->extclock,
		    event->time, event->smf_time);
#endif

		/*
		 * If the event is backlogged for more than BACKLOG_TIME ticks,
		 * don't play it
		 */
		if ((t - softc->prev_outgoing - event->time) > BACKLOG_TIME)
			quiet = 1;
		else
			quiet = 0;

		midi_write_event(softc, event, quiet);
		softc->prev_outgoing += event->time;
		stynamic_release(&event->data);
		if (!midi_peekq(softc->wqueue, &event))
			return;
		/* clear record timer? */
		if (softc->status & MIDI_RECONPLAY) {
			softc->prev_incoming = midi_get_timing_clock(softc);
			softc->status &= ~MIDI_RECONPLAY;
		}
	}
	sched_time = event->time - (t - softc->prev_outgoing);
	midi_schedule_timeout(softc, sched_time);
}

void
midi_dropped_SMPTE_sync(arg)
	unsigned long arg;
{
	struct midi_softc *softc = (struct midi_softc *)arg;
	struct SMPTE_frame *cf;

	cf = &softc->SMPTE_current;

	/*
	 * If we are here, then too many ticks went by without 
	 * a SMPTE frame.  Assume that sync is lost
	 */
	softc->SMPTE_current.status &= ~SMPTE_SYNC;
	softc->SMPTE_partial.status &= ~SMPTE_SYNC;

	/*
	 * We'll need to catch another set of 8 quarter-frame 
	 * messages before we can re-sync.  Make sure we get a
	 * full set, starting with the first one.
	 */
	softc->SMPTE_current.status &= ~SMPTE_FIRST0xF1;
	softc->SMPTE_partial.status &= ~SMPTE_FIRST0xF1;

#ifdef GJWDEBUG
	printk("SMPTE sync dropped at %2d:%2d:%2d:%2d:%2d\n",
	    cf->hour, cf->minute, cf->second, cf->frame, cf->fraction);
#endif
}

void
midi_write_event(softc, event, quiet)
	struct midi_softc *softc;
	struct event *event;
	int quiet;
{
	struct tempo_change *tchange, *prev_tchange;
	int i;
	u_char bytes[4], channel, command, *dataptr;

	switch (event->type) {
	case NOP:
		break;
	case SMPTE:
		break;
	case TEMPO:
		stynamic_copy(&event->data, &softc->rtempo,
		    sizeof(softc->rtempo));
		for (tchange = softc->tempo_changes, prev_tchange = NULL;
		    tchange != NULL;
		    prev_tchange = tchange, tchange = tchange->next) {
			if (tchange->smf_time <= event->smf_time)
				break;
		}
		if (tchange == NULL) {
			/* add the tempo change to our list */
			tchange = kmalloc(sizeof(struct tempo_change),
			    GFP_ATOMIC);
			if (tchange == NULL) {
				printk("Out of kernel memory: we're screwed\n");
				break;
			} else {
				tchange->time = event->time +
				    softc->prev_outgoing;
				tchange->smf_time = event->smf_time;
				tchange->tempo = softc->rtempo;
				tchange->next = NULL;
				/*
				 * Don't update through a null pointer.
				 * On the other hand, make sure to update
				 * softc->tempo_changes to reflect storage.
				 */
				if (prev_tchange != NULL)
					prev_tchange->next = tchange;
				else
					softc->tempo_changes = tchange;
			}
		}
		break;
	case NORMAL:
		/*
		 * fourth byte might not be valid, but who cares,
		 * we're only reading and in the kernel.  We'll
		 * ignore it if it isn't.
		 */
		stynamic_copy(&event->data, &bytes, 4);
		if (!(bytes[0] & 0x80))
			dataptr = &bytes[0];
		else {
			softc->noteonrs = bytes[0];
			dataptr = &bytes[1];
		}
		command = softc->noteonrs & 0xf0;
		channel = softc->noteonrs & 0x0f;

		/*
		 * In `quiet' mode, don't turn on any new notes but do
		 * turn them off if we're asked.
		 */
		if (command == 0x90) {
			/*
			 * set or clear appropriate bit in noteon
			 * array depending on velocity value
			 */
			if (dataptr[1] != 0)
				softc->noteon[dataptr[0]] |= 1 << channel;
			else
				softc->noteon[dataptr[0]] &= ~(1 << channel);
		}
		if (command == 0x80)
			/* clear bit */
			softc->noteon[dataptr[0]] &= ~(1 << channel);
		/* FALLTHRU */
	default:
		if (!quiet) {
			for (i = 0; i < event->data.len; i++) {
				if (!midi_wait_rdy_rcv(softc))
					break;
				OUTB(softc->addr + MIDI_DATA,
				    stynamic_get_byte(&event->data, i));
			}
		}
		break;
	}
	midi_send_sigio(softc);
}

/*
 * try to reset the midi devices as best we can
 */
void
midi_reset_devices(softc)
	struct midi_softc *softc;
{
	u_char channel, pitch;

	/* manual note off calls - turn off any note that is on */
	for (pitch = 0; pitch <= 0x7f; pitch++) {
		for (channel = 0; channel <= 0x0f; channel++) {
			if ((softc->noteon[pitch] >> (int)channel) & 0x01) {
				if (!midi_wait_rdy_rcv(softc))
					break;
				OUTB(softc->addr + MIDI_DATA, channel | 0x90);

				if (!midi_wait_rdy_rcv(softc))
					break;
				OUTB(softc->addr + MIDI_DATA, pitch);

				if (!midi_wait_rdy_rcv(softc))
					break;
				OUTB(softc->addr + MIDI_DATA, 0);
			}
		}
		softc->noteon[pitch] = 0;
	}
	for (channel = 0; channel <= 0x0f; channel++) {
		/*
		 * send paramter event for all notes off for redundancy
		 * some older synths don't support this
		 */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, channel | 0xb0);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x7b);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);

		/* modulation controller to zero */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x01);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);

		/* reset all controllers */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x79);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);

		/* lift sustain pedal */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x40);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);

		/* center pitch wheel */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0xe0 | channel);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x40);
	}
	softc->noteonrs = 0;
}

/*
 * An MTC event looks like 0xF1 0xNN.  Here code = 0xNN,
 * and this function interprets the data 0xNN as a quarter
 * frame, updating the partial frame structure with incoming
 * frame data.  We also update the current frame structure
 * with fractional frame data, and keep an eye out for loss
 * of tape sync via kernel timer.  See the MTC spec for info
 * on interpreting the way a SMPTE frame is encoded into 8
 * quarter frames of MTC.
 */
void
midi_parse_MTC_event(softc, code)
	struct midi_softc *softc;
	u_char code;
{
	struct SMPTE_frame *cf, *pf, tmp_frame;
	struct event *e;
	u_long offset;

	pf = &softc->SMPTE_partial;
	cf = &softc->SMPTE_current;

	switch (code & 0xf0) {
	case 0x00:
		/* Frame count LS nibble -- first quarter frame */
		pf->status |= SMPTE_FIRST0xF1;
		pf->frame = code & 0x0f;
		cf->fraction = 25;
		break;
	case 0x10:
		/* Frame count MS nibble */
		pf->frame |= ((code & 0x01) << 4);
		cf->fraction = 50;
		/* Reschedule our syncdrop every SMPTE_DROPOUT frames */
		if ((pf->frame % SMPTE_DROPCHK) == 0) {
			if (softc->SMPTE_current.status & SMPTE_SYNC)
				del_timer(&softc->timer);
			softc->timer.expires = SMPTE_DROPOUT;
			softc->timer.data = (unsigned long)softc;
			softc->timer.function = midi_dropped_SMPTE_sync;
			add_timer(&softc->timer);
 		}
	    break;
	case 0x20:
		/* second LS nibble */
		pf->second = code & 0x0f;
		cf->fraction = 75;
		break;
	case 0x30:
		/* second MS nibble */
		pf->second |= ((code & 0x03) << 4);
		cf->fraction = 0;
		cf->frame++;
		/* 25-frame has this problem */
		if ((cf->rate == 25) && (cf->frame == 25)) {
			cf->frame &= 0xe0;
			if (++cf->second == 60) {
				cf->second &= 0xc0;
				if (++cf->minute == 60) {
					cf->minute &= 0xc0;
					if (++cf->hour == 24)
						cf->hour &= 0xe0;
				}
			}
		}
		break;
	case 0x40:
		/* minute LS nibble */
		pf->minute = code & 0x0f;
		cf->fraction = 25;
		break;
	case 0x50:
		/* minute MS nibble */
		pf->minute |= ((code & 0x03) << 4);
		cf->fraction = 50;
		break;
	case 0x60:
		/* hour LS nibble */
		pf->hour = code & 0x0f;
		cf->fraction = 75;
		break;
	case 0x70:
		/* hour MS nibble */
		pf->hour |= ((code & 0x01) << 4);
		/*
		 * correct this two-frame bullshit here, once.
		 * that way all timing we have to deal with is
		 * clean, with no offsets at all, except for 
		 * right here as we pull it off the tape.
		 */
		pf->frame += 2;
		if (pf->frame >= pf->rate) {
			pf->second++;
			pf->frame -= pf->rate;
			if (pf->second == 60) {
				pf->minute++;
				pf->second = 0;
				if (pf->minute == 60) {
					pf->hour++;
					pf->minute = 0;
					if (pf->hour == 24)
						pf->hour = 0;
				}
			}
		}

		/* 
		 * This is the last of 8 quarter-frame MTC messages used
		 * to encode one SMPTE frame.  If we weren't sync'd before,
		 * we are now (assuming we caught all 8 quarter frames..)
		 */

		/* If we're not currently sync'd */
		if (!(cf->status & SMPTE_SYNC)) {
			/* and we have read all 8 frames */
			if (pf->status & SMPTE_FIRST0xF1)
				midi_register_SMPTE_sync(softc, code);
		}
		*cf = *pf;
		cf->fraction = 0;
		break;
	default:
		printk("Unknown MTC state\n");
		break;
	}

	/*
	 * check next event in queue.  If it is a MetaSMPTE event,
	 * don't call midi_timeout until the time in the event data
	 * matches the current time.  Actually check all events in
	 * queue that will go out together (those with zero delta)..
	 */
	offset = 0;
	do {
		if (!midi_browseq(softc->wqueue, offset++, &e)) {
			/* we didn't get a `next' event */
			if (offset == 1)
				/* bail out if there are *no* events */
				return;		
			else
				/* 
				 * output those that are here 
				 * they aren't SMPTE
				 */
				break;			
		}
		/* we did get an event -- check for SMPTE */
		if (e->type == SMPTE) {
			midi_midi2SMPTE(e, &tmp_frame);

			/*
			 * Don't call midi_timeout until the two times match.
			 * Maybe we should allow for cf >= &tmp_frame?? XXX
			 */
			if (midi_compare_SMPTE_time(cf, &tmp_frame) == -1) 
				return;

			/* 
			 * We have arrived -- set prev_outgoing to match the
			 * MetaSMPTE event time (not necessarily extclock..)
			 */
/*
			softc->prev_outgoing = midi_get_timing_clock(softc);
*/
			softc->prev_outgoing = midi_SMPTE2timing(&tmp_frame);
			break;
		}
	} while (e->time == 0);

	/* try to send some events */
	/*
	 * since we're only sending events when sync is enabled,
	 * and SMPTE_SYNC only gets set after we have a complete
	 * frame, we won't be able to play anything until after
	 * at least two frames have passed.  That's OK, though, 
	 * since we never know what time it really is until we
	 * have read two full frames ~= 1/15 of a second.
	 */

	if (cf->status & SMPTE_SYNC)
		midi_timeout((unsigned long)softc);
}

/*
 * SMPTE sync was just detected.. set everything up 
 */
void
midi_register_SMPTE_sync(softc, code)
	struct midi_softc *softc;
	u_char code;
{
	struct event *event;
	struct SMPTE_frame *cf, *pf;

	pf = &softc->SMPTE_partial;
	cf = &softc->SMPTE_current;

	/* indicate synchronization */
	softc->SMPTE_current.status |= SMPTE_SYNC;
	softc->SMPTE_partial.status |= SMPTE_SYNC;

	/* set up timer to check for (future) loss of sync */
	if (!(softc->status & MIDI_FIRST_SMPTE))
		del_timer(&softc->timer);
	softc->timer.expires = SMPTE_DROPOUT;
	softc->timer.data = (unsigned long)softc;
	softc->timer.function = midi_dropped_SMPTE_sync;
	add_timer(&softc->timer);

	/* get frame rate from upper bits of code */
	midi_SMPTE_framerate(pf, (code & 0x0e) >> 1);

	/*
	 * SMPTE_ORIGIN is no longer needed.  Instead we rely on a
	 * MetaSMPTE event to align SMPTE time with MIDI time.  Maybe
	 * we should think about default behavior in case there is
	 * no such event.. XXX
	 */

	/* set external clock to match the received frame */
	softc->extclock = midi_SMPTE2timing(pf);

	if (softc->status & MIDI_FIRST_SMPTE) {
		softc->status &= ~MIDI_FIRST_SMPTE;

		/* set external clock resolution based on frame rate */
		softc->hz = 4 * pf->rate;

		/* We have to play from the beginning first time around */
		softc->prev_outgoing = 0;
		softc->write_smf_time = midi_get_smf_time(softc);
	} else {
		/* check clock resolution */
		if (pf->rate != cf->rate) 
			printk("SMPTE frame rate has changed! All bets off.\n");

		/*
		 * at this point, we've received sync already, then
		 * lost it, and then regained it.  If the new time is
		 * greater than the old time, we need to scan the MIDI
		 * file for tempo changes.  If it's older, assume we
		 * already have them and do a time-warp instead.
		 */
		
		/* jump ahead */
		if (midi_compare_SMPTE_time(cf, pf) != -1) {
			/* these are no longer reliable */
			softc->premainder = 0;
			softc->rremainder = 0;

			softc->prev_outgoing = softc->extclock;
			softc->write_smf_time = midi_get_smf_time(softc);
			midi_time_warp(softc);

			while (midi_deq(softc->wqueue, &event))
				stynamic_release(&event->data);
			/* XXX should we dequeue rqueue too? */
			while (midi_deq(softc->rqueue, &event))
				stynamic_release(&event->data);
			softc->status |= MIDI_RD_BLOCK;
			/* remove any events already being timed */
			softc->status &= ~MIDI_WR_BLOCK;
			if (softc->status & MIDI_WR_SLEEP) {
				softc->status &= ~MIDI_WR_SLEEP;
				softc->status |= MIDI_WR_ABORT;
				wake_up_interruptible(&softc->write_waitq);
			}
		}
	}
#ifdef GJWDEBUG
	printk("SMPTE sync achieved at %2d:%2d:%2d:%2d\n",
	    pf->hour, pf->minute, pf->second, pf->frame);
	printk("Format is %u-frame", pf->rate);
	if (!(pf->status & SMPTE_FRAMEDROP))
		printk(" no-");
	printk("drop.\n");
#endif
}

/*
 * Convert a SMPTE frame into timing ticks.  We get 4 
 * quarter-frames per frame, and we get smpte->rate frames
 * per second.  (Frame-drop formats are handled in the 
 * hardware.. not my job ;-)  Two subtle points:
 *
 *   1) All SMPTE data is always two frames old, but the
 *      offset is corrected in parse_MTC_event -- as soon
 *      as it comes off the tape.  That's the only place
 *      you need to worry about it.
 *   2) Internal timing is proportional to SMPTE timing.
 *      Specifically SMPTE 00:00:00:00 is 0 ticks.
 */
u_long
midi_SMPTE2timing(smpte)
	struct SMPTE_frame *smpte;
{
	u_long ticks;
	u_long qrate = (u_long)(4 * smpte->rate);

	ticks = smpte->fraction / 25 + smpte->frame * 4 +
	    smpte->second * qrate + 
	    smpte->minute * 60 * qrate + 
	    smpte->hour * 60 * 60 * qrate;
	return (ticks);
}

int 
midi_midi2SMPTE(event, tf)
	struct event *event;
	struct SMPTE_frame *tf;
{
	u_char code;

	if (event->type != SMPTE) {
		printk(
		    "SMPTE error: midi2SMPTE called with non-SMPTE event.\n");
		return (-1);
	}

	tf->status   = 0;
	tf->hour     = stynamic_get_byte(&event->data, 0) & 0x1f;
	tf->minute   = stynamic_get_byte(&event->data, 1) & 0x3f;
	tf->second   = stynamic_get_byte(&event->data, 2) & 0x3f;
	tf->frame    = stynamic_get_byte(&event->data, 3) & 0x1f;
	tf->fraction = stynamic_get_byte(&event->data, 4);

	code = (stynamic_get_byte(&event->data, 0) & 0xe0) >> 5;
	if (midi_SMPTE_framerate(tf, code) == -1) {
    		printk(
		    "SMPTE error: frame rate is not encoded in hour bits.\n");
    		printk("Assuming 30-frame no-drop.\n");
    		tf->rate = 30;
		tf->status &= ~SMPTE_FRAMEDROP;
    		return (-1);
    	}
	return (0);
}

int
midi_SMPTE_framerate(tf, code)
	struct SMPTE_frame *tf;
	u_char code;
{

	/* get frame rate from code */
	switch (code) {
	case 0x00:	/* 24 frame */
		tf->rate = 24;
		tf->status &= ~SMPTE_FRAMEDROP;
		break;
	case 0x01:	/* 25 frame */
		tf->rate = 25;
		tf->status &= ~SMPTE_FRAMEDROP;
		break;
	case 0x02:	/* 30 frame drop */
		tf->rate = 30;
		tf->status |= SMPTE_FRAMEDROP;
		break;
	case 0x04:	/* 30 frame non-drop */
		tf->rate = 30;
		tf->status &= ~SMPTE_FRAMEDROP;
		break;
	default:
		return (-1);
	}
	return 0;
}

/*
 * return -1 if t1 < t2
 *         0 if t1 == t2
 *         1 if t1 > t2
 */
int
midi_compare_SMPTE_time(t1, t2)
	struct SMPTE_frame *t1;
	struct SMPTE_frame *t2;
{

	if (t1->hour < t2->hour)
		return (-1);
	else if (t2->hour > t2->hour)
		return (1);
	if (t1->minute < t2->minute)
		return (-1);
	else if (t1->minute > t2->minute)
		return (1);
	if (t1->second < t2->second)
		return (-1);
	else if (t1->second > t2->second)
		return (1);
	if (t1->frame < t2->frame)
		return (-1);
	else if (t1->frame > t2->frame)
		return (1);
	if (t1->fraction < t2->fraction)
		return (-1);
	else if (t1->fraction > t2->fraction)
		return (1);
	return (0);
}

/*
 * checks to see if we should send a signal and sends it if we
 * should.
 */
void
midi_send_sigio(softc)
	struct midi_softc *softc;
{

	if (softc->wqueue->count < MIDI_LOW_WATER) {
		softc->status &= ~MIDI_WR_BLOCK;
		if (softc->status & MIDI_WR_SLEEP) {
			softc->status &= ~MIDI_WR_SLEEP;
			wake_up_interruptible(&softc->write_waitq);
		}
		if (softc->status & MIDI_ASYNC &&
		    (softc->status & MIDI_SENDIO || softc->wqueue->count
		    == 0)) {
			if (softc->pgid < 0)
				kill_pg(-softc->pgid, SIGIO, 1);
			else
				kill_proc(softc->pgid, SIGIO, 1);
		}
		softc->status &= ~MIDI_SENDIO;
		/* notify select that writes will succeed */
		if (softc->status & MIDI_SELOUT) {
			softc->status &= ~MIDI_SELOUT;
			wake_up_interruptible(&softc->selout_waitq);
		}
	}
	if (softc->status & MIDI_FLUSH_SLEEP && softc->wqueue->count == 0) {
		softc->status &= ~MIDI_FLUSH_SLEEP;
		wake_up_interruptible(&softc->flush_waitq);
	}
}

void
midi_time_warp(softc)
	struct midi_softc *softc;
{

	/* turn off any notes that may be on */
	midi_reset_devices(softc);

	if (softc->status & MIDI_ASYNC) {
		/*
		 * Ugh!  Linux, for some unknown reason, defines SIGURG
		 * to be the same as SIGIO, thus we can't use both to
		 * mean different things.  So in this port of the driver,
		 * we use SIGUSR1 inplace of SIGURG.  It's ugly and I
		 * hate it, but it works.  Perhaps Linux developers will
		 * notice their oversight (?) and fix the SIGURG problem
		 * so we can use it.
		 */
		if (softc->pgid < 0)
			kill_pg(-softc->pgid, SIGUSR1, 1);
		else
			kill_proc(softc->pgid, SIGUSR1, 1);
	}
	if (softc->status & MIDI_SELEX) {
		softc->status &= ~MIDI_SELEX;
		wake_up_interruptible(&softc->selex_waitq);
	}
}

u_long
midi_get_timing_clock(softc)
	struct midi_softc *softc;
{
	u_long t;

	if (softc->status & MIDI_EXTCLK)
		t = softc->extclock;
	else
		t = jiffies - softc->start_time;
	return (t);
}

u_long
midi_get_smf_time(softc)
	struct midi_softc *softc;
{
 	long long numerator, denominator;
	u_long smf, tc, tdiff, tempo;
	struct tempo_change *prev, *tchange;

	/* 
	 * SMF time zero corresponds to SMPTE time 00:00:00:00 ??
	 * Not quite.  The way tclmidi seems to be working, we
	 * should do the following:
	 *
	 *   1) if extclock is less than the time corresponding to
	 *      the MetaSMPTE event, we should return SMF 0.
	 *   2) if extclock is past the time corresponding to the
	 *      MetaSMPTE event, we should return the SMF time 
	 *      represented by extclock - MetaSMPTE extclock.
	 */
	tc = midi_get_timing_clock(softc);

	/* find last tempo change */
	for (tchange = softc->tempo_changes, prev = NULL;
	    tchange != NULL;
	    prev = tchange, tchange = tchange->next) {
		if (tc <= tchange->time)
			break;
	}
	if (prev == NULL) {
		/* 
		 * no tempos found -- default tempo for MIDI files is
		 * 120 bpm.
		 */
		tempo = 500000;
		tdiff = 0;
		smf = 0;
	} else {
		tempo = prev->tempo;
		tdiff = tc - prev->time;
		smf = prev->smf_time;
	}
  
	numerator = (long long)softc->division * 1000000 * tdiff;
 	denominator = (long long)tempo * softc->hz;
 	smf += numerator / denominator;
#ifdef GJWDEBUG
	printk("midi_get_smf_time() returning %lu\n", smf);
#endif
  	return (smf); 
}

void
midi_initq(eq)
	struct event_queue *eq;
{

	eq->count = 0;
	eq->end = &eq->events[MIDI_Q_SIZE - 1];
	eq->head = eq->events;
	eq->tail = eq->events;
	/* zero events to clear stynamic stuctures */
	memset(eq->events, 0, MIDI_Q_SIZE * sizeof(struct event));
}

void
midi_copy_event(e1, e2)
	struct event *e1, *e2;
{

	e1->time = e2->time;
	e1->smf_time = e2->smf_time;
	e1->type = e2->type;
	e1->tempo = e2->tempo;
	stynamic_release(&e1->data);
	stynamic_append(&e1->data, &e2->data);
}

int
midi_deq(eq, event)
	struct event_queue *eq;
	struct event **event;
{

	cli();
	if (eq->count == 0) {
		sti();
		return (0);
	}
	if (event == NULL)
		eq->head++;
	else
		*event = eq->head++;
	if (eq->head > eq->end)
		eq->head = eq->events;
	eq->count--;
	sti();
	return (1);
}

int
midi_peekq(eq, event)
	struct event_queue *eq;
	struct event **event;
{

	cli();
	if (eq->count == 0) {
		sti();
		return (0);
	}
	*event = eq->head;
	sti();
	return (1);
}

int
midi_browseq(eq, offset, event)
	struct event_queue *eq;
	u_long offset;
	struct event **event;
{

	cli();
	if (eq->count <= offset) {
		sti();
		return (0);
	}
	*event = (eq->head + offset);
	sti();
	return (1);
}

int
midi_enq(eq, event)
	struct event_queue *eq;
	struct event *event;
{

	cli();
	if (eq->count == MIDI_Q_SIZE) {
		sti();
		return (0);
	}
	midi_copy_event(eq->tail++, event);
	if (eq->tail > eq->end)
		eq->tail = eq->events;
	eq->count++;
	sti();
	return (1);
}

void
stynamic_add_byte(sty, byte)
	struct stynamic *sty;
	u_char byte;
{
	u_char *new_ptr;

	cli();
	if (sty->len < STYNAMIC_SIZE)
		sty->datas[sty->len++] = byte;
	else {
		if (sty->len < sty->allocated) {
			sty->datad[sty->len++] = byte;
		} else {
			new_ptr = kmalloc(sty->allocated + STYNAMIC_ALLOC,
			    GFP_ATOMIC);
			if (new_ptr == NULL) {
				printk("midi: out of memory all hell"
				   " will break loose\n");
				return;
			}
			if (sty->allocated == 0)
				memcpy(new_ptr, sty->datas, sty->len);
			else {
				memcpy(new_ptr, sty->datad, sty->len);
				kfree_s(sty->datad, sty->allocated);
			}
			sty->datad = new_ptr;
			sty->datad[sty->len++] = byte;
			sty->allocated += STYNAMIC_ALLOC;
		}
	}
	sti();
}

void
stynamic_add_bytes(sty, bytes, num)
	struct stynamic *sty;
	u_char *bytes;
	int num;
{
	int size_inc;
	u_char *new_ptr;

	cli();
	if (sty->len + num <= STYNAMIC_SIZE) {
		memcpy(&sty->datas[sty->len], bytes, num);
		sty->len += num;
	} else {
		if (sty->len + num <= sty->allocated) {
			memcpy(&sty->datad[sty->len], bytes, num);
			sty->len += num;
		} else {
			size_inc = (num / STYNAMIC_ALLOC + 1) * STYNAMIC_ALLOC;
			new_ptr = kmalloc(sty->allocated + size_inc,
			    GFP_ATOMIC);
			if (new_ptr == NULL) {
				printk("midi: out of memory all hell"
				   " will break loose\n");
				return;
			}
			if (sty->allocated == 0)
				memcpy(new_ptr, sty->datas, sty->len);
			else {
				memcpy(new_ptr, sty->datad, sty->len);
				kfree_s(sty->datad, sty->allocated);
			}
			sty->datad = new_ptr;
			memcpy(&sty->datad[sty->len], bytes, num);
			sty->allocated += size_inc;
			sty->len += num;
		}
	}
	sti();
}

u_char
stynamic_get_byte(sty, index)
	struct stynamic *sty;
	int index;
{

	cli();
	if (sty->len <= 4)
		return (sty->datas[index]);
	else
		return (sty->datad[index]);
	sti();
}

void
stynamic_copy(sty, dest, len)
	struct stynamic *sty;
	void *dest;
	int len;
{

	cli();
	if (sty->len <= 4)
		memcpy(dest, sty->datas, len);
	else
		memcpy(dest, sty->datad, len);
	sti();
}

void
stynamic_append(dest, src)
	struct stynamic *dest;
	struct stynamic *src;
{

	if (src->len <= 4)
		stynamic_add_bytes(dest, src->datas, src->len);
	else
		stynamic_add_bytes(dest, src->datad, src->len);
}

#if 0
/* unused function */
void
stynamic_copy_from(sty, index, dest, len)
	struct stynamic *sty;
	int index, len;
	void *dest;
{

	cli();
	if (sty->len <= 4)
		memcpy(dest, &sty->datas[index], len);
	else
		memcpy(dest, &sty->datad[index], len);
	sti();
}
#endif


void
stynamic_release(sty)
	struct stynamic *sty;
{

	cli();
	if (sty->len > STYNAMIC_SIZE)
		kfree_s(sty->datad, sty->allocated);
	sty->len = 0;
	sty->allocated = 0;
	sty->datad = NULL;
	memset(sty->datas, 0, STYNAMIC_SIZE);
	sti();
}

void
stynamic_shift(sty, num)
	struct stynamic *sty;
	int num;
{
	int rem;
	u_char *ptr;

	if (sty->len <= num) {
		stynamic_release(sty);
		return;
	}
	cli();
	if (sty->len > STYNAMIC_SIZE)
		ptr = &sty->datad[num];
	else
		ptr = &sty->datas[num];
	rem = sty->len - num;
	if (rem > STYNAMIC_SIZE)
		memcpy(sty->datad, ptr, rem);
	else {
		memcpy(sty->datas, ptr, rem);
		if (sty->datad != NULL) {
			kfree_s(sty->datad, sty->allocated);
			sty->datad = NULL;
			sty->allocated = 0;
		}
	}
	sty->len = rem;
	sti();
}

void
stynamic_init(sty)
	struct stynamic *sty;
{

	cli();
	sty->len = 0;
	sty->allocated = 0;
	sty->datad = NULL;
	memset(sty->datas, 0, STYNAMIC_SIZE);
	sti();
}

#if 0
/* unused function */
void
stynamic_print(sty)
	struct stynamic *sty;
{

	printk("\t\tlen = %d\n", sty->len);
	printk("\t\tallocated = %d\n", sty->allocated);
	printk("\t\tdatad = 0x%x\n", (int)(sty->datad));
}
#endif
