/*-
 * Copyright (c) 1993, 1994 Michael B. Durian.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Michael B. Durian.
 * 4. The name of the the Author may be used to endorse or promote 
 *    products derived from this software without specific prior written 
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#ifndef MIDI_H
#define MIDI_H
/*
 * The SMPTE data structure
 */
struct SMPTE_frame {
	unsigned	char rate;
	unsigned	char hour;
	unsigned	char minute;
	unsigned	char second;
	unsigned	char frame;
	unsigned	char fraction;
	unsigned	char status;
};

/*
 * SMPTE status bits
 */
#define SMPTE_SYNC	(1 << 0)	/* indicates we are sync'd */
#define SMPTE_FRAMEDROP	(1 << 1)	/* this is a frame-drop format */
#define SMPTE_FIRST0xF1	(1 << 2)	/* we caught the first 1/4 frame */

/*
 * MIDI MPU401 ioctls (/dev/midi and /dev/smpte)
 */
#define MRESET		_IO('m', 0x01)
#define MDRAIN		_IO('m', 0x02)
#define MFLUSH		_IO('m', 0x03)
#define MGPLAYQ		_IOR('m', 0x04, int)
#define MGRECQ		_IOR('m', 0x05, int)
#define MSDIVISION	_IOW('m', 0x06, int)
#define MGDIVISION	_IOR('m', 0x07, int)
#define MGQAVAIL	_IOR('m', 0x08, int)
#define MASYNC		_IOW('m', 0x09, int)
#define MTHRU		_IOW('m', 0x0a, int)
#define MRECONPLAY	_IOW('m', 0x0b, int)
#define MGSMPTE		_IOR('m', 0x0c, struct SMPTE_frame)
#define MGSMFTIME	_IOR('m', 0x0d, unsigned long)

#ifdef __KERNEL__
/*
 * midi port offsets
 */
#define	MIDI_DATA			0
#define	MIDI_STATUS			1
#define	MIDI_COMMAND			1


/*
 * midi data transfer status bits
 */
#define	MIDI_RDY_RCV			(1 << 6)
#define	MIDI_DATA_AVL			(1 << 7)

/*
 * midi status flags
 */
#define MIDI_UNAVAILABLE	(1 << 0) /* not in uart mode */
#define MIDI_READING		(1 << 1) /* open for reading */
#define MIDI_WRITING		(1 << 2) /* open for writing */
#define MIDI_CALLBACK_ISSUED	(1 << 3) /* waiting for a timer to expire */
#define MIDI_RD_BLOCK		(1 << 4) /* read will block */
#define MIDI_WR_BLOCK		(1 << 5) /* write will block */
#define MIDI_WR_ABORT		(1 << 6) /* write should abort */
#define MIDI_OPEN		(1 << 7) /* device is open */
#define MIDI_FLUSH_SLEEP	(1 << 8) /* blocking on flush */
#define MIDI_RD_SLEEP		(1 << 9) /* blocking on read */
#define MIDI_WR_SLEEP		(1 << 10) /* blocking on write */
#define MIDI_BUFFER_SLEEP	(1 << 11) /* blocking on buffer */
#define MIDI_NEED_ACK		(1 << 12) /* command needs and ack */
#define MIDI_ASYNC		(1 << 13) /* send sigios */
#define MIDI_SENDIO		(1 << 14) /* a sigio should be send at low h2o */
#define MIDI_NONBLOCK		(1 << 15) /* non blocking mode */
#define MIDI_SELOUT		(1 << 16) /* wait for a select output */
#define MIDI_SELIN		(1 << 17) /* wait for a select input */
#define MIDI_SELEX		(1 << 18) /* wait for a select exceptional */
#define MIDI_THRU		(1 << 19) /* copy in port to out port? */
#define MIDI_RECONPLAY		(1 << 20) /* start record timer on play */
#define MIDI_EXTCLK		(1 << 21) /* use external clock */
#define MIDI_NEED_DATA		(1 << 22) /* need to read command response */
#define MIDI_FIRST_WRITE	(1 << 23) /* the next write will be the first */
#define MIDI_FIRST_SMPTE	(1 << 24) /* set if no SMPTE yet */

/*
 * These are the various input data states
 */
typedef enum {START, NEEDDATA1, NEEDDATA2, SYSEX, SYSTEM1, SYSTEM2, MTC}
    InputState;

/*
 * midi command values
 */
#define MIDI_RESET			0xff
#define MIDI_UART			0x3f
#define MIDI_ACK			0xfe
#define MIDI_VERSION			0xac
#define MIDI_REVISION			0xad
#define MIDI_TRIES			200000

/*
 * SMPTE configuration
 */
#define SMPTE_DROPCHK 10	/* check for loss of sync every 10 ticks */
#define SMPTE_DROPOUT 50	/* 50 ticks without 0xf1 ==> loss of sync */

/*
 * most events are short, so we use the static array,
 * but occationally we get a long event (sysex) and
 * we dynamically allocate that
 */
#define STYNAMIC_SIZE 4
#define STYNAMIC_ALLOC 256

struct stynamic {
	short	allocated;
	short	len;
	u_char	datas[4];
	u_char	*datad;
};

/*
 * data from the board that hasn't formed a complete event yet
 */
struct partial_event {
	struct	stynamic event;		/* the data */
	u_long	time;			/* event time */
	long	tempo;			/* tempo setting when event arrived */
	InputState	state;		/* what we are expecting next */
	u_char	rs;			/* the midi running state */
};

/*
 * keep a list of tempo changes
 */
struct tempo_change {
	u_long	time;		/* time tempo change occured in timing ticks */
	u_long	smf_time;	/* same but in SMF ticks */
	long	tempo;		/* the new tempo */
	struct	tempo_change *next;
};

/*
 * the internal representation of an event
 */
typedef enum {NORMAL, TEMPO, TIMESIG, SYSX, SMPTE, NOP} EventType;

struct event {
	u_long	time;		/* time until event in kernel clicks */
	u_long	smf_time;	/*
				 * absolute time of the event in smf ticks
				 * only used in writing
				 */
        long    tempo;		/*
				 * not used in play events, but contains
				 * the tempo setting current when the
				 * incoming event arrived.  Used for
				 * converting timing ticks to smf ticks
				 */
	EventType	type;
	struct	stynamic data;
};

/*
 * A event queue, used for both incoming and outgoing
 */
#define MIDI_Q_SIZE 150
#define MIDI_LOW_WATER 40

struct event_queue {
	int	count;
	struct	event events[MIDI_Q_SIZE];
	struct	event *end;
	struct	event *head;
	struct	event *tail;
};

/*
 * slop time
 * If we've fallen more than BACKLOG_TIME ticks behind, don't play
 * the events.  This is so we can "fastforward" through a song -
 * picking up important events like tempo changes, but not actually
 * playing any note on events.
 */
#define BACKLOG_TIME 5

/*
 * different MPU401 features
 */
#define SMPTE_EQUIP	(1 << 0)	/* equipped with MQ SMPTE */

struct midi_softc {      /* Driver status information */
	struct	wait_queue *flush_waitq;
	struct	wait_queue *read_waitq;
	struct	wait_queue *write_waitq;
	struct	wait_queue *selin_waitq;
	struct	wait_queue *selout_waitq;
	struct	wait_queue *selex_waitq;
	/* who's running us */
	struct	task_struct *owner;
	/* midi specific stuff */
	struct	event_queue *rqueue;
	struct	event_queue *wqueue;
	struct	stynamic rpartial;
	struct	stynamic wpartial;
	struct	partial_event partial_event;
	struct	SMPTE_frame SMPTE_current;
	struct	SMPTE_frame SMPTE_partial;
	struct	timer_list timer;
	struct	tempo_change *tempo_changes;
	volatile	long status;
	long long	premainder;
	long long	rremainder;
	u_long	extclock;
	u_long	start_time;
	u_long	prev_incoming;
	u_long	prev_outgoing;
	u_long	write_smf_time;		/* ab smf time of last event in write */
	long	features;
 	long	ptempo;			/* usec / beat */
 	long	rtempo;			/* usec / beat */
	long	prev_rtempo;
	int	hz;			/* timing clock rate, int or ext */
	int	pgid;
	int	addr;
	int	intr;
	int	division;
	int	wpartialpos;
	short	noteon[0x80];		/*
					 * each element is a different pitch
					 * each bit is a different channel
					 */
	u_char	readrs;
	u_char	writers;
	u_char	noteonrs;
};

struct midi_config {
	int	addr;
	int	intr;
};

/* for mem.c */
extern long midiinit(long);

#endif
#endif
