/*-
 * Copyright (c) 1993, 1994 Michael B. Durian.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Michael B. Durian.
 * 4. The name of the the Author may be used to endorse or promote 
 *    products derived from this software without specific prior written 
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#define INKERNEL 1
#define _KERNEL 1

#include <sys/sysmacros.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/file.h>
#include <sys/dir.h>
#include <sys/signal.h>
#include <sys/user.h>
#include <sys/systm.h>
#include <sys/errno.h>
#include <sys/open.h>
#include <sys/uio.h>
#include <sys/cred.h>
#include <sys/ddi.h>
#include <sys/poll.h>
#include <sys/kmem.h>
#include <sys/filio.h>
#include <sys/termios.h>

#include "midi.h"

extern struct midi_softc midi_sc[];
extern int midi_addrs[];
extern int midi_intrs[];
extern int NumMidi;

#if defined(__STDC__) || defined(__cplusplus)
#define __P(protos)	protos
#else
#define __P(protos)	()
#endif

/* I've been burned too many times by different byte ordering */
#define OUTB(x, y) outb((x), (y))
#define INB(x) inb((x))

void midiinit __P(());
int midiopen __P((dev_t *dev, int flag, int otyp, cred_t *cred_p));
int midiclose __P((dev_t dev, int flag, int otyp, cred_t *cred_p));
int midiread __P((dev_t dev, uio_t *uio_p, cred_t *cred_p));
int midiwrite __P((dev_t dev, uio_t *uio_p, cred_t *cred_p));
int midiioctl __P((dev_t dev, int cmd, int arg, int mode, cred_t *cred_p,
    int *rval_p));
int midipoll __P((dev_t dev, int /* short */ events, int anyyet,
    short *reventsp, struct pollhead **phpp));
int midiintr __P((int ivec));

static void midi_copy_event __P((struct event *, struct event *));
static void midi_initq __P((struct event_queue *));
static int midi_deq __P((struct event_queue *, struct event **));
static int midi_peekq __P((struct event_queue *, struct event **));
static int midi_browseq __P((struct event_queue *, u_long, struct event **));
static int midi_enq __P((struct event_queue *, struct event *));
static int midi_reset __P((struct midi_softc *));
static int midi_wait_rdy_rcv __P((struct midi_softc *softc));
static int midi_send_command __P((struct midi_softc *softc,
    /* u_char */ int comm));
static int midi_send_command_with_response __P((struct midi_softc *softc,
    /* u_char */ int comm, void *response, int resp_size));
static int midi_uart __P((struct midi_softc *));
static int midi_enter_smpte_mode __P((struct midi_softc *));
static int midi_event2smf __P((struct midi_softc *, struct event *,
    struct stynamic *));
static int midi_next_byte __P((struct midi_softc *, struct uio *));
static int midi_uio2event __P((struct midi_softc *, struct uio *,
    struct event *));
static int midi_fix2var __P((u_long, u_char *));
static int midi_var2fix __P((u_char *, u_long *));
static u_long midi_smf2timing_tick __P((struct midi_softc *, long));
static u_long midi_timing2smf_tick __P((struct midi_softc *, long, long));
static int midi_midi2SMPTE __P((struct event *, struct SMPTE_frame *));
static int midi_SMPTE_framerate __P((struct SMPTE_frame *, int /* u_char */));
static void midi_timeout __P((caddr_t arg));
static void midi_add_complete_event __P((struct midi_softc *));
static void midi_schedule_timeout __P((struct midi_softc *, u_long));
static void midi_write_event __P((struct midi_softc *, struct event *, int));
static void midi_reset_devices __P((struct midi_softc *));
static void midi_parse_MTC_event __P((struct midi_softc *, int /* u_char */));
static void midi_register_SMPTE_sync __P((struct midi_softc *,
    int /* u_char */));
static void midi_dropped_SMPTE_sync __P((caddr_t arg));
static u_long midi_SMPTE2timing __P((struct SMPTE_frame *));
static int midi_compare_SMPTE_time __P((struct SMPTE_frame *t1,
    struct SMPTE_Frame *t2));
static void midi_send_sigio __P((struct midi_softc *softc));
static void midi_time_warp __P((struct midi_softc *softc));
static u_long midi_get_timing_clock __P((struct midi_softc *));
static u_long midi_get_smf_time __P((struct midi_softc *softc));
static u_long midi_get_timing_clock __P((struct midi_softc *));
static void stynamic_add_byte __P((struct stynamic *, int /* u_char */));
static void stynamic_add_bytes __P((struct stynamic *, u_char *, int));
static u_char stynamic_get_byte __P((struct stynamic *, int));
static void stynamic_copy __P((struct stynamic *, void *, int));
static void stynamic_append __P((struct stynamic *, struct stynamic *));
static void stynamic_release __P((struct stynamic *));
static void stynamic_shift __P((struct stynamic *, int));
static void stynamic_init __P((struct stynamic *));

/* unused functions
static void stynamic_copy_from __P((struct stynamic *, int, void *, int));
static void stynamic_print __P((struct stynamic *));
*/

void
midiinit()
{
	struct midi_softc *softc;
	int i;
	u_char rev;

	for (i = 0; i < NumMidi; i++) {
		softc = &midi_sc[i];
		softc->addr = midi_addrs[i];
		softc->intr = midi_intrs[i];
		softc->status = MIDI_RD_BLOCK;
		rev = 0x01;
		if (!midi_reset(softc)) {
			if (!midi_reset(softc)) {
				printf("Couldn't reset MPU401\n");
				softc->status |= MIDI_UNAVAILABLE;
				continue;
			}
		}
		if (!midi_wait_rdy_rcv(softc)) {
			printf("MPU401 not acknowledging\n");
			softc->status |= MIDI_UNAVAILABLE;
			continue;
		}
		if (!midi_send_command_with_response(softc, MIDI_REVISION,
		    &rev, 1))
			printf("Couldn't get MPU401 revision\n");
		if (rev & 0x04)
			softc->features |= SMPTE_EQUIP;
		else
			softc->features &= ~SMPTE_EQUIP;
		if (!midi_uart(softc)) {
			printf("Couldn't put MPU401 into UART mode!\n");
			softc->status |= MIDI_UNAVAILABLE;
			continue;
		}

		/* allocate memory for event queues */
		if ((softc->rqueue = kmem_alloc(sizeof(struct event_queue),
		    KM_NOSLEEP)) == NULL) {
			printf("No memory for rqueue\n");
			softc->status |= MIDI_UNAVAILABLE;
			continue;
		}
		if ((softc->wqueue = kmem_alloc(sizeof(struct event_queue),
		    KM_NOSLEEP)) == NULL) {
			printf("No memory for wqueue\n");
			softc->status |= MIDI_UNAVAILABLE;
			kmem_free(softc->rqueue, sizeof(struct event_queue));
			continue;
		}
		/* zero read/write queue to clear stynamic structures */
		bzero(softc->rqueue, sizeof(struct event_queue));
		bzero(softc->wqueue, sizeof(struct event_queue));
		stynamic_init(&softc->rpartial);
		stynamic_init(&softc->wpartial);
		softc->wpartialpos = 0;
		printf("Found MPU401 #%d at 0x%02x irq %d", i, softc->addr,
		    softc->intr);
		if (softc->features & SMPTE_EQUIP)
			printf(" SMPTE Equipped");
		printf("\n");
	}
}

int
midiopen(dev, flag, otyp, cred_p)
	dev_t *dev;
	int flag, otyp;
	cred_t *cred_p;
{
	register struct midi_softc *softc;
	register int unit;
	register int smpte;
 
	unit = minor(*dev);
	smpte = unit & 0x80;
	unit &= 0x7f;
	if (unit >= NumMidi)
		return (ENXIO);
	softc = &midi_sc[unit];
	if (softc->status & MIDI_UNAVAILABLE)
		return (EIO);
	
	if (softc->status & MIDI_OPEN)
		return (EBUSY);
	else
		softc->status = MIDI_OPEN;

	/*
	 * reads will block until something appears
	 */
	softc->status |= MIDI_RD_BLOCK;

	/* initialize the queues */
	midi_initq(softc->rqueue);
	midi_initq(softc->wqueue);

	softc->partial_event.event.len = 0;
	softc->partial_event.event.datad = NULL;

	/* make sure we are in UART mode */
	if (!midi_fullreset(softc)) {
		printf("Couldn't put MPU401 into UART mode!\n");
		softc->status |= MIDI_UNAVAILABLE;
		kmem_free(softc->rqueue, sizeof(struct event_queue));
		kmem_free(softc->wqueue, sizeof(struct event_queue));
		return (EIO);
	}

	softc->pgid = u.u_procp->p_pid;
	softc->owner = u.u_procp;

	/* are we going to read, write or both? */
	if (flag & FREAD)
		softc->status |= MIDI_READING;
	else
		softc->status &= ~MIDI_READING;
	if (flag & FWRITE)
		softc->status |= MIDI_WRITING;
	else
		softc->status &= ~MIDI_WRITING;
	if (flag & FNONBLOCK)
		softc->status |= MIDI_NONBLOCK;
	else
		softc->status &= ~MIDI_NONBLOCK;

	/* if high bit is set, open device in SMPTE mode */
	if (!smpte)
		softc->status &= ~MIDI_EXTCLK;
	else {
		if (!(softc->features & SMPTE_EQUIP))
			return (ENXIO);
		/* reset again so we can issue commands */
		(void)midi_reset(softc);
		if (!midi_enter_smpte_mode(softc)) {
			printf("Couldn't enter SMPTE mode\n");
			return (EIO);
		}
		if (!midi_uart(softc)) {
			/* GJW maybe figure out which MQ board it is..*/
			printf("Couldn't put MQX-32M into UART mode\n");
			return (EIO);
		}
		softc->status |= MIDI_EXTCLK;
	}

	return (0);
}

int
midiclose(dev, flag, otyp, cred_p)
	dev_t dev;
	int flag, otyp;
	cred_t *cred_p;
{
	register struct midi_softc *softc;
	register int error, unit;

	unit = minor(dev) & 0x7f;
	softc = &midi_sc[unit];

	/*
	 * we're not going to finish closing until everything has
	 * been played.
	 */
	error = 0;
	if (softc->status & MIDI_WRITING) {
		/* same as MFLUSHQ ioctl */
		if (softc->wqueue->count != 0) {
			softc->status |= MIDI_FLUSH_SLEEP;
			do {
				if (error = sleep((caddr_t)&softc->status,
				    PWAIT | PCATCH))
					softc->status &= ~MIDI_FLUSH_SLEEP;
			} while (softc->status & MIDI_FLUSH_SLEEP);
		}
	}

	/* turn off any notes that might be stuck on */
	midi_reset_devices(softc);
	midi_fullreset(softc);

	softc->status &= ~MIDI_OPEN;
	return (error);
}

int
midiread(dev, uio, cred_p)
	dev_t dev;
	uio_t *uio;
	cred_t *cred_p;
{
	struct event *event;
	register struct midi_softc *softc;
	int error, in, num_to_move, unit;

	unit = minor(dev) & 0x7f;
	softc = &midi_sc[unit];

	if (softc->rqueue->count == 0 && softc->rpartial.len == 0) {
		if (softc->status & MIDI_NONBLOCK)
			return (EWOULDBLOCK);
		else {
			softc->status |= MIDI_RD_SLEEP;
			do {
				if (error = sleep((caddr_t)softc->rqueue,
				    PWAIT | PCATCH))
					return (error);
			} while (softc->status & MIDI_RD_SLEEP);
			/* XXX maybe check for abort here */
		}
	}
	while (uio->uio_resid) {
		/*
		 * dequeue an event if partial is empty
		 */
		if (softc->rpartial.len == 0) {
			if (!midi_deq(softc->rqueue, &event)) {
				softc->status |= MIDI_RD_BLOCK;
				return (0);
			}
			midi_event2smf(softc, event, &softc->rpartial);
			styanmic_release(&event->data);
		}
		/* read out as much of rpartial as possible */
		num_to_move = MIN(softc->rpartial.len, uio->uio_resid);
		if (softc->rpartial.len <= STYNAMIC_SIZE) {
			if (uiomove(softc->rpartial.datas, num_to_move,
			    UIO_READ, uio) == -1)
			return (-1);
		} else {
			if (uiomove(softc->rpartial.datad, num_to_move,
			    UIO_READ, uio) == -1)
			return (-1);
		}
		stynamic_shift(&softc->rpartial, num_to_move);
	}
	return (0);
}

int
midiwrite(dev, uio, cred_p)
	dev_t dev;
	uio_t *uio;
	cred_t *cred_p;
{
	register struct midi_softc *softc;
	struct event event, *e;
	int convert, error, unit, iovec, empty_queue;

	unit = minor(dev) & 0x7f;
	softc = &midi_sc[unit];

	if (softc->wqueue->count == 0)
		empty_queue = 1;
	else
		empty_queue = 0;

	/* check to see if we'll block */
	if (softc->status & MIDI_WR_BLOCK) {
		if (softc->status & MIDI_NONBLOCK)
			return (EWOULDBLOCK);
		else {
			softc->status |= MIDI_WR_SLEEP;
			do {
				if (error = sleep((caddr_t)softc->wqueue,
				    PWAIT | PCATCH))
					return (error);
			} while (softc->status & MIDI_WR_SLEEP);
		}
	}
	/* if returns from sleep and should abort because of DRAIN */
	if (softc->status & MIDI_WR_ABORT) {
		softc->status &= ~MIDI_WR_ABORT;
		return (0);
	}

	stynamic_init(&event.data);
	while (uio->uio_resid) {
		/* block if queue is full */
		if (softc->wqueue->count == MIDI_Q_SIZE) {
			softc->status |= MIDI_WR_BLOCK;
			if (empty_queue) {
				if (!midi_peekq(softc->wqueue, &e))
					return (0);
				midi_schedule_timeout(softc, e->time);
				empty_queue = 0;
			}
			if (softc->status & MIDI_NONBLOCK)
				return (0);
			else {
				softc->status |= MIDI_WR_SLEEP;
				do {
					if (error = sleep(softc->wqueue,
					    PWAIT | PCATCH))
						return (error);
				} while (softc->status & MIDI_WR_SLEEP);
			}
		}

		/*
		 * 1) get a complete event off queue
		 * 2) convert it from SMF to board format
		 * 3) deal with it
		 */
		convert = midi_uio2event(softc, uio, &event);
		switch (convert) {
		case 0:
			break;
		case -1:
			/* not a complete event - we'll get it next time */
			if (empty_queue) {
				if (!midi_peekq(softc->wqueue, &e))
					return (0);
				midi_schedule_timeout(softc, e->time);
			}
			return (0);
		default:
			return (convert);
		}

		if (midi_enq(softc->wqueue, &event) == -1)
			return (EIO);
		stynamic_release(&event.data);
		/* set flag so next time we cross LOW water we will SIGIO */
		if (softc->wqueue->count >= MIDI_LOW_WATER)
			softc->status |= MIDI_SENDIO;
	}
	/*
	 * we set the flag if the last write filled the queue, but
	 * we don't need to block
	 */
	if (softc->wqueue->count == MIDI_Q_SIZE)
		softc->status |= MIDI_WR_BLOCK;

	if (empty_queue) {
		if (!midi_peekq(softc->wqueue, &e))
			return (0);
		midi_schedule_timeout(softc, e->time);
	}

	/* XXX Do this the right way! */
	/*
	 * GJW i.e. change midiintr() so we can read 0xf1's but
	 * ignore everything else when in SMPTE play mode.. ??
	 */
	if (softc->status & MIDI_EXTCLK)
		softc->status |= MIDI_READING;

	softc->status |= MIDI_FIRST_WRITE;
	softc->status |= MIDI_FIRST_SMPTE;
	return (0);
}

/*
 * XXXINTR
 * I don't know what ivec is.  I'm really hoping it is something
 * useful like the unit number.
 * If it is the interrupt number, then we can loop through all the
 * midi_sc comparing ivec against softc->intr until we find the right
 * one.  No biggie, I just don't know which way it works.
 * For for I'll assume by default that it is the interrupt number.
 */
int
midiintr(ivec)
	int ivec;
{
	register struct midi_softc *softc;
	register u_char code;
	register struct partial_event *pe;

#if 1
	int i;

	for (i = 0; i < NumMidi && midi_sc[i].intr != ivec; i++);
	if (i == NumMidi) {
		printf("midi: ivec(%d) is not a valid MPU401 intr number\n");
		return;
	}
	softc = &midi_sc[i];
#else
	softc = &midi_sc[ivec];
#endif

	/* don't read the data in response to a command */
	if (softc->status & MIDI_NEED_DATA)
		return (1);

	code = INB(softc->addr + MIDI_DATA);
	if (code == 0xfe && (softc->status & MIDI_NEED_ACK)) {
		softc->status &= ~MIDI_NEED_ACK;
		return (1);
	}

	/* throw away data if no one has the device open */
	if (!(softc->status & MIDI_READING))
		return (1);

	/* pass input data to out port if necessary */
	if (softc->status & MIDI_THRU) {
		if (!midi_wait_rdy_rcv(softc))
			printf("Couldn't pass data thru\n");
		else
			OUTB(softc->addr + MIDI_DATA, code);
	}
	pe = &softc->partial_event;

	/* check for realtime events */
	if ((code & 0xf8) == 0xf8) {
		switch (code) {
		case 0xfa:	/* start */
			/* reset prev_incoming */
			softc->prev_incoming = midi_get_timing_clock(softc);
			break;
		case 0xff:	/* reset */
			/*
			 * I don't really want to do a full reset
			 * I'll just clear any event in progress
			 */
			stynamic_release(&pe->event);
			pe->state = START;
			break;
		case 0xf8:	/* timing clock */
		case 0xf9:	/* undefined */
		case 0xfb:	/* continue */
		case 0xfc:	/* stop */
		case 0xfd:	/* undefined */
		case 0xfe:	/* active sensing */
			break;
		}
		return (1);
	}
INTR_SWITCH:
	switch (pe->state) {
	case START:
		/* record when the time when the event started */
		pe->time = midi_get_timing_clock(softc);
		pe->tempo = softc->rtempo;
		/* start collecting the input */
		stynamic_release(&pe->event);
		/* decide what state is next */
		if (!(code & 0x80)) {
			stynamic_add_byte(&pe->event, code);
			switch (pe->rs) {
			case 0x80:
			case 0x90:
			case 0xa0:
			case 0xb0:
			case 0xe0:
				/*
				 * code is the first data byte, but
				 * we still need to get the second
				 */
				pe->state = NEEDDATA2;
				break;
			case 0xc0:
			case 0xd0:
				/* code is the only data byte */
				pe->state = START;
				midi_add_complete_event(softc);
				break;
			default:
				break;
			}
		} else {
			switch (code & 0xf0) {
			case 0x80:
			case 0x90:
			case 0xa0:
			case 0xb0:
			case 0xe0:
				pe->rs = code & 0xf0;
				stynamic_add_byte(&pe->event, code);
				pe->state = NEEDDATA1;
				break;
			case 0xc0:
			case 0xd0:
				pe->rs = code & 0xf0;
				stynamic_add_byte(&pe->event, code);
				pe->state = NEEDDATA2;
				break;
			default:
				switch (code) {
				case 0xf0: /* sysex */
					stynamic_add_byte(&pe->event, code);
					pe->state = SYSEX;
					break;
				case 0xf1: /* MTC quarter-frame */
					if (softc->status & MIDI_EXTCLK) {
						pe->state = MTC;
						softc->extclock++;
					}
					break;
				case 0xf4: /* undefined */
				case 0xf5: /* undefined */
				case 0xf6: /* tune request */
				case 0xf7: /* EOX (terminator) */
					/* ignore these */
					break;
				case 0xf2: /* song position */
					pe->state = SYSTEM2;
					break;
				case 0xf3: /* song select */
					pe->state = SYSTEM1;
					break;
				}
				break;
			}
		}
		break;
	case NEEDDATA1:
		stynamic_add_byte(&pe->event, code);
		pe->state = NEEDDATA2;
		break;
	case NEEDDATA2:
		stynamic_add_byte(&pe->event, code);
		pe->state = START;
		midi_add_complete_event(softc);
		break;
	case SYSEX:
		/* any non-data byte ends sysex */
		if (!(code & 0x80))
			stynamic_add_byte(&pe->event, code);
		else {
			stynamic_add_byte(&pe->event, 0xf7);
			midi_add_complete_event(softc);
			pe->state = START;
			if (code != 0xf7)
				goto INTR_SWITCH;
		}
		break;
	case SYSTEM1:
		/* throw away one data byte of a system message */
		pe->state = START;
		break;
	case SYSTEM2:
		/* throw away two data bytes of a system message */
		pe->state = SYSTEM1;
		break;
	case MTC:
		pe->state = START;
		midi_parse_MTC_event(softc, code);
		break;
	}
	return (1);
}

int
midiioctl(dev, cmd, arg, mode, cred_p, rval_p)
	dev_t dev;
	int cmd, arg, mode;
	cred_t *cred_p;
	int *rval_p;
{
	struct event *event;
	struct midi_softc *softc;
	register int error, unit;
	int i, ret, val;

	unit = minor(dev) & 0x7f;
	softc = &midi_sc[unit];

	switch (cmd) {
	case FIOASYNC:
	case MASYNC:
		/*
		 * Linux doesn't properly process the FIOASYNC
		 * ioctl entry point, thus we have two.  For
		 * compatibility purposes, I'm duplicating it
		 * here too.
		 */

		copyin((caddr_t)arg, (caddr_t)&val, sizeof(int));
		if (!val)
			softc->status &= ~MIDI_ASYNC;
		else {
			struct proc *p;

			softc->status |= MIDI_ASYNC;
			if (softc->wqueue->count < MIDI_LOW_WATER
			    && softc->status & MIDI_WRITING) {
				if (softc->pgid < 0)
					signal(-softc->pgid, SIGIO);
				else if ((p = prfind(softc->pgid)) != 0)
					psignal(p, SIGIO);
			}
		}
		break;
	case TIOCSPGRP:
		copyin((caddr_t)arg, (caddr_t)&softc->pgid, sizeof(int));
		break;
	case TIOCGPGRP:
		copyout((caddr_t)&softc->pgid, (caddr_t)arg, sizeof(int));
		break;
	case MRESET:
		if (!midi_fullreset(softc))
			return (EIO);
		break;
	case MSDIVISION:
		copyin((caddr_t)arg, (caddr_t)&val, sizeof(int));
  		/* must recalculate play remainder */
  		softc->premainder = softc->premainder * val / softc->division;
  		softc->division = val;
		break;
	case MGDIVISION:
		copyout((caddr_t)&softc->division, (caddr_t)arg, sizeof(int));
		break;
	case MDRAIN:
		/* dequeue all everything */
		while (midi_deq(softc->rqueue, &event))
			stynamic_release(&event->data);
		while (midi_deq(softc->wqueue, &event))
			stynamic_release(&event->data);
		/* remove any events already being timed */
		if (softc->timeout_id != -1)
			untimeout(softc->timeout_id);
		softc->status &= ~MIDI_WR_BLOCK;
		softc->status |= MIDI_RD_BLOCK;
		if (softc->status & MIDI_WR_SLEEP) {
			softc->status &= ~MIDI_WR_SLEEP;
			softc->status |= MIDI_WR_ABORT;
			wakeup((caddr_t)softc->wqueue);
		}
		midi_reset_devices(softc);
		break;
	case MFLUSH:
		if (softc->wqueue->count != 0) {
			softc->status |= MIDI_FLUSH_SLEEP;
			do {
				if (error = sleep((caddr_t)&softc->status,
				    PWAIT | PCATCH))
					return (error);
			} while (softc->status & MIDI_FLUSH_SLEEP);
		}
		break;
	case MGPLAYQ:
		copyout((caddr_t)&softc->wqueue->count, (caddr_t)arg,
		    sizeof(int));
		break;
	case MGRECQ:
		copyout((caddr_t)&softc->rqueue->count, (caddr_t)arg,
		    sizeof(int));
		break;
	case MGQAVAIL:
		val = MIDI_Q_SIZE - softc->wqueue->count;
		copyout((caddr_t)&val, (caddr_t)arg, sizeof(int));
		break;
	case MTHRU:
		copyin((caddr_t)arg, (caddr_t)&val, sizeof(int));
		if (val)
			softc->status |= MIDI_THRU;
		else
			softc->status &= ~MIDI_THRU;
		break;
	case MRECONPLAY:
		copyin((caddr_t)arg, (caddr_t)&val, sizeof(int));
		if (val)
			softc->status |= MIDI_RECONPLAY;
		else
			softc->status &= ~MIDI_RECONPLAY;
		break;
	case MGSMPTE:
		if (!(softc->status & MIDI_EXTCLK))
			return (ENODEV);
		*(struct SMPTE_frame *)data = softc->SMPTE_current;
		break;
	case MGSMFTIME:
		*(u_long *)data = midi_get_smf_time(softc);
		break;
	default:
		return (ENOTTY);
	}
	return (0);
}

/*
 * XXXPOLL
 * I don't know how this is supposed to work.  The SVR4/386
 * DDI/DKI reference manual doesn't explain it well enough for me.
 */
int
midipoll(dev, events, anyyet, reventsp, phpp)
	dev_t dev;
	short events;
	int anyyet;
	short *reventsp;
	struct pollhead **phpp;
{
	register struct midi_softc *softc;
	int unit;

	unit = minor(dev) & 0x7f;
	softc = &midi_sc[unit];

	*reventsp = 0;
	if (events & POLLIN && !(softc->status & MIDI_RD_BLOCK))
		*reventsp |= POLLIN;
	if (events & POLLOUT && !(softc->status & MIDI_WR_BLOCK))
		*reventsp |= POLLOUT;
	/* can't statisfy POLLPRI immediately, wait for timewarp */
	if (events & POLLPRI)
		;
	if (*reventsp == 0)
		if (!anyyet)
			*phpp = &softc->pollhead;
	return (0);
}

int
midi_fullreset(softc)
	struct midi_softc *softc;
{
	u_char pitch;
	struct event *event;
	struct tempo_change *tch, *nextch;

	/* dequeue all everything */
	while (midi_deq(softc->rqueue, &event))
		stynamic_release(&event->data);
	while (midi_deq(softc->wqueue, &event))
		stynamic_release(&event->data);
	/* free any storage allocated for tempo changes */
	for (tch = softc->tempo_changes; tch != NULL; tch = nextch) {
		nextch = tch->next;
		kmem_free(tch, sizeof(struct tempo_change));
	}
	softc->tempo_changes = NULL;
	/* remove any events already being timed */
	if (softc->timeout_id != -1)
		untimeout(softc->timeout_id);
	softc->timeout_id = -1;
	/* mark start time */
	softc->start_time = lbolt;
	softc->prev_incoming = midi_get_timing_clock(softc);
	softc->prev_outgoing = 0;
	/* reset fractional count */
	softc->premainder = 0;
	softc->rremainder = 0;
	/* initialize some variables */
	/* this is 120 bpm when a quarter note == 1 beat */
	softc->ptempo = 500000;
	softc->rtempo = 500000;
	softc->prev_rtempo = 500000;

	/* reset external clock */
	softc->extclock = 0;

	/* clear noteon */
	for (pitch = 0; pitch <= 0x7f; pitch++)
		softc->noteon[pitch] = 0;
	softc->noteonrs = 0;

	/* clear running play state */
	softc->writers = 0;

	/* reset partial event stuff */
	stynamic_release(&softc->partial_event.event);
	softc->partial_event.state = START;
	stynamic_release(&softc->rpartial);
	stynamic_release(&softc->wpartial);
	softc->wpartialpos = 0;

	/* defaults to 120 clocks per beat */
	softc->division = 120;
	/* default to kernel timing 100 */
	softc->hz = Hz;
	/* reset and enter uart mode */
	(void)midi_reset(softc);
	return (midi_uart(softc));
}

int
midi_wait_rdy_rcv(softc)
	struct midi_softc *softc;
{
	int flag;
	register int i;

	for (i = 0; i < MIDI_TRIES; i++) {
		flag = INB(softc->addr + MIDI_STATUS) & MIDI_RDY_RCV;
		if (flag == 0)
			break;
	}
	if (i == MIDI_TRIES)
		return (0);
	return(1);
}

int
midi_send_command(softc, comm)
	struct midi_softc *softc;
	u_char comm;
{
	int flag;
	register int i, s;
	unsigned char ack;

	/* writing command and setting flag must be atomic */
	s = spltty();
	OUTB(softc->addr + MIDI_COMMAND, comm);
	softc->status |= MIDI_NEED_ACK;
	splx(s);

	/* time out after MIDI_TRIES times */
	for (i = 0; i < MIDI_TRIES; i++) {
		/* did we pick up the ack via midiintr? */
		if (!(softc->status & MIDI_NEED_ACK))
			break;
		/* can we read a data byte? */
		flag = INB(softc->addr + MIDI_STATUS) & MIDI_DATA_AVL;
		if (flag == 0)
			break;
	}
	if (i == MIDI_TRIES)
		return (0);

	if (softc->status & MIDI_NEED_ACK) {
		ack = INB(softc->addr + MIDI_DATA);
		if (ack != MIDI_ACK)
			return (0);
	}
	softc->status &= ~MIDI_NEED_ACK;
	return (1);
}

int
midi_send_command_with_response(softc, comm, response, resp_size)
	struct midi_softc *softc;
	u_char comm;
	void *response;
	int resp_size;
{
	int flag;
	register int i, j, s;
	unsigned char ack, *resp;

	/* writing command and setting ack flag must be atomic */
	s = spltty();
	OUTB(softc->addr + MIDI_COMMAND, comm);
	softc->status |= MIDI_NEED_ACK;
	softc->status |= MIDI_NEED_DATA;
	splx(s);

	/* time out after MIDI_TRIES times */
	for (i = 0; i < MIDI_TRIES; i++) {
		/* did we pick up the ack via midiintr? */
		if (!(softc->status & MIDI_NEED_ACK))
			break;
		/* can we read a data byte? */
		flag = INB(softc->addr + MIDI_STATUS) & MIDI_DATA_AVL;
		if (flag == 0)
			break;
	}
	if (i == MIDI_TRIES)
		return (0);

	if (softc->status & MIDI_NEED_ACK) {
		ack = INB(softc->addr + MIDI_DATA);
		if (ack != MIDI_ACK)
			return (0);
	}
	softc->status &= ~MIDI_NEED_ACK;
	resp = response;
	for (j = 0; j < resp_size; j++) {
		for (i = 0; i < MIDI_TRIES; i++) {
			flag = INB(softc->addr + MIDI_STATUS) & MIDI_DATA_AVL;
			if (flag == 0)
				break;
		}
		if (i == MIDI_TRIES)
			return (0);
		resp[j] = INB(softc->addr + MIDI_DATA);
	}
	softc->status &= ~MIDI_NEED_DATA;
	return (1);
}

int
midi_reset(softc)
	struct midi_softc *softc;
{

	if (!midi_wait_rdy_rcv(softc))
		return (0);

	if (!midi_send_command(softc, MIDI_RESET))
		return (0);

	return (1);
}

int
midi_uart(softc)
	struct midi_softc *softc;
{

	if (!midi_wait_rdy_rcv(softc))
		return (0);

	if (!midi_send_command(softc, MIDI_UART))
		return (0);
	return (1);
}

int
midi_enter_smpte_mode(softc)
	struct midi_softc *softc;
{

	if (!(softc->features & SMPTE_EQUIP))
		return (0);
	bzero(&softc->SMPTE_current, sizeof(softc->SMPTE_current));
	softc->SMPTE_current.rate = 30;
	bzero(&softc->SMPTE_partial, sizeof(softc->SMPTE_partial));
	softc->SMPTE_partial.rate = 30;
	/* first guess for hz */
	softc->hz = 120;

	/* MQX on-board clock */
	if (!midi_wait_rdy_rcv(softc))
		return (0);
	if (!midi_send_command(softc, 0x80))
		return (0);
	/* SMPTE tape sync */
	if (!midi_wait_rdy_rcv(softc))
		return (0);
	if (!midi_send_command(softc, 0x3d))
		return (0);
	/* SMPTE reading active */
	if (!midi_wait_rdy_rcv(softc))
		return (0);
	if (!midi_send_command(softc, 0x2d))
		return (0);
	/* system-exclusive messages to PC */
	if (!midi_wait_rdy_rcv(softc))
		return (0);
	if (!midi_send_command(softc, 0x97))
		return (0);
	/* system common, real-time messages to PC */
	if (!midi_wait_rdy_rcv(softc))
		return (0);
	if (!midi_send_command(softc, 0x38))
		return (0);
	if (!midi_wait_rdy_rcv(softc))
		return (0);
	if (!midi_send_command(softc, 0x39))
		return (0);
	/* clock messages to PC */
	if (!midi_wait_rdy_rcv(softc))
		return (0);
	if (!midi_send_command(softc, 0x95))
		return (0);

	return (1);
}

int
midi_event2smf(softc, event, smf)
	struct midi_softc *softc;
	struct event *event;
	struct stynamic *smf;
{
	long tempo, smf_ticks;
	int tmp_len;
	u_char tmp_buf[4];

	/* convert from timing ticks to SMF ticks */
	smf_ticks = midi_timing2smf_tick(softc, event->tempo, event->time);
	tmp_len = midi_fix2var(smf_ticks, tmp_buf);
	stynamic_add_bytes(smf, tmp_buf, tmp_len);

	switch (event->type) {
	case NORMAL:
		stynamic_append(smf, &event->data);
		break;
	case TEMPO:
		/* this case just won't occur */
		stynamic_copy(&event->data, (void *)&tempo, sizeof(tempo));
		stynamic_add_byte(smf, 0xff);
		stynamic_add_byte(smf, 0x51);
		stynamic_add_byte(smf, 0x03);
		stynamic_add_byte(smf, (tempo & 0xff0000) >> 16);
		stynamic_add_byte(smf, (tempo & 0xff00) >> 8);
		stynamic_add_byte(smf, tempo & 0xff00);
		break;
	case SYSX:
		stynamic_add_byte(smf, 0xf0);
		/* skip over the leading 0xf0 */
		stynamic_shift(&event->data, 1);
		tmp_len = midi_fix2var(event->data.len, tmp_buf);
		stynamic_add_bytes(smf, tmp_buf, tmp_len);
		styanmic_append(smf, &event->data);
		break;
	}
	return (1);
}

int
midi_next_byte(softc, uio)
	struct midi_softc *softc;
	struct uio *uio;
{
	int byte;

	/* if we're not at the end of a partial event, read from it */
	if (softc->wpartialpos < softc->wpartial.len) {
		byte = stynamic_get_byte(&softc->wpartial,
		    softc->wpartialpos++);
		return (byte);
	} else {
		/* read from uio and copy uio onto partial event */
		if ((byte = uwritec(uio)) == -1) {
			/*
			 * reset partialpos so next time through
			 * we'll read from the partial event if
			 * it is non-zero in length
			 */
			softc->wpartialpos = 0;
			return (-1);
		}
		stynamic_add_byte(&softc->wpartial, byte);
		softc->wpartialpos = softc->wpartial.len;
		return (byte);
	}
}

int
midi_uio2event(softc, uio, event)
	struct midi_softc *softc;
	struct uio *uio;
	struct event *event;
{
	u_long smf_ticks, ulen;
	long tempo;
	int byte, error, extra_byte, i, len, num_data_bytes;
	int rs_change;
	u_char meta_type, tmp_buf[256];
#ifdef GJWDEBUG
	struct SMPTE_frame tmp_frame;
#endif

	/* copy in timing portion */
	len = 0;
	do {
		if ((byte = midi_next_byte(softc, uio)) == -1) {
			stynamic_release(&event->data);
			return (-1);
		}
		tmp_buf[len++] = byte;
	} while (byte & 0x80);

	/* compute time in smf ticks */
	midi_var2fix(tmp_buf, &smf_ticks);

	/* now convert from smf to timing */
	event->time = midi_smf2timing_tick(softc, smf_ticks);
	softc->write_smf_time += smf_ticks;
	event->smf_time = softc->write_smf_time;

	/* get first byte of event data */
	if ((byte = midi_next_byte(softc, uio)) == -1) {
		stynamic_release(&event->data);
		return (-1);
	}
	switch (byte) {
	case 0xf0:
		/* basic sysex type */
		event->type = SYSX;
		len = 0;
		do {
			if ((byte = midi_next_byte(softc, uio)) == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[len++] = byte;
		} while (byte & 0x80);
		midi_var2fix(tmp_buf, &ulen);
		stynamic_add_byte(&event->data, 0xf0);
		for (; ulen > 0; ulen--) {
			if ((byte = midi_next_byte(softc, uio)) == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			stynamic_add_byte(&event->data, byte);
		}
		break;
	case 0xf7:
		/* continued sysex type */
		event->type = SYSX;
		len = 0;
		do {
			if ((byte = midi_next_byte(softc, uio)) == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[len++] = byte;
		} while (byte & 0x80);
		midi_var2fix(smf, &ulen);
		for (; ulen > 0; ulen--) {
			if ((byte = midi_next_byte(softc, uio)) == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			stynamic_add_byte(&event->data, byte);
		}
		if (error = uiomove(event->data, event->length, UIO_WRITE,
		    uio))
			return (error);
		break;
	case 0xff:
		/* meta events */
		if ((byte = midi_next_byte(softc, uio)) == -1) {
			stynamic_release(&event->data);
			return (-1);
		}
		meta_type = byte;
		/* get length of meta data */
		len = 0;
		do {
			if ((byte = midi_next_byte(softc, uio)) == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[len++] = byte;
		} while (byte & 0x80);
		midi_var2fix(tmp_buf, &ulen);
		/* read it in  - meta events are not over 256 in size */
		for (i = 0; i < ulen; i++) {
			if ((byte = midi_next_byte(softc, uio)) == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
		}
		switch (meta_type) {
		default:
			/*
			 * we'll skip these events, but we need to
			 * save the timing info
			 */
			event->type = NOP;
			break;
		case 0x51:
			/* tempo event */
			event->type = TEMPO;
			tempo = 0;
			for (i = 0; i < 3; i++) {
				tempo = tempo << 8;
				tempo |= tmp_buf[i];
			}
			stynamic_add_bytes(&event->data, (u_char *)&tempo,
			    sizeof(tempo));
			/*
			 * change ptempo now, rtempo will change when
			 * the event's time comes up
			 */
			softc->ptempo = tempo;
			break;
		case 0x54:
			/* MetaSMPTE */
			event->type = SMPTE;
			stynamic_add_bytes(&event->data, tmp_buf, 5);

			/*
			 * MetaSMPTE must come before any non-zero delta
			 * events.  Therefore let MetaSMPTE *define* the
			 * SMF zero point
			 */
			softc->write_smf_time = 9;
#ifdef GJWDEBUG
			/*
			 * convert the MIDI format SMPTE frame into our
			 * format
			 */
			midi_midi2SMPTE(event, &tmp_frame);
			printf("MetaSMPTE %2d:%2d:%2d:%2d:%2d\n",
			    tmp_frame.hour, tmp_frame.minute,
			    tmp_frame.second, tmp_frame.frame,
			    tmp_frame.fraction);
			break;
#endif
		}
		break;
	default:
		if ((byte & 0xf0) == 0x80) {
			u_char rs_chan, rs_type;

			/* check note off events separately */
			tmp_buf[0] = byte;
			if ((byte = midi_next_byte(softc, uio)) == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[1] = byte;
			if ((byte = midi_next_byte(softc, uio)) == -1) {
				stynamic_release(&event->data);
				return (-1);
			}
			tmp_buf[2] = byte;
			len = 3;
			/*
			 * check to see if we can collapse and use
			 * running state
			 */
			rs_type = softc->writers & 0xf0;
			rs_chan = softc->writers & 0x0f;
			/*
			 * collapse to running state if time is 0
			 * and the running state is the same
			 * or the running state is note on for the
			 * same channel and the note off velocity is
			 * zero.
			 */
			if (event->time == 0 && (softc->writers == tmp_buf[0]
			    || (tmp_buf[2] == 0 && rs_type == 0x90
			    && rs_chan == (tmp_buf[0] & 0x0f)))) {
				tmp_buf[0] = tmp_buf[1];
				tmp_buf[1] = tmp_buf[2];
				len = 2;
			} else {
				softc->writers = tmp_buf[0];
			}
		} else {
			extra_byte = 0;
			rs_change = 0;
			if ((byte & 0x80) && (byte != softc->writers)) {
				softc->writers = byte;
				rs_change = 1;
			}
			len = 0;
			if (event->time != 0 || rs_change) {
				/*
				 * stick in a mode byte if time is non-zero
				 * This is so we don't confuse hardware that
				 * is turned on while we're playing
				 * also add it if the running state changes
				 */
				tmp_buf[0] = softc->writers;
				len = 1;
			}
			if (byte & 0x80)
				extra_byte = 1;
			else {
				tmp_buf[len] = byte;
				len++;
			}

			switch (softc->writers & 0xf0) {
			case 0x80:
			case 0x90:
			case 0xa0:
			case 0xb0:
			case 0xe0:
				num_data_bytes = 1;
				break;
			default:
				num_data_bytes = 0;
			}
			for (i = 0; i < num_data_bytes + extra_byte; i++) {
				if ((byte = midi_next_byte(softc, uio)) == -1) {
					stynamic_release(&event->data);
					return (-1);
				}
				tmp_buf[len++] = byte;
			}
		}
		event->type = NORMAL;
		stynamic_add_bytes(&event->data, tmp_buf, len);
	}

	stynamic_release(&softc->wpartial);
	softc->wpartialpos = 0;
	return (0);
}

int
midi_fix2var(fix, var)
	u_long fix;
	u_char *var;
{
	int i;
	unsigned char buf[4], *bptr;

	buf[0] = buf[1] = buf[2] = buf[3] = 0;
	bptr = buf;
	*bptr++ = fix & 0x7f;
	while ((fix >>= 7) > 0) {
		*bptr |= 0x80;
		*bptr++ += (fix & 0x7f);
        }

	i = 0;
	do {
		*var++ = *--bptr;
		i++;
	} while (bptr != buf);

	return (i);
}

int
midi_var2fix(var, fix)
	u_char *var;
	u_long *fix;
{
	int delta;

	*fix = 0;
	delta = 0;
	do {
		*fix = (*fix << 7) + (*var & 0x7f);
		delta++;
	} while (*var++ & 0x80);

	return (delta);
}

u_long
midi_smf2timing_tick(softc, smf)
	struct midi_softc *softc;
	long smf;
{
  	long long denominator, numerator;
  	u_long timing;
   
	numerator = (long long)softc->ptempo * softc->hz * smf
	    + softc->premainder;
	denominator = 1000000 * (long long)softc->division;
	time = numerator / denominator;
	softc->premainder = numerator % denominator;
	return (timing);
}

u_long
midi_timing2smf_tick(softc, tempo, timing)
	struct midi_softc *softc;
	long tempo, timing;
{
  	long long numerator, denominator;
  	u_long smf;
   
  	if (softc->prev_rtempo != tempo) {
  		/*
  		 * also update the rremainder to reflect tempo
  		 * change
  		 */
  		softc->rremainder = softc->rremainder * tempo
  		    / softc->prev_rtempo;
  		softc->prev_rtempo = tempo;
  	}
	numerator = (long long)softc->division * 1000000 * timing
	    + softc->rremainder;
	denominator = (long long)tempo * softc->hz;
  	smf = numerator / denominator;
  	softc->rremainder = numerator % denominator;
   	return (smf);
}

void
midi_add_complete_event(softc)
	struct midi_softc *softc;
{
	struct event event;
	struct partial_event *pe;
	long ticks;

	stynamic_init(&event.data);
	pe = &softc->partial_event;

	ticks = pe->time - softc->prev_incoming;
	if (ticks < 0) {
		ticks = 0;
		pe->time = softc->prev_incoming;
	}
	event.time = ticks;
	event.tempo = pe->tempo;
	switch (stynamic_get_byte(&pe->event, 0)) {
	case 0xf0:
		/* sysex */
		event.type = SYSX;
		break;
	default:
		event.type = NORMAL;
		break;
	}
	stynamic_append(&event.data, &softc->partial_event.event);
	/* enqueue new event */
	midi_enq(softc->rqueue, &event);
	stynamic_release(&event.data);
	/* readjust previous event time */
	softc->prev_incoming = pe->time;

	softc->status &= ~MIDI_RD_BLOCK;
	if (softc->status & MIDI_RD_SLEEP) {
		softc->status &= ~MIDI_RD_SLEEP;
		wakeup((caddr_t)softc->rqueue);
	}
	if (softc->status & MIDI_ASYNC) {
		struct proc *p;

		if (softc->pgid < 0)
			signal(-softc->pgid, SIGIO);
		else if ((p = prfind(softc->pgid)) != 0)
			psignal(p, SIGIO);
	}

	/* notify poll that there is data to be read */
	/*
	 * XXXPOLL - do I only want to do this if a poll is in
	 * progress and anyyet is 0?  Then I'll need to set
	 * a status flag, and how do I know when to unset it after
	 * the poll times out (ala select).
	 */
	pollwakeup(&softc->pollhead, POLLIN);

	stynamic_release(&event.data);
	return;
}

void
midi_schedule_timeout(softc, t)
	struct midi_softc *softc;
	u_long t;
{

	/* don't need this with external timing */
	if (!(softc->status & MIDI_EXTCLK)) {
		if (softc->status & MIDI_FIRST_WRITE) {
			softc->start_time = lbolt;
			softc->status &= ~MIDI_FIRST_WRITE;
		}
		softc->timeout_id = timeout(midi_timeout, (caddr_t)softc,
		    event->time);
	}
}

void
midi_timeout(arg)
	caddr_t arg;
{
	struct event *event;
	struct midi_softc *softc = (struct midi_softc *)arg;
	u_long sched_time, t;
	int quiet;
#ifdef GJWDEBUG
	struct SMPTE_frame *cf = &softc->SMPTE_current;
#endif

	if (!midi_peekq(softc->wqueue, &event))
		return (-1);
	while ((softc->prev_outgoing + event->time) <=
	    (t = midi_get_timing_clock(softc))) {
		midi_deq(softc->wqueue, NULL);
#ifdef GJWDEBUG
		/* Useful for debugging sync problems */
		printf("Writing: S:%2d:%2d:%2d:%2d P:%4lu E:%4lu/%3lu M:%4lu\n",
		    cf->minute, cf->second, cf->frame, cf->fraction,
		    softc->prev_outgoing, softc->extclock,
		    event->time, event->smf_time);
#endif

		/*
		 * If the event is backlogged for more than BACKLOG_TIME ticks,
		 * don't play it
		 */
		if ((t - softc->prev_outgoing - event->time) > BACKLOG_TIME)
			quiet = 1;
		else
			quiet = 0;

		midi_write_event(softc, event, quiet);
		softc->prev_outgoing += event->time;
		stynamic_release(&event->data);
		if (!midi_peekq(softc->wqueue, &event))
			return (-1);
		/* clear record timer? */
		if (softc->status & MIDI_RECONPLAY) {
			softc->prev_incoming = midi_get_timing_clock(softc);
			softc->status &= ~MIDI_RECONPLAY;
		}
	}
	sched_time = event->time - (t - softc->prev_outgoing);
	midi_schedule_timeout(softc, sched_time);
	return (0);
}

void
midi_dropped_SMPTE_sync(arg)
	caddr_t arg;
{
	struct midi_softc *softc = (struct midi_softc *)arg;
	struct SMPTE_frame *cf;

	cf = &softc->SMPTE_current;

	/*
	 * If we are here, then too many ticks went by without
	 * a SMPTE frame.  Assume that sync is lost
	 */
	softc->SMPTE_current.status &= ~SMPTE_SYNC;
	softc->SMPTE_partial.status &= ~SMPTE_SYNC;

	/*
	 * We'll need to catch another set of 8 quarter-frame
	 * messages before we can re-sync.  Make sure we get a
	 * full set, starting with the first one.
	 */
	softc->SMPTE_current.status &= ~SMPTE_FIRST0xF1;
	softc->SMPTE_partial.status &= ~SMPTE_FIRST0xF1;

#ifdef GJWDEBUG
	printf("SMPTE sync dropped at %2d:%2d:%2d:%2d:%2d\n",
	    cf->hour, cf->minute, cf->second, cf->frame, cf->fraction);
#endif
}


void
midi_write_event(softc, event, quiet)
	struct midi_softc *softc;
	struct event *event;
	int quiet;
{
	struct tempo_change *tchange, *prev_tchange;
	int i, j;
	u_char bytes[4], channel, command, *dataptr;

	switch (event->type) {
	case NOP:
		break;
	case SMPTE:
		break;
	case TEMPO:
		stynamic_copy(&event->data, &softc->rtempo,
		    sizeof(softc->rtempo));
		for (tchange = softc->tempo_changes, prev_tchange = NULL;
		    tchange != NULL;
		    prev_tchange = tchange, tchange = tchange->next) {
			if (tchange->smf_time < event->smf_time)
				break;
		}
		if (tchange == NULL) {
			/* add the tempo change to our list */
			tchange = kmem_alloc(sizeof(struct tempo_change),
			    KM_NOSLEEP);
			if (tchange == NULL) {
				printf("Out of kernel memory: we're screwed\n");
				break;
			} else {
				tchange->time = event->time +
				    softc->prev_outgoing;
				tchange->smf_time = event->smf_time;
				tchange->tempo = softc->rtempo;
				tchange->next = NULL;
				/*
				 * Don't update through a null pointer.
				 * On the other hand, make sure to update
				 * softc->tempo_changes to reflect storage.
				 */
				if (prev_tchange != NULL)
					prev_tchange->next = tchange;
				else
					softc->tempo_changes = tchange;
			}
		}
		break;
	case NORMAL:
		/*
		 * fourth byte might not be valid, but who cares,
		 * we're only reading and in the kernel.  We'll
		 * ignore it if it isn't.
		 */
		stynamic_copy(&event->data, &bytes, 4);
		if (!(bytes[0] & 0x80))
			dataptr = &bytes[0];
		else {
			softc->noteonrs = bytes[0];
			dataptr = &bytes[1];
		}
		command = softc->noteonrs & 0xf0;
		channel = softc->noteonrs & 0x0f;
		/*
		 * In `quiet' mode, don't turn on any new notes but do
		 * turn them off if we're asked.
		 */
		if (command == 0x90) {
			/*
			 * set or clear appropriate bit in noteon
			 * array depending on velocity value
			 */
			if (dataptr[1] != 0)
				softc->noteon[dataptr[0]] |= 1 << channel;
			else
				softc->noteon[dataptr[0]] &= ~(1 << channel);
		}
		if (command == 0x80)
			/* clear bit */
			softc->noteon[dataptr[0]] &= ~(1 << channel);
		/* FALLTHRU */
	default:
		if (!quiet) {
			for (i = 0; i < event->data.len; i++) {
				if (!midi_wait_rdy_rcv(softc))
					break;
				OUTB(softc->addr + MIDI_DATA,
				    stynamic_get_byte(&event->data, i));
			}
		}
		break;
	}
	midi_send_sigio(softc);
}


/*
 * try to reset the midi devices as best we can
 */
void
midi_reset_devices(softc)
	struct midi_softc *softc;
{
	int i;
	u_char channel, pitch;

	/* manual note off calls - turn off any note that is on */
	for (pitch = 0; pitch <= 0x7f; pitch++) {
		for (channel = 0; channel <= 0x0f; channel++) {
			if ((softc->noteon[pitch] >> (int)channel) & 0x01) {
				if (!midi_wait_rdy_rcv(softc))
					break;
				OUTB(softc->addr + MIDI_DATA, channel | 0x90);

				if (!midi_wait_rdy_rcv(softc))
					break;
				OUTB(softc->addr + MIDI_DATA, pitch);

				if (!midi_wait_rdy_rcv(softc))
					break;
				OUTB(softc->addr + MIDI_DATA, 0);
			}
		}
		softc->noteon[pitch] = 0;
	}
	for (channel = 0; channel <= 0x0f; channel++) {
		/*
		 * send paramter event for all notes off for redundancy
		 * some older synths don't support this
		 */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, channel | 0xb0);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x7b);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);

		/* modulation controller to zero */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x01);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);

		/* reset all controllers */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x79);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);

		/* lift sustain pedal */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x40);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);

		/* center pitch wheel */
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0xe0 | channel);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0);
		if (!midi_wait_rdy_rcv(softc))
			break;
		OUTB(softc->addr + MIDI_DATA, 0x40);
	}
	softc->noteonrs = 0;
}

/*
 * An MTC event looks like 0xF1 0xNN.  Here code = 0xNN,
 * and this function interprets the data 0xNN as a quarter
 * frame, updating the partial frame structure with incoming
 * frame data.  We also update the current frame structure
 * with fractional frame data, and keep an eye out for loss
 * of tape sync via kernel timer.  See the MTC spec for info
 * on interpreting the way a SMPTE frame is encoded into 8
 * quarter frames of MTC.
 */
void
midi_parse_MTC_event(softc, code)
	struct midi_softc *softc;
	u_char code;
{
	struct SMPTE_frame *cf, *pf, tmp_frame;
	struct event *e;
	u_long offset;

	pf = &softc->SMPTE_partial;
	cf = &softc->SMPTE_current;

	switch (code & 0xf0) {
	case 0x00:
		/* Frame count LS nibble  -- first quarter frame */
		pf->status |= SMPTE_FIRST0xF1;
		pf->frame = code & 0x0f;
		cf->fraction = 25;
		break;
	case 0x10:
		/* Frame count MS nibble */
		pf->frame |= ((code & 0x01) << 4);
		cf->fraction = 50;
		/* Reschedule our sync drop every SMPTE_DROPOUT frames */
		if ((pf->frame % SMPTE_DROPCHK) == 0) {
			if (softc->SMPTE_current.status & SMPTE_SYNC) {
				if (softc->timeout_id != -1)
					untimeout(softc->timeout_id);
			}
			softc->timeout_id = timeout(midi_dropped_SMPTE_sync,
			    (caddr_t)softc, SMPTE_DROPOUT);
		}
		break;
	case 0x20:
		/* second LS nibble */
		pf->second = code & 0x0f;
		cf->fraction = 75;
		break;
	case 0x30:
		/* second MS nibble */
		pf->second |= ((code & 0x03) << 4);
		cf->fraction = 0;
		cf->frame++;
		/* 25-frame has this problem */
		if ((cf->rate == 25) && (cf->frame == 25)) {
			cf->frame &= 0xe0;
			if (++cf->second == 60) {
				cf->second &= 0xc0;
				if (++cf->minute == 60) {
					cf->minute &= 0xc0;
					if (++cf->hour == 24)
						cf->hour &= 0xe0;
				}
			}
		}
		break;
	case 0x40:
		/* minute LS nibble */
		pf->minute = code & 0x0f;
		cf->fraction = 25;
		break;
	case 0x50:
		/* minute MS nibble */
		pf->minute |= ((code & 0x03) << 4);
		cf->fraction = 50;
		break;
	case 0x60:
		/* hour LS nibble */
		pf->hour = code & 0x0f;
		cf->fraction = 75;
		break;
	case 0x70:
		/* hour MS nibble */
		pf->hour |= ((code & 0x01) << 4);
		/*
		 * correct this two-frame bullshit here, one.
		 * that way all timing we have to deal with is
		 * clean, with no offsets at all, except for
		 * right here as we pull it off the tape.
		 */
		pf->frame += 2;
		if (pf->frame >= pf->rate) {
			pf->second++;
			pf->frame -= pf->rate;
			if (pf->second == 60) {
				pf->minute++;
				pf->second = 0;
				if (pf->minute == 60) {
					pf->hour++;
					pf->minute = 0;
					if (pf->hour == 24)
						pf->hour = 0;
				}
			}
		}

		/*
		 * This is the last of 8 quarter-frame MTC messages used
		 * to encode one SMPTE frame.  If we weren't sync'd before,
		 * we are now (assuming we caught all 8 quarter frames..)
		 */

		/* If we're not currently sync'd */
		if (!(cf->status & SMPTE_SYNC)) {
			/* and we have read all 8 frames */
			if (pf->status & SMPTE_FIRST0xF1)
				midi_register_SMPTE_sync(softc, code);
		}
		*cf = *pf;
		cf->fraction = 0;
		break;
	default:
		log(LOG_WARNING, "Unknown MTC state\n");
		break;
	}

	/*
	 * check next event in queue.  If it is a MetaSMPTE event,
	 * don't call midi_timeout until the time in the event data
	 * matches the current time.  Actually check all events in
	 * queue that will go out together (those with zero delta)..
	 */
	offset = 0;
	do {
		if (!midi_browseq(softc->wqueue, offset++, &e)) {
			/* we didn't get a `next' event */
			if (offset == 1)
				/* bail out if there are *no* events */
				return;
			else
				/*
				 * output those that are here
				 * they aren't SMPTE
				 */
				break;
		}
		/* we did get an event -- check for SMPTE */
		if (e->type == SMPTE) {
			midi_midi2SMPTE(e, &tmp_frame);

			/*
			 * Don't call midi_timeout util the two times match.
			 * Maybe we should allow for cf >= &tmp_frame?? XXX
			 */
			if (midi_compare_SMPTE_time(cf, &tmp_frame) == -1)
				return;

			/*
			 * We have arrived -- set prev_outgoing to match the
			 * MetaSMPTE event time (not necessarily extclock..)
			 */
			softc->prev_outgoing = midi_SMPTE2timing(&tmp_frame);
			break;
		}
	} while (e->time == 0);

	/* try to send some events */
	/*
	 * since we're only sending events when sync is enabled,
	 * and SMPTE_SYNC oly gets set after we have a complete
	 * frame, we won't be able to play anything until after
	 * at least two frames have passed.  That's OK, though,
	 * since we never know what time it really is until we
	 * have read two full frames ~= 1/15 of a second.
	 */

	if (cf->status & SMPTE_SYNC)
		midi_timeout((caddr_t)softc);
}

/*
 * SMPTE sync was just detected.. set everything up
 */
void
midi_register_SMPTE_sync(softc, code)
	struct midi_softc *softc;
	u_char code;
{
	struct event *event;
	struct SMPTE_frame *cf, *pf;

	pf = &softc->SMPTE_partial;
	cf = &softc->SMPTE_current;

	/* indicate synchronization */
	softc->SMPTE_current.status |= SMPTE_SYNC;
	softc->SMPTE_partial.status |= SMPTE_SYNC;

	/* set up timer to check for (future) loss of sync */
	if (!(softc->status & MIDI_FIRST_SMPTE)) {
		if (softc->timeout_id != -1)
			untimeout(softc->timeout_id);
	}
	softc->timeout_id = timeout(midi_dropped_SMPTE_sync, (caddr_t)softc,
	    SMPTE_DROPOUT);

	/* get frame rate from upper bits of code */
	midi_SMPTE_framerate(pf, (code & 0x0e) >> 1);

	/*
	 * SMPTE_ORIGIN is no longer needed.  Instead we rely on a
	 * MetaSMPTE event to align SMPTE time with MIDI time.  Maybe
	 * we should think about default behavior in case there is
	 * no such event.. XXX
	 */

	/* set external clock to match the received frame */
	softc->extclock = midi_SMPTE2timing(pf);

	if (softc->status & MIDI_FIRST_SMPTE) {
		softc->status &= ~MIDI_FIRST_SMPTE;

		/* set external clock resolution based on frame rate */
		softc->hz = 4 * pf->rate;

		/* We have to play from the beginning first time around */
		softc->prev_outgoing = 0;
		softc->write_smf_time = midi_get_smf_time(softc);
	} else {
		/* check clock resolution */
		if (pf->rate != cf->rate)
			printf("SMPTE frame rate has changed! All bets off.\n");

		/*
		 * at this point, we've received sync already, then
		 * lost it, and then regained it.  If the new time is
		 * greater than the old time, we need to scan the MIDI
		 * file for tempo changes.  If it's older, assume we
		 * already have them and do a time-warp instead.
		 */

		/* jump ahead */
		if (midi_compare_SMPTE_time(cf, pf) != -1) {
			/* these are no longe reliable */
			softc->premainder = 0;
			softc->rremainder = 0;

			softc->prev_outgoing = softc->extclock;
			softc->write_smf_time = midi_get_smf_time(softc);
			midi_time_warp(softc);

			while (midi_deq(softc->wqueue, &event))
				stynamic_release(&event->data);
			/* XXX should we dequeue rqueue too? */
			while (midi_deq(softc->rqueue, &event))
				stynamic_release(&event->data);
			softc->status |= MIDI_RD_BLOCK;
			/* remove any events already being timed */
			softc->status &= ~MIDI_WR_BLOCK;
			if (softc->status & MIDI_WR_SLEEP) {
				softc->status &= ~MIDI_WR_SLEEP;
				softc->status |= MIDI_WR_ABORT;
				wakeup(softc->wqueue);
			}
		}
	}
#ifdef GJWDEBUG
	printf("SMPTE sync achieved at %2d:%2d:%2d:%2d\n",
	    pf->hour, pf->minute, pf->second, pf->frame);
	printf("Format is %u-frame", pf->rate);
	if (!(pf->status & SMPTE_FRAMEDROP))
		printf(" no-");
	printf("drop.\n");
#endif
}

/*
 * Convert a SMPTE frame into timing ticks.  We get 4
 * quarter-frames per frame, and we get smpte->rate frames
 * per second.  (Frame-drop formats are handled in the
 * hardware.. not my job ;-)  Two subtle points:
 *
 *   1) All SMPTE data is always two frames old, but the
 *      offset is corrected in parse_MTC_event -- as soon
 *      as it comes off the tape.  That's the only place
 *      you need to worry about it.
 *   2) Internal timing is proportional to SMPTE timing.
 *      Specifically SMPTE 00:00:00:00 is 0 ticks.
 */
u_long
midi_SMPTE2timing(smpte)
	struct SMPTE_frame *smpte;
{
	u_long ticks;
	u_long qrate = (u_long)(4 * smpte->rate);

	ticks = smpte->fraction / 25 + smpte->frame * 4 +
	    smpte->second *qrate +
	    smpte->minute * 60 * qrate +
	    smpte->hour * 60 * 60 * qrate;
	return (ticks);
}

int
midi_midi2SMPTE(event, tf)
	struct event *event;
	struct SMPTE_frame *tf;
{
	u_char code;

	if (event->type != SMPTE) {
		printf(
		    "SMPTE error: midi2SMPTE called with non-SMPTE event.\n");
		return (-1);
	}

	tf->status   = 0;
	tf->hour     = stynamic_get_byte(&event->data, 0) & 0x1f;
	tf->minute   = stynamic_get_byte(&event->data, 1) & 0x3f;
	tf->second   = stynamic_get_byte(&event->data, 2) & 0x3f;
	tf->frame    = stynamic_get_byte(&event->data, 3) & 0x1f;
	tf->fraction = stynamic_get_byte(&event->data, 4);

	code = (stynamic_get_byte(&event->data, 0) & 0xe0) >> 5;
	if (midi_SMPTE_framerate(tf, code) == -1) {
		printf(
		    "SMPTE error: frame rate is not encoded in hour bits.\n");
		printf("Assuming 30-frame no-drop.\n");
		tf->rate = 30;
		tf->status &= ~SMPTE_FRAMEDROP;
		return (-1);
	}
	return (0);
}

int
midi_SMPTE_framerate(tf, code)
	struct SMPTE_frame *tf;
	u_char code;
{

	/* get frame rate from code */
	switch (code) {
	case 0x00:	/* 24 frame */
		tf->rate = 24;
		tf->status &= ~SMPTE_FRAMEDROP;
		break;
	case 0x01:	/* 25 frame */
		tf->rate = 25;
		tf->status &= ~SMPTE_FRAMEDROP;
		break;
	case 0x02:	/* 30 frame drop */
		tf->rate = 30;
		tf->status |= SMPTE_FRAMEDROP;
		break;
	case 0x04:	/* 30 frame non-drop */
		tf->rate = 30;
		tf->status &= ~SMPTE_FRAMEDROP;
		break;
	default:
		return (-1);
	}
	return (0);
}

/*
 * return -1 if t1 < t2
 *         0 if t1 == t2
 *         1 if t1 > t2
 */
int
midi_compare_SMPTE_time(t1, t2)
	struct SMPTE_frame *t1;
	struct SMPTE_frame *t2;
{

	if (t1->hour < t2->hour)
		return (-1);
	else if (t1->hour > t2->hour)
		return (1);
	if (t1->minute < t2->minute)
		return (-1);
	else if (t1->minute > t2->minute)
		return (1);
	if (t1->second < t2->second)
		return (-1);
	else if (t1->second > t2->second)
		return (1);
	if (t1->frame < t2->frame)
		return (-1);
	else if (t1->frame > t2->frame)
		return (1);
	if (t1->fraction < t2->fraction)
		return (-1);
	else if (t1->fraction > t2->fraction)
		return (1);
	return (0);
}

/*
 * checks to see if we should send a signal and sends it if we
 * should.
 */
void
midi_send_sigio(softc)
	struct midi_softc *softc;
{

	if (softc->wqueue->count < MIDI_LOW_WATER) {
		softc->status &= ~MIDI_WR_BLOCK;
		if (softc->status & MIDI_WR_SLEEP) {
			softc->status &= ~MIDI_WR_SLEEP;
			wakeup((caddr_t)softc->wqueue);
		}
		if (softc->status & MIDI_ASYNC &&
		    (softc->status & MIDI_SENDIO || softc->wqueue->count
		    == 0)) {
			struct proc *p;

			if (softc->pgid < 0)
				signal(-softc->pgid, SIGIO);
			else if ((p = prfind(softc->pgid)) != 0)
				psignal(p, SIGIO);
		}
		softc->status &= ~MIDI_SENDIO;
		/* notify poll that writes will succeed */
		/*
		 * XXXPOLL - do I only want to do this is a poll is in
		 * progress and anyyet is 0?  Then I'll need to set
		 * a status flag, and how do I know when to unset it after
		 * the poll times out (ala select).
		 */
		pollwakeup(&softc->pollhead, POLLOUT);
	}
	if (softc->status & MIDI_FLUSH_SLEEP && softc->wqueue->count == 0) {
		softc->status &= ~MIDI_FLUSH_SLEEP;
		wakeup((caddr_t)&softc->status);
	}
}

void
midi_time_warp(softc)
	struct midi_softc *softc;
{
	struct proc *p;

	/* turn off any notes that may be on */
	midi_reset_devices(softc);

	if (softc->status & MIDI_ASYNC) {
		if (softc->pgid < 0)
			signal(-softc->pgid, SIGURG);
		else if ((p = prfind(softc->pgid)) != 0)
			psignal(p, SIGURG);
	}
	pollwakeup(&softc->pollhead, POLLPRI);
}

u_long
midi_get_timing_clock(softc)
	struct midi_softc *softc;
{
	u_long t;

	if (softc->status & MIDI_EXTCLK)
		t = softc->extclock;
	else
		t = (time.tv_sec * hz +
		    time.tv_usec / (1000000 / hz) - softc->start_time);
	return (t);
}

u_long
midi_get_smf_time(softc)
	struct midi_softc *softc;
{
	long long numerator, denominator;
	u_long smf, tc, tdiff, tempo;
	struct tempo_change *prev, *tchange;

	/*
	 * SMF time zero corresponds to SMPTE time 00:00:00:00 ??
	 * Not quite.  The way tclmidi seems to be working, we
	 * should do the following:
	 *
	 *  1) if extclock is less than the time corresponding to
	 *     the MetaSMPTE event, we should return SMF 0.
	 *  2) if extclock is past the time corresponding to the
	 *     MetaSMPTE event, we should return the SMF time
	 *     represented by extclock - MetaSMPTE extclock.
	 */
	tc = midi_get_timing_clock(softc);

	/* find last tempo change */
	for (tchange = softc->tempo_changes, prev = NULL;
	    tchange != NULL;
	    prev = tchange, tchange = tchange->next) {
		if (tc <= tchange->time)
			break;
	}
	if (prev == NULL) {
		/*
		 * no tempos found -- default tempo for MIDI files is
		 * 120 bpm.
		 */
		tempo = 500000;
		tdiff = 0;
		smf = 0;
	} else {
		tempo = prev->tempo;
		tdiff = tc - prev->time;
		smf = prev->smf_time;
	}

	numerator = (long long)softc->division * 1000000 * tdiff;
	denominator = (long long)tempo * softc->hz;
	smf += numerator / denominator;
#ifdef GJWDEBUG
	printf("midi_get_smf_time() returning %lu\n", smf);
#endif
	return (smf);
}

void
midi_initq(eq)
	struct event_queue *eq;
{

	eq->count = 0;
	eq->end = &eq->events[MIDI_Q_SIZE - 1];
	eq->head = eq->events;
	eq->tail = eq->events;
	/* zero events to clear stynamic stuctures */
	bzero(eq->events, MIDI_Q_SIZE * sizeof(struct event));
}

void
midi_copy_event(e1, e2)
	struct event *e1, *e2;
{
	short i;

	e1->time = e2->time;
	e1->smf_time = e2->smf_time;
	e1->type = e2->type;
	e1->tempo = e2->tempo;
	stynamic_release(&e1->data);
	stynamic_append(&e1->data, &e2->data);
}

int
midi_deq(eq, event)
	struct event_queue *eq;
	struct event **event;
{
	int s;

	s = spltty();
	if (eq->count == 0) {
		splx(s);
		return (0);
	}
	if (event == NULL)
		eq->head++;
	else
		*event = eq->head++;
	if (eq->head > eq->end)
		eq->head = eq->events;
	eq->count--;
	splx(s);
	return (1);
}

int
midi_peekq(eq, event)
	struct event_queue *eq;
	struct event **event;
{
	int s;

	s = spltty();
	if (eq->count == 0) {
		splx(s);
		return (0);
	}
	*event = eq->head;
	splx(s);
	return (1);
}

int
midi_browseq(eq, offset, event)
	struct event_queue *eq;
	u_long offset;
	struct event **event;
{
	int s;

	s = spltty();
	if (eq->count <= offset) {
		splx(s);
		return (0);
	}
	*event = (eq->head + offset);
	splx(s);
	return (1);
}

int
midi_enq(eq, event)
	struct event_queue *eq;
	struct event *event;
{
	int s;

	s = spltty();
	if (eq->count == MIDI_Q_SIZE) {
		splx(s);
		return (0);
	}
	midi_copy_event(eq->tail++, event);
	if (eq->tail > eq->end)
		eq->tail = eq->events;
	eq->count++;
	splx(s);
	return (1);
}

void
stynamic_add_byte(sty, byte)
	struct stynamic *sty;
	u_char byte;
{
	int s;
	u_char *new_ptr;

	s = spltty();
	if (sty->len < STYNAMIC_SIZE)
		sty->datas[sty->len++] = byte;
	else {
		if (sty->len < sty->allocated) {
			sty->datad[sty->len++] = byte;
		} else {
			new_ptr = kmem_alloc(sty->allocated + STYNAMIC_ALLOC,
			    KM_NOSLEEP);
			if (new_ptr == NULL) {
				printf("midi: out of memory\n");
				return;
			}
			if (sty->allocated == 0)
				bcopy(sty->datas, new_ptr, sty->len);
			else {
				bcopy(sty->datad, new_ptr, sty->len);
				kmem_free(sty->datad, sty->allocated);
			}
			sty->datad = new_ptr;
			sty->datad[sty->len++] = byte;
			sty->allocated += STYNAMIC_ALLOC;
		}
	}
	splx(s);
}

void
stynamic_add_bytes(sty, bytes, num)
	struct stynamic *sty;
	u_char *bytes;
	int num;
{
	int s, size_inc;
	u_char *new_ptr;

	s = spltty();
	if (sty->len + num <= STYNAMIC_SIZE) {
		bcopy(bytes, &sty->datas[sty->len], num);
		sty->len += num;
	} else {
		if (sty->len + num <= sty->allocated) {
			bcopy(bytes, &sty->datad[sty->len], num);
			sty->len += num;
		} else {
			size_inc = (num / STYNAMIC_ALLOC + 1) * STYNAMIC_ALLOC;
			new_ptr = kmem_alloc(sty->allocated + size_inc,
			    KM_NOSLEEP);
			if (sty->allocated == 0)
				bcopy(sty->datas, new_ptr, sty->len);
			else {
				bcopy(sty->datad, new_ptr, sty->len);
				kmem_free(sty->datad, sty->allocated);
			}
			sty->datad = new_ptr;
			bcopy(bytes, &sty->datad[sty->len], num);
			sty->allocated += size_inc;
			sty->len += num;
		}
	}
	splx(s);
}

u_char
stynamic_get_byte(sty, index)
	struct stynamic *sty;
	int index;
{
	int s;

	s = spltty();
	if (sty->len <= 4)
		return (sty->datas[index]);
	else
		return (sty->datad[index]);
	splx(s);
}

void
stynamic_copy(sty, dest, len)
	struct stynamic *sty;
	void *dest;
	int len;
{
	int s;

	s = spltty();
	if (sty->len <= 4)
		bcopy(sty->datas, dest, len);
	else
		bcopy(sty->datad, dest, len);
	splx(s);
}

void
stynamic_append(dest, src)
	struct stynamic *dest;
	struct stynamic *src;
{

	if (src->len <= 4)
		stynamic_add_bytes(dest, src->datas, src->len);
	else
		stynamic_add_bytes(dest, src->datad, src->len);
}

#if 0
/* unused function */
void
stynamic_copy_from(sty, index, dest, len)
	struct stynamic *sty;
	int index, len;
	void *dest;
{
	int s;

	s = spltty();
	if (sty->len <= 4)
		bcopy(&sty->datas[index], dest, len);
	else
		bcopy(&sty->datad[index], dest, len);
	splx(s);
}
#endif


void
stynamic_release(sty)
	struct stynamic *sty;
{
	int s;

	s = spltty();
	if (sty->len > STYNAMIC_SIZE)
		kmem_free(sty->datad, sty->allocated);
	sty->len = 0;
	sty->allocated = 0;
	sty->datad = NULL;
	bzero(sty->datas, STYNAMIC_SIZE);
	splx(s);
}

void
stynamic_shift(sty, num)
	struct stynamic *sty;
	int num;
{
	int rem, s;
	u_char *ptr;

	if (sty->len <= num) {
		stynamic_release(sty);
		return;
	}
	s = spltty();
	if (sty->len > STYNAMIC_SIZE)
		ptr = &sty->datad[num];
	else
		ptr = &sty->datas[num];
	rem = sty->len - num;
	if (rem > STYNAMIC_SIZE)
		bcopy(ptr, sty->datad, rem);
	else {
		bcopy(ptr, sty->datas, rem);
		if (sty->datad != NULL) {
			kmem_free(sty->datad, sty->allocated);
			sty->datad = NULL;
			sty->allocated = 0;
		}
	}
	splx(s);
	sty->len = rem;
}

void
stynamic_init(sty)
	struct stynamic *sty;
{
	int s;

	s = spltty();
	sty->len = 0;
	sty->allocated = 0;
	sty->datad = NULL;
	bzero(sty->datas, STYNAMIC_SIZE);
	splx(s);
}

#if 0
/* unused function */
#endif
void
stynamic_print(sty)
	struct stynamic *sty;
{

	printf("\t\tlen = %d\n", sty->len);
	printf("\t\tallocated = %d\n", sty->allocated);
	printf("\t\tdatad = 0x%x\n", sty->datad);
}
